/* 
 * lndir - create shadow link tree
 *
 * By Christer Bernerus <bernerus@cs.chalmers.se> based on
 *     Stephen Gildea's lndir.sh
 * 
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <sysexits.h>
#include <strings.h>
#include <errno.h>

char *myname;
char curdir[2048];
char realtotop[2048];


int nflag = 0;
int paranoid = 1;
int verbose = 0;
int errflg = 0;
int allflag = 0;
int nofork = 1;
int update=0;
int parallell=0;
#define MAX_SKIPCNT 256
char *skiplist[MAX_SKIPCNT];
int skipcount=0;
char *fromtop, *totop;
extern int errno;

usage()
{
    fprintf(stderr,"Usage: %s [-anvfpu] [-s file [-s file] ... ] fromdir [todir]\n",myname);
    fprintf(stderr,"       -n          - Don't do it, produce a shell script\n");
    fprintf(stderr,"       -v          - Verbose output\n");
    fprintf(stderr,"       -F          - Do subdirs in separate processes\n");
    fprintf(stderr,"       -p          - Do not wait for child processes\n");
    fprintf(stderr,"       -f          - Force. Disable consinstency checks\n");
    fprintf(stderr,"       -u          - Update, tree might contain links\n");
    fprintf(stderr,"       -s filename - Skip filename.(Max %d items) \n",MAX_SKIPCNT);
}

main(argc,argv) char **argv; int argc;
{
    extern int optind, opterr;
    extern char* optarg;
    char c;

    myname = argv[0];

    while((c = getopt(argc, argv, "nvpFfus:")) != EOF)
    {
	switch(c)
	{
		case 'F':
			nofork=0;
			break;
		case 'p':
			parallell=1;
			break;
		case 'n':
			nflag++;
			break;
		case 'v':	
			verbose++;
			break;
		case 'u':
			update=1;
			break;
		case 'f':
			paranoid=0;
			break;
		case 's':
			if(skipcount >= MAX_SKIPCNT)
			{
				printf("Too many items in skip list, max is %d\n",MAX_SKIPCNT);
				exit(EX_USAGE);
			}
			skiplist[skipcount++]=optarg;
			break;
		case '?':
			errflg++;
			break;
	}
    }
    
    if(optind  >= argc)
	errflg++;
    if(argc - optind > 2 )
	errflg++;

    if(errflg)
    {
	usage();
	  exit(EX_USAGE);
    }
    
    fromtop = argv[optind++];
   
    if(optind - argc > 0)
	totop = argv[optind];
    else
	totop = ".";

    if ( ! isdir(totop))
    {
	fprintf(stderr,"%s: %s is not a directory\n",myname,totop);
	usage();
	exit(EX_USAGE);
    }

    if ( ! isdir(totop))
    {
	fprintf(stderr,"%s: %s is not a directory\n",myname,fromtop);
	usage();
	exit(EX_USAGE);
    }

    /* Find out the real name for totop */
    getwd(curdir);
    if(chdir(totop) < 0)
    {
	perror("chdir");
	exit(EX_OSERR);
    }
    getwd(realtotop);
    if(chdir(curdir) < 0)
    {
	perror("chdir");
	exit(EX_OSERR);
    }
	
    strcpy(curdir,totop);

    lndir(fromtop,totop);
    exit(EX_OK);
}


lndir(dirfrom,dirto) 
char *dirfrom, *dirto;
{
    DIR *dir;
    struct direct *dp;
    int pid;
    char *ri;

    if(!nflag && paranoid && issame(dirfrom,realtotop))
    {
	fprintf(stderr,"%s: FROM (%s) and TOP (%s) are equal\n",
		myname,dirfrom,realtotop);
	fprintf(stderr,"This would have caused an infinite recursion\n");
	exit(EX_DATAERR);
	
    }
    if(paranoid && issame(dirfrom,dirto))
    {
	fprintf(stderr,"%s: FROM (%s) and TO (%s) are identical!\n",myname,dirfrom,dirto);
	exit(EX_DATAERR);
    }

    if(verbose)
	printf("# lndir %s %s\n",dirfrom,dirto);

    dir = opendir(dirfrom);
    if(dir == NULL)
    {
	printf("Cannot open directory %s\n",dirfrom);
	return;
    }

    while((dp=readdir(dir)) != NULL)
    {
	char fromfile[1024];
	char fromdir[1024];
	char thisdir[1024];

	int skipit = 0;

	if(strcmp(dp->d_name,".") == 0 ) 
	    continue;
	if(strcmp(dp->d_name,"..") == 0 ) 
	    continue;

	sprintf(fromfile,"%s/%s",dirfrom,dp->d_name);


	if(skipcount)
	{
	    int i;

	    for(i=0; i<skipcount; i++)
	    {
		if(strcmp(dp->d_name,skiplist[i]) == 0)
		{
		    if(verbose)
		    {
			printf("# Skipped %s/%s\n",curdir,dp->d_name);
		    }
		    skipit = 1;
		    break;
		}
	    }
	    if(skipit)
		continue;
	}

	if( ! isdir(fromfile) ) /* Plain file */
	{
	    if(!paranoid  || isnt(dp->d_name))
	    {
		if(nflag)
		    printf("ln -s %s %s\n",fromfile, dp->d_name);
		else
		    if(symlink(fromfile, dp->d_name) < 0)
		    {
			if(!update || errno != EEXIST)
			    perror(myname);
		    }
		    else
		    {
			if(verbose>2 || update)
			    printf("# %s -> %s\n",dp->d_name,fromfile);
		    }
	    }

	}
	else /* Directory */
	{

	    if(nflag)
		printf("mkdir %s/%s\n",curdir,dp->d_name);
	    else
		if(mkdir(dp->d_name,0777) < 0)
		{
		    if(!update || errno != EEXIST)
			perror(myname);
		}
		else
		    if(verbose>1 || update)
			printf("# Made directory %s/%s\n",curdir,dp->d_name);

	    if(!nflag && !nofork)
	    {
	    	pid = fork();

	    	if(pid < 0)
	    	{
			perror(myname);
			exit(EX_OSERR);
	    	}

	    }
	    else 
		pid=0;

	    if(pid == 0)
	    {
		/* Child code */
		if(nflag)
		{
			printf("cd %s\n",dp->d_name);
		}
		else
		{
		    if(chdir(dp->d_name) < 0)
		    {
			perror("chdir");
			exit(EX_OSERR);
		    }
		}

		strcat(curdir,"/");
		strcat(curdir,dp->d_name);

		if(!nflag)
		{
		    char * df=dirfrom;
		    strcpy(fromdir,dirfrom);

		    if(dirfrom[0]!='/')
		    {
			sprintf(fromdir,"../%s",dirfrom);
			df = fromdir;
		    }

		    sprintf(fromfile,"%s/%s",df,dp->d_name);

		    if(paranoid)
		    {
			getwd(thisdir);

			if(issame(thisdir,fromfile))
			{
			    fprintf(stderr,"%s: FROM (%s) and TO (%s) are identical!\n",myname,thisdir,fromfile);
			    exit(EX_DATAERR);
			}
		    }
		}

		lndir(fromfile,".");

		if(nflag)
			printf("cd ..\n");
		else
		    if(nofork)
		    {
			if(chdir("..") < 0)
			{
			    perror("chdir");
			    exit(EX_OSERR);
			}
		    }

		if(ri=rindex(curdir,'/'))
			*ri = '\0';
		if(!nflag && !nofork)
		    exit(EX_OK);
	    }
	    else
	    {
		/* Parent */
		if(!parallell)
		wait(0);
	    }

	}
    }
    closedir(dir);
}

isdir(s)
char *s;
{
    struct stat buf;

    if(lstat(s,&buf) < 0)
	return 0;
    if(buf.st_mode & S_IFDIR)
	return 1;
    return 0;
}

isnt(s)
char *s;
{
    struct stat buf;

    if(stat(s,&buf) < 0)
	return 1;
    return 0;
}

issame(a,b)
char *a, *b;
{
    struct stat abuf, bbuf;

    if(stat(a,&abuf) < 0)
	return 0;
    if(stat(b,&bbuf) < 0)
	return 0;
    if(abuf.st_ino != bbuf.st_ino)
	return 0;
    if(abuf.st_dev != bbuf.st_dev)
	return 0;
    return 1;
}
