/*
 * ninetd - an alternative inetd.  This one implements restriction
 *	    lists, per-connect logging, usage statistics logging,
 *	    and per interface services.  The latter allows UDP
 *	    servers to be implemented properly on routers.
 */
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <errno.h>
#include <stdio.h>
#include <ctype.h>
#include <strings.h>
#include <signal.h>
#include <netdb.h>
#include <syslog.h>
#include <pwd.h>

#define	STREQ(a, b)	(*(a) == *(b) && strcmp((a), (b)) == 0)
#define	STRNEQ(a, b, n)	(*(a) == *(b) && strncmp((a), (b), (n)) == 0)

/*
 * We retry things we can't get open right away.  This is the
 * retry period.
 */
#define	RETRYTIME	(60*5)

/*
 * We expect child and alarm interrupts, and use the hangup
 * interrupt to reconfigure.
 */
#define	BLOCKMASK	(sigmask(SIGCHLD)|sigmask(SIGHUP)|sigmask(SIGALRM))
#define	BLOCKSIGS(osig)		(osig) = sigblock(BLOCKMASK)
#define	UNBLOCKSIGS(osig)	(void) sigsetmask((osig))

/*
 * This server really only understands TCP and UDP sockets.  These
 * are the types.
 */
#define	T_TCP		1
#define	T_UDP		2


/*
 * Length of an interface name.  This is hardcoded in struct ifreq's.
 */
#define	INTER_NAME_LEN	16

/*
 * Maximum length of an argument list for an external server.
 */
#define	MAXARGV		8

/*
 * Maximum concatenated length of a line in the configuration file
 */
#define MAXLINE		2048

/*
 * Absolute maximum number of interfaces.  We actually declare
 * space as needed.
 */
#define	MAXINTERFACES	64

/*
 * Maximum number of restrictions allowed
 */
#define	MAXRESTRICT	512

/*
 * Buffer size we use for things
 */
#define	BUFFERSIZE	4096


/*
 * Masks to use as defaults for address-and-mask restrictions
 */
#define	DEFMASK		0
#define	HOSTMASK	0xffffffff


/*
 * Configuration keyword table
 */
struct keyword {
	char *word;
	int numargs;
	int key;
};

/*
 * Generic identifiers
 */
#define	KEY_ERROR	(-1)
#define	KEY_NONE	0

/*
 * Configuration keyword identifiers
 */
#define	CF_INTERFACE	1
#define	CF_STATS	2
#define	CF_CONNECT	3
#define	CF_LOG		4
#define	CF_RESTRICT	5
#define	CF_USER		6
#define	CF_PROGRAM	7
#define	CF_INTERNAL	8

/*
 * Configuration keyword tables
 */
struct keyword config_keywords[] = {
	{ "interface",	1,	CF_INTERFACE },
	{ "stats",	1,	CF_STATS },
	{ "connect",	0,	CF_CONNECT },
	{ "log",	0,	CF_LOG },
	{ "restrict",	1,	CF_RESTRICT },
	{ "user",	1,	CF_USER },
	{ "program",	0,	CF_PROGRAM },
	{ "internal",	1,	CF_INTERNAL },
	{ "",		0,	KEY_NONE }
};


/*
 * Restriction keywords and table
 */
#define	RS_ALLOW	1
#define	RS_DISALLOW	2

struct keyword restrict_keywords[] = {
	{ "allow",	1,	RS_ALLOW },
	{ "disallow",	1,	RS_DISALLOW },
	{ "",		0,	KEY_NONE }
};


/*
 * Some simple services are provided internally.  These are defined
 * following.
 */
int tcp_echo(), udp_echo(), tcp_discard(), udp_discard();
int tcp_time(), udp_time(), tcp_daytime(), udp_daytime();
int tcp_chargen(), udp_chargen(), tcp_dropconn();

struct internal {
	char	*service;		/* internally provided service name */
	short	type;			/* type of service supported */
	short	fork;			/* 1 if should fork before call */
	int	(*doit_rtn)();		/* function which performs it */
} internal_servers[] = {
	/* Echo received data */
	"tcp_echo",	T_TCP,	1,	tcp_echo,
	"udp_echo",	T_UDP,	0,	udp_echo,

	/* Internet /dev/null */
	"tcp_discard",	T_TCP,	1,	tcp_discard,
	"udp_discard",	T_UDP,	0,	udp_discard,

	/* Return 32 bit time since 1970 */
	"tcp_time",	T_TCP,	0,	tcp_time,
	"udp_time",	T_UDP,	0,	udp_time,

	/* Return human-readable time */
	"tcp_daytime",	T_TCP,	0,	tcp_daytime,
	"udp_daytime",	T_UDP,	0,	udp_daytime,

	/* Familiar character generator */
	"tcp_chargen",	T_TCP,	1,	tcp_chargen,
	"udp_chargen",	T_UDP,	0,	udp_chargen,

	/* Service to simply drop the connection, for logging */
	"tcp_dropconn",	T_TCP,	0,	tcp_dropconn,
	NULL,		0,	0,	0
};


/*
 * An external server is known by the executable to run and
 * the argv list it is to be run with.
 */
struct external {
	char *executable;
	char *argv[MAXARGV+1];
};


/*
 * We keep a list of interfaces we can see on the machine for
 * occasions when we have to open multiple sockets.
 */
struct interface {
	char name[INTER_NAME_LEN];	/* length is hardcoded in ifreq */
	struct in_addr address;		/* address of interface */
	int flags;			/* flags for the interface */
};

#define	INT_DEFAULT	0x1		/* this is default interface */
#define	INT_INCOMPLETE	0x2		/* incomplete address */

/*
 * Restrictions.  These are done by address-and-mask matches.
 */
struct restrict {
	u_long address;
	u_long mask;
	int allow;
};


/*
 * Server description.  An individual server is identified by
 * the triple {protocol, port#, interface}.  Protocol is always
 * T_TCP or T_UDP, while the interface is either one of the
 * interfaces on the machine, or default.
 */
struct server {
	struct server *next;
	int type;			/* TCP or UDP */
	char *name;			/* name of service */
	struct sockaddr_in address;	/* address & port number bound */
	int interface_index;		/* index into interface array */
	int flags;			/* flags for this server */
	char *user;			/* name of user to use when running */
	struct internal *internal_handler;	/* internal handler, if any */
	struct external	*external_handler;	/* external handler otherwise */
	int fd;				/* file descriptor */
	struct timeval laststattime;	/* last time stats printed */
	int statinterval;		/* stat print interval */
	int statcount;			/* count of requests since last time */
	struct restrict *reslist;	/* restrict list */
	int len_reslist;		/* length of the restrict list */
	int child_pid;			/* single-threaded server */
};


/*
 * Flags used in the server description
 */
#define	SERVE_DELETE		0x1	/* delete after merge operation */
#define	SERVE_LOG		0x2	/* connection logging set */
#define	SERVE_STATS		0x4	/* print stats once in a while */
#define	SERVE_CONNECT		0x8	/* datagram service does connect */
#define	SERVE_HAVE_CHILD	0x10	/* we have an active child */
#define	SERVE_DOWN		0x20	/* this service is down (retry) */


/*
 * Clean environment for running things with
 */
char *clean_envp[] = {
	"TERM=network",
	"HOME=/",
	0
};


/*
 * The interface and server list heads.
 */
struct server *servlist = 0;
struct interface *interlist = 0;
int ninterfaces;

/*
 * File descriptor mask maintenance.  For select.
 */
int nsockets;			/* number of open sockets */
int maxsockets;			/* highest open fd number */
fd_set allsockets;		/* mask of sockets to check */


/*
 * Counters so the timer routines will know we have something to do.
 */
int numserverdown;		/* number of servers down */
int numlogging;			/* number of servers requiring logging */
int isalarmed;			/* set when alarm is set */
int waitingforchild;		/* keep count of children being waited upon */


/*
 * Configuration file path
 */
char *config_file = "/etc/ninetd.conf";

/*
 * Miscellany
 */
char *progname;
int debug = 0;
extern int errno;

char **Argv;
char *LastArg;


/*
 * main - parse arguments and handle options
 */
main(argc, argv, envp)
int argc;
char *argv[];
char *envp[];
{
	register struct server *svp;
	int c;
	int errflg = 0;
	int osig;
	fd_set readable;
	extern int optind;
	extern char *optarg;
	void alarmed();
	void configure();
	void catch_child();
	void detach();
	void setsignals();
	void process();
	void getinterfaces();

	Argv = argv;
	if (envp == 0 || *envp == 0)
		envp = argv;
	while (*envp)
		envp++;
	LastArg = envp[-1] + strlen(envp[-1]);

	progname = argv[0];
	while ((c = getopt(argc, argv, "df:")) != EOF)
		switch (c) {
		case 'd':
			++debug;
			break;
		case 'f':
			config_file = optarg;
			break;
		default:
			errflg++;
			break;
		}
	if (errflg || optind != argc) {
		(void) fprintf(stderr,
		    "usage: %s [-d] [-f config]\n", progname);
		exit(2);
	}

	/*
	 * If not debugging, detach us from the controlling terminal
	 */
	if (!debug)
		detach();

	/*
	 * Open the log file
	 */
#ifdef	LOG_DAEMON
	openlog("ninetd", LOG_PID | LOG_NOWAIT, LOG_DAEMON);
#else
	/* OH joy, 4.2 BSD syslog. Grump. */
	openlog("ninetd", LOG_PID);
#endif

	/*
	 * Get interface list for host
	 */
	getinterfaces();

	/*
	 * Block signals, set up signal handlers and read the
	 * configuration file.
	 */
	BLOCKSIGS(osig);
	setsignals(alarmed, configure, catch_child);
	configure();
	UNBLOCKSIGS(osig);

	/*
	 * Now the loop.  Select on the current mask, see who
	 * is ready when we wake up, and go to it.
	 */
	for (;;) {

		/*
		 * If no sockets are open, wait for someone to
		 * reconfigure us.
		 */
		if (nsockets == 0) {
			BLOCKSIGS(osig);
			while (nsockets == 0)
				(void) sigpause(0L);
			UNBLOCKSIGS(osig);
		}

		/*
		 * Wait for something to do
		 */
		readable = allsockets;
		c = select(maxsockets+1, &readable, (fd_set *)NULL,
		    (fd_set *)NULL, (struct timeval *)NULL);
		if (c <= 0) {
			if (c < 0 && errno != EINTR) {
				syslog(LOG_WARNING, "select: %m");
				sleep(1);
			}
			continue;
		}

		/*
		 * Search the server list.  Block signals while doing this.
		 * Process anything which is ready.
		 */
		BLOCKSIGS(osig);
		for (svp = servlist; svp != 0; svp = svp->next)
			if (svp->fd > 0 && FD_ISSET(svp->fd, &readable))
				process(svp);
		UNBLOCKSIGS(osig);
	}
}


/*
 * process - process a service request
 */
void
process(serverp)
	struct server *serverp;
{
	register struct server *svp;
	int n;
	int pid;
	int svc_fd;
	int dofork;
	int findid;
	int size;
	struct sockaddr_in peerid;
	int okayhost();
	void eatpacket();
	char *hosttoa();

	svp = serverp;

	/*
	 * Discover a couple of processing options
	 */
	if (svp->internal_handler == NULL ||
	    svp->internal_handler->fork)
		dofork = 1;
	else
		dofork = 0;

	if ((svp->flags & SERVE_LOG) || svp->len_reslist > 0)
		findid = 1;
	else
		findid = 0;

	/*
	 * Do protocol specific processing
	 */
	switch (svp->type) {
	case T_TCP:
		if (debug)
			(void) fprintf(stderr,
			    "process: someone wants %s/tcp\n", svp->name);
		svc_fd = accept(svp->fd, (struct sockaddr *)0, (int *)0);
		if (svc_fd < 0) {
			if (errno != EINTR)
				syslog(LOG_WARNING, "accept: %m");
			return;
		}

		if (findid) {
			size = sizeof(struct sockaddr_in);
			if (getpeername(svc_fd, (struct sockaddr *)&peerid,
			    &size) < 0) {
				syslog(LOG_WARNING, "getpeername: %m");
				(void) close(svc_fd);
				return;
			}
		}
		break;

	case T_UDP:
		(void) fprintf(stderr, "someone wants %s/udp\n", svp->name);
		svc_fd = svp->fd;
		if (findid) {
			size = sizeof(peerid);
			n = recvfrom(svc_fd, (char *)&n, 0, MSG_PEEK,
			    (struct sockaddr *)&peerid, &size);
			if (n < 0) {
				syslog(LOG_WARNING, "recvfrom(MSG_PEEK): %m");
				eatpacket(svc_fd);
				return;
			}
		}
		break;

	default:
		syslog(LOG_ERR, "hideous internal error! type = %d", svp->type);
		exit(1);
		/*NOTREACHED*/
	}

	/*
	 * Check for restrictions.  If none, but logging is on, log
	 * the attempt.  If counting accesses, do that too.
	 */
	if (svp->len_reslist > 0 && svp->reslist != NULL) {
		if (!okayhost(&peerid, svp->reslist, svp->len_reslist)) {
			if (svp->flags & SERVE_LOG)
				syslog(LOG_ERR,
				    "%s/%s: %s.%d denied access on %s",
				    svp->name,
				    ((svp->type == T_TCP) ? "tcp" : "udp"),
				    hosttoa(&peerid), ntohs(peerid.sin_port),
				    interlist[svp->interface_index].name);
			if (svp->type == T_TCP)
				(void) close(svc_fd);
			else
				eatpacket(svc_fd);
			return;
		}
	}

	if (svp->flags & SERVE_LOG) {
		syslog(LOG_INFO, "%s/%s: %s.%d on %s", svp->name,
		    ((svp->type == T_TCP) ? "tcp" : "udp"),
		    hosttoa(&peerid), ntohs(peerid.sin_port),
		    interlist[svp->interface_index].name);
	}

	if (svp->flags & SERVE_STATS) {
		svp->statcount++;
	}


	/*
	 * If we have to fork, do it now.
	 */
	if (dofork) {
		pid = fork();
		if (pid < 0) {
			/* boo-boo */
			if (svp->type == T_TCP)
				(void) close(svc_fd);
			else
				eatpacket(svc_fd);
			sleep(1);
			return;
		}

		if (pid > 0) {
			/* parent */
			if (svp->type == T_UDP &&
			    !(svp->flags & SERVE_CONNECT)) {
				svp->flags |= SERVE_HAVE_CHILD;
				svp->child_pid = pid;
				FD_CLR(svp->fd, &allsockets);
				nsockets--;
				waitingforchild++;
			}
			if (svp->type == T_TCP)
				(void) close(svc_fd);
			return;
		}

		if (debug) {
			/* detach us from terminal */
			if ((n = open("/dev/tty", O_RDWR)) >= 0) {
				(void) ioctl(n, TIOCNOTTY, 0);
				(void) close(n);
			}
			(void) setpgrp(0, 0);
			(void) signal(SIGTSTP, SIG_IGN);
			(void) signal(SIGTTIN, SIG_IGN);
			(void) signal(SIGTTOU, SIG_IGN);
		}

		/*
		 * close everything we aren't going to need
		 */
		for (n = maxsockets; n > 2; n--) {
			if (n != svc_fd)
				(void) close(n);
		}
	}

	/*
	 * If using an internal service, dispatch this now.
	 */
	if (svp->internal_handler != NULL) {
		svp->internal_handler->doit_rtn(svc_fd, svp);
		if (dofork)
			_exit(0);
		if (svp->type == T_TCP)
			(void) close(svc_fd);
		return;
	}

	/*
	 * Now we're to the nitty-gritty.  We have to exec
	 * a server.  Move the service fd onto the first
	 * three file descriptors.  If there was a user
	 * associated with this, set our gid to that.  Then
	 * run the service.
	 */
	(void) sigsetmask(0L);
	(void) dup2(svc_fd, 0);
	(void) close(svc_fd);
	(void) dup2(0, 1);
	(void) dup2(0, 2);

	if (svp->user) {
		struct passwd *pwd;

		pwd = getpwnam(svp->user);
		if (pwd == NULL) {
			syslog(LOG_ERR, "getpwnam: %s: no such user",
			    svp->user);
			if (svp->type != T_TCP)
				eatpacket(0);
			_exit(1);
		}

		if (pwd->pw_uid > 0) {
			(void) setgid((gid_t)pwd->pw_gid);
			initgroups(pwd->pw_name, pwd->pw_gid);
			(void) setuid((uid_t)pwd->pw_uid);
		}
	}

	execve(svp->external_handler->executable, svp->external_handler->argv,
	    clean_envp);

	/*
	 * If we got here, we're in trouble.  Print a message and
	 * exit.
	 */
	syslog(LOG_ERR, "execve %s: %m", svp->external_handler->executable);
	if (svp->type != T_TCP)
		eatpacket(0);
	_exit(1);
}


/*
 * eatpacket - read a packet from a datagram socket and toss it
 */
void
eatpacket(fd)
	int fd;
{
	char buf[64];

	(void) recv(fd, buf, sizeof buf, 0);
}



/*
 * detach - detach us from the controlling terminal
 */
void
detach()
{
	register int i;
	register int res;

	/*
	 * Fork us off.  Let the parent commit hara kiri.
	 */
	i = 5;
	for (;;) {
		res = fork();
		if (res == -1) {
			if (i-- > 0)
				sleep(3);
			else {
				perror(progname);
				exit(1);
			}
		} else if (res > 0) {
			exit(0);
		} else {
			break;
		}
	}

	/*
	 * Close all descriptors
	 */
	res = getdtablesize();
	for (i = 0; i < res; i++)
		(void) close(i);

	/*
	 * Open garbage to keep 0, 1 and 2 clear
	 */
	(void) open("/", O_RDONLY);
	(void) dup2(0, 1);
	(void) dup2(0, 2);

	/*
	 * Now detach us
	 */
	i = open("/dev/tty", O_RDWR);
	if (i > 0) {
		(void) ioctl(i, TIOCNOTTY, (char *)0);
		(void) close(i);
	}

	/*
	 * Restrain unnecessary signals
	 */
	(void) signal(SIGTSTP, SIG_IGN);
	(void) signal(SIGTTIN, SIG_IGN);
	(void) signal(SIGTTOU, SIG_IGN);
}



/*
 * setsignals - set up signal handlers
 */
void
setsignals(alarm_hndlr, hup_hndlr, child_hndlr)
	void (*alarm_hndlr)();
	void (*hup_hndlr)();
	void (*child_hndlr)();
{
	struct sigvec sv;

	bzero((char *)&sv, sizeof(sv));
	sv.sv_mask = BLOCKMASK;

	if (alarm_hndlr != 0) {
		sv.sv_handler = alarm_hndlr;
		sigvec(SIGALRM, &sv, (struct sigvec *)0);
	}
	if (hup_hndlr != 0) {
		sv.sv_handler = hup_hndlr;
		sigvec(SIGHUP, &sv, (struct sigvec *)0);
	}
	if (child_hndlr != 0) {
		sv.sv_handler = child_hndlr;
		sigvec(SIGCHLD, &sv, (struct sigvec *)0);
	}
}


/*
 * catch_child - catch SIGCHLD interrupts, inform waiting servers
 */
void
catch_child()
{
	register struct server *svp;
	union wait status;
	int pid;

	for (;;) {
		pid = wait3(&status, WNOHANG, (struct rusage *)0);
		if (pid <= 0)
			break;

		if (debug)
			(void) fprintf(stderr,
			    "catch_child: process %d terminated\n", pid);

		if (waitingforchild > 0) {
			for (svp = servlist; svp != NULL; svp = svp->next)
				if ((svp->flags & SERVE_HAVE_CHILD) &&
				    svp->child_pid == pid)
					break;
			if (svp != NULL) {
				if (status.w_status != 0)
				    syslog(LOG_WARNING,
					"%s/%s: exit status 0x%x", svp->name,
					((svp->type == T_TCP) ? "tcp" : "udp"),
					status.w_status);
				FD_SET(svp->fd, &allsockets);
				if (svp->fd > maxsockets)
					maxsockets = svp->fd;
				svp->flags &= ~SERVE_HAVE_CHILD;
				svp->child_pid = 0;
				nsockets++;
				waitingforchild--;
			}
		}
	}
}



/*
 * alarmed - handle an alarm interrupt
 */
void
alarmed()
{
	register struct server *svp;
	register long nowsec, nexttime, temp;
	int nsdown, nslogging, fd;
	struct timeval now;
	int opensocket();
	void printstats();

	nsdown = nslogging = 0;
	(void) gettimeofday(&now, (struct timezone *)0);
	nowsec = now.tv_sec;
	nexttime = -1;

	/*
	 * Run through the list looking for servers which
	 * need to be restarted and stats which need to be
	 * logged.  Keep a count of what you find.
	 */
	for (svp = servlist; svp != 0; svp = svp->next) {
		temp = -1;
		if (svp->flags & SERVE_DOWN) {
			nsdown++;
			temp = svp->laststattime.tv_sec + RETRYTIME - nowsec;
			if (temp <= 0) {
				svp->laststattime = now;
				fd = opensocket(svp->type, &svp->address);
				if (fd != 0) {
					/*
					 * Got this one open.  Record
					 * this fact.
					 */
					svp->flags &= ~SERVE_DOWN;
					svp->fd = fd;
					nsdown--;
				} else {
					temp = RETRYTIME;
				}
			}
		}

		if (!(svp->flags & SERVE_DOWN) && (svp->flags & SERVE_STATS)) {
			nslogging++;
			temp = svp->laststattime.tv_sec + svp->statinterval
			    - nowsec;
			if (temp < 0) {
				printstats(svp);
				svp->laststattime = now;
				temp = svp->statinterval;
			}
		}

		if (temp > 0) {
			if (nexttime < 0 || temp < nexttime)
				nexttime = temp;
		}
	}

	numserverdown = nsdown;
	numlogging = nslogging;

	if (nexttime > 0) {
		isalarmed = 1;
		alarm((int)nexttime);
	} else {
		isalarmed = 0;
	}
}



/*
 * setalarm - set an alarm interrupt for the next propitious moment
 */
void
setalarm(now)
	struct timeval *now;
{
	register struct server *svp;
	register long nowsec, nexttime, temp;

	nowsec = now->tv_sec;
	nexttime = -1;

	for (svp = servlist; svp != NULL; svp = svp->next) {
		if (svp->flags & SERVE_DOWN) {
			temp = svp->laststattime.tv_sec + 1 + RETRYTIME
			    - nowsec;
			if (temp <= 0)
				temp = 1;
			if (nexttime < 0 || temp < nexttime)
				nexttime = temp;
		} else if (svp->flags & SERVE_STATS) {
			temp = svp->laststattime.tv_sec + 1 + svp->statinterval
			    - nowsec;
			if (temp <= 0)
				temp = 1;
			if (nexttime < 0 || temp < nexttime)
				nexttime = temp;
		}
	}

	if (nexttime > 0) {
		isalarmed = 1;
		alarm((int)nexttime);
	} else {
		isalarmed = 0;
	}
}


/*
 * configure - read the configuration file, make our internal
 *	       configuration match
 */
void
configure()
{
	register char *tok;
	register struct server *svp;
	int n;
	int key;
	FILE *fp;
	char *linebp;
	char *cf_resfile;
	char *cf_interfaces;
	char *proto;
	char *arg1, *arg2;
	u_long ltemp;
	char linebuf[MAXLINE];
	struct restrict reslist[MAXRESTRICT];
	struct server svr;
	struct external ext;
	struct timeval now;
	char okayinter[MAXINTERFACES];
	char *getline();
	char *gettoken();
	struct external *get_external();
	struct internal *find_internal();
	int getrestrict();
	int getkeyword();
	int atouint();
	void addservice();
	void rmservice();
	void makeinterlist();
	void setalarm();
	void printservlist();

	/*
	 * Open the configuration file
	 */
	fp = fopen(config_file, "r");
	if (fp == NULL) {
		syslog(LOG_ERR, "can't open %s!: %m", config_file);
		return;
	}

	/*
	 * Mark all existing entries for deletion
	 */
	for (svp = servlist; svp != NULL; svp = svp->next)
		svp->flags |= SERVE_DELETE;

	/*
	 * Set number of failed opens and number of stat-printers
	 * to zero.  We'll count them again.
	 */
	numserverdown = 0;
	numlogging = 0;
	if (isalarmed) {
		alarm(0);
		isalarmed = 0;
	}

	/*
	 * Get current time
	 */
	(void) gettimeofday(&now, (struct timezone *)0);

	/*
	 * Read lines of input, processing until we hit the end
	 * of the file.
	 */
	while ((linebp = getline(fp, linebuf, MAXLINE)) != NULL) {
		bzero((char *)&svr, sizeof(svr));
		svr.laststattime = now;
		cf_resfile = cf_interfaces = NULL;

		/*
		 * Fill in the service spec
		 */
		if ((tok = gettoken(&linebp)) == NULL)
			continue;
		if (!getservice(tok, &svr.name, &proto, &svr.type,
		    &svr.address.sin_port)) {
			syslog(LOG_ERR,
		"invalid service specification %s in file %s, ignored",
			    tok, config_file);
			continue;
		}

		/*
		 * Okay.  Next are all keyword-argument combinations.
		 * Run through decoding these.
		 */
		while ((key = getkeyword(config_keywords, &linebp, &arg1,
		    &arg2)) != KEY_NONE && key != KEY_ERROR) {
			switch(key) {
			case CF_INTERFACE:
				cf_interfaces = arg1;
				break;
			
			case CF_STATS:
				if (!atouint(arg1, &ltemp)) {
					syslog(LOG_ERR,
				"stats interval %s undecodable", arg1);
					key = KEY_ERROR;
				} else {
					svr.statinterval = 60 * (int)ltemp;
					svr.flags |= SERVE_STATS;
				}
				break;

			case CF_CONNECT:
				if (svr.type == T_TCP) {
					syslog(LOG_ERR,
			"connect keyword inappropriate for a TCP service");
					key = KEY_ERROR;
				} else {
					svr.flags |= SERVE_CONNECT;
				}
				break;

			case CF_LOG:
				svr.flags |= SERVE_LOG;
				break;

			case CF_RESTRICT:
				cf_resfile = arg1;
				break;

			case CF_USER:
				svr.user = arg1;
				break;

			case CF_PROGRAM:
				if (get_external(&ext, &linebp) == NULL)
					key = KEY_ERROR;
				else
					svr.external_handler = &ext;
				break;

			case CF_INTERNAL:
				svr.internal_handler = find_internal(arg1);
				if (svr.internal_handler == NULL) {
					syslog(LOG_ERR,
				"internal server %s unknown", arg1);
					key = KEY_ERROR;
				}
				break;
			}

			if (key == KEY_ERROR || key == CF_PROGRAM ||
			    key == CF_INTERNAL)
				break;
		}

		if (key == KEY_ERROR) {
			syslog(LOG_ERR, "%s/%s ignored due to error",
			    svr.name, proto);
			continue;
		} else if (key == KEY_NONE) {
			syslog(LOG_ERR, "%s/%s: no program specification",
			    svr.name, proto);
			continue;
		}

		/*
		 * If we get to here, we have a valid service as far as
		 * this has gone, anyway.  If we have a restrict file,
		 * read it now.
		 */
		if (cf_resfile != NULL) {
			svr.len_reslist = getrestrict(cf_resfile, reslist,
			    MAXRESTRICT);
			if (svr.len_reslist > 0)
				svr.reslist = reslist;
		}

		/*
		 * Now the tricky part.  Run through the interface
		 * list, determining whether we should set up this
		 * service on the interface.  For each one we must
		 * open on, either update the existing server data
		 * or open a new service by copying what we have.
		 * Note that if no interface selection was given,
		 * we use the default.  Note also that the default
		 * is always opened first, if at all, this to avoid
		 * troubles with older multicasting kernels.
		 */
		makeinterlist(cf_interfaces, okayinter);
		for (n = 0; n < ninterfaces; n++)
			if (okayinter[n])
				addservice(n, &svr);
	}

	(void) fclose(fp);

	/*
	 * If we're here, the configuration has been completely read.
	 * Go through the server list looking for guys to delete.
	 */
	while (servlist != NULL && (servlist->flags & SERVE_DELETE)) {
		svp = servlist;
		servlist = svp->next;
		rmservice(svp);
	}

	if (servlist != NULL) {
		svp = servlist;
		while (svp->next != NULL) {
			if (svp->next->flags & SERVE_DELETE) {
				struct server *osvp;

				osvp = svp->next;
				svp->next = osvp->next;
				rmservice(osvp);
			} else {
				svp = svp->next;
			}
		}
	}

	/*
	 * Finally, if there is work to do on a timeout, set it up.
	 */
	if (numlogging > 0 || numserverdown > 0)
		setalarm(&now);

	if (debug)
		printservlist();
}



/*
 * makeinterlist - make a list of interfaces we are permitted to use
 */
void
makeinterlist(str, list)
	char *str;
	char *list;
{
	register int i;
	register char *cp;
	register char *ep;
	int not;
	u_long netnum;
	char entry[256];
	int decodenetnum();

	/*
	 * zero out the list
	 */
	cp = list;
	for (i = 0; i < ninterfaces; i++)
		cp[i] = 0;

	/*
	 * If str is null, use default interface.  Otherwise,
	 * use the list as given.
	 */
	if (str == NULL || *str == '\0') {
		/*
		 * Default interface always first
		 */
		list[0] = 1;
		return;
	}

	cp = str;

	for (;;) {
		/*
		 * Make positive assumption
		 */
		not = 0;

		/*
		 * Space past leading ','
		 */
		while (*cp == ',')
			cp++;

		/*
		 * If next is a !, this is instuction not to use
		 * interface.
		 */
		if (*cp == '!') {
			cp++;
			if (*cp == ',')
				continue;
			not++;
		}

		/*
		 * If this is the end of the list, return.  Nothing more
		 * to do.
		 */
		if (*cp == '\0')
			break;

		/*
		 * Got one.  Copy it out into temp storage
		 */
		ep = entry;
		while (*cp != ',' && *cp != '\0')
			*ep++ = *cp++;
		*ep = '\0';

		/*
		 * Check to see if this is "all".  If so, mark
		 * all interfaces that haven't been !'d as
		 * golden.
		 */
		if (STREQ("all", entry)) {
			if (not)
				continue;
			for (i = 1; i < ninterfaces; i++)
				if (list[i] != 2)
					list[i] = 1;
			continue;
		}

		/*
		 * Assume it is an interface name.  Search
		 * through the list looking for it.  If found, record
		 * this fact in the list.
		 */
		for (i = 0; i < ninterfaces; i++)
			if (STRNEQ(entry, interlist[i].name, INTER_NAME_LEN))
				break;
		if (i < ninterfaces) {
			if (not)
				list[i] = 2;
			else if (list[i] != 2)
				list[i] = 1;
			continue;
		}

		/*
		 * Maybe a network address.  See if we have a match.
		 */
		if (decodenetnum(entry, &netnum)) {
			for (i = 0; i < ninterfaces; i++)
				if (!(interlist[i].flags & INT_INCOMPLETE) &&
				    interlist[i].address.s_addr == netnum)
					break;
			if (i < ninterfaces) {
				if (not)
					list[i] = 2;
				else if (list[i] != 2)
					list[i] = 1;
				continue;
			}
		}

		/*
		 * Maybe do symbolic addresses here.
		 */
	}

	/*
	 * Go through, turning all 2's into 0's
	 */
	for (i = 0; i < ninterfaces; i++)
		if (list[i] == 2)
			list[i] = 0;
}


/*
 * addservice - add a service to the list, deleting the old version
 *		if appropriate.
 */
void
addservice(inter, newserver)
	int inter;
	struct server *newserver;
{
	register struct server *ns;
	register struct server *os;
	char *emalloc();
	char *strsave();
	struct external *copy_external();
	void free_external();
	void printstats();
	int opensocket();
	void closesocket();

	/*
	 * Search through the current list to see if
	 * we can find a match for this.
	 */
	ns = newserver;
	for (os = servlist; os != NULL; os = os->next)
		if (os->type == ns->type &&
		    os->interface_index == inter &&
		    os->address.sin_port == ns->address.sin_port)
			break;

	/*
	 * If we haven't got a structure for this, create one and
	 * link it to the list.  Otherwise, free up the details.
	 */
	if (os == NULL) {
		os = (struct server *)emalloc(sizeof(struct server));
		bzero((char *)os, sizeof(struct server));
		os->type = ns->type;
		os->interface_index = inter;
		os->address.sin_port = ns->address.sin_port;
		os->next = servlist;
		servlist = os;
	} else {
		if ((os->flags & SERVE_STATS) &&
		    os->statcount > 0)
			printstats(os);
		if (os->address.sin_addr.s_addr !=
		    interlist[inter].address.s_addr) {
			if (os->fd != 0) {
				closesocket(os->fd);
				os->fd = 0;
			}
			os->child_pid = 0;
		}
		(void) free(os->name);
		if (os->user != NULL)
			(void) free(os->user);
		os->internal_handler = NULL;
		if (os->external_handler != NULL) {
			free_external(os->external_handler);
			os->external_handler = NULL;
		}
		os->statcount = 0;
		if (os->len_reslist > 0 && os->reslist != NULL) {
			(void) free((char *)os->reslist);
			os->len_reslist = 0;
		}
	}

	/*
	 * By the time we get here we have an entry linked into
	 * the list which has barely been initialized.  Copy stuff
	 * from the new entry into this one.
	 */
	os->name = strsave(ns->name);
	os->flags = ns->flags;
	if (ns->user != NULL)
		os->user = strsave(ns->user);
	if (ns->external_handler != NULL) {
		os->external_handler = copy_external(ns->external_handler);
	} else {
		os->internal_handler = ns->internal_handler;
	}
	os->laststattime = ns->laststattime;
	os->statinterval = ns->statinterval;
	if (ns->len_reslist > 0) {
		os->reslist = (struct restrict *)emalloc(
		    ns->len_reslist * sizeof(struct restrict));
		bcopy((char *)ns->reslist, (char *)os->reslist,
		    ns->len_reslist * sizeof(struct restrict));
		os->len_reslist = ns->len_reslist;
	}

	/*
	 * If stats printing is on, count this in
	 */
	if (os->flags & SERVE_STATS)
		numlogging++;

	/*
	 * Finally, worry about the socket for I/O.  If it isn't
	 * open already, try to do so now.  If we can't get it
	 * open, mark for retry.
	 */
	if (os->fd == 0) {
		os->address.sin_family = AF_INET;
		if (!(interlist[inter].flags & INT_INCOMPLETE)) {
			os->address.sin_addr = interlist[inter].address;
			os->fd = opensocket(os->type, &os->address);
		}

		if (os->fd == 0) {
			os->flags |= SERVE_DOWN;
			numserverdown++;
		}
	}
}


/*
 * rmservice - free all resources associated with this service entry
 */
void
rmservice(svr)
	struct server *svr;
{
	void closesocket();
	void free_external();
	void printstats();

	if ((svr->flags & SERVE_STATS) && svr->statcount > 0)
		printstats(svr);

	(void) free(svr->name);
	if (svr->user != NULL)
		(void) free(svr->user);
	if (svr->external_handler != NULL)
		free_external(svr->external_handler);
	if (svr->len_reslist > 0 && svr->reslist != NULL)
		(void) free((char *)svr->reslist);
	if (svr->fd != 0)
		closesocket(svr->fd);
	(void) free((char *)svr);
}


/*
 * opensocket - open a socket and add it to the set we will
 *		select on.
 */
int
opensocket(type, addr)
	int type;
	struct sockaddr_in *addr;
{
	int sock_type;
	int fd;
	int on;

	if (type == T_TCP)
		sock_type = SOCK_STREAM;
	else if (type == T_UDP)
		sock_type = SOCK_DGRAM;
	else {
		syslog(LOG_ERR, "internal error: socket type %d unknown!",
		    type);
		exit(1);
		/*NOTREACHED*/
	}

	fd = socket(AF_INET, sock_type, 0);
	if (fd < 0) {
		syslog(LOG_ERR, "can't open socket type %s: %m",
		    ((type == T_TCP) ? "stream" : "datagram"));
		return 0;
	}

	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
	    < 0) {
		syslog(LOG_ERR, "setsockopt(SO_REUSEADDR): %m");
		(void) close(fd);
		return 0;
	}

	if (bind(fd, (struct sockaddr *)addr, sizeof(*addr)) < 0) {
		syslog(LOG_ERR, "%d/%s: bind: %m", htons(addr->sin_port),
		    ((type == T_TCP) ? "tcp" : "udp"));
		(void) close(fd);
		return 0;
	}

	if (type == T_TCP)
		(void) listen(fd, 10);

	FD_SET(fd, &allsockets);
	nsockets++;
	if (fd > maxsockets)
		maxsockets = fd;
	return fd;
}


/*
 * closesocket - close a socket and remove its bit from the select mask
 */
void
closesocket(fd)
	int fd;
{
	register int n;

	(void) close(fd);
	FD_CLR(fd, &allsockets);
	nsockets--;

	if (fd == maxsockets) {
		for (n = maxsockets - 1; n > 0; n--)
			if (FD_ISSET(n, &allsockets))
				break;
		if (n == 0 && nsockets > 0) {
			syslog(LOG_ERR,
			    "internal error: nsockets %d, no mask bits set",
			    nsockets);
			exit(1);
		}
		maxsockets = n;
	}
}




/*
 * getrestrict - read a restriction file and compile a restrict list
 */
int
getrestrict(file, list, maxentry)
	char *file;
	struct restrict *list;
	int maxentry;
{
	register struct restrict *lp;
	register struct restrict *lplast;
	register char *tok;
	register int key;
	char buf[1024];
	char *bufp, *arg1, *arg2;
	FILE *fp;
	u_long netnum;
	u_long netmask;
	char *getline();
	char *gettoken();
	int getkeyword();
	int getnetnum();
	int decodenetnum();

	/*
	 * Open the file if we can
	 */
	fp = fopen(file, "r");
	if (fp == NULL) {
		syslog(LOG_ERR, "can't open restrict file %s: %m", file);
		return 0;
	}

	/*
	 * Add a default entry first.  We assume initially that the default
	 * is to allow hosts.
	 */
	lp = list;
	lp->address = DEFMASK;
	lp->mask = DEFMASK;
	lp->allow = 1;
	lplast = lp + 1;

	/*
	 * Read the file line-by-line until we hit the end.
	 */
	while ((bufp = getline(fp, buf, sizeof buf)) != NULL) {
		key = getkeyword(restrict_keywords, &bufp, &arg1, &arg2);
		if (key == KEY_ERROR || key == KEY_NONE)
			continue;

		if (!getnetnum(arg1, &netnum)) {
			syslog(LOG_ERR, "restrict file %s: can't find host %s",
			    file, arg1);
			continue;
		}
		netnum = ntohl(netnum);

		netmask = HOSTMASK;
		tok = gettoken(&bufp);
		if (tok != NULL) {
			if (!STREQ(tok, "mask")) {
				syslog(LOG_ERR,
			"restrict file %s: inappropriate keyword %s",
				    file, tok);
				continue;
			}

			tok = gettoken(&bufp);
			if (tok == NULL) {
				syslog(LOG_ERR,
			"restrict file %s: no mask following keywords",
				    file);
				continue;
			}

			if (!decodenetnum(tok, &netmask)) {
				syslog(LOG_ERR,
			"restrict file %s: invalid net mask %s",
				    file, tok);
				continue;
			}
			netmask = ntohl(netmask);
		}
		netnum &= netmask;

		/*
		 * At this point we have a valid number and mask in
		 * host byte order.  Sort it into the existing array
		 * if we have room.
		 */
		if (lplast == list + maxentry) {
			syslog(LOG_ERR,
			    "restrict file %s: too many entries (> %d)",
			    file, maxentry);
			break;
		}

		for (lp = list + 1; lp < lplast; lp++)
			if (netnum < lp->address ||
			    (netnum == lp->address && netmask < lp->mask))
				break;
		
		if (lp < lplast) {
			register struct restrict *rp;

			for (rp = lplast; rp > lp; rp--)
				*rp = *(rp-1);
		}
		lplast++;

		lp->address = netnum;
		lp->mask = netmask;
		if (key == RS_ALLOW) {
			lp->allow = 1;
			list->allow = 0;
		} else {
			lp->allow = 0;
		}
	}

	/*
	 * Finally, done.  Return number of entries.
	 */
	return (lplast - list);
}


/*
 * okayhost - determine if a host is okay with the restriction list
 */
int
okayhost(hostp, list, nres)
	struct sockaddr_in *hostp;
	struct restrict *list;
	int nres;
{
	register struct restrict *lp;
	register struct restrict *lplast;
	register u_long netnum;
	register int allow;

	lp = list;
	if (lp == NULL || nres <= 0)
		return 1;
	lplast = lp + nres;

	netnum = ntohl(hostp->sin_addr.s_addr);

	allow = lp->allow;
	for (lp++; lp < lplast; lp++) {
		if ((netnum & lp->mask) == lp->address)
			allow = lp->allow;
		else if (netnum < lp->address)
			break;
	}

	return allow;
}


/*
 * Macros for classing characters
 */
#define	ISWHITE(c)	((c) == ' ' || (c) == '\t' || (c) == '\r')
#define	ISEOL(c)	((c) == '\n' || (c) == '#' || (c) == '\0')
#define	BSLASH		('\\')

/*
 * getline - read a line from the configuration file
 */
char *
getline(fp, linebuf, maxline)
	FILE *fp;
	char *linebuf;
	int maxline;
{
	register char *cp, *bp, *bq;
	register int continued;
	char buf[1024];

	cp = linebuf;
	*cp = '\0';

	while (&linebuf[maxline] - cp > 0) {
		if (fgets(buf, sizeof(buf), fp) == NULL) {
			if (cp == linebuf)
				return NULL;
			return linebuf;
		}

		/*
		 * Collect tokens
		 */
		bp = buf;
		for (;;) {
			/*
			 * Space past white space at begining
			 */
			while (ISWHITE(*bp))
				bp++;
			/*
			 * If we've hit the end of a line, quit
			 */
			if (ISEOL(*bp)) {
				continued = 0;
				break;
			}

			/*
			 * If we've found a backslash, determine whether
			 * the line is continued.
			 */
			if (*bp == BSLASH) {
				bq = bp+1;
				while (ISWHITE(*bq))
					bq++;
				if (ISEOL(*bq)) {
					continued = 1;
					break;
				}
			}

			/*
			 * Got a token, pointed at by bp.  Copy it into
			 * the line buffer and append a space
			 */
			do {
				if (cp >= &linebuf[maxline]) {
					continued = 1;
					break;
				}
				*cp++ = *bp++;
			} while (!ISWHITE(*bp) && !ISEOL(*bp));
			*cp++ = ' ';
			*cp = '\0';
		}

		/*
		 * Here we've ended the line.  If it was continued,
		 * or if we've collected nothing at all, go around
		 * again.  Otherwise, return what we have.
		 */
		if (continued == 0 && cp != linebuf)
			return linebuf;
	}

	/*
	 * If we're here the line was too long.  This is so
	 * unlikely we don't even try to recover.  Print an
	 * error and pretend we hit the end of the file.
	 */
	linebuf[32] = '\0';
	syslog(LOG_ERR, "processing of %s terminated, line too long: %s...",
	    config_file, linebuf);
	return NULL;
}


/*
 * gettoken - return the next token on the line
 */
char *
gettoken(line_nextc)
	char **line_nextc;
{
	register char *cp, *tp;

	tp = *line_nextc;
	while (*tp == ' ')
		tp++;

	cp = tp;
	while (*cp != ' ' && *cp != '\0')
		cp++;

	if (cp == tp) {
		*line_nextc = tp;
		if (debug >= 3)
			(void) fprintf(stderr, "gettoken: returning NULL\n");
		return NULL;
	}

	*cp++ = '\0';
	*line_nextc = cp;
	if (debug >= 3)
		(void) fprintf(stderr, "gettoken: returning `%s'\n", tp);
	return tp;
}


/*
 * getkeyword - read the next token and return the keyword it
 *		corresponds to
 */
int
getkeyword(keylist, line_nextc, arg1, arg2)
	struct keyword *keylist;
	char **line_nextc;
	char **arg1;
	char **arg2;
{
	register struct keyword *list;
	register char *cp;
	char *gettoken();

	cp = gettoken(line_nextc);
	if (cp == NULL)
		return KEY_NONE;

	for (list = keylist; list->key != KEY_NONE; list++)
		if (STREQ(cp, list->word))
			break;
	
	if (list->key == KEY_NONE) {
		syslog(LOG_ERR, "keyword %s unknown", cp);
		return KEY_ERROR;
	}

	if (list->numargs > 0) {
		*arg1 = gettoken(line_nextc);
		if (*arg1 == NULL) {
			syslog(LOG_ERR, "keyword %s requires an argument", cp);
			return KEY_ERROR;
		}
	} else {
		*arg1 = NULL;
	}

	if (list->numargs > 1) {
		*arg2 = gettoken(line_nextc);
		if (*arg2 == NULL) {
			syslog(LOG_ERR,
			    "keyword %s requires a second argument", cp);
			return KEY_ERROR;
		}
	} else {
		*arg2 = NULL;
	}

	return list->key;
}


/*
 * getservice - return a service specification
 */
int
getservice(str, name, proto, type, port)
	char *str;
	char **name;
	char **proto;
	int *type;
	u_short *port;
{
	register char *s;
	register struct servent *sp;
	u_long p;
	int atouint();

	/*
	 * Services look like telnet/tcp or 123/udp.  Find the
	 * '/' and split it there.
	 */
	for (s = str; *s != '/' && *s != '\0'; s++)
		/* nothing */;
	if (*s != '/')
		return 0;
	*s++ = '\0';

	/*
	 * The second part should be either tcp or udp.  Check
	 * this.
	 */
	if (STREQ(s, "tcp"))
		*type = T_TCP;
	else if (STREQ(s, "udp"))
		*type = T_UDP;
	else
		goto nogood;

	/*
	 * So far, so good.  Next is the port number, which is
	 * either an integer or a string to be given to
	 * getservbyname().  Try the former first, if no
	 * go try the latter.
	 */
	if (atouint(str, &p)) {
		if (p > 65535 || p == 0)
			goto nogood;
		*port = htons((u_short)p);
	} else {
		sp = getservbyname(str, s);
		if (sp == NULL)
			goto nogood;
		*port = (u_short)sp->s_port;
	}

	*name = str;
	*proto = s;
	return 1;

nogood:
	*(--s) = '/';
	return 0;
}



/*
 * find_internal - given a name, find an internal service
 */
struct internal *
find_internal(name)
	char *name;
{
	register struct internal *isp;

	for (isp = internal_servers; isp->service != NULL; isp++) {
		if (STREQ(name, isp->service))
			return isp;
	}
	return NULL;
}


/*
 * get_external - read the tokens from the input and store the
 *		  data in an external structure.
 */
struct external *
get_external(extern_pt, line_nextc)
	struct external *extern_pt;
	char **line_nextc;
{
	register char *cp;
	register struct external *exp;
	register int i;
	char *gettoken();

	cp = gettoken(line_nextc);
	if (cp == NULL) {
		syslog(LOG_ERR, "no arguments to `program' keyword");
		return NULL;
	}

	exp = extern_pt;
	exp->executable = cp;

	for (i = 0; i < MAXARGV; i++) {
		cp = gettoken(line_nextc);
		if (cp == NULL) {
			exp->argv[i] = NULL;
			break;
		}
		exp->argv[i] = cp;
	}

	if (i == 0) {
		exp->argv[0] = exp->executable;
		exp->argv[1] = NULL;
	} else if (i == MAXARGV) {
		exp->argv[MAXARGV] = NULL;
		if (gettoken(line_nextc) != NULL) {
			syslog(LOG_ERR,
			    "too many arguments for `program' keyword (> %d)",
			    MAXARGV);
			return NULL;
		}
	}
	return exp;
}


/*
 * copy_external - copy an external specification into malloc'd space
 */
struct external *
copy_external(extern_pt)
	struct external *extern_pt;
{
	register struct external *old_exp;
	register struct external *new_exp;
	register int i;
	char *emalloc();
	char *strsave();

	old_exp = extern_pt;
	new_exp = (struct external *)emalloc(sizeof(struct external));

	new_exp->executable = strsave(old_exp->executable);
	for (i = 0; i < MAXARGV && old_exp->argv[i] != NULL; i++)
		new_exp->argv[i] = strsave(old_exp->argv[i]);
	for (; i <= MAXARGV; i++)
		new_exp->argv[i] = NULL;
	return new_exp;
}


/*
 * free_external - free all memory associated with an external server
 */
void
free_external(exp)
	struct external *exp;
{
	register char **cpp;

	if (exp->executable != NULL)
		(void) free(exp->executable);

	cpp = exp->argv;
	while (*cpp != NULL) {
		(void) free(*cpp);
		cpp++;
	}

	(void) free(exp);
}


/*
 * getinterfaces - get the interface list for host
 */
void
getinterfaces()
{
	register struct ifreq *ifr;
	register int i;
	register int num;
	int vs;
	struct ifreq confbuf[MAXINTERFACES];
	struct interface ibuf[MAXINTERFACES];
	struct ifconf ifc;
	char *emalloc();

	if ((vs = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		syslog(LOG_ERR, "vs=socket(AF_INET, SOCK_DGRAM) %m");
		exit(1);
	}

	ifc.ifc_len = sizeof(confbuf);
	ifc.ifc_buf = (char *)confbuf;
	if (ioctl(vs, SIOCGIFCONF, (char *)&ifc) < 0) {
		syslog(LOG_ERR, "get interface configuration: %m");
		exit(1);
	}
	(void) close(vs);
	num = ifc.ifc_len/sizeof(struct ifreq);

	/*
	 * Fill in the default interface
	 */
	bzero((char *)ibuf, sizeof(ibuf));
	(void) strncpy(ibuf[0].name, "default", sizeof(ibuf[0].name));
	ibuf[0].address.s_addr = INADDR_ANY;
	ibuf[0].flags = INT_DEFAULT;
	i = 0;

	/*
	 * Go through the list looking for unique interface names.
	 * Record all of them even if there isn't (yet) an IP
	 * address associated with them.  We assume there will
	 * never be an interface named "default".  We also make
	 * illicit knowledge of the fact that the kernel groups
	 * different addresses for the same interface together.
	 */
	for (ifr = ifc.ifc_req; num > 0; num--, ifr++) {
		/*
		 * Record the name if we haven't seen
		 * this one before.
		 */
		if (!STRNEQ(ifr->ifr_name, ibuf[i].name, INTER_NAME_LEN)) {
			i++;
			strncpy(ibuf[i].name, ifr->ifr_name, INTER_NAME_LEN);
			ibuf[i].flags = INT_INCOMPLETE;
		}

		if (ifr->ifr_addr.sa_family == AF_INET) {
			if (!(ibuf[i].flags & INT_INCOMPLETE)) {
				/*
				 * Second IP address for same interface?
				 */
				i++;
				strncpy(ibuf[i].name, ifr->ifr_name,
				    INTER_NAME_LEN);
			}
			ibuf[i].address = ((struct sockaddr_in *)
			    &(ifr->ifr_addr))->sin_addr;
			ibuf[i].flags &= ~INT_INCOMPLETE;
		}
	}

	/*
	 * Got it.  Allocate permanent space for this and copy the
	 * list over.
	 */
	ninterfaces = i + 1;
	interlist = (struct interface *)emalloc(
	    ninterfaces * sizeof(struct interface));
	bcopy((char *)ibuf, (char *)interlist,
	    ninterfaces * sizeof(struct interface));
}


/*
 * Printable months
 */
char *months[] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};


/*
 * printstats - print request stats
 */
void
printstats(svp)
	struct server *svp;
{
	struct tm *tmp;
	char buf[128];

	if (svp->statcount > 0) {
		tmp = localtime(&(svp->laststattime.tv_sec));
		(void) sprintf(buf, "%d requests since %s %d %02d:%02d:%02d",
		    svp->statcount, months[tmp->tm_mon], tmp->tm_mday,
		    tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
		syslog(LOG_INFO, "%s/%s on %s: %s",
		    svp->name, ((svp->type == T_TCP) ? "tcp" : "udp"),
		    interlist[svp->interface_index].name, buf);
		svp->statcount = 0;
	}
}



/*
 * emalloc - malloc with error checking
 */
char *
emalloc(len)
	int len;
{
	char *cp;
	extern char *malloc();

	cp = malloc(len);
	if (cp == NULL) {
		syslog(LOG_ERR, "no memory! Aborting...");
		exit(1);
	}
	return cp;
}


/*
 * strsave - save a string in permanent storage
 */
char *
strsave(str)
	char *str;
{
	register char *cp;
	register int len;

	if (str == NULL)
		return NULL;
	len = strlen(str) + 1;
	cp = emalloc(len);
	bcopy(str, cp, len);
	return cp;
}


/*
 * atouint - return an unsigned long int, with appropriate checking
 */
int
atouint(str, uval)
	char *str;
	u_long *uval;
{
	register u_long u;
	register char *cp;

	cp = str;
	if (*cp == '\0')
		return 0;

	u = 0;
	while (*cp != '\0') {
		if (!isdigit(*cp))
			return 0;
		if (u > 429496729 || (u == 429496729 && *cp >= '6'))
			return 0;	/* overflow */
		u = (u << 3) + (u << 1);
		u += *cp++ - '0';	/* ascii dependent */
	}

	*uval = u;
	return 1;
}


/*
 * getnetnum - given a host name, return its net number
 */
int
getnetnum(host, num)
	char *host;
	u_long *num;
{
	struct hostent *hp;
	int decodenetnum();

	if (decodenetnum(host, num)) {
		return 1;
	} else if ((hp = gethostbyname(host)) != 0) {
		bcopy(hp->h_addr, (char *)num, sizeof(u_long));
		return 1;
	}
	return 0;
}

/*
 * decodenetnum - return a net number (this is crude, but careful)
 */
int
decodenetnum(num, netnum)
	char *num;
	u_long *netnum;
{
	register char *cp;
	register char *bp;
	register int i;
	register int temp;
	register int eos;
	char buf[80];		/* will core dump on really stupid stuff */

	cp = num;
	*netnum = 0;

	if (*cp == '[') {
		eos = ']';
		cp++;
	} else {
		eos = '\0';
	}

	for (i = 0; i < 4; i++) {
		bp = buf;
		while (isdigit(*cp))
			*bp++ = *cp++;
		if (bp == buf)
			break;

		if (i < 3) {
			if (*cp++ != '.')
				break;
		} else if (*cp != eos)
			break;

		*bp = '\0';
		temp = atoi(buf);
		if (temp > 255)
			break;
		*netnum <<= 8;
		*netnum += temp;
	}
	
	if (i < 4)
		return 0;
	*netnum = htonl(*netnum);
	return 1;
}


/*
 * hosttoa - turn a sockaddr_in into a printable host number
 */
char *
hosttoa(addr)
	struct sockaddr_in *addr;
{
	u_long netnum;
	static char numbuf[32];

	netnum = (u_long)ntohl(addr->sin_addr.s_addr);
	(void) sprintf(numbuf, "%d.%d.%d.%d", ((netnum >> 24) & 0xff),
	    ((netnum >> 16) & 0xff), ((netnum >> 8) & 0xff), netnum & 0xff);
	return numbuf;
}


/*
 * setproctitle - rewrite the argument list so that ps can see
 *		  what we are doing.
 */
void
setproctitle(svp, s)
	struct server *svp;
	int s;
{
	int size;
	register char *cp;
	struct sockaddr_in sin;
	char buf[80];

	cp = Argv[0];
	size = sizeof(sin);

	if (getpeername(s, &sin, &size) == 0)
		(void) sprintf(buf, "-%s/tcp [%s]", svp->name, hosttoa(&sin));
	else
		(void) sprintf(buf, "-%s/tcp", svp->name);
	strncpy(cp, buf, LastArg - cp);
	cp += strlen(cp);
	while (cp < LastArg)
		*cp++ = ' ';
}


/*
 * Simple services provided internally
 */

/*
 * tcp_echo - echo data stream back
 */
/* ARGSUSED */
tcp_echo(s, svp)
	int s;
	struct server *svp;
{
	char buffer[BUFFERSIZE];
	int i, n, done;

	setproctitle(svp, s);
	while ((i = read(s, buffer, sizeof(buffer))) > 0) {
		done = 0;
		do {
			n = write(s, &buffer[done], i - done);
			if (n <= 0)
				exit(0);
			done += n;
		} while (done < i);
	}
	exit(0);
}


/*
 * udp_echo - echo datagrams back
 */
/* ARGSUSED */
udp_echo(s, svp)
	int s;
	struct server *svp;
{
	char buffer[BUFFERSIZE];
	int i, size;
	struct sockaddr sa;

	size = sizeof(sa);
	i = recvfrom(s, buffer, sizeof(buffer), 0, &sa, &size);
	if (i < 0)
		return;
	(void) sendto(s, buffer, i, 0, &sa, sizeof(sa));
}


/*
 * tcp_discard - read and discard incoming data
 */
/* ARGSUSED */
tcp_discard(s, svp)
	int s;
	struct server *svp;
{
	char buffer[BUFFERSIZE];
	int i;

	setproctitle(svp, s);

	do {
		i = read(s, buffer, sizeof(buffer));
	} while (i > 0);

	exit(0);
}


/*
 * udp_discard - read and discard an incoming datagram
 */
/* ARGSUSED */
udp_discard(s, svp)
	int s;
	struct server *svp;
{
	char buffer[64];

	(void)recv(s, buffer, sizeof(buffer), 0);
}


/*
 * Unix time has a later epoch (Jan 1, 1970) than the TCP/UDP
 * time format (Jan 1, 1900).  This is the conversion.
 */

#define	JAN_1970	2208988800L	/* 1970 - 1900 in seconds */


/*
 * tcp_time - return the time in machine readable format
 */
/* ARGSUSED */
tcp_time(s, svp)
	int s;
	struct server *svp;
{
	struct timeval tv;
	u_long toc;

	(void) gettimeofday(&tv, (struct timezone *)0);

	toc = htonl((u_long)tv.tv_sec + JAN_1970);
	(void) write(s, (char *) &toc, sizeof(toc));
}


/*
 * udp_time - return the time in machine readable format
 */
/* ARGSUSED */
udp_time(s, svp)
	int s;
	struct server *svp;
{
	struct timeval tv;
	u_long toc;
	struct sockaddr sa;
	int size;


	size = sizeof(sa);
	if (recvfrom(s, (char *)&toc, sizeof(toc), 0, &sa, &size) < 0)
		return;

	(void) gettimeofday(&tv, (struct timezone *)0);
	toc = htonl((u_long)tv.tv_sec + JAN_1970);
	(void) sendto(s, (char *)&toc, sizeof(toc), 0, &sa, sizeof(sa));
}


/*
 * Month and day-of-week strings useful for forming a date
 */
char *day_month[] = {
	"January", "February", "March", "April", "May", "June", "July",
	"August", "September", "October", "November", "December"
};

char *day_day[] = {
	"Sunday", "Monday", "Tuesday", "Wednesday",
	"Thursday", "Friday", "Saturday"
};


/*
 * day_makedate - make a nice looking date
 */
int
day_makedate(tbuf)
	char *tbuf;
{
	struct tm *tmp;
	char *tzp;
	struct timeval tv;
	struct timezone tz;
	extern char *timezone();

	(void) gettimeofday(&tv, &tz);
	tmp = localtime(&tv.tv_sec);
	tzp = timezone(tz.tz_minuteswest, tmp->tm_isdst);

	(void) sprintf(tbuf, "%s, %s %d, %d %02d:%02d:%02d %s\r\n",
	    day_day[tmp->tm_wday], day_month[tmp->tm_mon], tmp->tm_mday,
	    tmp->tm_year + 1900, tmp->tm_hour, tmp->tm_min, tmp->tm_sec, tzp);
	return strlen(tbuf);
}


/*
 * tcp_daytime - return a human readable time of day
 */
/* ARGSUSED */
tcp_daytime(s, svp)
	int s;
	struct server *svp;
{
	char buffer[100];
	int i;

	i = day_makedate(buffer);
	(void) write(s, buffer, i);
}


/*
 * udp_daytime - return a human readable time of day
 */
/* ARGSUSED */
udp_daytime(s, svp)
	int s;
	struct server *svp;
{
	char buffer[100];
	struct sockaddr sa;
	int size;

	size = sizeof(sa);
	if (recvfrom(s, buffer, sizeof(buffer), 0, &sa, &size) < 0)
		return;
	size = day_makedate(buffer);
	(void) sendto(s, buffer, size, 0, &sa, sizeof(sa));
}


/*
 * Pattern data for chargen routines
 */
#define	NPCHARS		95
#define	NPERLINE	72
#define	LINELEN		(NPERLINE + 2)

char cg_pattern[168] = {
	' ', '!', '"', '#', '$', '%', '&', '\'', '(', ')', '*', '+',
	',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C',
	'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
	'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[',
	'\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g',
	'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's',
	't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~', ' ',
	'!', '"', '#', '$', '%', '&', '\'', '(', ')', '*', '+', ',',
	'-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8',
	'9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D',
	'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
	'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\',
	']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h'
};


/*
 * cg_fillbuf - fill a buffer with character data
 */
void
cg_fillbuf(buffer, len)
	char *buffer;
	int len;
{
	register char *bp;
	register int left;
	register int i;

	bp = buffer;
	left = len;
	for (i = 0; left >= LINELEN; left -= LINELEN, i = (i+1) % NPCHARS) {
		bcopy(&cg_pattern[i], bp, NPERLINE);
		bp += NPERLINE;
		*bp++ = '\r';
		*bp++ = '\n';
	}

	if (left > 0) {
		if (left == 1) {
			bp--;
		} else if (left > 2) {
			bcopy(&cg_pattern[i], bp, left-2);
			bp += left-2;
		}
		*bp++ = '\r';
		*bp++ = '\n';
	}
}


/*
 * tcp_chargen - print a pretty pattern
 */
/* ARGSUSED */
tcp_chargen(s, svp)
	int s;
	struct server *svp;
{
	char buffer[NPCHARS * LINELEN];

	setproctitle(svp, s);

	cg_fillbuf(buffer, sizeof(buffer));

	for (;;) {
		if (write(s, buffer, sizeof(buffer)/2) < sizeof(buffer)/2)
			break;
		if (write(s, &buffer[sizeof(buffer)/2], sizeof(buffer)/2)
		    < sizeof(buffer)/2)
			break;
	}

	exit(0);
}


/*
 * udp_chargen - send back a random sized packet full of stuff
 */
/* ARGSUSED */
udp_chargen(s, svp)
	int s;
	struct server *svp;
{
	char buffer[512];
	struct timeval tv;
	int len;
	struct sockaddr sa;
	int size;

	/*
	 * Get origin of packet
	 */
	size = sizeof(sa);
	if (recvfrom(s, buffer, sizeof(buffer), 0, &sa, &size) < 0)
		return;

	/*
	 * Use time to generate pseudo-random number
	 */
	(void) gettimeofday(&tv, (struct timezone *)0);
	len = (int)((tv.tv_sec ^ (tv.tv_usec >> 10)) % 509);
	if ((len % LINELEN) < 2)
		len += 3;
	cg_fillbuf(buffer, len);

	(void) sendto(s, buffer, len, 0, &sa, sizeof(sa));
}


/*
 * tcp_dropconn  - do absolutely nothing
 */
/* ARGSUSED */
tcp_dropconn(s, svp)
	int s;
	struct server *svp;
{
	/* nothing */
}


/*
 * printservlist - print the server list, for debugging purposes
 */
void
printservlist()
{
	register struct server *svp;
	register int i;
	char *hosttoa();

	if (servlist == 0) {
		(void) fprintf(stderr, "No servers in list\n");
		return;
	}

	for (svp = servlist; svp != 0; svp = svp->next) {
		(void) fprintf(stderr, "%s/%s %s.%d %s ",
		    svp->name, ((svp->type == T_TCP) ? "tcp" : "udp"),
		    hosttoa(&svp->address), ntohs(svp->address.sin_port),
		    interlist[svp->interface_index].name);
		(void) fprintf(stderr, "flags:");
		if (svp->flags & SERVE_DELETE)
			(void) fprintf(stderr, " delete");
		if (svp->flags & SERVE_LOG)
			(void) fprintf(stderr, " log");
		if (svp->flags & SERVE_STATS)
			(void) fprintf(stderr, " stats");
		if (svp->flags & SERVE_CONNECT)
			(void) fprintf(stderr, " connect");
		if (svp->flags & SERVE_HAVE_CHILD)
			(void) fprintf(stderr, " have_child");
		if (svp->flags & SERVE_DOWN)
			(void) fprintf(stderr, " down");
		(void) fprintf(stderr, "\n");

		if (svp->len_reslist > 0) {
			for (i = 0; i < svp->len_reslist; i++) {
				(void) fprintf(stderr, "   %sallow %d.%d.%d.%d",
				    (svp->reslist[i].allow ? "" : "dis"),
				    ((svp->reslist[i].address >> 24) & 0xff),
				    ((svp->reslist[i].address >> 16) & 0xff),
				    ((svp->reslist[i].address >>  8) & 0xff),
				    (svp->reslist[i].address & 0xff));
				(void) fprintf(stderr, " mask %d.%d.%d.%d\n",
				    ((svp->reslist[i].mask >> 24) & 0xff),
				    ((svp->reslist[i].mask >> 16) & 0xff),
				    ((svp->reslist[i].mask >>  8) & 0xff),
				    (svp->reslist[i].mask & 0xff));
			}
		}
	}
}
