/* 
 * Mach Operating System
 * Copyright (c) 1988 Carnegie-Mellon University
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/* 
 * HISTORY
 * $Log:	kern_exec.c,v $
 * Revision 2.12  88/12/19  02:33:31  mwyoung
 * 	ca, sun3: Removed lint.
 * 	[88/12/17            mwyoung]
 * 	
 * 	Removed old MACH conditionals.
 * 	[88/12/13            mwyoung]
 * 	
 * 	Use copyout on *all* architectures to zero the partial page at
 * 	the end of the data region, to prevent a pagein error from being
 * 	improperly attributed to the kernel.
 * 	[88/11/15            mwyoung]
 * 	
 * 	Apply inode locking changes to all architectures.
 * 	[88/11/03  19:19:47  mwyoung]
 * 	
 * 	Unlock the inode after calling inode_pager_setup(), so that
 * 	the subsequent vm_map call and the copyout of zeroes can
 * 	take faults.
 * 	[88/11/03            mwyoung]
 * 
 * Revision 2.11  88/10/18  03:14:11  mwyoung
 * 	Watch out for zero return value from kmem_alloc_wait.
 * 	[88/09/13            mwyoung]
 * 
 * Revision 2.10  88/10/18  00:27:36  mwyoung
 * 	Oct-10-88 Mary Thompson (mrt) @ Carnegie Mellon
 * 	Changed program_loader for romp to use USRTEXT for
 * 	the start of the text area.
 * 	[88/10/17  16:51:54  mrt]
 * 
 * Revision 2.9  88/10/11  12:06:17  rpd
 * 	Removed the reserving of segment 0 on the RT
 * 	when ROMP+SHARED_SEG option is enabled. Lisp
 * 	counts on using that space. (From sanzi.)
 * 	[88/10/11  12:00:15  rpd]
 * 
 * Revision 2.8  88/08/25  18:09:48  mwyoung
 * 	Corrected include file references.
 * 	[88/08/22            mwyoung]
 * 	
 * 	Fix up arguments to vm_protect.
 * 	[88/08/17  19:54:11  mwyoung]
 * 	
 * 	Convert all architectures to use vm_map.  Correctly release
 * 	inode_pager when errors occur.  Use only user-visible VM calls.
 * 	Reserve all of segment 0 on the RT/PC regardless whether
 * 	the ROMP_SHARED_SEG option is enabled.
 * 	[88/08/15  02:24:35  mwyoung]
 * 	
 * 	Changed Vax getxfile to use vm_map.  Other architectures will
 * 	follow shortly.
 * 	Add inode_pager_release() calls to relinquish port
 * 	rights/reference acquired by inode_pager_setup().
 * 	[88/08/11  18:42:13  mwyoung]
 * 
 * Revision 2.7  88/07/17  17:47:33  mwyoung
 * Use new memory object types.
 * 
 * Revision 2.6  88/07/15  15:28:16  mja
 * Flushed obsolete cs_security.h include.
 * 
 * 16-Jun-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Record stack start, end, and direction of growth.
 *
 * 18-Apr-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Handle zero-size data regions on Sun, Multimax, RT/PC.  [Vax and
 *	Balance both map in text and data together.]  It's unclear to me
 *	why there are two vm_allocate_with_pager's in those getxfile's;
 *	it's probably just for historical reasons.
 *
 *  1-Mar-88  David Black (dlb) at Carnegie-Mellon University
 *	Changed for updated sysV header files.  Use cpp symbols for
 *	multimax and balance.
 *
 * 19-Jan-88  Robert Baron (rvb) at Carnegie-Mellon University
 *	sun3 only: Force page 0 to be allocated and VM_PROTECT_NONE so
 *	noone can use it.  Also change vm_map_find's to vm_allocate.
 *
 * 30-Dec-87  David Golub (dbg) at Carnegie-Mellon University
 *	Delinted.
 *
 *  6-Dec-87  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Changed inode_pager_setup() calling sequence again.
 *	
 *	Removed ancient history.  Contributors so far: avie, dbg,
 *	mwyoung, bolosky, jjc, dlb, rvb, beck, jjk.
 *
 *  4-Nov-87  David Black (dlb) at Carnegie-Mellon University
 *	Changed multimax to copyout an array of zeroes instead of using
 *	user_zero.
 *
 * 28-Oct-87  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Added check for security of kernel ports before allowing
 *	setuid/gid changes.
 *
 * 24-Sep-87  David Black (dlb) at Carnegie-Mellon University
 *	Added unix_master(), unix_release() to load_init_program() so it
 *	stays on master throughout exec.
 *
 *  6-Apr-87  David Golub (dbg) at Carnegie-Mellon University
 *	MACH: In vax_getxfile, round up text size separately from data+bss
 *	size.  Some text files (410 format) don't round up sizes
 *	correctly in the image file, but expect exec to do it for them.
 *
 * 05-Mar-87  Jonathan J. Chew (jjc) at Carnegie-Mellon University
 *	Modified execve() to save a.out header in pcb so it can be
 *	dumped into a core file and used by the debuggers.
 *
 * 12-Jan-87  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Added "load_init_program()" routine to set up and exec /etc/init.
 *
 *  8-Jan-87  Robert Beck (beck) at Sequent Computer Systems, Inc.
 *	Changed most "#ifdef ns32000"'s to multimax -- COFF stuff.
 *	Mods for balance a.out's -- add balance_getxfile() case and
 *	ZMAGIC magic number recognition.
 *
 *  2-Dec-86  Jay Kistler (jjk) at Carnegie-Mellon University
 *	VICE:  added hooks for ITC/Andrew remote file system.
 *
 *  21-Oct-86  Jonathan J. Chew (jjc) and David Golub (dbg)
 *		at Carnegie-Mellon University
 *	Merged in changes for Sun:
 *		1) Created version of getxfile() for the Sun 3.
 *		2) Don't need signal trampoline code in execve() for Sun.
 *
 *  7-Oct-86  David L. Black (dlb) at Carnegie-Mellon University
 *	ns32000: Merged in Multimax changes; Multimax uses coff format.
 *
 * 11-Jun-86  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	romp: Added stupid exect() call for adb on the RT.
 *
 * 28-Mar-86  David Golub (dbg) at Carnegie-Mellon University
 *	Remember that the loader's page-size is still
 *	(CLSIZE*NBPG), and that text, data and bss end on the old
 *	page boundaries, not the new ones (or we'd have to relink all
 *	programs whenever we changed the page size!).
 *
 * 23-Nov-85  Mike Accetta (mja) at Carnegie-Mellon University
 *	CS_RFS:  enabled remote namei() processing for all
 *	routines in this module.
 *	[V1(1)]
 *
 * 21-May-85  Glenn Marcy (gm0w) at Carnegie-Mellon University
 *	Upgraded from 4.1BSD.  Carried over changes below.
 *
 *	CMUCS:  Added setting of execute only bit and clearing of
 *	trace bit in process status if image is not readable (V3.00).
 *	[V1(1)]
 *
 **********************************************************************
 */
 
#include <cmucs_rfs.h>
#include <cmucs.h>
#include <mach.h>
#include <vice.h>

/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)kern_exec.c	7.1 (Berkeley) 6/5/86
 */

#include <machine/reg.h>
#if	MACH
#else	MACH
#include <machine/pte.h>
#endif	MACH
#ifdef	romp
#include <ca/scr.h>
#endif	romp

#ifdef	vax
#include <vax/psl.h>
#endif	vax

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/map.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/kernel.h>
#include <sys/proc.h>
#include <sys/buf.h>
#include <sys/inode.h>
#if	MACH
#else	MACH
#include <sys/seg.h>
#endif	MACH
#include <sys/vm.h>
#if	MACH
#else	MACH
#include <sys/text.h>
#endif	MACH
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/acct.h>
#include <sys/exec.h>

#ifdef	multimax
/* 
 * If Multimax, handle COFF files.
 */
#include <sysV/scnhdr.h>
#include <sysV/aouthdr.h>
#include <sysV/filehdr.h>

#define SECTALIGN 1024
#define SECT_ROUNDUP(x) (((x) + (SECTALIGN - 1)) & ~(SECTALIGN - 1))
#endif	multimax
 
/*
 *  Force all namei() calls to permit remote names since this module has
 *  been updated.
 */
#if	CMUCS_RFS
#undef	namei
#define	namei	rnamei
#endif	CMUCS_RFS

#ifdef	romp
#include <ca/debug.h>
#endif	romp

#if	MACH
#else	MACH
#ifdef vax
#include <vax/mtpr.h>
#endif
#endif	MACH

#if	MACH
#include <sys/signal.h>
#include <sys/task.h>
#include <sys/thread.h>

#include <vm/vm_param.h>
#include <vm/vm_map.h>
#include <vm/vm_object.h>
#include <vm/memory_object.h>
#include <vm/inode_pager.h>
#include <vm/vm_kern.h>
#include <vm/vm_user.h>
#include <sys/zalloc.h>

#include <kern/parallel.h>


/*
 * All of these silly loader_page_size's should be moved into a machine
 * dependent directory, for obvious reasons.  -BB
 */
#ifdef	romp
#define	LOADER_PAGE_SIZE	(2048)
#endif	romp
#ifdef	vax
#define	LOADER_PAGE_SIZE	(1024)
#endif	vax
#ifdef	sun
#define	LOADER_PAGE_SIZE	(8192)
#endif	sun
#ifdef	multimax
#define	LOADER_PAGE_SIZE	(1024)
#endif	multimax
#ifdef	balance
#define	LOADER_PAGE_SIZE	(2048)
#endif	balance
#define loader_round_page(x)	((vm_offset_t)((((vm_offset_t)(x)) \
						+ LOADER_PAGE_SIZE - 1) \
					& ~(LOADER_PAGE_SIZE-1)))
#define loader_trunc_page(x)	((vm_offset_t)(((vm_offset_t)(x)) \
					& ~(LOADER_PAGE_SIZE-1)))

#endif	MACH

#if	MACH
/*
 *	A corrupted fileheader can cause getxfile to decide to bail
 *	out without setting up the address space correctly.  It is
 *	essential in this case that control never get back to the
 *	user.  The following error code is used by getxfile to tell
 *	execve that the process must be killed.
 */

#define	EGETXFILE	126

#endif	MACH

/*
 * exec system call, with and without environments.
 */
struct execa {
	char	*fname;
	char	**argp;
	char	**envp;
};

execv()
{
	((struct execa *)u.u_ap)->envp = NULL;
	execve();
}

#ifdef	romp
exect()	/* New RXTUnix system call for execve with single step active */
{
	execve();
	if( u.u_error );
	else u.u_ar0[ICSCS] |= ICSCS_INSTSTEP;
}
#endif	romp

execve()
{
	register nc;
	register char *cp;
#if	MACH
#else	MACH
	register struct buf *bp;
#endif	MACH
	register struct execa *uap;
	int na, ne, ucp, ap, cc;
	unsigned len;
	int indir, uid, gid;
	char *sharg;
	struct inode *ip;
#if	VICE
	struct file *fp = NULL;
#endif	VICE
#if	MACH
	vm_offset_t exec_args;
#else	MACH
	swblk_t bno;
#endif	MACH
	char cfname[MAXCOMLEN + 1];
#define	SHSIZE	32
	char cfarg[SHSIZE];


#ifdef	multimax
/* 
 * If Multimax code, handle COFF files.
 */
	struct aouthdr ahdr;
	int    aouthdr_offset;
	union {
		char	ex_shell[SHSIZE];  /* #! and name of interpreter */
		struct	filehdr	fhdr;
	} exdata;
#else	multimax
	union {
		char	ex_shell[SHSIZE];  /* #! and name of interpreter */
		struct	exec ex_exec;
	} exdata;
#endif	multimax


	register struct nameidata *ndp = &u.u_nd;
	int resid, error;

	ndp->ni_nameiop = LOOKUP | FOLLOW;
	ndp->ni_segflg = UIO_USERSPACE;
	ndp->ni_dirp = ((struct execa *)u.u_ap)->fname;
#if	VICE
	u.u_rmt_requested = 1;
#endif	VICE
	if ((ip = namei(ndp)) == NULL)
#if	VICE
		if (u.u_error == EVICEOP)
		{
			int	fd = u.u_r.r_val1;

			u.u_error = 0;
		 	GETF(fp, fd);
			ip = (struct inode *)fp->f_data;
			ip->i_count++;	/* Because fp points to the inode also */
			u.u_ofile[fd] = NULL;	/* free user fd */
		}
		else
#endif	VICE
		return;
#if	MACH
	exec_args = 0;
#else	MACH
	bno = 0;
	bp = 0;
#endif	MACH
	indir = 0;
	uid = u.u_uid;
	gid = u.u_gid;
#if	MACH
	/*
	 *	Can't let setuid things get abused by others through the
	 *	IPC interface
	 */

	if (ip->i_mode & (ISUID | ISGID)) {
	  if (task_secure(current_task())) {
#endif	MACH
	if (ip->i_mode & ISUID)
		uid = ip->i_uid;
	if (ip->i_mode & ISGID)
		gid = ip->i_gid;
#if	MACH
	  } else {
		uprintf("%s: privileges disabled because of outstanding IPC access to task\n",
			u.u_comm);
	  }
	}
#endif	MACH

  again:
	if (access(ip, IEXEC))
		goto bad;
	if ((u.u_procp->p_flag&STRC) && access(ip, IREAD))
		goto bad;
	if ((ip->i_mode & IFMT) != IFREG ||
	   (ip->i_mode & (IEXEC|(IEXEC>>3)|(IEXEC>>6))) == 0) {
		u.u_error = EACCES;
		goto bad;
	}

#ifdef	multimax
	/*
	 * Read in the COFF header.
	 *
	 *	510 = old RO text
	 *	515 = old demand paged RO text
	 *	524 = RO text
	 *	525 = demand paged RO text
	 */
	exdata.ex_shell[0] = '\0';	/* for zero length files */
	u.u_error = rdwri(UIO_READ, ip, (caddr_t)&exdata,
			sizeof(exdata), (off_t)0, 1, &resid);
	if (u.u_error)
		goto bad;
	if (resid && exdata.ex_shell[0] != '#') {
		u.u_error = ENOEXEC;
		goto bad;
	}

	/*
	 * If no COFF magic number then falling through
	 * to the default code handles shell scripts.
	 */
	switch(exdata.fhdr.f_magic) {
	    case N16WRMAGIC:
	    case N16ROMAGIC:
		aouthdr_offset = N16FILHSZ;
		break;
	    case NS32GMAGIC:
	    case NS32SMAGIC:
		aouthdr_offset = FILHSZ;
		break;

#else	multimax

	/*
	 * Read in first few bytes of file for segment sizes, magic number:
	 *	407 = plain executable
	 *	410 = RO text
	 *	413 = demand paged RO text
	 * Also an ASCII line beginning with #! is
	 * the file name of a ``shell'' and arguments may be prepended
	 * to the argument list if given here.
	 *
	 * SHELL NAMES ARE LIMITED IN LENGTH.
	 *
	 * ONLY ONE ARGUMENT MAY BE PASSED TO THE SHELL FROM
	 * THE ASCII LINE.
	 */
	exdata.ex_shell[0] = '\0';	/* for zero length files */
	u.u_error = rdwri(UIO_READ, ip, (caddr_t)&exdata, sizeof (exdata),
	    (off_t)0, 1, &resid);
	if (u.u_error)
		goto bad;
#ifndef lint
	if (resid > sizeof(exdata) - sizeof(exdata.ex_exec) &&
	    exdata.ex_shell[0] != '#') {
		u.u_error = ENOEXEC;
		goto bad;
	}
#endif lint
	switch ((int)exdata.ex_exec.a_magic) {

#ifdef	romp
	/* Complain and I'll fix it.  When I get time, that is. -BjB */
#else	romp
	case 0407:
		exdata.ex_exec.a_data += exdata.ex_exec.a_text;
		exdata.ex_exec.a_text = 0;
		break;
#endif	romp

#ifdef	balance
	case 0x10ea:				/* ZMAGIC: 0@0 */
						/* no XMAGIC yet */
		exdata.ex_exec.a_magic = 0413;	/* make other code easier */
		/* Fall through... */
#endif	balance
	case 0413:
	case 0410:
		if (exdata.ex_exec.a_text == 0) {
			u.u_error = ENOEXEC;
			goto bad;
		}
		break;

#endif	multimax

	default:
		if (exdata.ex_shell[0] != '#' ||
		    exdata.ex_shell[1] != '!' ||
		    indir) {
			u.u_error = ENOEXEC;
			goto bad;
		}
		cp = &exdata.ex_shell[2];		/* skip "#!" */
		while (cp < &exdata.ex_shell[SHSIZE]) {
			if (*cp == '\t')
				*cp = ' ';
			else if (*cp == '\n') {
				*cp = '\0';
				break;
			}
			cp++;
		}
		if (*cp != '\0') {
			u.u_error = ENOEXEC;
			goto bad;
		}
		cp = &exdata.ex_shell[2];
		while (*cp == ' ')
			cp++;
		ndp->ni_dirp = cp;
		while (*cp && *cp != ' ')
			cp++;
		cfarg[0] = '\0';
		if (*cp) {
			*cp++ = '\0';
			while (*cp == ' ')
				cp++;
			if (*cp)
				bcopy((caddr_t)cp, (caddr_t)cfarg, SHSIZE);
		}
		indir = 1;
		iput(ip);
#if	VICE
		if (fp) {
			closef(fp);
			fp = NULL;
		}
#endif	VICE
		ndp->ni_nameiop = LOOKUP | FOLLOW;
		ndp->ni_segflg = UIO_SYSSPACE;
#if	VICE
		u.u_rmt_requested = 1;
#endif	VICE
		ip = namei(ndp);
		if (ip == NULL)
#if	VICE
		if (u.u_error == EVICEOP)
		{
			int	fd = u.u_r.r_val1;

			u.u_error = 0;
		 	GETF(fp, fd);
			ip = (struct inode *)fp->f_data;
			ip->i_count++;	/* Because fp points to the inode also */
			u.u_ofile[fd] = NULL;	/* free user fd */
		}
		else
#endif	VICE
			return;
		bcopy((caddr_t)ndp->ni_dent.d_name, (caddr_t)cfname,
		    MAXCOMLEN);
		cfname[MAXCOMLEN] = '\0';
		goto again;
	}

#ifdef	multimax
	/*
	 * Read in the a.out header for segment sizes and magic number:
	 *	407 = plain executable
	 *	410 = RO text
	 *	413 = demand paged RO text
	 * (The Opus tools currently create separate text and data 
	 * sections even for 407 files.)
	 */
	u.u_error = rdwri(UIO_READ, ip, (caddr_t)&ahdr,
		sizeof(struct aouthdr), aouthdr_offset, 1, &resid);
	if (u.u_error)
		goto bad;

	switch (ahdr.magic) {
	    case 0407:
	    case 0410:
	    case 0413:
		if (ahdr.tsize == 0) {
			u.u_error = ENOEXEC;
			goto bad;
		}
		break;

	    default:
		u.u_error = ENOEXEC;
		goto bad;
	}
#endif	multimax

	/*
	 * Collect arguments on "file" in swap space.
	 */
	na = 0;
	ne = 0;
	nc = 0;
	cc = 0;
	uap = (struct execa *)u.u_ap;
#if	MACH
	iunlock(ip);
	exec_args = kmem_alloc_wait(kernel_pageable_map, NCARGS);
	if (exec_args == 0)
		panic("execve: cannot allocate arguments");
	cp = (char *) exec_args;	/* running pointer for copy */
	cc = NCARGS;			/* size of exec_args */
#else	MACH
	bno = rmalloc(argmap, (long)ctod(clrnd((int)btoc(NCARGS))));
	if (bno == 0) {
		swkill(u.u_procp, "exec: no swap space");
		goto bad;
	}
	if (bno % CLSIZE)
		panic("execa rmalloc");
#endif	MACH
	/*
	 * Copy arguments into file in argdev area.
	 */
	if (uap->argp) for (;;) {
		ap = NULL;
		sharg = NULL;
		if (indir && na == 0) {
			sharg = cfname;
			ap = (int)sharg;
			uap->argp++;		/* ignore argv[0] */
		} else if (indir && (na == 1 && cfarg[0])) {
			sharg = cfarg;
			ap = (int)sharg;
		} else if (indir && (na == 1 || na == 2 && cfarg[0]))
			ap = (int)uap->fname;
		else if (uap->argp) {
			ap = fuword((caddr_t)uap->argp);
			uap->argp++;
		}
		if (ap == NULL && uap->envp) {
			uap->argp = NULL;
			if ((ap = fuword((caddr_t)uap->envp)) != NULL)
				uap->envp++, ne++;
		}
		if (ap == NULL)
			break;
		na++;
		if (ap == -1) {
			u.u_error = EFAULT;
			break;
		}
		do {
#if	MACH
			if (nc >= NCARGS-1) {
				error = E2BIG;
				break;
			}
#else	MACH
			if (cc <= 0) {
				/*
				 * We depend on NCARGS being a multiple of
				 * CLSIZE*NBPG.  This way we need only check
				 * overflow before each buffer allocation.
				 */
				if (nc >= NCARGS-1) {
					error = E2BIG;
					break;
				}
				if (bp)
					bdwrite(bp);
				cc = CLSIZE*NBPG;
				bp = getblk(argdev, bno + ctod(nc/NBPG), cc);
				cp = bp->b_un.b_addr;
			}
#endif	MACH
			if (sharg) {
				error = copystr(sharg, cp, (unsigned)cc, &len);
				sharg += len;
			} else {
				error = copyinstr((caddr_t)ap, cp, (unsigned)cc,
				    &len);
				ap += len;
			}
			cp += len;
			nc += len;
			cc -= len;
		} while (error == ENOENT);
		if (error) {
			u.u_error = error;
#if	MACH
			ilock(ip);
#else	MACH
			if (bp)
				brelse(bp);
			bp = 0;
#endif	MACH
			goto badarg;
		}
	}
#if	MACH
	ilock(ip);
#else	MACH
	if (bp)
		bdwrite(bp);
	bp = 0;
#endif	MACH
	nc = (nc + NBPW-1) & ~(NBPW-1);
#ifdef	sun
	/*
	 *	Save a.out header for Sun debuggers
	 */
	current_thread()->pcb->pcb_exec = exdata.ex_exec;
#endif	sun
#ifdef	multimax
	mmax_getxfile(ip, &ahdr, nc + (na+4)*NBPW, uid, gid);
#else	multimax
	getxfile(ip, &exdata.ex_exec, nc + (na+4)*NBPW, uid, gid);
#endif	multimax
	if (u.u_error) {
badarg:
#if	MACH
	/*
	 *	NOTE: to prevent a race condition, getxfile had
	 *	to temporarily unlock the inode.  If new code needs to
	 *	be inserted here before the iput below, and it needs
	 *	to deal with the inode, keep this in mind.
	 */
#else	MACH
		for (cc = 0; cc < nc; cc += CLSIZE*NBPG) {
			bp = baddr(argdev, bno + ctod(cc/NBPG), CLSIZE*NBPG);
			if (bp) {
				bp->b_flags |= B_AGE;		/* throw away */
				bp->b_flags &= ~B_DELWRI;	/* cancel io */
				brelse(bp);
				bp = 0;
			}
		}
#endif	MACH
		goto bad;
	}
	iput(ip);
	ip = NULL;
#if	VICE
	if (u.u_textfile)
		closef(u.u_textfile);
	u.u_textfile = fp;
	fp = NULL;
#endif	VICE

	/*
	 * Copy back arglist.
	 */
	ucp = USRSTACK - nc - NBPW;
	ap = ucp - na*NBPW - 3*NBPW;
	u.u_ar0[SP] = ap;
	(void) suword((caddr_t)ap, na-ne);
	nc = 0;
	cc = 0;
#if	MACH
	cp = (char *) exec_args;
	cc = NCARGS;
#endif	MACH
	for (;;) {
		ap += NBPW;
		if (na == ne) {
			(void) suword((caddr_t)ap, 0);
			ap += NBPW;
		}
		if (--na < 0)
			break;
		(void) suword((caddr_t)ap, ucp);
		do {
#if	MACH
#else	MACH
			if (cc <= 0) {
				if (bp)
					brelse(bp);
				cc = CLSIZE*NBPG;
				bp = bread(argdev, bno + ctod(nc / NBPG), cc);
				bp->b_flags |= B_AGE;		/* throw away */
				bp->b_flags &= ~B_DELWRI;	/* cancel io */
				cp = bp->b_un.b_addr;
			}
#endif	MACH
			error = copyoutstr(cp, (caddr_t)ucp, (unsigned)cc,
			    &len);
			ucp += len;
			cp += len;
			nc += len;
			cc -= len;
		} while (error == ENOENT);
		if (error == EFAULT)
			panic("exec: EFAULT");
	}
	(void) suword((caddr_t)ap, 0);

	/*
	 * Reset caught signals.  Held signals
	 * remain held through p_sigmask.
	 */
	while (u.u_procp->p_sigcatch) {
		nc = ffs((long)u.u_procp->p_sigcatch);
		u.u_procp->p_sigcatch &= ~sigmask(nc);
		u.u_signal[nc] = SIG_DFL;
	}
	/*
	 * Reset stack state to the user stack.
	 * Clear set of signals caught on the signal stack.
	 */
	u.u_onstack = 0;
	u.u_sigsp = 0;
	u.u_sigonstack = 0;

	for (nc = u.u_lastfile; nc >= 0; --nc) {
		if (u.u_pofile[nc] & UF_EXCLOSE) {
			closef(u.u_ofile[nc]);
			u.u_ofile[nc] = NULL;
			u.u_pofile[nc] = 0;
		}
		u.u_pofile[nc] &= ~UF_MAPPED;
	}
	while (u.u_lastfile >= 0 && u.u_ofile[u.u_lastfile] == NULL)
		u.u_lastfile--;
#ifdef	multimax
	setregs(ahdr.entry, ahdr.mod_start);
#else	multimax
	setregs(exdata.ex_exec.a_entry);
#endif	multimax
#if	MACH
#ifdef	vax
	{
		/*
		 *	This belongs in vax.setregs()
		 */
		extern int nsigcode[5];

		bcopy((caddr_t)nsigcode,
		      (caddr_t)(VM_MAX_ADDRESS - sizeof(nsigcode)),
		      sizeof(nsigcode));
	}
#endif	vax
#ifdef	romp
	{
	    	/*
		 *	sigcode[] must agree with declaration in pcb.h
		 *
		 *	sigcode goes at the bottom of the user_stack,
		 *	where, of course, the user's stack can grow
		 *	down on top of it, but this seems unlikely.
		 *	Putting it at the top makes ps(1) unhappy.
		 */
		extern int sigcode[3];
		bcopy((caddr_t)sigcode,
		      (caddr_t)SIGCODE_ADDRESS,
		      sizeof(sigcode));
	}
#endif	romp
#endif	MACH

	/*
	 * Remember file name for accounting.
	 */
	u.u_acflag &= ~AFORK;
	if (indir)
		bcopy((caddr_t)cfname, (caddr_t)u.u_comm, MAXCOMLEN);
	else {
		if (ndp->ni_dent.d_namlen > MAXCOMLEN)
			ndp->ni_dent.d_namlen = MAXCOMLEN;
		bcopy((caddr_t)ndp->ni_dent.d_name, (caddr_t)u.u_comm,
		    (unsigned)(ndp->ni_dent.d_namlen + 1));
	}
bad:
#if	MACH
	if (exec_args != 0)
		kmem_free_wakeup(kernel_pageable_map, exec_args, NCARGS);
#else	MACH
	if (bp)
		brelse(bp);
	if (bno)
		rmfree(argmap, (long)ctod(clrnd((int) btoc(NCARGS))), bno);
#endif	MACH
	if (ip)
		iput(ip);
#if	VICE
	if (fp)
		closef(fp);
#endif	VICE
#if	MACH
	if (u.u_error == EGETXFILE) {
		struct proc	*p;

		/* 
		 *	getxfile failed, kill the current process.
		 *	Send SIGKILL, blow away other pending signals.
		 */
		p = u.u_procp;
		p->p_sig = sigmask(SIGKILL);
		p->p_cursig = SIGKILL;
		u.u_sig = 0;
		u.u_cursig = 0;
		psig();		/* Bye */
	}
#endif	MACH
}

#ifdef	multimax
#else	multimax
/*
 * Read in and set up memory for executed file.
 */
getxfile(ip, ep, nargc, uid, gid)
	struct inode *ip;
	struct exec *ep;
	int nargc, uid, gid;
{
#if	MACH
	size_t ts, ds, ss;
	int pagi;
	vm_size_t	text_size, data_size;
#else	MACH
	size_t ts, ds, ids, uds, ss;
	int pagi;
#endif	MACH

	if (ep->a_magic == 0413)
		pagi = SPAGI;
	else
		pagi = 0;

#if	MACH
	/*
	 *	The vm system handles text that is modified
	 *	for tracing - we don't have to worry about it.
	 */
#else	MACH
	if (ip->i_text && (ip->i_text->x_flag & XTRC)) {
		u.u_error = ETXTBSY;
		goto bad;
	}
#endif	MACH

	if (ep->a_magic != 0407 && (ip->i_flag&ITEXT) == 0 &&
	    ip->i_count != 1) {
		register struct file *fp;

		for (fp = file; fp < fileNFILE; fp++) {
#if	VICE
			if ((fp->f_type == DTYPE_INODE ||
			     fp->f_type == DTYPE_VICEINO) &&
#else	VICE
			if (fp->f_type == DTYPE_INODE &&
#endif	VICE
			    fp->f_count > 0 &&
			    (struct inode *)fp->f_data == ip &&
			    (fp->f_flag&FWRITE)) {
				u.u_error = ETXTBSY;
				goto bad;
			}
		}
	}

	/*
	 * Compute text and data sizes and make sure not too large.
	 * NB - Check data and bss separately as they may overflow 
	 * when summed together.
	 */
#if	MACH
	text_size = loader_round_page(ep->a_text);	/* bytes */
	ts = btoc(text_size);				/* machine pages */
	data_size = loader_round_page(ep->a_data + ep->a_bss);
							/* bytes */
	ds = btoc(data_size);				/* machine pages */
	ss = SSIZE + btoc(loader_round_page(nargc));
#else	MACH
	ts = clrnd(btoc(ep->a_text));
	ids = clrnd(btoc(ep->a_data));
	uds = clrnd(btoc(ep->a_bss));
	ds = clrnd(btoc(ep->a_data + ep->a_bss));
	ss = clrnd(SSIZE + btoc(nargc));

	if (chksize((unsigned)ts, (unsigned)ids, (unsigned)uds, (unsigned)ss))
		goto bad;

	/*
	 * Make sure enough space to start process.
	 */
	u.u_cdmap = zdmap;
	u.u_csmap = zdmap;
	if (swpexpand(ds, ss, &u.u_cdmap, &u.u_csmap) == NULL)
		goto bad;

	/*
	 * At this point, committed to the new image!
	 * Release virtual memory resources of old process, and
	 * initialize the virtual memory of the new process.
	 * If we resulted from vfork(), instead wakeup our
	 * parent who will set SVFDONE when he has taken back
	 * our resources.
	 */
	if ((u.u_procp->p_flag & SVFORK) == 0)
		vrelvm();
	else {
		u.u_procp->p_flag &= ~SVFORK;
		u.u_procp->p_flag |= SKEEP;
		wakeup((caddr_t)u.u_procp);
		while ((u.u_procp->p_flag & SVFDONE) == 0)
			sleep((caddr_t)u.u_procp, PZERO - 1);
		u.u_procp->p_flag &= ~(SVFDONE|SKEEP);
	}
#endif	MACH
#if	CMUCS
	u.u_procp->p_flag &= ~(SPAGI|SSEQL|SUANOM|SOUSIG|SXONLY);
	if (access(ip, IREAD))
	{
		u.u_procp->p_flag |= SXONLY;
		u.u_procp->p_flag &= ~STRC;
		u.u_error = 0;
	}
#else	CMUCS
	u.u_procp->p_flag &= ~(SPAGI|SSEQL|SUANOM|SOUSIG);
#endif	CMUCS
	u.u_procp->p_flag |= pagi;
#if	MACH

#define	unix_stack_size	(u.u_rlimit[RLIMIT_STACK].rlim_cur)

#ifndef	multimax
	program_loader(ip, ep, pagi);
#endif	multimax
#else	MACH
	u.u_dmap = u.u_cdmap;
	u.u_smap = u.u_csmap;
	vgetvm(ts, ds, ss);

	if (pagi == 0)
		u.u_error =
		    rdwri(UIO_READ, ip,
			(char *)ctob(dptov(u.u_procp, 0)),
			(int)ep->a_data,
			(off_t)(sizeof (struct exec) + ep->a_text),
			(off_t)0, (int *)0);
	xalloc(ip, ep, pagi);
	if (pagi && u.u_procp->p_textp)
		vinifod((struct fpte *)dptopte(u.u_procp, 0),
		    PG_FTEXT, u.u_procp->p_textp->x_iptr,
		    (long)(1 + ts/CLSIZE), (size_t)btoc(ep->a_data));

#ifdef vax
	/* THIS SHOULD BE DONE AT A LOWER LEVEL, IF AT ALL */
	mtpr(TBIA, 0);
#endif

	if (u.u_error)
		swkill(u.u_procp, "exec: I/O error mapping pages");
#endif	MACH
	/*
	 * set SUID/SGID protections, if no tracing
	 */
	if ((u.u_procp->p_flag&STRC)==0) {
		u.u_uid = uid;
		u.u_procp->p_uid = uid;
		u.u_gid = gid;
	} else
		psignal(u.u_procp, SIGTRAP);
	u.u_tsize = ts;
	u.u_dsize = ds;
	u.u_ssize = ss;
	u.u_prof.pr_scale = 0;
bad:
	return;
}
#endif	multimax

#if	MACH
boolean_t	getxfile_use_map = FALSE;

#ifdef	multimax

mmax_getxfile(ip, ap, nargc, uid, gid)
	struct inode *ip;
	struct aouthdr *ap;
	int nargc, uid, gid;
{
	size_t ts, ds, ss;
	int pagi;
	vm_offset_t	addr;
	vm_size_t	size;
	vm_map_t	my_map;
	vm_offset_t	data_end;
	vm_offset_t	vm_text_start, vm_text_end;
	vm_offset_t	vm_data_start, vm_data_end;

	if (ap->magic == 0413)
		pagi = SPAGI;
	else
		pagi = 0;

	/*
	 *	The vm system handles text that is modified
	 *	for tracing - we don't have to worry about it.
	 */

	if (ap->magic != 0407 && (ip->i_flag&ITEXT) == 0 &&
	    ip->i_count != 1) {
		register struct file *fp;

		for (fp = file; fp < fileNFILE; fp++) {
			if (fp->f_type == DTYPE_INODE &&
			    fp->f_count > 0 &&
			    (struct inode *)fp->f_data == ip &&
			    (fp->f_flag&FWRITE)) {
				u.u_error = ETXTBSY;
				goto bad;
			}
		}
	}

	/*
	 * Compute text, data and stack sizes.
	 */

	ts = btoc(loader_round_page(ap->tsize));
	ds = btoc(loader_round_page(ap->bsize + ap->dsize));
	ss = SSIZE + btoc(loader_round_page(nargc));

#if	CMUCS
	u.u_procp->p_flag &= ~(SPAGI|SSEQL|SUANOM|SOUSIG|SXONLY);
#else	CMUCS
	u.u_procp->p_flag &= ~(SPAGI|SSEQL|SUANOM|SOUSIG);
#endif	CMUCS
#if	CMUCS
	if (access(ip, IREAD))
	{
		u.u_procp->p_flag |= SXONLY;
		u.u_procp->p_flag &= ~STRC;
		u.u_error = 0;
	}
#endif	CMUCS
	u.u_procp->p_flag |= pagi;

#define	unix_stack_size	(u.u_rlimit[RLIMIT_STACK].rlim_cur)

	my_map = current_task()->map;
	/*
	 *	Even if we are exec'ing the same image (the RFS server
	 *	does this, for example), we don't have to unlock the
	 *	inode; deallocating it doesn't require it to be locked.
	 */
	(void) vm_deallocate(my_map, vm_map_min(my_map),
		vm_map_max(my_map)-vm_map_min(my_map));
	
	/*
	 *	Allocate low-memory stuff: text, data, bss.
	 *	Read text and data into lowest part, then make text read-only.
	 */

	addr = VM_MIN_ADDRESS;
	size = round_page(ap->data_start + ap->dsize + ap->bsize);

	/*
	 *	Remember where text and data start.
	 */
	u.u_text_start = (caddr_t) ap->text_start;
	u.u_data_start = (caddr_t) ap->data_start;

	/*
	 *	Note vm boundaries for data and text segments.  If data
	 *	and text overlap a page, that is considered data.
	 */
	vm_text_start = trunc_page(ap->text_start);
	vm_text_end = round_page(ap->text_start + ap->tsize);
	vm_data_start = trunc_page(ap->data_start);
	vm_data_end = round_page(ap->data_start + ap->dsize);
	if (vm_text_end > vm_data_start)
		vm_text_end = vm_data_start;

	u.u_error = 0;

	if (pagi == 0) {	/* not demand paged */
		if (vm_allocate(my_map, &addr, size, FALSE) != KERN_SUCCESS) {
			uprintf("Cannot find space for exec image.\n");
			goto suicide;
		}

		/*
		 *	Read in the data segment (0407 & 0410).  It goes on
		 *	the next loader_page boundary after the text.
		 */
		u.u_error = rdwri(UIO_READ, ip,
				(caddr_t) ap->data_start,
				(int) ap->dsize,
				(off_t) SECT_ROUNDUP(sizeof(struct filehdr) +
						ap->tsize),
				0, (int *)0);
		/*
		 *	Read in text segment if necessary (0410), 
		 *	and read-protect it.
		 */
		if ((u.u_error == 0) && (ap->tsize > 0)) {
			u.u_error = rdwri(UIO_READ, ip,
				(caddr_t) ap->text_start,
				(int) ap->tsize,
				(off_t) SECT_ROUNDUP(sizeof(struct filehdr)),
				2, (int *) 0);
			if (u.u_error == 0) {
				(void) vm_protect(my_map,
					vm_text_start,
					vm_text_end - vm_text_start,
					FALSE,
					VM_PROT_READ|VM_PROT_EXECUTE);
			}
		}
	}
	else {
		memory_object_t	pager;

		/*
		 *	Allocate a region backed by the exec'ed inode.
		 */

		pager = (memory_object_t)inode_pager_setup(ip, TRUE, TRUE);
		iunlock(ip);

		if (vm_map(my_map, &vm_text_start, vm_text_end - vm_text_start,
			0, FALSE,
			pager, (vm_offset_t)SECTALIGN, TRUE,
			VM_PROT_READ|VM_PROT_EXECUTE, VM_PROT_ALL,
			VM_INHERIT_COPY)
		     != KERN_SUCCESS) {
			uprintf("Cannot map text into user space.\n");
			inode_pager_release(pager);
			ilock(ip);
			goto suicide;
		}

		if (vm_data_end > vm_data_start) {
			if (vm_map(my_map, &vm_data_start,
			    vm_data_end - vm_data_start, 0, FALSE, pager,
			    (vm_offset_t) SECTALIGN + SECT_ROUNDUP(ap->tsize),
			    TRUE, VM_PROT_ALL, VM_PROT_ALL, VM_INHERIT_COPY)
			      != KERN_SUCCESS) {
				uprintf("Cannot map data into user space.\n");
				inode_pager_release(pager);
				ilock(ip);
				goto suicide;
			}
		}

		inode_pager_release(pager);
		ilock(ip);

		/* BSS goes on the end of data */

		addr = data_end;
		size =
			(ap->dsize + ap->bsize) - 	/* Size of data +bss */
			(vm_data_end - vm_data_start);	/* What data has now */

		if (vm_allocate(my_map, &addr, size, FALSE) != KERN_SUCCESS) {
			uprintf("Cannot find space for bss.\n");
			goto suicide;
		}

		/*
		 *	If the data segment does not end on a VM page
		 *	boundary, we have to clear the remainder of the VM
		 *	page it ends on so that the bss segment will
		 *	(correctly) be zero.
		 *	The loader has already guaranteed that the (text+data)
		 *	segment ends on a loader_page boundary.
		 */

		data_end = ap->data_start + loader_round_page(ap->dsize);
		if (vm_data_end > data_end) {
			if (vm_data_end - data_end > PAGE_SIZE) {
			    	uprintf("Can't clear front of bss.\n");
				goto suicide;
			}
			iunlock(ip);
			if (copyout(vm_kern_zero_page, (caddr_t)data_end,
						copy_end - data_end)) {
				ilock(ip);
				uprintf("Cannot zero partial data page\n");
				goto suicide;
			}
			ilock(ip);
		}
	}

	/*
	 *	Create the stack.  (Deallocate the old one and create a 
	 *	new one).
	 */

	size = round_page(unix_stack_size);
	u.u_stack_start = (caddr_t) (addr = trunc_page(VM_MAX_ADDRESS - size));
	u.u_stack_end = u.u_stack_start + size;
	u.u_stack_grows_up = FALSE;
	(void) vm_deallocate(my_map, addr, size);
	if (vm_allocate(my_map, &addr, size, FALSE) != KERN_SUCCESS) {
		uprintf("Cannot find space for stack.\n");
		goto suicide;
	}

	/*
	 * set SUID/SGID protections, if no tracing
	 */
	if ((u.u_procp->p_flag&STRC)==0) {
		u.u_uid = uid;
		u.u_procp->p_uid = uid;
		u.u_gid = gid;
	} else
		psignal(u.u_procp, SIGTRAP);
	u.u_tsize = ts;
	u.u_dsize = ds;
	u.u_ssize = ss;
	u.u_prof.pr_scale = 0;
bad:
	return;
suicide:
	u.u_error = EGETXFILE;
	return;
}
#endif	multimax

#ifdef	vax
program_loader(ip, ep, pagi)
	struct inode	*ip;
	struct exec	*ep;
	int		pagi;
{
	vm_offset_t	addr;
	vm_size_t	size;
	vm_map_t	my_map;
	vm_size_t	copy_size;
	vm_offset_t	copy_end, data_end;
	vm_size_t	text_size, data_size;

	my_map = current_task()->map;

	/*
	 *	Even if we are exec'ing the same image (the RFS server
	 *	does this, for example), we don't have to unlock the
	 *	inode; deallocating it doesn't require it to be locked.
	 */
	(void) vm_deallocate(my_map, vm_map_min(my_map),
			vm_map_max(my_map) - vm_map_min(my_map));
	
	/*
	 *	Allocate low-memory stuff: text, data, bss, space for brk
	 *	calls.
	 *	Read text&data into lowest part, then make text read-only.
	 */

	addr = VM_MIN_ADDRESS;
	text_size = loader_round_page(ep->a_text);
	data_size = loader_round_page(ep->a_data + ep->a_bss);
	size = round_page(text_size + data_size);

	/*
	 *	Remember text and data starting points
	 */
	u.u_text_start = USRTEXT;
	u.u_data_start = (caddr_t) text_size;

	u.u_error = 0;

	if (pagi == 0) {	/* not demand paged */
		if (vm_allocate(my_map, &addr, size, FALSE) != KERN_SUCCESS) {
			uprintf("Cannot find space for exec image.\n");
			goto suicide;
		}

		/*
		 *	Read in the data segment (0407 & 0410).  It goes on
		 *	the next loader_page boundary after the text.
		 */
		u.u_error = rdwri(UIO_READ, ip,
				u.u_data_start,
				(int)ep->a_data,
				(off_t)(sizeof(struct exec)+ep->a_text),
				0, (int *)0);
		/*
		 *	Read in text segment if necessary (0410), 
		 *	and read-protect it.
		 */
		if ((u.u_error == 0) && (ep->a_text > 0)) {
			u.u_error = rdwri(UIO_READ, ip,
				(caddr_t) VM_MIN_ADDRESS,
				(int)ep->a_text,
				(off_t)sizeof(struct exec), 2, (int *) 0);
			if (u.u_error == 0) {
				(void) vm_protect(my_map,
					 VM_MIN_ADDRESS,
					 trunc_page(ep->a_text),
					 FALSE,
					 VM_PROT_READ|VM_PROT_EXECUTE);
			}
		}
	}
	else {
		memory_object_t	pager;

		/*
		 *	Allocate a region backed by the exec'ed inode.
		 */

		copy_size = round_page(ep->a_text + ep->a_data);

		pager = (memory_object_t)inode_pager_setup(ip, TRUE, TRUE);
		iunlock(ip);

		addr = VM_MIN_ADDRESS;

		/* TEXT and DATA */
		if (vm_map(my_map, &addr, copy_size, 0, FALSE,
			pager, (vm_offset_t) 1024, TRUE,
			VM_PROT_ALL, VM_PROT_ALL, VM_INHERIT_COPY)
		    != KERN_SUCCESS) {
			uprintf("Cannot map text into user address space\n");
			inode_pager_release(pager);
			ilock(ip);
			goto suicide;
		}
		inode_pager_release(pager);
		ilock(ip);

		/* BSS */
		addr += copy_size;
		if ((size != copy_size) &&
		    vm_allocate(my_map, &addr, size - copy_size, FALSE)
		     != KERN_SUCCESS) {
		    	uprintf("Cannot allocate BSS in user address space\n");
			goto suicide;
		}

		/*
		 *	Read-protect just the text region.  Do this before 
		 *	we zero the bss area, so that we have only one copy
		 *	of the text.
		 */

		(void) vm_protect(my_map,
			 VM_MIN_ADDRESS,
			 trunc_page(ep->a_text),
			 FALSE,
			 VM_PROT_READ|VM_PROT_EXECUTE);

		/*
		 *	If the data segment does not end on a VM page
		 *	boundary, we have to clear the remainder of the VM
		 *	page it ends on so that the bss segment will
		 *	(correctly) be zero.
		 *	The loader has already guaranteed that the (text+data)
		 *	segment ends on a loader_page boundary.
		 */

		data_end = VM_MIN_ADDRESS + 
				loader_round_page(ep->a_text + ep->a_data);
		copy_end = VM_MIN_ADDRESS + copy_size;
		if (copy_end > data_end) {
			iunlock(ip);
			if (copyout(vm_kern_zero_page, (caddr_t)data_end,
						copy_end - data_end)) {
				ilock(ip);
				uprintf("Cannot zero partial data page\n");
				goto suicide;
			}
			ilock(ip);
		}
	}

	/*
	 *	Create the stack.  (Deallocate the old one and create a 
	 *	new one).
	 */

	size = round_page(unix_stack_size);

	u.u_stack_start = (caddr_t) (addr = trunc_page(VM_MAX_ADDRESS - size));
	u.u_stack_end = u.u_stack_start + size;
	u.u_stack_grows_up = FALSE;

	(void) vm_deallocate(my_map, addr, size);
	if (vm_allocate(my_map, &addr, size, FALSE) != KERN_SUCCESS) {
		uprintf("Cannot find space for stack.\n");
		goto suicide;
	}
	return;

suicide:
	u.u_error = EGETXFILE;
	return;
}
#endif	vax
#ifdef	romp
program_loader(ip, ep, pagi)
	struct inode	*ip;
	struct exec	*ep;
	int		pagi;
{
	vm_map_t	my_map;
	memory_object_t	pager;
	vm_offset_t	addr;
	long		data_end, copy_end;
	long		text_size, data_size;
	vm_size_t	bss_size;
	vm_size_t	stack_size;

#ifdef	lint
	pagi++;
#endif	lint

	my_map = current_task()->map;

	/*
	 *	This is how unix processes are mapped onto romp/rosetta:
	 *
	 *	0 - 0x10000000				text
	 *	0x10000000 - 0x10800000			data/bss
	 *	USRSTACK - unix_stack_size - USRSTACK	stack
	 */

#define DATA_START	0x10000000

	/*
	 *	remember where text and data start
	 */
	u.u_text_start = USRTEXT;
	u.u_data_start = (caddr_t) DATA_START;

	/*
	 *	Deallocate the previous text, data, bss & stack segments.
	 */

	(void) vm_deallocate(my_map, vm_map_min(my_map),
			VM_MAX_ADDRESS - vm_map_min(my_map));

	u.u_error = 0;

	/*
	 *	Map the text into the temp map.
	 */

	pager = (memory_object_t)inode_pager_setup(ip, TRUE, TRUE);
	iunlock(ip);
	addr = VM_MIN_ADDRESS;
	text_size = round_page(ep->a_text);
	if (vm_map(my_map, &addr, text_size, 0, FALSE,
			pager, (vm_offset_t) 2048, TRUE,
			VM_PROT_READ|VM_PROT_EXECUTE, VM_PROT_ALL,
			VM_INHERIT_COPY) != KERN_SUCCESS) {
		uprintf("Cannot map text into user space.\n");
		ilock(ip);
		inode_pager_release(pager);
		goto suicide;
	}
#ifdef	not_on_your_life
	/*
	 *	Reserve space from the top of text to the top of the 
	 *	rosetta 0 segment.
	 */

	addr = round_page(ep->a_text);
	if (vm_map(my_map, &addr,(u_int)DATA_START - (u_int)addr, 0, FALSE,
		   MEMORY_OBJECT_NULL, (vm_offset_t) 0, FALSE,
		   VM_PROT_NONE, VM_PROT_NONE, VM_INHERIT_SHARE)
			!= KERN_SUCCESS) {
		uprintf("Cannot allocate hole in text segment!\n");
	}
#endif	not_on_your_life	

	/*
	 *	Now copy the data segment.
	 */
	addr = DATA_START;
	data_size = round_page(ep->a_data);
	if (data_size != 0) {
		if (vm_map(my_map, &addr, data_size, 0, FALSE, pager,
			(vm_offset_t) loader_round_page(ep->a_text) + 2048,
			TRUE, VM_PROT_ALL, VM_PROT_ALL, VM_INHERIT_COPY)
		     != KERN_SUCCESS) {
			uprintf("Cannot map data into user space.\n");
			ilock(ip);
			inode_pager_release(pager);
			goto suicide;
		}
		addr += data_size;
	}

	ilock(ip);
	inode_pager_release(pager);

	/*
	 *	Lastly, allocate the BSS region
	 */

	bss_size = round_page(ep->a_data + ep->a_bss) - data_size;
	if (bss_size != 0) {
		if (vm_allocate(my_map,	&addr, bss_size, FALSE) != KERN_SUCCESS) {
			uprintf("Cannot find space for data/bss.\n");
			goto suicide;
		}
	}

	/*
	 *	If the data segment does not end on a VM page boundary,
	 *	we have to clear the remainder of the VM page it ends on
	 *	so that the bss segment will (correctly) be zero.
	 *	The loader has already guaranteed that the (text+data)
	 *	segment ends on a loader_page boundary.
	 */

	data_end = DATA_START + loader_round_page(ep->a_data);
	copy_end = DATA_START + round_page(ep->a_data);
	if (copy_end > data_end) {
		iunlock(ip);
		if (copyout(vm_kern_zero_page, (caddr_t)data_end, copy_end - data_end)) {
			ilock(ip);
			uprintf("Cannot zero partial data page\n");
			goto suicide;
		}
		ilock(ip);
	}

	/*
	 *	Create the stack.  (Deallocate the old one and create a new one).
	 */

	stack_size = round_page(unix_stack_size);
	u.u_stack_start = (caddr_t) (addr = trunc_page(VM_MAX_ADDRESS - stack_size));
	u.u_stack_end = u.u_stack_start + stack_size;
	u.u_stack_grows_up = FALSE;
	(void) vm_deallocate(my_map, addr, stack_size);
	if (vm_allocate(my_map, &addr, stack_size, FALSE) != KERN_SUCCESS) {
		uprintf("Cannot find space for stack.\n");
		goto suicide;
	}

	 /*
	  *	Allocate space for the signal trampoline code.
	  */
	{
		vm_size_t	size;

		extern int sigcode[3];
		size = round_page(sizeof(sigcode));
		addr = trunc_page((vm_offset_t)VM_MAX_ADDRESS - size);
		(void) vm_deallocate(my_map, addr, size);
		if (vm_allocate(my_map, &addr, size, FALSE) != KERN_SUCCESS) {
		 	uprintf("Cannot allocate space for sigcode.\n");
			 goto suicide;
		}
	}

	return;
suicide:
	u.u_error = EGETXFILE;
	return;
}
#endif	romp
#ifdef	sun
/*
 *	Loader for SUN.
 *	Only handles ZMAGIC (413) files for SUN3.	XXX
 */
program_loader(ip, ep, pagi)
	struct inode	*ip;
	struct exec	*ep;
	int		pagi;
{
	vm_map_t	my_map;
	memory_object_t	pager;
	vm_offset_t	text_start, data_start, addr;
	long		data_end, copy_end, size, text_size, data_size;
	vm_size_t	bss_size;

/*
 *	Round the beginning of the (loaded) data segment up to the next
 *	SUN segment boundary.
 */
#define	round_sun_segment(x) (((x) + SGOFSET) & ~ SGOFSET)

	my_map = current_task()->map;

	/*
	 *	This is how unix processes are mapped onto sun:
	 *
	 *	0x2000 - segment boundary		text
	 *	(from location 0 in file!)
	 *	next segment				data/bss
	 *	USRSTACK - unix_stack_size - USRSTACK	stack
	 */

	/*
	 *	Deallocate the previous text, data, bss & stack segments.
	 */

	(void) vm_deallocate(my_map, vm_map_min(my_map),
			vm_map_max(my_map) - vm_map_min(my_map));

	if (pagi == 0) {	/* not demand paged */
		/*
		 *	Figure out where text and data should start and how
		 *	big each should be
		 */
		text_size = ep->a_text;
		text_start = USRTEXT;
		data_size = ep->a_data;
		data_start = text_size
				? round_sun_segment(text_size + (vm_offset_t) USRTEXT)
				: (vm_offset_t) USRTEXT;

		/*
		 *	Allocate space for exec image
		 */
		size = round_page(text_size + data_size + ep->a_bss);
		if (vm_allocate(my_map, &text_start, size, FALSE) != KERN_SUCCESS) {
			uprintf("Cannot find space for exec image.\n");
			goto suicide;
		}

		/*
		 *	Read in the data segment (0407 & 0410).  It starts on
		 *	the next Sun segment boundary after the text.
		 */

		u.u_error = rdwri(UIO_READ, ip,
				(caddr_t)data_start,
				(int)data_size,
				(off_t)(sizeof(struct exec)+text_size),
				0, (int *)0);
		/*
		 *	Read in text segment if necessary (0410), 
		 *	and read-protect it.
		 */
		if ((u.u_error == 0) && (text_size > 0)) {
			u.u_error = rdwri(UIO_READ, ip,
				(caddr_t) text_start,
				(int)text_size,
				(off_t)sizeof(struct exec), 2, (int *) 0);
			if (u.u_error == 0) {
				(void) vm_protect(my_map,
					 text_start,
					 trunc_page(text_size),
					 FALSE,
					 VM_PROT_READ|VM_PROT_EXECUTE);
			}
		}
	}
	else {		/* demand paged */
		/*
		 *	Allocate enough space for everything (esp. since
		 *	we don't really know what to allocate anyway).
		 */

		text_start = USRTEXT;
		if (trunc_page(text_start) != text_start) {
			uprintf("text doesn't start on page boundary.\n");
			goto suicide;
		}
		text_size = round_page(ep->a_text);

		data_start = round_sun_segment(text_start + text_size);
		data_size = round_page(ep->a_data);

		u.u_error = 0;

		/*
		 *	Copy in the text segment
		 */

		pager = (memory_object_t)inode_pager_setup(ip, TRUE, TRUE);
		iunlock(ip);
		addr = text_start;
		if (vm_map(my_map, &addr, text_size, 0, FALSE,
				pager, (vm_offset_t) 0, TRUE,
				VM_PROT_ALL, VM_PROT_ALL, VM_INHERIT_COPY)
		     != KERN_SUCCESS) {
			ilock(ip);
			inode_pager_release(pager);
			uprintf("Cannot map text into user space.\n");
			goto suicide;
		}

		/*
		 *	Now copy in the data segment.
		 */
		addr = data_start;
		if (data_size != 0) {
			if (vm_map(my_map, &addr, data_size, 0, FALSE,
				pager, (vm_offset_t) loader_round_page(ep->a_text),
				TRUE, VM_PROT_ALL, VM_PROT_ALL, VM_INHERIT_COPY)
			     != KERN_SUCCESS) {
				ilock(ip);
				inode_pager_release(pager);
				uprintf("Cannot map data into user space.\n");
				goto suicide;
			}
			addr += data_size;
		}

		ilock(ip);
		inode_pager_release(pager);

		/*
		 *	Allocate BSS
		 */
		bss_size = round_page(ep->a_data + ep->a_bss) - data_size;
		if (vm_allocate(my_map, &addr, bss_size, FALSE)!= KERN_SUCCESS) {
			uprintf("Cannot find space for bss segment.\n");
			goto suicide;
		}

		/*
		 *	Reprotect just the text region.
		 *	Does nothing if there isn't one (407 execs).
		 */

		(void) vm_protect(my_map,
			 text_start,
			 trunc_page(ep->a_text),
			 FALSE,
			 VM_PROT_READ|VM_PROT_EXECUTE);

		/*
		 *	If the data segment does not end on a VM page
		 *	boundary, we have to clear the remainder of the VM 
		 *	page it ends on so that the bss segment will 
		 *	(correctly) be zero.
		 *	The loader has already guaranteed that the (text+data)
		 *	segment ends on a loader_page boundary.
		 */

		data_end = data_start + loader_round_page(ep->a_data);
		copy_end = data_start + data_size;
		if (copy_end > data_end) {
			iunlock(ip);
			if (copyout(vm_kern_zero_page, (caddr_t)data_end,
						copy_end - data_end)) {
				ilock(ip);
				uprintf("Cannot zero partial data page\n");
				goto suicide;
			}
			ilock(ip);
		}
	}

	/*
	 *	Reprotect just text page 0 to NONE
	 */
	addr = 0;
	if (vm_map(my_map, &addr, PAGE_SIZE, 0, FALSE,
		MEMORY_OBJECT_NULL, (vm_offset_t) 0, FALSE,
		VM_PROT_NONE, VM_PROT_NONE, VM_INHERIT_SHARE)
	     != KERN_SUCCESS) {
		uprintf("Cannot reserve user page 0.\n");
		goto suicide;
	}

	/*
	 *	Remember where text and data start
	 */
	u.u_text_start = (caddr_t)text_start;
	u.u_data_start = (caddr_t)data_start;

	/*
	 *	Create the stack.  (Deallocate the old one and create a new one).
	 */

	size = round_page(unix_stack_size);
	u.u_stack_start = (caddr_t) (addr = trunc_page(USRSTACK - size));
	u.u_stack_end = u.u_stack_start + size;
	u.u_stack_grows_up = FALSE;
	(void) vm_deallocate(my_map, addr, size);
	if (vm_allocate(my_map, &addr, size, FALSE) != KERN_SUCCESS) {
		uprintf("Cannot find space for stack");
		goto suicide;
	}

	return;
suicide:
	u.u_error = EGETXFILE;
	return;
}
#endif	sun

#ifdef	balance
/*
 * balance_getxfile()
 *	Loader for Sequent Balance (ns32000) object files.
 *
 * Code derived from VAX version in getxfile().
 *
 * Magic number ZMAGIC has already been converted to 0413.  No XMAGIC yet.
 * Assumes "pagi".
 *
 * Balance a.out's assume the code is loaded at 0x800, and text/data are
 * rounded to 2k boundaries.  The header is loaded at 0x800 as part of the
 * text, to avoid wasting space.  0 -> 0x7ff are filled with read-only
 * zero pages (ZMAGIC).  When/if there is XMAGIC, 0->0x7ff needs to
 * be totally invalid.
 *
 * ep->a_text includes the low 2k of address space; the file doesn't, however.
 */

#define	LOADER_LOWBYTES	LOADER_PAGE_SIZE	/* start of address space not */
						/* backed by file */

program_loader(ip, ep, pagi)
	struct inode	*ip;
	struct exec	*ep;
	int		pagi;
{
	register vm_map_t	my_map;
	register kern_return_t	ret;
	register vm_size_t	copy_size;
	register vm_offset_t	copy_end;
	register vm_offset_t	data_end;
	vm_size_t	bss_size;
	memory_object_t	pager;
	vm_offset_t	addr;
	vm_offset_t	low_delta;
	long		size;

#ifdef	lint
	pagi++;
#endif	lint

	/*
	 * Need to know how much larger MACH page is than LOADER_LOWBYTES.
	 */

	if (PAGE_SIZE > LOADER_LOWBYTES)
		low_delta = PAGE_SIZE - LOADER_LOWBYTES;
	else
		low_delta = 0;

	my_map = current_task()->map;

	(void) vm_deallocate(my_map, vm_map_min(my_map),
			vm_map_max(my_map) - vm_map_min(my_map));
	
	/*
	 *	Remember text and data starting points
	 */
	u.u_text_start = USRTEXT;
	u.u_data_start = (caddr_t) loader_round_page(ep->a_text);

	u.u_error = 0;

	/*
	 * Allocate a region backed by the exec'ed inode.
	 *
	 * copy_size is set to that part of the file that will be page-aligned
	 * in the addresss space (ie, after LOADER_LOWBYTES).  Thus, if
	 * LOADER_LOWBYTES < PAGE_SIZE, the beginning of the file is not
	 * part of the "copy" map.
	 */

	copy_size = round_page(ep->a_text + ep->a_data - PAGE_SIZE);
	if (ep->a_text+ep->a_data > PAGE_SIZE) {
		pager = (memory_object_t)inode_pager_setup(ip, TRUE, TRUE);
		iunlock(ip);
		addr = VM_MIN_ADDRESS + round_page(LOADER_LOWBYTES),
		ret = vm_map(my_map, &addr, copy_size, 0, FALSE,
			     pager, low_delta, TRUE,
			     VM_PROT_ALL, VM_PROT_ALL, VM_INHERIT_COPY);

		ilock(ip);
		inode_pager_release(pager);

		if (ret != KERN_SUCCESS) {
			uprintf("Unable to map text/data.\n");
			goto suicide;
		}
	}

	/*
	 *	Allocate the blank area preceding the text
	 */

	addr = VM_MIN_ADDRESS;
	if (vm_allocate(my_map, &addr, round_page(LOADER_LOWBYTES), FALSE)
	    != KERN_SUCCESS) {
	    	uprintf("Cannot allocate low bytes region\n");
		goto suicide;
	}

	/*
	 * If the loader page-size < PAGE_SIZE, need to read the
	 * first part of the file into place.  Do this before write-protext
	 * the text, since we must write on it.
	 */

	if (low_delta) {
		int	resid;
		ret = rdwri(UIO_READ, ip,
				(caddr_t) LOADER_LOWBYTES, (int) low_delta,
				(off_t) 0, UIO_USERISPACE, &resid);
		if (ret != KERN_SUCCESS) {
			uprintf("Could not read first page of text.\n");
			goto suicide;
		}
	}

	/*
	 * Read-protect just the text region.  Do this before we zero
	 * the bss area, so that we have only one copy of the text.
	 */

	(void) vm_protect(my_map,
		 VM_MIN_ADDRESS,
		 trunc_page(ep->a_text),
		 FALSE,
		 VM_PROT_READ|VM_PROT_EXECUTE);

	/*
	 * If the data segment does not end on a VM page boundary,
	 * we have to clear the remainder of the VM page it ends
	 * on so that the bss segment will (correctly) be zero.
	 * The loader has already guaranteed that the (text+data)
	 * segment ends on a loader_page boundary.
	 */

	data_end = VM_MIN_ADDRESS + loader_round_page(ep->a_text + ep->a_data);
	copy_end = VM_MIN_ADDRESS + round_page(LOADER_LOWBYTES) + copy_size;
	if (copy_end > data_end) {
		if(copy_end-data_end > PAGE_SIZE) {
			uprintf("Cannot clear front of bss segment.\n");
			goto suicide;
		}
		iunlock(ip);
		if (copyout(vm_kern_zero_page, (caddr_t)data_end, copy_end - data_end)) {
			ilock(ip);
			uprintf("Cannot zero partial data page\n");
			goto suicide;
		}
		ilock(ip);
	}

	/*
	 *	Allocate the BSS region
	 */

	bss_size = round_page(ep->a_text + ep->a_data + ep->a_bss) - copy_end;
	addr = copy_end;
	if (bss_size != 0) {
		if (vm_allocate(my_map, &addr, bss_size, FALSE) != KERN_SUCCESS) {
			uprintf("Cannot allocate BSS region\n");
			goto suicide;
		}
	}

	/*
	 * Create the stack.  (Deallocate the old one and create a new one).
	 *
	 * Is it really necessary to deallocate the old stack?  The
	 * vm_map_remove() done above should have deleted the entire
	 * address space.  This one might make some data/bss dissappear,
	 * though.
	 */

	size = round_page(unix_stack_size);
	u.u_stack_start = (caddr_t) (addr = trunc_page(VM_MAX_ADDRESS - size));
	u.u_stack_end = u.u_stack_start + size;
	u.u_stack_grows_up = FALSE;
	(void) vm_deallocate(my_map, addr, size);
	if (vm_allocate(my_map, &addr, size, FALSE) != KERN_SUCCES) {
		uprintf("Cannot create stack.\n");
		goto suicide;
	}
	return;

suicide:
	u.u_error = EGETXFILE;
	return;
}
#endif	balance

#include <sys/reboot.h>

char		init_program_name[128] = "/etc/mach_init\0";

char		init_args[128] = "-xx\0";
struct execa	init_exec_args;
int		init_attempts = 0;

void load_init_program()
{
	vm_offset_t	init_addr;
	vm_size_t	init_size;
	int		*old_ap;
	char		*argv[3];

	unix_master();

	u.u_error = 0;

	init_args[1] = (boothowto & RB_SINGLE) ? 's' : 'x';
	init_args[2] = (boothowto & RB_ASKNAME) ? 'a' : 'x';
	
	do {
#if	defined(balance) || defined(multimax)
		if (init_attempts == 2)
			panic("Can't load init");
#else	balance || multimax
		if (boothowto & RB_INITNAME) {
			printf("init program? ");
			gets(init_program_name);
		}

#endif	balance || multimax
		if (u.u_error && ((boothowto & RB_INITNAME) == 0) &&
					(init_attempts == 1)) {
			static char other_init[] = "/etc/init";
			u.u_error = 0;
			bcopy(other_init, init_program_name,
							sizeof(other_init));
		}

		init_attempts++;

		if (u.u_error) {
			printf("Load of %s failed, errno %d\n",
					init_program_name, u.u_error);
			u.u_error = 0;
			boothowto |= RB_INITNAME;
		}

		/*
		 *	Copy out program name.
		 */

		init_size = round_page(sizeof(init_program_name) + 1);
		init_addr = VM_MIN_ADDRESS;
		(void) vm_allocate(current_task()->map, &init_addr, init_size, TRUE);
		if (init_addr == 0)
			init_addr++;
		(void) copyout((caddr_t) init_program_name,
				(caddr_t) (init_addr),
				(unsigned) sizeof(init_program_name));

		argv[0] = (char *) init_addr;

		/*
		 *	Put out first (and only) argument, similarly.
		 */

		init_size = round_page(sizeof(init_args) + 1);
		init_addr = VM_MIN_ADDRESS;
		(void) vm_allocate(current_task()->map, &init_addr, init_size, TRUE);
		if (init_addr == 0)
			init_addr++;
		(void) copyout((caddr_t) init_args,
				(caddr_t) (init_addr),
				(unsigned) sizeof(init_args));

		argv[1] = (char *) init_addr;

		/*
		 *	Null-end the argument list
		 */

		argv[2] = (char *) 0;
		
		/*
		 *	Copy out the argument list.
		 */
		
		init_size = round_page(sizeof(argv));
		init_addr = VM_MIN_ADDRESS;
		(void) vm_allocate(current_task()->map, &init_addr, init_size, TRUE);
		(void) copyout((caddr_t) argv,
				(caddr_t) (init_addr),
				(unsigned) sizeof(argv));

		/*
		 *	Set up argument block for fake call to execve.
		 */

		init_exec_args.fname = argv[0];
		init_exec_args.argp = (char **) init_addr;
		init_exec_args.envp = 0;

		old_ap = u.u_ap;
		u.u_ap = (int *) &init_exec_args;
		execve();
		u.u_ap = old_ap;
	} while (u.u_error);

	unix_release();
}

#endif	MACH
