/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/* 
 * HISTORY
 * $Log:	kern_mman.c,v $
 * Revision 2.9  88/12/13  12:02:28  mikeg
 * 	Enclosed modifications to smmap for Suntools in "#ifdef sun"
 * 	conditionals.
 * 
 * Revision 2.8  88/12/08  15:35:24  mikeg
 * 	Modified smmap so that a call to mmap that invokes winmmap
 * 	will have the requested memory marked as MAP_PRIVATE.  This
 * 	allows Suntools to run under Mach.
 * 	[88/11/30  10:18:55  mikeg]
 * 
 * Revision 2.6  88/10/25  03:38:26  mwyoung
 * 	MACH_XP: Get the right offset for objects managed by the
 * 	device_pager.
 * 	[88/10/25            mwyoung]
 * 
 * Revision 2.5  88/10/11  12:07:00  rpd
 * 	Correct mmap() to copy data for MAP_PRIVATE; allow private
 * 	mappings to later increase privileges on the mapped region.
 * 	(From mwyoung.)
 * 	[88/10/11  12:01:46  rpd]
 * 
 * Revision 2.4  88/08/25  18:10:27  mwyoung
 * 	Corrected include file references.
 * 	[88/08/22            mwyoung]
 * 	
 * 	Eliminate old variables in smmap.
 * 	[88/08/21            mwyoung]
 * 	
 * 	Use memory_object_special when not MACH_XP; condense all of
 * 	the mapping operations to a single vm_map call.
 * 	[88/08/21  18:17:24  mwyoung]
 * 	
 * 	Vast simplification of mmap() through use of vm_map().
 * 	Further simplification would be possible if vm_object_special()
 * 	returned a memory object suitable for use in vm_map() instead of
 * 	a vm_object.
 * 	[88/08/20  03:07:54  mwyoung]
 * 
 * Revision 2.3  88/07/17  17:38:39  mwyoung
 * Use new memory object types.
 * 
 * Print the name of the process that can't sbrk.
 * 
 * Revision 2.2.1.2  88/07/04  15:18:40  mwyoung
 * Use new memory object types.
 * 
 * Revision 2.2.1.1  88/06/28  20:14:25  mwyoung
 * Print the name of the process that can't sbrk.
 * 
 *
 *  9-May-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Print the name of the process that can't sbrk.
 *
 * 16-Feb-88  David Golub (dbg) at Carnegie-Mellon University
 *	Correct fix below to return proper U*X return code.  Check that
 *	mapfun is not NULL, nulldev, or nodev when mapping devices.
 *
 * 30-Jan-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added check to smmap to only work on local inodes.
 *
 * 30-Dec-87  David Golub (dbg) at Carnegie-Mellon University
 *	Delinted.
 *
 *  6-Dec-87  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Changed inode_pager_setup() use to new interface.
 *	Removed meaningless history.
 *
 *  9-Apr-87  William Bolosky (bolosky) at Carnegie-Mellon University
 *	MACH_XP: Turned off device file mmaping for the interim.
 *
 *  2-Mar-87  David Golub (dbg) at Carnegie-Mellon University
 *	Made mmap handle both special devices and files for all machines.
 *
 * 14-Oct-86  David Golub (dbg) at Carnegie-Mellon University
 *	Made mmap work for character devices to support (sun) frame
 *	buffers.
 *
 **********************************************************************
 */

#include <mach.h>
#include <mach_xp.h>
/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)kern_mman.c	7.1 (Berkeley) 6/5/86
 */

#include <machine/reg.h>
#ifdef	romp
#else	romp
#include <machine/psl.h>
#endif	romp
#if	MACH
#else	MACH
#include <machine/pte.h>
#endif	MACH

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/map.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/buf.h>
#include <sys/inode.h>
#if	MACH
#else	MACH
#include <sys/seg.h>
#endif	MACH
#include <sys/acct.h>
#include <sys/wait.h>
#include <sys/vm.h>
#if	MACH
#else	MACH
#include <sys/text.h>
#endif	MACH
#include <sys/file.h>
#include <sys/vadvise.h>
#if	MACH
#else	MACH
#include <sys/cmap.h>
#endif	MACH
#include <sys/trace.h>
#include <sys/mman.h>
#include <sys/conf.h>
#if	MACH
#include <sys/kern_return.h>
#include <sys/task.h>
#include <vm/vm_param.h>
#include <vm/vm_map.h>
#include <vm/memory_object.h>
#include <vm/inode_pager.h>
#endif	MACH
#if	MACH_XP
#include <vm/device_pager.h>
#endif	MACH_XP

sbrk()
{

}

sstk()
{

}

getpagesize()
{

	u.u_r.r_val1 = NBPG * CLSIZE;
}

smmap()
{
#if	MACH
	/*
	 *	Map in special device (must be SHARED) or file
	 */
	struct a {
		caddr_t	addr;
		int	len;
		int	prot;
		int	share;
		int	fd;
		off_t	pos;
	} *uap = (struct a *)u.u_ap;
	register struct file *fp;
	register struct inode *ip;
	vm_map_t	user_map = current_task()->map;
	kern_return_t	result;
	vm_offset_t	user_addr = (vm_offset_t) uap->addr;
	vm_size_t	user_size = uap->len;
	off_t		file_pos = uap->pos;
	memory_object_t	pager;
	vm_offset_t	pager_offset;
	vm_prot_t	user_prot =
				(uap->prot & PROT_WRITE) ?
					VM_PROT_ALL :
					VM_PROT_READ|VM_PROT_EXECUTE;
#if	MACH
	int		realshare = uap->share; /* For winmmap kluge */
#endif	MACH

	extern struct file	*getinode();
#if	MACH_XP
#else	MACH_XP
	extern memory_object_t	memory_object_special();
#endif	MACH_XP

	/*
	 *	Only work with local inodes.  Do not use getinode - it will
	 *	try to invoke remote filesystem code, which will panic on smmap.
	 */

	GETF(fp, uap->fd);
	if (fp->f_type != DTYPE_INODE) {
		u.u_error = EINVAL;
		return;
	}
	ip = (struct inode *)fp->f_data;

	/*
	 *	We bend a little - round the start and end addresses
	 *	to the nearest page boundary.
	 */
	user_addr = trunc_page(user_addr);
	user_size = round_page(user_size);

	/*
	 *	File can be COPIED at an arbitrary offset.
	 *	File can only be SHARED if the offset is at a
	 *	page boundary.
	 *
	if (uap->share == MAP_SHARED &&
	    (trunc_page(file_pos) != (vm_offset_t)(file_pos))) {
		u.u_error = EINVAL;
		return;
	}

	/*
	 *	File must be writable if mapping will be.
	 */
	if ((uap->prot & PROT_WRITE) && (fp->f_flag&FWRITE) == 0) {
		u.u_error = EINVAL;
		return;
	}
	if ((uap->prot & PROT_READ) && (fp->f_flag&FREAD) == 0) {
		u.u_error = EINVAL;
		return;
	}

	/*
	 *	memory must exist and be writable (even if we're
	 *	just reading)
	 *
	 *	XXX Is this really part of the BSD spec for mmap?
	 *	We'll be replacing the address range anyway.
	 */

	if (!vm_map_check_protection(user_map, user_addr,
				(vm_offset_t)(user_addr + user_size),
				VM_PROT_READ|VM_PROT_WRITE)) {
		u.u_error = EINVAL;
		return;
	}

	if ((ip->i_mode & IFMT) == IFCHR) {
		/*
		 *	Map in special device memory.
		 *	Must be shared, since we can't assume that
		 *	this memory can be made copy-on-write.
		 */
		int		off;
		dev_t		dev;
		int		(*mapfun)();
		extern int	nodev();
		extern int	nulldev();
#if	MACH
#ifdef	sun
		extern int	winmmap(); /* Needed for Suntools kluge */
#endif	sun
#endif	MACH

		dev = ip->i_rdev;
		mapfun = cdevsw[major(dev)].d_mmap;
		if (mapfun == nulldev || mapfun == nodev || mapfun == 0) {
			u.u_error = EINVAL;
			return;
		}

#if	MACH
#ifdef	sun

	/*
	 *	The following modification is required for running Suntools.  If
	 *	this is a call to winmmap, the Suntools task is trying to map
	 *	in a data structure in the window driver.  That is, it is using
	 *	the mmap call to map some kernel memory into its address space!!
	 *	This klugey use of mmap breaks the Mach VM system, but the
	 *	problem can be avoided by simply not having the memory marked
	 *	as shared.
	 */
		if( mapfun == winmmap )
			realshare = MAP_PRIVATE;
#endif	sun
#endif	MACH

		for (off=0; off<uap->len; off += NBPG) {
			if ((*mapfun)(dev, file_pos+off, uap->prot) == -1) {
				u.u_error = EINVAL;
				return;
			}
		}

		if (uap->share != MAP_SHARED) {
			u.u_error = EINVAL;
			return;
		}
	
#if	MACH_XP
		pager_offset = (vm_offset_t) file_pos;
		pager = device_pager_create(dev, mapfun, uap->prot,
				pager_offset, user_size);
#else	MACH_XP
		pager = memory_object_special(dev, mapfun, uap->prot,
				(vm_offset_t) file_pos, user_size);
		pager_offset = 0;
#endif	MACH_XP
		if (pager == PORT_NULL) {
			u.u_error = EINVAL;
			return;
		}
	}
	else {
		/*
		 *	Map in a file.  May be PRIVATE (copy-on-write)
		 *	or SHARED (changes go back to file)
		 */

		/*
		 *	Only allow regular files for the moment.
		 */
		if ((ip->i_mode & IFMT) != IFREG) {
			u.u_error = EINVAL;
			return;
		}
		
		if ((pager = inode_pager_setup(ip, FALSE, FALSE))
				== MEMORY_OBJECT_NULL) {
			u.u_error = KERN_INVALID_ARGUMENT;
			return;
		}
		pager_offset = (vm_offset_t) file_pos;
	}

	/*
	 *	Deallocate the existing memory, then map the appropriate
	 *	memory object into the space left.
	 */

	result = vm_deallocate(user_map, user_addr, user_size);
	if (result == KERN_SUCCESS)
		result = vm_map(user_map, &user_addr, user_size, 0, FALSE,
				pager, pager_offset,
				(uap->share != MAP_SHARED),
				user_prot,
				(uap->share == MAP_SHARED) ?
					user_prot : VM_PROT_ALL,
				((realshare == MAP_SHARED) ?
					VM_INHERIT_SHARE : VM_INHERIT_COPY)
				);
	
	/*
	 *	Throw away references to the appropriate pager
	 */

	if ((ip->i_mode & IFMT) == IFCHR) {
#if	MACH_XP
		port_release(pager);
#else	MACH_XP
		vm_object_t	object = vm_object_lookup(pager);
		
		if (object != VM_OBJECT_NULL) {
			/* Once for the lookup, once for real */
			vm_object_deallocate(object);
			vm_object_deallocate(object);
		}
#endif	MACH_XP
	} else {
		inode_pager_release(pager);
	}

	if (result != KERN_SUCCESS) {
		u.u_error = EINVAL;
		return;
	}

	u.u_pofile[uap->fd] |= UF_MAPPED;
#else	MACH
#ifdef notdef
	struct a {
		caddr_t	addr;
		int	len;
		int	prot;
		int	share;
		int	fd;
		off_t	pos;
	} *uap = (struct a *)u.u_ap;
	register struct file *fp;
	register struct inode *ip;
	register struct fpte *pte;
	int off;
	int fv, lv, pm;
	dev_t dev;
	int (*mapfun)();
	extern struct file *getinode();

	fp = getinode(uap->fd);
	if (fp == NULL)
		return;
	ip = (struct inode *)fp->f_data;
	if ((ip->i_mode & IFMT) != IFCHR) {
		u.u_error = EINVAL;
		return;
	}
	dev = ip->i_rdev;
	mapfun = cdevsw[major(dev)].d_mmap;
	if (mapfun == NULL) {
		u.u_error = EINVAL;
		return;
	}
	if (((int)uap->addr & CLOFSET) || (uap->len & CLOFSET) ||
	    (uap->pos & CLOFSET)) {
		u.u_error = EINVAL;
		return;
	}
	if ((uap->prot & PROT_WRITE) && (fp->f_flag&FWRITE) == 0) {
		u.u_error = EINVAL;
		return;
	}
	if ((uap->prot & PROT_READ) && (fp->f_flag&FREAD) == 0) {
		u.u_error = EINVAL;
		return;
	}
	if (uap->share != MAP_SHARED) {
		u.u_error = EINVAL;
		return;
	}
	fv = btop(uap->addr);
	lv = btop(uap->addr + uap->len - 1);
	if (lv < fv || !isadsv(u.u_procp, fv) || !isadsv(u.u_procp, lv)) {
		u.u_error = EINVAL;
		return;
	}
	for (off=0; off<uap->len; off += NBPG) {
		if ((*mapfun)(dev, uap->pos+off, uap->prot) == -1) {
			u.u_error = EINVAL;
			return;
		}
	}
	if (uap->prot & PROT_WRITE)
		pm = PG_UW;
	else
		pm = PG_URKR;
	for (off = 0; off < uap->len; off += NBPG) {
		pte = (struct fpte *)vtopte(u.u_procp, fv);
		u.u_procp->p_rssize -= vmemfree(pte, 1);
		*(int *)pte = pm;
		pte->pg_v = 1;
		pte->pg_fod = 1;
		pte->pg_fileno = uap->fd;
		pte->pg_blkno = (*mapfun)(dev, uap->pos+off, uap->prot);
		fv++;
	}
	u.u_procp->p_flag |= SPTECHG;
	u.u_pofile[uap->fd] |= UF_MAPPED;
#endif
#endif	MACH
}

mremap()
{

}

munmap()
{
#if	MACH
	register struct a {
		caddr_t	addr;
		int	len;
	} *uap = (struct a *)u.u_ap;
	vm_offset_t	user_addr;
	vm_size_t	user_size;
	kern_return_t	result;

	user_addr = (vm_offset_t) uap->addr;
	user_size = (vm_size_t) uap->len;
	if ((user_addr & page_mask) ||
	    (user_size & page_mask)) {
		u.u_error = EINVAL;
		return;
	}
	result = vm_deallocate(current_task()->map, user_addr, user_size);
	if (result != KERN_SUCCESS) {
		u.u_error = EINVAL;
		return;
	}
#else	MACH
#ifdef notdef
	register struct a {
		caddr_t	addr;
		int	len;
	} *uap = (struct a *)u.u_ap;
	int off;
	int fv, lv;
	register struct pte *pte;

	if (((int)uap->addr & CLOFSET) || (uap->len & CLOFSET)) {
		u.u_error = EINVAL;
		return;
	}
	fv = btop(uap->addr);
	lv = btop(uap->addr + uap->len - 1);
	if (lv < fv || !isadsv(u.u_procp, fv) || !isadsv(u.u_procp, lv)) {
		u.u_error = EINVAL;
		return;
	}
	for (off = 0; off < uap->len; off += NBPG) {
		pte = vtopte(u.u_procp, fv);
		u.u_procp->p_rssize -= vmemfree(pte, 1);
		*(int *)pte = (PG_UW|PG_FOD);
		((struct fpte *)pte)->pg_fileno = PG_FZERO;
		fv++;
	}
	u.u_procp->p_flag |= SPTECHG;
#endif
#endif	MACH
}

munmapfd(fd)
{
#ifdef notdef
	register struct fpte *pte;
	register int i;

	for (i = 0; i < u.u_dsize; i++) {
		pte = (struct fpte *)dptopte(u.u_procp, i);
		if (pte->pg_v && pte->pg_fod && pte->pg_fileno == fd) {
			*(int *)pte = (PG_UW|PG_FOD);
			pte->pg_fileno = PG_FZERO;
		}
	}
#endif
	u.u_pofile[fd] &= ~UF_MAPPED;
	
}

mprotect()
{

}

madvise()
{

}

mincore()
{

}

/* BEGIN DEFUNCT */
obreak()
{
	struct a {
		char	*nsiz;
	};
#if	MACH
	vm_offset_t	old, new;
	kern_return_t	ret;

	new = round_page(((struct a *)u.u_ap)->nsiz);
	if ((int)(new - (vm_offset_t)u.u_data_start) > u.u_rlimit[RLIMIT_DATA].rlim_cur) {
		u.u_error = ENOMEM;
		return;
	}
	old = round_page(u.u_data_start + ctob(u.u_dsize));
	if (new > old) {
		ret = vm_allocate(current_task()->map, &old, new - old, FALSE);
		if (ret != KERN_SUCCESS)
			uprintf("%s: could not sbrk, return = %d\n", u.u_comm, ret);
		else
			u.u_dsize += btoc(new - old);
	}
#else	MACH
	register size_t n, d, ds;

	/*
	 * set n to new data size
	 */
	n = btoc(((struct a *)u.u_ap)->nsiz) - ctos(u.u_tsize) * stoc(1);

	if (n < 0)
		n = 0;
	/*
	 * since we can't pass a -ve argument for the difference to chksize,
	 * if d is negative, make ds equal to the final value and clear d.
	 * keep the real difference in n for later use in expand.
	 */
	ds = u.u_dsize;
	if ((n = d = clrnd(n - u.u_dsize)) < 0) {
		ds += d;
		d = 0;
	}
	if (ctob(ds + d) > u.u_rlimit[RLIMIT_DATA].rlim_cur) {
		u.u_error = ENOMEM;
		return;
	}
	if (chksize((u_int)u.u_tsize, (u_int)ds, (u_int)d, (u_int)u.u_ssize))
		return;
	if (swpexpand(ds + d, u.u_ssize, &u.u_dmap, &u.u_smap) == 0)
		return;
	expand((int)n, 0);
#endif	MACH
}

int	both;

ovadvise()
{
#if	MACH
#else	MACH
	register struct a {
		int	anom;
	} *uap;
	register struct proc *rp = u.u_procp;
	int oanom = rp->p_flag & SUANOM;
	register struct pte *pte;
	register struct cmap *c;
	register unsigned i;
#endif	MACH

#ifdef lint
	both = 0;
#endif
#if	MACH
#else	MACH
	uap = (struct a *)u.u_ap;
	trace(TR_VADVISE, uap->anom, u.u_procp->p_pid);
	rp->p_flag &= ~(SSEQL|SUANOM);
	switch (uap->anom) {

	case VA_ANOM:
		rp->p_flag |= SUANOM;
		break;

	case VA_SEQL:
		rp->p_flag |= SSEQL;
		break;
	}
	if ((oanom && (rp->p_flag & SUANOM) == 0) || uap->anom == VA_FLUSH) {
		for (i = 0; i < rp->p_dsize; i += CLSIZE) {
			pte = dptopte(rp, i);
			if (pte->pg_v) {
				c = &cmap[pgtocm(pte->pg_pfnum)];
				if (c->c_lock)
					continue;
				pte->pg_v = 0;
				if (anycl(pte, pg_m))
					pte->pg_m = 1;
				distcl(pte);
			}
		}
	}
	if (uap->anom == VA_FLUSH) {	/* invalidate all pages */
		for (i = 1; i < rp->p_ssize; i += CLSIZE) {
			pte = sptopte(rp, i);
			if (pte->pg_v) {
				c = &cmap[pgtocm(pte->pg_pfnum)];
				if (c->c_lock)
					continue;
				pte->pg_v = 0;
				if (anycl(pte, pg_m))
					pte->pg_m = 1;
				distcl(pte);
			}
		}
		for (i = 0; i < rp->p_tsize; i += CLSIZE) {
			pte = tptopte(rp, i);
			if (pte->pg_v) {
				c = &cmap[pgtocm(pte->pg_pfnum)];
				if (c->c_lock)
					continue;
				pte->pg_v = 0;
				if (anycl(pte, pg_m))
					pte->pg_m = 1;
				distcl(pte);
				distpte(rp->p_textp, i, pte);
			}
		}
	}
#ifdef vax
#include <vax/mtpr.h>		/* XXX */
	mtpr(TBIA, 0);
#endif
#endif	MACH
}
/* END DEFUNCT */

/*
 * grow the stack to include the SP
 * true return if successful.
 */
grow(sp)
	unsigned sp;
{
	register int si;

	if (sp >= USRSTACK-ctob(u.u_ssize))
		return (0);
#if	MACH
	si = round_page((USRSTACK - sp) - ctob(u.u_ssize) + btoc(SINCR));
	if (si > u.u_rlimit[RLIMIT_STACK].rlim_cur)
		return (0);
#else	MACH
	si = clrnd(btoc((USRSTACK-sp)) - u.u_ssize + SINCR);
	if (ctob(si) > u.u_rlimit[RLIMIT_STACK].rlim_cur)
		return (0);
	if (chksize((u_int)u.u_tsize, (u_int)u.u_dsize, (u_int)0,
	    (u_int)u.u_ssize+si))
		return (0);
	if (swpexpand(u.u_dsize, u.u_ssize+si, &u.u_dmap, &u.u_smap)==0)
		return (0);
	
	expand(si, 1);
#endif	MACH
	return (1);
}
