#if	CMU
/*  
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/* 
 * HISTORY
 * $Log:	kern_synch.c,v $
 * Revision 2.7  89/01/15  16:17:38  rpd
 * 	Include <sys/callout.h> instead of declaring callout_lock by hand.
 * 	[89/01/15  14:48:15  rpd]
 * 
 * Revision 2.6  88/08/24  01:20:45  mwyoung
 * 	Corrected include file references.
 * 	[88/08/23  01:00:57  mwyoung]
 * 
 * Revision 2.5  88/07/17  17:24:32  mwyoung
 * Converted use of kernel_only to kernel_vm_space in schedcpu().
 * It's unclear that this is the best choice, but it's compatible.
 * 
 * Revision 2.3.1.1  88/06/28  20:16:26  mwyoung
 * Converted use of kernel_only to kernel_vm_space in schedcpu().
 * It's unclear that this is the best choice, but it's compatible.
 * 
 *
 * 21-Jun-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Converted use of kernel_only to kernel_vm_space in schedcpu().
 *	It's unclear that this is the best choice, but it's compatible.
 *
 * Revision 2.4  88/07/15  18:22:10  mja
 * Changed to include <sys/table.h> instead of <sys/syscalls.h> for
 * setmodes bit definitons.
 * 
 *  2-Jun-88  David Golub (dbg) at Carnegie-Mellon University
 *	MACH: added missing check for UMODE_NONICE mode bit in schedcpu()
 *	to disable auto-nice mechanism.
 *	[ V5.1(XF27) ]
 *
 *  4-May-88  David Black (dlb) at Carnegie-Mellon University
 *	MACH: Cleaned up conditionals in sleep().
 *	      Moved autonice code here from softclock().
 *	      No more SSLEEP state for P_STAT.
 *
 * 21-Apr-88  Mike Accetta (mja) at Carnegie-Mellon University
 *	Revised rpsleep() routine to support polled resource pause
 *	operation and moved continuation message display into new
 *	rpcont() routine.
 *	[ V5.1(XF23) ]
 *
 * 18-Feb-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Check for use of kernel_pmap to determine whether to (de,)activate.
 *
 * 26-Jan-88  David Black (dlb) at Carnegie-Mellon University
 *	Don't activate pmaps for kernel_only tasks.
 *
 * 21-Nov-87  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Reduced conditionals, purged history.
 *
 * 30-Jan-87  Mike Accetta (mja) at Carnegie-Mellon University
 *	CS_RPAUSE: Renamed rpause() to be rpsleep() since rpause()
 *	is now the system call used to enable/disable resource pausing;
 *
 * 03-Aug-85  Mike Accetta (mja) at Carnegie-Mellon University
 *	CS_RPAUSE:  Added rpause() routine.
 *	[V1(1)]
 *
 * 10-Jun-85  Mike Accetta (mja) at Carnegie-Mellon University
 *	CS_COMPAT:  Redid tsleep() to use a private timeout()
 *	routine since piggybacking off the alarm (real interval) timer
 *	is no longer appropriate.
 *	[V1(1)]
 *
 * 15-May-85  Glenn Marcy (gm0w) at Carnegie-Mellon University
 *	Upgraded from 4.1BSD.  Carried over changes below:
 *
 *	CS_COMPAT:  Carry over tsleep() routine from 4.1BSD.
 *	[V1(1)]
 *
 **********************************************************************
 */

#include <cputypes.h>

#include <cmucs.h>
#include <mach.h>

#include <cpus.h>
#endif	CMU

/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)kern_synch.c	7.1 (Berkeley) 6/5/86
 */

#if	MACH
#else	MACH
#include <machine/pte.h>
#endif	MACH

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/file.h>
#include <sys/inode.h>
#include <sys/vm.h>
#include <sys/kernel.h>
#include <sys/buf.h>
#if	CMUCS
#include <sys/table.h>
#endif	CMUCS

#if	MACH
#include <sys/callout.h>
#include <sys/queue.h>
#include <sys/lock.h>
#include <sys/thread.h>
#include <sys/sched.h>
#include <sys/sched_prim.h>
#include <sys/machine.h>
#include <kern/parallel.h>

#include <machine/cpu.h>
#include <vm/pmap.h>
#include <vm/vm_kern.h>

#endif	MACH
#ifdef vax
#include <vax/mtpr.h>	/* XXX */
#endif

#if	MACH
#include <sys/task.h>
#include <sys/time_value.h>
#endif	MACH

#if	MACH
	/*
	 *	Quantum scheduling replaces roundrobin().
	 */
#else	MACH
/*
 * Force switch among equal priority processes every 100ms.
 */
roundrobin()
{

	runrun++;
	aston();
	timeout(roundrobin, (caddr_t)0, hz / 10);
}
#endif	MACH

/* fraction for digital decay to forget 90% of usage in 5*loadav sec */
#define	filter(loadav) ((2 * (loadav)) / (2 * (loadav) + 1))

double	ccpu = 0.95122942450071400909;		/* exp(-1/20) */

/*
 * Recompute process priorities, once a second
 */
schedcpu()
{
#if	MACH
	register thread_t th;
#else	MACH
	register double ccpu1 = (1.0 - ccpu) / (double)hz;
	register struct proc *p;
	register int s, a;
	float scale = filter(avenrun[0]);
#endif	MACH

	wakeup((caddr_t)&lbolt);

#if	MACH
	/*
	 *	Autonice code moved here from kern_clock.c
	 */
	th = current_thread();
	if (!(th->task->kernel_vm_space)) {
	    register struct proc *p;

	    p = u.u_procp;
#if	CMUCS
	    if (p->p_uid && p->p_nice == NZERO &&
		(u.u_modes & UMODE_NONICE) == 0)
#else	CMUCS
	    if (p->p_uid && p->p_nice == NZERO)
#endif	CMUCS
	    {
		time_value_t	user_time;

		timer_read(&(th->user_timer), &user_time);
		if (user_time.seconds > 10 * 60) {
		    register int	s;

		    p->p_nice = NZERO+4;
		    th->priority = BASEPRI_USER + 2*p->p_nice;
		    s = splsched();
		    thread_lock(th);
		    compute_my_priority(th);
		    thread_unlock(th);
		    splx(s);
		}
	    }
	}
#else	MACH
	for (p = allproc; p != NULL; p = p->p_nxt) {
		if (p->p_time != 127)
			p->p_time++;
		if (p->p_stat==SSLEEP || p->p_stat==SSTOP)
			if (p->p_slptime != 127)
				p->p_slptime++;
		/*
		 * If the process has slept the entire second,
		 * stop recalculating its priority until it wakes up.
		 */
		if (p->p_slptime > 1) {
			p->p_pctcpu *= ccpu;
			continue;
		}
		/*
		 * p_pctcpu is only for ps.
		 */
		p->p_pctcpu = ccpu * p->p_pctcpu + ccpu1 * p->p_cpticks;
		p->p_cpticks = 0;
		a = (int) (scale * (p->p_cpu & 0377)) + p->p_nice;
		if (a < 0)
			a = 0;
		if (a > 255)
			a = 255;
		p->p_cpu = a;
		(void) setpri(p);
		s = splhigh();	/* prevent state changes */
		if (p->p_pri >= PUSER) {
#define	PPQ	(128 / NQS)
			if ((p != u.u_procp || noproc) &&
			    p->p_stat == SRUN &&
			    (p->p_flag & SLOAD) &&
			    (p->p_pri / PPQ) != (p->p_usrpri / PPQ)) {
				remrq(p);
				p->p_pri = p->p_usrpri;
				setrq(p);
			} else
				p->p_pri = p->p_usrpri;
		}
		splx(s);
	}
	vmmeter();
	if (runin!=0) {
		runin = 0;
		wakeup((caddr_t)&runin);
	}
	if (bclnlist != NULL)
		wakeup((caddr_t)&proc[2]);
#endif	MACH
	timeout(schedcpu, (caddr_t)0, hz);
}

#if	MACH
#else	MACH
/*
 * Recalculate the priority of a process after it has slept for a while.
 */
updatepri(p)
	register struct proc *p;
{
	register int a = p->p_cpu & 0377;
	float scale = filter(avenrun[0]);

	p->p_slptime--;		/* the first time was done in schedcpu */
	while (a && --p->p_slptime)
		a = (int) (scale * a) /* + p->p_nice */;
	if (a < 0)
		a = 0;
	if (a > 255)
		a = 255;
	p->p_cpu = a;
	(void) setpri(p);
}

#define SQSIZE 0100	/* Must be power of 2 */
#define HASH(x)	(( (int) x >> 5) & (SQSIZE-1))
struct slpque {
	struct proc *sq_head;
	struct proc **sq_tailp;
} slpque[SQSIZE];

#endif	MACH
/*
 * Give up the processor till a wakeup occurs
 * on chan, at which time the process
 * enters the scheduling queue at priority pri.
 * The most important effect of pri is that when
 * pri<=PZERO a signal cannot disturb the sleep;
 * if pri>PZERO signals will be processed.
 * Callers of this routine must be prepared for
 * premature return, and check that the reason for
 * sleeping has gone away.
 */
sleep(chan, pri)
	caddr_t chan;
	int pri;
{
	register struct proc *rp;
#if	MACH
#else	MACH
	register struct slpque *qp;
#endif	MACH
	register s;

	rp = u.u_procp;
	s = splhigh();
	if (panicstr) {
		/*
		 * After a panic, just give interrupts a chance,
		 * then just return; don't run any other procs 
		 * or panic below, in case this is the idle process
		 * and already asleep.
		 * The splnet should be spl0 if the network was being used
		 * by the filesystem, but for now avoid network interrupts
		 * that might cause another panic.
		 */
		(void) splnet();
		splx(s);
		return;
	}
#if	MACH
#else	MACH
	if (chan==0 || rp->p_stat != SRUN || rp->p_rlink)
		panic("sleep");
	rp->p_wchan = chan;
#endif	MACH
	rp->p_slptime = 0;
	rp->p_pri = pri;
#if	MACH
	assert_wait((int) chan, pri > PZERO);
#else	MACH
	qp = &slpque[HASH(chan)];
	if (qp->sq_head == 0)
		qp->sq_head = rp;
	else
		*qp->sq_tailp = rp;
	*(qp->sq_tailp = &rp->p_link) = 0;
#endif	MACH
	if (pri > PZERO) {
#if	MACH
		/*
		 * If wakeup occurs while in issig, thread_block()
		 * below is a no-op.  If ISSIG finds a signal, clear
		 * sleep condition before going to process it.
		 */
		if (ISSIG(rp)) {
			clear_wait(current_thread(), THREAD_INTERRUPTED,
					TRUE);
			(void) spl0();
			goto psig;
		}
		(void) spl0();
		u.u_ru.ru_nvcsw++;
		if (cpu_number() != master_cpu) {
			printf("unix sleep: on slave?");
		}
		thread_block();
		if (ISSIG(rp))
			goto psig;
#else	MACH
		/*
		 * If we stop in issig(), wakeup may already have happened
		 * when we return (rp->p_wchan will then be 0).
		 */
		if (ISSIG(rp)) {
			if (rp->p_wchan)
				unsleep(rp);
			rp->p_stat = SRUN;
			(void) spl0();
			goto psig;
		}
		if (rp->p_wchan == 0)
			goto out;
		rp->p_stat = SSLEEP;
		(void) spl0();
		u.u_ru.ru_nvcsw++;
		swtch();
		if (ISSIG(rp))
			goto psig;
#endif	MACH
	} else {
#if	MACH
		(void) spl0();
		u.u_ru.ru_nvcsw++;
		if (cpu_number() != master_cpu) {
			printf("unix sleep: on slave?");
		}
		thread_block();
#else	MACH
		rp->p_stat = SSLEEP;
		(void) spl0();
		u.u_ru.ru_nvcsw++;
		swtch();
#endif	MACH
	}
#if	MACH
#else	MACH
	curpri = rp->p_usrpri;
out:
#endif	MACH
	splx(s);
	return;

	/*
	 * If priority was low (>PZERO) and
	 * there has been a signal, execute non-local goto through
	 * u.u_qsave, aborting the system call in progress (see trap.c)
#if	CMUCS
	 * (or finishing a tsleep, see below)
#endif	CMUCS
	 */
psig:
	longjmp(&u.u_qsave);
	/*NOTREACHED*/
}

#if	CMUCS
/* 
 *  rpsleep - perform a resource pause sleep
 *
 *  rsleep = function to perform resource specific sleep
 *  arg1   = first function parameter
 *  arg2   = second function parameter
 *  mesg1  = first component of user pause message
 *  mesg2  = second component of user pause message
 *
 *  Display the appropriate pause message on the user's controlling terminal.
 *  Save the current non-local goto information and establish a new return
 *  environment to transfer here.  Invoke the supplied function to sleep
 *  (possibly interruptably) until the resource becomes available.  When the
 *  sleep finishes (either normally or abnormally via a non-local goto caused
 *  by a signal), restore the old return environment and display a resume
 *  message on the terminal.  The notify flag bit is set when the pause message
 *  is first printed.  If it is cleared on return from the function, the
 *  continue message is printed here.  If not, this bit will remain set for the
 *  duration of the polling process and the rpcont() routine will be called
 *  directly from the poller when the resource pause condition is no longer
 *  pending.
 *
 *  Return: true if the resource has now become available, or false if the wait
 *  was interrupted by a signal.
 */

boolean_t
rpsleep(rsleep, arg1, arg2, mesg1, mesg2)
int (*rsleep)();
int arg1;
int arg2;
char *mesg1;
char *mesg2;
{
    label_t lsave;
    boolean_t ret = TRUE;

    if ((u.u_rpswhich&URPW_NOTIFY) == 0)
    {
        u.u_rpswhich |= URPW_NOTIFY;
	uprintf("[%s: %s%s, pausing ...]\r\n", u.u_comm, mesg1, mesg2);
    }
    
    bcopy((caddr_t)&u.u_qsave, (caddr_t)&lsave, sizeof(lsave));
    if (setjmp(&u.u_qsave) == 0)
	(*rsleep)(arg1, arg2);
    else
	ret = FALSE;
    bcopy((caddr_t)&lsave, (caddr_t)&u.u_qsave, sizeof(lsave));

    if ((u.u_rpswhich&URPW_NOTIFY) == 0)
	rpcont();
    return(ret);
}
 
 
 
/* 
 *  rpcont - continue from resource pause sleep
 *
 *  Clear the notify flag and print the continuation message on the controlling
 *  terminal.  When this routine is called, the resource pause condition is no
 *  longer pending and we can afford to clear all bits since only the notify
 *  bit should be set to begin with.
 */

rpcont()
{
    u.u_rpswhich = 0;
    uprintf("[%s: ... continuing]\r\n", u.u_comm);
}
#endif	CMUCS
#if	CMUCS

/* 
 * Sleep on chan at pri.
 * Return in no more than the indicated number of seconds.
 * (If seconds==0, no timeout implied)
 * Return	TS_OK if chan was awakened normally
 *		TS_TIME if timeout occurred
 *		TS_SIG if asynchronous signal occurred
 */

#if	MACH
#else	MACH
tsleepexpire(p)
	register struct proc *p;
{
	p->p_flag &= ~STIMO;
	wakeup((caddr_t)&p->p_wchan);
}
#endif	MACH

tsleep(chan, pri, seconds)
	caddr_t	chan;
{
#if	MACH
 	struct timeval when;
	int	s;
	register struct proc *p = u.u_procp;

	s = splhigh();
	assert_wait((int)chan, pri > PZERO);
	if (seconds) {
		when = time;
		when.tv_sec += seconds;
		thread_set_timeout(hzto(&when));
	}
	if (pri > PZERO) {
		if (ISSIG(p)) {
			clear_wait(current_thread(), THREAD_INTERRUPTED, TRUE);
			splx(s);
			return (TS_SIG);
		}
		thread_block();
		if (ISSIG(p)) {
			splx(s);
			return (TS_SIG);
		}
	}
	else {
		thread_block();
	}
	if (current_thread()->wait_result == THREAD_TIMED_OUT) {
		splx(s);
		return(TS_TIME);
	}
	splx(s);
	return(TS_OK);
#else	MACH
	label_t lqsave;
	register struct proc *pp;
	register n, rval;
	struct timeval when;

	pp = u.u_procp;
	n = spl7();
	if (seconds)
	{
		when = time;
		when.tv_sec += seconds;
		pp->p_flag |= STIMO;
		timeout(tsleepexpire, (caddr_t)pp, hzto(&when));
	}
	bcopy((caddr_t)&u.u_qsave, (caddr_t)&lqsave, sizeof (label_t));
	if (setjmp(&u.u_qsave))
		rval = TS_SIG;
	else {
		sleep(chan, pri);
		if ((pp->p_flag&STIMO)==0 && seconds)
			rval = TS_TIME;
		else
			rval = TS_OK;
		untimeout(tsleepexpire, (caddr_t)pp);
	}
	pp->p_flag &= ~STIMO;
	bcopy((caddr_t)&lqsave, (caddr_t)&u.u_qsave, sizeof (label_t));
	splx(n);
	return(rval);
#endif	MACH
}

#endif	CMUCS
#if	MACH
#else	MACH
/*
 * Remove a process from its wait queue
 */
unsleep(p)
	register struct proc *p;
{
	register struct slpque *qp;
	register struct proc **hp;
	int s;

	s = splhigh();
	if (p->p_wchan) {
		hp = &(qp = &slpque[HASH(p->p_wchan)])->sq_head;
		while (*hp != p)
			hp = &(*hp)->p_link;
		*hp = p->p_link;
		if (qp->sq_tailp == &p->p_link)
			qp->sq_tailp = hp;
		p->p_wchan = 0;
	}
	splx(s);
}
#endif	MACH

/*
 * Wake up all processes sleeping on chan.
 */
wakeup(chan)
	register caddr_t chan;
{
#if	MACH
#else	MACH
	register struct slpque *qp;
	register struct proc *p, **q;
#endif	MACH
	int s;

	s = splhigh();
#if	MACH
	thread_wakeup((int) chan);
#else	MACH
	qp = &slpque[HASH(chan)];
restart:
	for (q = &qp->sq_head; p = *q; ) {
		if (p->p_rlink || p->p_stat != SSLEEP && p->p_stat != SSTOP)
			panic("wakeup");
		if (p->p_wchan==chan) {
			p->p_wchan = 0;
			*q = p->p_link;
			if (qp->sq_tailp == &p->p_link)
				qp->sq_tailp = q;
			if (p->p_stat == SSLEEP) {
				/* OPTIMIZED INLINE EXPANSION OF setrun(p) */
				if (p->p_slptime > 1)
					updatepri(p);
				p->p_slptime = 0;
				p->p_stat = SRUN;
				if (p->p_flag & SLOAD)
					setrq(p);
				/*
				 * Since curpri is a usrpri,
				 * p->p_pri is always better than curpri.
				 */
				runrun++;
				aston();
				if ((p->p_flag&SLOAD) == 0) {
					if (runout != 0) {
						runout = 0;
						wakeup((caddr_t)&runout);
					}
					wantin++;
				}
				/* END INLINE EXPANSION */
				goto restart;
			}
			p->p_slptime = 0;
		} else
			q = &p->p_link;
	}
#endif	MACH
	splx(s);
}

/*
 * Initialize the (doubly-linked) run queues
 * to be empty.
 */
rqinit()
{
	register int i;

	for (i = 0; i < NQS; i++)
		qs[i].ph_link = qs[i].ph_rlink = (struct proc *)&qs[i];
#if	MACH
	simple_lock_init(&callout_lock);
#endif	MACH
}

#if	MACH
#else	MACH
/*
 * Set the process running;
 * arrange for it to be swapped in if necessary.
 */
setrun(p)
	register struct proc *p;
{
	register int s;

	s = splhigh();
	switch (p->p_stat) {

	case 0:
	case SWAIT:
	case SRUN:
	case SZOMB:
	default:
		panic("setrun");

	case SSTOP:
	case SSLEEP:
		unsleep(p);		/* e.g. when sending signals */
		break;

	case SIDL:
		break;
	}
	if (p->p_slptime > 1)
		updatepri(p);
	p->p_stat = SRUN;
	if (p->p_flag & SLOAD)
		setrq(p);
	splx(s);
	if (p->p_pri < curpri) {
		runrun++;
		aston();
	}
	if ((p->p_flag&SLOAD) == 0) {
		if (runout != 0) {
			runout = 0;
			wakeup((caddr_t)&runout);
		}
		wantin++;
	}
}
#endif	MACH

#if	MACH
#else	MACH
/*
 * Set user priority.
 * The rescheduling flag (runrun)
 * is set if the priority is better
 * than the currently running process.
 */
setpri(pp)
	register struct proc *pp;
{
	register int p;

	p = (pp->p_cpu & 0377)/4;
	p += PUSER + 2 * pp->p_nice;
	if (pp->p_rssize > pp->p_maxrss && freemem < desfree)
		p += 2*4;	/* effectively, nice(4) */
	if (p > 127)
		p = 127;
	if (p < curpri) {
		runrun++;
		aston();
	}
	pp->p_usrpri = p;
	return (p);
}
#endif	MACH
#if	MACH
slave_start()
{
	register struct thread	*th;
	register int		mycpu;
	extern int		should_exit[];

	/*	Find a thread to execute */

	mycpu = cpu_number();

	splhigh();
	th = choose_thread();
	if (th == NULL) {
		printf("Slave %d failed to find any threads.\n", mycpu);
		printf("Should have at least found idle thread.\n");
		halt_cpu();
	}

	/*
	 *	Show that this cpu is using the kernel pmap
	 */
	PMAP_ACTIVATE(kernel_pmap, th, mycpu);

	active_threads[mycpu] = th;

	if (vm_map_pmap(th->task->map) != kernel_pmap) {
		PMAP_ACTIVATE(vm_map_pmap(th->task->map), th, mycpu);
	}

	/*
	 *	Clock interrupt requires that this cpu have an active
	 *	thread, hence it can't be done before this.
	 */
	startrtclock();
	load_context(th);
	/*NOTREACHED*/
}
#endif	MACH
