/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/* 
 * HISTORY
 *  4-May-88  David Black (dlb) at Carnegie-Mellon University
 *	Make sure p_stat is SSTOP in ptrace.
 *
 * 29-Mar-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	MACH: Don't include "sys/vm.h".
 *
 * 16-Oct-87  David Black (dlb) at Carnegie-Mellon University
 *	MACH_TT: Improved exit logic for PT_KILL.
 *
 * 13-Oct-87  David Black (dlb) at Carnegie-Mellon University
 *	MACH_TT: run_state --> user_stop_count.
 *
 * 30-Sep-87  David Black (dlb) at Carnegie-Mellon University
 *	Replaced proc lock with sig lock
 *
 * 13-Jul-87  David Black (dlb) at Carnegie-Mellon University
 *	MACH: fixed exit sequence for PT_KILL.  If delivering a
 *	thread signal, set thread's u.u_cursig.
 *
 *  2-Jul-87  David Black (dlb) at Carnegie-Mellon University
 *	multimax: added thread argument to mmax_{get,set}_ptrace_u.
 *	Changed to cpp symbols for multimax/balance.  Removed PHYSOFF
 *	definition for machines that don't use it.
 *
 * 23-May-87  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Include vm_param.h for ROMP to pick up KERNEL_STACK_SIZE.
 *
 * 20-May-87  David Golub (dbg) at Carnegie-Mellon University
 *	MACH: in ptrace(), fetch fields from U-area or top of thread's
 *	kernel stack to emulate old u_area.
 *
 * 15-May-87  David Black (dlb) at Carnegie-Mellon University
 *	MACH: set p_stat to SRUN when resuming child for sanity.
 *
 * 27-Apr-87  David Black (dlb) at Carnegie-Mellon University
 *	MACH: check task->run_state instead of thread->state.
 *
 *  8-Mar-87  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Make ptrace really work for tasks with a single thread under
 *	MACH.  What a hack.  Seems to work well enough for adb
 *	(without even recompiling it).  (Note: A bit more work will be
 *	needed for the RT when I make MACH available there).
 *
 * 04-Mar-87  Jonathan J. Chew (jjc) at Carnegie-Mellon University
 *	Commented out whole file for Sun which uses sun_sys_process.c.
 *
 *  5-Feb-87  David L. Black (dlb) at Carnegie-Mellon University
 *	MACH: Check thread/process state to determine if it is ok
 *	to ptrace.  
 *
 *  4-Feb-87  David L. Black (dlb) at Carnegie-Mellon University
 *	MACH: Changed clear_wait to task_resume because stop()
 *	does a task_hold.  This really isn't right for multiple threads,
 *	but then neither is this module.
 *
 * 31-Jan-87  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Multiple thread support (still incorrect as this module is a
 *	total abortion).
 *
 * 08-Jan-87  Robert Beck (beck) at Sequent Computer Systems, Inc.
 *	Add BALANCE case definition of ipcreg[] and NIPCREG.
 *
 * 17-Dec-86  David Golub (dbg) at Carnegie-Mellon University
 *	Removed include of pte.h, text.h for MACH.
 *
 * 30-Oct-86  David L. Black (dlb) at Carnegie-Mellon University
 *	Added Multimax changes.  Multimax uses a special interface
 *	for junk in uarea.
 *
 * 23-Oct-86  Jonathan J. Chew (jjc) at Carnegie-Mellon University
 *	Merged in changes for Sun ie. added 68000 dependent code.
 *	Also, included Sun in list of machines to define "PHYSOFF" for.
 *
 * 15-Oct-86  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Put psl.h and reg.h includes under not romp again since Multimax
 *	also wants these.
 *
 * 14-Oct-86  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Merged Multimax, RT and Vax versions.  What a mess.
 *
 *  7-Oct-86  David L. Black (dlb) at Carnegie-Mellon University
 *	Merged in Multimax changes
 *
 *  7-Sep-86  William Bolosky (bolosky) at Carnegie-Mellon University
 *	Fixed romp UAREA_ADDR macro to not use hard wired u-area address.
 *
 *  7-Jul-86  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Removed unnecessary include of pte.h
 *
 * 16-Feb-86  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Added changes to procxmt.
 *
 **********************************************************************
 */
 
#include "mach.h"
/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)sys_process.c	7.1 (Berkeley) 6/5/86
 */

#ifdef	sun
/* Use Sun version of ptrace */
#else	sun
#include "machine/reg.h"
#ifdef	romp
#include "ca/scr.h"
#include "ca/vm_param.h"		/* for KERNEL_STACK_SIZE */
#else	romp
#include "machine/psl.h"
#endif	romp
#if	MACH
#else	MACH
#include "vax/pte.h"
#endif	MACH

#include "sys/param.h"
#include "sys/systm.h"
#include "sys/dir.h"
#include "sys/user.h"
#include "sys/proc.h"
#include "sys/inode.h"
#if	MACH
#else	MACH
#include "sys/text.h"
#include "sys/seg.h"
#include "sys/vm.h"
#endif	MACH
#include "sys/buf.h"
#include "sys/acct.h"
#include "sys/ptrace.h"

#if	MACH
#include "sys/task.h"

#define	RO	(VM_PROT_READ)
#define	RW	(VM_PROT_READ|VM_PROT_WRITE)
#endif	MACH

/*
 * Priority for tracing
 */
#define	IPCPRI	PZERO

/*
 * Tracing variables.
 * Used to pass trace command from
 * parent to child being traced.
 * This data base cannot be
 * shared and is locked
 * per user.
 */
struct {
	int	ip_lock;
	int	ip_req;
	int	*ip_addr;
	int	ip_data;
} ipc;

/*
 * sys-trace system call.
 */
ptrace()
{
	register struct proc *p;
	register struct a {
		int	req;
		int	pid;
		int	*addr;
		int	data;
	} *uap;

	uap = (struct a *)u.u_ap;
	if (uap->req <= 0) {
		u.u_procp->p_flag |= STRC;
		return;
	}
	p = pfind(uap->pid);
#if	MACH
	if (p == 0 || p->task->user_stop_count == 0 ||
		p->p_stat != SSTOP || p->p_ppid != u.u_procp->p_pid ||
#else	MACH
	if (p == 0 || p->p_stat != SSTOP || p->p_ppid != u.u_procp->p_pid ||
#endif	MACH
	    !(p->p_flag & STRC)) {
		u.u_error = ESRCH;
		return;
	}
	while (ipc.ip_lock)
		sleep((caddr_t)&ipc, IPCPRI);
	ipc.ip_lock = p->p_pid;
	ipc.ip_data = uap->data;
	ipc.ip_addr = uap->addr;
	ipc.ip_req = uap->req;
	p->p_flag &= ~SWTED;
	while (ipc.ip_req > 0) {
#if	MACH
		task_resume(p->task);	/* XXX */
		p->p_stat = SRUN;	/* XXX */
#else	MACH
		if (p->p_stat==SSTOP)
			setrun(p);
#endif	MACH
		sleep((caddr_t)&ipc, IPCPRI);
	}
	u.u_r.r_val1 = ipc.ip_data;
	if (ipc.ip_req < 0)
		u.u_error = EIO;
	ipc.ip_lock = 0;
	wakeup((caddr_t)&ipc);
}

#if defined(vax)
#define	NIPCREG 16
int ipcreg[NIPCREG] =
	{R0,R1,R2,R3,R4,R5,R6,R7,R8,R9,R10,R11,AP,FP,SP,PC};
#endif
#ifdef mc68000
#define NIPCREG 17
int ipcreg[NIPCREG] =
        {R0,R1,R2,R3,R4,R5,R6,R7,AR0,AR1,AR2,AR3,AR4,AR5,AR6,AR7,PC};
#endif	mc68000
#ifdef	romp
#define NIPCREG 18	 /* allow modification of only these u area variables */
int ipcreg[NIPCREG] =
	{R0,R1,R2,R3,R4,R5,R6,R7,R8,R9,R10,R11,R12,R13,R14,R15,IAR,MQ};

#define UAREA_ADDR(o) ((int *)(ENDOFP1 - UPAGES*NBPG + (int)(o)))
#endif	romp
#ifdef	multimax
/* 
 *	Multimax does this differently; see mmax/mmax_ptrace.c
 *	and mmax/mmax_ptrace.h for details.
 */
#endif	multimax
#ifdef	balance
int	ipcreg[] = {R0,R1,R2,R3,R4,R5,R6,R7,FP,SP,PC};
#define	NIPCREG	(sizeof(ipcreg)/sizeof(ipcreg[0]))
#endif	balance
#if	defined(vax) || defined(balance)
#define	PHYSOFF(p, o) \
	((physadr)(p)+((o)/sizeof(((physadr)0)->r[0])))
#endif	defined(vax) || defined(balance)

#if	!defined(vax)
/* These are handled by inline on the vax */
extern char fubyte(), fuibyte();
extern int fuword(), fuiword();
extern int subyte(), suibyte(), suword(), suiword();
#endif	!defined(vax)
/*
 * Code that the child process
 * executes to implement the command
 * of the parent process in tracing.
 */
procxmt()
{
	register int i;
	register *p;
#if	MACH
	struct user	fake_uarea;
#else	MACH
	register struct text *xp;
#endif	MACH

	if (ipc.ip_lock != u.u_procp->p_pid)
		return (0);
	u.u_procp->p_slptime = 0;
	i = ipc.ip_req;
	ipc.ip_req = 0;
	switch (i) {

	case PT_READ_I:			/* read the child's text space */
		if (!useracc((caddr_t)ipc.ip_addr, 4, B_READ))
			goto error;
#ifdef	romp
		ipc.ip_data = getubits((caddr_t)ipc.ip_addr, fuiword, fuibyte);
#else	romp
		ipc.ip_data = fuiword((caddr_t)ipc.ip_addr);
#endif	romp
		break;

	case PT_READ_D:			/* read the child's data space */
		if (!useracc((caddr_t)ipc.ip_addr, 4, B_READ))
			goto error;
#ifdef	romp
		ipc.ip_data = getubits((caddr_t)ipc.ip_addr, fuword, fubyte);
#else	romp
		ipc.ip_data = fuword((caddr_t)ipc.ip_addr);
#endif	romp
		break;

	case PT_READ_U:			/* read the child's u. */
#ifdef	multimax
		if (mmax_get_ptrace_u(ipc.ip_addr, &ipc.ip_data,
		    current_thread()) != 0)
			goto error;
#else	multimax
		i = (int)ipc.ip_addr;
		if (i<0 || i >= ctob(UPAGES))
			goto error;
#ifdef	romp
		/*
		 *	RT can only fetch word-aligned data
		 */
		if (i & (sizeof(int)-1))
			goto error;
#if	MACH
		i -= (ctob(UPAGES) - sizeof(struct user));
		if (i >= 0) {
		    /*
		     *	We want data from the U area.  Fake it up,
		     *	then pull out the desired int.
		     */
		    bzero((caddr_t)&fake_uarea, sizeof(struct user));
		    fake_u(&fake_uarea, current_thread());
		    ipc.ip_data = *(int *)((caddr_t)&fake_uarea + i);
		}
		else {
		    /*
		     *	Assume we want data from the kernel stack -
		     *	most likely the user's registers.
		     *
		     *	i is (negative) offset below base of kernel
		     *	stack
		     */
		    ipc.ip_data = *(int *)(
		    	(caddr_t)current_thread()->kernel_stack
			+ KERNEL_STACK_SIZE
			+ i);
		}
#else	MACH
		ipc.ip_data = *UAREA_ADDR(i);
#endif	MACH
#else	romp
#if	MACH
		if (i < sizeof(struct user)) {
		    /*
		     *	We want data from the U area.  Fake it up,
		     *	then pull out the desired int.
		     */
		    bzero((caddr_t)&fake_uarea, sizeof(struct user));
		    fake_u(&fake_uarea, current_thread());
		    ipc.ip_data = *(int *)PHYSOFF(&fake_uarea, i);
		}
		else {
		    /*
		     *	Assume we want data from the kernel stack, most
		     *	likely the user's registers.
		     */
		    ipc.ip_data = *(int *)
		    	PHYSOFF(current_thread()->kernel_stack
				+ (KERNEL_STACK_SIZE - ctob(UPAGES)), i);
		}
#else	MACH
		ipc.ip_data = *(int *)PHYSOFF(&u, i);
#endif	MACH
#endif	multimax
#endif	romp
		break;

	case PT_WRITE_I:		/* write the child's text space */
		/*
		 * If text, must assure exclusive use
		 */
#if	MACH
#else	MACH
		if (xp = u.u_procp->p_textp) {
			if (xp->x_count!=1 || xp->x_iptr->i_mode&ISVTX)
				goto error;
			xp->x_flag |= XTRC;
		}
#endif	MACH
#ifdef	romp
		/* Must set up to allow	writing */
		if ((i = setubits((caddr_t)ipc.ip_addr, ipc.ip_data, suiword, suibyte ))) {
			if( chgprot( (caddr_t)ipc.ip_addr, RW ) && chgprot( (caddr_t)ipc.ip_addr+3, RW ))
				i = setubits((caddr_t)ipc.ip_addr, ipc.ip_data, suiword, suibyte);
			(void) chgprot((caddr_t)ipc.ip_addr, RO);
			(void) chgprot((caddr_t)ipc.ip_addr+3, RO);
#else	romp
		i = -1;
		if ((i = suiword((caddr_t)ipc.ip_addr, ipc.ip_data)) < 0) {
			if (chgprot((caddr_t)ipc.ip_addr, RW) &&
			    chgprot((caddr_t)ipc.ip_addr+(sizeof(int)-1), RW))
				i = suiword((caddr_t)ipc.ip_addr, ipc.ip_data);
			(void) chgprot((caddr_t)ipc.ip_addr, RO);
			(void) chgprot((caddr_t)ipc.ip_addr+(sizeof(int)-1), RO);
#endif	romp
		}
		if (i < 0)
			goto error;
#if	MACH
#else	MACH
		if (xp)
			xp->x_flag |= XWRIT;
#endif	MACH
		break;

	case PT_WRITE_D:		/* write the child's data space */
#ifdef	romp
		if (setubits((caddr_t)ipc.ip_addr, 0, suword, subyte))
			goto error;
		(void) setubits((caddr_t)ipc.ip_addr, ipc.ip_data, suword, subyte);
		break;
#else	romp
		if (suword((caddr_t)ipc.ip_addr, 0) < 0)
			goto error;
		(void) suword((caddr_t)ipc.ip_addr, ipc.ip_data);
		break;
#endif	romp

	case PT_WRITE_U:		/* write the child's u. */
#ifdef	multimax
		if (mmax_set_ptrace_u(ipc.ip_addr, ipc.ip_data,
		    current_thread()) != 0)
			goto error;
		break;
#else	multimax
#ifdef	romp
		i = (int)(caddr_t)ipc.ip_addr;
#if	MACH
		i -= (ctob(UPAGES) - sizeof(struct user));
		p = (int *)(
		    	(caddr_t)current_thread()->kernel_stack
			+ KERNEL_STACK_SIZE
			+ i);
#else	MACH
		p = UAREA_ADDR(i);
#endif	MACH
#else	romp
		i = (int)ipc.ip_addr;
#if	MACH
		/*
		 *	Write one of the user's registers.
		 *	Convert the offset (in old-style Uarea/kernel stack)
		 *	into the corresponding offset into the saved
		 *	register set.
		 */
		p = (int *)PHYSOFF(current_thread()->kernel_stack
				+ (KERNEL_STACK_SIZE - ctob(UPAGES)), i);
#else	MACH
		p = (int *)PHYSOFF(&u, i);
#endif	MACH
#endif	romp
		for (i=0; i<NIPCREG; i++)
			if (p == &u.u_ar0[ipcreg[i]])
				goto ok;
#ifdef	romp
		if (p == &u.u_ar0[ICSCS]) {
			ipc.ip_data |= ICSCS_USERSET;
			ipc.ip_data &= ~ICSCS_USERCLR;
#else	romp
		if (p == &u.u_ar0[PS]) {
			ipc.ip_data |= PSL_USERSET;
			ipc.ip_data &=  ~PSL_USERCLR;
#endif	romp
			goto ok;
		}
		goto error;

	ok:
		*p = ipc.ip_data;
		break;
#endif	multimax
	case PT_STEP:			/* single step the child */
	case PT_CONTINUE:		/* continue the child */
		if ((int)ipc.ip_addr != 1)
#ifdef	romp
			u.u_ar0[IAR] = (int)ipc.ip_addr;
#else	romp
			u.u_ar0[PC] = (int)ipc.ip_addr;
#endif	romp
		if ((unsigned)ipc.ip_data > NSIG)
			goto error;
#if	MACH
		if (sigmask(u.u_procp->p_cursig) & threadmask)
			u.u_cursig = 0;
#endif	MACH
		u.u_procp->p_cursig = ipc.ip_data;	/* see issig */
#if	MACH
		if (sigmask(ipc.ip_data) & threadmask)
		    u.u_cursig = ipc.ip_data;
#endif	MACH
		if (i == PT_STEP) 
#ifdef	romp
			u.u_ar0[ICSCS] |= ICSCS_INSTSTEP;  /* see locore */
#else	romp
			u.u_ar0[PS] |= PSL_T;
#endif	romp
		wakeup((caddr_t)&ipc);
		return (1);

	case PT_KILL:			/* kill the child process */
		wakeup((caddr_t)&ipc);
#if	MACH
		/*
		 *	Tell other threads that we're about to exit.
		 */
		sig_lock_to_exit(u.u_procp);
#endif	MACH
		exit(u.u_procp->p_cursig);

	default:
	error:
		ipc.ip_req = -1;
	}
	wakeup((caddr_t)&ipc);
	return (0);
}
#ifdef	romp

getubits(cp, wsubr, bsubr )
	register char *cp;
	register int (*wsubr)();
	register char (*bsubr)();
{
	register i, fourbytes;

	if ((int)cp & 3 ) {
		for (i=0; i<4; ++i) {
			fourbytes <<= 8;
			fourbytes |= (*bsubr)(cp+i);
		}
		return (fourbytes);
	}
	return ((*wsubr)(cp));
}

setubits(cp, bits, wsubr, bsubr)
	register char *cp;
	register int bits;
	register int (*wsubr)();
	register int (*bsubr)();
{
	register i, r;

	if ((int)cp & 3) {
		for (i=3; i>=0; --i) {
			if (r = (*bsubr)(cp+i, bits & 0xFF))
				return (r);
			bits >>= 8;
		}
		return (r);
	}
	return((*wsubr)(cp, bits));
}
#endif	romp
#endif	sun
