/* 
 **********************************************************************
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 **********************************************************************
 * HISTORY
 * 01-Oct-86  Mike Accetta (mja) at Carnegie-Mellon University
 *	Added interim hack to omit specific bad blocks from swap map at
 *	initialization time.  This should really be handled by bad block
 *	revectoring in the disk drivers but since that doesn't really seem to
 *	work right at the moment and MACH is "... just around the corner ..."
 *	we'll do it the simple way for now.
 *
 * 27-Sep-86  Mike Accetta (mja) at Carnegie-Mellon University
 *	Added include of "generic.h"
 *
 **********************************************************************
 */ 
 
#include "generic.h"

#include "cs_generic.h"
#include "mach.h"
/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)vm_sw.c	7.1 (Berkeley) 6/5/86
 */

#if	MACH
#else	MACH
#include "param.h"
#include "systm.h"
#include "buf.h"
#include "conf.h"
#include "dir.h"
#include "user.h"
#include "inode.h"
#include "map.h"
#include "uio.h"
#include "file.h"

struct	buf rswbuf;
/*
 * Indirect driver for multi-controller paging.
 */
swstrategy(bp)
	register struct buf *bp;
{
	int sz, off, seg;
	dev_t dev;

#ifdef GENERIC
	/*
	 * A mini-root gets copied into the front of the swap
	 * and we run over top of the swap area just long
	 * enough for us to do a mkfs and restor of the real
	 * root (sure beats rewriting standalone restor).
	 */
#define	MINIROOTSIZE	4096
	if (rootdev == dumpdev)
		bp->b_blkno += MINIROOTSIZE;
#endif
	sz = howmany(bp->b_bcount, DEV_BSIZE);
	if (bp->b_blkno+sz > nswap) {
		bp->b_flags |= B_ERROR;
		iodone(bp);
		return;
	}
	if (nswdev > 1) {
		off = bp->b_blkno % dmmax;
		if (off+sz > dmmax) {
			bp->b_flags |= B_ERROR;
			iodone(bp);
			return;
		}
		seg = bp->b_blkno / dmmax;
		dev = swdevt[seg % nswdev].sw_dev;
		seg /= nswdev;
		bp->b_blkno = seg*dmmax + off;
	} else
		dev = swdevt[0].sw_dev;
	bp->b_dev = dev;
	if (dev == 0)
		panic("swstrategy");
	(*bdevsw[major(dev)].d_strategy)(bp);
}

swread(dev, uio)
	dev_t dev;
	struct uio *uio;
{

	return (physio(swstrategy, &rswbuf, dev, B_READ, minphys, uio));
}

swwrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{

	return (physio(swstrategy, &rswbuf, dev, B_WRITE, minphys, uio));
}

/*
 * System call swapon(name) enables swapping on device name,
 * which must be in the swdevsw.  Return EBUSY
 * if already swapping on this device.
 */
swapon()
{
	struct a {
		char	*name;
	} *uap = (struct a *)u.u_ap;
	register struct inode *ip;
	dev_t dev;
	register struct swdevt *sp;
	register struct nameidata *ndp = &u.u_nd;

	if (!suser())
		return;
	ndp->ni_nameiop = LOOKUP | FOLLOW;
	ndp->ni_segflg = UIO_USERSPACE;
	ndp->ni_dirp = uap->name;
	ip = namei(ndp);
	if (ip == NULL)
		return;
	if ((ip->i_mode&IFMT) != IFBLK) {
		u.u_error = ENOTBLK;
		iput(ip);
		return;
	}
	dev = (dev_t)ip->i_rdev;
	iput(ip);
	if (major(dev) >= nblkdev) {
		u.u_error = ENXIO;
		return;
	}
	for (sp = &swdevt[0]; sp->sw_dev; sp++)
		if (sp->sw_dev == dev) {
			if (sp->sw_freed) {
				u.u_error = EBUSY;
				return;
			}
			swfree(sp - swdevt);
			return;
		}
	u.u_error = EINVAL;
}

#if	CS_GENERIC
/* 
 *  Indirect the freeing of parts of the swap area through here.  This routine
 *  checks the various pieces against a fixed list of bad sectors known to
 *  exist in the associated disk partitions and doesn't free the parts
 *  of the swap map corresponding to those areas.  This prevents the system
 *  from trying to use them which almost always results in a process killed
 *  on swap error diagnostics and disgruntled users.
 *
 *  I don't dispute that this is a hack.
 */
 
struct swbadt
{
    dev_t	swb_dev;
    swblk_t	swb_bad;
} swbadt[20] = {0};
 
int swbaddebug=0;
 
swfree_bad(index, freeit, dvbase, map, blk, vsbase, name, size)
    int (*freeit)();
    register swblk_t dvbase;
    struct map *map;
    register long blk;
    register swblk_t vsbase;
    char *name;
{
    register struct swbadt *swb;

    if (swbaddebug)
	printf("%d:%d %D %D\n", index, blk, vsbase, dvbase);
    for (swb=swbadt; swb < &swbadt[sizeof(swbadt)/sizeof(swbadt[0])]; swb++)
    {
	swblk_t bad;

	if (swb->swb_dev == 0)
	    break;
	if (swb->swb_dev != swdevt[index].sw_dev)
	    continue;
	bad = clbase(swb->swb_bad);
	if (bad < dvbase || bad >= (dvbase+blk))
	    continue;
	printf("[ disabled bad swap sectors %D-%D on device %d,%d ]\n",
	       bad, bad+CLSIZE-1, major(swb->swb_dev), minor(swb->swb_dev));
	if (bad != dvbase)
	    (*freeit)(map, bad-dvbase, vsbase, name, size);
	bad += CLSIZE;
	if (bad != (dvbase+blk))
	    swfree_bad(index, freeit, bad, map, blk-(bad-dvbase), vsbase+(bad-dvbase), name, size);
	return;
    }
    (*freeit)(map, blk, vsbase, name, size);
}

swrminit_bad(map, blk, vsbase, name, size)
    struct map *map;
    long blk;
    swblk_t vsbase;
    char *name;
{
    rminit(map, blk, vsbase, name, size);
}

swrmfree_bad(map, blk, vsbase)
    struct map *map;
    long blk;
    swblk_t vsbase;
{
    rmfree(map, blk, vsbase);
}

#define	rminit(map, blk, vsbase, name, size) \
     swfree_bad(index, swrminit_bad, dvbase, map, blk, vsbase, name, size)
#define	rmfree(map, blk, vsbase) \
     swfree_bad(index, swrmfree_bad, dvbase, map, blk, vsbase)

#endif	CS_GENERIC
/*
 * Swfree(index) frees the index'th portion of the swap map.
 * Each of the nswdev devices provides 1/nswdev'th of the swap
 * space, which is laid out with blocks of dmmax pages circularly
 * among the devices.
 */
swfree(index)
	int index;
{
	register swblk_t vsbase;
	register long blk;
	dev_t dev;
	register swblk_t dvbase;
	register int nblks;

	dev = swdevt[index].sw_dev;
	(*bdevsw[major(dev)].d_open)(dev, FREAD|FWRITE);
	swdevt[index].sw_freed = 1;
	nblks = swdevt[index].sw_nblks;
	for (dvbase = 0; dvbase < nblks; dvbase += dmmax) {
		blk = nblks - dvbase;
		if ((vsbase = index*dmmax + dvbase*nswdev) >= nswap)
			panic("swfree");
		if (blk > dmmax)
			blk = dmmax;
		if (vsbase == 0) {
			/*
			 * Can't free a block starting at 0 in the swapmap
			 * but need some space for argmap so use 1/2 this
			 * hunk which needs special treatment anyways.
			 */
			argdev = swdevt[0].sw_dev;
			rminit(argmap, (long)(blk/2-ctod(CLSIZE)),
			    (long)ctod(CLSIZE), "argmap", ARGMAPSIZE);
			/*
			 * First of all chunks... initialize the swapmap
			 * the second half of the hunk.
			 */
			rminit(swapmap, (long)blk/2, (long)blk/2,
			    "swap", nswapmap);
		} else
			rmfree(swapmap, blk, vsbase);
	}
}
#endif	MACH
