/* Mach Operating System
 * Copyright (c) 1988 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 *	File:	ca/cache.h
 *
 *	RT Version:
 *
 *	Shared memory alias cache header routines.
 *
 *	Author:	Richard N. Sanzi Jr.
 *
 * HISTORY:
 * $Log:	cache.c,v $
 * Revision 2.4  88/12/19  02:35:04  mwyoung
 * 	Remove lint.
 * 	[88/12/18            mwyoung]
 * 	
 * 	Fix fencepost error in cache_remove_segment, from Rich Sanzi.
 * 	[88/12/08            mwyoung]
 * 	
 * 	Corrected include file references.
 * 	[88/11/22            mwyoung]
 * 	
 * 	Added cache_verify_free().
 * 	[88/11/13  19:16:32  mwyoung]
 * 
 * Revision 2.3  88/10/06  13:29:06  sanzi
 * 	Complete reworking of cache handling.  Added cache_entries list for
 * 	fast removal from cache.  
 * 
 * Revision 2.2.1.4  88/10/03  09:31:21  sanzi
 * 	Fix units in call to pmap_protect_phys_page().
 * 
 * Revision 2.2.1.3  88/08/30  15:14:18  sanzi
 * 	Turn on cache_fault in common case.
 * 
 * 22-Jan-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Created.
 *
 */

#include <vm/pmap.h>
#include <sys/queue.h>
#include <romp_debug.h>
#include <ca/rosetta.h>
#include <ca/cache.h>
#include <ca/pmap.h>

int cache_free_num 	= 0;
int cache_debug 	= 0;
int cache_initialized 	= 0;
int default_cache_size 	= 65536;
int cache_max_chain_length = 2;
extern	int mi_page_mask;		/* machine independent page mask */
extern	int ppages_per_vmpage;

struct cache_entry cache_free_list_header;
struct cache_entry *cache_free;

extern struct pmap_list_entry *pmap_list_entries; /* XXX should be in pmap.h */
#if	CACHE_STATISTICS
int cache_collisions 	= 0;
int cache_chain_length 	= 0;
int cache_reclaims 	= 0;
int cache_free_list_hit	= 0;
int cache_inserts	= 0;
int cache_redundant	= 0;
#define	cache_event(event)	(event++)
#else	CACHE_STATISTICS
#define	cache_event(event)	
#endif	CACHE_STATISTICS

#define	CACHE_ENTER	0x01
#define	CACHE_LOOKUP	0x02
#define	CACHE_REMOVE	0x04

#define	LOOP_CHECK(sid,vpage)

#if	ROMP_DEBUG
#define	CACHE_DEBUG(x,stmt)	if (x&cache_debug) printf stmt
#else	ROMP_DEBUG
#define	CACHE_DEBUG(x,stmt)
#endif	ROMP_DEBUG

/*
 *	Clear all the next/prev pointers for the headers array.
 *	Set up a free list linked through the next field of the
 *	alias array.
 */

cache_init(load_start, avail_end)
vm_offset_t load_start, *avail_end;
{
    int i;

    int size;
    extern int rose_page_shift;
    vm_offset_t ending_addr = *avail_end;
    
    if (default_cache_size == 0) {
	default_cache_size = PAGE_SIZE;
    }

    size = default_cache_size;
    
    CACHE_DEBUG(0x01,("cache_init() start.\n"));

    cache_size = (size / 2)  / sizeof(struct cache_entry);

    ending_addr -= size;
    
    headers = (struct cache_entry *) (ending_addr + load_start);
    *avail_end = ending_addr;
    alias = headers + cache_size;

    cache_free = &cache_free_list_header;
    cache_free->next = &alias[0];
    
    for (i=0;i<cache_size;i++) {
	headers[i].next = CACHE_NULL;
	headers[i].prev = CACHE_NULL;
	alias[i].prot = ENTRY_FREE;
	
	if (i < cache_size-1) alias[i].next = &alias[i+1];
	else	alias[i].next = CACHE_NULL;
	
	alias[i].prev = CACHE_NULL;
    }

    mi_page_mask = (~ (PAGE_SIZE - 1)) >> rose_page_shift ;
    cache_initialized = 1;
    CACHE_DEBUG(0x01,("cache_init() end.\n"));
}

cache_free_entry(ent)
struct cache_entry *ent;
{
    queue_head_t *list;
    pmap_t pmap = ent->pmap;    
    /*
     * Sanity check in case something just happens to match
     *	either cache_remove_page''s test for phys (implies
     *	that we''re removing ppage 0 which would be
     *	bad) or cache_remove_range''s test for sid and vpage (implies that
     *	some ppage has a mapping whose sid==0 and vpage==0, which is ok.)
     */
    if (ent->prot & ENTRY_FREE) {
	return(FALSE);
    }
    if (ent->pmap == PMAP_NULL) panic("cache: free bonk pmap");
    /*
     * Remove the entry from the owner''s pmap list,
     * and if the owner doesn''t currently own the mapping in the
     * pagetable, then remove it from the HATIPT.
     */
#define	mmu_to_vm_page(addr)	\
	((addr) >> ( PAGE_SHIFT - rose_page_shift))
	
#define	mmu_extract_addrtag(ppage) \
	(RTA_HATIPT[(ppage)].key_addrtag & RTA_ADDRTAG_MASK)
	
    if ((ent->pmap != pmap_list_entries[mmu_to_vm_page(ent->ppage)].pmap) &&
        (mmu_extract_addrtag(ent->ppage) == ent->addrtag)) {
    	mapout(ent->ppage, ppages_per_vmpage);
	tlb_invalidate_all();
    }
    
    list = &(ent->pmap->cache_entries);
    queue_remove(list,ent,cache_entry_t, cache_entries);
    pmap->cache_count--;    

    if (ent->next != CACHE_NULL) ent->next->prev = ent->prev;
    if (ent->prev != CACHE_NULL) ent->prev->next = ent->next;
	else panic("cache: ENTRY_FREE: previous entry NULL!");
    /*
     * Link entry onto free list.
     */
    ent->pmap = PMAP_NULL;
    ent->prot = ENTRY_FREE;
    ent->ppage = 0;
    ent->addrtag = 0;
    ent->next = cache_free->next;
    cache_free->next = ent;
    return(TRUE);
}

/*
 * Flush an entry, and return its index.
 *
 *	Try the free list first, and if that fails, resort to
 *	stealing one using a fifo reclaim algorithm.
 */

struct cache_entry *cache_find_free()
{
    struct cache_entry *ent;
    int i;

    /*
     * Check the free list.
     */

    if (cache_free->next != CACHE_NULL) {
	cache_event(cache_free_list_hit);
	ent = cache_free->next;
	cache_free->next = ent->next;
	return(ent);
    }
    cache_event(cache_reclaims);

    i = cache_free_num++;
    ent = &alias[i];
    
    if (cache_free_num == cache_size) cache_free_num = 0;
    cache_free_entry(ent);

    /*
     * pull it from the free list
     *
     */
    cache_free->next = cache_free->next->next;
    
    return(ent);
}

/*
 *	This does a cache lookup for the sid,vpage combination,
 * 	and returns the entry.
 */

struct cache_entry *cache_lookup_entry(sid,vpage)
vm_offset_t sid, vpage;
{
#if	USE_ASSEMBLY
    register struct cache_entry *ent;
    struct cache_entry *cache_lookup_entry_();

    CACHE_DEBUG(CACHE_LOOKUP,
    	("cache_lookup(%x %x)",sid,vpage));
	
    ent = cache_lookup_entry_(sid,vpage); /* call assembly version */
    
    CACHE_DEBUG(CACHE_LOOKUP,
        ("returns %x.\n",ent));
    return(ent);
#else	USE_ASSEMBLY
    register int index = (sid ^ vpage) & (cache_size-1);
    register struct cache_entry *ent = headers[index].next;
    register vm_offset_t addrtag = sid << RTA_VPAGE_BITS | vpage;
    register int chain_length = 0;

    CACHE_DEBUG(CACHE_LOOKUP,
    	("cache_lookup(%x %x)",sid,vpage));
    while (ent != CACHE_NULL) {
	cache_event(chain_length);
	if (ent->addrtag == addrtag) {
	    	CACHE_DEBUG(CACHE_LOOKUP,
			("returns %x.\n",ent));
#if	CACHE_STATISTICS
		if (chain_length > cache_chain_length)
			cache_chain_length = chain_length;
#endif	CACHE_STATISTICS	
		return(ent);
	}
	ent = ent->next;
    }
    CACHE_DEBUG(CACHE_LOOKUP,
        ("returns %x.\n",0));
#if	CACHE_STATISTICS
		if (chain_length > cache_chain_length)
			cache_chain_length = chain_length;
#endif	CACHE_STATISTICS	
    return((struct cache_entry *) 0);
#endif USE_ASSEMBLY

}

/*
 * THis is no longer in use.
 */
vm_offset_t
cache_lookup(sid,vpage,prot,ppage)
vm_offset_t sid, vpage, prot, *ppage;
{
    struct cache_entry *ent;
    if (! cache_initialized) return(FALSE);
    ent = cache_lookup_entry(sid,vpage);

    if ((ent != (struct cache_entry *) 0) &&
        (ent->prot == prot)) {
	*ppage = ent->ppage;
	return(TRUE);
    } 
    *ppage = 0;
    return(TRUE);
}

cache_enter(pmap,sid,vpage,ppage,prot)
pmap_t pmap;
vm_offset_t sid, vpage, ppage, prot;
{
    struct cache_entry *new,*ent,*victim;
    int index;
    struct cache_entry *header;
    int i;
    
    if (! cache_initialized) return(FALSE);

    cache_event(cache_inserts);

    CACHE_DEBUG(CACHE_ENTER,
    	("cache_enter(%x %x %x %x)\n",sid,vpage,ppage,prot));

    if ((ent = cache_lookup_entry(sid,vpage)) != CACHE_NULL ) {
#if	CACHE_STATISTICS
	if ((ent->prot == prot) && (ent->ppage == ppage))
		cache_event(cache_redundant);
#endif	CACHE_STATISTICS
	ent->prot = prot;
	ent->ppage = ppage;
	return(TRUE);
    }

    /*
     * Otherwise ...
     * 
     * Get a free entry.
     */
    new = cache_find_free();
    /*
     * hash to find the header chain.
     */
    index = (sid ^ vpage) & (cache_size-1);
    header = &headers[index];

    if (! (new->prot & ENTRY_FREE)) panic("cache: cache_find_free");

    new->addrtag = sid << RTA_VPAGE_BITS | vpage;
    new->ppage = ppage;
    new->prot = prot & ~ENTRY_FREE;

    /*
     * insert the entry at the front of the list.
     */
    new->next = header->next;    
    if (new->next != CACHE_NULL) new->next->prev = new;
    header->next = new;
    new->prev = header;
    new->pmap = pmap;

    /*
     * Remember this entry in our pmap cache entry list.
     */
    queue_enter(&pmap->cache_entries,new,cache_entry_t,cache_entries);
    pmap->cache_count++;

    /*
     * Keep the hash chains short (ie 17 is a little too long)
     */
    victim = new->next;
    i = cache_max_chain_length - 1 ;
    while ( victim != CACHE_NULL ) {
	if ( i-- == 0 ) {
	    cache_event(cache_collisions);
	    cache_free_entry(victim);
	    break;
	}	
	victim = victim->next;
    }

    return(TRUE);
}

cache_remove(pmap,sid,vpage)
	pmap_t pmap;
	vm_offset_t sid, vpage;
{
    struct cache_entry *ent;

#ifdef	lint
	pmap++;
#endif	lint

    if (! cache_initialized) return(FALSE);
    CACHE_DEBUG(CACHE_LOOKUP,("cache_remove(%x %x)\n",sid,vpage));
    ent = cache_lookup_entry(sid,vpage);
    if (ent != CACHE_NULL) {
	cache_free_entry(ent);	
	return(TRUE);
    }
    LOOP_CHECK(sid,vpage);        
    return(FALSE);
}
#define	ROSETTA_SEGMENT_SIZE 0x10000000

/*
 * Step through the range in units of segments and remove
 * all known cache entries for the region.
 *
 */

cache_remove_range(pmap,start,end)
pmap_t pmap;
vm_offset_t start,end;
{

    	vm_offset_t s,e;

	s = atoseg(start);	
	e = atoseg(end);
	
	/*
	 * Remove from start -> to end if less than segment.
	 */
	if (s == e) {
	    cache_remove_segment(pmap,pmap->sidtab[s],
	    		a_to_rosetta_page(segoffset(start)),
			a_to_rosetta_page(segoffset(end)));
	    return;
	}
	/*
	 * Removing more than one segment:
	 *	Remove from start -> atoseg(start) + 1
	 */

	cache_remove_segment(pmap,pmap->sidtab[s],
			a_to_rosetta_page(segoffset(start)),
			a_to_rosetta_page(0xfffffff)); /* XXX */

	/*
	 * Remove all whole segments from (atoseg(start) + 1) -> atoseg(end)
	 */
	
	for (s += 1; s < e; s += atoseg(ROSETTA_SEGMENT_SIZE) ) {
	    cache_remove_segment(pmap,
	    			pmap->sidtab[s],
				(vm_offset_t) 0,
				a_to_rosetta_page(0xfffffff)); /* XXX */
	}

	if (segtoa(e) != end) {
	    /*
	     * Remove from atoseg(end) to end.
	     */
	    cache_remove_segment(pmap,
	    			pmap->sidtab[e],
				(vm_offset_t) 0,
				a_to_rosetta_page(segoffset(end)));
	}
}

cache_remove_segment(pmap,sid,spage,epage)
pmap_t pmap;
vm_offset_t sid,spage,epage;
{
    struct cache_entry *ent, *next_ent;
    queue_head_t *list;
    register vm_offset_t entry_sid, entry_vpage;    

    CACHE_DEBUG(CACHE_LOOKUP,
    	("cache_rremove(%x %x %x %x)\n",pmap,sid,spage,epage));

    list = &pmap->cache_entries;
    ent = (cache_entry_t) queue_first(list);
    while (!queue_end(list,(queue_entry_t) ent)) {
	entry_sid = ent->addrtag >> RTA_VPAGE_BITS;
	entry_vpage = ent->addrtag & (RTA_VPAGE_MASK);
	next_ent = (cache_entry_t) queue_next(&ent->cache_entries);
	if ((entry_sid == sid)
		&& (entry_vpage >= spage)
		&& (entry_vpage <= epage)) {
	    cache_free_entry(ent);		
	}
	ent = next_ent;
    }
}

/*
 * Step through the range in units of segments and
 * reprotect all known cache entries for the region.
 *
 */

cache_protect_range(pmap,start,end,mmu_prot)
	pmap_t pmap;
	vm_offset_t start,end;
	int mmu_prot;
{
#ifdef	lint
	mmu_prot++;
#endif	lint

cache_remove_range(pmap,start,end);

#ifdef	notyet
    	vm_offset_t s,e;

	s = atoseg(start);	
	e = atoseg(end);
	
	/*
	 * Remove from start -> to end if less than segment.
	 */
	if (s == e) {
	    cache_protect_segment(pmap,pmap->sidtab[s],
	    		a_to_rosetta_page(segoffset(start)),
			a_to_rosetta_page(segoffset(end)),
			mmu_prot);
	    return;
	}
	/*
	 * Removing more than one segment:
	 *	Remove from start -> atoseg(start) + 1
	 */

	cache_protect_segment(pmap,pmap->sidtab[s],
			a_to_rosetta_page(segoffset(start)),
			a_to_rosetta_page(0xfffffff), /* XXX */
			mmu_prot); 

	/*
	 * Remove all whole segments from (atoseg(start) + 1) -> atoseg(end)
	 */
	
	for (s += 1; s < e; s += atoseg(ROSETTA_SEGMENT_SIZE) ) {
	    cache_protect_segment(pmap,
	    			pmap->sidtab[s],
				(vm_offset_t) 0,
				a_to_rosetta_page(0xfffffff), /* XXX */
				mmu_prot);
	}

	if (segtoa(e) != end) {
	    /*
	     * Remove from atoseg(end) to end.
	     */
	    cache_protect_segment(pmap,
	    			pmap->sidtab[e],
				(vm_offset_t) 0,
				a_to_rosetta_page(segoffset(end)),
				mmu_prot);				
	}
#endif	notyet	
}

cache_protect_segment(pmap,sid,spage,epage,mmu_prot)
pmap_t pmap;
vm_offset_t sid,spage,epage;
int mmu_prot;
{
    struct cache_entry *ent, *next_ent;
    queue_head_t *list;
    register vm_offset_t entry_sid, entry_vpage;    
    CACHE_DEBUG(CACHE_LOOKUP,
    	("cache_rprotect(%x %x %x %x)\n",pmap,sid,spage,epage));

    list = &pmap->cache_entries;
    ent = (cache_entry_t) queue_first(list);
    while (!queue_end(list,(queue_entry_t) ent)) {
	entry_sid = ent->addrtag >> RTA_VPAGE_BITS;
	entry_vpage = ent->addrtag & (RTA_VPAGE_MASK);
	next_ent = (cache_entry_t) queue_next(&ent->cache_entries);
	if ((entry_sid==sid) && (entry_vpage>=spage) && (entry_vpage < epage)){
	    ent->prot = mmu_prot;
	    /*
	     * we reprotect all pages, even though we could be lazy
	     * and only protect the writeable pages when we are
	     * downgrading from write to read.
	     */
	    if (mmu_extract_addrtag(ent->ppage) == ent->addrtag) {
		pmap_protect_phys_page(ent->ppage << rose_page_shift,
					mmu_prot);		
	    }
	}
	ent = next_ent;
    }
}

/*
 *	Remove all references to a page in the cache.
 *
 * 	For now, we must scan the whole page table, but
 *	we could maintain pv-lists in the pmap module.
 */
cache_remove_page(addr)
vm_offset_t addr;
{
    int i;
    vm_offset_t phys = a_to_rosetta_page(segoffset(addr));
    struct cache_entry *ent;

    CACHE_DEBUG(CACHE_LOOKUP,
    	("cache_premove(%x)\n",addr));

    for (i=0;i<cache_size;i++) {
	ent = &alias[i];
	if (ent->ppage == phys) {
	    cache_free_entry(ent);
	    LOOP_CHECK(ent->addrtag>>RTA_VPAGE_BITS,
	    	ent->addrtag&RTA_VPAGE_MASK);
	}
    }
}

boolean_t cache_verify_free(addr)
vm_offset_t addr;
{
    int i;
    vm_offset_t phys = a_to_rosetta_page(segoffset(addr));
    struct cache_entry *ent;

    if (!cache_initialized)
	return(TRUE);

    for (i=0;i<cache_size;i++) {
	ent = &alias[i];
	if (ent->ppage == phys) {
	    if (ent->prot & ENTRY_FREE)
		printf("cache_verify_free: marked free, but has page address");
	    return(FALSE);
	}
    }
    return(TRUE);
}

/*
 *	cache_loop_check(sid,vpage)
 *
 *	check for loops in the cache hash chains.
 */
cache_loop_check(sid,vpage)
vm_offset_t sid, vpage;
{
    register int index = (sid ^ vpage) & (cache_size-1);
    register struct cache_entry *ent = headers[index].next;
    register struct cache_entry *trailer = ent;
    int toggle = 0;

    while (ent != CACHE_NULL) {
	ent = ent->next;	
	if (ent == trailer) Debugger("Loop in cache!");	
	if (toggle) {
	    trailer = trailer->next;
	}
        toggle = !toggle;
    }
    
}

#include <vm/vm_prot.h>
int	try_c_cache = 0;
boolean_t 
cache_fault(pmap,virt,fault_type)
pmap_t pmap;
vm_offset_t virt;
int fault_type;
{
    struct cache_entry *ent;
    int prot;
    extern int lo_tries, lo_hits;
    int orig_sid, orig_vpage;

    if (!try_c_cache) return(FALSE);
    CACHE_DEBUG(CACHE_LOOKUP,
    	("cache_fault(%x %x %x)\n",pmap,virt,fault_type));

    lo_tries ++;
    
    ent = cache_lookup_entry(orig_sid = pmap->sidtab[atoseg(virt)],
    			orig_vpage = a_to_rosetta_page(segoffset(virt)));

    if (fault_type & VM_PROT_WRITE) prot = 2;
    else prot = 3;
    
    if ((ent != CACHE_NULL) &&  (prot >= ent->prot)) {
	int sid = ent->addrtag >> RTA_VPAGE_BITS;
	int vpage = ent->addrtag & RTA_VPAGE_MASK;

	/* sanity */
	if ((sid != orig_sid) || ( vpage != orig_vpage))
		panic("cache is out to lunch");
	/* end sanity */
	
	mapout(ent->ppage,ppages_per_vmpage);
	mapin(ent->ppage,vpage,ent->prot,ppages_per_vmpage,sid);
	lo_hits++;
	return(TRUE);	    
    }
    return(FALSE);
}

