/* 
 * Mach Operating System
 * Copyright (c) 1988 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	float.c,v $
 * Revision 2.8  88/12/19  02:36:00  mwyoung
 * 	Move declarations of _FPemulate() and fpgen() outside any routine.
 * 	Avoid bogus lint by declaring _FPemulate() as an integer routine.
 * 	[88/12/17            mwyoung]
 * 	
 * 	Remove old MACH conditionals.
 * 	[88/12/14            mwyoung]
 * 
 * Revision 2.7  88/11/23  16:18:55  rpd
 * 	More Acis merge and cleanup.
 * 	[88/11/13  22:17:35  rpd]
 * 
 * Revision 2.6  88/10/10  22:21:58  sanzi
 * 	Fix include of mach_exception.h.  
 * 
 * Revision 2.5  88/10/06  13:37:55  sanzi
 * 	Fix __HIGHC__ conditional reference, fix include references.
 * 	
 * Revision 2.4  88/07/20  16:11:10  rpd
 * Fix pmap_pr_phys() call.
 * 
 * 25-Apr-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added a pcc related bug work-around.  
 *
 * 29-Jan-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	MACH_EXCEPTION changes.
 *
 * 15-Dec-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added float_pcb_init() routine for initializing the fpa state.
 *
 * 23-Sep-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Fixed a bug in float_pci which failed  when the user
 *	tried to acquire the mc881.
 *
 * 29-Jul-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	In fpa_pci(): determine if the reason for the program check
 *	was that the user executed a priveledged instruction, and if
 *	so, return 0.
 *
 * 24-Jul-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added call to pmap_pr_phys() to set User Read
 *	protection on kernel page containing fpa_valid variable.
 *
 *  5-Jul-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Eliminated FPA option, changed from ROMPC to ROMP_APC.
 *
 ***********************************************************************
 */
#include <romp_apc.h>

/*
 * 5799-CGZ (C) COPYRIGHT IBM CORPORATION 1987
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/* $Header: float.c,v 2.8 88/12/19 02:36:00 mwyoung Exp $ */
/* $ACIS:float.c 9.1$ */
/* $Source: /afs/cs.cmu.edu/source_mach/rcs/kernel/ca/float.c,v $ */

#ifndef lint
static char *rcsid = "$Header: float.c,v 2.8 88/12/19 02:36:00 mwyoung Exp $";
#endif

/*
 * This file contains those routines which (should be) the only
 * routines the kernel needs to call in order to perform floating
 * point support.  All knowledge of the actual floating point
 * hardware configuration is confined to this file.  From this
 * file, we call routines in fpa.c, mc881.c, and (someday)
 * fpa_emul.c.
 *
 * The following, in alphabetical order, are the routines
 * which are called from the outside world.
 */
#include <sys/types.h>
#include <sys/param.h>
#include <sys/dir.h>
#include <sys/signal.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/thread.h>

#include <ca/sigframe.h>
#include <ca/fp.h>
#include <ca/float.h>
#include <ca/rosetta.h>		/* for pmap_pr_phys() call */
#include <vm/vm_kern.h>		/* for declaration of kernel_map */
#include <vm/vm_param.h>	/* for round_page(), trunc_page() */
#include <sys/exception.h>
#include <sys/ux_exception.h>

struct floatsave *floatsave;	/* NPROC of these allocated */
int	float_hardware;		/* What hardware is on the system */

/*
 *	Routines provided by "lib/libfp.a":
 */

extern void _FPemulate();	/* Floating point emulator */
extern void fpgen();		/* Floating point code generator */

/*
 * float_core(thread) :
 *
 * Called to get the register set (if it exists)
 * into the user address space so that a debugger can look at it.
 */

void
float_core(thread)
thread_t thread;
{
    int floatmask = thread->pcb->floatmask;

#if	ROMP_APC
    if (float_has_881(floatmask)) {
	mc881_core(thread);
    }
#endif	ROMP_APC
    if (float_has_fpa_or_afpa(floatmask)) {
	fpa_core(thread);
    }
    if (float_has_emul(floatmask)) {
	emul_core(thread);
    }
}

/*
 * float_exit:
 *
 * When a process exits, we may have to do some clean up.
 */

 
void
float_exit(th)
struct thread *th;			/* exiting proc */
{
    int floatmask = th->pcb->floatmask;

#if	ROMP_APC
    if (float_has_881(floatmask)) {
	mc881_exit(th);
    }
#endif	ROMP_APC
    if (float_has_fpa_or_afpa(floatmask)) {
	fpa_exit(th);
    }
    if (float_has_emul(floatmask)) {
	emul_exit(th);
    }
}

/*
 * On a fork, make sure that floating point registers are copied down.
 *
 */

void
float_fork(oldth, newth)
struct thread	*oldth,		/* parent thread */
		*newth;		/* child thread */
{
    int floatmask = oldth->pcb->floatmask;

#if	ROMP_APC
    if (float_has_881(floatmask)) {
	mc881_fork(oldth, newth);
    }
#endif	ROMP_APC

    /*
     * The FPA code, which leaves stuff lying around in the proc struct,
     * needs to be called on EVERY fork.
     */
    fpa_fork(oldth, newth);

    if (float_has_emul(floatmask)) {
	emul_fork(oldth, newth);
    }
}
/*
 * This routine initializes the floating point state for
 * a thread which is created with thread_create(), rather
 * than created by forking.
 */
 
void float_pcb_init(pcb)
struct pcb *pcb;		/* new pcb */
{
    pcb->pcb_fpastatus = 0;
    pcb->pcb_floatinfo = 0;
    pcb->floatmask = 0;
    emul_pcb_init(pcb);
    fpa_pcb_init(pcb);
    mc881_pcb_init(pcb);
}
/*
 * float_getreg:
 *
 * Return the value of a floating point register.
 *
 * The value returned is through one of the arguments.  The actual
 * value of the function is:
 *	0		Good return
 *	non-0		Error value.
 */

int
float_getreg(reg, value)
int	reg;			/* Which floating point register */
int	*value;			/* Where value should be placed */
{
    int floatmask = current_thread()->pcb->floatmask;
    int registers;

#if	ROMP_APC
    if (float_has_881(floatmask)) {
	if ((registers = mc881_getreg(reg, value)) == 0) {
	    return 0;
	} else {
	    reg -= registers;
	}
    }
#endif	ROMP_APC

    if (float_has_fpa_or_afpa(floatmask)) {
	if ((registers = fpa_getreg(reg, value)) == 0) {
	    return 0;
	} else {
	    reg -= registers;
	}
    }

    if (float_has_emul(floatmask)) {
	if ((registers = emul_getreg(reg, value)) == 0) {
	    return 0;
	} else {
	    reg -= registers;
	}
    }
    return EINVAL;
}

/*
 * 1. call each initialization routine in turn.
 * 2. if any FPA hardware is present allocate the floatsave structure.
 *
 */
float_init()
{
    fpa_init();

#if	ROMP_APC
    mc881_init();
#endif	ROMP_APC
    emul_init();
    if (float_has_any(float_hardware)
    		|| float_has_afpa_hardware(float_hardware)) {
	/*
	 *	Set the protection on the page containing
	 *	the fpa_valid variable such that the user can read it.
	 *
	 *	XXX Protection Value 1 == Rosetta Kernel RW/ User R 
	 */
	extern int fpa_valid;		/* to make getfpemulator() work */
	pmap_pr_phys(kernel_pmap,
		     trunc_page(&fpa_valid),
		     round_page(&fpa_valid),
		     RTA_KEY_URKW << RTA_KEY_SHIFT);
	}
}

/*
 * trap has detected an invalid data reference that MIGHT be fpa related.
 * We return
 *	0	if it isn't ours (register set already allocated)
 *	1	if it is ours (we also allocate the appropriate register set)
 */

int
float_pci(mcs_pcs, info, locr0)
int	mcs_pcs;
int	info;
int	*locr0;		/* Where user registers are located */
{
    int result = 0;
    int floatmask = current_thread()->pcb->floatmask;
    int oldmask = floatmask;
    
#if	ROMP_APC
    result |= mc881_pci(mcs_pcs, info, locr0);
#endif	ROMP_APC
    result |= fpa_pci(mcs_pcs, info, locr0);

    floatmask = current_thread()->pcb->floatmask;
    
    if (result == 0) {
	return 0;
    } else {
	result = 0;
	if (float_has_881(floatmask)) {
	    result++;
	}
	if (float_has_emul(floatmask)) {
	    result++;
	}
	if (result > 1) {
	    uprintf("float: attempt to acquire %b when already using %b.\n",
	    	floatmask&(~oldmask), FLOAT_FMT,
		oldmask, FLOAT_FMT);
	    return 0;
	} else {
	    return 1;
	}
    }
}


/*
 * float_putreg :
 *
 * For a given floating point register number, set the value.
 */

int
float_putreg(reg, value)
int	reg;
int	*value;
{
    int floatmask = current_thread()->pcb->floatmask;
    int registers;
    
#if	ROMP_APC
    if (float_has_881(floatmask)) {
	if ((registers = mc881_putreg(reg, value)) == 0) {
	    return 0;
	} else {
	    reg -= registers;
	}
    }
#endif	ROMP_APC

    if (float_has_fpa_or_afpa(floatmask)) {
	if ((registers = fpa_putreg(reg, value)) == 0) {
	    return 0;	
	} else {
	    reg -= registers;
	}
    }

    if (float_has_emul(floatmask)) {
	if ((registers = emul_putreg(reg, value)) == 0) {
	    return 0;
	} else {
	    reg -= registers;
	}
    }
    return EINVAL;
}


/*
 * float_sendsig
 *
 * We are sending a floating point signal to a process.  Add
 * the floatsave structure to the stack.
 *
 * Returns
 *	0	We didn't save anything (integer divide case)
 *	1	We saved a floatsave structure
 */

int
float_sendsig(fp)
struct sigframe *fp;
{
    int floatmask = current_thread()->pcb->floatmask;
    int result = 0;

    if (u.u_code != FP_INT_DIVIDE) {
#if	ROMP_APC
	if (float_has_881(floatmask)) {
	    result |= mc881_sendsig(&fp->sf_floatsave);
	}
#endif	ROMP_APC

	if (float_has_fpa_or_afpa(floatmask)) {
	    result |= fpa_sendsig(&fp->sf_floatsave);
	}

	if (float_has_emul(floatmask)) {
	    result |= emul_sendsig(&fp->sf_floatsave);
	}
	return result;
    } else {
	return 0;
    }
}



/*
 * float_sigcleanup:
 *
 * Do some very hardware specific things (at the
 * request of the user) during signal cleanup time.
 */

void
float_sigcleanup(sf_scp)
struct sigcontext *sf_scp;
{
    int floatmask = current_thread()->pcb->floatmask;

#if	ROMP_APC
    if (float_has_881(floatmask)) {
	mc881_sigcleanup(sf_scp);
    }
#endif	ROMP_APC
#ifdef	FPA
    if (float_has_fpa_or_afpa(floatmask)) {
	fpa_sigcleanup(sf_scp);
    }
#endif	/* FPA */
    if (float_has_emul(floatmask)) {
	emul_sigcleanup(sf_scp);
    }
}

/*
 * getfloatstate()
 *
 * This returns, to the user, information which can be used to determine
 * which floating point hardware exists on the system, and which floating
 * point hardware is in use by the current (invoking) process.
 */

getfloatstate()
{
    struct a {
	caddr_t	state;		/* address of structure to be filled in */
	caddr_t	size;		/* size of structure pointed to by 'state' */
    } *uap;
    struct floatstate ourstate;
    int usersize;

    /* Set up argument list */
    uap = (struct a *)u.u_ap;

    /* Fetch (and validate, by the way) the size */
    u.u_error = copyin(uap->size, (caddr_t) &usersize, sizeof usersize);
    if (u.u_error) {
	return;
    }

    /* We could access size; is it OK? */
    if (usersize != sizeof (struct floatstate)) {
		/* Aside from sizeof (struct floatstate),
		 * what are the legal sizes.
		 */
	static int legalsizes[] = { 16 };
	int i;

	u.u_error = EINVAL;
	for (i = 0; i < sizeof legalsizes/sizeof legalsizes[0]; i++) {
	    if (usersize == legalsizes[i]) {
		u.u_error = 0;		/* No error */
		break;
	    }
	}
	if (u.u_error) {
	    return;
	}
    }

    /* OK.  Now, set up our version of the structure. */
    ourstate.hardware_state = float_hardware;
    ourstate.process_state = current_thread()->pcb->floatmask;
    ourstate.emulator = _FPemulate;
    ourstate.code_generator = fpgen;
    ourstate.fpa_registerset = current_thread()->pcb->fparegs;

    /* Now, copy our version up to the user process (validating the pointer) */
    u.u_error = copyout((caddr_t) &ourstate, uap->state, usersize);

    /* Done.  If an error on the above copyout, u.u_error is already set */
}

/*
 * system call to set return fpaemulator address
 */
getfpemulator()
{
    extern int fpa_valid;

    if (float_has_fpa_or_afpa(float_hardware)) {
	u.u_r.r_val1 = (int) &fpa_valid;
    } else {
	if (float_has_881(current_thread()->pcb->floatmask)) {
	    thread_doexception(current_thread(),
	    		EXC_ARITHMETIC,
			EXC_ROMP_FLOAT_SPEC,
			current_thread()->pcb->floatmask);
			
 	} else {
	    u.u_r.r_val1 = (int) ((int (*)())_FPemulate);
	    /* We assume if getfpemulator, and no fpa, then will use emul */
	    current_thread()->pcb->floatmask |= FLOAT_EMUL;/* Using emulator */
	}
	u.u_r.r_val2 = (int) ((int (*)())_FPemulate);
    }
}


/*
 * Kernel call to get fp
 */
kgetfpemulator()
{
    return (int) ((int (*)())_FPemulate);
}


/*
 * setfloatstate
 *
 * This allows a user process to set the state of the floating
 * point hardware in use by the invoking process.
 *
 * In addition, the super user is able to change the state of
 * the hardware configuration using this syscall.  This would
 * most typically be used to enable the afpa (advanced floating
 * point adapter) after a user-level process has loaded micro-
 * code into it.
 */

setfloatstate()
{
    struct a {
	caddr_t	mask;		/* Pointer to structure containing mask */
	caddr_t state;	/* Pointer to structure containing new value */
	caddr_t	size;		/* size of structure pointed to by 'state' */
    } *uap;
    struct floatstate newstate;
    struct floatstate newmask;
    int usersize;
    int hardware, user;

    /* Set up argument list */
    uap = (struct a *)u.u_ap;

    /* Fetch (and validate, by the way) the size */
    u.u_error = copyin(uap->size, (caddr_t) &usersize, sizeof usersize);
    if (u.u_error) {
	return;
    }

    /* We could access size; is it OK? */
    if (usersize != sizeof (struct floatstate)) {
	u.u_error = EINVAL;
	return;
    }

    /* Copyin the mask and new values */
    u.u_error = copyin(uap->mask, (caddr_t) &newmask, usersize);
    u.u_error = copyin(uap->state, (caddr_t) &newstate, usersize);

    /* Check for super user */
    if (!suser() && (newmask.hardware_state != 0)) {
	return;			/* suser set EPERM */
    }

    hardware = (float_hardware&~newmask.hardware_state) |
			    (newstate.hardware_state&newmask.hardware_state);
    user = (current_thread()->pcb->floatmask&~newmask.process_state) |
			    (newstate.process_state&newmask.process_state);
	    /*
	     * We are about to change the state of the hardware.
	     * So, to keep things cool, let the affected hardware
	     * handlers keep track of what is going on.
	     *
	     * In addition, the hardware handlers are allowed to
	     * change bits in the input (if, for example, the
	     * afpa handler decides that the afpa isn't really
	     * working very well; or if the particular handler
	     * doesn't allow the "dropping" of a piece of hardware
	     * on the fly).
	     */
    mc881_setfloatstate(&newmask, &newstate, hardware, user);
    fpa_setfloatstate(&newmask, &newstate, hardware, user);

    /* The hardware handlers may have modified the mask, so recompute */
    hardware = (float_hardware&~newmask.hardware_state) |
			    (newstate.hardware_state&newmask.hardware_state);
    user = (current_thread()->pcb->floatmask&~newmask.process_state) |
			    (newstate.process_state&newmask.process_state);

    /* Update the data areas */

    float_hardware = hardware;
    current_thread()->pcb->floatmask = user;

    /* OK.  Now, set up our version of the structure. */
    newstate.hardware_state = float_hardware;
    newstate.process_state = current_thread()->pcb->floatmask;
    newstate.emulator = _FPemulate;
    newstate.code_generator = fpgen;

    /* Now, copy our version up to the user process (validating the pointer) */
    u.u_error = copyout((caddr_t) &newstate, uap->state, usersize);

    /* Done.  If an error on the above copyout, u.u_error is already set */
}
