/* 
 * Mach Operating System
 * Copyright (c) 1988 Carnegie-Mellon University
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/***********************************************************************
 * HISTORY
 * $Log:	fpa.c,v $
 * Revision 2.8  89/01/12  07:39:31  rpd
 * 	Renamed fpa_counters calls, type with ibmrt_fpa_counters.
 * 	[89/01/12  06:40:10  rpd]
 * 	
 * 	Moved fpa_counters.h from kern/ to ca/.
 * 	Added MACH_DEBUG_CA conditional around fpa_counters kernel calls.
 * 	[89/01/12  04:29:02  rpd]
 * 
 * Revision 2.7  89/01/04  13:32:58  rpd
 * 	Added more counters (pci_kernel, pci_from_save) to fpa_pci.
 * 	[89/01/03  16:42:50  rpd]
 * 	
 * 	Removed Acis fpa_count code; replaced with fpa_counter() calls.
 * 	Added (under ROMP_FPA_COUNT) kernel calls to reset/retrieve
 * 	the fpa counters.
 * 	[89/01/01  14:56:23  rpd]
 * 
 * Revision 2.6  88/12/20  13:39:16  rpd
 * 	Fixed to use fpa_rdscx() when allocating/stealing
 * 	an fp register set. (From APAR A2005.)
 * 	[88/12/19  17:49:42  rpd]
 * 
 * Revision 2.5  88/12/19  02:36:17  mwyoung
 * 	Remove lint.
 * 	[88/12/18  23:28:36  mwyoung]
 * 	
 * 	Remove old MACH conditionals.
 * 	[88/12/14            mwyoung]
 * 
 * Revision 2.4  88/11/23  17:19:55  rpd
 * 	Picked up a DEBUGF change from Acis.
 * 	[88/11/23  10:26:23  rpd]
 * 	
 * 	More Acis merge.
 * 	[88/11/13  22:20:21  rpd]
 * 	
 * 	Changed includes to the new style.
 * 	Merged with Acis (135 support, etc).
 * 	[88/11/04  17:02:18  rpd]
 * 
 * Revision 2.3  88/10/06  13:40:29  sanzi
 * 	Added fpa changes for AFPA from ACIS.
 * 
 * 25-Apr-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added AFPA support.  Changed the fpa_pci() routine to ignore the 
 *	info parameter passed from trap() when trying to handle DMA
 *	faults.  The info value seems to be wrong.
 *
 *  4-Feb-88  David Golub (dbg) at Carnegie-Mellon University
 *	Fixed backward conditional in fpa_save.  Fixed include file
 *	names.
 *
 *  1-Feb-88  David Golub (dbg) at Carnegie-Mellon University
 *	Made addressing error return invalid address.
 *
 * 29-Jan-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	MACH_EXCEPTION changes.
 *
 * 12-Dec-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	In fpa_pci(), before testing for direct mode compilation of FPA
 *	code, make sure the fault was in user mode.  Also added
 * 	fpa_pcb_init() for initializing a thread's pcb state.
 *
 * 21-Sep-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Since floating point exceptions are associated with threads
 *	rather than processes, I replaced all uses of u.u_procp->p_flag 
 *	with th->pcb->pcb_status, since this is where we maintain the
 *	rest of the floating point information for a thread.
 *
 * 26-Aug-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	A number of changes, specifically:
 *		1: in fpa_pci() call vm_fault() instead of vtop() and
 *			pagein().
 *		2: in fpa_trap() change the parameters to agree with the
 *			way that fault() calls trap().
 *		3: in fpa_pci() fixed the code which tried to read user
 *			space directly (see section which does the direct
 *			simulation of FPA opcodes using fpput and fpget).
 *		4: merged latest 4.3 ACIS changes for the support of the
 *			FPA and	AFPA.
 *
 *************************************************************************
 */
/*
 * 5799-CGZ (C) COPYRIGHT IBM CORPORATION  1986,1987
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/* $Header: fpa.c,v 2.8 89/01/12 07:39:31 rpd Exp $ */
/* $ACIS:fpa.c 9.5$ */
/* $Source: /afs/cs.cmu.edu/source_mach/rcs/kernel/ca/fpa.c,v $ */


#include <mach.h>
#include <romp_debug.h>
#include <romp_rdb.h>
#include <romp_apc.h>
#include <romp_135.h>
#include <romp_fpa_count.h>
#include <mach_debug_ca.h>

#include <ca/debug.h>
#include <ca/scr.h>
#include <ca/mmu.h>
#include <ca/reg.h>

#include <sys/param.h>
#include <sys/vmparam.h>
#include <sys/systm.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/acct.h>
#include <sys/kernel.h>
#include <ca/io.h>
#include <ca/ioim.h>
#include <ca/float.h>
#include <ca/fp.h>
#include <ca/fpa.h>
#include <ca/fpavar.h>
#include <sys/buf.h>
#include <sys/proc.h>
#include <sys/thread.h>
#include <vm/vm_param.h>
#if	MACH
#include <sys/exception.h>
#include <sys/ux_exception.h>
#endif	MACH
#include <ca/fpa_counters.h>

#if	ROMP_DEBUG || ROMP_RDB
#define	STATIC
#else	ROMP_DEBUG || ROMP_RDB
#define	STATIC	static
#endif	ROMP_DEBUG || ROMP_RDB

int fpa_valid = 0;

int fpa_num_regs = NUM_FPA_FLOATS;	/* How many registers to save */
int fpa_num_sets = NUM_FPA_SETS;

STATIC void fpa_restore();

/*
 * Variables used (mostly) internal to the fpa code.
 */

thread_t fpa_thread[NUM_FPA_SETS];	/* current owner of this set */

STATIC int
	fpa_nextgrab = 0,	/* next register to grab */
	fpa_status;		/* Status of last exception */

STATIC int fpa_regs = 0;		/* register allocation bit map */
#define	REG_SET_IN_USE(i)	(fpa_regs&(1<<(i)))
#define	REG_SET_FREE(i)		(fpa_regs &= ~(1<<(i)))
#define	REG_SET_ALLOCATE(i)	(fpa_regs |= 1<<(i))

STATIC int
	fpa_unused = FPA_NONE;	/* What the default case is */

#define	FPA_CUR_NONE()		fpa_curreg = FPA_NONE
#define	FPA_CUR_UNUSED()	fpa_curreg = fpa_unused
#define	FPA_CUR_REGISTER(i)	fpa_curreg = (i)

/* The initial state values for the FPA */
STATIC struct
				/* data Ilr	fpasr	ier rer  flags */
	fpa_state fpa_initial = { { 0 }, 0 , FP_S_unsgd, 0, 0,  FPA_EIGHTH_REG};

/* fpa exceptions in string form */
STATIC char *fpa_tft[8] = {
    "no exception", "underflow", "overflow", "zero devide",
    "reserved", "operand error", "inexact result", "illegal command" };

#define LOADREGSET
#ifdef LOADREGSET
static long zeros[NUM_AFPA_FLOATS] = { 0 };

static long regset24[NUM_AFPA_FLOATS] = {
0x00000000, 0x00036000, 0x00016000, 0x0003F800,
0x00016000, 0x00016800, 0x00017000, 0x00017800,
0x0001E000, 0x0001E800, 0x0001F000, 0x0001F800,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00100000, 0x00000000, 0x00800000, 0x00000000,
0x40000040, 0x00000000, 0xFFFFEFFF, 0x40808000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0xBFA8B9FD, 0xE410C874, 0x3FB5D9D9, 0x7886D172,
0xBFBC4F7F, 0x0223ED73, 0x3FC24865, 0xC0060469,
0xBFC99996, 0x886FF101, 0x3FD55555, 0x544E0375
};

static long regset27[NUM_AFPA_FLOATS] = {
0x00000000, 0x00000000, 0x3FC45F30, 0x6DC9C883,
0x401921FB, 0x40000000, 0x3E94442D, 0x18000000,
0x3FD45F30, 0x6DC9C883, 0x400921FB, 0x40000000,
0x3E84442D, 0x18000000, 0x00000000, 0x000FFFFF,
0x7FFFFFFF, 0xFFFFFFFF, 0x41800000, 0x00000000,
0x00000000, 0x00000000, 0x40038000, 0x00000000,
0x3FF30000, 0x3FE60000, 0x3FDC0000, 0x40080000,
0x3FEF730B, 0xD281F69B, 0xBC7C23DF, 0xEFEAE6B5,
0x3FDDAC67, 0x0561BB4F, 0x3C55543B, 0x8F253271,
0x43ABC16D, 0x674EC800, 0x3E112E0B, 0xE826D695,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000
};

static long regset28[NUM_AFPA_FLOATS] = {
0x00000000, 0x00000000, 0xBDA8BD59, 0x86B2A52E,
0x3E21EE3B, 0x60DDDC8C, 0xBE927E4F, 0x1314AD1A,
0x3EFA01A0, 0x1971CAEB, 0xBF56C16C, 0x16C14199,
0x3FA55555, 0x5555553E, 0x00000000, 0x40B00000,
0x40862000, 0x000003E8, 0x3BFD83C9, 0x4FB6D2AC,
0x00000000, 0x00000000, 0x7FF00000, 0x00000000,
0x400921FB, 0x54442D18, 0x3FF921FB, 0x54442D18,
0x3FE921FB, 0x54442D18, 0x4002D97C, 0x7F3321D2,
0x3FE00000, 0x00000000, 0x3FD0B70C, 0x6D604DD4,
0x3FF00000, 0x00000000, 0x3E49C511, 0xDC3A41DF,
0x3DE5CF61, 0xDF672B13, 0xBE5AE5C0, 0xE319A4EF,
0x3EC71DE3, 0x209CDCD9, 0xBF2A01A0, 0x19746345,
0x3F811111, 0x1110C461, 0xBFC55555, 0x5555550C,
0xFFFFFE7F, 0x54442D19, 0x3F015317, 0xDF4526C4,
0x3F8C70FF, 0x8B3CC2CF, 0x3F503FC4, 0xCB8C98E8,
0x3FBC7195, 0x38248597, 0x3FF71547, 0x652B82FE
};

static long regset29[NUM_AFPA_FLOATS] = {
0x00000000, 0x00000000, 0x000007FD, 0x00000000,
0x00100000, 0x00000000, 0x3FF6A09E, 0x667F3BCD,
0x41300000, 0x000003FF, 0x3DEA39EF, 0x35793C76,
0x3FE62E42, 0xFEE00000, 0x00000000, 0x40000000,
0x3FDBCB7B, 0x1526E50E, 0x40877AF8, 0xEBEAE354,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x3FC2F039, 0xF0085122, 0x3FC39A1E, 0xC014045B,
0x3FC74663, 0xCC94342F, 0x3FCC71C5, 0x2150BEA6,
0x3FD24924, 0x941E07B4, 0x3FD99999, 0x9997FF24,
0x3FE55555, 0x55555592, 0x3F90D521, 0x74A1BB54,
0xBFA2CA6C, 0x04C6937A, 0x3FA98605, 0x24BDD807,
0xBFADE125, 0xFDDBD793, 0x3FB10DA9, 0x24597FFF,
0xBFB3B0FA, 0x54BEC400, 0x3FB745CE, 0x5AA35DB2,
0xBFBC71C7, 0x059AF280, 0x3FC24924, 0x921EC1D7,
0xBFC99999, 0x99997CCD, 0x3FD55555, 0x555555C3,
0x7FF00000, 0x408FF800, 0xFFFFFE7F, 0x00000000
};

static long regset30[NUM_AFPA_FLOATS] = {
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00000000,
0x00400000, 0x7F800000, 0x7FF00000, 0x60008000,
0xBBBBBBBB, 0xAAAAAAAA, 0x90000000, 0x00000000,
0x00000000, 0xFFFFFFFF, 0x7FFFFFFF, 0x00080000,
0x7FC00000, 0x7FF80000, 0x00010000, 0x00033000,
0x7F800000, 0x7FF00000, 0x00012000, 0x0003F000,
0x00000C00, 0x00030C00, 0x00004800, 0x0003C800
};

static long regset31[NUM_AFPA_FLOATS] = {
0x00000000, 0x00000000, 0x00000004, 0x41000000,
0x00005800, 0xFFC00000, 0x41E00000, 0x43300000,
0x00001006, 0x00000000, 0x80000000, 0x7F800000,
0x7FF00000, 0x00000000, 0x00000000, 0x000FFFFF,
0x00028000, 0x00100000, 0x81000000, 0x00000000,
0x00000000, 0x00000000, 0x00000000, 0x00008000,
0x84000000, 0x80000020, 0x0003F800, 0x00012000,
0x7FF00000, 0x48008000, 0x84000800, 0x40008040,
0x80000820, 0xFFFF7FFF, 0x42008000, 0x81000800,
0xFFFFDFFF, 0x00000000, 0xFFFFFFF8, 0x00004000,
0x00000007, 0x00000005, 0xFFFFFFF8, 0x00000000,
0x00023C00, 0x00027000, 0x00002000, 0x00000010,
0x00000040, 0x7FFFFFFF, 0x0002DC00, 0x0003FC00,
0x00001000, 0xFFFFF9F8, 0x00000607, 0x00000000,
0x00000000, 0x00007000, 0x80000000, 0x00000180,
0x00000080, 0x00000000, 0xFFFFFE7F, 0xFFFF8FFF
};

static long	*registersets[NUM_FPA_SETS] = {
		zeros, zeros, zeros, zeros,
		zeros, zeros, zeros, zeros,
		zeros, zeros, zeros, zeros,
		zeros, zeros, zeros, zeros,
		zeros, zeros, zeros, zeros,
		zeros, zeros, zeros, zeros,
		regset24, zeros, zeros, regset27,
		regset28, regset29, regset30, regset31
		};
#endif LOADREGSET

/*
 * The following section consists of the code and variables necessary
 * to detect the fpa and afpa at system startup time, bring the fpa 
 * online at system startup time, and bring the afpa online at
 * a later time (when a user program has loaded the microcode).
 */

extern void (*_trap)();			/* Pointer to trap slih */

STATIC label_t fpa_label;
/*
 * nofpa may be patched to cause FPA to be ignored 
 * during boot.
 */
STATIC int
	nofpa = 0,				/* assume that fpa is present */
	fpa_reset_done = 0,			/* 0 until fpa has been reset */
	fpa_mcs_pcs = 0;			/* what was the mcs_pcs? */

STATIC enum fpa_trap_state {
    state_illegal, state_longjmp, state_return
} fpa_trap_state = state_illegal;

/*
 * trap intercept routine in effect during fpa initialization.
 * this is done so that we can test to see if the fpa is there, and
 * also to handle the machine check that the fpa produces if it is 
 * reset more than once (sigh).
 */

STATIC void
fpa_trap(mcs_pcs, info, locr0)
	register mcs_pcs, info;
	register int *locr0;
{
#define	ics_cs locr0[ICSCS]

	fpa_mcs_pcs = mcs_pcs;		/* Save state */

	DEBUGF(fpadebug&SHOW_INIT,
	    prstate("fpa_trap", mcs_pcs, info, locr0[IAR], ics_cs, locr0));
#if	ROMP_APC
	DEBUGF ((fpadebug&SHOW_INIT) && cpu_model != CPU_SGP, {
	    int i;

	    for (i = IOIM1; i <= IOIM1+15; i++) {
		printf("0x%x", ior(i));
		if ((i%4) == 3) {
		    printf("\n");
		} else {
		    printf("\t");
		}
	    }
	});
#endif	ROMP_APC
	if (mcs_pcs & MCS_CHECK && fpa_reset_done == 0) {
		fpa_reset_done = 1;
		printf("FPA caused machine check ignored\n");
		klsreset();
		return;		/* try it again */
	}
	switch (fpa_trap_state) {
	case state_illegal:
		panic("fpa_trap:  illegal trap\n");
		/*NOTREACHED*/
	case state_longjmp:
		longjmp(&fpa_label);
		/*NOTREACHED*/
	case state_return:
		return;
	default:
		panic("fpa_trap:  illegal value in fpa_trap_state\n");
		/*NOTREACHED*/
	}
}


/*
 * initialize the FPA, including determining if it is present.
 * we determine if it is present by trapping faults (via _trap)
 * and marking the fpa not present.
 *
 * For the afpa, we just detect it, and then exit (since the micro
 * code needs to be loaded before we do anything exciting to it).
 *
 * note that this routine MUST be invoked before any fork's have
 * been done in order to set p_fpareg appropriately.
 * WARNING:
 *	it is arranged that fpa_curreg = FPA_NONE, and
 *		fpareg = FPA_NONE for the startup thread
 *	during startup. This prevents 'resume' from changing the fpa 
 *	register set or locking the fpa. Changing fpa_curreg or p_fpareg's 
 *	value is done here (in sync) to FPA_NONE if there is no fpa, or to
 *	FPA_UNUSED (fpa present, not being used by this thread).
 */

/*
 * This routine gets called if fpa_init decides we are running without
 * an afpa.
 */

STATIC void
fpa1_init()
{
	register void (*oldtrap)() = _trap;
	register int i, x;

		    /* assume no FPA present */
	FPA_CUR_NONE();

	fpa_num_regs = NUM_FPA_FLOATS;

	_trap = fpa_trap;
	fpa_trap_state = state_longjmp;

	if (setjmp(&fpa_label)) {	/* no FPA present */
		FPA_CUR_NONE();
		printf("Neither AFPA nor FPA present.\n");
		fpa_trap_state = state_illegal;
		_trap = oldtrap;
		return;
	}
	x = fpa_rdscx();
	FPA_SYNC(x);

	fpa_lockfp();
/* 
 * for each register set do a taskswitch and read status & clear exception 
 * and then write the status register.
 */
	for (i=0; i<NUM_FPA_SETS; ++i) {
		fpa_tskswu(i);
		x = fpa_rdscx();
		FPA_SYNC(x);
		fpa_wtstr(0);
		if (fpa_rdstr() != 0) {
			printf(
			    "FPA fails 'zero status' check.  Marked down.\n");
			fpa_trap_state = state_illegal;
			_trap = oldtrap;
			return;
		}
	}
	if (!fpa_test()) {
		fpa_lockfp();		/* lock the register set again */
		printf("FPA fails basic arithmetic checkout.  Marked down.\n");
		fpa_trap_state = state_illegal;
		_trap = oldtrap;
		return;
	}

#if	ROMP_APC
#if	ROMP_135
	if ((cpu_model == CPU_ROMPC) || (cpu_model == CPU_MOD135))
#else	ROMP_135
	if (cpu_model == CPU_ROMPC)
#endif	ROMP_135
	{
		/* Now, enable user access to subsegment F */
		x = ior(IOIM1+IOIM_SUB_PROB);		/* Get old value */
		iow(IOIM1+IOIM_SUB_PROB, x | SEG_F);	/* Turn on user access */
	}
#endif	ROMP_APC	

	fpa_lockfp();

				/* FPA present */
	 fpa_unused = FPA_UNUSED;
	FPA_CUR_UNUSED();

	float_hardware |= FLOAT_FPA;		/* We have an fpa */

	printf("FPA initialized %s\n",
		fpa_reset_done != 2 ? "normally" : " (after machine check)");

	fpa_trap_state = state_illegal;
	_trap = oldtrap;
	return;
}

#define FLOAT_ONE		0x3f800000
#define FLOAT_TWO		0x40000000
#define AFPA_PASS_UNKNOWN	0
#define AFPA_PASS_C		1
#define AFPA_PASS_D		2

#ifdef	notdef
static int	afpa_diag_ucode_offset[4] = {3988, 3992, 3999, 4028};
static int	afpa_diag_ucode_hi[47] = {
			0x39800020, 0x07080000, 0x0E003FF0, 0x0B000000,
			0x13800000, 0x09000C00, 0x03800000, 0x0F800000,
			0x0B800000, 0x0EA00240, 0x06800000, 0x06800000,
			0x06800000, 0x06800000, 0x06800000, 0x06800000,
			0x06800000, 0x06800000, 0x06800000, 0x06800000,
			0x00800000, 0x0980F070, 0x1B000000, 0x00007C0A,
			0x3DE0F00A, 0x3DE0F00A, 0x3DE0F00A, 0x3980018A,
			0xC0450000, 0x3B870000, 0x398001FE, 0x3980020A,
			0x39800219, 0x00000009, 0xF86A000A, 0x320703FA,
			0x3980018E, 0x3D870000, 0x3980018A, 0x39800180,
			0x00000000, 0x0000000A, 0x39800180, 0x398001C0,
			0x39800280, 0x39800270, 0x39800258
			};
static int	afpa_diag_ucode_lo[47] = {
			0x01210000, 0x00000010, 0x01000000, 0x01000030,
			0x010000D0, 0x01000050, 0x01000070, 0x010000B8,
			0x010000A0, 0x01000000, 0x010000B0, 0x01000008,
			0x01000018, 0x01000080, 0x01000010, 0x01000000,
			0x01000000, 0x01000000, 0x01000000, 0x01000090,
			0xA1000000, 0x01000000, 0x01000078, 0xA0000000,
			0xA10800B8, 0xA1080098, 0xA1080008, 0xA1080000,
			0x01000000, 0x91000000, 0x01540000, 0xA1080000,
			0x81400000, 0x8D40F088, 0xA1103FF0, 0x11080078,
			0xA1540090, 0xA1000008, 0xA1080090, 0x01000090,
			0xA1000020, 0x01080008, 0xA1000090, 0x9120AFA0,
			0x014000A0, 0x010000A0, 0x014100A0
			};

/*
 * Tesing for AFPA pass_c or pass_d:
 *
 * Write float 1 to reg0.  Write using these diag ucode will
 * first write, and then add the number to itself.  Pass_c
 * use ucode to write, so will read back twice what is written.
 * Pass_d doesn't.
 */
static int
afpa_init()
{
	int	i, hi_word, lo_word;
	volatile int result;

	for (i=0; i<43; i++) {		/* load diag ucode for afpa */
		hi_word = AFPA_UCODE_HIGH | (i << 2);
		*(int volatile *)hi_word = afpa_diag_ucode_hi[i];
		DELAY(1);			/* force delay */
		lo_word = AFPA_UCODE_LOW | (i << 2);
		*(int volatile *)lo_word = afpa_diag_ucode_lo[i];
	}
	DEBUGF(fpadebug&SHOW_INIT,
		printf("afpa_init: 43 diag ucode words loaded\n"));

	for (i=43; i<47; i++) {		/* some special location for ucode */
		hi_word = AFPA_UCODE_HIGH | (afpa_diag_ucode_offset[i-43] << 2);
		*(int volatile *)hi_word = afpa_diag_ucode_hi[i];
		DELAY(1);			/* force delay */
		lo_word = AFPA_UCODE_LOW | (afpa_diag_ucode_offset[i-43] << 2);
		*(int volatile *)lo_word = afpa_diag_ucode_lo[i];
	}
	DEBUGF(fpadebug&SHOW_INIT,
		printf("afpa_init: 47 diag ucode words loaded\n"));

	DELAY(1);			/* force delay */
	afpa_disable_control_store();	/* MACRO */
	DELAY(1);			/* force delay */
	DEBUGF(fpadebug&SHOW_INIT, printf("afpa_init: disable_control\n"));

	fpa_tskswu(0);
	DEBUGF(fpadebug&SHOW_INIT, printf("afpa_init: fpa_tskswu\n"));

	afpa_write_register(0, FLOAT_ONE);	/* float 1 to reg 0 */
	DEBUGF(fpadebug&SHOW_INIT, printf("afpa_init: write\n"));

	i = afpa_read_register(0);		/* read it back */
	result = i;				/* sync */
	result = i;				/* sync */
	DEBUGF(fpadebug&SHOW_INIT, printf("afpa_init: read\n"));

	afpa_enable_control_store();	/* MACRO */
	DEBUGF(fpadebug&SHOW_INIT, printf("afpa_init: enable_control\n"));

	i = result;
	if (i == FLOAT_TWO) {
		return(AFPA_PASS_C);
	} else if (i == FLOAT_ONE) {
		return(AFPA_PASS_D);
	} else {
		printf("afpa_init, i = 0x%x\n",i);
		return(AFPA_PASS_UNKNOWN);
	}

}
#endif	notdef

/* see above comment */
fpa_init()
{
	register void (*oldtrap)() = _trap;
	register int x;

		    /* assume no FPA present */
	FPA_CUR_NONE();
	fpa_unused = FPA_NONE;	
	if (nofpa) {
		printf("FPA ignored\n");
		return;
	}

	_trap = fpa_trap;
	fpa_trap_state = state_longjmp;

	/* First, what kind of fpa is out there? */
	/* Read control store high, location 0 */
	if (setjmp(&fpa_label)) {		/* if we trap, it is fpa1 */
		fpa_trap_state = state_illegal;
		_trap = oldtrap;
		fpa1_init();			/* Go check for fpa1 */
		return;
	}
	/* if we didn't trap, it is afpa */
	DEBUGF(fpadebug&SHOW_INIT, printf("fpa_init: before AFPA_UCODE_HIGH\n"));
	x = *(int volatile *) AFPA_UCODE_HIGH;
	FPA_SYNC(x);				/* Synchronize */
	    /* Do NOT set 'FLOAT_AFPA' until microcode loaded (afpacode(8)) */
	float_hardware |= (FLOAT_AFPA_HARDWARE|FLOAT_AFPA_CONTROL_STORE_ENABLE);
	fpa_num_regs = NUM_AFPA_FLOATS;

	/* is it AFPA (pass_c) or E_AFPA (pass_d)? */
	x = AFPA_PASS_C;	/* for now!! it should be x=afpa_init() */
	if (x == AFPA_PASS_C) {
		printf("AFPA");
	} else if (x == AFPA_PASS_D) {
		float_hardware |= FLOAT_AFPA_PASS_D;
		printf("E_AFPA");
	} else {
		printf("UN_AFPA");
	}
	printf(" marked down pending microcode load and initialization.\n");

	fpa_trap_state = state_illegal;
	_trap = oldtrap;
	return;
}



/*
 * Respond to a user (superuser) request to enable the afpa's
 * control store.  In this (new) state, the user program will
 * be able to load (and dump) the afpa's control store.
 */

STATIC int
afpa_enable_control()
{
	register void (*oldtrap)() = _trap;
	int x;

	DEBUGF(fpadebug&SHOW_INIT, printf("afpa_enable_control\n"));

	_trap = fpa_trap;
	fpa_trap_state = state_longjmp;

	if (setjmp(&fpa_label)) {
		fpa_trap_state = state_illegal;
		_trap = oldtrap;
		uprintf("AFPA unable to enable control store.\n");
		return 0;
	}

	afpa_enable_control_store();	/* MACRO */

	x = *(int volatile *)AFPA_UCODE_HIGH;	/* Test to see if it worked */
	FPA_SYNC(x);

	return 1;
}


/*
 * The user (superuser) has requested that access to the afpa's control
 * store be disabled.  This automatically starts the afpa running.
 *
 * What we need to do is:
 *	o	Disable Control Store enable (this starts the microcode).
 *
 * After we finish, the user code still needs to:
 *	o	Initialize the "reserved" registers.
 *	o	Let us know how many registers are reserved for use
 *		by the microcode.
 */

STATIC int
afpa_disable_control()
{
	register void (*oldtrap)() = _trap;
	int s = spl7();
	int i,j;
#ifdef LOADREGSET
	long *regs;
#endif LOADREGSET

	DEBUGF(fpadebug&SHOW_INIT, printf("afpa_disable_control\n"));
	_trap = fpa_trap;
	fpa_trap_state = state_longjmp;

	if (setjmp(&fpa_label)) {
		fpa_trap_state = state_illegal;
		_trap = oldtrap;
		splx(s);
		uprintf("AFPA control store disable failed.\n");
		return 0;
	}

	afpa_disable_control_store();	/* MACRO */

	/*
	 * Now, initialize all the registers.
	 */

	if (setjmp(&fpa_label)) {
		fpa_trap_state = state_illegal;
		_trap = oldtrap;
		splx(s);
		uprintf("AFPA:  Unable to zero register sets.\n");
		return 0;
	}

#ifdef LOADREGSET
	for (i = 0; i < NUM_FPA_SETS; i++) {
		DEBUGF(fpadebug&SHOW_REGS, printf("afpa_disable_control: load register set %d\n",i));
		afpa_diag_tskswu(i);
		DELAY(1);			/* force delay */
		regs = registersets[i];
		for (j = 0; j < NUM_AFPA_FLOATS; j++) {
			afpa_write_register(j, *regs);
			DELAY(1);	/* force a delay */
			DEBUGF(fpadebug&SHOW_REGS, printf("r%d=%x%c",j,*regs,((j+1)&07) ? ' ' : '\n'));
			regs++;
		}
	}
#else LOADREGSET
	for (i = 0; i < NUM_FPA_SETS; i++) {
		DEBUGF(fpadebug&SHOW_REGS, printf("afpa_disable_control: zero register set %d\n",i));
		afpa_diag_tskswu(i);
		for (j = 0; j < NUM_AFPA_FLOATS; j++) {
			afpa_write_register(j, 0);
		}
	}
#endif LOADREGSET

#if	ROMP_APC
#if	ROMP_135
    if ((cpu_model == CPU_ROMPC) || (cpu_model == CPU_MOD135))
#else	ROMP_135
    if (cpu_model == CPU_ROMPC)
#endif	ROMP_135
	{
	/* Now, enable user access to subsegments E and F */
	i = ior(IOIM1+IOIM_SUB_PROB);		/* Get old value */
	iow(IOIM1+IOIM_SUB_PROB, i | SEG_E | SEG_F);/* Turn on user access */
	}
#endif ROMP_APC

	fpa_trap_state = state_illegal;
	_trap = oldtrap;
	splx(s);
	DEBUGF(fpadebug&SHOW_INIT, printf("afpa_disable_control (i=%x) - return 1\n",i));
	return 1;
}

/*
 * Here to bring the afpa online.
 */

STATIC int
afpa_online()
{
	register void (*oldtrap)() = _trap;
	int x, i;
	int s = spl7();

	DEBUGF(fpadebug&SHOW_INIT, printf("afpa_online\n"));

	_trap = fpa_trap;
	fpa_trap_state = state_longjmp;
/* 
 * for each register set do a taskswitch and read status & clear exception 
 * and then write the status register.
 */

	DEBUGF(fpadebug&SHOW_INIT, printf("fpa_num_sets = %d\n", fpa_num_sets));
	for (i=0; i < fpa_num_sets; i++) {

		if (setjmp(&fpa_label)) {
			fpa_trap_state = state_illegal;
			_trap = oldtrap;
			splx(s);
			uprintf("AFPA error resetting register sets.\n");
			return 0;
		}
		fpa_tskswu(i);
		if (setjmp(&fpa_label)) {
		    if (fpa_mcs_pcs & MCS_DATA_TO) {
			fpa_trap_state = state_illegal;
			_trap = oldtrap;
			splx(s);
			uprintf("AFPA error reading and clearing status.");
			uprintf("(register set %d)\n", i);
			return 0;
		    }
		    /* If no timeout, keep trying the reset... */
		}
		x = fpa_rdscx();
		FPA_SYNC(x);
		if (setjmp(&fpa_label)) {
			fpa_trap_state = state_illegal;
			_trap = oldtrap;
			splx(s);
			uprintf("AFPA error resetting register sets.\n");
			return 0;
		}
		fpa_wtstr(0);
		if ((x = fpa_rdstr()) != 0) {
			fpa_trap_state = state_illegal;
			_trap = oldtrap;
			splx(s);
			uprintf(
			"AFPA register set %d failed 'zero status' check.\n",
			i, x);
			return 0;
		}
	}
	fpa_trap_state = state_illegal;
	_trap = oldtrap;
	splx(s);
	fpa_unused = FPA_UNUSED;
	FPA_CUR_UNUSED();
	return 1;
}

/*
 * The following are the internal routines called by the externally-callable
 * fpa (and afpa) routines.
 *
 * They are listed in alphabetical order.
 */

/*
 * allocate a register set to the current thread
 * 1. find a set
 * 2. set it up
 */

STATIC void
fpa_alloc()
{
	STATIC void fpa_register_set();

	fpa_register_set(fpa_findreg());
}

/*
 * Bump the current owner of a register set.
 */

STATIC void
fpa_bump(set)
int	set;
{
	thread_t th;
	void fpa_save();
	int x;
	
	DEBUGF(fpadebug&SHOW_REGS,
			printf("fpa_bump: bumping register set %d\n",set));
	th = fpa_thread[set];
	if ((th == THREAD_NULL) || (th->pcb->fparegs != set)) {
		panic("fpa_bump");	/* oops */
	}
	fpa_tskswu(set);		/* make it current */
			/* Save the register set.  Don't access user space */
	fpa_save(th->pcb->fpasave.fpa_intreg, th, 0);
	x = fpa_rdscx();
	FPA_SYNC(x);
	REG_SET_FREE(th->pcb->fparegs);
	th->pcb->fparegs = FPA_SAVED;	/* we saved it */
	FPA_CUR_UNUSED();
	fpa_counter(bump);
}

/*
 * locate a free register if possible - otherwise steal one
 */
STATIC int
fpa_findreg()
{
	register int i;
	int x;
	
	for (i = 0; i < fpa_num_sets; ++i) {
		if (!REG_SET_IN_USE(i))  {
			break;
		}
	}
	if (i == fpa_num_sets) {
	    i = fpa_grabreg();
	}
	DEBUGF(fpadebug&SHOW_REGS, printf("fpa_findreg: pid %d reg set %d\n",
		u.u_procp->p_pid,i));
	fpa_tskswu(i);			/* switch to it */
	x = fpa_rdscx();
	FPA_SYNC(x);
	return i;
}

/*
 * grab a register set from some other thread
 */
STATIC int
fpa_grabreg()
{
	register int set = fpa_nextgrab;

	if (++set >= fpa_num_sets) {
		set = 0;
	}
	fpa_nextgrab = set;
	DEBUGF(fpadebug&SHOW_REGS,
			printf("fpa_grabreg: grabbing register set %d\n",set));
	fpa_bump(set);
	fpa_counter(grabreg);
	REG_SET_ALLOCATE(set);
	return set;
}

/*
 * fpa_register_set(i)
 *
 * Set up register set i for the indicated thread
 * (including loading the correct contents into registers)
 */

STATIC void
fpa_register_set(i)
int i;
{
	thread_t th = current_thread();
	STATIC void fpa_restore();
	int oldset = th->pcb->fparegs;

	REG_SET_ALLOCATE(i);		/* Allocate this set */
	th->pcb->floatmask |=
		(float_hardware&(FLOAT_FPA|FLOAT_AFPA|FLOAT_AFPA_DMA|FLOAT_AFPA_PASS_D));
	th->pcb->fparegs = i;
	FPA_CUR_REGISTER(i);
	fpa_thread[i] = th;		/* remember who has it */
	fpa_restore((oldset == FPA_SAVED) ?
	    (struct fpa_state *)th->pcb->fpasave.fpa_intreg: &fpa_initial, 0);
}


/*
 * Restore the current register set from restore area provided
 *
 * As a side-effect, we may restore the eighth register to
 * the user area (in the case of an fpa).
 */

STATIC void
fpa_restore(restorefp, save_eighth)
struct fpa_state *restorefp;
int	save_eighth;
{
	int i;
	u_int *data = &restorefp->programmer.data[0];

	if (fpa_num_regs <= NUM_FPA_FLOATS) {	/* FPA1 mode */
		int volatile *FPA_ptr = (int volatile *) (FPA_BASE + 0x025000);
		for (i=0; i < fpa_num_regs; ++i) {
			* FPA_ptr = *data++;
			FPA_ptr += 1;		/* actually add 2^2 or 4 */
		}
		/*
		 * On an fpa, restore the eighth register to the USER_FPM.
		 */
		if (save_eighth & (restorefp->flags&FPA_EIGHTH_REG)) {
			((FP_MACH *)USER_FPM)->dreg[NUM_FPA_FLOATS/2].dfracth =
				   restorefp->programmer.data[NUM_FPA_FLOATS];
			((FP_MACH *)USER_FPM)->dreg[NUM_FPA_FLOATS/2].dfractl =
				   restorefp->programmer.data[NUM_FPA_FLOATS+1];
		}
		fpa_wtier(restorefp->ier);	/* AFPA doesn't support this */
	} else {
		for (i=0; i < fpa_num_regs; ++i) {
			afpa_write_register(i, *data);
			data++;
		}
	}
		    
	fpa_wtstr(restorefp->fpasr);		/* store status */
}


/*
 * Save the current register set into save area provided
 *
 * If the current thread is using the fpa, and if we have
 * access to the user stack (ie: the user hasn't blown it away),
 * then save the eighth double register (maintained
 * by the code generator).
 *
 * Since we are sometimes called only to make sure the eighth soft
 * register is updated, we check the thread structure to see if
 * we should actually save the hardware registers.
 */

void fpa_save(savefp, th, save_eighth)
struct fpa_state *savefp;
thread_t	th;
int		save_eighth;		/* Are we running in the user context */
{
	int i;
	u_int *data = &savefp->programmer.data[0];

	if (fpa_hasreg(th->pcb->fparegs)) {
		savefp->fpasr = fpa_rdstr();		/* store status */
		if (fpa_num_regs <= NUM_FPA_FLOATS) {
		    int volatile *FPA_ptr = (int volatile *) (FPA_BASE + 0x02f000);
		    savefp->ier = fpa_rdier();	/* AFPA doesn't support this */
		    for (i=0; i < fpa_num_regs; ++i) {
			    *data++ = * FPA_ptr;
			    FPA_ptr += 16;	/* actually add 2^6 or 64 */
		    }
		} else {
		    for (i=0; i < fpa_num_regs; ++i) {
			    *data++ = afpa_read_register(i);
		    }
		}
	}
	if ((fpa_num_regs <= NUM_FPA_FLOATS)
		&& (fpa_hasreg(th->pcb->fparegs)
			|| (th->pcb->fparegs == FPA_SAVED))) {
		/*
		 * If we are running on the user's stack, and if this is
		 * an fpa user (as opposed to an afpa user), then
		 * pull in the software (code generator) maintained
		 * eighth data register.
		 */
#if	MACH 
/*
 * START UNFORTUNATE CODE
 *
 *  This is where having multiple threads in a task and
 * not rewriting all of this code is a big lose.
 */
if (current_thread() == th) {
#endif	MACH    
		if (save_eighth && float_has_fpa(th->pcb->floatmask)) {
			savefp->programmer.data[NUM_FPA_FLOATS] =
			  ((FP_MACH *)USER_FPM)->dreg[NUM_FPA_FLOATS/2].dfracth;
			savefp->programmer.data[NUM_FPA_FLOATS+1] =
			  ((FP_MACH *)USER_FPM)->dreg[NUM_FPA_FLOATS/2].dfractl;
			savefp->flags |= FPA_EIGHTH_REG;
		} else {
			/* If we can't save the state, note that fact */
			savefp->flags &= ~FPA_EIGHTH_REG;
		}
#if	MACH		
} else  {
	/* If we can t save the state, note that fact */
	savefp->flags &= ~FPA_EIGHTH_REG;
}
/*
 * END UNFORTUNATE CODE
 */
#endif	MACH
			}
}

/*
 * do FPA unit test to determine if FPA is operational.
 */
STATIC int
fpa_test()
{
	int result;

	* (int volatile *) 0xff025000 = 0x3f800000;	/* fr0 = 1.0 */
	* (int volatile *) 0xff090480 = 0x40000000;	/* fr0 += (fr2 = 2.0) */

	result = * (int volatile *) 0xff02f000;		/* result = fr0 */
	return(result == 0x40400000);
}

/*
 * The following routines, presented in alphabetical order, comprise
 * the set of routines called from float.c.
 */


/*
 * realize the fpa register set into user address space for inspection
 * by debugger. 
 * if the fpa is present and allocated we store it into the emulator
 * data structure on the top of the user stack
 * if saved we copy the saved structure.
 */
fpa_core(th)
thread_t th;
{
	register int fp_reg = th->pcb->fparegs;
	struct fpa_state *state = (struct fpa_state *)&th->pcb->fpasave;

	if (fpa_hasreg(fp_reg))	{ /* has register set */
		register int status = fpa_rdscx();	/* paranoia */
		if (status == 4) {
		    status = 3;		/* Synch up hardware */
		}
	}
	fpa_save((struct fpa_state *)state, th, 1);	/* Always save */
#ifdef	notdef
	bcopy((caddr_t) state, (caddr_t) (USER_FPM), sizeof *state);
#endif	notdef
}


/*
 * invoked at exit or exec to release a register set
 */
fpa_exit(th)
thread_t th;
{
	register int reg = th->pcb->fparegs;

	DEBUGF(fpadebug&SHOW_INIT, printf("fpa_exit called.\n"));
	if (fpa_hasreg(reg)) {
		register int status = fpa_rdscx();	/* paranoia */

		if (status == 4) {
		    status = 3;		/* Synch up hardware */
		}
		DEBUGF(fpadebug&SHOW_REGS,
			printf("fpa_exit: release thread 0x%x regs %d\n",
				th,reg));
		fpa_lockfp();
		REG_SET_FREE(reg);		/* release register set */
		fpa_thread[reg] = THREAD_NULL;		/* forget it */
	}
	th->pcb->fparegs = fpa_unused;		/* mark as unused */
	FPA_CUR_UNUSED();
	th->pcb->floatmask &= ~FLOAT_FPA;
}

/*
 * fork -
 *
 *	Save the old register set.  This will save the actual hardware
 * registers if the old register set was in the hardware.  Else,
 * it will just update the eighth (software) register.  Then, copy
 * the old area into the new area.
 */

void
fpa_fork(oldth,newth)
register thread_t oldth, newth;
{
	struct fpa_state *newsavep = (struct fpa_state *)&newth->pcb->fpasave;
	struct fpa_state *oldsavep = (struct fpa_state *)&oldth->pcb->fpasave;

	/*
	 * Don't do the assignment if we are dealing with someone who
	 * doesn't have anything (since when we do the initial system
	 * fork(s) at init_main time we can't afford to take page
	 * faults, etc.).
	 */
	if (fpa_hasreg(oldth->pcb->fparegs) || 
	    (oldth->pcb->fparegs == FPA_SAVED)) {
		/*
		 * Note: ACIS doesn't copy in the eighth register, since
		 * they maintain it at/near/above the user stack, but
		 * that can only work for single threaded processes.
		 */
		fpa_save(oldsavep, oldth, 1); 
		*newsavep = *oldsavep;
	}
	if (fpa_hasreg(oldth->pcb->fparegs)) {
		newth->pcb->fparegs = FPA_SAVED;
	} else {
		newth->pcb->fparegs = oldth->pcb->fparegs;
	}
}

void
fpa_pcb_init(pcb)
struct pcb *pcb;
{
   
    pcb->fparegs = FPA_UNUSED;
}

/*
 * utility routine to get floating point register contents
 * (invoked from ptrace)
 * return:	0	ok
 *		~0	failed
 * note that if an FPA register set has not yet been allocated
 * referencing the FPA will cause one to be.
 */

int
fpa_getreg(reg,value)
u_int reg;
int *value;
{
	thread_t th = current_thread();
	struct fpa_state *newsavep = (struct fpa_state *)&th->pcb->fpasave;

	fpa_save(newsavep, th, 1);

	if (fpa_hasreg(th->pcb->fparegs)) {	/* has register set */
		REG_SET_FREE(th->pcb->fparegs);
		fpa_lockfp();			/* Lock the FPA */
		th->pcb->fparegs = FPA_SAVED;	/* mark as saved */
		FPA_CUR_UNUSED();
	}
	if ((reg < fpa_num_regs) ||
		(float_has_fpa(th->pcb->floatmask) && 
		  (reg < NUM_FPA_FLOATS+2))) {
		*value = newsavep->programmer.data[reg];
		return 0;		/* succeeded */
	} else {
		/* invalid register number */
		if (float_has_fpa(th->pcb->floatmask)) {
		    return NUM_FPA_FLOATS+1;
		} else {
		    return fpa_num_regs;
		}
	}
}

/*
 * trap has detected a invalid data reference that might be 
 * fpa related.
 * we return 0 if it isn't ours (fpa reg set already allocated)
 *	     1 if we have now allocated register set
 * to be done:
 *	check to see if the reference is to '0xff......'
 */
fpa_pci(mcs_pcs,info,locr0)
	register int mcs_pcs;
	int info;
	register int *locr0;
{
	extern unsigned int a_save_context, a_longjmp;
	extern short resume_rdstr;	/* read status instruction */
	static int level = 0;	/* to detect recursive fpa exceptions */
	thread_t th = current_thread();
	register int fp_reg = th->pcb->fparegs;
	register int floatmask = th->pcb->floatmask;
	register int fixup_rc;
	kern_return_t result;

	fpa_counter(pci_total);

	if (!(mcs_pcs & USER)) {
	    fpa_counter(pci_kernel);

	    if ((a_save_context <= (unsigned int) locr0[IAR]) &&
		((unsigned int) locr0[IAR] < a_longjmp)) {
		/* did we trap while trying to lock fpa in save_context? */
		fpa_counter(pci_from_save);
	    }
	}

	if ((GETSEG(info) != GETSEG(FPA_BASE)) ||
	    ((GETSUBSEG(info) != GETSUBSEG(FPA_BASE)) &&
	     (GETSUBSEG(info) != GETSUBSEG(FPA_DMA_BASE)))) {
	    fpa_counter(pci_info_ret);
#ifdef	notdef
	    return 0;
#endif	notdef
	}

	DEBUGF(fpadebug&SHOW_PCI,
		prstate("fpa_pci", mcs_pcs, info, locr0[IAR], locr0[ICSCS],locr0));
	if (float_has_fpa_or_afpa(float_hardware) || fpa_hasreg(fp_reg)) {
		register int status = fpa_rdscx();

		fpa_counter(pci_has_fpa);

		DEBUGF((fpadebug&SHOW_STATUS) || level,
			printf("fpa_status=0x%b tft=%s\n", status, "\20\4UNDER-ENABLE\5UNDERFLOW\6INEXACT-ENABLE\7INEXACT\10ROUND0\11ROUND1\12COND-Z\13COND-N\14PCK-DISABLE\15TASK-EXCEPTION\16BUS-EXCEPTION\17ABORTED\20PZERO",fpa_tft[status&0x07]));
		if (level++)
			panic("recursive FPA exception");
		fpa_status = status;		/* remember the status */
/*
 * test to see if we had a page fault on a dma operation
 */
		if (((status & FPA_TASK_EXCEPTION) == 0) &&
		    (GETSUBSEG(info) == GETSUBSEG(FPA_DMA_BASE)) &&
		    (float_has_afpa(float_hardware)
				|| float_has_afpa(floatmask))) {
			int cps_status = ior(IOIM1+IOIM_CPS);
			int addr = (locr0[EX1_DATA] & ~0xff)
						| ((cps_status>>8) &0xfc);

			fpa_counter(pci_dma);

			/* XXX see below
			 * int fault_type = (info & RTA_EX_HACK) ?
 	    		 *  (VM_PROT_READ) : (VM_PROT_READ|VM_PROT_WRITE);
			 */
			if (addr < locr0[EX1_DATA]) {
				fpa_counter(pci_dma_carry);
				addr += 0x100;		/* carry */
			}
			if (cps_status&0x100) {
				fpa_counter(pci_dma_ahead);
				addr -= 4;	/* address was ahead by 4 */
			}
			DEBUGF(fpadebug&SHOW_PCI, printf("dma_addr=%x ", addr));

			/*
			 * Need to decrement level here, since
			 * vm_fault() may sleep.
			 *
			 * XXX Check whether the fault_type is correct.
			 */
			--level;
			if ((result = vm_fault(current_task()->map,
					trunc_page(addr),
/* fault_type, XXX */ 			VM_PROT_READ|VM_PROT_WRITE,
					FALSE))
				!= KERN_SUCCESS) {
				fpa_counter(pci_dma_except);
				thread_doexception(current_thread(),
					EXC_BAD_ACCESS,
					result,
					addr);
					
			}
			return 1;
		}
		
		if ((status & FPA_TASK_EXCEPTION) == 0) {
			fpa_counter(pci_not_fpa);
			--level;
			return(0);		/* not fpa related */
		}

		if (!fpa_hasreg(fp_reg)) {
			fpa_counter(pci_no_reg);
			fpa_alloc();	/* get a register set */
			if ((th->pcb->pcb_floatinfo&SOWEFPA)==0){
				fpa_counter(pci_no_reg_ok);
				--level;
				return(1);	/* exception handled */
			}
		}
		if (th->pcb->pcb_floatinfo&SOWEFPA) {
			fpa_counter(pci_sowefpa);

			status = th->pcb->pcb_fpastatus;	/* remembered stat */
			DEBUGF(fpadebug&(SHOW_STATUS|SHOW_DELAY),
				printf("delayed fpa_status=0x%b tft=%s\n", status, "\20\4UNDER-ENABLE\5UNDERFLOW\6INEXACT-ENABLE\7INEXACT\10ROUND0\11ROUND1\12COND-Z\13COND-N\14PCK-DISABLE\15TASK-EXCEPTION\16BUS-EXCEPTION\17ABORTED\20PZERO",fpa_tft[status&0x07]));
			th->pcb->pcb_floatinfo &= ~SOWEFPA; 
				/* no longer owed to us */
		}
		fpa_status = status;		/* remember the status */

/*
 * if fpa exception happened during critical code in resume then remember
 * the exception and process it later 
 * by saving the register set as if it was grabbed by another thread.
 * we then release the register set (though we don't actually lock
 * the fpa as that will be done by the resume code, which will also
 * set fpa_curreg).
 */
		if ((short *) (locr0[IAR]) == &resume_rdstr ||
				(short *) (locr0[IAR]) == &resume_rdstr + 1 ) {
			struct fpa_state *newsavep = 
				(struct fpa_state *)&th->pcb->fpasave;

			fpa_counter(pci_resume);

				    /* Save fpa state in new area */
			fpa_save(newsavep, th, 0);
			th->pcb->pcb_floatinfo |= SOWEFPA; 
				/* owed an fpa exception */
			th->pcb->fparegs = FPA_SAVED;	/* we saved it */
			FPA_CUR_UNUSED();
			REG_SET_FREE(fp_reg);		/* release regs */
			fpa_thread[fp_reg] = THREAD_NULL;/* forget it */
			th->pcb->pcb_fpastatus = status;
			DEBUGF(fpadebug&(SHOW_PCI|SHOW_DELAY), 
				printf("delaying FPA exception in resume\n"));
			--level;
			return(1);
		}

/*
 * test to see if the problem was an invalid reference to segment F
 * (indicated by a either no-exception (shouldn't happen) or
 * an illegal command in the status register); return to trap which
 * will deliver an error to the thread.
 */
		DEBUGF(fpadebug&SHOW_PCI,
			printf("fpa_status=0x%x\n", status));
		if ((status&07) == 7) {
			fpa_counter(pci_illegal);

			--level;
			return(0);	/* illegal command - give error */
		}
		
	/* Call IEEE fixup routine for arithmetic fixups*/
		/*
		 * Mask software level intr to get around the optimization
		 * bug that softclock() is being called directly from
		 * hardclock(), thus fp regs are not properly saved/restored.
		 * softclock() should be called from slih6().
		 */
		{
		int	s = spl1();
		fixup_rc = _fpfpx(status, th->pcb->floatmask);
		splx(s);
		}
		if (!fixup_rc) {
			fpa_counter(pci_fixed_up);

			DEBUGF(fpadebug&SHOW_PCI, 
				printf("FPA/IEEE fixup done\n"));
			DEBUGF(fp_reg == FPA_SAVED && 
					(fpadebug&(SHOW_STATUS|SHOW_DELAY)),
				{	/* display actual status */
				status = fpa_rdstr();
				printf("fpa_status=0x%b tft=%s\n", status, "\20\4UNDER-ENABLE\5UNDERFLOW\6INEXACT-ENABLE\7INEXACT\10ROUND0\11ROUND1\12COND-Z\13COND-N\14PCK-DISABLE\15TASK-EXCEPTION\16BUS-EXCEPTION\17ABORTED\20PZERO",fpa_tft[status&0x07]); } );
			--level;
			return(1); 		/* handled */
		}
		--level;			/* copyout may sleep; */
		copyout((caddr_t)(&machine),(caddr_t)USER_FPM,sizeof(FP_MACH));
		fpa_counter(pci_not_fixed);
#if	MACH
		thread_doexception(current_thread(),
			EXC_ARITHMETIC,
			EXC_ROMP_FPA_EMUL,
			status);
#else 	MACH
		u.u_code = status;		/* pass to handler */
		psignal(u.u_procp, SIGFPE);		/* signal FPE */
#endif	MACH		
		return(1);			/* now handled */
	}

	fpa_counter(pci_dir_check);

#define FPaGET0_DIRECT 0xcd230000	/* L r2,0(r2) */
#define FPaPUT0_DIRECT 0xdd230000	/* ST r2,0(r2) */
/*
 * check to see if this is a program compiled for direct mode running
 * without an FPA.
 * we assume this is so if the instruction is the correct format 
 * and the value in r3 has 0xff in the high-order byte.
 * in this case we arrange for control to be transfered to the floating
 * point emulator just as if it was called from user mode in compatible
 * mode.
 * we do this by setting the iar to the appropriate fpget or fpput
 * and r15 to be the (return) address following the offending instruction.
 */
 	if (mcs_pcs & USER) {
		register int iar = locr0[IAR];
		extern fpget, fpput;
		register u_short x1, x2;
		label_t jmpbuf;
		int s;

		fpa_counter(pci_dir_user);

		s = splvm();
		if (setjmp(&jmpbuf)) {
		    fpa_counter(pci_dir_fault);

		    /*
		     * We faulted trying to read the instruction at
		     * 'iar' or 'iar+2'
		     *
		     *	hence the iar must be invalid.
		     */
		    current_thread()->recover = NULL;
		    splx(s);
		    return 0;	/* send a SIGBUS */
		}
		current_thread()->recover = (vm_offset_t) &jmpbuf;
		x1 = * (u_short *) iar;
		x2 = * (u_short *) (iar+2);
		current_thread()->recover = NULL;
		splx(s);

		if ((x1 == ((unsigned) FPaGET0_DIRECT >> 16) || 
				x1 == ((unsigned) FPaPUT0_DIRECT >> 16)) &&
				(unsigned) locr0[R3] >> 24 == 0xff &&
				x2 == (FPaGET0_DIRECT & 0xffff)) {
			
			fpa_counter(pci_dir_simul);

			DEBUGF(nfldebug&0x02,
				prstate("fpa direct simulation", mcs_pcs, 
					info, locr0[IAR], locr0[ICSCS],
					locr0));
			locr0[R15] = iar+4;		/* return address */
			locr0[IAR] = (x1 == ((unsigned)FPaGET0_DIRECT >> 16)) ?
				(int) &fpget :
				(unsigned) &fpput; /* code address */
			return(1);		   /* handled! */
		}

		fpa_counter(pci_dir_nosimul);
	}

	fpa_counter(pci_unrelated);
	return(0);		/* not FPA related */
}

/*
 * utility routine to set floating point register contents
 * (invoked from ptrace)
 * return:	0	ok
 *		~0	failed
 */

fpa_putreg(reg,value)
u_int reg;
int *value;
{
	thread_t th = current_thread();
	struct fpa_state *newsavep = (struct fpa_state *)&th->pcb->fpasave;

	fpa_save(newsavep, th, 1);

	if (fpa_hasreg(th->pcb->fparegs)) {	/* has register set */
		REG_SET_FREE(th->pcb->fparegs);
		fpa_lockfp();			/* Lock the FPA */
		th->pcb->fparegs = FPA_SAVED;	/* mark as saved */
		FPA_CUR_UNUSED();
	}
	if ((reg < fpa_num_regs) ||
		(float_has_fpa(th->pcb->floatmask) && 
		  (reg < NUM_FPA_FLOATS+2))) {
		newsavep->programmer.data[reg] = *value;
		return 0;		/* succeeded */
	} else {
			    /* invalid register number */
		if (float_has_fpa(th->pcb->floatmask)) {
			return NUM_FPA_FLOATS+1;
		} else {
			return fpa_num_regs;
		}
	}
}


/*
 * We are in the process of sending a floating point signal
 * to a user thread.  Kindly place information on the stack
 * to help the user's floating point signal handler.
 */

int
fpa_sendsig(exc_floatsave)
struct floatsave *exc_floatsave;
{
    return emul_sendsig(exc_floatsave);
}

/*
 * Alter floating point registers 
 * (should be available to signal handler in the sigcontext structure;
 * an "FP_MACH" (defined in ieeetrap.h) which user can modify and
 * results get put back into emulator machine (if no FPA) or back
 * on card (if FPA)).
 */
fpa_sigcleanup(sf_scp)
struct sigcontext *sf_scp;
{
	FP_MACH usermach;
	int destreg;
	int update_status, result_size;

		/* Call to emulator to handle the updating of USER_FPM */
	emul_handle_cleanup(sf_scp, &destreg, &result_size, &update_status);

	copyin((caddr_t)USER_FPM,(caddr_t)(&usermach),sizeof(FP_MACH));

	/* 
	 * NOTE:  I am not sure that we really want to update the status
	 * register with that returned by the signal handler.  User can
	 * easily screw it up, and if he/she uses the swapfptrap function
	 * from the signal handler to do what seems reasonable, this
	 * would tromp on that one...
	 */

	switch (result_size) {
	case 0:
		break;		/* Don't update result */
	case sizeof (FLOAT):
		_FPAws(destreg, usermach.dreg[destreg/2].dfracth);
		break;
	case sizeof (DOUBLE):
		_FPAwl(destreg, usermach.dreg[destreg/2]);
		break;
	default:
		panic("fpa_sigcleanup: unknown result size.\n");
		/*NOTREACHED*/
	}

	if (update_status) {
		_FPAputs(usermach.status);
	}
}

/*
 * So, this works like this.  If the user wants to get rid of their
 * own register set, fine.  If the user wants to get a brand new
 * register set, no way (they have to go touch the hardware).
 *
 * If the superuser (though WE don't check) wants to change the
 * afpa:
 *	o	If they turn off AFPA_CONTROL_STORE_ENABLE, we disable
 *		the control store, etc. (via afpa_disable_control()).
 *	o	If they turn on AFPA_CONTROL_STORE_ENABLE, we
 *		enable the control store.
 *	o	If they go from !AFPA to AFPA, then we set up the
 *		various variables to say the afpa is present.
 */

void
fpa_setfloatstate(mask, state, newhardware, newuser)
struct floatstate *mask, *state;
int	newhardware,		/* What the new hardware will be */
	newuser;		/* What the new user will be */
{
	thread_t th = current_thread();

#ifdef	lint
	newuser++;
#endif	lint

		/*
		 * First, (silently) disallow changing either fpa state
		 * or the existence of the afpa hardware.
		 */
    mask->hardware_state &= ~(FLOAT_FPA|FLOAT_AFPA_HARDWARE|FLOAT_AFPA_DMA|FLOAT_AFPA_PASS_D);

    DEBUGF(fpadebug&SHOW_INIT, printf("fpa_setfloatstate: float_hardware=%b mask->hardware_state=%b newhardware=%b\n",float_hardware,FLOAT_FMT,mask->hardware_state,FLOAT_FMT,newhardware,FLOAT_FMT));

    /* If new configuration is AFPA_CONTROL_STORE_ENABLE and AFPA, complain */
    if ((newhardware&(FLOAT_AFPA_CONTROL_STORE_ENABLE|FLOAT_AFPA)) ==
				(FLOAT_AFPA_CONTROL_STORE_ENABLE|FLOAT_AFPA)) {
	uprintf( "fpasetfloatstate: Illegal state \n");
	uprintf("\t(FLOAT_AFPA and FLOAT_AFPA_CONTROL_STORE_ENABLE).\n");
	mask->hardware_state &= ~(FLOAT_AFPA|FLOAT_AFPA_CONTROL_STORE_ENABLE);
	newhardware = float_hardware;
    }

    /* Only allow one state to change at a time */

    if ((mask->hardware_state&FLOAT_AFPA) &&
		(mask->hardware_state&FLOAT_AFPA_CONTROL_STORE_ENABLE)) {
	uprintf("fpasetfloatstate: Attempt to disable control store and\n");
	uprintf("\tbring afpa online simultaneously not allowed by kernel.\n");
	mask->hardware_state &= ~(FLOAT_AFPA|FLOAT_AFPA_CONTROL_STORE_ENABLE);
	newhardware = float_hardware;
    }

    /*
     * Changing Control Store Enable
     */

    if ((newhardware&FLOAT_AFPA_CONTROL_STORE_ENABLE)
			!= (float_hardware&FLOAT_AFPA_CONTROL_STORE_ENABLE)) {
	if (newhardware&FLOAT_AFPA_CONTROL_STORE_ENABLE) {	/* enable */
	    if (afpa_enable_control() == 0) {
		mask->hardware_state &= ~FLOAT_AFPA_CONTROL_STORE_ENABLE;
		newhardware = float_hardware;
	    }
	} else {				/* Turning "down" off */
	    if (afpa_disable_control() == 0) {
		mask->hardware_state &= ~FLOAT_AFPA_CONTROL_STORE_ENABLE;
		newhardware = float_hardware;
	    }
	}
    } else if ((newhardware&FLOAT_AFPA) != (float_hardware&FLOAT_AFPA)) {
	if (newhardware&FLOAT_AFPA) {	/* Wants to turn on afpa */
	    if (((unsigned)state->fpa_registerset) > fpa_num_sets) {
		u.u_error = EIO;
		mask->hardware_state &= ~FLOAT_AFPA;
		newhardware = float_hardware;
	    } else {			/* Turn it online if possible */
		fpa_num_sets = state->fpa_registerset;

		if (afpa_online() == 0) {	/* failed */
			mask->hardware_state &= ~FLOAT_AFPA;
			newhardware = float_hardware;
	    	} else {			/* succeeded */
			/*
			 * if we have just enabled AFPA,then enable AFPA_DMA if
			 * we are on a CPU that supports that feature.
			 */
#if	ROMP_135
			if ((cpu_model == CPU_ROMPC) || (cpu_model == CPU_MOD135))
#else	ROMP_135
			if (cpu_model == CPU_ROMPC)
#endif	ROMP_135
			{   /* we have DMA if ROMPC */
			    mask->hardware_state |= FLOAT_AFPA_DMA;	
			    state->hardware_state |= FLOAT_AFPA_DMA;
			}
		}
	    }	    
	} else {	/* We don't support turning off afpa */
	    mask->hardware_state &= ~FLOAT_AFPA;
	}
    }
    
    /*
     * Enabling/Disabling the AFPA or FPA for this process.
     */

    if ((mask->process_state&(FLOAT_FPA|FLOAT_AFPA)) &&
		(float_has_fpa_or_afpa(
			th->pcb->floatmask^state->process_state))) {
	/* The user is changing state.  Handle this. */

	if (float_has_fpa_or_afpa(state->process_state)) {
	    /* If they want it, let them touch it (then fpa_pci) */
	    mask->process_state &= ~(FLOAT_FPA|FLOAT_AFPA);
	} else {			/* They want to let the fpa go */
	    fpa_exit(th);
	}
    }

    /*
     * Changing register sets
     */
     
    if (mask->fpa_registerset) {	
	if (suser()) {
	    int i;

 	    DEBUGF(fpadebug&SHOW_INIT, printf("fpa_setfloatstate: registerset=%d\n",state->fpa_registerset));
	    if (fpa_hasreg(th->pcb->fparegs)) {
		fpa_save(
		    (struct fpa_state *) th->pcb->fpasave.fpa_intreg, th, 0);
		REG_SET_FREE(th->pcb->fparegs);
		th->pcb->fparegs = FPA_SAVED;	/* mark as saved */
		FPA_CUR_UNUSED();
	    }
	    i = state->fpa_registerset;
	    if ((th->pcb->fparegs == FPA_NONE) &&
		((float_hardware&
			(FLOAT_AFPA_HARDWARE|FLOAT_AFPA_CONTROL_STORE_ENABLE))
			!= FLOAT_AFPA_HARDWARE)) {
		u.u_error = EIO;
		mask->fpa_registerset = 0;
	    } else if ((i < 0) || (i >= fpa_num_sets)) {
		u.u_error = EIO;
		mask->fpa_registerset = 0;
	    } else {
		if (REG_SET_IN_USE(i)) {
		    fpa_bump(i);		/* Free the register set */
		}
		fpa_register_set(i);		/* Get new set */
		th->pcb->floatmask |= FLOAT_AFPA;
						/* Mark us using AFPA_FLOAT */
	    }
	} else {
	    /* EPERM already set */
	    mask->fpa_registerset = 0;
	}
    }
}

#if	ROMP_FPA_COUNT
ibmrt_fpa_counters_t fpa_counters;

#if	MACH_DEBUG_CA
kern_return_t
host_ibmrt_fpa_counters(task, counters)
	task_t task;
	ibmrt_fpa_counters_t *counters;
{
	*counters = fpa_counters;
	return KERN_SUCCESS;
}

kern_return_t
host_ibmrt_fpa_counters_reset(task)
	task_t task;
{
	bzero((char *) &fpa_counters, sizeof fpa_counters);
	return KERN_SUCCESS;
}
#endif	MACH_DEBUG_CA
#endif	ROMP_FPA_COUNT
