/*
 ****************************************************************
 * Mach Operating System
 * Copyright (c) 1986 Carnegie-Mellon University
 *  
 * This software was developed by the Mach operating system
 * project at Carnegie-Mellon University's Department of Computer
 * Science. Software contributors as of May 1986 include Mike Accetta, 
 * Robert Baron, William Bolosky, Jonathan Chew, David Golub, 
 * Glenn Marcy, Richard Rashid, Avie Tevanian and Michael Young. 
 * 
 * Some software in these files are derived from sources other
 * than CMU.  Previous copyright and other source notices are
 * preserved below and permission to use such software is
 * dependent on licenses from those institutions.
 * 
 * Permission to use the CMU portion of this software for 
 * any non-commercial research and development purpose is
 * granted with the understanding that appropriate credit
 * will be given to CMU, the Mach project and its authors.
 * The Mach project would appreciate being notified of any
 * modifications and of redistribution of this software so that
 * bug fixes and enhancements may be distributed to users.
 *
 * All other rights are reserved to Carnegie-Mellon University.
 ****************************************************************
 */
/*
 * HISTORY
 * 16-Aug-87  David Golub (dbg) at Carnegie-Mellon University
 *	Make kdbgetprocess return a useful map pointer if proc->task is
 *	TASK_NULL (a zombie process).
 *
 * 11-Jun-87  Daniel Julin (dpj) at Carnegie-Mellon University
 *	Avoided printing the trap type for single-step and breakpoints.
 *	This should allow Bill to set a breakpoint at printf without 
 *	running into trouble (most of the time).
 *
 *  6-Jun-87  Daniel Julin (dpj) at Carnegie-Mellon University
 *	Fixed a bug in kdbwlong for non-aligned references.
 *	Made kdb_trap not return blindly in case of recursive entry.
 *
 *  3-Feb-87  Daniel Julin (dpj) at Carnegie-Mellon University
 *	Created.
 *
 */

#include "mach.h"

#include "../h/param.h"
#include "../h/reboot.h"
#include "../h/systm.h"
#include "../h/vmmac.h"

#include "../h/ioctl.h"
#include "../h/tty.h"
#include "../cacons/screen_conf.h"
#include "../cacons/consvars.h"

#include "../ca/reg.h"
#include "../ca/pcb.h"
#include "../ca/scr.h"

#if	MACH
#include "../vm/vm_param.h"
#include "../vm/vm_map.h"
#include "../h/dir.h"		/* for user.h (!) */
#include "../h/user.h"		/* for u.u_procp  */
#include "../h/proc.h"
#include "../h/task.h"
#include "../h/thread.h"
#endif	MACH
#include "../machine/mmu.h"
#include "../machine/cpu.h"

#ifdef	notdef
#define	Mask_MOVL	0xF000
#define Op_MOVL		0x2000
#define	Mask_CLRL	0xFFC0
#define Op_CLRL		0x4280
#endif	notdef

#ifdef	lint
struct pcb kdbpcb;
char *kdberrflg;
#endif	lint

extern int kdbintr;

extern char *kdbsbrk();



#ifdef	notdef
char getmemc(pa)
{
	return(0);
}

putmemc(pa,val)
{
	return;
}
#endif	notdef


/*
 *  kdb_init - initialize kernel debugger
 */

kdb_init()
{
	kdbsetsym(); 
#ifdef	lint
	(void) kdbread(0, (char *)0, 0);
	kdbwrite(0, (char *)0, 0);
	kdbrlong((long *)0, (long *)0);
	kdbwlong((long *)0, (long *)0);
	(void) kdbsbrk(0);
#endif	lint
}



/*
 *  kdb_kintr - received keyboard interrupt sequence
 *
 *  Queue software interrupt to transfer into the debugger at the
 *  first opportunity.  This is not done if we are already in the
 *  debugger or if the system was booted without the KDB boot flag.
 */

int kdbactive = 0;

kdb_kintr()
{
	kdbintr = 1;
	mtsr(SCR_IRB,32768);	/* cause an interrupt at level 0 */
#ifdef	notdef
    if (kdbactive == 0 && (boothowto&RB_NOSYNC))
    {
	extern void softcall(), enter_kdb();
	/*
	 *	Schedule a software interrupt to get to KDB.
	 */
	current_thread()->pcb->pcb_flag &= ~TRACE_AST;
	current_thread()->pcb->pcb_flag |= TRACE_KDB;
	softcall(enter_kdb, (caddr_t) 0);
    }
#endif	notdef
}



/*
 *  kdb_trap - field a TRACE or BPT trap
 */

int kdbtrapok = 0;
int kdbreturn = 1;

kdb_trap(type,locr0)
	int		type;
	int		*locr0;
{
    extern struct pcb kdbpcb;
    register struct pcb *pcb = &kdbpcb;

    int spl;

    int cnarg = 0;

#ifdef	notdef
    if ((boothowto&RB_NOSYNC) == 0)
	return(0);
#endif	notdef

    switch (type)
    {
	case STEP:
	case BKPT:
	    break;

	default:
	{
#ifdef	notdef
	    int	i;
	    extern char *trap_type[];
	    extern int TRAP_TYPES;
	    extern char *kdberrflg;

	    i = type / sizeof(int);
	    printf("kernel: ");
	    if ((unsigned)i >= TRAP_TYPES)
		printf("type %d", type);
	    else
		printf("%s", trap_type[i]);
	    printf(" trap\n");
#endif	notdef
	    printf("kernel debugger: MCS_PCS = %x\n",type);
	    if (kdbtrapok)
	    {
		{
/*		    kdberrflg = "";*/
		    printf(" *** recursive entry to kernel debugger ***\n");
#ifdef	notdef
		    /* clear bus error frame! */
		    return(1);
#endif	notdef
		}
	    }
	}
    }

#if	0	/* ? do we need this ? */
    /*
     *  We'd rather run the debugger while on the interrupt stack so as not to
     *  run into kernel stack overflow problems depending on how deep we were
     *  when the trap occurred.  Thus, if we aren't already on the interrupt
     *  stack schedule a software interrupt at IPL 0xf to get us there.  First,
     *  raise the IPL to the maximum, though, to insure that the software
     *  interrupt won't happen until the REI which returns from the current
     *  trap completes and it can appears to happen at the same PC.
     */
    if ((movpsl()&PSL_IS) == 0)
    {
	int ipl;

	ipl = splx(0x1f);
	if (ipl < 0xf)
	{
	    mtpr(SIRR, 0xf);
	    return(1);
	}
	splx(ipl);
    }
#endif	0	/* ? do we need this ? */

    spl = spl7();
    kdbpcb.pcb_r0 = locr0[R0];
    kdbpcb.pcb_r1 = locr0[R1];
    kdbpcb.pcb_r2 = locr0[R2];
    kdbpcb.pcb_r3 = locr0[R3];
    kdbpcb.pcb_r4 = locr0[R4];
    kdbpcb.pcb_r5 = locr0[R5];
    kdbpcb.pcb_r6 = locr0[R6];
    kdbpcb.pcb_r7 = locr0[R7];
    kdbpcb.pcb_r8 = locr0[R8];
    kdbpcb.pcb_r9 = locr0[R9];
    kdbpcb.pcb_r10 = locr0[R10];
    kdbpcb.pcb_r11 = locr0[R11];
    kdbpcb.pcb_r12 = locr0[R12];
    kdbpcb.pcb_r13 = locr0[R13];
    kdbpcb.pcb_r14 = locr0[R14];
    kdbpcb.pcb_r15 = locr0[R15];
    kdbpcb.pcb_iar = locr0[IAR];
    kdbpcb.pcb_icscs = locr0[ICSCS];

    kdbactive++;

#ifdef	notdef
    (void) spl2();	/* set IPL below keyboard interrupt level (3) */
#endif	notdef

    kdbintr = 0;

    kdb(type, u.u_procp);

    kdbactive--;

    locr0[R0] = kdbpcb.pcb_r0;
    locr0[R1] = kdbpcb.pcb_r1;
    locr0[R2] = kdbpcb.pcb_r2;
    locr0[R3] = kdbpcb.pcb_r3;
    locr0[R4] = kdbpcb.pcb_r4;
    locr0[R5] = kdbpcb.pcb_r5;
    locr0[R6] = kdbpcb.pcb_r6;
    locr0[R7] = kdbpcb.pcb_r7;
    locr0[R8] = kdbpcb.pcb_r8;
    locr0[R9] = kdbpcb.pcb_r9;
    locr0[R10] = kdbpcb.pcb_r10;
    locr0[R11] = kdbpcb.pcb_r11;
    locr0[R12] = kdbpcb.pcb_r12;
    locr0[R13] = kdbpcb.pcb_r13;
    locr0[R14] = kdbpcb.pcb_r14;
    locr0[IAR] = kdbpcb.pcb_iar;
    locr0[ICSCS] = kdbpcb.pcb_icscs;

    splx(spl);

    cnioctl(cons[CONS_GEN].t_dev,TIOCFLUSH,&cnarg,0);

    return(kdbreturn);
}



/*
 *  kdbread - read character from input queue
 */

kdbread(x, cp, len)
register char *cp;
{
#ifdef lint
    while (x--) len--;
#endif lint
    *cp = getchar();
    return(1);
}



/*
 *  kdbwrite - send characters to terminal
 */

/* ARGSUSED */
kdbwrite(x, cp, len)
register char *cp;
register int len;
{
#ifdef lint
   while (x--);
#endif lint
   while (len--)
	cnputc(*cp++);
}



/*
 *   kdbrlong - read long word from kernel address space
 */

kdbrlong(addr, p)
register unsigned long *addr;
register unsigned long *p;
{
    *p = 0;
    kdbtrapok = 1;
    switch(((unsigned long)addr) % 4) {
	case 0:
		*p = *addr;
		break;
	case 1:
		addr = (unsigned long *)(((unsigned long)addr) & ~0x3);
		*p = (*addr << 8) | (*(addr + 1) >> 24);
		break;
	case 2:
		addr = (unsigned long *)(((unsigned long)addr) & ~0x3);
		*p = (*addr << 16) | (*(addr + 1) >> 16);
		break;
	case 3:
		addr = (unsigned long *)(((unsigned long)addr) & ~0x3);
		*p = (*addr << 24) | (*(addr + 1) >> 8);
		break;
    }
    kdbtrapok = 0;
}



/*
 *  kdbwlong - write long word to kernel address space
 */

kdbwlong(addr, p)
register unsigned long *addr;
register unsigned long *p;
{
#ifdef	notdef
    int	pme, oldpme = 0;
    extern char etext;

    if (addr >= (unsigned long *)VM_MIN_KERNEL_ADDRESS && addr <= (unsigned long *)&etext)
    {
	oldpme = getpgmap(addr);
	setpgmap(addr, (oldpme & ~PG_PROT) | PG_KW);
    }
#endif	notdef
    kdbtrapok = 1;
    switch(((unsigned long)addr) % 4) {
	case 0:
		*addr = *p;
		break;
	case 1:
		addr = (unsigned long *)(((unsigned long)addr) & ~0x3);
		*addr = (*addr & 0xff000000) | (*p >> 8);
		*(addr + 1) = (*(addr + 1) & 0xffffff) | (*p << 24);
		break;
	case 2:
		addr = (unsigned long *)(((unsigned long)addr) & ~0x3);
		*addr = (*addr & 0xffff0000) | (*p >> 16);
		*(addr + 1) = (*(addr + 1) & 0xffff) | (*p << 16);
		break;
	case 3:
		addr = (unsigned long *)(((unsigned long)addr) & ~0x3);
		*addr = (*addr & 0xffffff00) | (*p >> 24);
		*(addr + 1) = (*(addr + 1) & 0xff) | (*p << 8);
		break;
    }
    kdbtrapok = 0;
#ifdef	notdef
    if (oldpme)
    {
	setpgmap(addr, oldpme);
    }
#endif	notdef
}



/*
 *  kdbsbrk - extended debugger dynamic memory
 */

static char kdbbuf[1024];
char *kdbend = kdbbuf; 

char *
kdbsbrk(n)
unsigned n;
{
    char *old = kdbend;

    if ((kdbend+n) >= &kdbbuf[sizeof(kdbbuf)])
    {
	return((char *)-1);
    }
    kdbend += n;
    return(old);
}




#if	MACH
/*
 *	Return the map and pcb for a process.
 */
void kdbgetprocess(p, map, pcb)
	struct proc	*p;
	vm_map_t	*map;	/* OUT */
	struct pcb	**pcb;	/* OUT */

{
	/*
	 *	special case for current process
	 */
	if (p == u.u_procp) {
		*map = current_task()->map;
		*pcb = current_thread()->pcb;
	}
	else {
		if (p->task)
			*map = p->task->map;
		else
			*map = VM_MAP_NULL;
		if (p->thread)
			*pcb = p->thread->pcb;
		else
			*pcb = (struct pcb *)0;
	}
}


/*
 *	Reads or writes a longword to/from the specified address
 *	in the specified map.  The map is forced to be the kernel
 *	map if the address is in the kernel address space.
 *
 *	Returns 0 if read/write OK, -1 if not.
 */

int kdbreadwrite(map, addr, value, rd)
	vm_map_t	map;
	vm_offset_t	addr;
	long		*value;	/* IN/OUT */
	boolean_t	rd;
{
	if (addr >= VM_MIN_KERNEL_ADDRESS) {
		/*
		 *	in kernel
		 */
		if (rd)
			kdbrlong(addr, value);
		else
			kdbwlong(addr, value);
		return (0);
	}
	else {
		vm_offset_t	pa;	/* physical address */
		unsigned long	temp;
		int		i, ret;

		do {
			pa = pmap_extract(vm_map_pmap(map), addr);
			if (pa == 0) {
				ret = vm_fault(map, trunc_page(addr),
					rd ? VM_PROT_READ : VM_PROT_READ|VM_PROT_WRITE,
					FALSE);
				if (ret != KERN_SUCCESS)
					return (-1);
			}
		} while (pa == 0);

		if (rd) {
			temp = 0;
			copy_from_phys(pa,&temp,4);
#ifdef	notdef
			for (i = 0; i <= 3; i++) {
				temp <<= 8;
				temp |= (getmemc(pa + i)) & 0xFF;
			}
#endif	notdef
			*value = (int) temp;
		}
		else {
			temp = *value;
			copy_to_phys(&temp,pa,4);
#ifdef	notdef
			for (i = 3; i >= 0; i--) {
				putmemc(pa + i, temp & 0xFF);
				temp >>= 8;
			}
#endif	notdef
		}
		return (0);
	}
}
#endif	MACH

