/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 **********************************************************************
 * HISTORY
 * $Log:	kdb_print.c,v $
 * Revision 2.7  88/12/19  02:36:40  mwyoung
 * 	Put the 'm' print cases in the common file.
 * 	[88/12/06  01:00:50  mwyoung]
 * 
 * Revision 2.6  88/10/26  05:34:05  mwyoung
 * 	Correct include file references, conditional error.
 * 	Move 'k' to machine-independent cases.
 * 	[88/10/25            mwyoung]
 * 
 * Revision 2.5  88/10/25  03:39:35  mwyoung
 * 	Pick up printing improvements.
 * 	[88/10/25            mwyoung]
 * 
 * Revision 2.4  88/07/20  16:13:42  rpd
 * Support for varying the length of symbol names.
 * 
 * 17-May-88  David Golub (dbg) at Carnegie-Mellon University
 *	Changed thread times printout to match new thread timer fields.
 *
 * 29-Oct-87  David Golub (dbg) at Carnegie-Mellon University
 *	Changed thread state printout to match new scheduling state
 *	machine.
 *
 * 20-Aug-87  Daniel Julin (dpj) at Carnegie-Mellon University
 *	Added $K to trace back any thread, specified by its thread
 *	pointer. Modified for traceback() with a pcb argument.
 *
 * 11-Jun-87  Daniel Julin (dpj) at Carnegie-Mellon University
 *	Modified $l for MACH.
 *
 *  5-Jun-87  Daniel Julin (dpj) at Carnegie-Mellon University
 *	Fixed $c and $C to not crash the machine by default. sigh!
 *
 *  2-Apr-87  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Temporary hack to allow compilation under MACH.
 *
 *  8-Mar-87  Daniel Julin (dpj) at Carnegie-Mellon University
 *	Hacked $k to call the traceback routine copied from rdb.
 *
 * 31-Jan-87  Daniel Julin (dpj) at Carnegie-Mellon University
 *	Modified for kernel debugger KDB.
 *
 **********************************************************************
 */ 

#ifndef lint
static char *rcsid = "$ Header: print.c,v 5.3 86/02/17 17:24:21 zittin Exp $";
#endif
/*
 *
 *      UNIX debugger
 *
 */

#include <mach.h>
#include <ca/kdb/kdb_defs.h>

#if	MACH
#include <sys/task.h>
#include <sys/thread.h>

#include <kern/kdb_print.h>
#endif	MACH

#ifdef	KDB
#include <ca/frame.h>
#else	KDB
#include <frame.h>
#endif	KDB

MSG             LONGFIL;
MSG             NOTOPEN;
MSG             BADMOD;
MAP             txtmap;
MAP             datmap;
INT             infile;
INT             outfile;
CHAR            *lp;
L_INT           maxoff;
L_INT           maxpos;
INT             radix;
/* symbol management */
L_INT           localval;

/* error management -RNS */
extern char *errflg;

/* breakpoints */
BKPTR           bkpthead;
#ifdef	KDB
REGLIST reglist [] = {
/*        "error",UERROR, 0,*/
        "icscs",ICSCS,  &pcb.pcb_icscs,
        "iar",  IAR,    &pcb.pcb_iar,
/*        "mq",   MQ,     0,*/
        "r15",  R15,    &pcb.pcb_r15,
        "r14",  R14,    &pcb.pcb_r14,
        "r13",  R13,    &pcb.pcb_r13,
        "r12",  R12,    &pcb.pcb_r12,
        "r11",  R11,    &pcb.pcb_r11,
        "r10",  R10,    &pcb.pcb_r10,
        "r9",   R9,     &pcb.pcb_r9,
        "r8",   R8,     &pcb.pcb_r8,
        "r7",   R7,     &pcb.pcb_r7,
        "r6",   R6,     &pcb.pcb_r6,
        "r5",   R5,     &pcb.pcb_r5,
        "r4",   R4,     &pcb.pcb_r4,
        "r3",   R3,     &pcb.pcb_r3,
        "r2",   R2,     &pcb.pcb_r2,
        "r1",   R1,     &pcb.pcb_r1,
        "r0",   R0,     &pcb.pcb_r0,
};
#else	KDB
REGLIST reglist [] = {
        "error",UERROR, 0,
        "icscs",ICSCS,  0,
        "iar",  IAR,    0,
        "mq",   MQ,     0,
        "r15",  R15,    0,
        "r14",  R14,    0,
        "r13",  R13,    0,
        "r12",  R12,    0,
        "r11",  R11,    0,
        "r10",  R10,    0,
        "r9",   R9,     0,
        "r8",   R8,     0,
        "r7",   R7,     0,
        "r6",   R6,     0,
        "r5",   R5,     0,
        "r4",   R4,     0,
        "r3",   R3,     0,
        "r2",   R2,     0,
        "r1",   R1,     0,
        "r0",   R0,     0,
};
#endif	KDB
/*#define	NUM_REGS_IN_REGLIST	20*/
#define	NUM_REGS_IN_REGLIST	18

int	num_regs_in_reglist	= NUM_REGS_IN_REGLIST;
char            lastc;
INT             fcor;
INT             signo;
INT             sigcode;
L_INT           dot;
L_INT           var[];
STRING          symfil;
STRING          corfil;
INT             pid;
L_INT           adrval;
INT             adrflg;
L_INT           cntval;
INT             cntflg;
STRING          signals[] = {
                "",
                "hangup",
                "interrupt",
                "quit",
                "illegal instruction",
                "trace/BPT",
                "IOT",
                "EMT",
                "floating exception",
                "killed",
                "bus error",
                "memory fault",
                "bad system call",
                "broken pipe",
                "alarm call",
                "terminated",
                "signal 16",
                "stop (signal)",
                "stop (tty)",
                "continue (signal)",
                "child termination",
                "stop (tty input)",
                "stop (tty output)",
                "input available (signal)",
                "cpu timelimit",
                "file sizelimit",
                "signal 26",
                "signal 27",
                "signal 28",
                "signal 29",
                "signal 30",
                "signal 31",
};

#define	GETREG(i)  (*(reglist[num_regs_in_reglist - i - 1].rkern))
extern	ADDR	userpc;

char Error_buf[128];

char *getNameFromText();
ADDR nextPc(), getRegVal(), srchForRegs(), validSavedLink();

/* Number of words we are will search down instruction space for tracetable */
#define MAXTRACECNT	4000

/* Get left and right nibbles of a char. */
#define LNIBBLE(c)	(((c & 0xF0) >> 4) & 0x0F)
#define RNIBBLE(c)	(c & 0x0F)

#define NOTFOUND	0
#define ONWORD		1
#define ONHALFWORD	2

#define NOT		!
#define ERROR_OK	1

/* Over estimate of trace table size in words. */
#define TTSIZE	8
static union TTbuf {		/* Buffer to offload t-table from text. */
	L_INT	tt_wbuf[TTSIZE];			/* Word access. */
	char	tt_cbuf[TTSIZE * sizeof(L_INT)];	/* Byte access. */
} TTbuf;

		/* Max no. word pairs to search up stack before quitting.
		 * Multiply by 2 for number of words. */
#define MAXFRAMESIZ	1500
extern int errno;

#define UPPERSTACK	(pid ? (USRSTACK)  : datmap.e2)
#define LOWERSTACK	(pid ? StackBottom : datmap.b2)
	/* Is the address x a valid stack address? */
#define inStack(x)	(LOWERSTACK <= (x) && (x) < UPPERSTACK)

#define UPPERDATA	(pid ? txtmap.e2 : datmap.e1)
#define LOWERDATA	(pid ? txtmap.b2 : datmap.b1)
	/* Is the address x a valid data address? */
#define inData(x)	(LOWERDATA <= (x) && (x) < UPPERDATA)

#define UPPERINSTR	txtmap.e1
#define LOWERINSTR	txtmap.b1
	/* Is the address x a valid instruction address? */
#define inInstr(x)	(LOWERINSTR <= (x) && (x) < UPPERINSTR)

int getwordCounter;	/* Debugging and profiling adb. */
ADDR validLinkReg();

/* MAXTRACE is the max number of tracebacks to take if cntval has
 * not been set by user (e.g.  ,N$c). */
#define MAXTRACE	24

/* Fudgy way of determining if control is still processing call prologue.
 * inPrologue should evaluate to true if the pc is below the address
 * where the drop of the frame pointer occurs. This info is used
 * to determine whether the active frame is incomplete. */
/* #define inPrologue(pc,functionEntry)	(pc < functionEntry + 12) */
 
/* Register containing return address of caller. */
#define LINKREG		15


/* Register containing the constant table pointer. */
#define CONST_TABL_REG	14

/* Register containing the stack pointer. */
#define STACKREG	1

/* Absolute max num of autos to print. For pretty sakes, make it multple
 * of 8. */
#define MAXAUTOWORDS	64

struct funcName { 			/* info about current function. */
	char		*fn_name;	/* Function's name. */
	struct nlist	*fn_symPtr;	/* Ptr to functions symtab entry. */
} FuncName, *getFuncName();

	

/* Macro to get to top of current frame. */
#define topOfFrame(fp,offset) (fp + offset)

/* Address of 1st arg word.
 * size area on top of callee's frame which can contain <= ARG_SAVE_SZ
 * arg words.  Words always start at the lowest address of this area and
 * continue toward higher addresses into the previous frame. */

#define firstArgOnFrame(fp,offset) (topOfFrame(fp,offset) - ARG_SAVE_SZ)


#define GETWORD(addr,space)	(access(RD, addr, space, 0))
#define WORDEQ(w1,w2,mask)	((w1 & mask) == w2)

	/* Sanity check - frame pointers are always in these registers: */
#define validFrameRegNum(frameRegNum) (frameRegNum == 1 || frameRegNum == 13)


/* Adb will only do tracebacks throught functions exibiting a certain
 * routine type code. This macro tests for these. */
#define validRtype(rtype) (rtype == 2 || rtype == 3 || rtype == 7)

/* Based on the tracetable rtype, is the iar in a glue routine. */
#define inGlueRoutine(rtype)	(rtype == 2)

	/* Sanity check - given a number, can it be a register number? */
#define validRegNum(regnum)	(0 <= regnum && regnum <= 15)

	/* Location of regnum in the relist array.
	 * DAMN BOGUS MAGIC NUMBERS!			*/
#ifdef	KDB
#define regInRegList(regnum)	(2 + 15 - regnum)
#else	KDB
#define regInRegList(regnum)	(4 + 15 - regnum)
#endif	KDB

ADDR pcOffStack();
	
ADDR CallersFp;		/* Fp of previous function, via r15. */
L_INT ConstTblPtr;	/* Data ptr from register or stack. */


#ifdef	dumb_code
/* Get stack address of a saved register. */
#define getSavedRegAddr(fp,offset,regnum)\
	(topOfFrame(fp,offset) + REG_OFFSET + regnum * sizeof(int))
#else	dumb_code /* RNS bug fix for variable size register frames.*/

int getSavedRegAddr(fp,tt,regnum)
ADDR fp;
register TTABLE *tt;
int regnum;
{
#define	MAX_REG_NUM	15    
    int regbottom= topOfFrame(fp,tt->local_offset)
    		   - ARG_SAVE_SZ
		   - LINK_SAVE_SZ
		   - (MAX_REG_NUM - tt->first_gpr + 1) * sizeof(int);

    if (regnum<tt->first_gpr)
        printf("r%d not saved on stack.\n",tt->first_gpr);
    dprintf(("getSavedRegAddr:R%d saved at addr 0x%x on stack\n",regnum,
    	    regbottom + ((regnum - tt->first_gpr) * sizeof (int))));
    return(regbottom + ((regnum - tt->first_gpr) * sizeof (int)));
}
    
#endif	dumb_code
    
    


/* general printing routines ($) */
printtrace(modif)
{
        INT             i;
        REG BKPTR       bkptr;
        STRING          comptr;
        register struct nlist *sp;
        INT             stack;
        IF cntflg==0 THEN cntval = -1; FI
        switch (modif) {
#ifdef	KDB
#if	MACH
	    case 'p':
		/*
		 *	pid$p	sets process to pid
		 *	$p	sets process to the one that invoked
		 *		kdb
		 */
		{
			struct proc 	*p;
			extern struct proc *	pfind();

			extern void	kdbgetprocess();

			if (adrflg) {
				p = pfind(dot);
			}
			else {
				p = pfind(curpid);	/* pid at entry */
			}
			if (p) {
				printf("proc entry at 0x%X\n", p);
				kdbgetprocess(p, &curmap, &curpcb);
				var[varchk('m')] = (int)curmap;
				pid = p->p_pid;
				if (pid == curpid) {
					/*
					 * pcb for entry process is saved
					 * register set
					 */
					 curpcb = &pcb;
				}
			}
			else {
				printf("no such process");
			}
		}
		break;

#endif	MACH
#else	KDB
            case '<':
                IF cntval == 0
                THEN    WHILE readchar() != EOR
                        DO OD
                        lp--;
                        break;
                FI
                IF rdc() == '<'
                THEN    stack = 1;
                ELSE    stack = 0; lp--;
                FI
                                                        /* fall through */
            case '>':
                {CHAR           file[64];
                CHAR            Ifile[128];
                extern CHAR     *Ipath;
                INT             index;
                index=0;
                IF rdc()!=EOR
                THEN    REP file[index++]=lastc;
                            IF index>=63 THEN error(LONGFIL); FI
                        PER readchar()!=EOR DONE
                        file[index]=0;
                        IF modif=='<'
                        THEN    IF Ipath THEN
                                        (void) strcpy(Ifile, Ipath);
                                        (void) strcat(Ifile, "/");
                                        (void) strcat(Ifile, file);
                                FI
                                IF strcmp(file, "-")!=0
                                THEN    iclose(stack, 0);
                                        infile=open(file,0);
                                        IF infile<0
                                        THEN    infile=open(Ifile,0);
                                        FI
                                ELSE    lseek(infile, 0L, 0);
                                FI
                                IF infile<0
                                THEN    infile=0; error(NOTOPEN);
                                ELSE    IF cntflg
                                        THEN    var[9] = cntval;
                                        ELSE    var[9] = 1;
                                        FI
                                FI
                        ELSE    oclose();
                                outfile=open(file,1);
                                IF outfile<0
                                THEN    outfile=creat(file,0644);
#ifndef EDDT
                                ELSE    lseek(outfile,0L,2);
#endif
                                FI
                        FI
                ELSE    IF modif == '<'
                        THEN    iclose(-1, 0);
                        ELSE    oclose();
                        FI
                FI
                lp--;
                }
                break;
            case 'p':
                IF kernel == 0 THEN
                        printf("not debugging kernel\n");
                ELSE
#ifdef vax
                        IF adrflg THEN
                                int pte = access(RD, dot, DSP, 0);
                                masterpcbb = (pte&PG_PFNUM)*512;
                        FI
                        getpcb();
#endif
#ifdef ibm032
                        IF adrflg THEN
                                masterpcbb = dot;
#ifdef DEBUG
				if (debug) printf("masterpcbb = %X\n", masterpcbb);
#endif
                        FI
                        getproc();
#endif
                FI
                break;
#endif	KDB

            case 'd':
                if (adrflg) {
                        if (adrval<2 || adrval>16){
				printf("must have 2 <= radix <= 16");
				break;
			}
                        printf("radix=%d base ten",radix=adrval);
                }
                break;

#ifdef	KDB
#else	KDB
            case 'q': case 'Q': case '%':
                done();
#endif	KDB

#ifdef	KDB
	    case 'w':
#else	KDB
            case 'w': case 'W':
#endif	KDB
                maxpos=(adrflg?adrval:MAXPOS);
                break;

            case 's': case 'S':
                maxoff=(adrflg?adrval:MAXOFF);
                break;

            case 'v': case 'V':
                prints("variables\n");
                FOR i=0;i<=35;i++
                DO IF var[i]
                   THEN printc((i<=9 ? '0' : 'a'-10) + i);
                        printf(" = %X\n",var[i]);
                   FI
                OD
                break;

#if	MACH
#else	MACH
            case 'm': case 'M':
                printmap("? map",&txtmap);
                printmap("/ map",&datmap);
                break;
#endif	MACH

            case 0: case '?':
                IF pid
                THEN printf("pcs id = %d\n",pid);
                ELSE prints("no process\n");
                FI
                sigprint(); flushbuf();

            case 'r': case 'R':
               	printregs();
                break;
	
#ifdef DEBUG
case 'Y':
	{
	int t;
	errno = 0;
	t = GETWORD(adrval,ISP);
	printf("errno=%d word=%d\n", errno, t);
	}
	break;

case 'X':	/* Hindsight:should use the adrval to set debug levels. */
	debug = !debug;		/* Switch debug off/on */
	break;
#endif

#ifdef	KDB
#else	KDB
	case 'D':		/* Dump entire symbol table(D). */
	{
		struct nlist *sp;
		for (sp = symtab; sp < esymtab; sp++) {
			int type = sp->n_type;
			printf("%-18s %10X %d ",
					sp->n_un.n_name,sp->n_value,type);
			if((type & 0xfe) == 0x1f)
				printf("%s", "filename");
			else {
				if((type & 0xfe) == 0x2)
					printf("%s","abs ");
				else if((type & 0xfe) == 0x4)
					printf("%s", "text");
				else if((type & 0xfe) == 0x6)
					printf("%s", "data");
				else if((type & 0xfe) == 0x8)
					printf("%s", "bss ");
				else if((type & 0xfe) == 0x12)
					printf("%s", "comm");
				if((type & 0x1) == 0x1)
					printf("%s", " extern");
			}
			printf("\n");		
		}
	}
	break;
#endif	KDB

	case 'c':	/* Backtrace. Expect problems if STACKREG */
	case 'C':	/* is used and still in prologue. see adb(1). */
		printf("This does not work, and may well crash the system...\n");
		printf("   Use \"50$c\" to go ahead anyway.\n");
		if (cntval < 50)
			break;
		else
			cntval -=50;

		{
		int maxTrace = cntval >= 0 ? cntval : MAXTRACE;
		if(adrflg)
			(void) traceFromStackAddr(adrval, maxTrace, modif);
		else
			(void) traceFromRegs(maxTrace,modif);
			
		break;
		}
	
#ifdef	KERNEL
	KDB_PRINT_CASES
#endif	KERNEL

	case 'e':  case 'E':
#ifdef	KDB
	sp = symtab;
	for (ALLSYMS(sp)) {
		printf("%.*s:%12t%R\n", SYMNAMELEN, sp->n_un.n_name, get(SYMVAL(sp->n_value),DSP));
	}
#else	KDB
              for (sp = symtab; sp < esymtab; sp++) {
                 if (sp->n_type==(N_DATA|N_EXT) ORF sp->n_type==(N_BSS|N_EXT))
                      printf("%s:%12t%R\n", sp->n_un.n_name, get(sp->n_value,DSP));
              }
#endif	KDB
              break;

        case 'a': case 'A':
                error("No algol 68 here");

        /*print breakpoints*/
        case 'b': case 'B':
                printf("breakpoints\ncount%8tbkpt%24tcommand\n");
                for (bkptr=bkpthead; bkptr; bkptr=bkptr->nxtbkpt)
                        if (bkptr->flag) {
                                printf("%-8.8d",bkptr->count);
                                psymoff(bkptr->loc,ISYM,"%24t");
                                comptr=bkptr->comm;
                                WHILE *comptr DO printc(*comptr++); OD
                        }
                break;

        default: error(BADMOD);
        }
}

#ifdef	KDB
#else	KDB
printmap(s,amap)
STRING  s; MAP *amap;
{
        int file;
        file=amap->ufd;
        printf("%s%12t`%s'\n",s,(file<0 ? "-" : (file==fcor ? corfil : symfil)));
        printf("b1 = %-16R",amap->b1);
        printf("e1 = %-16R",amap->e1);
        printf("f1 = %-16R",amap->f1);
        printf("\nb2 = %-16R",amap->b2);
        printf("e2 = %-16R",amap->e2);
        printf("f2 = %-16R",amap->f2);
        printc(EOR);
}
#endif	KDB

printregs()
{
        REG REGPTR      p;
        REG L_INT       v;
	REG int		firstreg, lastreg;

#ifdef	KDB
#else	KDB
	if(!kcore && !adrflg) {
		printf("%s%6t%R -> %R\n", "stack",
			USRSTACK,(pid?StackBottom:datmap.b2));
		printf("%s%6t%R -> %R\n",
			"data", DATABASE, (pid?DataTop:datmap.e2));
	}
#endif	KDB

	if(adrflg) {
		lastreg = cntflg ? (adrval + cntval) : (adrval + 1);
		if(!validRegNum(adrval) || !validRegNum(lastreg-1)) {
			printf("Error: %D-%D not in range.\n",adrval,lastreg);
			return;
		}
			/* Note: regs in descending order in reglist */
		firstreg = regInRegList(adrval) + 1;	/* Last printed. */
		lastreg = cntflg ? (adrval + cntval) : (adrval + 1);
		lastreg = regInRegList(lastreg) + 1;	/* 1st printed. */
	}
	else {
		firstreg = NUM_REGS_IN_REGLIST;	/* Ugly constant == elements in reglist. */
		lastreg  = 0;	/* All C arrays start at zero. */
	}
        FOR p = &reglist[lastreg]; p < &reglist[firstreg]; p++
       	DO      v = kcore ? *p->rkern:*(ADDR *)(((ADDR)&u)+p->roffs);
               	printf("%s%6t%R %16t", p->rname, v);
                valpr(v,(p->roffs==PC?ISYM:DSYM));
       	        printc(EOR);
        OD
/*RNS Debug */
#ifdef	notdef
	{
	    int i;
	    for (i=lastreg;i<firstreg;i++) {
		int v= getRegVal(i);
               	printf("%s%6t%R %16t", reglist[i].rname,v);
		valpr(v,(reglist[i].roffs==PC?ISYM:DSYM));
		printc(EOR);
	    }
	}
#endif	notdef
        printpc();
}

getreg(regnam) {
        REG REGPTR      p;
        REG STRING      regptr;
        CHAR    *olp;
        olp=lp;
        FOR p=reglist; p < &reglist[24]; p++
        DO      regptr=p->rname;
                IF (regnam == *regptr++)
                THEN
                        WHILE *regptr
                        DO IF (readchar()) != *regptr++
                                THEN --regptr; break;
                                FI
                        OD
                        IF *regptr
                        THEN lp=olp;
                        ELSE
                                int i = kcore ? (int)p->rkern : p->roffs;
                                return (i);
                        FI
                FI
        OD
        lp=olp;
        return(0);
}
printpc()
{
#ifdef	KDB
	dot = pcb.pcb_iar;
#else	KDB
        dot= *(ADDR *)(((ADDR)&u)+PC);
#endif	KDB
        psymoff(dot,ISYM,":%16t"); printins(dot,chkget(dot,ISP));
        printc(EOR);
}

char    *illinames[] = {
        "reserved addressing fault",
        "priviliged instruction fault",
        "reserved operand fault"
};

char    *fpenames[] = {
	"floating operation invalid",
	"floating overflow exception",
	"floating underflow exception",
	"floating divide by zero exception",
	"floating inexact result exception",
	"integer divide by zero exception"
};

sigprint()
{
        IF (signo>=0) ANDF (signo<sizeof signals/sizeof signals[0])
        THEN prints(signals[signo]); FI
        switch (signo) {
        case SIGFPE:
                IF (sigcode >= 0 &&
                    sigcode < sizeof fpenames / sizeof fpenames[0]) THEN
                        prints(" ("); prints(fpenames[sigcode]); prints(")");
                FI
                break;
        case SIGILL:
                IF (sigcode >= 0 &&
                    sigcode < sizeof illinames / sizeof illinames[0]) THEN
                        prints(" ("); prints(illinames[sigcode]); prints(")");
                FI
                break;
        }
}


/* Arbitrary distance (bytes) from frame top to handcrafted frame pointer. */
#define ARB_FP_DISPLACEMENT	1000

/* Set up for a stackWalk. Starting at stackAddr, scan up stack to find a
 * stored r14/r15 combo. Pass off info to stackWalk(). */
traceFromStackAddr(stackAddr, maxTrace, modif)
ADDR stackAddr;
int maxTrace, modif;
{
	register TTABLE *tt = &TraceTable;	/* Pointer to trace table. */
	register ADDR framePtr;			/* Current frame pointer. */
	register ADDR pc;			/* Text address. */
	ADDR stackPtr;				/* frame's stackptr. */

dprintf(("traceFromStackAddr(0x%x,%d,%d)\n",stackAddr,maxTrace,modif));

	
					/* Scan up stack for r14/r15 pair. */
	if((pc = pcOffStack(stackAddr)) == 0) {
		printf("Can't trace - absurd stack!\n");
		return(0);
	}
	
	/* Handcraft a frame pointer. Calc to top of frame from address of
	 * saved r15, then drop an arb amount. As long as dropped distance
	 * includes the auto area of frame, all should be ok. */
	framePtr = (AddrR15 + sizeof(int)
		+ LINK_SAVE_SZ + ARG_SAVE_SZ) - ARB_FP_DISPLACEMENT;

	if(getTraceTable(pc) == 0)
		return(0);
				/* Handcraft trace table. */
	tt->local_offset = ARB_FP_DISPLACEMENT;
	
				/* Handcraft stack pointer. Excessive, but
				 * only needed for printing autos. Will be
				 * truncated by the printAuto function. */
	stackPtr = ARB_FP_DISPLACEMENT;
					/* Try to walk up stack. */
	return(stackWalk(pc, framePtr, stackPtr, tt,  maxTrace, modif));
}

/* Set up for a stack walk using register and stack info as starting point.
 * As per the caveat in adb(1), funny things can happen if the pc is
 * somewhere in prologue because the stack is incomplete at this point. */
traceFromRegs(maxTrace, modif)
int maxTrace, modif;
{
	register TTABLE *tt = &TraceTable;
	register ADDR framePtr, tmpAddr, entryAddr;
	register ADDR stackPtr;
	struct funcName *fn;

dprintf(("traceFromRegs(%d,%d)\n",maxTrace,modif));

	if(getTraceTable(*(ADDR *)(((ADDR)&u)+PC)) == 0)
		return(0);
	dprintf(("\t+tracefromregs:tt->frame_reg=%D\n",tt->frame_reg));
	dprintf(("\t+tracefromregs:regval(framereg)=%X\n",getRegVal(tt->frame_reg)));
	if(inGlueRoutine(tt->rtype)) {	/* Special case: type 2 functions. */
		tmpAddr = getRegVal(LINKREG);	    /* Get return address. */
		
			/* "Normal" type2 should not have linkreg in stack,
			 * but if the LINKREG does not look reasonable, then
			 * try to get link addr off stack and see if it looks
			 * reasonable. */
		if(validSavedLink(tmpAddr, ERROR_OK) == 0) {
			/* Next statement is a special getSavedRegAddr(). */
			tmpAddr = getRegVal(STACKREG)	/* Top of frame.  */
				+ REG_OFFSET		/* Down to regsv. */
				+ LINKREG * sizeof(int);/* Up to linkreg. */
				
				/* Try to get linkreg off stack. */
			tmpAddr = GETWORD(tmpAddr, DSP);
/* RNS */		if (errflg) return(0);
			
				/* Test for reasonable LINKREG address. */
			validSavedLink(tmpAddr, NOT ERROR_OK);
		}
		fn = getFuncName((*(ADDR *)(((ADDR)&u)+PC)));
		tt->param_words = 0;
		printArgs(fn->fn_name, (ADDR) 0, tt, fn->fn_symPtr);

				/* Get caller's tracetable. */
		if(getTraceTable(tmpAddr) == 0)
			return(0);
	}
	framePtr = getRegVal(tt->frame_reg);
	stackPtr = getRegVal(STACKREG);
	dprintf(("\t+regtrace:stackPtr=%X\n",stackPtr));
#ifdef	dumb_code
	tmpAddr = getSavedRegAddr(framePtr, tt->local_offset, LINKREG);
#else	dumb_code
	tmpAddr = getSavedRegAddr(framePtr, tt, LINKREG);
#endif	dumb_code	
	dprintf(("\t+addr of link reg=%X\n",tmpAddr));
	dprintf(("\t+word at link reg addr=%X\n",GETWORD(tmpAddr,DSP)));
	tmpAddr = GETWORD(tmpAddr,DSP);
/*RNS*/ if (errflg) return(0);
	ConstTblPtr = getRegVal(CONST_TABL_REG);
	if((entryAddr = validSavedLink(tmpAddr, NOT ERROR_OK)) == 0)
		return(0);
	return(stackWalk(entryAddr, framePtr, stackPtr, tt, maxTrace, modif));
}
	
/* Given the entry point of a function, the associated frame pointer and
 * trace table, march up to top of stack printing available info. */
stackWalk(pc, framePtr, stackPtr, tt, maxTrace, modif)
register ADDR pc;		/* Current function's entry address. */
register ADDR framePtr;
ADDR stackPtr;
register TTABLE *tt;
int maxTrace;			/* Max no. frames to walk up. */
int modif;			/* Has to be 'c' or 'C'. */
{
	register int activeFrame;
	register struct funcName *fn;

	dprintf(("\t+stackwalk: pc=%X framePtr=%X\n", pc, framePtr));
 	for(activeFrame=0; activeFrame < maxTrace; ++activeFrame) {
		fn = getFuncName(pc);
		printArgs(fn->fn_name, framePtr, tt, fn->fn_symPtr);
		if(modif == 'C') {
			printAutos(framePtr, stackPtr, tt);
			/* Next frame's stackPtr is top of current frame. */
			stackPtr = topOfFrame(framePtr,tt->local_offset);
		}

		if((pc = nextPc(CalledFromAddr, framePtr, tt, pc)) == 0)
			return(0);
		framePtr = CallersFp;
	}
	return(1);	/* Success (if caller cares). */
}

/* Given a return address, current frame pointer and a current trace table,
 * Output the calling function's framepointer (CallersFp) and
 * the entry address of the caller. "Next" frame refers to next highest
 * frame on the stack. Next pc, return address-balix of associated funct. */
ADDR
nextPc(retAddr, framePtr, tt, oldEntryAddr)
register ADDR retAddr, framePtr, oldEntryAddr;
register TTABLE *tt;
{
	register ADDR regAddr;
	register ADDR linkAddr, newEntryAddr;
	TTABLE oldTT;
	oldTT = TraceTable;		/* Save the current trace table. */
/* Start switching reference to next frame up stack. */
/* Get next trace table, if routine type indicates _.start, then done. */
	if(getTraceTable(retAddr) == 0 || tt->rtype == 3)
		return(0);
#ifdef	dumb_code		
	regAddr = getSavedRegAddr(framePtr,oldTT.local_offset,tt->frame_reg);
#else	dumb_code
	regAddr = getSavedRegAddr(framePtr,&oldTT,tt->frame_reg);
#endif	dumb_code	
	CallersFp = GETWORD(regAddr, DSP);
	dprintf(("\t+nextPc:framePtr=%X\n",framePtr));
dprintf(("\t+nextPC:regAddr(prevFrPtr)=%X callersfp=%X\n",regAddr,CallersFp));
						/* Next link address (pc). */
#ifdef	dumb_code						
	regAddr =   getSavedRegAddr(CallersFp, tt->local_offset, LINKREG);
#else	dumb_code
	regAddr =   getSavedRegAddr(CallersFp, tt, LINKREG);
#endif	dumb_code
	linkAddr =  GETWORD(regAddr, DSP);
dprintf(("\t+nextPC:linkAddr=%X regAddr=%X Callersfp=%X\n",linkAddr,regAddr,CallersFp));
#ifdef	dumb_code
	regAddr = getSavedRegAddr(framePtr,oldTT.local_offset,CONST_TABL_REG);
#else	dumb_code
	regAddr = getSavedRegAddr(framePtr,&oldTT,CONST_TABL_REG);
#endif	dumb_code	
	ConstTblPtr = GETWORD(regAddr, DSP);
				/* Really get & validate next link address. */
	if((newEntryAddr = validSavedLink(linkAddr, NOT ERROR_OK))==(ADDR) 0)
		return(0);

/* Verify: will the next frame's entry address equal the first word in its
 * constant table.  Get the stored CONST_TABLE_REG off the stack
 * and follow it to get the const table value at _foo. */
	if(validSavedConstPtr(oldEntryAddr,newEntryAddr,framePtr,&oldTT,tt)==0)
		return(0);
	return(newEntryAddr);
}

/* Given a text address, find the name of the function. */
struct funcName *
getFuncName(pc)
register ADDR pc;
{
	static char nameBuf[16];	/* Holds 10 hex chars (0x00000000). */
	FuncName.fn_symPtr = (struct nlist *) 0;   /* Null if no sym tabl. */
	FuncName.fn_name   = (char *) 0;	   /* Always expect a name. */
	if (symTableIsPresent()) {
		staticSym(pc);
		FuncName.fn_symPtr = cursym;
		FuncName.fn_name = FuncName.fn_symPtr->n_un.n_name;
	}
	else
		FuncName.fn_name = getNameFromText(pc);
		
	if(FuncName.fn_name == (char *) 0) {
		(void) sprintf(nameBuf, "%#x", pc);
		FuncName.fn_name = nameBuf;
	}
	return(&FuncName);
}

printArgs(name, framePtr, tt, symPtr)
char *name;
register ADDR framePtr;
TTABLE *tt;
struct nlist *symPtr;
{
	register int thereAreArgwords = tt->param_words;
	register ADDR argAddress = firstArgOnFrame(framePtr,tt->local_offset);
	
#ifdef	KDB
	printf("%.8s(", name);
#else	KDB
	printf("%s(", name);
#endif	KDB
	for(;thereAreArgwords; --thereAreArgwords) {
		printf("%X", GETWORD(argAddress, DSP));
		if(thereAreArgwords > 1)
			printf(", ");
		argAddress += sizeof(int);
	}
	printf(")  Called from ");
	psymoff(CalledFromAddr, ISP, "");
	dprintf(("+symPtr=%X\n",symPtr));
#ifdef	KDB
#else	KDB
	printf("%s\n", 
		(symPtr && (symPtr->n_type&N_EXT)==0) ? ", [static]" : "");
#endif	KDB
}

/* Get the value stored in registerN. There are 8 reglist elements into
 * reglist[] to r15 element. */
ADDR
getRegVal(regNum)
{
	register REGLIST *p = &reglist[regInRegList(regNum)];
	return(kcore ? *(ADDR *)p->rkern : *(ADDR *)(((ADDR)&u) + p->roffs));
}

/* Print autos. Does not work if prologue is incomplete - let the user have
 * it anyway. */
printAutos(framePtr, stackPtr, tt)
ADDR framePtr;
register ADDR stackPtr;
register TTABLE *tt;
{
	register ADDR fromAddr, toAddr;
	register int i;

		/* Find lowest addr of regsave area and 1 wrd lower. */
#ifdef	dumb_code		
	toAddr = getSavedRegAddr(framePtr,tt->local_offset,tt->first_gpr);
#else	dumb_code
	toAddr = getSavedRegAddr(framePtr,tt,tt->first_gpr);
#endif	dumb_code	
	dprintf(("\t+printautos:framptr=%X stackPtr=%X\n",framePtr,stackPtr));
	fromAddr = toAddr - MAXAUTOWORDS * sizeof(int);
	fromAddr = stackPtr > fromAddr ? stackPtr : fromAddr;
	
		/* Start from lowest reg saved, print downward. */
	for(i=0; fromAddr < toAddr; fromAddr += sizeof(int), ++i) {
		if(i == 0)
			printf("\tAutomatics from %X to %X",
					fromAddr, toAddr - sizeof(char));
		printf("%s%X",
			(i%8)!=0 ? ", " : "\n\t  ",
			GETWORD(fromAddr,DSP));
	}
			
	printf("\n");
}

#define STM_OPCODE 0xd9000000
#define STM_MASK   0xff000000

/* Given a functions entry address and the new (next) and old (prev) trace
 * tables, determine if the first word in the constant pool is the entry
 * address of the function. */
validSavedConstPtr(oldEntryAddr, newEntryAddr, framePtr, oldTT, newTT)
register ADDR oldEntryAddr, newEntryAddr;
ADDR framePtr;
register TTABLE *oldTT, *newTT;
{
	register int firstGpr;			/* First gpr saved. */
	L_INT instr;			    /* Hopefully a stm instruction. */
	ADDR entryAddrFromConstTabl;
	
	instr = GETWORD(oldEntryAddr, ISP);
	if(!WORDEQ(instr, STM_OPCODE, STM_MASK)) {  /* stm instruction? */
		dprintf(("\t+Got %X instead, addr=%X\n",instr,oldEntryAddr));
		error("Expecting stm instruction - not found!");
	}
		
		/* ASSERTION:The current adb trace back code relies on the
		 *  frame register being saved on the stack between calls. */
	firstGpr = (instr >> 20) & 0x0000000f;	/* Reg. # of 1st saved GPR. */
	if(newTT->frame_reg < firstGpr) {
		error("Expecting frame register on stack-not there");
	}
				/* Get entry address at _foo. */
	entryAddrFromConstTabl = GETWORD(ConstTblPtr,DSP);
	
	dprintf(("\t+cons entry=%X\n",entryAddrFromConstTabl));
	dprintf(("\t+new entry=%X\n",newEntryAddr));
	
				/* Validate (_foo) == entry address. */
	if(entryAddrFromConstTabl != newEntryAddr) {
		sprintf(Error_buf,"Expecting first word of constant pool(%X) to equal entry address(%X)",
			entryAddrFromConstTabl, newEntryAddr);
		error(Error_buf);
	}

	return(1);		/* Looks good up to here. */
}


/* Given a stack addr, return entry address of the function that is
 * using what appears to be the frame immediately above the input
 * address. Frame determination is made by looking for a r14/r15 save
 * combination on the stack. The r15 (link register) is dereferenced
 * into the callers text to get the entry address of the called
 * function (operand of bal* instruction). */
ADDR
pcOffStack(addr)
ADDR addr;
{
	ADDR entryVal;
	dprintf(("\t+Looking for pc on stack, start srch at %X\n",addr));
	entryVal = srchForRegs(addr);
	dprintf(("\t+stack wrds read to r14-15 combo=%D\n",getwordCounter));
	return(entryVal);
}
	
/* Search up stack for plausible r14 and r15 starting at stack
 * address == saddr. Leap frog 2 words at a time to save time.
 * Return entry point of function on whose frame is the stored r15. */
ADDR
srchForRegs(saddr)
register ADDR saddr;
{
	register ADDR w, w1, w2;
	register int i = MAXFRAMESIZ;
	ADDR oldAddr = saddr;
	getwordCounter=0;
	saddr = WORDALIGN(saddr);
	if(!inStack(saddr)) {
#ifdef	KDB
#else	KDB
		sprintf(Error_buf,"Error:%X not in stack range %X to %X\n",
			saddr, UPPERSTACK-sizeof(int), LOWERSTACK);
#endif	KDB
		error(Error_buf);
	}
	errno = 0;
	for(i=0; saddr < USRSTACK && --i; saddr += sizeof(int) * 2) {
#ifdef DEBUG
		++getwordCounter;
#endif
		w = GETWORD(saddr, DSP);
			/* Observation: stacks can contain lots of zeroes.
			 * The srch values never zero so ignore 0s. */
		if(w == 0)
			continue;
			/* Check errno from ptrace in access(). */
		if(w == -1 && errno != 0)
			break;
#ifdef DEBUG
		debug_printtype(w,saddr);
#endif
		if(inInstr(w)) {
			ConstTblPtr = w1 = GETWORD(saddr - sizeof(int), DSP);
			if(inData(w1) && 
					(w2=validSavedLink(w,NOT ERROR_OK))) {
				dprintf(("\t+!FOUND:stack addr=%X",saddr));
				dprintf((" link addr on stack(r15)=%X", w));
				dprintf((" data addr on stack(r14)=%X",w1));
				dprintf((" link addr points to %X\n",w2));
				AddrR15 = saddr;
				return(w2);
			}
		}
		if(inData(w)) {
			ConstTblPtr = w;
			w1 = GETWORD(saddr + sizeof(int), DSP);
			if(inInstr(w1) &&
					(w2=validSavedLink(w1,NOT ERROR_OK))){
				dprintf(("\t+!FOUND:stack addr=%X",saddr));
				dprintf((" link addr on stack(r15)=%X", w1));
				dprintf((" data addr on stack(r14)=%X",w));
				dprintf((" link addr points to %X\n",w2));
				AddrR15 = saddr + sizeof(int);
				return(w2);
			}
		}
	}

/* Error-Warning msgs: */
	dprintf(("\t+reg search broke %D %X %X\n",i,saddr,w));
	if(i == 0)
		printf("Searched %d word pairs on stack - no r14-r15 pair",
			MAXFRAMESIZ);
	if(errno && w == -1)
		printf("stack scan from %X to non-stack address=%X",
			oldAddr, saddr);
	if(saddr >= USRSTACK)
		printf("Hit the top of the stack");
	error("Back trace attempt halted!");
	/* NOTREACHED */
}

	/* Functions to test which form of the bal* instruction
	 * is present in text. */
	 
/* Bal* instructions to look for in text. */
#define BALIX	0x8d000000
#define BALI	0x8c000000
#define BALRX	0xed000000

ADDR tryBalix(), tryBalrx();	/* Bal* search functions. */
struct balTest {
	ADDR (*tryBal)();
	unsigned int instr;
} balTest[] = {			/* Order may be important. */
	tryBalix, BALIX,
	tryBalrx, BALRX,
	tryBalix, BALI,
	(ADDR (*)())0
};

char *Bal_error = "Error:bal%s expected %s %X, but not there (is iar in prologue?)!";

/* Input: a link value (as in a saved r15 off stack) and
 * a pointer into a function's data (const) area (global).
 * Return the entry address of the function on whose frame we
 * found the stored r15. Return zero if something is wrong.
 * Set the address of the bal* instruction we were called from
 * in CalledFromAddr as a side effect. */

/* WARNING: this function is myopic, assumes all C calls are from
 * certain bal* instructions. May not be true in the future. */
ADDR
validSavedLink(linkAddr, errorOk)
register ADDR linkAddr;
register int errorOk;
{
	register ADDR tmpAddr, (*tryFunct)();
	register int i;
	for(i=0; (tryFunct = balTest[i].tryBal); i++) {
		if((tmpAddr = tryFunct(linkAddr,balTest[i].instr))) {
			return(tmpAddr);
		}
	}
	if(errorOk)	
		return((ADDR) 0);
	balError("*", "near", linkAddr);	/* Bail out! */
	/* NOTREACHED */
}

ADDR
tryBalrx(linkAddr, instr)
register ADDR linkAddr;
register unsigned instr;
{
	register ADDR entryAddr;
	register L_INT bal_instr;

	CalledFromAddr = linkAddr - sizeof(int) - 2; /* balrx specific. */
	
	dprintf(("\t+Trying for balrx at %X\n",CalledFromAddr));
	errno = 0;
	bal_instr = GETWORD(CalledFromAddr, ISP);
	if(bal_instr == -1 && errno)
		balError("rx", "at", CalledFromAddr);
	dprintf(("bal_instr=%X errno=%d\n",bal_instr,errno));
	if(WORDEQ(bal_instr, instr, 0xff000000)) {
		dprintf(("\t+tryBalrx:ConstTblPtr=%X\n",ConstTblPtr));
		entryAddr = GETWORD(ConstTblPtr, DSP);
		dprintf(("\t+Got balrx, returning entryaddr=%X\n",entryAddr));
		return(entryAddr);
	}
	return((ADDR) 0);
}

ADDR
tryBalix(linkAddr, instr)
register ADDR linkAddr;
register unsigned int instr;
{
	register ADDR entryAddr;
	L_INT bal_instr;
	dprintf(("\t+Trying for balix at %X\n",linkAddr));
	dprintf(("\t+Old calledfrom addr(1st garbage)=%X\n",CalledFromAddr));
	CalledFromAddr = linkAddr - sizeof(int) * ((instr == BALIX) ? 2 : 1);
	dprintf(("\t+New calledfrom addr=%X\n",CalledFromAddr));
	errno = 0;
	bal_instr = GETWORD(CalledFromAddr, ISP);
	if(bal_instr == -1 && errno)
		balError("ix", "at", CalledFromAddr);
	if(WORDEQ(bal_instr, instr, 0xff000000)) {
				/* Add sign extended address operand to balix addr. */
			entryAddr = CalledFromAddr +
			((((0x000fffff & bal_instr) ^ 0x00080000)
							- 0x00080000) << 1);
		dprintf(("\t+balxx=%X Entryaddr=%X\n",bal_instr,entryAddr));
		return(entryAddr);
	}
	return((ADDR) 0);
}		

balError(s, proximity, addr)
char *s, *proximity;
ADDR addr;
{
	sprintf(Error_buf, Bal_error, s, proximity, addr);
	error(Error_buf);
}
		
#ifdef DEBUG
debug_printtype(w,saddr)
ADDR w, saddr;
{
	dprintf(("\t+testing for r15/r14 combo: "));
	if(debug && inInstr(w))
		printf("text address in stack=%X stackaddr=%X\n",w,saddr);
	if(debug && inData(w))
		printf("data address in stack=%X stackaddr=%X\n",w,saddr);
	if(debug && inStack(w))
		printf("stack address in stack=%X stckaddr%X\n",w,saddr);
}
#endif

	/* Find a trace table in text. Start scan at pc. */
getTraceTable(pc)
ADDR pc;
{
	int foundState;
	ADDR oldPc = pc;
	setDefaults(&TraceTable);	/* Just in case TT is not found. */
	
	pc = WORDALIGN(pc);
	
	if((foundState = scanForTable(&pc)) == NOTFOUND) {
		sprintf(Error_buf,"Scanned from %x to %x - no trace table!",
				oldPc, pc);
		error(Error_buf);
	}
	
	unPack(pc, foundState, &TraceTable, &TTbuf);
	
#ifdef DEBUG
	if(debug) dumpTraceTable(&TraceTable, pc);
#endif	
	return(1);
}

/* Scan text for magic numbers that indicate a trace table, read it in.
 * Replace startingPc with the address at which the trace table is found. */
scanForTable(startingPc)
ADDR *startingPc;
{
	register ADDR pc = *startingPc;
	register L_INT w, prevWord = 0;
	register int count = 0, isfound = NOTFOUND;
	
	dprintf(("\t+Starting TT search at pc(word aligned)=%x\n",pc));
	for(; count++ < MAXTRACECNT; pc += sizeof(int)) {
		if((w = GETWORD(pc, ISP)) == 0) {
			isfound = NOTFOUND;
			break;
		}
		if(prevWord && WORDEQ(w, 0xdf000000, 0xff000000)) {
			isfound = ONHALFWORD;
			break;
		}
		if(WORDEQ(w, 0xdf00df00, 0xff00ff00)) {
			isfound = ONWORD;		/* Word Aligned. */
			break;
		}
		if(WORDEQ(w, 0x0000df00, 0x0000ff00)) {
			prevWord = w;
			continue;
		}
		prevWord = 0;
	}
	
	if(isfound == NOTFOUND)
		return(NOTFOUND);
	if(isfound == ONHALFWORD)
		pc -= sizeof(int);
	*startingPc = pc;	/* Trace table at this text address. */
	return(isfound);
}

/* Read in the trace table from text into the ttbuf. Unpack from
 * ttbuf into the global trace table structure.
 * Unpacking is relative to the byte containing the routine-type.
 */
unPack(pc, found, tt, ttBuf)
register ADDR pc;
int found;
register TTABLE *tt;
register union TTbuf *ttBuf;
{
	register int i;
	char *flgNotImplemented="%c Trace flag found, but not implemented!\n";
	int whichFlag;
	int size;
				/* Read in trace table and a little extra. */
	for(i=0; i < TTSIZE; i++, pc += sizeof(int))
		ttBuf->tt_wbuf[i] = get(pc, ISP);
	
	i = (found == ONWORD) ? 1 : 3;		/* routine-type index. */
	
						/* Unpack. */
	tt->rtype =	ttBuf->tt_cbuf[i];
			/* Magic constants should be in a syswide .h file.
			 * 2 = glue functions.
			 * 3 = start (trace back termination).
			 * 7 = run of the mill trace table. */
	if(!validRtype(tt->rtype)) {
		sprintf(Error_buf,"Don't handle trace tables with %d code.",
				tt->rtype);
		error(Error_buf);
	}
	i += 2;
	tt->first_gpr =	LNIBBLE(ttBuf->tt_cbuf[i]);
	tt->wflg =	RNIBBLE(ttBuf->tt_cbuf[i]) & 0x8;
	tt->xflg =	RNIBBLE(ttBuf->tt_cbuf[i]) & 0x4;
	tt->yflg =	RNIBBLE(ttBuf->tt_cbuf[i]) & 0x2;
	tt->zflg =	RNIBBLE(ttBuf->tt_cbuf[i]) & 0x1;
	if(tt->wflg) {
		++i;
		tt->param_words =	LNIBBLE(ttBuf->tt_cbuf[i]);
		tt->frame_reg =		RNIBBLE(ttBuf->tt_cbuf[i]);
	}
	else if(!inGlueRoutine(tt->rtype))	/* Something is wrong! */
		printf("Warning:no frame register number in trace table!\n");
	else
		return;		/* Glue routines needs no extra unpacking. */

	if(!validFrameRegNum(tt->frame_reg))
		error("Bad frame register number in trace table-got %X\n",
			tt->frame_reg);
	whichFlag = 0;
	
	if(tt->xflg) {		/* x, y and z flags are for the future. */
		++i;
		whichFlag = 'x';
	}
	if(tt->yflg) {
		++i;	
		whichFlag = 'y';
	}
	if(tt->zflg) {
		++i;
		whichFlag = 'z';
	}
	if(whichFlag)	  /* Not clear for user if >1 flag not implemented. */
		printf(flgNotImplemented, whichFlag);
	
			/* bits 0 and 1 right shifted 6 is offset_size. */
	tt->offset_size = ((ttBuf->tt_cbuf[++i] & 0xc0) >> 6);
	
			/* Bits 2-7 always present for local offset. */
	dprintf(("\t+offset byte byte[%d] = %x\n", i,ttBuf->tt_cbuf[i]));
	tt->local_offset = ttBuf->tt_cbuf[i] & 0x3f;
	
			/* If size !0, then get remaining offset bits. */
	for(size = tt->offset_size; size--; ) {
		++i;
		dprintf(("\t+offset byte byte[%d]=%x\n",i,ttBuf->tt_cbuf[i]));
		tt->local_offset=(tt->local_offset << 8) | ttBuf->tt_cbuf[i];
	}
	
		/* Local offset in words, make it bytes. */
	tt->local_offset <<= 2;
}


/* Put reasonable defaults in trace table structure fields.*/
setDefaults(tt)
register TTABLE *tt;
{
	tt->rtype = -1;		/* If < 0, then other fields invalid. */
	tt->frame_reg = -1;	/* Maybe someday there will be a r-1?? */
	tt->wflg  =  0;
	tt->xflg  =  0;
	tt->yflg  =  0;
	tt->zflg  =  0;
}

#ifdef DEBUG
dumpTraceTable(tt,pc) TTABLE *tt; ADDR pc;{
	if(debug) {
		int i;
		printf("--- trace table found(%x) ---\n",pc);
		for(i=0; i < TTSIZE/2; i++)
			dprintf((" %x", TTbuf.tt_wbuf[i]));
		dprintf(("\n"));
		printf("\t+== Unpacked trace table ==>\n");
		dprintf(("\t+rtype=%d",		tt->rtype));
		dprintf((" first_gpr=%d",	tt->first_gpr));
		dprintf((" param_words=%d",tt->param_words));
		dprintf((" frame_reg=%d\n",	tt->frame_reg));
		dprintf(("\t+wflg=%x",	tt->wflg));
		dprintf((" xflg=%x",	tt->xflg));
		dprintf((" yflg=%x",	tt->yflg));
		dprintf((" zflg=%x\n",	tt->zflg));
		dprintf(("\t+offset_size=%d",tt->offset_size));
		dprintf((" local_offset=%x\n",tt->local_offset));
		dprintf(("\t+===============\n"));
	}
}
#endif


/* MAXNAMEWORDS*sizeof(int) is a few chars smaller than the max name length 
 * of a function's name in text. */
#define MAXNAMEWORDS 20
#define MAXNAMECHARS (MAXNAMEWORDS * sizeof(int))

union WordBuf {
	char	charname[MAXNAMECHARS];
	int	intname [MAXNAMEWORDS];
} Wbuf;

/* Look for function name in text, just before entry point. Expecting ascii
 * string "<functioname>". The '>' character must be within sizeof(int) bytes
 * before the function entry point. */
char *
getNameFromText(entryAddr)
register ADDR entryAddr;
{
	register int i, c;
	register char *s;	/* Function name pointer. */
	register char *e;	/* End address to look for '>' char. */

	for(i = MAXNAMEWORDS - 1; i >= 0; --i) {
		entryAddr -= sizeof(int);
		Wbuf.intname[i] = GETWORD(entryAddr, ISP);
	}

	dprintf(("Wbuf[MAX]=%X\n",Wbuf.intname[MAXNAMEWORDS - 1]));

	e = &Wbuf.charname[MAXNAMECHARS];
	for(s = e - sizeof(int); s < e; ++s)
		if(*s == '>')
			break;

	if(*s != '>') {		/* Did we find a function name string? */
		dprintf(("\t+Did not find '>' above entry point!\n"));
		return((char *)0);
	}

	*s = '\0';		/* Replace '>' with null char. */
	
	for(c = *--s; s >= Wbuf.charname; c = *--s)
		if(c == '<' || !isprint(c))
			break;
	if(!isprint(c)) {
		dprintf(("\t+Bad funcname:non printable character!\n"));
		return((char *)0);
	}
	if(s < Wbuf.charname) {
		dprintf(("\t+Bad funcname:no '<' above entry point!\n"));
		return((char *)0);
	}
	return(++s);
}

#ifdef	KERNEL
print_stack_trace(thread)
	thread_t	thread;
{
	long		thiar,thsp;
	struct pcb	*thpcb;

	thpcb = (struct pcb *) chkget(&thread->pcb, DSP);
	thiar = chkget(&(thpcb->pcb_r15),DSP);
	thsp = chkget(&(thpcb->pcb_r1),DSP);
	traceback(thiar,thsp,thpcb);
}
#endif	KERNEL
