/* 
 * Mach Operating System
 * Copyright (c) 1988 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	mc881.c,v $
 * Revision 2.6  88/12/19  02:37:16  mwyoung
 * 	Remove old MACH conditionals.
 * 	[88/12/14            mwyoung]
 * 
 * Revision 2.5  88/11/23  16:25:17  rpd
 * 	Changed includes to the new style.
 * 	[88/11/04  17:16:39  rpd]
 * 
 * 25-Apr-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Upgraded from ACIS release 3 the sigcleanup() and sendsig()
 *	routines.
 *
 * 10-Mar-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	In mc881_exception():  Decode the operation that the 881 was
 *	trying to perform to determine whether it was a read or write
 *	fault.  The info word from Rosetta is unreliable and often
 *	wrong.  Hardware bug perhaps?
 *
 *  1-Feb-88  David Golub (dbg) at Carnegie-Mellon University
 *	Made addressing error return the invalid address.
 *
 * 24-Jan-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	MACH_EXCEPTION changes to mc881_exception().  In mc881_abort(),
 *	check for the CPS_RESPONSE register signalling the combination
 *	busy/timeout/exception otherwise the CA bit (Come-Again) never
 *	is cleared and the processor just sits there spinning awaiting
 *	that bit to clear. 
 *
 * 23-Jan-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Corrected fault_type parameter to vm_fault().
 *
 * 15-Dec-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added mc881_pcb_init() to initialize the floating point state
 *	for at thread.
 *
 * 23-Sep-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Fixed a bug in mc881_pci() which incorrectly saved the current
 *	thread's 881 state to the *next* thread to acquire the 881.
 *	Uggh.
 *
 * 26-Aug-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Merged in latest 4.3 ACIS changes.
 *
 *  5-Jul-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Associate floating point state with threads, not processes.
 *	Also call vm_fault instead of vtop and pagein.
 *
 ***********************************************************************
 */

/*
 * 5799-CGZ (C) COPYRIGHT IBM CORPORATION 1987
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/* $ACIS:mc881.c 9.3$ */

#ifndef lint
/* Old RCSid string here */
#endif

#include <romp_apc.h>
#include <romp_debug.h>
#include <mach.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/dir.h>
#include <sys/signal.h>
#include <sys/user.h>

#include <ca/float.h>

#include <ca/ioim.h>
#include <ca/mmu.h>
#include <ca/mc881.h>
#include <ca/mc881var.h>

#include <ca/cpu.h>
#include <ca/reg.h>
#include <ca/debug.h>
#include <sys/thread.h>
#include <vm/vm_param.h>

#include <ca/scr.h>
#include <ca/frame.h>
#if	MACH
#include <sys/exception.h>
#include <sys/ux_exception.h>
#endif	MACH


#if	ROMP_DEBUG
#define	STATIC
#else	ROMP_DEBUG
#define	STATIC static
#endif 	ROMP_DEBUG

extern thread_t mc881_thread;		/* Who owns mc881 registers */
int mc881_status;			/* What last error status was */
int nomc881 = 0;			/* set to disable use of mc881 */

#define MC881_ABORT()	MC_CIR_WRITE_16(MC_CIR_CONTROL, MC_CIR_CONTROL_AB)

STATIC void
mc881_busy()
{
    /* Loop waiting for the 68881 to finish after a DMA operation */
    while (ior(IOIM2+IOIM_CPS)&IOIM_CPS_68881_BUSY) {
	;
    }
}


STATIC void
mc881_save(statep)
struct mc881_state *statep;
{
    u_int i, *ptr;

    /* First, save ioim length register */
    statep->ioim_length_register = IOIM_881_READ_LENGTH_REGISTER();
    do {
	i = MC_CIR_READ_16(MC_CIR_SAVE);
    } while ((i&MC_STATE_MASK) == MC_STATE_NOT_READY);

    statep->internal.format = i;
    switch ((i&MC_STATE_MASK)) {
    case MC_STATE_NULL:
	i = MC_NULL_FRAME_SIZE;
	break;
    case MC_STATE_NOT_READY:
	panic("mc881_save: Should not occur!\n");
	/*NOTREACHED*/
    case MC_STATE_ILLEGAL:
	panic("mc881_save: Illegal!\n");
	/*NOTREACHED*/
    default:
	i &= MC_NON_NULL_FRAME_SIZE_MASK;
    }
    i /= 4;
    ptr = &statep->internal.data[0];
    while (i-- != 0) {
	*ptr = MC_CIR_READ_32(MC_CIR_OPERAND);
	ptr++;
    }

    /* Now, save the programmer's model */

    IOIM_881_WRITE_LENGTH_REGISTER(96/32*8);		/* How much to save */
    IOIM_881_WRITE(IOIM_881_XFER_PATH_FROM_881,
				MC_SAVE_DATA_REGS, IOIM_881_LENGTH_N_WORDS,
			(u_int) &statep->programmer.data[0]);
    mc881_busy();				/* Wait for op to complete */
    IOIM_881_WRITE_LENGTH_REGISTER(32/32*3);		/* How much to save */
    IOIM_881_WRITE(IOIM_881_XFER_PATH_FROM_881,
			MC_SAVE_CONTROL_REGS, IOIM_881_LENGTH_N_WORDS,
			(u_int) &statep->programmer.control[0]);
    mc881_busy();				/* Wait for op to complete */
}

STATIC void
mc881_restore(statep)
struct mc881_state *statep;
{
    u_int i, *ptr;

    /* First, restore the programmer's model */

    if (statep->internal.format != MC_STATE_NULL) {
	IOIM_881_WRITE_LENGTH_REGISTER(96/32*8);	/* How much to save */
	IOIM_881_WRITE(IOIM_881_XFER_PATH_TO_881, MC_RESTORE_DATA_REGS,
	    IOIM_881_LENGTH_N_WORDS, (u_int) &statep->programmer.data[0]);
	mc881_busy();				/* Wait for op to complete */
	IOIM_881_WRITE_LENGTH_REGISTER(32/32*3);	/* How much to save */
	IOIM_881_WRITE(IOIM_881_XFER_PATH_TO_881, MC_RESTORE_CONTROL_REGS,
	    IOIM_881_LENGTH_N_WORDS, (u_int) &statep->programmer.control[0]);
	mc881_busy();				/* Wait for op to complete */
    }

    /* Next, restore the internal state */

    MC_CIR_WRITE_16(MC_CIR_RESTORE, statep->internal.format);

    do {
	i = MC_CIR_READ_16(MC_CIR_RESTORE);
    } while ((i&MC_STATE_MASK) == MC_STATE_NOT_READY);

    switch ((i&MC_STATE_MASK)) {
    case MC_STATE_NULL:
	i = MC_NULL_FRAME_SIZE;
	break;
    case MC_STATE_NOT_READY:
	panic("mc881_restore: Should not occur!\n");
	/*NOTREACHED*/
    case MC_STATE_ILLEGAL:
	panic("mc881_restore: Illegal!\n");
	/*NOTREACHED*/
    default:
	i &= MC_NON_NULL_FRAME_SIZE_MASK;
    }
    i /= 4;

    /*
     * We need to "pop" these off the "stack" (ie: we restore in the
     * opposite order they were saved).
     */
    ptr = &statep->internal.data[i-1];
    while (i-- != 0) {
	MC_CIR_WRITE_32(MC_CIR_OPERAND, *ptr);
	ptr--;
    }

    /* Last, restore ioim length register */
    IOIM_881_WRITE_LENGTH_REGISTER(statep->ioim_length_register);
}

STATIC label_t mc881_label;
void mc881_trap();
/*
 * mc881_test(): called to do a unit test to make sure that the 881 
 * is operational.
 */
STATIC int
mc881_test()
{
    STATIC double x = 1.0, y = 2.0;	/* static to avoid use of floating point code */
    static double z;
    static double result = 3.0;
    int n;
    struct mc881_state state;
    extern void (*_trap)();			/* Pointer to trap slih */
    void (*oldtrap)() = _trap;

    _trap = mc881_trap;
    if (setjmp(&mc881_label)) {
	_trap = oldtrap;
	return(0);				/* program check - not operational */
    }
    bzero((char *)&state, sizeof state);	/* Zero registers XXX */
    state.internal.format = MC_STATE_NULL;
    mc881_restore(&state);			/* intialize 881 */

    *(int *)(IOIM_881_HW_BASE + 0x015202) = (int) (&x);	/* transfer x to fr1 */
    *(int *)(IOIM_881_HW_BASE + 0x01528a) = (int) (&y);	/* add y to fr1 */
    *(int *)(IOIM_881_HW_BASE + 0x3dd202) = (int) (&z);	/* transfer fr1 to z */

    n = IOIM_881_READ_LENGTH_REGISTER();	/* get in sync */
    if (n == 3) {
	n = 4;
    }
    mc881_save(&state);
    _trap = oldtrap;
    return(*(int *) (&z) == * (int *) (&result));
}

/*
 * trap intercept function in case mc881 causes a program or
 * machine check.
 */
STATIC void
mc881_trap(mcs_pcs, info, locr0)
	int mcs_pcs, info;
	int *locr0;
{
#define	ics_cs locr0[ICSCS]

	DEBUGF(mc881debug&SHOW_INIT,
	    prstate("mc881_trap", mcs_pcs, info, locr0[IAR], ics_cs, (int *)0));
	printf("68881 program check - ");	/* give reason not operational */
	longjmp(&mc881_label);
}


/*
 * The following, in alphabetical order, are the routines
 * which are called from the outside world.
 */

/*
 * This is very hardware specific.  What has happened is that the second
 * exception in the exception stack is for the 68881, and the exception
 * sequence is not store/load (in the store/load case, when the store gives
 * an exception, the CPU always gives an exception to the load, even if
 * it succeeds).
 *
 * The 68881 (or IOIM2) has given the exception for one of two reasons:
 *	First, the 68881 decided that there was a problem and issued a
 *		"take exception" primitive.  There are two cases:
 *		1. mid-instruction exception - issue abort unless the
 *			exception was a protocol violation which does
 *			regenerate.
 *		2. pre-instruction exception - reset BIU
 *			"instruction-pending" bit.
 * 
 *	Second, issue an 881 abort.
 */

int
mc881_abort(locr0)
int *locr0;			/* Registers */
{
    int i = MC_CIR_READ_16(MC_CIR_RESPONSE);	/*In case IOIM did not do it */
    int ioim2_cps;

    ioim2_cps = ior(IOIM2+IOIM_CPS);

     /*
      * 	If the 881 is busy, and it has not timed out nor
      *		reported an exception then wait for the CA bit to
      *		become unasserted.
      */

    if ((ioim2_cps & IOIM_CPS_68881_BUSY) &&
    	! (ioim2_cps & (IOIM_CPS_PMUC_TIMEOUT|IOIM_CPS_PMUC_EXCEPTION))) {
	do {
	    i = MC_CIR_READ_16(MC_CIR_RESPONSE);
	} while ((i&MC_RESPONSE_CA)
		&& ((i&MC_RESPONSE_NULL_MASK) == MC_RESPONSE_NULL_VALUE));
    }

    if ((i&MC_RESPONSE_TAKE_EXCEPT_MASK) == MC_RESPONSE_TAKE_MID_EXCEPT) {
	if ((i&MC_RESPONSE_PARAMETER_MASK) == MC_RESPONSE_PROTOCOL_VIOLATION)
		return;		/* protocol violations regenerate */
	/* otherwise fall thru to do abort */
    } else 
    	if ((i&MC_RESPONSE_TAKE_EXCEPT_MASK) == MC_RESPONSE_TAKE_PRE_EXCEPT) {
	/* reset BIU "instruction pending flag" */
	struct mc881_state *state = (struct mc881_state *)
			    current_thread()->pcb->fpasave.mc881_intreg;
	mc881_save(state);
	if ((state->internal.format&MC_NON_NULL_FRAME_SIZE_MASK) != MC_IDLE_FRAME_SIZE)
	 	panic("mc881_abort: not in idle state");
	if ((state->internal.biu_flags & MC_BIU_INST_PENDING) != 0)
		panic("mc881_abort: instn not pending");
	state->internal.biu_flags |= MC_BIU_INST_PENDING;	/* reset instruction pending */
	mc881_restore(state);
	DEBUGF(mc881debug&SHOW_PCI,printf("mc881_abort:cleared BIU instn pending\n"));
	return;
    }
    MC881_ABORT();		/* abort current function */
    DEBUGF(mc881debug&SHOW_PCI,printf("mc881_abort:abort command issued\n"));
}


/*
 * mc881_core() : called to get the register set (if it exists)
 * into the user address space so that a debugger can look at it.
 */

void
mc881_core(thread)
thread_t thread;
{
    if (mc881_thread == thread) {
	mc881_save((struct mc881_state*)
			thread->pcb->fpasave.mc881_intreg);
	iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_DISABLE);
	mc881_thread = THREAD_NULL;
	
    } /* otherwise 881 state already in pcb. */
}

void
mc881_exit(th)
thread_t th;
{
    if (mc881_thread == th) {
	mc881_save((struct mc881_state *)
			th->pcb->fpasave.mc881_intreg);	
	iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_DISABLE);
	mc881_thread = THREAD_NULL;			/* No more thread */
    }
    th->pcb->floatmask  &= ~FLOAT_MC881;	/* We are not using 68881 */
    DEBUGF(mc881debug&SHOW_REGS,
	printf("mc881_exit: release th 0x%x\n", th));
}

/*
 * Duplicate the thread floating point register into the next thread.
 */

void
mc881_fork(oldth, newth)
thread_t	oldth,		/* parent thread */
		newth;		/* child thread */
{
    if (mc881_thread == oldth) {
	mc881_save((struct mc881_state *)
			oldth->pcb->fpasave.mc881_intreg);
	iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_DISABLE);
	mc881_thread = THREAD_NULL;			/* Force reload next time referenced */
    }
    newth->pcb->fpasave = oldth->pcb->fpasave;	/* copy it */
}

/*
 * Called to initialize any floating point state required
 * by a thread.
 */
void mc881_pcb_init(pcb)
struct pcb *pcb;		/* new thread */
{
	bzero(pcb->fpasave,sizeof(struct floatsave));
}

int
mc881_getreg(reg, value)
int reg;			/* register number to get */
int *value;			/* Where to return it */
{
    struct mc881_state *state = (struct mc881_state *)
				current_thread()->pcb->fpasave.mc881_intreg;

    if (reg >= MC881_NUM_REG) {
	return EINVAL;
    }
    if (mc881_thread == current_thread()) {
	mc881_save(state);
	iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_DISABLE);
	mc881_thread = THREAD_NULL;			/* Force reload next time referenced */
    }
    *value = state->programmer.data[reg];
    return 0;
}

/*
 * Check to see if the mc881 exists.  This simply involves the null
 * case if we are "ROMPC" compiled, but not "SGP" compiled.  If we
 * are both "ROMPC" and "SGP" compiled, then we check the CPU type.
 * ROMPC cpu's "always" have the 68881.
 *
 * If the 68881 does exist, enable it's use.
 * If it doesn't exist, disable the use of it (by setting mc881_thread
 * to "-1").
 */

void
mc881_init()
{
    if (cpu_model != CPU_ROMPC) {
	mc881_thread = (thread_t) -1;
	return;
    } else {
	if (nomc881) {
	    printf("6881 disabled.\n");
	} else {
	    mc881_thread = THREAD_NULL;
	    float_hardware |= FLOAT_MC881;
	}
    }
    if (float_hardware&FLOAT_MC881) {
        if (GETSEG(IOIM_881_SW_BASE) != GETSEG(IOIM_881_HW_BASE))
		panic("mc881_init: HW_BASE != SW_BASE");
	if (sizeof (struct floatsave) < sizeof (struct mc881_state)) {
	    printf("Kernel inconsistency (mc881_init): 68881 not enabled.\n");
	    float_hardware &= ~FLOAT_MC881;
	} 
	else if (!mc881_test()){
	    printf("68881 not operational.\n");
	    float_hardware &= ~FLOAT_MC881;
	} 
	else {
	    printf("68881 enabled.\n");
	}
    }
}

/*
 * Is the interrupt from the 881?
 *
 * If so, then why did we get it?

 * If we got it because the 881 was locked, then allocate a register set
 * to this thread, possibly store the old process's registers, and
 * then unlock the IOIM.
 */

int
mc881_pci(mcs_pcs, info, locr0)
int	mcs_pcs;
int	info;
int	*locr0;				/* Registers at interrupt time */
{
    if ((GETSEG(info) == GETSEG(IOIM_881_HW_BASE)) && 
		    ((GETSUBSEG(info) == GETSUBSEG(IOIM_881_HW_BASE)) ||
		    (GETSUBSEG(info) == GETSUBSEG(IOIM_881_SW_BASE)))) {
	thread_t th = current_thread();
	struct mc881_state *statep = (struct mc881_state *)
					th->pcb->fpasave.mc881_intreg;

	DEBUGF(mc881debug&SHOW_OPEN,
	    prstate("mc881_pci", mcs_pcs, info, locr0[IAR], locr0[ICSCS],locr0));

	if (mc881_thread == th) {			/* 68881 exception */
	    return(mc881_exception(mcs_pcs,info,locr0,th));
	} 

	if (!float_has_881(float_hardware)) {
	    DEBUGF(mc881debug&SHOW_REGS,
		printf("mc881_pci: no 881 available th 0x%x\n",th));
		return(0);		/* no 881 - give an exception */
	}
	if (! float_has_881(th->pcb->floatmask)) {
	    DEBUGF(mc881debug&SHOW_REGS,
		printf("mc881_pci: allocate 881 th 0x%x\n", th));
	    bzero((char *)statep, sizeof *statep);	/* Zero registers XXX */
	    statep->internal.format = MC_STATE_NULL;
	    current_thread()->pcb->floatmask |= FLOAT_MC881;
			/* We are using 68881 */
	}
	if (mc881_thread != THREAD_NULL) {
	    /* Save current registers */
	    mc881_save((struct mc881_state *)
				    mc881_thread->pcb->fpasave.mc881_intreg);
	}
	mc881_thread = th;
	mc881_restore(statep);
	iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_ENABLE);
	return 1;			/* handled */
    } else {
	return 0;			/* not ours */
    }
}

/*
 * handle actual exceptions reported from the 881 
 * these include:
 * 1. floating point exceptions
 * 2. DMA exceptions (page faults)
 * 3. other exceptions (parity errors) due to bad hardware.
 */

int
mc881_exception(mcs_pcs, info, locr0, th)
int		mcs_pcs;
int		info;
int		*locr0;			/* Registers at interrupt time */
thread_t	th;
{
    int status = ior(IOIM2 + IOIM_CPS);	/* co-processor status register */
    kern_return_t result;
#if	MACH
    int type = EXC_ARITHMETIC,
    	code = 0,
	subcode = 0;
#else	MACH
    int i = SIGFPE;			/* assume numeric exception */
#endif	MACH    
    struct mc881_state *state = (struct mc881_state *)
				th->pcb->fpasave.mc881_intreg;


    mc881_status = status;		/* RAS */
    DEBUGF(mc881debug&SHOW_OPEN,printf("CPS=%b ",status,CPS_FMT));

    MC881_ABORT();

    if ((locr0[ICSCS] & ICSCS_PROBSTATE) == 0) {
	return(0);	/* no kernel mode traps allowed, sorry. */
    }
    
    switch (status & IOIM_CPS_MASK) {
    case IOIM_CPS_68881_EXCEPTION:
#if	MACH
	code = EXC_ROMP_68881;
	subcode = status;
#else 	MACH    
	i = SIGFPE;
#endif	MACH    	
	/* Save the last 68881 operation performed */
	state->lastop = ior(IOIM2+IOIM_LASTOP);
	break;
    case IOIM_CPS_PMUC_TIMEOUT+IOIM_CPS_PMUC_EXCEPTION:
    /*
     * The TIMEOUT bit can, because of hardware idiosyncracies,
     * come on in conjunction with the EXCEPTION bit.  So,
     * treat the combination as an EXCEPTION unless the
     * data packet was bound for segnemt F.
     */
	if (GETSUBSEG(info) == GETSUBSEG(IOIM_881_HW_BASE) &&
			GETSEG(locr0[EX1_DATA]) == GETSEG(IOIM_881_HW_BASE)) {
#if	MACH
	code = EXC_ROMP_68881_TIMEOUT;
	subcode = status;
#else	MACH
	    i = SIGBUS;
#endif	MACH	    
	    break;
	}	/* else: fall thru */
    case IOIM_CPS_PMUC_EXCEPTION:
#define	MC881_TRANSFER_TO_MEMORY	0xf
#define	MC881_TRANSFER_FROM_MEMORY	0x0
	if (GETSUBSEG(info) == GETSUBSEG(IOIM_881_HW_BASE)) {
	    int addr = (locr0[EX1_DATA] & ~0xff) | ((status>>8) & 0xfc);
	    int fault_type;
	    if (((locr0[EX1_ADDR] >> 18) & 0xf) == MC881_TRANSFER_TO_MEMORY)
	    	fault_type = VM_PROT_READ | VM_PROT_WRITE;
	    else
	    	fault_type = VM_PROT_READ;
		
	    if (addr < locr0[EX1_DATA])
		addr += 0x100;		/* carry */
	    if (status & 0x100)
		addr -= 4;		/* address was ahead by 4 */
	    DEBUGF(mc881debug&SHOW_OPEN,printf("dma_addr=%x ",addr));
	    if ((result = vm_fault(current_task()->map,
			trunc_page((vm_offset_t)addr),
			fault_type,
			FALSE))
		== KERN_SUCCESS)
#if	MACH
		type = 0;
	    else {
		type = EXC_BAD_ACCESS;
		code = result;
		subcode = addr;
	    }
#else	MACH
		i = 0;	/* was a handled page fault */
	    else {
		printf("failed 68881 dma to address [%x]\n",addr);
		i = SIGSEGV;
	    }
#endif	MACH
	}
	break;
    /*
     * The user has set up for some wayout DMA to occur.
     */
    case IOIM_CPS_PMUC_TIMEOUT:
#if	MACH
	code = EXC_ROMP_68881_TIMEOUT;
	subcode = status;
#else	MACH
    	i = SIGBUS;
#endif	MACH	
	break;
    /*
     * Other exceptions should only happen if there is a hardware failure
     */
    default:
    	printf("CPS=%b ",status,CPS_FMT);
	panic("mc881_exception: fatal status");
    }
#if	MACH
    DEBUGF(mc881debug&SHOW_PCI,printf("mc881_pci(type %x code %x sub %x)\n",
    			type,code,subcode));
    if (type)
        thread_doexception(current_thread(),
	    type,
	    code, 
	    subcode);	
#else	MACH
    DEBUGF(mc881debug&SHOW_PCI,printf("signal=%d\n",i));

    if (i)
	psignal(u.u_procp, i);		/* Signal the user */
#endif	MACH	
    return(1);			/* its ours */
}

/*
 * mc881_putreg :
 *
 * For a given floating point register number, set the value.
 */

int
mc881_putreg(reg, value)
int	reg;
int	*value;
{
    struct mc881_state *state = (struct mc881_state *)
				current_thread()->pcb->fpasave.mc881_intreg;

    if (reg >= MC881_NUM_REG) {
	return MC881_NUM_REG;
    }
    if (mc881_thread == current_thread()) {
	mc881_save(state);
	iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_DISABLE);
	mc881_thread = THREAD_NULL;  /* Force reload next time referenced */
	/* So, now in each case, we are using from save area */
    }
    state->programmer.data[reg] = *value;
    return 0;
}


/*
 * Copy programmer state into the sigframe structure.
 *
 * Return 1
 */


int
mc881_sendsig(floatsave)
struct floatsave *floatsave;
{
    struct mc881_state *state = (struct mc881_state *)
				current_thread()->pcb->fpasave.mc881_intreg;
    struct mc881_state *tostate = (struct mc881_state *) floatsave;
    int status, code;

    DEBUGF(mc881debug&SHOW_PCI,printf("mc881_sendsig: fs=%x state=%x\n",
    		floatsave, state));
		
    if (mc881_thread == current_thread()) {
	mc881_save(state);
	iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_DISABLE);
	mc881_thread = THREAD_NULL; /* Force reload next time referenced */
    }

    *tostate = *state;		/* Copy things over */
    /* Set u.u_code */
    status = state->programmer.control[1];	/* Get status register */    
    u.u_code = mc881_except_to_code(status);
    return 1;
}


mc881_except_to_code(status)
int status;
{
    int code;

    if (status&MC_EXCEPT_BSUN) {
	code = FP_INV_OPER;
    } else if (status&MC_EXCEPT_SNAN) {
	code = FP_INV_OPER;
    } else if (status&MC_EXCEPT_OPERR) {
	code = FP_INV_OPER;
    } else if (status&MC_EXCEPT_OVFL) {
	code = FP_OVERFLOW;
    } else if (status&MC_EXCEPT_UNFL) {
	code = FP_UNDERFLOW;
    } else if (status&MC_EXCEPT_DZ) {
	code = FP_DIVIDE;
    } else if (status&MC_EXCEPT_INEX2) {
	code = FP_INEXACT;
    } else if (status&MC_EXCEPT_INEX1) {
	code = FP_INEXACT;
    } else {
	code = FP_INV_OPER;			/* ??? */
    }
    
    return(code);
}

/*
 * mc881_sigcleanup -
 *
 * After a signal handler returns from a SIGFPE, restore the
 * registers.
 *
 * On a signal, we send up the entire register set.  On a return,
 * we restore ONLY the programmer's model (and the length register).
 */

void
mc881_sigcleanup(sf_scp)
struct sigcontext *sf_scp;
{
    struct mc881_state
		*sigstate = (struct mc881_state *)sf_scp->sc_floatsave,
		*state = (struct mc881_state *)
				current_thread()->pcb->fpasave.mc881_intreg;

    /* Ensure that the current state is saved */
    if (mc881_thread == current_thread()) {
	mc881_save(state);
	iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_DISABLE);
	mc881_thread = THREAD_NULL;	/* Force reload next time referenced */
    }
    state->programmer = sigstate->programmer;
    state->ioim_length_register = sigstate->ioim_length_register;
}

void
mc881_setfloatstate(mask, state)
struct floatstate *mask, *state;
{
    mask->hardware_state &= ~FLOAT_MC881;	/* No changes in hardware */
    if ((mask->process_state&FLOAT_MC881) &&
		(float_has_881(
		  current_thread()->pcb->floatmask^state->process_state))) {
	/* The user is changing state.  Handle this. */
	if (float_has_881(state->process_state)) {
	    /* If they want it, let them touch it (then mc881_pci) */
	    mask->process_state &= ~FLOAT_MC881;
	} else {			/* They want to let the 881 go */
	    mc881_exit(current_thread());	/* Release hardware */
	}
    }
}

