/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 *	File:	ca/pcb.c
 *	Author:	Avadis Tevanian, Jr., Michael Wayne Young, David Golub,
 *			William J. Bolosky
 *
 *	Copyright (C) 1986, Avadis Tevanian, Jr., Michael Wayne Young,
 *			David Golub, William J. Bolosky
 *
 *	Romp PCB management
 *
 * HISTORY
 * $Log:	pcb.c,v $
 * Revision 2.4  88/11/23  16:26:04  rpd
 * 	Fixed cpu_model test in pcb_set_except_state for 135s.
 * 	[88/11/23  10:29:51  rpd]
 * 
 * Revision 2.3  88/08/25  18:12:58  mwyoung
 * 	Corrected include file references.
 * 	[88/08/22            mwyoung]
 * 	
 * 	Implement THREAD_STATE_FLAVOR_LIST.
 * 	[88/08/11  18:45:04  mwyoung]
 * 
 *
 * 25-Apr-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Let the user set the ICSCS_INSTSTEP bit for a thread.
 *
 * 25-Jan-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Fixed pcb_synch() to do the right thing.  Also converted to the
 *	new "approved" thread_set_status() and thread_get_status()
 *	interface.
 *
 *  8-Jan-88  David Golub (dbg) at Carnegie-Mellon University
 *	Changed for new thread_status interface.
 *
 * 15-Dec-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added pcb_terminate() function.  Added call to float_fork() from
 *	thread_dup().  Added a call to float_pcb_init() from
 *	pcb_init().  Changed pcb parameter in pcb_init() to be a thread.
 *
 *  9-Sep-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	MACH_TT: corrected value of ksp in pcb_init() just before calling
 *	thread_bootstrap().  Also corrected bcopy() in thread_dup()
 *	which depended on the layout of the register saved state on
 * 	the kernel stack (which has changed, for the better).
 *
 * 28-Jul-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Made thread_setstatus() return a value.
 *
 * 10-Jul-87  David Black (dlb) at Carnegie-Mellon University
 *	Added pcb_synch() stub.  Should be fixed!!
 *
 * 25-Apr-87  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Be sure that thread_start is only enabled for MACH_TT since it
 *	uses some externals that are only in MACH_TT.
 *
 * 16-Apr-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Fixed thread_dup() to copy the romp saved state correctly.
 *
 *  1-Apr-87  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Preliminary support for MACH_TT.  Needs a bit more work.
 *
 */


#include <ca/pcb.h>
#include <ca/scr.h>
#include <sys/thread.h>
#include <sys/thread_status.h>
#include <sys/kern_return.h>
#include <vm/vm_param.h>
#include <ca/pmap.h>	/* for KERNSTACK */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/proc.h>

/*
 *	pcb_init [exported to kernel]:
 *
 *	Initialize the pcb for a thread.  The RT version
 *	also initializes the floating point state for the
 *	RT.
 */

void pcb_init(thread, ksp)
	struct thread *thread;
	vm_offset_t	ksp;
{
	struct pcb	*pcb = thread->pcb;
	register struct romp_saved_state	*saved_state;
	extern int	thread_bootstrap();
	int * tmp = (int *) thread_bootstrap;

	bzero((caddr_t)pcb, sizeof(struct pcb));
	pcb->pcb_r15 = * tmp;
	saved_state = (struct romp_saved_state *) (ksp + KERNEL_STACK_SIZE);
	saved_state--;
	/*
	 * For the RT (in locore) the ksp always points at R1 when accessing
	 * user registers on the stack.
	 */
	pcb->pcb_usp = pcb->pcb_ksp = (int) &saved_state->r1;

	pcb->pcb_icscs =	ICSCS_STGPROT |	/* memory protect */
			ICSCS_TRANSLATE |      	/* virtual mode */
			ICSCS_CHECKSTOP;	/* check stop mask */
	
	saved_state->icscs = ICSCS_USERSET & ~ICSCS_USERCLR;

	pcb->pcb_ssp = -1;
	pcb->pcb_esp = -1;
	/*
	 * Initialize the floating point state for the new pcb.
	 */
	float_pcb_init(pcb);
}


/*
 *  	pcb_terminate [Exported to Kernel]:
 *
 *	Shutdown any state associated with a thread's pcb.
 *	On the RT,  this means release any floating point
 *	register state.
 */
 
void pcb_terminate(thread)
struct thread *thread;
{
    float_exit(thread);
}

/*
 *
 *	initial_context: [Exported to kernel]:
 *
 *	Set up the context for the very first thread to execute
 *	(which is passed in as the parameter).
 */
void initial_context(thread)
	thread_t	thread;
{
	extern	int	after_lc;

	active_threads[cpu_number()] = thread;
	pmap_activate(vm_map_pmap(thread->task->map), thread, cpu_number());
	thread->pcb->pcb_r15 = (int)(&after_lc);
	thread->pcb->pcb_icscs = ICSCS_STGPROT |	/* memory protect */
			 	ICSCS_TRANSLATE |      	/* virtual mode */
			 	ICSCS_CHECKSTOP |	/* check stop mask */
				ICSCS_INTERRUPT;
}

/*
 *	thread_start [Exported to kernel]:
 *
 *	Start a kernel thread at the specified routine.  The thread must
 *	be in a suspended state.
 *
 *	start is really of type void (*start)(), but hc complains.
 */

thread_start(thread, start, mode)
	register thread_t	thread;
	vm_offset_t		start;
	int			mode;
{
	register struct pcb	*pcb;
	extern int thread_tramp;
	pcb = thread->pcb;
	pcb->pcb_r0 = (int) start;		/* data area pointer */
	pcb->pcb_iar = * ((int * ) start );
	pcb->pcb_icscs = ICSCS_STGPROT |	/* memory protect */
			 ICSCS_TRANSLATE |      /* virtual mode */
			 ICSCS_CHECKSTOP;	/* check stop mask */
	pcb->pcb_r15 = (int) &thread_tramp;			 
	pcb->pcb_ksp = thread->kernel_stack + KERNEL_STACK_SIZE;
}

/*
 *	thread_dup [Exported to UNIX] :
 *
 *	Duplicate the user's state of a thread.  This is only used to perform
 *	the Unix fork operation.
 */
thread_dup(parent, child)
	register thread_t	parent, child;
{
	register struct romp_saved_state	*child_state, *parent_state;

	/*
	 *	Parent state is NOT at top of stack - initial process
	 *	has put other call frames on stack.
	 */
	parent_state = (struct romp_saved_state *)
			 (parent->u_address.uthread->uu_ar0);
	child_state = USER_REGS(child);
	*child_state = *parent_state;

	child_state->r2 = proc[child->task->proc_index].p_pid;
	child_state->r0 = 1;		/* must agree with library fork code.*/
	child_state->icscs = ICSCS_USERSET & ~ICSCS_USERCLR;
	float_fork(parent,child);	/* copy parent's saved fp state */
}


/*
 *	pcb_synch [Exported to Kernel]:
 *
 *	Stores registers that aren't saved by the kernel at context
 *	switch time into the pcb.  On the romp, this means the
 *	floating point registers.
 *
 */

pcb_synch(thread)
thread_t thread;
{
    float_core(thread);
}

/*
 *	thread_setstatus [Exported to user]:
 *
 *	Set the status of the specified thread.
 *
 *	On the RT, we support setting three distinct flavors
 *	of thread state: register state, exception state (applicable
 *	to the APC), and floating point state.
 */

kern_return_t pcb_set_regs(), pcb_set_except_state(),
		pcb_set_float_state();

kern_return_t pcb_get_regs(), pcb_get_except_state(),
		pcb_get_float_state();		

kern_return_t thread_setstatus(thread, flavor, tstate, count)
	thread_t		thread;
	int			flavor;
	thread_state_data_t	*tstate;
	unsigned int		count;
{
    	kern_return_t r;

	switch(flavor) {
		case  ROMP_THREAD_STATE:
		    if (count < ROMP_THREAD_STATE_COUNT) {
			return(KERN_INVALID_ARGUMENT);
		    }
		    r = pcb_set_regs(thread,
		    	(struct romp_thread_state *) tstate);
		break;
		
		case ROMP_EXCEPT_STATE:
		    if (count < ROMP_EXCEPT_STATE_COUNT) {
			return(KERN_INVALID_ARGUMENT);
		    }
		    r = pcb_set_except_state(thread,
		    	(struct romp_except_state *) tstate);		    
		break;
		
		case ROMP_FLOAT_STATE:
		    if (count < ROMP_FLOAT_STATE_COUNT) {
			return(KERN_INVALID_ARGUMENT);
		    }
		    r = pcb_set_float_state(thread,
		    	(struct romp_except_state *) tstate);
		break;

		default:
			return(KERN_INVALID_ARGUMENT);
	}
	return(r);
}

/*
 *	thread_getstatus [Exported to User] :
 *
 *	Get the status of the specified thread.  As in thread_setstatus,
 *	we support setting of the three distinct flavors of thread
 *	state.
 */

kern_return_t thread_getstatus(thread, flavor, tstate, count)
	register thread_t	thread;
	int			flavor;
	thread_state_t		tstate;	/* pointer to OUT array */
	unsigned int		*count;	/* IN/OUT */
{
    	kern_return_t r;
	
	switch(flavor) {
		case THREAD_STATE_FLAVOR_LIST:
			if (*count < 3)
				return(KERN_INVALID_ARGUMENT);
			tstate[0] = ROMP_THREAD_STATE;
			tstate[1] = ROMP_EXCEPT_STATE;
			tstate[2] = ROMP_FLOAT_STATE;
			*count = 3;
			r = KERN_SUCCESS;
			break;

		case  ROMP_THREAD_STATE:
		    if (*count < ROMP_THREAD_STATE_COUNT) {
			return(KERN_INVALID_ARGUMENT);
		    }
		    r = pcb_get_regs(thread,
		    	(struct romp_thread_state *) tstate,
			count);
		break;
		
		case ROMP_EXCEPT_STATE:
		    if (*count < ROMP_EXCEPT_STATE_COUNT) {
			return(KERN_INVALID_ARGUMENT);
		    }
		    r = pcb_get_except_state(thread,
		    	(struct romp_except_state *) tstate,
			count);		    
		break;
		
		case ROMP_FLOAT_STATE:
		    if (*count < ROMP_FLOAT_STATE_COUNT) {
			return(KERN_INVALID_ARGUMENT);
		    }
		    r = pcb_get_float_state(thread,
		    	(struct romp_except_state *) tstate,
			count);
		break;

		default:
			return(KERN_INVALID_ARGUMENT);
	}
	return(r);
}
/*
 *	pcb_get_regs: [Internal to PCB module]
 *
 *	Get the user register state for the specified thread.
 */

kern_return_t
pcb_get_regs(thread,state,count)
thread_t thread;
register struct romp_thread_state	*state;
int *count;
{
	register struct romp_saved_state	*saved_state;

	saved_state = USER_REGS(thread);

	state->r0 = saved_state->r0;
	state->r1 = saved_state->r1;
	state->r2 = saved_state->r2;
	state->r3 = saved_state->r3;
	state->r4 = saved_state->r4;
	state->r5 = saved_state->r5;
	state->r6 = saved_state->r6;
	state->r7 = saved_state->r7;
	state->r8 = saved_state->r8;
	state->r9 = saved_state->r9;
	state->r10 = saved_state->r10;
	state->r11 = saved_state->r11;
	state->r12 = saved_state->r12;
	state->r13 = saved_state->r13;
	state->r14 = saved_state->r14;
	state->r15 = saved_state->r15;
	state->iar = saved_state->iar;
	state->icscs = saved_state->icscs;
	state->mq = saved_state->mq;

	*count = ROMP_THREAD_STATE_COUNT;
	return(KERN_SUCCESS);
}

/*
 *	pcb_set_regs: [Internal to PCB module]
 */

kern_return_t
pcb_set_regs(thread,state)
thread_t thread;
struct romp_thread_state *state;
{
	register struct romp_saved_state	*saved_state;    
    
	saved_state = USER_REGS(thread);
	saved_state->r0 = state->r0;
	saved_state->r1 = state->r1;
	saved_state->r2 = state->r2;
	saved_state->r3 = state->r3;
	saved_state->r4 = state->r4;
	saved_state->r5 = state->r5;
	saved_state->r6 = state->r6;
	saved_state->r7 = state->r7;
	saved_state->r8 = state->r8;
	saved_state->r9 = state->r9;
	saved_state->r10 = state->r10;
	saved_state->r11 = state->r11;
	saved_state->r12 = state->r12;
	saved_state->r13 = state->r13;
	saved_state->r14 = state->r14;
	saved_state->r15 = state->r15;
	saved_state->iar = state->iar;
	saved_state->icscs = (state->icscs & (~(ICSCS_USERCLR)|ICSCS_INSTSTEP))
				| ICSCS_USERSET;
	saved_state->mq = state->mq;
	return(KERN_SUCCESS);	
}

/*
 *	pcb_get_except_state: [Internal to PCB module]
 *
 *	Get the user exception state for the specified thread.
 */
kern_return_t
pcb_get_except_state(thread,state,count)
thread_t thread;
struct romp_except_state *state;
int *count;
{
    register struct romp_saved_state	*saved_state;    
    saved_state = USER_REGS(thread);
    
    state->ecr_count =  saved_state->ecr_count;

    state->ex1_ctl =  saved_state->ex1_ctl ;
    state->ex1_addr =  saved_state->ex1_addr ;
    state->ex1_data =  saved_state->ex1_data ;        
    state->ex1_rsv =  saved_state->ex1_rsv;    

    state->ex2_ctl =  saved_state->ex2_ctl ;
    state->ex2_addr =  saved_state->ex2_addr ;
    state->ex2_data =  saved_state->ex2_data ;        
    state->ex2_rsv =  saved_state->ex2_rsv;    

    state->mcspcs =  saved_state->mcspcs;
    state->info = saved_state->info;
    return(KERN_SUCCESS);    
}

/*
 *	pcb_get_except_state: [Internal to PCB module]
 *
 *	Get the user exception state for the specified thread.
 */
kern_return_t 
pcb_set_except_state(thread,state)
thread_t thread;
struct romp_except_state *state;
{
    register struct romp_saved_state	*saved_state;    
    saved_state = USER_REGS(thread);

    if (cpu_model == CPU_SGP)
	return(KERN_INVALID_ARGUMENT);
#define	APC_MAX_EXCEPT	2	/* should be defined somewhere else! */
    if ((state->ecr_count > APC_MAX_EXCEPT)
    	|| (state->ecr_count < 0)) {
	return(KERN_INVALID_ARGUMENT);
    }
    
    saved_state->ecr_count =  state->ecr_count;

    saved_state->ex1_ctl =  state->ex1_ctl ;
    saved_state->ex1_addr =  state->ex1_addr ;
    saved_state->ex1_data =  state->ex1_data ;        
    saved_state->ex1_rsv =  state->ex1_rsv;    

    saved_state->ex2_ctl =  state->ex2_ctl ;
    saved_state->ex2_addr =  state->ex2_addr ;
    saved_state->ex2_data =  state->ex2_data ;        
    saved_state->ex2_rsv =  state->ex2_rsv;    

    saved_state->mcspcs =  state->mcspcs;
    saved_state->info = state->info;
    return(KERN_SUCCESS);
}

/*
 *	pcb_get_float: [Internal to PCB module]
 *
 *	Get the user floating point state for the specified thread.
 */
kern_return_t 
pcb_get_float_state(thread,state,count)
thread_t thread;
struct romp_float_state *state;
int *count;
{
    /*
     * get floating point info in core now.
     */
    float_core(thread);
    
    switch(state->float_type) {
	case FLOAT_MC881: {
	    bcopy(thread->pcb->fpasave.mc881_intreg,
		state->fpregs.mc881_intreg,
		sizeof(state->fpregs.mc881_intreg));
	    *count = sizeof(state->fpregs.mc881_intreg)/sizeof(int);
	break;
	}
	
	case FLOAT_FPA: {
	case FLOAT_AFPA: 	    
	    bcopy(thread->pcb->fpasave.fpa_intreg,
		state->fpregs.fpa_intreg,
		sizeof(state->fpregs.fpa_intreg));
	    *count = sizeof(state->fpregs.fpa_intreg)/sizeof(int);
	break;
	}
	case FLOAT_EMUL:
	/* fall through for now */
	default:
		return(KERN_INVALID_ARGUMENT);
    }
    return(KERN_SUCCESS);
    
}

/*
 *	pcb_get_float: [Internal to PCB module]
 *
 *	Get the user floating point state for the specified thread.
 *
 *	XXX Check the values the user sticks into the fpa status registers.
 *	Can the user muck them up sufficiently well to crash the kernel?
 */
kern_return_t 
pcb_set_float_state(thread,state)
thread_t thread;
struct romp_float_state *state;
{
    /*
     * get floating point info in core now.
     */
    float_core(thread);
    
    switch(state->float_type) {
	case FLOAT_MC881: {
	    bcopy(state->fpregs.mc881_intreg,
	    	thread->pcb->fpasave.mc881_intreg,
		sizeof(state->fpregs.mc881_intreg));
	break;
	}
	
	case FLOAT_FPA: {
	case FLOAT_AFPA: 	    
	    bcopy(state->fpregs.fpa_intreg,
	    	thread->pcb->fpasave.fpa_intreg,
		sizeof(state->fpregs.fpa_intreg));
	break;
	}
	case FLOAT_EMUL:
	/* fall through for now */
	default:
		return(KERN_INVALID_ARGUMENT);
    }
    return(KERN_SUCCESS);
	
}

