/* 
 * Mach Operating System
 * Copyright (c) 1988 Carnegie-Mellon University
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
#define	DISPLAY_DEBUG 0
/*
 *	File:	ca/pmap.c
 *	Author:	William J. Bolosky
 *
 *	RT PC (romp)/Rosetta Version:
 *	Copyright (c) 1986, 1987, William J. Bolosky
 *
 *	Romp/Rosetta Physical memory management routines
 *
 * HISTORY
 * $Log:	pmap.c,v $
 * Revision 2.10  88/12/20  13:40:39  rpd
 * 	Changed includes to the new style.
 * 	Fixed PMAP_DEBUGF in non-ROMP_DEBUG case.
 * 	[88/11/26  21:15:33  rpd]
 * 
 * Revision 2.9  88/12/19  05:29:44  mwyoung
 * 	Only call cache_verify_free when ROMP_CACHE is selected.
 * 	[88/12/19            mwyoung]
 * 
 * Revision 2.8  88/12/19  02:37:36  mwyoung
 * 	Prepend DEBUG_ to pmap debug tags, to avoid overlap with pmap
 * 	module definitions.  Remove lint.
 * 	[88/12/18            mwyoung]
 * 	
 * 	Make rtakey a macro, and document its use and limitations.  Fix
 * 	pmap_protect (and pmap_enter) to check for VM_PROT_NONE.
 * 	[88/12/13            mwyoung]
 * 	
 * 	Add pmap_valid_page.
 * 	[88/12/13  03:20:33  mwyoung]
 * 	
 * 	Remove pmap_reflect_reference garbage.
 * 	[88/12/09            mwyoung]
 * 	
 * 	More include file fixups.
 * 	[88/11/22            mwyoung]
 * 	
 * 	Fix include file references.
 * 	[88/11/22  02:04:58  mwyoung]
 * 	
 * 	Remove old cache entries when installing a new mapping at the
 * 	same virtual address as an old mapping.  [sanzi]
 * 	[88/11/16  20:06:05  mwyoung]
 * 	
 * 	Map out non-kernel pages in pmap_bootstrap.  Add pmap_verify_free.
 * 	[88/11/13  19:17:41  mwyoung]
 * 
 * Revision 2.7  88/11/23  16:26:56  rpd
 * 	Fixed the log.  Picked up changes from sanzi:
 * 	  Renamed pmap_ppage_remove_phys to pmap_remove_ppage_entry.
 * 	  Fixed spl problem in pmap_remove.
 * 	  Fixed lock/spl ordering problem in pmap_pr_phys.
 * 	  Added missing pmap_cache_remove call to pmap_enter.
 * 	  Fixed spl problem in pmap_phys_find_address.
 * 	  Fixed spl problem in pmap_protect, also
 * 	  modifying pmap_protect_phys_page to spl.
 * 	[88/11/23  15:15:35  rpd]
 * 	
 * 	Redid copystr/copyinstr/copyoutstr to make them all similar.
 * 	Fixed bugs in copyinstr (exited at splvm, etc).  Fixed all three
 * 	to work when maxlength <= 0.
 * 	[88/11/23  10:34:58  rpd]
 * 
 * Revision 2.6  88/10/06  13:55:19  sanzi
 * 	Add implementation of pmap_resident_count().  I made this a function
 * 	because	I may want to change it for the ROMP_CACHE case in the
 *	near future.
 * 
 *      Mapin/Mapout now use a back pointer kept in the HATIPT.  
 * 
 * 	Added pmap_lists to keep track of physical pages in a pmap.
 *
 *	Fixed (for the nth time) an uninitialized variable bug in kernacc().
 * 	[88/10/04  09:52:53  sanzi]
 * 
 * Revision 2.4  88/07/20  16:17:52  rpd
 * Several fixes for simple-locking calls.
 * 
 *  6-May-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	MACH_XP: Don't use vm_page_set_modified.  The pageout daemon
 *	correctly collects modified bits later.
 *	
 *	Use "sys/" rather than "h/" for include files.
 *
 * 15-Dec-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Deleted ROMP_SHARED_SEG code since it wasn't general enough, and
 *	the ROMP_CACHE is coming.  Also completed an implementation for
 *	kernacc(), though all places which use this should be converted
 *	to use the copyin/copyout mechanism. 
 *
 * 14-Dec-87  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Added pmap_phys_address() routine.
 *
 *  6-Oct-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Moved several splx() statements from before lock_read_done()
 *	statements to after the lock_read_done() statement.  This 
 *	prevents an ill timed interrupt which allocates mbuf data 
 *	(such as a lan or un0 interrupt) from deadlocking.   Also 
 *	added splvm() .. splx() calls to pmap_copy_on_write() for
 *	the same reason.  
 *
 *  3-Oct-87  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Corrected pmap_page_protect.
 *
 * 24-Jul-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Modified the way which pmap_protect() used the protection
 *	value to allow routines outside of pmap to call pmap_pr_phys()
 *	directly instead of going through pmap_protect().  This is a
 *	machine dependent hack to support the X buffer and the current
 *	implementation of FPA support that require certain regions of
 *	the kernel be either User Read/Write or User Read.
 *
 * 23-Jul-87  William Bolosky (bolosky) at Carnegie-Mellon University
 *	Fixed up ROMP_DEBUG conditionals around references to 'magic'
 *	variables.
 *
 * 23-Jun-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	ROMP_SHARED_SEG: Added code to support shared (read only)
 *	rosetta sized segments.
 *
 * 16-Jun-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Fixed a ROMP_DEBUG conditional in pmap_alloc_seg() which
 *	prevented it from returning a value when ROMP_DEBUG was off.  
 *	Also placed ROMP_DEBUG conditionals around some other debugging
 *	code.
 *
 * 31-Mar-87  William Bolosky (bolosky) at Carnegie-Mellon University
 *	Turned off trans_virt_addr in favor of an assembly language
 *	version.
 *
 * 26-Mar-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Added pmap_page_protect.
 *
 * 22-Feb-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Changed pmap_pr_phys and pmap_phys_remove to know about memory
 *	holes.
 *
 * 14-Feb-87  Rich Sanzi (sanzi) at Carnegie-Mellon University
 *	Found the "last bug" in Avie's version of the code to maintain the
 *	the resident_count statistic to make the RSS field in ps(1) happy.
 *
 *  1-Feb-87  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Call lock_init with new parameters.
 *
 * 25-Jan-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Eliminated many unused variables.
 *
 * 19-Jan-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Moved warning about "segment not mapped in" in pmap_pr_phys
 *	under "vmdebug" conditional.
 *
 * 12-Dec-86  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Replaced uses of page_shift and page_size with PAGE_SHIFT and
 *	PAGE_SIZE (machine dependent implementations should not know
 *	that these are run time variables, since they may actually *not*
 *	be run time variables).  Also did some other minor cleanup.
 *
 *  8-Apr-86  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Created.
 *
 */

/*
 *	Manages physical address maps.
 *
 *	In addition to hardware address maps, this
 *	module is called upon to provide software-use-only
 *	maps which may or may not be stored in the same
 *	form as hardware maps.  These pseudo-maps are
 *	used to store intermediate results from copy
 *	operations to and from address spaces.
 *
 *	Since the information managed by this module is
 *	also stored by the logical address mapping module,
 *	this module may throw away valid virtual-to-physical
 *	mappings at almost any time.  However, invalidations
 *	of virtual-to-physical mappings must be done as
 *	requested.
 *
 *	In order to cope with hardware architectures which
 *	make virtual-to-physical map invalidates expensive,
 *	this module may delay invalidate or reduced protection
 *	operations until such time as they are actually
 *	necessary.  This module is given full information as
*	to which processors are currently using which maps,
 *	and to when physical maps must be made correct.
 */

#include <cpus.h>
#include <mach_xp.h>
#include <romp_cache.h>
#include <romp_debug.h>

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/map.h>
#include <sys/buf.h>
#include <sys/errno.h>
#include <sys/vm.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/types.h>
#include <sys/thread.h>
#include <sys/zalloc.h>
#include <sys/lock.h>

#include <vm/pmap.h>
#include <vm/vm_map.h>
#include <vm/vm_kern.h>
#include <vm/vm_param.h>
#include <vm/vm_prot.h>
#include <vm/vm_page.h>

#if	ROMP_DEBUG
#include <ca/debug.h>
#endif	ROMP_DEBUG
#include <ca/pcb.h>
#include <ca/rosetta.h>

#include <ca/cache.h>

#if	DISPLAY_DEBUG
#define	DDISPLAY(number) display(((number/10) << 4) + (number%10))
#else	DISPLAY_DEBUG
#define	DDISPLAY(number) /*NOOP*/
#endif	DISPLAY_DEBUG

#define	HACK_SEG 7
/*
 * Note to future hackers: Several places in this code we want to have rosetta
 * do a virtual address translation on an address which isn't in the currently
 * active pmap.  To do this, we munge the segment register represented by
 * HACK_SEG and then coerce the virtual address to be translated into this
 * segment.  If you should, for some strange reason, decide to change the
 * value of HACK_SEG, you should not use segments e (where the kernel lives),
 * f (which is used for IO), and 0 (in case virtual = real is turned on).
 */


pmap_t		active_pmap;	/* XXX should be an array for multiprocessors*/

struct pmap	kernel_pmap_store;	/* statically allocated kernel pmap */
pmap_t		kernel_pmap;

struct zone	*pmap_zone;		/* zone of pmap structures 	*/

struct pmap_list_entry *pmap_list_entries; /* one ppage_list_entry per page */

lock_data_t	pmap_lock;		/* read/write lock for pmap system */

extern int	rose_page_size;		/* hardware page size 		*/
extern int	rose_page_shift;
int		pmap_threshold;		/* pmap_remove's and pmap_protects on
					   more than this # of pages scan the 
					   ipt rather than the virtual address
					   range.  This is initialized in 
					   romp_init.
					 */
vm_offset_t	min_user_pa;		/* Minimum user physical page 	*/
extern long 	*is_in;			/* Initialized by bootstrap 	*/
extern int	physmem;		/* Physical end of memory 	*/
extern int 	ppages_per_vmpage;	/* declared in locore 		*/

/*
 *	Bit array for in-use segments. 
 */
long 		pmap_seg_tab[(RTA_NSEGS + 31) / 32];
short		max_free_seg = RTA_NSEGS - 1;

/* forward declarations */
vm_offset_t 	trans_virt_addr();
short 		pmap_alloc_seg();
void 		pmap_phys_remove();


#if	ROMP_DEBUG
int		magic_paddr = 0;
int 		magic_sid = 0;
int		magic_voff = 0;
int		magic_vaddr = -1;

#define MAGIC_PPAGE(_ppage_,str) \
	if (pmap_debug && ( _ppage_ == magic_ppage)) Debugger(str);

#define MAGIC_VPAGE(_vpage_,str) \
	if (pmap_debug && ( _vpage_ == magic_ppage)) Debugger(str);	
	
#define	PMAP_DEBUGF(option,args)	if (vmdebug&(option)) 	printf args
#else	ROMP_DEBUG
#define	PMAP_DEBUGF(option,args)
#endif	ROMP_DEBUG

#define	DEBUG_PMAP_MISC		0x01
#define	DEBUG_PMAP_PROTECT	0x02
#define	DEBUG_PMAP_LIST		0x04
#define	DEBUG_PMAP_INIT		0x08
#define	DEBUG_PMAP_REMOVE	0x10
#define	DEBUG_PMAP_ENTER	0x20
#define	DEBUG_PMAP_SEGMENT	0x40
#define	DEBUG_PMAP_MAPOUT	0x80
#define	DEBUG_PMAP_MAPIN	DEBUG_PMAP_MAPOUT

#if	ROMP_CACHE

#define	pmap_cache_init(ls,av)			\
				cache_init(ls,av)

#define	pmap_cache_remove(pmap,sid,page)	\
				cache_remove(pmap,sid,page)

#define	pmap_cache_remove_range(pmap,s,e) 	\
				cache_remove_range(pmap,s,e)

#define	pmap_cache_protect_range(pmap,s,e,prot) 	\
				cache_protect_range(pmap,s,e,prot)

#define	pmap_cache_remove_page(addr)		\
				cache_remove_page(addr)

#define	pmap_cache_enter(pmap,sid,vp,pp,prot)	\
				cache_enter(pmap,sid,vp,pp,prot)

#else	ROMP_CACHE
#define	pmap_cache_init(ls,av)
#define	pmap_cache_remove(pmap,sid,page)		
#define	pmap_cache_remove_range(pmap,s,e)
#define	pmap_cache_protect_range(pmap,s,e,prot) 
#define	pmap_cache_remove_page(addr)		
#define	pmap_cache_enter(pmap,sid,vp,pp,prot)	

#endif	ROMP_CACHE

/*
 * pmap_add_ppage_entry
 */
pmap_add_ppage_entry(pmap,v,p)
struct pmap *pmap;
vm_offset_t v;
vm_offset_t p;
{
    pmap_list_entry_t entry;

    /*
     * before memory allocator initialized?
     */
 
    if (pmap_list_entries == 0) return;

    PMAP_DEBUGF(DEBUG_PMAP_LIST,("pmap_add_ppage_entry(pmap,v,p) (0x%x,0x%x,0x%x)\n",
		pmap,v,p));
		
    entry = &pmap_list_entries[atop(p)];
    if (entry->in_use) {
	printf("pmap_add_ppage_entry pmap %x, entry %x",
		pmap,entry);
	panic("pmap_add_ppage_entry : in_use page");
    }

    entry->vaddr = v;
    entry->paddr = p;
    entry->in_use = TRUE;
    entry->pmap = pmap;
    queue_enter(&pmap->ppage_list,entry,pmap_list_entry_t,pmap_pages);
    pmap->ppage_count++;
}

pmap_remove_ppage_entry(paddr)
vm_offset_t paddr;
{
    	pmap_t pmap;
	queue_head_t *list;
	struct pmap_list_entry *entry;


    PMAP_DEBUGF(DEBUG_PMAP_LIST,("pmap_remove_ppage_entry(paddr) (0x%x)\n",
		paddr));

#ifdef	notyet
	entry = &pmap_list_entries[atop(paddr)];
	pmap = entry->pmap;
	queue_remove(&pmap->ppage_list,entry,pmap_list_entry_t, pmap_pages);
	pmap->ppage_count--;
	entry->in_use = FALSE;
#else	notyet
	/*
	 * Sanity check
	 */
	 
	pmap = pmap_list_entries[atop(paddr)].pmap;

	/*
	 * Its entirely possible that with the cache in place,
	 * we get here with no pmap associated with this page.
	 * Probably we should find the current owner and reset
	 * this field, but for now, we'll just punt.
	 */
	if (pmap == PMAP_NULL) return;

	list = &pmap->ppage_list;

	entry = (struct pmap_list_entry *) queue_first(list);
	while (!queue_end(list,(queue_entry_t) entry)) {
	    if (entry->paddr == paddr) {
		goto sanity_ok;
	    }
	    entry = (pmap_list_entry_t) queue_next(&entry->pmap_pages);
	}
	printf("pmap=%x, list=%x, paddr=%x, entry=%x\n",
		pmap,&pmap->ppage_list,paddr,&pmap_list_entries[atop(paddr)]);
	panic("pmap_remove_ppage_entry failed sanity check!");

sanity_ok:
	/*
	 * remove it from the pmap ppage list,
	 * and then free it.
	 */
	queue_remove(list,entry,pmap_list_entry_t, pmap_pages);
	pmap->ppage_count--;	
	entry->in_use = FALSE;
	entry->pmap = PMAP_NULL;
#endif	notyet

}

pmap_ppage_chain_check(paddr,s)
vm_offset_t paddr;
char *s;
{
    	pmap_t pmap;
	queue_head_t *list;
	struct pmap_list_entry *entry;
	int count;
	boolean_t saw_ppage = FALSE;

	pmap = pmap_list_entries[atop(paddr)].pmap;
	if (pmap == PMAP_NULL) return;
	count = pmap->ppage_count;

	list = &pmap->ppage_list;
	entry = (struct pmap_list_entry *) queue_first(list);
	while (!queue_end(list,(queue_entry_t) entry)) {
	    if (entry->paddr == paddr) {
		saw_ppage = TRUE;
	    }
	    count--;
	    entry = (pmap_list_entry_t) queue_next(&entry->pmap_pages);	    
	}

	if (count != 0) {
	    printf("pmap %x, list %x, paddr %x\n",
	    	pmap,list,paddr);
	    printf("pmap_ppage_chain_check %s\n",s);
	    panic("bonk chain check!");
	}
	if (!saw_ppage) {
	    printf("pmap %x, list %x, paddr %x\n",
	    	pmap,list,paddr);	    
	    printf("pmap_ppage_chain_check %s\n",s);
	    panic("bonk chain ppage!");
	}
		
	list = &pmap->ppage_list;
}


/*
 *	Initialize the physical map module.
 *
 *	For now, VM is already on, we only need to map the
 *	specified memory.
 */
vm_offset_t pmap_map(virt, start, end, prot)
	register vm_offset_t	virt;
	register vm_offset_t	start;
	register vm_offset_t	end;
	int		prot;
{

	PMAP_DEBUGF(DEBUG_PMAP_INIT,
		("Entering pmap_map(v,s,e,prot) (0x%x,0x%x,0x%x,0x%x)\n",
			virt,start,end,prot));

	while (start < end) {
		pmap_enter(kernel_pmap, virt, start, prot, FALSE);
		virt += PAGE_SIZE;
		start += PAGE_SIZE;
	}
	return(virt);
}

/*	
 *	Allocate a segment number which is currently unused.
 *	Current algorithm allocates by doing a linear scan of an
 *	array.  XXX - Should be done with a free list.
 */
short pmap_alloc_seg()

{
	register short	cur_seg;

	cur_seg = max_free_seg;
	while (cur_seg > 0) {
		if (isclr(pmap_seg_tab, cur_seg))
			break;
		cur_seg--;
	}

	if (cur_seg <= 0)
		panic("out of segments");

	setbit(pmap_seg_tab, cur_seg);
	max_free_seg = cur_seg;

        PMAP_DEBUGF(DEBUG_PMAP_SEGMENT,("pmap_alloc_seg: 0x%x.\n",cur_seg));
	return(cur_seg);
}


/*
 *	Deallocate a formerly used segment (duplicate deallocates are NOT ok).
 */
pmap_dealloc_seg(segid)
	short	segid;
{
    	PMAP_DEBUGF(DEBUG_PMAP_SEGMENT,("pmap_dealloc_seg: 0x%x\n",segid));

	if (isclr(pmap_seg_tab,segid))
		panic("pmap_dealloc_seg: dup free");

	clrbit(pmap_seg_tab,segid);
	if (max_free_seg < segid)
		max_free_seg = segid;
}

/*
 *	Initialize the segment allocation system.
 */
void pmap_init_seg_alloc()

{
	register int	current_seg = RTA_NSEGS;

	while (current_seg >= 0) {
		clrbit(pmap_seg_tab,current_seg);
		current_seg--;
	}

	setbit(pmap_seg_tab, RTA_SID_SYSTEM);	/* Allocate system segment */
}

/*
 *	Bootstrap the system enough to run with virtual memory.
 *	Called with mapping ON.
 *
 *	Parameters:
 *	load_start:	PA where kernel was loaded
 *	avail_start	PA of first available physical page
 *	avail_end	PA of last available physical page
 *	virtual_avail	VA of first available page (after kernel bss)
 *	virtual_end	VA of last available page (end of kernel addr space)
 *
 *	&start		start of kernel text
 *	&etext		end of kernel text
 *
 */
void pmap_bootstrap(load_start, avail_start, avail_end,
			virtual_avail, virtual_end)
	vm_offset_t	load_start;
	vm_offset_t	*avail_start;	/* IN/OUT */
	vm_offset_t	*avail_end;	/* IN/OUT */
	vm_offset_t	*virtual_avail;	/* IN/OUT */
	vm_offset_t	*virtual_end;	/* OUT */
{
	int			index;
	void			pmap_pr_phys();
	int 			i;
	vm_size_t		s;

	/*
	 *	Initialize the "is it mapped in" table.  lohatipt makes
	 *	sure that all pages are indeed mapped in when we start up.
	 *
	 * 	Also initialize the table of back pointers.  Initially,
	 *	the code in lohatipt.s sets up the page table such that
	 * 	system page i maps to real page i.  	
	 */
	for (index = 0;index < physmem;index++) {
	    	struct hatipt_entry *ipte, *next;
		ipte = &RTA_HATIPT[index];
		if ((ipte->hat_ptr != RTA_UNDEF_PTR) &&
		    (!RTA_ENDCHAIN(ipte->hat_ptr))) {
			next = &RTA_HATIPT[ipte->hat_ptr];
			next->rev_ptr = IPT_HATLINK | index;
		} 
		setbit(is_in,index);
	}

	/*
	 *	The kernel's pmap is statically allocated so we don't
	 *	have to use pmap_create, which is unlikely to work
	 *	correctly at this part of the boot sequence.
	 */

	kernel_pmap = &kernel_pmap_store;
	lock_init(&pmap_lock, FALSE);
	simple_lock_init(&kernel_pmap->lock);
	queue_init(&kernel_pmap->ppage_list);
	queue_init(&kernel_pmap->cache_entries);
	kernel_pmap->ppage_count = 0;
	kernel_pmap->cache_count = 0;

	pmap_init_seg_alloc();/* Initialize the rosetta segment allocator. */


	for (index = 0; index < NUSR_SEGS; index++)
		kernel_pmap->sidtab[index] = 0;
	/*
 	 *	We do not need to set the system segment in the kernel pmap
 	 *	because the system segment (SYS_SEG) is not considered part
	 * 	of a pmap, but rather is a constant (and ser(SYS_SEG) is never
	 *	changed after it is initialized by lohatipt.s).
	 */
	kernel_pmap->ref_count = 1;
	
	*virtual_avail = round_page(VM_MIN_KERNEL_ADDRESS + mem_size); /*XXX*/
	*virtual_end = trunc_page(VM_MAX_KERNEL_ADDRESS);
	
	ppages_per_vmpage = (PAGE_SIZE / rose_page_size);

	/*
	 * Grab some physical memory for the auxiallary page table
	 * information.
	 */
	s = round_page(atop(mem_size) * sizeof(struct pmap_list_entry));
	*avail_end -= s;
	
	pmap_list_entries = (pmap_list_entry_t) (*avail_end + load_start);

	for (i = 0;  i < atop(mem_size); i++) {
	    	pmap_list_entry_t entry = &pmap_list_entries[i];
		vm_offset_t p_addr = (vm_offset_t) ( i << PAGE_SHIFT);
		entry->pmap = kernel_pmap;
		entry->in_use = TRUE;
		entry->vaddr = p_addr + load_start;
		entry->paddr = p_addr;
		queue_enter(&kernel_pmap->ppage_list,
				entry,
				pmap_list_entry_t,
				pmap_pages);
		kernel_pmap->ppage_count++;
	}

	/*
	 *	Initialize the shared-memory cache hash table.
	 */
	pmap_cache_init(load_start,avail_end);
	
	min_user_pa = round_page((vm_offset_t)*avail_end) >> rose_page_shift;
				 /*No user physical pages yet*/
				 
	/*
	 *	Remove unused mappings so that consistency code will work.
	 */

	mapout(a_to_rosetta_page(*avail_start),
	       a_to_rosetta_page((*avail_end) - (*avail_start)));

	return;
}

/*
 *	Initialize the pmap module.
 *	Called by vm_init, to initialize any structures that the pmap
 *	system needs to map virtual memory.
 */
void pmap_init(phys_start, phys_end)
	vm_offset_t	phys_start, phys_end;
{
	vm_size_t		s;
	extern vm_size_t	mem_size;
	extern void		pmap_make_kernel_text_readable();/* forward */

#ifdef	lint
	phys_start++; phys_end++;
#endif	lint

	/*
	 *	Create the zone of physical maps.
	 */

	s = (vm_size_t) sizeof(struct pmap);
	pmap_zone = zinit(s, 400*s, 4096, FALSE, "pmap"); /* XXX */

	pmap_make_kernel_text_readable();
}

/*
 *	Indicate which physical addresses can be used for
 *	user memory.  [Don't use the hole.]
 */
boolean_t pmap_valid_page(phys_addr)
	vm_offset_t	phys_addr;
{
	return((phys_addr < holestart) || (phys_addr >= holeEnd));
}

/*
 *	Create and return a physical map.
 *
 *	If the size specified for the map
 *	is zero, the map is an actual physical
 *	map, and may be referenced by the
 *	hardware.
 *
 *	If the size specified is non-zero,
 *	the map will be used in software only, and
 *	is bounded by that size.
 */
pmap_t pmap_create(size)
	vm_size_t	size;
{
	register pmap_t			p;
	register pmap_statistics_t	stats;
	int				index;

	/*
	 *	A software use-only map doesn't even need a map.
	 */

	if (size != 0) {
		return(PMAP_NULL);
	}

	p = (pmap_t) zalloc(pmap_zone);

	/*
	 *	Initialize the sidtab.
	 */
	for (index = 0; index < NUSR_SEGS; index++) {
		p->sidtab[index] = 0;
	}

	p->ref_count = 1;

	simple_lock_init(&p->lock);

	/*
	 *	Initialize statistics.
	 */

	stats = &p->stats;
	stats->resident_count = 0;
	stats->wired_count = 0;

	queue_init(&p->ppage_list);
	queue_init(&p->cache_entries);
	p->ppage_count = 0;
	p->cache_count = 0;

	PMAP_DEBUGF(DEBUG_PMAP_INIT,("pmap_create returns 0x%x.\n",p));

	return(p);
}

/*
 *	Retire the given physical map from service.
 *	Should only be called if the map contains
 *	no valid mappings.
 */

void pmap_destroy(p)
	pmap_t	p;
{
	int		c, index, pl;

	DDISPLAY(1);
	if (p != PMAP_NULL) {
		simple_lock(&p->lock);
		c = --p->ref_count;
		simple_unlock(&p->lock);
		if (c == 0) {
			if (p->ppage_count != 0)
				panic("pmap_destroy: ppage count not zero");
			if (p->cache_count != 0)
				panic("pmap_destroy: cache count not zero");
			pl = splvm();
			for (index = 0;index < NUSR_SEGS;index++) 
				if (p->sidtab[index]) 
					pmap_dealloc_seg(p->sidtab[index]);
			zfree(pmap_zone, p);
			splx(pl);
		}
	}
	DDISPLAY(0);
}

/*
 *	Add a reference to the specified pmap.
 */

void pmap_reference(p)
	pmap_t	p;
{
	if (p != PMAP_NULL) {
		simple_lock(&p->lock);
		p->ref_count++;
		simple_unlock(&p->lock);
	}
}

/*
 *	Remove the given range of (virtual) addresses
 *	from the specified map.
 *
 *	It is assumed that the start and end are properly
 *	rounded to the ROMP page size.
 *
 *	This routine requires that the start and end address both
 *	be in the same segment.
 *
 *	All pmap_remove does is to decide whether we are removing enough
 *	of a range to warrent scanning through the ipt as opposed to checking
 *	each page in the range and then calls the appropriate (internal)
 *	version of pmap_remove.
 */

void pmap_remove(map, s, e)
	pmap_t		map;
	vm_offset_t	s, e;
{
	queue_head_t *list;
	struct pmap_list_entry *entry, *next_entry;
	int			spl;

    	PMAP_DEBUGF(DEBUG_PMAP_REMOVE,("pmap_remove(%x,%x,%x)\n",map,s,e));

	if (map == PMAP_NULL) return;

	spl = splvm();
    	if (atoseg(s) != SYS_SEG) { pmap_cache_remove_range(map,s,e);} 
	
	list = &map->ppage_list;
	entry = (struct pmap_list_entry *) queue_first(list);
	while (!queue_end(list,(queue_entry_t) entry)) {
	    next_entry = (pmap_list_entry_t) queue_next(&entry->pmap_pages);
	    if ((entry->vaddr >= s) && (entry->vaddr < e)) {
		/*
		 * remove it from the hatipt
		 * 	(might not be there since the cache fault
		 *	 handler could have remapped the page)
		 * remove it from the TLBs,
		 * remove it from the pmap ppage list,
		 * and then free it.
		 */
#ifdef	notdef
		/* the following should work, plus it should be faster */
		/* it needs the mmu_extract_addrtag macro added to pmap.h*/
		
		if (mmu_extract_addrtag(a_to_rosetta_page(entry->paddr)) ==
			( pmap->sidtab[atoseg(entry->vaddr)] << RTA_VPAGE_BITS
			  || a_to_rosetta_page(segoffset(entry->vaddr))))
#endif	notdef		    
		if (pmap_extract(map,entry->vaddr) != 0) {

		    mapout(a_to_rosetta_page(entry->paddr),ppages_per_vmpage);
		    tlb_invalidate_seg(atoseg(entry->vaddr));
		}
		queue_remove(list,entry,pmap_list_entry_t, pmap_pages);
		map->ppage_count--;
		entry->pmap = PMAP_NULL;
		entry->in_use = FALSE;
	    }
	    entry = next_entry;
	}
	splx(spl);

}

boolean_t pmap_verify_free(phys)
	register
	vm_offset_t	phys;
{
	register
	int		i;

#if	ROMP_CACHE
	if (!cache_verify_free(phys)) {
		printf("pmap_verify_free: it's in the cache\n");
		return(FALSE);
	}
#endif	ROMP_CACHE

#if	0
	/* I'm not sure why this is not a valid check to make */
	if (pmap_list_entries[atop(phys)].pmap != PMAP_NULL) {
		printf("pmap_verify_free: pmap_list_entry is bad\n");
		return(FALSE);
	}
#endif	0

	for (i = ppages_per_vmpage, phys >>= rose_page_shift; --i >= 0; phys++)
		if (isset(is_in, phys))
			return(FALSE);
	return(TRUE);
}

/*
 *	Routine:	pmap_remove_all
 *	Function:
 *		Removes this physical page from
 *		all physical maps in which it resides.
 *		Reflects back modify bits to the pager.
 *
 *		Since a physical page may have at most one virtual
 * 		address on the rosetta, just map it out.
 *
 *		XXX - pmap locking.
 */
void pmap_remove_all(phys)
	vm_offset_t	phys;
{
	vm_offset_t		addr = phys;
	int			s = splvm();

	DDISPLAY(5);

	PMAP_DEBUGF(DEBUG_PMAP_REMOVE,("pmap_remove_all: 0x%x.\n",phys));

	tlb_invalidate_all();

	pmap_cache_remove_page(addr);

	/*
	 *  If the cache is in place, the is_in bit can
	 *  be clear, but the page still can belong to a pmap.
	 */
	pmap_remove_ppage_entry(addr);
	
	if (isset(is_in,addr>>rose_page_shift)) {
		DDISPLAY(35);
		mapout(a_to_rosetta_page(addr),ppages_per_vmpage);
		DDISPLAY(5);
	}

#if	MACH_XP
#else	MACH_XP
	for (;addr < PAGE_SIZE + phys; addr += rose_page_size) {
		if (get_mod_bit(addr>>0xb))
			vm_page_set_modified(PHYS_TO_VM_PAGE(addr));
	}
#endif	MACH_XP

	DDISPLAY(0);
	splx(s);
}

/*
 *	Macro:		rtakey
 *	Function:
 *		Map machine-independent VM protection to MMU access key.
 *	Note:
 *		Write capability implies read capability; execute capability
 *		is ignored.  The Mach interface allows these reductions.
 *
 *		Total lack of capability (VM_PROT_NONE) must be handled
 *		separately, as no access code applies.  The page in question
 *		must not be mapped at all.
 *
 *		To make memory read-only to the kernel, it must be readable
 *		by user programs.  This makes it impossible to send kernel
 *		memory copy-on-write from a segment that is accessible by
 *		users.  [XXX THIS PRECAUTION IS NOT CURRENTLY TAKEN.]
 */
#define	rtakey(map, prot)						\
		(((prot) & VM_PROT_WRITE) ?				\
		 (((map) == kernel_pmap) ? RTA_KEY_KW : RTA_KEY_UW) :	\
		 RTA_KEY_URKR)						


/*
 *	Routine:	pmap_copy_on_write
 *	Function:
 *		Remove write privileges from all
 *		physical maps for this physical page.
 */
void pmap_copy_on_write(phys)
	vm_offset_t	phys;
{
	int	ppage;
	struct hatipt_entry *ipte;
	int s = splvm();
	
	DDISPLAY(6);
	/*
	 *	Lock the entire pmap system, since we may be changing
	 *	several maps.
	 */
	lock_write(&pmap_lock);
	for (ppage = phys; ppage < PAGE_SIZE + phys; ppage += rose_page_size)
		if (isset(is_in,ppage>>rose_page_shift)) {
			ipte = &RTA_HATIPT[ppage>>rose_page_shift];

			/*
			 *	Because a mapping exists, it must already
			 *	have read permission; therefore, it is safe
			 *	to give out read permission here.
			 *
			 *	It is not possible to make kernel memory
			 *	read-only without letting users also read it.
			 *	XXX The kernel should only let users have access
			 *	to segments that don't contain sensitive data!
			 */

			ipte->key_addrtag &= ~(3 << RTA_KEY_SHIFT);
			ipte->key_addrtag |= RTA_KEY_URKR << RTA_KEY_SHIFT;
		}
	pmap_cache_remove_page(phys);

	tlb_invalidate_all();
	lock_write_done(&pmap_lock);
	DDISPLAY(0);
	splx(s);
}

void pmap_pr_phys_range(min_ppage, max_ppage, sid, s, e, map, internal_prot)
	int			min_ppage;
	register int		max_ppage, sid;
	register vm_offset_t	s, e;
	pmap_t			map;
	int			internal_prot;

{
	register int		pagenum;
	struct  hatipt_entry	*ipte;
	long			key_addrtag;
	
#ifdef	lint
	map++; /* XXX This routine is internal-use-only, but has unused arguments? */
#endif	lint

	for (pagenum = min_ppage; pagenum < max_ppage; pagenum++) {
		ipte = &RTA_HATIPT[pagenum];
		key_addrtag = ipte->key_addrtag;
		if ((isset(is_in,pagenum))
		  && (((key_addrtag & RTA_SID_MASK) >> RTA_VPAGE_BITS) == sid)
		  && ((key_addrtag & RTA_VPAGE_MASK) >= s)
		  && ((key_addrtag & RTA_VPAGE_MASK) < e)) {
			ipte->key_addrtag &= ~(3 << RTA_KEY_SHIFT);
			ipte->key_addrtag |= internal_prot;
		  }
	     }
}

void pmap_pr_phys(map, start, end, internal_prot)
	pmap_t		map;
	vm_offset_t	start, end;
	int		internal_prot;	
{
	int		pl, seg = atoseg(start);
	int		sid;
	vm_offset_t	s, e;	
	
	if (atoseg(start) != atoseg(end-1))
		panic("pmap_pr_phys (too big)");

	pl = splvm();

	lock_read(&pmap_lock);
	simple_lock(&map->lock);

	DDISPLAY(8);

	PMAP_DEBUGF(DEBUG_PMAP_PROTECT,
		("pmap_pr_phys: map= 0x%x, s = 0x%x, e=0x%x, prot=0x%x.\n",
				map,start,end,internal_prot));

	if (seg == SYS_SEG)
		sid = RTA_SID_SYSTEM;
	else
		sid = map->sidtab[seg];

	if (sid == 0) {
	    	PMAP_DEBUGF(DEBUG_PMAP_PROTECT,
			  ("pmap_pr_phys: seg %x not mapped in, pmap=0x%x.\n",
			  	seg,map));
		simple_unlock(&map->lock);
		lock_read_done(&pmap_lock);
		splx(pl);
		return;
	}
	/*
	 * Why aren't the following shifts using rose_page_size??
	 */

	if (atoseg(end) == atoseg(start))
		e = segoffset(end) >> 0xb;
	else
		e = (0x10000000) >> 0xb; /*XXX*/

	s = segoffset(start) >> rose_page_shift;		

	if (seg != SYS_SEG) { pmap_cache_remove_range(map,start,end); } 

	/*
 	 *	Optimization:  Do not check the kernel text
	 *	if we are removing from a non-system segment (we probably
	 *	do not need to anyway, but just for safety's sake...)
	 */
	pmap_pr_phys_range(((sid == RTA_SID_SYSTEM) ? 0 : min_user_pa),
		a_to_rosetta_page(holestart),
		sid, s, e, map, internal_prot);

	if (ptoa(physmem) > holestart) 
		pmap_pr_phys_range(a_to_rosetta_page(holeEnd),
			vm_to_rosetta_page(physmem) +
			a_to_rosetta_page(holeEnd - holestart),
			sid, s, e, map, internal_prot);

	tlb_invalidate_seg(seg);
	simple_unlock(&map->lock);
	lock_read_done(&pmap_lock);

	DDISPLAY(0);
	splx(pl);
}


pmap_protect_phys_page(phys_addr,prot)
vm_offset_t phys_addr;
int prot;
{
    	vm_offset_t p;
	struct hatipt_entry *ipte;
        int     spl;

	spl = splvm();
	
	for (p = phys_addr; p < phys_addr + PAGE_SIZE; p += rose_page_size) {
		ipte = &RTA_HATIPT[a_to_rosetta_page(p)];
		ipte->key_addrtag &= ~(3 << RTA_KEY_SHIFT);
		ipte->key_addrtag |= prot << RTA_KEY_SHIFT;
	}
	splx(spl);	
}

/*
 *	Set the physical protection on the
 *	specified range of this map as requested.
 */
void pmap_protect(map, s, e, prot)
	pmap_t		map;
	vm_offset_t	s, e;
	vm_prot_t	prot;
{
    queue_head_t *list;
    struct pmap_list_entry *entry;
    int     mmu_prot = rtakey(map, prot);

    if (map == PMAP_NULL)
	return;

    PMAP_DEBUGF(DEBUG_PMAP_PROTECT, ("pmap_protect(%x,%x,%x,%x)\n", map, s, e, prot));

    if (prot == VM_PROT_NONE) {
	/*
	 *	There's no equivalent MMU protection for "none".
	 *	Remove the mappings.
	 */

    	pmap_remove(map, s, e);
	return;
    }

    if (atoseg(s) != SYS_SEG) {
        int     spl;
	spl = splvm();
	pmap_cache_protect_range(map, s, e, prot);
	splx(spl);
    }
    list = &map->ppage_list;
    entry = (struct pmap_list_entry *) queue_first(list);
    while (!queue_end(list, (queue_entry_t) entry)) {
	if ((entry->vaddr >= s) && (entry->vaddr < e)) {
	    if (pmap_extract(map, entry->vaddr) != 0) {
		pmap_protect_phys_page(entry->paddr, mmu_prot);
		tlb_invalidate_seg(atoseg(entry->vaddr));
	    }
	}
	entry = (pmap_list_entry_t) queue_next(&entry->pmap_pages);
    }
}


/*
 *
 * mapout - remove a given physical address from the hat/ipt.
 *
 */
 
mapout(ppage,count)
	vm_offset_t	ppage;
	int count;
{

#if	USE_ASSEMBLY
#else	USE_ASSEMBLY
	register struct hatipt_entry *ipte, *prev, *next;
	int i;
#endif	USE_ASSEMBLY

	PMAP_DEBUGF(DEBUG_PMAP_MAPOUT,("mapout(0x%x,0x%x)\n",ppage,count));

#if	USE_ASSEMBLY
	lo_mapout(ppage,count);
#else	USE_ASSEMBLY	
	

	for (i=0; i < count; i++, ppage += 1) {

	    if (isclr(is_in,ppage)) {
		if (vmdebug)
		    printf("mapout: already out ppage %x count %d\n",
		    	ppage,count);
#if	ROMP_CACHE
	/*
	 * It is ok to try to mapout a page which is already out.
	 *  lo_lookup() may do this.
	 *  pmap_remove() may also do it.
	 */
		return;
#else	ROMP_CACHE
		panic("mapout: already out");
#endif	ROMP_CACHE
	    }
	    clrbit(is_in,ppage);
  	    ipte = &RTA_HATIPT[ppage];

	    if (!RTA_ENDCHAIN(ipte->ipt_ptr)) {
	    	next = &RTA_HATIPT[ipte->ipt_ptr];
	    	next->rev_ptr = ipte->rev_ptr;
	    }
	    
	    if (RTA_ENDCHAIN(ipte->rev_ptr)) {
		/*
		 * The reverse link points to a hat entry
		 */
		prev = &RTA_HATIPT[ipte->rev_ptr&IPT_HATLINK_MASK];
		prev->hat_ptr = ipte->ipt_ptr;
	    } else {
		/*
		 * The reverse link points to just another ipt entry.
		 */
		prev = &RTA_HATIPT[ipte->rev_ptr];
		prev->ipt_ptr = ipte->ipt_ptr;
	    }
	    ipte->ipt_ptr = RTA_UNDEF_PTR;
	    ipte->rev_ptr = RTA_UNDEF_PTR;	    
	}
#endif	USE_ASSEMBLY	
}

mapin(ppage, vpage, prot, count, sid)
	vm_offset_t ppage, vpage;
	int prot;
	int count, sid;
{
#if	USE_ASSEMBLY
#else	USE_ASSEMBLY
	register struct hatipt_entry *ipte, *head, *next_ipte;
	u_int key;
	int i;
	int hash_head;
#endif	USE_ASSEMBLY

    	PMAP_DEBUGF(DEBUG_PMAP_MAPIN,("mapin(0x%x,0x%x,0x%x,0x%x,0x%x)\n",
		    ppage, vpage, prot, count, sid));

#if	USE_ASSEMBLY
	lo_mapin(ppage, vpage, prot, count, sid);
#else	USE_ASSEMBLY
	if (isset(is_in,ppage)) {
	        printf("mapin: ppage %x, vpage %x, sid %x, count %d\n",
			ppage,vpage,sid,count);
		panic("mapin: page already in");
	}

	if ((sid != RTA_SID_SYSTEM) && (ppage < min_user_pa))
		min_user_pa = ppage;

	for (i = 0; i < count ; i++, ppage += 1, vpage += 1) {
		/* point to the ipte for this page frame */
		ipte = &RTA_HATIPT[ppage];

		/* determine the key, write, tid, lockbits */
		key = prot;	  /* non-special segments only */

		/* fill in the fields to reflect the new virt addr and prot */
		ipte->key_addrtag = (key << RTA_KEY_SHIFT)
			|(sid << RTA_VPAGE_BITS)
			|((rose_page_size == 0x1000) ? vpage << 1 : vpage);
			
		ipte->w = 1;		  /* special segments only */
		ipte->tid = 0 ;		  /* special segments only */
		ipte->lockbits = -1;	  /* special segments only */

		/* hash to find the head of the ipte chain */
		head = &RTA_HATIPT[hash_head = RTA_HASH(sid, vpage)];

		if (!RTA_ENDCHAIN(head->hat_ptr)) {
		    next_ipte = &RTA_HATIPT[head->hat_ptr];
		    next_ipte->rev_ptr = ppage;
		}
		ipte->ipt_ptr = head->hat_ptr;
		ipte->rev_ptr = hash_head | IPT_HATLINK;
		
		set_hatptr(head, ppage);

		/* Mark it in in our external table. */
		setbit(is_in,ppage);

	}
#endif	USE_ASSEMBLY	
}

remap(sid,vpage,ppage,prot)
int sid;
vm_offset_t vpage;
vm_offset_t ppage;
int prot;
{
#if	USE_ASSEMBLY
#else	USE_ASSEMBLY
    struct hatipt_entry *ipte, *prev, *next, *head;
    int i;
    int hash_head;
#endif	USE_ASSEMBLY

    PMAP_DEBUGF(DEBUG_PMAP_MAPIN,("remap(0x%x,0x%x,0x%x,0x%x)\n",
		    sid, vpage, ppage, prot));
#if	USE_ASSEMBLY
	lo_remap(sid, vpage, ppage, prot);
	PMAP_DEBUGF(DEBUG_PMAP_MAPIN,("remap done(0x%x,0x%x,0x%x,0x%x)\n",
		    sid, vpage, ppage, prot));	
#else	USE_ASSEMBLY
    for (i = 0; i < ppages_per_vmpage; ppage++, vpage++, i++) {
  	    ipte = &RTA_HATIPT[ppage];

	    if (isset(is_in,ppage)) {
		if (!RTA_ENDCHAIN(ipte->ipt_ptr)) {
		    next = &RTA_HATIPT[ipte->ipt_ptr];
		    next->rev_ptr = ipte->rev_ptr;
		}

		if (RTA_ENDCHAIN(ipte->rev_ptr)) {
		    /*
		     * The reverse link points to a hat entry
		     */
		    prev = &RTA_HATIPT[ipte->rev_ptr&IPT_HATLINK_MASK];
		    prev->hat_ptr = ipte->ipt_ptr;
		} else {
		    /*
		     * The reverse link points to just another ipt entry.
		     */
		    prev = &RTA_HATIPT[ipte->rev_ptr];
		    prev->ipt_ptr = ipte->ipt_ptr;
		}
	    } else {
		setbit(is_in,ppage);
	    }

	    /*
	     * fill in the fields to reflect the new virt addr and prot
	     */
	     
	    ipte->key_addrtag = (prot << RTA_KEY_SHIFT)
			|(sid << RTA_VPAGE_BITS)
			|((rose_page_size == 0x1000) ? vpage << 1 : vpage);

	    ipte->w = 1;		  /* special segments only */
	    ipte->tid = 0 ;		  /* special segments only */
	    ipte->lockbits = -1;	  /* special segments only */

	    /* hash to find the head of the ipte chain */
	    head = &RTA_HATIPT[hash_head = RTA_HASH(sid, vpage)];
  	    if (!RTA_ENDCHAIN(head->hat_ptr)) {
		    next = &RTA_HATIPT[head->hat_ptr];
		    next->rev_ptr = ppage;
	    }
	    ipte->ipt_ptr = head->hat_ptr;	    
	    ipte->rev_ptr = hash_head | IPT_HATLINK;
	    set_hatptr(head, ppage);
    }
#endif	USE_ASSEMBLY
}


int shm_faults = 0;
/*
 *	Insert the given physical page (p) at
 *	the specified virtual address (v) in the
 *	target physical map with the protection requested.
 *
 *	The 'wired' parameter is silliness from the vax, and is
 *	ignored here.
 *
 *	NB:  This is the only routine which MAY NOT lazy-evaluate
 *	or lose information.  That is, this routine must actually
 *	insert this page into the given map NOW.
 */
void pmap_enter(map, v, p, prot, wired)
	register pmap_t		map;
	register vm_offset_t	v;
	register vm_offset_t	p;
	vm_prot_t		prot;
	boolean_t		wired;
{
	int			sid;
	int			old_sr;
	int			seg = atoseg(v);
	int			pl, this_page;
	int			count;
	vm_offset_t 		vpage =  a_to_rosetta_page(segoffset(v));
	vm_offset_t 		ppage =  a_to_rosetta_page(p);

	DDISPLAY(9);
#if	ROMP_RDB
	if (v == magic_vaddr)
		Debugger("magic vaddr hit");
#endif	ROMP_RDB

	PMAP_DEBUGF(DEBUG_PMAP_ENTER,("pmap_enter, p = 0x%x, v=0x%x, prot = %d, wired = %d.\n",
		p, v, prot, wired));

	if ((map == PMAP_NULL) || (prot == VM_PROT_NONE))
		return;

	pl = splvm();

	lock_read(&pmap_lock);
	simple_lock(&map->lock);

	if (seg != SYS_SEG) {
		sid = map->sidtab[seg];
		if (sid == 0) {
			sid = map->sidtab[seg] = pmap_alloc_seg();

			/* If we are entering in the currently active pmap,
			 * physically enter the new segment id in the rosetta
			 * segment register.
			 *
			 * No need for TLB invalidate here (or it
			 * wouldn''t have faulted).
			 */

			if (map == active_pmap)
				set_rosetta_seg(seg,sid_to_seg(sid));
		}
	} else {
		sid = RTA_SID_SYSTEM;
	}

	/*
 	 * This fakes sr0 into pointing at the segment on which we're
	 * working so that the tva will tell us if the virtual address
	 * is REALLY mapped in.
	 */
	 
	old_sr = get_rosetta_seg(HACK_SEG);
	set_rosetta_seg(HACK_SEG,sid_to_seg(sid));
	tlb_invalidate_seg(HACK_SEG);

	prot = rtakey(map,prot);
	count = PAGE_SIZE/rose_page_size;

#define	pmap_ppage_in_use(p)	(pmap_list_entries[atop(p)].in_use)

	/*
	 * remove the physical page from the pmap->ppages_list
	 */

	if (pmap_ppage_in_use(p)) {
	    pmap_remove_ppage_entry(p);
	    /*
	     * Only add a cache entry if we are sharing this
	     * page with another process
	     */
	    if (sid != RTA_SID_SYSTEM) {
		pmap_cache_enter(map,sid,vpage,ppage,prot);
	    }
	} else {
	    pmap_cache_remove(map,sid,vpage);
	}
	
	if (isset(is_in,ppage)) {
		mapout(ppage,ppages_per_vmpage);
		tlb_invalidate_all();
	}

	/*
	 * the vm system assumes that it can remap a virtual address
	 * without first removing an old mapping.
	 */
	if ((this_page = 
	    trans_virt_addr((HACK_SEG << 28) | segoffset(v))) != -1) {
		mapout(a_to_rosetta_page(this_page),count);
		pmap_remove_ppage_entry(this_page);
	}
	
	mapin(ppage, vpage, prot,count, sid);
	pmap_add_ppage_entry(map,v,p);
	
	set_rosetta_seg(HACK_SEG,old_sr);
	tlb_invalidate_seg(HACK_SEG);
	tlb_invalidate_seg(atoseg(v));

	simple_unlock(&map->lock);
	lock_read_done(&pmap_lock);
	DDISPLAY(0);
	splx(pl);
}

/*
 *	Routine:	pmap_resident_count
 *	Function:
 *		Return the number of resident pages in this pmap.
 *
 *	XXX 	We could return some estimate of the number of shared
 *		pages by adding in the size of the cache_entries list,
 *		but this list may contain duplicate mappings, making
 *		the returned value too high.  Therefore, we error
 *		on the pessimistic side, only reporting the number of
 *		pages that this pmap actually has in the mmu.
 */
int	pmap_resident_count(pmap)
pmap_t	pmap;
{
    	return(pmap->ppage_count);
}
	 
/*
 *	Routine:	pmap_extract
 *	Function:
 *		Extract the physical page address associated
 *		with the given map/virtual_address pair.
 */

vm_offset_t pmap_extract(pmap, va)
	pmap_t		pmap;
	vm_offset_t	va;
{
	register int	phys;
	int		pl;
	int 		old_sr;

	pl = splvm();
	simple_lock(&pmap->lock);
	DDISPLAY(11);

	if (atoseg(va) == SYS_SEG) {
		phys = trans_virt_addr(va);
		DDISPLAY(0);
		simple_unlock(&pmap->lock);
		splx(pl);
		return(phys == -1 ? 0 : phys);
	}
	old_sr = get_rosetta_seg(HACK_SEG);
	set_rosetta_seg(HACK_SEG,sid_to_seg(pmap->sidtab[atoseg(va)]));
	tlb_invalidate_seg(HACK_SEG);
	
	phys = trans_virt_addr((HACK_SEG<<28) | segoffset(va));

	set_rosetta_seg(HACK_SEG,old_sr);
	tlb_invalidate_seg(HACK_SEG);

	DDISPLAY(0);
	simple_unlock(&pmap->lock);
	splx(pl);
	return(phys == -1 ? 0 : phys);
}

/*
 *	Routine:	pmap_access
 *	Function:
 *		Returns whether there is a valid mapping for the
 *		given virtual address stored in the given physical map.
 */

boolean_t pmap_access(pmap, va)
	pmap_t		pmap;
	vm_offset_t	va;
{
	boolean_t	ok;
	int		pl, old_sr;

	if (pmap == PMAP_NULL)
		return(FALSE);

	pl = splvm();
	simple_lock(&pmap->lock);
	DDISPLAY(12);

	old_sr = get_rosetta_seg(HACK_SEG);
	set_rosetta_seg(HACK_SEG,sid_to_seg(pmap->sidtab[atoseg(va)]));
	tlb_invalidate_seg(HACK_SEG);

	if (trans_virt_addr((HACK_SEG << 28) | segoffset(va)) == (unsigned)(-1)) 
			/* is it not mapped in? */
		ok = FALSE;
	else
		ok = TRUE;

	set_rosetta_seg(HACK_SEG,old_sr);
	tlb_invalidate_seg(HACK_SEG);
	
	simple_unlock(&pmap->lock);
	splx(pl);
	DDISPLAY(0);

	return (ok);
}

/*
 *	Routine:	pmap_copy
 *	Function:
 *		Copy the range specified by src_addr/len
 *		from the source map to the range dst_addr/len
 *		in the destination map.
 */
void pmap_copy(dst_pmap, src_pmap, dst_addr, len, src_addr)
	pmap_t		dst_pmap;
	pmap_t		src_pmap;
	vm_offset_t	dst_addr;
	vm_size_t	len;
	vm_offset_t	src_addr;
{
#ifdef lint
	pmap_copy(dst_pmap, src_pmap, dst_addr, len, src_addr);
#endif lint
}

/*
 *	Require that all active physical maps contain no
 *	incorrect entries NOW.  [This update includes
 *	forcing updates of any address map caching.]
 *
 *	Generally used to insure that a thread about
 *	to run will see a semantically correct world.
 */
void pmap_update()
{
	/*
	 * everyting should properly invalidate the TLB when it runs, so this
	 * should be OK.
	 *
	 */

}

/*
 *	Routine:	pmap_collect
 *	Function:
 *		Garbage collects the physical map system for
 *		pages which are no longer used.
 *		Success need not be guaranteed -- that is, there
 *		may well be pages which are not referenced, but
 *		others may be collected.
 *	Usage:
 *		Called by the pageout daemon when pages are scarce.
 */
void pmap_collect()
{
}

/*
 *	Routine:	pmap_activate
 *	Function:
 *		Binds the given physical map to the given
 *		processor.  (Since there are only uniprocessor
 *		romp/rosetta's, all this does is remember which
 *		pmap is currently active for pmap_enter.)
 *
 */
void pmap_activate(pmap, th, cpu)
	pmap_t		pmap;
	thread_t	th;
	int		cpu;
{
#ifdef	lint
	th++; cpu++;
#endif	lint
#if	NCPUS > 1
	;YOU LOSE;
#endif	NCPUS > 1
	active_pmap = pmap;
}

/*
 *	Routine:	pmap_deactivate
 *	Function:
 *		Indicates that the given physical map is no longer
 *		in use on the specified processor.
 */
void pmap_deactivate(pmap, th, cpu)
	pmap_t		pmap;
	thread_t	th;
	int		cpu;
{
	active_pmap = PMAP_NULL;
#ifdef lint
	pmap++; th++; cpu++;
#endif lint
}

/*
 *	Routine:	pmap_kernel
 *	Function:
 *		Returns the physical map handle for the kernel.
 */
pmap_t pmap_kernel()
{
    	return (kernel_pmap);
}


/*
 *	pmap_zero_page zeros the specified (machine independent)
 *	page.  
 */
pmap_zero_page(phys)
	register vm_offset_t	phys;
{
	register int s = splvm(); /* Can't interrupt with virt_eq_real on! */
	register int old_real;

	DDISPLAY(13);
	old_real = virt_eq_real(TRUE);
	bzero(phys,PAGE_SIZE);
	set_mod_bit(phys>>0xb,0); /* Zero fill page is clean...*/
	virt_eq_real(old_real);
	DDISPLAY(0);
	splx(s);
}

/*
 *	pmap_copy_page copies the specified (machine independent)
 *	pages.  
 */

pmap_copy_page(src, dst)
	register vm_offset_t	src, dst;
{
	register int s = splvm();
	register int old_real;

	DDISPLAY(14);
	old_real = virt_eq_real(TRUE);
	bcopy(src,dst,PAGE_SIZE);
	virt_eq_real(old_real);
	splx(s);
	DDISPLAY(0);
}

/*
 *	Routine:	pmap_pageable
 *	Function:
 *		Make the specified pages (by pmap, offset)
 *		pageable (or not) as requested.
 *
 *		A page which is not pageable may not take
 *		a fault; therefore, its page table entry
 *		must remain valid for the duration.
 *
 *		This routine is merely advisory; pmap_enter
 *		will specify that these pages are to be wired
 *		down (or not) as appropriate.
 */
pmap_pageable(pmap, start, end, pageable)
	pmap_t		pmap;
	vm_offset_t	start;
	vm_offset_t	end;
	boolean_t	pageable;
{
#ifdef	lint
	pmap++; start++; end++; pageable++;
#endif	lint
}

/*
 *	Routine:	pmap_redzone
 *	Function:
 *		Make the particular address specified invalid
 *		in the hardware map.  Note that the hardware
 *		page size (not the logical page size) determines
 *		how much memory is made invalid.
 *
 *		[Perhaps only meaningful for wired down pages.]
 */
void pmap_redzone(pmap, addr)
	pmap_t		pmap;
	vm_offset_t	addr;
{
#ifdef	lint
	addr++;
#endif	lint

	simple_lock(&pmap->lock);
	/* XXX - Do something! */
	simple_unlock(&pmap->lock);
}

#if	0
This was found to be wanting for speed, and has been moved to loutil.s
/*
 *	This function takes a virtual address and returns the corresponding
 *	physical address if it is mapped in, else returns -1.
 *
 */
vm_offset_t trans_virt_addr(va)
	register vm_offset_t va;
{
	int	phys;

	rosetta_write_lrar(va);
	phys = rosetta_read_trar();
	if (phys < 0)
		return(-1);
	return(phys);
}
#endif	0

/*
 *	copystr - copy a string from kernel vm to kernel vm subject
 *		  to length constraints.
 *
 */
copystr(fromaddr, toaddr, maxlength, lencopied)
	register char *fromaddr, *toaddr;
	register int	maxlength, *lencopied;
{
	register int bytes_copied = 0;
	register int ret = 0;

	DDISPLAY(15);

	while (bytes_copied < maxlength) {
		bytes_copied++;
		if ((*toaddr++ = *fromaddr++) == '\0')
			goto done;
	}
	ret = ENOENT;

      done:
	if (lencopied != NULL) *lencopied = bytes_copied;
	DDISPLAY(0);
	return(ret);
}

/*
 *	Copy routines for moving data between kernel vm and user vm
 *	subject to length and validation checks.
 */

copyinstr(fromaddr, toaddr, maxlength, lencopied)
	register char *fromaddr, *toaddr;
	register int  maxlength, *lencopied;
{
	register int bytes_copied = 0;
	label_t	jmpbuf;
	int pl = splvm();
	int ret = 0;

	DDISPLAY(16);

	if (setjmp(&jmpbuf)) {
		ret = EFAULT;
		goto done;
	}
	current_thread()->recover = (vm_offset_t)&jmpbuf;

	while (bytes_copied < maxlength) {
		bytes_copied++;
		if ((*toaddr++ = *fromaddr++) == '\0')
			goto done;
	}
	ret = ENOENT;

      done:
	if (lencopied != NULL) *lencopied = bytes_copied;
	current_thread()->recover = NULL;
	DDISPLAY(0);
	splx(pl);
	return(ret);
}

copyoutstr(fromaddr, toaddr, maxlength, lencopied)
	register char *fromaddr, *toaddr;
	register int  maxlength, *lencopied;
{
	register int bytes_copied = 0;
	label_t	jmpbuf;
	int pl = splvm();
	int ret = 0;

	DDISPLAY(17);

	if (setjmp(&jmpbuf)) {
		ret = EFAULT;
		goto done;
	}
	current_thread()->recover = (vm_offset_t)&jmpbuf;

	while (bytes_copied < maxlength) {
		bytes_copied++;
		if ((*toaddr++ = *fromaddr++) == '\0')
			goto done;
	}
	ret = ENOENT;

      done:
	if (lencopied != NULL) *lencopied = bytes_copied;
	current_thread()->recover = NULL;
	DDISPLAY(0);
	splx(pl);
	return(ret);
}

copyin(from,to,length)
	register caddr_t from, to;
	register unsigned length;
{
	label_t	jmpbuf;
	int	pl = splvm();

	DDISPLAY(18);
	if (setjmp(&jmpbuf)) {
		current_thread()->recover = NULL;
		DDISPLAY(0);
		splx(pl);
		return(EFAULT);
	}
	current_thread()->recover = (vm_offset_t)&jmpbuf;
	bcopy(from,to,length);
	current_thread()->recover = NULL;
	DDISPLAY(0);
	splx(pl);
	return(0);
}

copyout(from,to,length)
	register caddr_t from,to;
	register unsigned length;

{
	label_t	jmpbuf;
	int	pl = splvm();

	DDISPLAY(19);
	if (setjmp(&jmpbuf)) {
		current_thread()->recover = NULL;
		DDISPLAY(0);
		splx(pl);
		return(EFAULT);
	}
	current_thread()->recover = (vm_offset_t)&jmpbuf;
	bcopy(from,to,length);
	current_thread()->recover = NULL;
	DDISPLAY(0);
	splx(pl);
	return(0);
}


/*
 * copy_to_phys and copy_from_phys work by a hack--they use the
 * rosetta virtual-equal-to-physical mode.  Thus, things wouldn't
 * work quite right if the stuff to be copied was in segment-register
 * zero, so if it is we save the value of segment register 1, zot the
 * value of sr0 over it, fix up the virtual address, turn on v_eq_p
 * mode, do the bcopy, zot the register back and are happy.
 *
 */


/*
 *	Copy virtual memory to physical memory.  The virtual memory
 *	must be resident (e.g. the buffer pool).
 */

copy_to_phys(src_addr_v, dst_addr_p, count)
	register vm_offset_t src_addr_v, dst_addr_p;
	register unsigned count;
{
	register old_sr, seg_hack = 0, pl = splhigh();
	int old_real;

	DDISPLAY(20);
	if ((src_addr_v >> 28) == 0) {
		old_sr = get_rosetta_seg(HACK_SEG);
		seg_hack = 1;
		set_rosetta_seg(HACK_SEG,get_rosetta_seg(0));
		tlb_invalidate_seg(HACK_SEG);
		src_addr_v |= HACK_SEG<<28;	/* Make src addr in vm */
	}

	old_real = virt_eq_real(TRUE);
	bcopy(src_addr_v,dst_addr_p,count);
	virt_eq_real(old_real);

	if (seg_hack) {
	    	set_rosetta_seg(HACK_SEG,old_sr);
		tlb_invalidate_seg(HACK_SEG);
	}

	DDISPLAY(0);
	splx(pl);	/* Everything's back to normal. */
}



/*
 *	Copy physical memory to virtual memory.  The virtual memory
 *	must be resident (e.g. the buffer pool).
 */

copy_from_phys(src_addr_p, dst_addr_v, count)
	register vm_offset_t src_addr_p, dst_addr_v;
	register unsigned count;
{
	register old_sr, seg_hack = 0, pl = splhigh();
	int old_real;
 
	DDISPLAY(21);
	if ((dst_addr_v >> 28) == 0) {
		old_sr = get_rosetta_seg(HACK_SEG);
		seg_hack = 1;
		set_rosetta_seg(HACK_SEG,get_rosetta_seg(0));
		tlb_invalidate_seg(HACK_SEG);
		dst_addr_v = HACK_SEG<<28 | (segoffset(dst_addr_v));
	}

	old_real = virt_eq_real(TRUE);
	bcopy(src_addr_p,dst_addr_v,count);
	virt_eq_real(old_real);

	if (seg_hack) {
	    	set_rosetta_seg(HACK_SEG,old_sr);
		tlb_invalidate_seg(HACK_SEG);
	}
	DDISPLAY(0);
	splx(pl);	/* Everything's back to normal. */
}

pagemove(from, to, size)
	register vm_offset_t from, to;
	register int size;
{
	int pl = splvm();
	register caddr_t ra;
	DDISPLAY(21);
	PMAP_DEBUGF(DEBUG_PMAP_ENTER,("pagemove(from=0x%x,to=0x%x,size0x%x)\n",
				from, to, size));	
#if	ROMP_DEBUG
	if (size % rose_page_size)
		panic("pagemove");
	if ((((unsigned)from >> 28) != SYS_SEG) 
		|| (((unsigned)to >> 28) != SYS_SEG))
		panic("pagemove seg reg");
#endif	ROMP_DEBUG

	while ((size -= rose_page_size) >= 0) {
		if ((ra = (caddr_t)trans_virt_addr(from)) == (caddr_t) - 1)
			panic("pagemove vtop");
		mapout((vm_offset_t)a_to_rosetta_page((unsigned int)ra),1);
		mapin((int)(ra)>>rose_page_shift, 
			((segoffset(to))>>rose_page_shift), 0, 1,
			RTA_SID_SYSTEM); /* XXX - Protection */
		from += rose_page_size;
		to += rose_page_size;
	}
  tlb_invalidate_seg(atoseg(pagemove));	/*This is easier than doing it right */
					/*  Don't ask */
  DDISPLAY(0);
  splx(pl);
}

/*
 *
 * Glorified tva to make the drivers happy.
 *
 */
caddr_t
real_buf_addr(bp, bufaddr)
	register struct buf *bp;
	register vm_offset_t bufaddr;
{
	caddr_t		phys;
	pmap_t		map;

	if ((bp->b_flags & B_PHYS) && ((atoseg(bufaddr)) != SYS_SEG))
		map = vm_map_pmap(bp->b_proc->task->map);
	else
		map = kernel_pmap;

	if (((phys = (caddr_t)pmap_extract(map,bufaddr)) != (caddr_t)0))
		return(phys);
	else
		return((caddr_t)((int)bufaddr & (rose_page_size - 1)));
}

kernacc(va,cnt,rw)
	vm_offset_t	va;
	vm_offset_t cnt;
	int rw;
{
	vm_offset_t	v;
	vm_offset_t	end;

#ifdef	lint
	rw++;
#endif	lint

	for (v = trunc_page(va), end = round_page(va+cnt-1);
	     v < end;
	     v += rose_page_size) {
		if (trans_virt_addr(v) == (unsigned)(-1) )
		    	return(FALSE);
	}
	return(TRUE);
}

/*
 *	Routine:	pmap_change_wiring
 *	Function:	Change the wiring attribute for a map/virtual-address
 *			pair.
 *	Note:		This routine is a dummy for Rosetta.
 */
void pmap_change_wiring(map, v, wired)
	pmap_t		map;
	vm_offset_t	v;
	boolean_t	wired;
{
#ifdef	lint
	map++; v++; wired++;
#endif	lint
}

/*
 *	Routine:	pmap_clear_modify
 *	Function:	Clear the hardware modified ("dirty") bit for one
 *			machine independant page starting at the given
 *			physical address.  phys must be aligned on a machine
 *			independant page boundary.
 */

void pmap_clear_modify(phys)
	vm_offset_t	phys;
{
	vm_offset_t	addr = phys;

	DDISPLAY(25);
#if	ROMP_DEBUG && ROMP_RDB
	if (phys == magic_paddr)
		Debugger("pmap_clear_modify: magic paddr hit");
#endif	ROMP_DEBUG && ROMP_RDB
	for (;addr < PAGE_SIZE + phys;addr += rose_page_size)
		set_mod_bit(addr>>0xb,0);
	DDISPLAY(0);
}

/*
 *	Routine: pmap_is_referenced
 *	Function: Returns TRUE iff the given physical page has
 *		been referenced since the last pmap_clear_reference.
 */

boolean_t pmap_is_referenced(phys)
	register vm_offset_t	phys;
{
	register vm_offset_t	addr = phys;

	for (;addr < PAGE_SIZE + phys; addr += rose_page_size) {
		if (get_ref_bit(addr>>0xb))
			return(TRUE);
	}
	return(FALSE);
}

/*
 *	Routine: pmap_is_modified
 *	Function: Returns TRUE iff the given physical page has
 *		been referenced since the last pmap_clear_reference.
 */

boolean_t pmap_is_modified(phys)
	register vm_offset_t	phys;
{
	vm_offset_t	addr = phys;

	for (;addr < PAGE_SIZE + phys; addr += rose_page_size) {
		if (get_mod_bit(addr>>0xb))
			return(TRUE);
	}
	return(FALSE);
}

/*
 *	Routine:	pmap_clear_reference
 *	Function:
 *		Clear the hardware referenced bit in the given machine
 *		independant physical page.  If referenced bits are not
 *		available in a particular hardware architecture, do a
 *		pmap_remove_all instead.
 */
void pmap_clear_reference(phys)
	vm_offset_t	phys;
{
	vm_offset_t	addr = phys;

	DDISPLAY(27);
#if	1
	for (;addr < PAGE_SIZE + phys; addr += rose_page_size) 
		set_ref_bit(addr>>0xb,0);
#else	0
	pmap_remove_all(phys);
#endif	0
	DDISPLAY(0);
}


/*
 *	Routine:	virt_eq_real [internal use only]
 *	Function:
 *		Set segment 0 virtual_equal_real mode if turnon,
 *		clear otherwise; in any case, return the old 
 *		value.
 */

virt_eq_real(turnon)
	register int turnon;
{
	register int tcr = ior (ROSEBASE + ROSE_TCR);

	if (turnon)
		iow(ROSEBASE+ROSE_TCR,tcr | TCR_V);
	else
		iow(ROSEBASE+ROSE_TCR,tcr & (~TCR_V));

	return((tcr & TCR_V) ? 1 : 0);
}

/*
 *	pmap_page_protect:
 *
 *	Lower the permission for all mappings to a given page.
 */
void	pmap_page_protect(phys, prot)
	vm_offset_t	phys;
	vm_prot_t	prot;
{
	switch (prot) {
		case VM_PROT_READ:
		case VM_PROT_READ|VM_PROT_EXECUTE:
			pmap_copy_on_write(phys);
			break;
		case VM_PROT_ALL:
			break;
		default:
			pmap_remove_all(phys);
			break;
	}
}

#include <sys/vmmac.h>

vm_offset_t	pmap_phys_address(frame)
	int		frame;
{
	return(ptoa(frame));
}

#if	ROMP_CACHE
/***DEBUG***
 *
 *	This function is called when pmap_extract fails on a wired
 * 	page.
 */

pmap_phys_find_address(map,virt)
pmap_t map;
vm_offset_t virt;
{
    queue_head_t *list;
    struct pmap_list_entry *entry, *next_entry;
    int			old_spl_level = splvm(); 
    int entry_in_plist = 0;
    int entry_in_clist = 0;
    vm_offset_t vpage = a_to_rosetta_page(segoffset(virt));
    
    if (map == PMAP_NULL) {
	splx(old_spl_level);
	return;
    }

    list = &map->ppage_list;
    entry = (struct pmap_list_entry *) queue_first(list);

    printf("pmap_phys_find_address");
    
    while (!queue_end(list,(queue_entry_t) entry)) {
	    next_entry = (pmap_list_entry_t) queue_next(&entry->pmap_pages);
	    if (entry->vaddr == virt) {
		entry_in_plist = TRUE; break;
	    }
	    entry = next_entry;
    }
    entry_in_clist = (cache_lookup_entry(map->sidtab[atoseg(virt)],vpage));
					
    printf("entry_in_plist %x  entry_in_clist %x\n",
    		entry_in_plist, entry_in_clist);

    Debugger("pmap_phys_find_address");
    
    splx(old_spl_level);

}
#endif	ROMP_CACHE	

/*
 *	Routine:	pmap_make_kernel_text_readable
 *	Function:
 *		Allow users that have access to the kernel
 *		text segment to execute code from that segment.
 *		[There is no added privilege gained -- this is just
 *		a cheap shared library mechanism.]
 *	Note:
 *		This routine should be placed near the end of the
 *		code file, to avoid lint complaints about re-using
 *		the external symbol "start".  Obviously lint should
 *		be fixed, but that is not a kernel hacker's job.
 */
void	pmap_make_kernel_text_readable()
{
	extern int start, edata;

	/*
	 * protect all the kernel text as world readable.
	 */
	pmap_pr_phys(kernel_pmap,
		trunc_page(&start),
		round_page(&edata),
		RTA_KEY_URKW << RTA_KEY_SHIFT);
}

#ifdef	notdef
unsigned char do_loop_check = 0;
unsigned char do_chain_check = 0;
int hats_set, ipts_set, max_chain;
loop_check()		/* This debugging routine checks for loops in the 
			    HAT/IPT. */

{
 static long chain_check[(physmem + 31)/32];
 register struct hatipt_entry *ipte;
 int hat_index,ipt_index, len_this_chain;

 if (!do_loop_check) goto check_chain;

/* Check for circular ipt chains. */
 max_chain = 0;
 for (hat_index = 0;hat_index < physmem;hat_index++) {
	ipte = &RTA_HATIPT[hat_index];
	if (ipte->hat_ptr & 0x8000)
		continue;
	for (ipt_index = 0;ipt_index < physmem;ipt_index++)
		clrbit(chain_check,ipt_index);
	ipt_index = ipte->hat_ptr & 0x07ff;
	len_this_chain = 0;
	do {
		len_this_chain++;
		if (isset(chain_check,ipt_index)) {
			printf("chain_check = 0x%x, hat_index = 0x%x.\n",
				chain_check,ipt_index);
			panic("circular chain");
		}
		setbit(chain_check,ipt_index);
		ipte = &RTA_HATIPT[ipt_index];
		ipt_index = ipte->ipt_ptr & 0x07ff;
	} while (!(ipte->ipt_ptr & 0x8000));
 	max_chain = (max_chain > len_this_chain) ? max_chain : len_this_chain;
 }

check_chain:
 if (!do_chain_check)

	return;
/* Check for duplicate ipt pointed to's. */

 hats_set = ipts_set = 0;

 for (ipt_index = 0;ipt_index < physmem;ipt_index++)
	clrbit(chain_check,ipt_index);
 for (hat_index = 0;hat_index < physmem;hat_index++) {
	ipte = &RTA_HATIPT[hat_index];
	if (!(ipte->hat_ptr & 0x8000)) {
		hats_set++;
		if (isset(chain_check,ipte->hat_ptr & 0x07ff)) {
			printf("chain_check = 0x%x, hat_index = 0x%x., hat = 0x%x\n",
				chain_check,hat_index,ipte->hat_ptr);
			panic("dup hat ptrs");
		}
		setbit(chain_check,ipte->hat_ptr & 0x07ff);
	}
 }
 for (ipt_index = 0;ipt_index < physmem;ipt_index++) {
	ipte = &RTA_HATIPT[ipt_index];
	if (!(ipte->ipt_ptr & 0x8000)) {
		ipts_set++;
		if (isset(chain_check,ipte->ipt_ptr & 0x07ff)) {
			printf("chain_check = 0x%x, ipt_index = 0x%x, ipt = 0x%x\n",
				chain_check,ipt_index,ipte->ipt_ptr);
			panic("dup hat/ipt ptrs");
		}
		setbit(chain_check,ipte->ipt_ptr & 0x07ff);
	}
 }
}
#endif	notdef	

#ifdef	now_defunct
void pmap_range_remove(map, s,e)
	pmap_t		map;
	vm_offset_t	s,e;
{
	vm_offset_t	pa;
	int		seg = atoseg(s), old_sr;
	int		sid;
	int		pl;
	register int	count;

	panic("pmap_range_remove");

	if (atoseg(s) != atoseg(e-1))  
		panic("pmap_range_remove (too big)");

	DDISPLAY(3);
	PMAP_DEBUGF(DEBUG_PMAP_REMOVE,
		("pmap_remove_rng: map= 0x%x, s = 0x%x, e=0x%x.\n",
				map,s,e));

	pl = splvm();

	lock_read(&pmap_lock);
	simple_lock(&map->lock);

	if (seg == SYS_SEG)
		sid = RTA_SID_SYSTEM;
	else
		sid = map->sidtab[seg];

	if (sid == 0) {
		PMAP_DEBUGF(DEBUG_PMAP_REMOVE,
			("pmap_remove: segment %x not mapped in, pmap=0x%x.\n",
			seg,map))
		simple_unlock(&map->lock);
		lock_read_done(&pmap_lock);
		splx(pl);
		return;
	}
	if (seg != SYS_SEG) { pmap_cache_remove_range(map,s,e); }

	/*
	 *	map the correct segment into HACK_SEG.
	 */
	
	old_sr = get_rosetta_seg(HACK_SEG);
	set_rosetta_seg(HACK_SEG,sid_to_seg(sid));
	tlb_invalidate_seg(HACK_SEG);

	count = 0;
	while (s < e) {
		if ((pa = 
		    trans_virt_addr((HACK_SEG << 28) | segoffset(s))) != -1) {
			mapout(a_to_rosetta_page(pa),ppages_per_vmpage);
			count++;
		};
		s += PAGE_SIZE;
	}

	set_rosetta_seg(HACK_SEG,old_sr);
	tlb_invalidate_seg(HACK_SEG);
	tlb_invalidate_seg(seg);

	simple_unlock(&map->lock);
	lock_read_done(&pmap_lock);
	DDISPLAY(0);
	splx(pl);
}

/*
 * Scan the ipt from physical page min_ppage to physical page max_ppage
 * for virtual addresses in (12-bit) segment sid from offset s to e
 * (s and e must be segment relative) and remove these mappings.
 */
#ifdef	isset
#undef	isset
#endif	isset
#define	isset(a,i)	(*(((char *)(a)) + ((i)>>3 )) & (1<<((i)&0x7)))
void pmap_phys_range_remove(min_ppage,max_ppage,sid,s,e,map)
	int			min_ppage;
	register int		max_ppage, sid;
	register vm_offset_t	s, e;
	pmap_t			map;
{
	register int	pagenum, key_addrtag;
	register vm_offset_t	vpage;
	int		count = 0;

	panic("pmap_phys_range_remove");
	
	for (pagenum = min_ppage; pagenum < max_ppage;
		pagenum += ppages_per_vmpage) {
		key_addrtag = RTA_HATIPT[pagenum].key_addrtag;
		vpage = key_addrtag & RTA_VPAGE_MASK;
		if ((isset(is_in,pagenum))
		  && (((key_addrtag & RTA_SID_MASK) >> RTA_VPAGE_BITS) == sid)
		  && (vpage >= s)
		  && (vpage < e)) {
			mapout(pagenum,ppages_per_vmpage);
			count++ ; 
		  }
#if	PMAP_STATISTICS 		  
		/*
		 * if we have found all of the mappings in the
		 * hash table, we can stop scanning.
		 */
		 
		if (initial_resident - count == 0) break;
#endif	PMAP_STATISTICS		
        }

}



/* 
 * Remove a range of memory from virt. addr. s to e from the given pmap.
 * s and e must be in the same segment.  Operates by scanning the ipt
 * (using pmap_phys_range_remove).  Knows about memory holes and avoids scanning
 * them.
 */
void pmap_phys_remove(map, start, end)
	register pmap_t		map;
	register vm_offset_t	start, end;
{
	int		seg = atoseg(start);
	register int	sid;
	int		pl;
	register vm_offset_t	s, e;

	DDISPLAY(4);

	panic("pmap_phys_remove");

	if (atoseg(start) != atoseg(end-1))  
		panic("pmap_phys_remove (too big)");

	PMAP_DEBUGF(DEBUG_PMAP_REMOVE,
		("pmap_remove_physical: map= 0x%x, s = 0x%x, e=0x%x.\n",
			map,start,end));

	pl = splvm();

	lock_read(&pmap_lock);
	simple_lock(&map->lock); 


	if (seg == SYS_SEG)
		sid = RTA_SID_SYSTEM;
	else
		sid = map->sidtab[seg];

	if (sid == 0) {
	    	PMAP_DEBUGF(DEBUG_PMAP_REMOVE,
 	 		("pmap_remove: segment %x not mapped in, pmap=0x%x.\n",
			seg,map));

		simple_unlock(&map->lock);
		lock_read_done(&pmap_lock);
		splx(pl);
		return;
	}

	/*
	 * Why aren't the following shifts using rose_page_size???
	 */

	if (atoseg(end) == seg) 
		e = segoffset(end) >> 0xb;
	else 
		e = (0x10000000) >> 0xb /*XXX*/;
	s = (segoffset(start) >> 0xb);

	if (seg != SYS_SEG) { pmap_cache_remove_range(map,start,end); }
	
/* When the system is using 4K pages the lowest bit of key_addrtag in the
   ipte is unused, and the next however many bits are the page number.
   So, we shift s and e by the 2K size (which is 0xb bits) and if we are
   using 4K pages then add one to e so as to correctly map out the last
   page in the case that the low bit is set in the ipte by some magic.
 */
	if (rose_page_size == 0x1000) {
		s &= ~0x1;	/* Clear unused bit */
		e |= 0x1;	/* Set unused bit */
	}


	/*
 	 *	Optimization:  Do not check the kernel text
	 *	if we are removing from a non-system segment (we probably
	 *	do not need to anyway, but just for safety's sake...)
	 */
	pmap_phys_range_remove(((sid == RTA_SID_SYSTEM) ? 0 : min_user_pa),
				a_to_rosetta_page(holestart), 
				sid, s, e, map);

	if (ptoa(physmem) > holestart)
		pmap_phys_range_remove(a_to_rosetta_page(holeEnd),
			vm_to_rosetta_page(physmem) + 
			a_to_rosetta_page(holeEnd - holestart),
			sid, s, e, map);

	tlb_invalidate_seg(seg);

	simple_unlock(&map->lock);
	lock_read_done(&pmap_lock);

	DDISPLAY(0);
	splx(pl);
}


void pmap_pr_range(map, s, e, internal_prot)
	pmap_t		map;
	vm_offset_t	s, e;
	int 		internal_prot;
{
	int		pl, pg, old_sr, seg = atoseg(s);
	struct  hatipt_entry	*ipte;

	DDISPLAY(7);

	if (atoseg(s) != atoseg(e-1))
		panic("pmap_pr_range (too big)");

	if (seg != (atoseg(e - 1)))
		panic("pmap_protect");

	pl = splvm();
	
	lock_read(&pmap_lock);
	simple_lock(&map->lock);

	old_sr = get_rosetta_seg(HACK_SEG);
	set_rosetta_seg(HACK_SEG,sid_to_seg(map->sidtab[seg]));
	tlb_invalidate_seg(HACK_SEG);

	if (seg != SYS_SEG) { pmap_cache_remove_range(map,s,e); } 
	
	while (s < e) {
		if ((pg = 
		   trans_virt_addr((HACK_SEG << 28) | segoffset(s))) != -1) {
			ipte = &RTA_HATIPT[pg>>rose_page_shift];
			ipte->key_addrtag &= ~(3 << RTA_KEY_SHIFT);
			ipte->key_addrtag |= internal_prot;
		}
		s += rose_page_size;
	}

	set_rosetta_seg(HACK_SEG,old_sr);
	tlb_invalidate_seg(HACK_SEG);
	tlb_invalidate_seg(seg);

	simple_unlock(&map->lock);
	lock_read_done(&pmap_lock);

	DDISPLAY(0);
	splx(pl);
}
#endif	now_defunct
