/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 *
 ****************************************************************
 *	File:	romp_init.c
 *	Author:	William J. Bolosky
 *	Copyright (c) 1986, 1987, William J. Bolosky
 *	
 *	Unlike vax_init on the vax, this DOES NOT run in physical mode.  Thus,
 *	the hat/ipt is already setup and active, so we can't randomly munge it
 *	anymore.
 */

/*
 ***********************************************************************
 * HISTORY
 * $Log:	romp_init.c,v $
 * Revision 2.3  88/12/19  02:38:00  mwyoung
 * 	Added missing call to vm_set_page_size.
 * 	[88/12/12            mwyoung]
 * 
 * Revision 2.2  88/11/23  16:27:45  rpd
 * 	Changed includes to the new style.
 * 	Picked up 135 support from Acis.
 * 	[88/11/04  17:19:43  rpd]
 * 
 * 12-Nov-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Minor modification to conditionally use loop mode on the new
 *	APC.  This allows us to turn off loop mode on back-level hardware. 
 *
 * 25-Aug-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added update from 4.3 ACIS official release to initially set the
 *	FPA to be unavailable to the user.
 *
 *  3-Jul-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	ROMP_APC: do init_ioims if on ROMPC.
 *
 * 10-Mar-87  Avadis Tevanian (avie) at Carnegie-Mellon University
 *	Change initial banner to not print that we are using VM, this is
 *	of course the case.
 *
 * 29-Jan-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Moved in INIT_INTR() so that the new kls controller works when
 *	cnatch() is called, also added include of "../ca/io.h" for its
 *	definition.
 *
 * 26-Jan-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Removed external declaration of loadpt to make hc happy.
 *
 * 26-Aug-86  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Added code to initialize channel control register.
 *
 * 24-Jul-86  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Added code to initialize pmap_threshold.
 ***********************************************************************
 */
#include <romp_apc.h>
#include <romp_135.h>

#include <sys/types.h>

#if	ROMP_APC
#include <ca/cpu.h>
#include <ca/ioim.h>
#endif	ROMP_APC
#include <ca/machparam.h>
#include <ca/pmap.h>
#include <ca/rosetta.h>
#include <ca/scr.h>
#include <ca/io.h>
#include <sys/msgbuf.h>
#include <vm/vm_param.h>
#include <vm/vm_prot.h>

vm_size_t	mem_size;
vm_offset_t	first_addr;
vm_offset_t	last_addr;

vm_offset_t	avail_start, avail_end;
vm_offset_t	virtual_avail, virtual_end;

romp_init()
{
	extern	char	end;
	extern	int	physmem;	/* Set by lohatipt */
	extern	caddr_t	is_in;
	extern	int pmap_threshold;

	iow(ROSEBASE+ROSE_TCR,ior(ROSEBASE+ROSE_TCR) | TCR_I);
		/* Debugging - TLIPT */
	INIT_INTR();	/* cnatch needs interrupts...so we initialize now */
#if	ROMP_APC
#if	ROMP_135
	if (cpu_model == CPU_ROMPC || cpu_model == CPU_MOD135)
#endif	ROMP_135
	if (cpu_model == CPU_ROMPC)
		init_ioims(cpu_model);
#endif	ROMP_APC
	cnatch(0);
#if	ROMP_APC
	check_mem_config();
#endif	ROMP_APC
	/* Initialize CCR */
	* (char *) CCR = CCR_DEFAULT;

	vm_set_page_size();

	mem_size = physmem * rose_page_size;
	first_addr  = (vm_offset_t)(&end) & 0x0fffffff; /* Make it physical */
	first_addr  = round_page(first_addr);
	last_addr   = (((int)RTA_HATIPT) & 0x0fffffff) - 
			(((sizeof (struct msgbuf) + 2047) >> 11) << 11) - 
			((((mem_size / (rose_page_size * 8)) + 2047) >> 11)
				<< 11);
			/* Assumption1: hat/ipt is at top of memory */
			/* Assumption2: machdep will allocate msgbuf here */
			/* Leave space for the is_in table here, too. */
	last_addr   = trunc_page(last_addr);
	/*Plunk the is_in table right above available memory. */
	is_in = (caddr_t)((int)last_addr + (int)loadpt);  /*Make it virtual.*/

	pmap_threshold = (last_addr - first_addr) >> (rose_page_shift + 2);
	  /* There should be some better way to calculate the threshold... */

	avail_start = first_addr;
	avail_end   = last_addr;
	printf("RT/PC Mach, avail_start = 0x%x, avail_end = 0x%x.\n",
		avail_start, avail_end);
	pmap_bootstrap(loadpt, &avail_start, 
			   &avail_end, &virtual_avail,
			   &virtual_end);

}


#if	ROMP_APC

int ioim1_config;		/* IOIM 1 configuration */
int ioim2_config;		/* IOIM 2 configuration */
int ioim2;			/* non-zero if IOIM2 present */
label_t ioim_label;

#define MEM_2_CYCLE 0x80	/* On if 2 Cycle Memory */
/*
 * initialize both IOIM1 and IOIM2 (IOIM2 requires complete setup
 * IOIM1 has already been initialized by the ROS)
 */
init_ioims(cpu_model)
int cpu_model;
{
	extern int memconfig;	/* memory configuration register (locore) */
	int mem_cycle = (memconfig & MEM_2_CYCLE) != 0 ? 0 : IOIM_2_CYCLE;

	iow(IOIM1_BASE, (BUS1 << 8) + (IOIM1 >> 16));
	ioim1_config = ior(IOIM1 + IOIM_MCC);	/* get configuration */
	/* for now disable parity (chips don't work at 100ns) */
	iow(IOIM1+IOIM_GCR, ior(IOIM1+IOIM_GCR) | IOIM_NOPARITY | mem_cycle);
	/* Turn off user access to FPA, but leave IOCC controlled alone */
	iow(IOIM1+IOIM_SUB_PROB, SEG_0+SEG_4);

#if	ROMP_135
	if (cpu_model != CPU_MOD135)
#endif	ROMP_135
	{
		extern int (*_trap)();
		extern int ioim_trap();
		register int (*oldtrap)() = _trap;
		_trap = ioim_trap;
		if (setjmp(&ioim_label)) {
			_trap = oldtrap;
			return;
		}
		/* initialize IOIM2 for 68881 operation */
		iow(IOIM2_BASE, (BUS2 << 8) + (IOIM2 >> 16));
		ioim2_config = ior(IOIM2 + IOIM_MCC);	/* get configuration */
		iow(IOIM2+IOIM_MCC, IOIM2_MCC);		/* set TAG and ARB */
		iow(IOIM2+IOIM_GCR, IOIM_NOPARITY | IOIM_LOOPMODE);
		iow(IOIM2+IOIM_MISC_0, IOIM2_MISC_0);	/* XXX obsolete hardware */
		iow(IOIM2+IOIM_MISC_1, IOIM2_MISC_1);
		iow(IOIM2+IOIM_SUB_SEG_1, IOIM2_VALUE1);
		iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_DISABLE);	/* */
/*		iow(IOIM2+IOIM_SUB_PROB, IOIM2_PROB_ENABLE);	/* for testing */
		iow(IOIM2+IOIM_SUB_SEG_3, IOIM2_VALUE3);
		iow(IOIM2+IOIM_SUB_SEG_4, IOIM2_VALUE4);
		iow(IOIM2+IOIM_SUB_SEG_5, IOIM2_VALUE5);
		_trap = oldtrap;
		ioim2 = IOIM2;
	}
}

ioim_trap()
{
	longjmp(&ioim_label);
}

check_mem_config()
{
 extern int memconfig;	/* memory configuration register (locore) */
 int mem_cycle = (memconfig & MEM_2_CYCLE);
 int x;

 if (cpu_model == CPU_ROMPC)
	{
	if (mem_cycle) {
		printf("Warning: slow memory cards are not supported on APC\n");
		printf("	machine checks are likely\n");
	}
	/* following tests if APC has an 8meg card in first slot - which 
	 * means that either the cards should be swapped so that the 8meg
	 * card is in the second slot, or that the two 8meg cards are present
	 * which is wasteful as the first could be replaced by a 4meg card.
	 */
	if ((memconfig & 0xf07) == 0x804)
		printf("Warning: 4MB of 8MB memory card unusable - %s\n",
			((memconfig & 0x070) == 0x040) ? 
			"4MB card recommended" : "exchange card order");
	/*
	 * test to see if the reply register exists
	 * if it does then the value will change when the
	 * base register is read.
	 */
	if ((ioim1_config & 0x0f00) != 0x0900)
		printf("IOIM1 EC level different than expected\n");
	x = ior(IOIM1+IOIM_REPLY);
	(void) ior(IOIM1+IOIM_BASE);
	if (ior(IOIM1+IOIM_REPLY) == x)
		printf("Warning: back-level IOIM - no reply register\n");
	}
}
#endif	ROMP_APC
