/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 ***********************************************************************
 * HISTORY $Log:	kls.c,v $
 * Revision 2.3  88/10/06  18:10:14  sanzi
 * 	Added include of "device_base.h".  Merge with ACIS to minimize differences.
 * 	Fix includes.
 * 
 ***********************************************************************
 */

/*
 * 5799-CGZ (C) COPYRIGHT IBM CORPORATION 1987
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/* $Header: kls.c,v 2.3 88/10/06 18:10:14 sanzi Exp $ */
/* $ACIS:kls.c 9.0$ */
/* $Source: /afs/cs.cmu.edu/source_mach/rcs/kernel/cacons/kls.c,v $ */

#ifndef lint
static char *rcsid = "$Header: kls.c,v 2.3 88/10/06 18:10:14 sanzi Exp $";
#endif

/* 
 * 8051 scheduler. Controls all the access to the planar 8051 to ensure the
 * keyboard, mouse, and speaker drivers do not step on each other's toes.
 *
 */
#include "romp_debug.h"
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/tty.h>
#include <ca/io.h>
#include <sys/time.h>
#include <sys/kernel.h>
#include <cacons/consdefs.h>
#include <cacons/kls.h>
#include "ms.h"

int klsdone();
caddr_t		klsminit();
volatile struct klsq 	*klsalloc();
struct clist	kbd_data,ms_block,ms_uart; /* character "queues" */
volatile struct klsq	*klshead,*klstail;	    /* pointers to the xmit queues */
volatile struct klsq	*klsfreel;		    /* free list */
volatile struct klsq	klspool[KLSPOOLSZ];
int		klsdebug = 0;
int		klsinreset = 0;

static char *klserrormsgs[] = {
	"Keyboard transmit timed out",				/* E0 */
	"Keyboard receive timed out",				/* E1 */
	"Keyboard acknowledge was not received",		/* E2 */
	"Unexpected keyboard acknowledge was received",		/* E3 */
	"Hard error was detected on keyboard frame receive",	/* E4 */
	"Hard error was detected on keyboard frame transmit",	/* E5 */
	"Keyboard clock pin not plus",				/* E6 */
	"Keyboard clock pin not minus",				/* E7 */
	"Uart interrupt received without tran/rec identifier",	/* E8 */
	"Uart transmit timed out",				/* E9 */
	"Uart acknowledge timed out",				/* EA */
};

#ifdef DEBOUNCE
int		klslastscan = 0x100;
struct		timeval	klslasttime = { 0, 0};
#endif DEBOUNCE

/* These are things that happen only once during boot time */
klsinit()
{
	klshead=NULL;
	klsfreel = (volatile struct klsq *) klsminit(klspool,sizeof(struct klsq),KLSPOOLSZ);
	kbdinit();
#if NMS > 0
	msinit();
#endif NMS
	spkinit();

	/* put a dummy pointer on the queue to prevent kls from starting */
	/* if individual reset routines queue a command */
	klsinreset = 1;

	/* turn off other devices */
	(void) kls_raw_cmd(EXTCMD,CLRSPKMOD);
	(void) kls_raw_cmd(UARTCNT,MSDIS);
	(void) kls_raw_cmd(KYBDCMD,NKSCAN);

	/* The following routines have full control over the adapter 
	 * as long as they don't enable unsolicited input. They may
	 * place a command on the queue to do so. These routines are
	 * also responsible for any special cleanup caused by the
	 * rejection of the commands commands before the reset.
	 */

	kbdreset();	/* do keyboard first */
	kls_pflush(4);
	spkreset();
	kls_pflush(4);	/* clean up any strays */
#if NMS > 0
	msreset();
	kls_pflush(4);	/* clean up any strays */
#endif NMS
	
	beep();			       /* beep AFTER speaker has been inited! */

	/* start commands that have been queued */
	klsinreset = 0;
	if (klshead != NULL)
		klsstart();
	out(KLS_CNTIW, 0x09);	       
	
}

/* kls_status, kls_pflush, and kls_raw_cmd may only be called by xxxreset routines! */

/*
 *	This routine waits for the status of a klscmd to be returned before continuing
 *	Legal statuses are 1) an info byte with is not an informational interupt (iid =
 *	KLS_INFO, c & KLSSRI != 0). 2) a request return bye (iid = KLS_REQ). All
 *	other statuses are ignored. If a timeout occurs, kls_status will return a KLS_SOFT_ERROR
 */
kls_status()
{
	register volatile int iid,c;
	register timeout = KLSMAXTIME;

	do {
		/* wait for something on in the input buffer */
		while ( (((iid=((volatile int) in(KLS_CNTIR))) & KLS_IBF) == 0) && (timeout-- > 0) )
			/* void */ DELAY(1);
		if (timeout <= 0)
			return(KLS_SOFT_ERROR);
		/* read it in */
		c=((volatile int) in(KLS_READ));
		iid &= IID_MASK;
	/* stop reading if it matches the conditions specified above */
	} while(!( ((iid == KLS_INFO)&&((c & KLSSRI) == 0)) 
			|| (iid == KLS_REQ) ));
	return ((iid << 8) + c);
}


kls_raw_read(iid)
	register volatile int *iid;
{
	register volatile int c;
	register timeout = KLSMAXTIME;

	/* wait for something on in the input buffer */
	while ( (((*iid=(volatile int) in(KLS_CNTIR)) & KLS_IBF) == 0) && (timeout-- > 0) )
		DELAY(1);
	if (timeout <= 0) {
		*iid = KLS_SOFT_ERROR >> 8;
		return(KLS_SOFT_ERROR & 0xff);
	}
	/* read it in */
	c= (volatile int) in(KLS_READ);
	*iid &= IID_MASK;
	return(c);
}

/*	Kls_pflush trys to clear the input buffer. by doing successive reads. */
kls_pflush(count) 
	register int count;
{
	register int tmp;
	for (; count >= 0 ; count--) {
		tmp = in(KLS_READ);
		delay(1);
	}
}

/*	Kls_raw_cmd sends a command to the adapter and waits for a return status */
kls_raw_cmd(dest,cmd)
	register int dest,cmd;
{
	register int x;
	while((in(KLS_CNTIR) & KLS_OBF) == 0)
		/* void */ DELAY(1);
	outw(KLS_WRITE,CMD(dest, cmd));
	x = kls_status();
	return(x);
}
	


/* following allows us to get rid of bell and keyboard clicks easily */
#ifndef NOBELL
#define NOBELL 0
#endif

int nobell = NOBELL;

/* WARNING: THIS MAY BE CALLED AFTER A USER MODE MACHINE CHECK! NO SLEEPING */
klsreset()
{
	register int s=KLSSPL();	/* don't let anyone interupt us! */
	register int x;
	register int i;
	int	tmp;

	if (klshead != NULL) { /* commands have been queued */
		/* clear queue & wake everyone up */
		while (klshead != NULL) {
			klshead->qp_ret=KLS_SOFT_ERROR;
			if (klshead->qp_callback)
				(* klshead->qp_callback) (klshead);
			klshead = klshead->qp_next;
		}
		/* flush out pipeline */
		kls_pflush(10);
	}

	x = in(CRRB);
	x &= KLS_ADAPTOR_RESET;
	DELAY(1);
	out(CRRB, x);

	delay(200);
	out(KLS_CNTIW, KLS_CONFIG);    /* configure 8255 */
	delay(200);		       /* delay 200 ms */

	out(KLS_CNTIW, 0x09);	       /* enable interrupt request */
	delay(200);

	x = in(CRRB);
	x |= KLS_ADAPTOR_RELEASE;
	DELAY(1);
	out(CRRB, x);

	delay(200);

	kls_pflush(40);	       /* clear pipeline */
	if (in(KLS_CNTIR) & KLS_IBF) {
		printf("adapter pipeline won't empty\n");
		printf("probable backlevel planar, trying old init\n");
		out(KLS_CNTIW, KLS_CONFIG);    /* configure 8255 */
		delay(200);		       /* delay 200 ms */

		out(KLS_CNTIW, 0x09);	       /* enable interrupt request */
		delay(200);
		kls_pflush(40);
		if (in(KLS_CNTIR) & KLS_IBF) {
			printf("adapter still won't empty!!\n");
			FAIL(1);
		}
	}
	x = kls_raw_cmd(EXTCMD,RD1C);
	if (x != ((KLS_REQ << 8) + 0xAE)) {
		printf("adapter self test failed: IID/data = %x\n", x);
		FAIL(3);
	}
#ifdef notdef
	kls_pflush(10);		  /* do 10 reads to empty pipeline */
	if (in(KLS_CNTIR) & KLS_IBF) {
		printf("adapter pipeline won't empty\n");
		FAIL(3);
	}
#endif
	x = kls_raw_cmd(EXTCMD,ENKYBD);
	if (x != ((KLS_INFO << 8) + 0x00)) {
		printf("keyboard failed to clear: IID/data = %x\n", x);
		FAIL(4);
	}
#ifdef notdef
	kls_pflush(10);		    /* do 10 reads to empty pipeline */
	if (in(KLS_CNTIR) & KLS_IBF) {
		printf("adapter pipeline won't empty\n");
		FAIL(5);
	}
#endif
	x = kls_raw_cmd(KYBDCMD,KRESET);
	if (x != ((KLS_INFO << 8) + 0x00)) {
		printf("keyboard failed to reset: IID/data = %x\n", x);
		FAIL(6);
	}

	delay(500);

	for (i = 0; i < 4; ++i) {
		x = kls_raw_read(&tmp); /* pick up the BAT etc. */
		if (x == 0xAA)
			break;
	}

	if (x != 0xAA)
		printf("BAT was not found\n");
#ifdef notdef
	kls_pflush(10);
	x = kls_raw_cmd(EXTCMD,KEYCLICK);
	kls_pflush(10);
#endif notdef
	/* put a dummy pointer on the queue to prevent kls from starting */
	/* if individual reset routines queue a command */
	klsinreset = 1;

	/* turn off other devices */
	(void) kls_raw_cmd(EXTCMD,CLRSPKMOD);
	(void) kls_raw_cmd(UARTCNT,MSDIS);
	(void) kls_raw_cmd(KYBDCMD,NKSCAN);

	/* The following routines have full control over the adapter 
	 * as long as they don't enable unsolicited input. They may
	 * place a command on the queue to do so. These routines are
	 * also responsible for any special cleanup caused by the
	 * rejection of the commands commands before the reset.
	 */

	spkreset();
	kls_pflush(4);	/* clean up any strays */
#if NMS > 0
	msreset();
	kls_pflush(4);	/* clean up any strays */
#endif NMS
	kbdreset();
	kls_pflush(4);
	
	beep();			       /* beep AFTER speaker has been inited! */

	/* start commands that have been queued */
	klsinreset = 0;
	if (klshead != NULL)
		klsstart();
	splx(s);

	s = spl0();		/* Go fully interruptable */
	while (klshead != NULL) delay(200); /* wait for stuf to happen */
	splx(s);
}
	
/* ARGSUSED */
klsint(unit, icscs, iar)
	int unit;
	int icscs;
	int iar;
{
	register int c;
	register int iid;
	register int klsret=KLSINVRET;
	register int not_our_int = 1;
	register int spkr,kbdsend;
	int lastscan = 0;
#ifdef SECURE
	int kbdchgsend = 0;
#endif SECURE

	spkr = kbdsend = 0;

/* empty the 8051 input buffer into clists */
	while (((iid=in(KLS_CNTIR)) & KLS_IBF) != 0) {
		if (!(iid & KLS_INT))
			break;
		not_our_int = 0;
		c = in(KLS_READ) & 0xff;
		switch (iid & IID_MASK) {
		case KLS_KBD:
			/* XXX need to check if we run out of clists */
			if (c != lastscan)
				putc(c,&kbd_data);
			lastscan = c;
			break;
#if NMS > 0
		case KLS_UART:
			putc(c,&ms_uart);
			break;
		case KLS_BLOCK:
			putc(c,&ms_block);
			break;
#endif NMS
		case KLS_INFO:
			if (c & KLSSRI)  { /* unsolicited info report */
			    if (c & SPKRDONE)
				spkr++;
			    if (c & KEYBDDONE)
				kbdsend++;
#ifdef SECURE
			    if (c & KEYBDLOCK)
				kbdchgsend++;
#endif SECURE
			    break;
			}		  /* fall through to KLS_REQ */
		case KLS_REQ:
			if (klsret != KLSINVRET) {
			    printf ("klsint: extra return status found \n");
#if	ROMP_DEBUG
			    printf (" old=0x%x new=0x%x dest=0x%x cmd=0x%x call=0x%x\n"
				,klsret,(((iid & IID_MASK) << 8) + c)
				,klshead->qp_dest,klshead->qp_cmd
				,klshead->qp_callback);
#endif ROMP_DEBUG
			}
			klsret = ((iid & IID_MASK) << 8) + c;
			break;
		case KLS_ERROR:
			klserror(c);
			break;
		case KLS_RESERVED:
		case KLS_SELF_TEST:
		default:
			printf("klsint: received unexpected iid = %x\n",iid & IID_MASK);
		}
	}

	/* call each of the interrupt routines if data is for them */
	if (klsret != KLSINVRET)
		klssendint(klsret);
	if (kbd_data.c_cc > 0) 
		kbdint();
#if NMS > 0
	if ((ms_block.c_cc > 0) || (ms_uart.c_cc > 0))
		msrint();
#endif NMS
	if (spkr)
		spkrint();
	if (kbdsend)
		kbdsint();
#ifdef SECURE
	if (kbdchgsend)
		kbd_lock_chg();
#endif SECURE
	return(not_our_int);
}

/* get queue command up from earlier level */
klsstrategy(qp)
	register volatile struct klsq *qp;
{
	register int	s=KLSSPL();

	qp->qp_next = NULL;
	qp->qp_ret = KLSINVRET;
	if (klshead == NULL) {
		klshead =qp;
		klstail =qp;
		if (!klsinreset)
			klsstart();
	} else {
		klstail->qp_next=qp;
		klstail = qp;
	}
	splx(s);
}

/* send command to 8051 */
klsstart()
{
	if (klshead == NULL)
		return;
	while ((in(KLS_CNTIR) & KLS_OBF) == 0)
		/* void */ DELAY(1);
	outw(KLS_WRITE, CMD(klshead->qp_dest,klshead->qp_cmd));
}

/* handle return status & start next command on queue */
klssendint(retstat)
	register int retstat;
{
	register volatile struct klsq *current=klshead;

	if (current == NULL)
		panic("klssendint: no commands");
	current->qp_ret=retstat;
	if ((klshead = current->qp_next) != NULL)
		klsstart();
	if (current->qp_callback)
		(* current->qp_callback) (current);
	else
		panic("klssendint: no callback ");
}

/* General command routine, used for queueing commands not requiring
 * special attention from the drivers 
 */
klscmd(dest,cmd,callback)
	register	char dest,cmd;
	register	int (*callback)();
{
	register volatile struct klsq *qp;

	qp=(volatile struct klsq *) klsalloc(&klsfreel);
	if (qp == NULL)
		return(-1);
	qp->qp_dest=dest;
	qp->qp_cmd=cmd;
	qp->qp_callback = (callback ? callback : klsdone);
	klsstrategy(qp);
	return(0);
}

klsdone(qp)
	register volatile struct klsq *qp;
{
	klsfree(klsfreel,qp);
}

caddr_t
klsminit(pool,struct_size,pool_size)
	caddr_t pool;
	int	 struct_size,pool_size;
{
	int	count;
	caddr_t	current=pool;

	/*
	 * the following cryptic loop links the elements of the pool together. This
	 * method is used to allow linking pools of different sizes
	 */
	for (count = 0 ; count < pool_size-1;  count++) {
		*(caddr_t *) current = (current + struct_size);
		current += struct_size;
	}
	*(caddr_t *) current = NULL;
	return(pool);
}
	
volatile struct klsq *
klsalloc(free)
	register volatile struct klsq **free;
{
	register volatile struct klsq *tmp = *free;

	*free = (tmp ? tmp->qp_next : NULL);
	return (tmp);
}

klserror(c)
	int c;
{
	printf("klsint: Error code detected 0x%x.\n",c);
	if ( (c >= KLS_ERROR_LOW) && (c <= KLS_ERROR_HIGH) )
		printf("%s.\n",klserrormsgs[c-KLS_ERROR_LOW]);
	switch (c) {
		case KBD_XMIT_TIMEOUT:
		case KBD_ACK_TIMEOUT:
		case KBD_XMIT_ERROR:
			kbdsint();
			break;
#if NMS > 0
		case MS_XMIT_TIMEOUT:
		case MS_ACK_TIMEOUT:
			msrint();
			break;
#endif NMS
		default:
			break;
	};
}

/*
 * Used by autoconf to wait for kls initialization to complete before
 * starting to probe controllers and devices.
 */
klswait()
{
	spkwait();
#if NMS > 0
	mswait();
#endif NMS
	kbdwait();
	while (klshead != NULL)
		DELAY(1);
}

