/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 **********************************************************************
 * HISTORY
 * $Log:	if_un.c,v $
 * Revision 2.2  88/08/24  01:46:35  mwyoung
 * 	Corrected include file references.
 * 	[88/08/22            mwyoung]
 * 
 *
 * 23-Apr-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	hc: turned off optimization.
 *
 *  7-Jul-86  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	Removed unnecessary include of pte.h
 *
 * 11-Jun-86  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	CS_GENERIC: Upgraded to 4.3 (quite a while ago).
 *
 ***********************************************************************
 */
#include <cmucs.h>
#include "romp_debug.h"

#ifdef	hc
pragma	off (optimize);
#endif	hc

/*
 * Ungermann-Bass PC-NIC (Ethernet) Adapter
 */

#include "un.h"
#if NUN > 0

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/buf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/vmmac.h>
#include <sys/ioctl.h>
#include <sys/errno.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/if_ether.h>
#if	CMUCS
#include <netinet/in_var.h>
#endif	CMUCS

#include <ca/io.h>
#include <caif/if_unreg.h>
#include <caio/ioccvar.h>
#include <ca/debug.h>

int	unprobe(), unattach();

caddr_t unstd[] = { (caddr_t) 0xf4080000, 0 };
struct	iocc_device *uninfo[NUN];

int	unint(),  uninit(), unioctl(), unoutput(), unreset();

struct	iocc_driver undriver =
	{ unprobe, 0, unattach, 0, unstd, "un", uninfo,
		0, 0, unint, UN_EADDROFF };

struct	mbuf *unget();

/*
 * Ethernet software status per adapter.
 */
struct	un_softc {
	struct	arpcom us_ac;		/* common Ethernet structures */
#define	us_if	us_ac.ac_if		/* network-visible interface */
#define	us_addr	us_ac.ac_enaddr		/* hardware Ethernet address */
	short	us_oactive;		/* is output active? */
	short	us_nextpage;		/* next receive buffer page */
	short	us_xbuf;		/* in-use xmt buf (if output active) */
	short	us_xfull[2];		/* 1 => a full xmt buf */
	short	us_xstart[2];		/* start address used in unstart */
} un_softc[NUN];

#if	ROMP_DEBUG
char undebug = 0;
#endif	ROMP_DEBUG

/*
 *  unprobe - try to generate an interrupt (to see if the board is there)
 */
unprobe(addr)
	register caddr_t addr;
{
	register struct undevice *undevicep = (struct undevice *) addr;

	unzap(undevicep);
	undevicep->un_csr = UN_GSFTINT;	/* generate software interrupt */
	PROBE_DELAY(100000);
	undevicep->un_csr = 0;
	return(PROBE_OK);
}

/*
 *  unattach - make the interface available to the network software
 *  (if the auto-configuration software determines that the interface
 *  exists).  The system will initialize the interface when it is
 *  ready to accept packets.
 */
unattach(iod)
	register struct iocc_device *iod;
{
	register struct un_softc *us = &un_softc[iod->iod_unit];
	register struct ifnet *ifp = &us->us_if;
	register struct undevice *addr = (struct undevice *)iod->iod_addr;
	register int i;
#if	CMUCS
#else	CMUCS
	struct in_addr arpmyaddr();
#endif	CMUCS

	ifp->if_unit = iod->iod_unit;
	ifp->if_name = "un";
	ifp->if_mtu = ETHERMTU;

	/*
	 * Read the ethernet address off the board.
	 * Save it and also write it to the edlc chip.
	 */
	for (i = 0; i < ETH_ADDR_SIZE; i++){
		addr->un_edlc.nodeID[i] =
			us->us_addr[i] = addr->un_eprom[UN_EADDROFF+i];
	}
	DEBUGF( 1,
		printf("un%d: ethernet address ", ifp->if_unit);
		for (i = 0; i < ETH_ADDR_SIZE; i++) {
			if (i != 0) printf(":");
			printf("%x", us->us_addr[i]);
		}
		printf("\n");
	)
#if	CMUCS
#else	CMUCS
	sin = (struct sockaddr_in *)&us->us_if.if_addr;
	sin->sin_family = AF_INET;
	sin->sin_addr = arpmyaddr((struct arpcom *)0);
#endif	CMUCS
	ifp->if_init = uninit;
	ifp->if_ioctl = unioctl;
	ifp->if_output = unoutput;
	ifp->if_reset = unreset;
	ifp->if_flags |= IFF_BROADCAST;
	if_attach(ifp);
	DEBUGF(undebug, printf("un%d: attached\n", iod->iod_unit);)
}

/*
 *  unreset - reset interface
 */
unreset(unit)
	register unsigned int unit;
{
	register struct iocc_device *iod;

	if (unit < NUN && (iod = uninfo[unit]) != 0 && iod->iod_alive != 0){
		uninit(unit);
		DEBUGF(undebug, printf("un%d: reset\n", unit);)
	}
}

/*
 *  uninit - initialize interface, enable packet reception, start any
 *  pending writes
 */
uninit(unit)
	register int unit;
{
	register struct un_softc *us = &un_softc[unit];
	register struct ifnet *ifp = &us->us_if;
	int s;
	register struct undevice *addr;
	register int i;

#if	CMUCS
	if (ifp->if_addrlist == (struct ifaddr *)0)
		return;
	if (ifp->if_flags & IFF_RUNNING)
		return;
#else	CMUCS
	sin = (struct sockaddr_in *)&ifp->if_addr;
	if (sin->sin_addr.s_addr == 0)		/* address still unknown */
		return;
	if ((us->us_if.if_flags & IFF_RUNNING) == 0) {
#endif	CMUCS
		addr = (struct undevice *) (uninfo[unit]->iod_addr);
		s = splimp();
		unzap(addr);	/* initialize hardware */
		for (i = 0; i < ETH_ADDR_SIZE; i++){
			addr->un_edlc.nodeID[i] = us->us_addr[i];
		}
		us->us_nextpage = 0;	/* nextpage for packet reception */
		us->us_oactive = 0;	/* output not active */
		us->us_if.if_flags |= IFF_UP|IFF_RUNNING;
		if (us->us_if.if_snd.ifq_head){	/* anything on send queue */
			struct mbuf *m;

			IF_DEQUEUE(&us->us_if.if_snd, m);
			unput(us, addr, m, 0);
			unstart(us, addr, 0);
			if (us->us_if.if_snd.ifq_head){
				IF_DEQUEUE(&us->us_if.if_snd, m);
				unput(us, addr, m, 1);
			}
		}
		splx(s);
#if	CMUCS
#else	CMUCS
	}
	if_rtinit(&us->us_if, RTF_UP);	/* initialize routing table entry */
	arpattach(&us->us_ac);	/* attach interface to ARP */
	arpwhohas(&us->us_ac, &sin->sin_addr);	/* broadcast my IP address */
						/* look for duplicates */
#endif	CMUCS
	DEBUGF(undebug, printf("un%d: init'ed\n", unit);)
}

/*
 *  unstart - start output from one of the adapter's 2 transmit buffers
 */
unstart(us, addr, xbuf)
	register struct un_softc *us;
	register struct undevice *addr;
	register int xbuf;
{
	us->us_oactive = 1;
	us->us_xbuf = xbuf;
	addr->un_csr = UN_IENABLE;	/* enable transmit done interrupt */
	UN_XMIT(addr, us->us_xstart[xbuf]);
}

/*
 *  unint - interrupt handler.  find the cause of the interrupt and
 *  dispatch an appropriate handler routine.
 */
unint(unit)
	register int unit;
{
	register struct un_softc *us = &un_softc[unit];
	register struct undevice *addr =
		(struct undevice *)uninfo[unit]->iod_addr;
	register status;
	register int rc = 1;

	while ((status = ~(addr->un_csr)) & UN_PAVINT) {
		DEBUGF(undebug & 0x2, printf("unint: unit = %d, csr = %b",
		  unit, status & 0xff, UN_CSRBITS);)
		unrint(unit, us, addr);
		rc = 0;
	}
	if (status & UN_TXRINT) {
		DEBUGF(undebug & 0x2, printf("unint: unit = %d, csr = %b",
		  unit, status & 0xff, UN_CSRBITS);)
		unxint(unit, us, addr);
		rc = 0;
	}
	UN_GLOBIENB(us->us_nextpage);	/* global interrrupt (re)enable */
	return(rc);
}

/*
 *  unrint - interrupt handler for packet reception.
 *
 *  log error if error bits are latched,  examine packet to determine
 *  type, if can't determine packet length from type, drop packet.
 *  otherwise decapsulate packet based on type and pass to an appropriate
 *  higher-level input routine.
 */

unrint(unit, us, addr)
	int unit;
	register struct un_softc *us;
	register struct undevice *addr;
{
	register struct ether_header *eh;
    	register struct mbuf *m;
	register int len;
	register int off;
	int resid;
	struct ifqueue *inq;
	char status = addr->un_edlc.rstat;
	u_short ungetushortatoff();

	addr->un_edlc.rstat = status;	/* clear status */
	/* (the hardware xor's in the value of status setting rstat to 0) */
	DEBUGF(undebug & 0x2, printf(" rstat = %b", status, RS_BITS);)
	/*
	 *  Latch errors.  (Errors found correspond to packets
	 *  that were received prior to the current packet
	 *  since packet available interrupts are generated
	 *  for good packets only.)
	 */
	if (status & RS_ERR){
		DEBUGF(undebug, printf("unrint: input error\n");)
		us->us_if.if_ierrors++;
	}
	us->us_if.if_ipackets++;

	/*
	 *  determine the length of the received packet.
	 */
	len = 0;
	off = us->us_nextpage;

#define BUMP(page)	if (++(page) == UN_NUMRBUFS) page = 0
	while ((addr->un_pram[us->us_nextpage] & UN_LAST_PAGE) == 0) {
		BUMP(us->us_nextpage);
		len += UN_RBUFSIZE;
	}
	len += (addr->un_pram[us->us_nextpage] & UN_PAGE_LENGTH_MASK) + 1;
	BUMP(us->us_nextpage);
	if (len > UN_XBSIZE) {
		printf("un%d: huge packet!\n",unit);
		len = UN_XBSIZE;
	}
	DEBUGF(undebug & 0x2, printf(" len = %d ", len);)
#undef BUMP
	/*
	 * Process the packet
	 */
	eh = (struct ether_header *) &addr->un_rcvbuf[off][0];
	len -= sizeof(struct ether_header);
	eh->ether_type = ntohs((u_short)eh->ether_type);
	/*
	 *  The ETHERPUP_NTRAILER packet types starting at ETHERPUP_TRAIL
	 *  have (type - ETHERPUP_TRAIL) * 512 bytes of data followed by
	 *  a type field and then a (variable length) header
	 */
#if	CMUCS
	if (eh->ether_type >= ETHERTYPE_TRAIL &&
	    eh->ether_type < ETHERTYPE_TRAIL+ETHERTYPE_NTRAILER) {
		off = (eh->ether_type - ETHERTYPE_TRAIL) * 512;
#else	CMUCS
	if (eh->ether_type >= ETHERPUP_TRAIL &&
	    eh->ether_type < ETHERPUP_TRAIL+ETHERPUP_NTRAILER) {
		off = (eh->ether_type - ETHERPUP_TRAIL) * 512;
#endif	CMUCS
		if (off >= ETHERMTU){
			goto chuckit;
		}
		eh->ether_type = ungetushortatoff(addr, eh, off);
		resid = ungetushortatoff(addr, eh, off + 2);
		if (off + resid > len){
			goto chuckit;
		}
		len = off + resid;
	} else {
		off = 0;
	}
	if (len == 0){
		goto chuckit;
	}
	/*
	 *  pull packet off interface.  if off is non-zero, the
	 *  packet has a trailing "header".  unget will move this
	 *  header to the front, but we still have to remove the
	 *  type and length fields from the front of the data.
	 */
#if	CMUCS
	m = unget(addr, (char *) eh, len, off, &us->us_if);
#else	CMUCS
	m = unget(addr, (char *) eh, len, off);
#endif	CMUCS
	/*
	 *  update the full page pointer and clear the packet available
	 *  flag if necessary.  update the fpp here to free the on-board
	 *  receive pages as soon as possible.
	 */
	unupdatefpp(addr, us->us_nextpage);
	if (m != 0){
		if (off) {
			m->m_off += 2 * sizeof(u_short);
			m->m_len -= 2 * sizeof(u_short);
		}
		switch (eh->ether_type) {
#ifdef INET
		case ETHERTYPE_IP:
			schednetisr(NETISR_IP);
			inq = &ipintrq;
			if (IF_QFULL(inq)) {
				DEBUGF(undebug & 0x2, printf(" qfull\n");)
				IF_DROP(inq);
				m_freem(m);
			} else {
				IF_ENQUEUE(inq, m);
				DEBUGF(undebug & 0x2, printf(" queued\n");)
			}
			break;

		case ETHERTYPE_ARP:
			arpinput(&us->us_ac, m);  /* arpinput frees m */
			break;
#endif
		default:
			m_freem(m);
			break;
		}
	}
	return;
chuckit:
	DEBUGF(undebug, printf("unrint: packet dropped\n");)
	unupdatefpp(addr, us->us_nextpage);
}

/*
 *  unxint -  interrupt handler for transmit ready
 */
unxint(unit, us, addr)
	register int unit;
	register struct un_softc *us;
	register struct undevice *addr;
{
	register int status;
	register int next_buf;

	/*
	 *  collect stats on last packet
	 */
	status = addr->un_edlc.xstat;
	addr->un_edlc.xstat = status;	/* clear status bits */
	DEBUGF(undebug & 0x2, printf(" unxint: xstat = %b\n",
	  status & 0xff, XS_BITS);)
	if (status & XS_16CL) {
		us->us_if.if_collisions += 16;
		us->us_if.if_oerrors++;
		printf("un%d: ethernet jammed\n", unit);
	}
	else if (status & XS_SHRT) {
		us->us_if.if_oerrors++;
		printf( "un%d: ethernet not responding (is it connected?)\n",
			unit);
	}
	else {
		us->us_if.if_opackets++;
		us->us_if.if_collisions += UN_NCOLL(addr);
	}
	DEBUGF(undebug & 0x2,
	  printf(" ipkt = %d ierr = %d okt = %d oerr = %d coll = %d\n",
	    us->us_if.if_ipackets, us->us_if.if_ierrors,
	    us->us_if.if_opackets, us->us_if.if_oerrors,
	    us->us_if.if_collisions);)
	/*  mark the current transmit buffer empty */
	us->us_xfull[us->us_xbuf] = 0;
	/*  switch to the other transmit buffer */
	next_buf = 1 - us->us_xbuf;
	if (us->us_xfull[next_buf]){	/*  if it's full */
		unstart(us, addr, next_buf);	/* start output from it */
		if (us->us_if.if_snd.ifq_head){	/*  if more on out queue */
			struct mbuf *m;

			IF_DEQUEUE(&us->us_if.if_snd, m); /* fill empty buf */
			unput(us, addr, m, 1 - next_buf);
		}
	}
	else {	/*  the other transmit buffer is empty */
		us->us_oactive = 0;
		addr->un_csr = UN_PAVIENB;	/* Turn off TxRIENB */
	}
}

/*
 *  unoutput - ethernet output routine.  encapsulate a packet of type
 *  family for the local net.  use trailer local net encapsulation if
 *  the number of bytes in the mbufs after the first is a multiple of
 *  512.
 */
unoutput(ifp, m0, dst)
	register struct ifnet *ifp;
	register struct mbuf *m0;
	register struct sockaddr *dst;
{
	int type;
	int s;
	int error;
	char edst[ETH_ADDR_SIZE];
	struct in_addr idst;
	register struct un_softc *us = &un_softc[ifp->if_unit];
	register struct mbuf *m = m0;
	register struct ether_header *eh;
	int off;
	struct mbuf *m_get();
	int usetrailers;

	switch (dst->sa_family) {

#ifdef INET
	case AF_INET:
		idst = ((struct sockaddr_in *)dst)->sin_addr;
		if (!arpresolve(&us->us_ac, m, &idst, edst,
			&usetrailers)){ /* not resolved */
			return(0);
		}
		off = ntohs((u_short)mtod(m, struct ip *)->ip_len) - m->m_len;
		if (((ifp->if_flags & IFF_NOTRAILERS) == 0) /* trailers OK */
		 && off > 0 && (off & 0x1ff) == 0 &&
		  m->m_off >= MMINOFF + 2 * sizeof(u_short)
		 && usetrailers) {
			type = ETHERTYPE_TRAIL + (off>>9);
			m->m_off -= 2 * sizeof(u_short);
			m->m_len += 2 * sizeof(u_short);
			*mtod(m, u_short *) = htons((u_short)ETHERTYPE_IP);
			*(mtod(m, u_short *) + 1) = htons((u_short)m->m_len);
			/*
			 *  Packet to be sent with trailer, move first packet
			 *  (control information) to end of chain.
			 */
			while (m->m_next)
				m = m->m_next;
			m->m_next = m0;
			m = m0->m_next;
			m0->m_next = 0;
			m0 = m;
		}
		else {
			type = ETHERTYPE_IP;
		}
		break;
#endif
	case AF_UNSPEC:
		eh = (struct ether_header *)dst->sa_data;
		bcopy((char *)eh->ether_dhost, (caddr_t)edst, sizeof(edst));
		type = eh->ether_type;
		break;
	default:
		printf("un%d: can't handle af%d\n", ifp->if_unit,
		  dst->sa_family);
		error = EAFNOSUPPORT;
		goto bad;
	}
	/*
	 * Add local net header.  If no space in first mbuf,
	 * allocate another.
	 */
	if (m->m_off > MMAXOFF ||
	    MMINOFF + sizeof(struct ether_header) > m->m_off) {
		m = m_get(M_DONTWAIT, MT_HEADER);
		/*
		 *  Note:  m_get, m_freem etc. guard against concurrent
		 *  updates to the list of free mbufs.
		 */
		if (m == 0) {
			error = ENOBUFS;
			goto bad;
		}
		m->m_next = m0;
		m->m_off = MMINOFF;
		m->m_len = sizeof(struct ether_header);
	} else {
		m->m_off -= sizeof(struct ether_header);
		m->m_len += sizeof(struct ether_header);
	}
	eh = mtod(m, struct ether_header *);
	bcopy((caddr_t)edst, (caddr_t)eh->ether_dhost, sizeof(edst));
	eh->ether_type = htons((u_short)type);
	bcopy((caddr_t)us->us_addr, (caddr_t)eh->ether_shost, ETH_ADDR_SIZE);
	/*
	 *  queue packet for transmission.  if there is an empty
	 *  transmit buffer on the adapter, use it.
	 */
	s = splimp();
	if (IF_QFULL(&ifp->if_snd)) {
		IF_DROP(&ifp->if_snd);
		error = ENOBUFS;
		goto qfull;
	}
	if (us->us_xfull[0] == 0 || us->us_xfull[1] == 0){ /* empty xmt buf */
		struct undevice *addr = (struct undevice *)
		  uninfo[ifp->if_unit]->iod_addr;
		int next_buf;

		if (us->us_xfull[0] == 0){
			next_buf = 0;
		}
		else {
			next_buf = 1;
		}
		unput(us, addr, m, next_buf);
		if (us->us_oactive == 0){
			unstart(us, addr, next_buf);
		}
	}
	else {
		IF_ENQUEUE(&ifp->if_snd, m);
	}
	splx(s);
	return(0);
qfull:
	m0 = m;
	splx(s);
bad:
	m_freem(m0);
	return(error);
}

/*
 *  unput -  copy packet from an  mbuf chain to one of the adapter's
 *  transmit buffers.  the packet is extended to the minimum legal
 *  size if necessary.  the extra bytes could be zeroed out to improve
 *  security but are not to maximize performance.
 */
unput(us, addr, m, xbuf)
	struct un_softc *us;
	struct undevice *addr;
	register struct mbuf *m;
	register int xbuf;
{
	register unsigned off;
	register struct mbuf *mp;
	register char *bp;

	/*
	 *  compute starting address in transmit buffer.  packets must be
	 *  "end_aligned".
	 */
	for (off = UN_XBSIZE, mp = m; mp; mp = mp->m_next){
		off -= mp->m_len;
	}
	if (UN_XBSIZE - off < ETHERMIN + sizeof(struct ether_header)){
		/*  packet too short => extend it */
		off = UN_XBSIZE - ETHERMIN - sizeof(struct ether_header);
	}
	if (xbuf == 1){		/* use the second buffer */
		off += UN_XBSIZE;	/* the 2 buffers are adjacent */
	}
	bp = ((char *)(addr->un_xmtbuf)) + off;
	for (mp = m; mp; mp = mp->m_next) {
		register unsigned len = mp->m_len;

		bcopy(mtod(mp, char *), bp, len);
		bp += len;
	}
	/* save starting address so interrupt handler can find it */
	us->us_xstart[xbuf] = off;  /* start address to be passed to adapter */
	us->us_xfull[xbuf] = 1;	/* mark buffer full */
	m_freem(m);
}

/*
 *  unget - copy packet from adapter's receive buffers into a chain of mbufs
 *
 */
struct mbuf *
#if	CMUCS
unget(addr, unbuf, totlen, off0, ifp)
#else	CMUCS
unget(addr, unbuf, totlen, off0)
#endif	CMUCS
	struct undevice *addr;
	char *unbuf;
	register int totlen;
	int off0;
#if	CMUCS
	struct ifnet *ifp;
#endif	CMUCS
{
	register struct mbuf *m;
	struct mbuf *top = 0;
	register struct mbuf **mp = &top;
	register int off = off0;
	register int len;
	register char *cp;

	cp = unbuf + sizeof(struct ether_header);
	while (totlen > 0) {
		MGET(m, M_DONTWAIT, MT_DATA);
		if (m == 0)
			goto bad;
		if (off) {	/* trailer exists */
			len = totlen - off;
			cp = unbuf + sizeof(struct ether_header) + off;
		} else
			len = totlen;
		if (len >= CLBYTES) {

#if	CMUCS
			if (ifp) {
				len = 0;
				goto nopage;
			}
			MCLGET(m);
			if (m->m_len != CLBYTES)
				goto nopage;
			len = CLBYTES;
			goto copy;
		}
nopage:
		m->m_off = MMINOFF;
		if (ifp) {
			m->m_len = len = MIN(MLEN - sizeof(ifp), len);
			m->m_off += sizeof(ifp);
		} else
			m->m_len = len = MIN(MLEN, len);
copy:
#else	CMUCS

			MCLGET(p, 1);
			if (p != 0) {
				m->m_len = len = CLBYTES;
				m->m_off = (int)p - (int)m;
			} else {
				m->m_len = len = MIN(MLEN, len);
				m->m_off = MMINOFF;
			}
		} else {
			m->m_len = len = MIN(MLEN, len);
			m->m_off = MMINOFF;
		}
#endif	CMUCS
		unbcopy(addr, cp, mtod(m, char *), len);
		cp += len;
		*mp = m;
		mp = &m->m_next;
#if	CMUCS
		if (off) {
			/* sort of an ALGOL-W style for statement... */
			off += len;
			if (off == totlen) {
				cp = unbuf + sizeof(struct ether_header);
				off = 0;
				totlen = off0;
			}
		} else
			totlen -= len;
		if (ifp) {
			m->m_len += sizeof(ifp);
			m->m_off -= sizeof(ifp);
			*(mtod(m, struct ifnet **)) = ifp;
			ifp = (struct ifnet *)0;
		}
#else	CMUCS
		if (off == 0) {
			totlen -= len;
			continue;
		}
		off += len;
		if (off == totlen) {
			cp = unbuf + sizeof(struct ether_header);
			off = 0;
			totlen = off0;
		}
#endif	CMUCS
	}
	return(top);
bad:
	m_freem(top);
	return(0);
}


/*
 *  ioctl - process an ioctl request.
 */
unioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	register int cmd;
	register caddr_t data;
{
	register struct ifreq *ifr = (struct ifreq *)data;
	register int s = splimp();
	register int error = 0;
#if	CMUCS
	register struct ifaddr *ifa = (struct ifaddr *)data;
#endif	CMUCS;

	switch (cmd) {
	case SIOCSIFADDR:
#if	CMUCS
		ifp->if_flags |= IFF_UP;
		uninit(ifp->if_unit);
		switch (ifa->ifa_addr.sa_family) {
#ifdef	INET
		case AF_INET:
			((struct arpcom *)ifp)->ac_ipaddr =
				IA_SIN(ifa)->sin_addr;
			arpwhohas((struct arpcom *)ifp, 
					&IA_SIN(ifa)->sin_addr);
			break;
#endif	INET
		default:
			error = EINVAL;
		}
#else	CMUCS
		if (ifp->if_flags & IFF_RUNNING)
			if_rtinit(ifp, -1);	/* delete previous route */
		unsetaddr(ifp, (struct sockaddr_in *)&ifr->ifr_addr);
		uninit(ifp->if_unit);
#endif	CMUCS
		break;
	default:
		error = EINVAL;
	}
	splx(s);
	return(error);
}

/*
 *  unsetaddr - set adapter's internet address
 */
#if	CMUCS
unsetaddr(ifa, sin)
	register struct ifaddr *ifa;
	register struct sockaddr_in *sin;
{
	ifa->ifa_addr = *(struct sockaddr *)sin;
}
#else	CMUCS
unsetaddr(ifp, sin)
	register struct ifnet *ifp;
	register struct sockaddr_in *sin;
{
	struct in_addr if_makeaddr();

	ifp->if_addr = *(struct sockaddr *)sin;
	ifp->if_net = in_netof(sin->sin_addr);
	ifp->if_host[0] = in_lnaof(sin->sin_addr);
	sin = (struct sockaddr_in *)&ifp->if_broadaddr;
	sin->sin_family = (short) AF_INET;
	sin->sin_addr = if_makeaddr(ifp->if_net, INADDR_ANY);
	ifp->if_flags |= IFF_BROADCAST;
}
#endif	CMUCS

/*
 *  unzap - initialize adapter
 */
unzap(addr)
	register struct undevice *addr;
{
	register int i;
	register int junk;

	addr->un_csr = 0;		/* disable interrupts */
	addr->un_edlc.reset = RESET_ON; /* set reset bit while init'ing */
	addr->un_edlc.rstat = RS_CLEAR;
	addr->un_edlc.xstat = XS_CLEAR;
	addr->un_edlc.rmask = 0;
	addr->un_edlc.xmask = 0;
	addr->un_edlc.rmode = RM_NORMAL;
	/*
	 *  the next line puts the transmitter in loopback mode so
	 *  that a spurious packet is not sent when the reset bit is
	 *  cleared.  it also allows us to reset the epp as described below
	 */
	addr->un_edlc.tmode = TM_NORMAL - TM_LBC;
	addr->un_edlc.reset = RESET_OFF; /* clear reset bit */
	addr->un_fppepp = 0;	/* set the fpp to 0 */
	/*
	 *  set the epp to 0 -  to do this send a bunch of packets to self.
	 *  this will force the epp to catch up with the fpp (which is already
	 *  0).  since we are in loopback mode, the following loop sends packets
	 *  to self
	 */
	for (i = 0; i < UN_NUMRBUFS; i++){
		UN_XMIT(addr, UN_XBSIZE - 100);	/* send 100 byte junk packet */
	}
	/*
	 *  clear the packet available flag.  to do this, read the empty
	 *  page pointer and then execute UN_CLRPAV.  The sequence is
	 *  explained as follows.  If a packet has come in since the empty
	 *  page pointer has been read, the adapter's "I know something
	 *  you don't know" flag is set and the UN_CLRPAV won't clear the
	 *  flag.  by reading the epp, the processor clears the "IKSYDK"  
	 *  flag so that the UN_CLRPAV will have the desired effect.
	 */
	junk = addr->un_fppepp;		/* clear the IKSYDK flag */
	UN_CLRPAV(addr);		/* clear the PAV flag */
	addr->un_edlc.tmode = TM_NORMAL; /* put transmitter in normal mode */
	addr->un_csr = UN_PAVIENB;  /* Allow packet available interrupts */
	UN_GLOBIENB(0);	/* global interrrupt enable */
}

/*
 *  unupdatefpp - update adapter's full page pointer and clear packet available
 *  flag if appropriate
 */
unupdatefpp(addr, nextpage)
	register struct undevice *addr;
	register int nextpage;
{
	if (nextpage == /* EPP */(addr->un_fppepp & UN_PAGE_MASK)) 
		UN_CLRPAV(addr);
	addr->un_fppepp /* FPP */ = nextpage;
}

/*
 *  unbcopy - similar to bcopy but can deal with packets that wrap
 *  around from the high end of the adapter's receive buffer to the
 *  low end
 */
unbcopy(addr, from, to, len)
	register struct undevice *addr;
	register char *from;
	register char *to;
	register int len;
{
	register char *high_end = &addr->un_rcvbuf[UN_LASTRBUF][UN_RBUFSIZE];
	register int n;

	if (from + len <= high_end){
		bcopy(from, to, len);
	}
	else if (from >= high_end){
		from -= sizeof(addr->un_rcvbuf);
		bcopy(from, to, len);
	} else {
		n = high_end - from;
		bcopy(from, to, n);
		to += n;
		bcopy((char *)addr->un_rcvbuf, to, len - n);
	}
}

/*
 *  ungetushortatoff - return the u_short at offset in the received packet,
 *  handling wrap-around in the receive buffer and conversion between network
 *  and host formats as necessary.
 */
u_short ungetushortatoff(addr, eh, off)
	register struct undevice *addr;
	register struct ether_header *eh;
	register int off;
{
	register char *high_end = &addr->un_rcvbuf[UN_LASTRBUF][UN_RBUFSIZE];
	register char *p;

	p = (caddr_t)(eh + 1) + off;
	if (p >= high_end){
		p -= sizeof(addr->un_rcvbuf);
	}
	return(ntohs(*((u_short *) p)));
}

#endif NUN > 0


