/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	asy.c,v $
 * Revision 2.2  88/09/08  16:19:26  parker
 * 	Added code from IBM ACIS at Palo Alto to use FIFO on asy card.
 * 	[88/09/08  16:08:03  parker]
 * 
 * 30-Sep-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added include of ../caio/cbuf.h and rearranged the way that the
 *	circular buffers maintained their information.  
 *
 * 22-Aug-87  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Turned off optimization, eliminated machine{,io,cons} style of
 *	includes, eliminated IBMRTPC conditionals, and deleted all code
 *	in #ifdef ATR ... #endif ATR conditionals.
 */
/*
 * 5799-CGZ (C) COPYRIGHT IBM CORPORATION 1987
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/* $ Header: asy.c,v 2.1 87/09/30 17:20:24 mrt Exp $ */
/* $ACIS:asy.c 9.3$ */
/* $ Source: /afs/cs.cmu.edu/bsd4.3/nsource/rcs/kernel/caio/RCS/asy.c,v $ */

#ifdef	hc
pragma	off (optimize);
#endif	hc

/*
 * Multiport asynchronous driver
 */
#include "asy.h"
#if NASY > 0
#include "param.h"
#include "conf.h"
#include "dir.h"
#include "user.h"
#include "proc.h"
#include "ioctl.h"
#include "tty.h"
#include "systm.h"
#include "uio.h"
#include "file.h"
#include "ca/debug.h"

#include "caio/asyreg.h"
#include "caio/cbuf.h"
#include "ca/io.h"
#include "buf.h"
#include "caio/ioccvar.h"

#include "ap.h"
#if NAP > 0
#include "caio/apdefs.h"
#endif

#include "ca/softint.h"

#define ASY_SETBAUD(addr, baud) { \
	    IOOUT( &(addr)->lcr, (IOIN( &(addr)->lcr ) | LCR_DLAB) );\
	    IOOUT( &(addr)->dllsb, asybaudtbl[(baud)].lsb);\
	    IOOUT( &(addr)->dlmsb, asybaudtbl[(baud)].msb);\
	    IOOUT( &(addr)->lcr, (IOIN( &(addr)->lcr ) & ~LCR_DLAB) );}

static struct asybaudtbl {
	unsigned char msb;
	unsigned char lsb;
} asybaudtbl[NSPEEDS] = {
	 /* B0     */ 	0x00, 0x00,
	 /* B50    */ 	0x09, 0x00,
	 /* B75    */ 	0x06, 0x00,
	 /* B110   */ 	0x04, 0x17,
	 /* B134   */ 	0x03, 0x5b,
	 /* B150   */ 	0x03, 0x00,
	 /* B200   */ 	0x02, 0x40,
	 /* B300   */ 	0x01, 0x80,
	 /* B600   */ 	0x00, 0xc0,
	 /* B1200  */ 	0x00, 0x60,
	 /* B1800  */ 	0x00, 0x40,
	 /* B2400  */ 	0x00, 0x30,
	 /* B4800  */ 	0x00, 0x18,
	 /* B9600  */ 	0x00, 0x0c,
	 /* B19200 */ 	0x00, 0x06,
	 /* EXTB   */ 	0x00, 0x00
};

#define ISPEED B9600		 /* default input and output speed */
/* #define ISPEED EXTB		 default input and output speed */
 
 /*
  * Encoding of the minor device number.
  * Numbering from the right to left:
  * 	Bits 0 & 1 encode the port number (0-3) on the multiport adapter
  * 	Bits 2, 3 & 4 encode the adapter number as follows:
  *			(0-3) for the multiport adapters
  *			(4-5) for the COM adapters
  *
  *			 4 3 2 1 0
  *			___________
  *			|     |   |
  *			-----------
  */
#define NPORTS	4		/* number of serial port per multiport card */
#define MAXUNIT (NASY * NPORTS)	/* total number of ports configured */
struct cbuf asycbuf[MAXUNIT];
 /*
  * Terminology:
  *
  *	ctlr    -- refers to the card number and can have a value from 0 to NASY
  *	port    -- always has a value from 0 to 3 to indicate slot A, B, C, 
  *		   or D	on a given ctlr
  *	unit    -- refers to the index into the tty struct array for a port
  *	           on any ctlr.
  */

 /*
  * Encoding of index into tty structure array:  
  *		2 2 2 2 | 1 1 1 1 | 1 1 1 1 | 1 1
  *	unit    3 2 1 0 | 9 8 7 6 | 5 4 3 2 | 1 0 9 8 | 7 6 5 4 | 3 2 1 0
  *		________|_________|_________|_________|_________|________
  *	port	      0 |       0 | 3 2 1 0 | 3 2 1 0 | 3 2 1 0 | 3 2 1 0
  *	        --------|---------|---------|---------|---------|--------
  *	ctlr       5         4	       3         2	   1         0
  *
  * Note that 17-19 and 21-23 are not used since there is only one port on
  * each of the COM cards.
  *
  * To calculate the index number:
  *			UNIT = (ctlr# * NPORTS ) + port#
  */
#define UNIT(ctlr, port)	(((ctlr) << 2) + (port))
#define CTLR(unit)		((unit) >> 2)
#define PORT(unit)		((unit) & 0x03)
#define PORTMASK(port)		((1 << (port)) & 0x0f) 
 
  
struct tty asy[MAXUNIT];
int	nasy = MAXUNIT;

#if NAP > 0
int asyoutc();
#endif
int asystart(), ttrstrt(), asyattach(), asyprobe(), asyint();

/* following is a hokey test for a quad card */
#define isquad(addr) (0x0000f000 & (int) (addr) )

/* define standard addresses for autoconf */
caddr_t asystd[] = {
	(caddr_t)0xf0001230,	       /* Multiport adapter 1 */
	(caddr_t)0xf0002230,	       /* Multiport adapter 2 */
	(caddr_t)0xf0003230,	       /* Multiport adapter 3 */
	(caddr_t)0xf0004230,	       /* Multiport adapter 4 */
/* we do not define the COM cards here for an RT because this screws up the 
 * the ctlr number in autoconf if there are no multiport cards present
 */
	0
};

#define BADD_ADDR (caddr_t) 0xeeeeeeee /* to catch bad IRQ number */

caddr_t asyreset[16] = {
	BADD_ADDR, BADD_ADDR, BADD_ADDR, BADD_ADDR, /* 0 - 3 */
	BADD_ADDR, BADD_ADDR, BADD_ADDR, BADD_ADDR, /* 4 - 7 */
	BADD_ADDR, (caddr_t)0xf00002f2,	
	  (caddr_t)0xf00006f2, (caddr_t)0xf00006f3, /* 8 - 11 */
	BADD_ADDR, BADD_ADDR, BADD_ADDR, BADD_ADDR, /* 12 - 15 */
};

char asysoftCAR[NASY];	    /*  software carrier detect  */
unsigned char asysoftFIFO[NASY];	/*  software - FIFO mode */
#define fifomode 0xff 

struct iocc_device *asydinfo[NASY];
struct iocc_driver asydriver = {
	/* probe, slave, attach, dgo, addr, dname, dinfo, mname, csr, intr */
	   asyprobe, 0, asyattach, 0, asystd, "asy", asydinfo, 0, 0, asyint
};

/*   asyprobe will try to generate an interrupt so that the system can
 *   determine whether or not the async ctlrs specified in the config
 *   file are truly present.  It will also try to determine the ctlrs
 *   interrupt request level (IRQ).  The interrupts are generated using 
 *   the loopback feature provided by the INS8250.
 */

asyprobe(addr)
	register caddr_t addr;
{
	register struct asydevice *asyp = (struct asydevice *)addr;
	char saveiir;

	DEBUGF(ttydebug, printf("asyprobe(%x):\n", addr));

	if (isquad(asyp)) {
		IOOUT( &asyp->ppir, 0xff); /* initialize ppir - ONLY ONCE */
		DEBUGF(ttydebug, 
		printf("ppir=%x, iir=%x\n",IOIN(&asyp->ppir),IOIN(&asyp->iir)));

		/* reset all shared interrupts */
		IOOUT(asyreset[9], 0);
		IOOUT(asyreset[10], 0);
		IOOUT(asyreset[11], 0);
	}
	
	DEBUGF(ttydebug, printf(" lsr=%x,", IOIN( &asyp->lsr ) ) );
	DEBUGF(ttydebug, printf(" msr=%x\n", IOIN( &asyp->msr ) ) );

	IOOUT(&asyp->ier, IER_MSR);     	/* enable modem status int. */
	IOOUT(&asyp->mcr, MCR_IEN);		/* enable modem ints (only */
	DELAY(1);				/* for COM card; does not */
						/* affect Multiport card) */
	IOOUT(&asyp->mcr, MCR_IEN | MCR_LOOP) ; /* enable LOOP mode */
	DELAY(1);
	IOOUT(&asyp->mcr, MCR_IEN); 		/* disable LOOP mode */
	PROBE_DELAY(10000);
	DEBUGF(ttydebug, printf(" iir=%x", IOIN( &asyp->iir )));
 
	IOOUT(&asyp->ier,0);	        	/* disable all interrupts */
	saveiir = IOIN(&asyp->msr);		/* reset modem interrupts */
	DEBUGF(ttydebug, printf(" msr=%x\n", saveiir));

	/* reset all shared interrupts */
	if (isquad(addr)) {
		IOOUT(asyreset[9], 0);
		IOOUT(asyreset[10], 0);
		IOOUT(asyreset[11], 0);
		PROBE_DELAY(10000);
	}
	return (PROBE_OK);
}


asyattach(iod)
	register struct iocc_device *iod;
{
	register struct tty *tp;
	register int ctlr = iod->iod_unit;
	register int port;  /* line number on this ctlr */
	register struct asydevice *addr = (struct asydevice *)iod->iod_addr;

	DEBUGF(ttydebug & SHOW_INIT, printf("asyattach: ctlr=%d addr=%x\n", ctlr,addr));
	/* initializes the tty struct for each port on this ctlr */
	for (port = 0, tp = &asy[UNIT(ctlr,port)]; port < NPORTS; port++, addr++, tp++) {
		DEBUGF(ttydebug & SHOW_INIT, 
		printf("port = %d, tp = %x, addr = %x\n", port, tp, addr ));
		tp->t_addr = (caddr_t)&asycbuf[UNIT(ctlr,port)];
		asycbuf[UNIT(ctlr,port)].t_addr = (caddr_t)addr;
		tp->t_state = 0;
		if (!isquad(addr))
			break;	       /* only one port per PC card */
	}
	asysoftCAR[iod->iod_unit] = iod->iod_flags; /* carrier detect */
 
}


/*ARGSUSED*/
asyopen(dev, flag)
	dev_t dev;
{
	register struct tty *tp;
	register int unit = minor(dev);
	register int s, error;
	register struct asydevice *asyaddr;
	register struct iocc_device *iod;
	int ctlr = CTLR(unit);
	register unsigned char saveiir;

	DEBUGF(ttydebug & SHOW_OPEN, printf("asyopen%x: \n", unit));
	if (unit >=MAXUNIT || (iod=asydinfo[ctlr])==0 || iod->iod_alive==0)
		return (ENXIO);

	tp = &asy[unit];
	if (tp->t_state & TS_XCLUDE && u.u_uid != 0)
		return (EBUSY);

	asyaddr = (struct asydevice *)((struct cbuf *)tp->t_addr)->t_addr;
	tp->t_state |= TS_WOPEN;

	/* if this is first open, initialize tty state to defaults */
	if ((tp->t_state & TS_ISOPEN) == 0) {
		ttychars(tp);
		tp->t_oproc = asystart;
		tp->t_ospeed = tp->t_ispeed = ISPEED;
		tp->t_flags = ODDP | EVENP | ECHO;
		asyparam(unit);

		s = spl_asy();

		/* turn DTR RTS on */
		IOOUT(&asyaddr->mcr, MCR_IEN | MCR_DTR | MCR_RTS);

		/* enable receive,line and modem interrupts */
		IOOUT(&asyaddr->ier, IER_DR | IER_LSR | IER_MSR);

		/* enable FIFO mode */
                /* This has no effect for a non-buffered card. */
		IOOUT(&asyaddr->fcr, FCR_FCR0);

		(void) splx(s);
	} 
	IOOUT(&asyaddr->fcr, FCR_FCR0);   

	s = spl_asy();
	/* wait for carrier to come up */
	/* check if modem control is on or the carrier is up */
	if ( ( (IOIN(&asyaddr->msr)) & MSR_DCD) ||
	     ( asysoftCAR[ctlr] & PORTMASK(PORT(unit)) ) ) {
	      		tp->t_state |= TS_CARR_ON;
			saveiir=IOIN(&asyaddr->iir) ;
			if (saveiir & (IIR_FIFO1 | IIR_FIFO2)) {
			asysoftFIFO[ctlr] = fifomode;	
			DEBUGF(ttydebug & SHOW_OPEN,
				printf("asyopen - fifomode\n"));
		}
         }

	while ( (tp->t_state & TS_CARR_ON) == 0 ) {
		tp->t_state |= TS_WOPEN;
		sleep((caddr_t) & tp->t_rawq, TTIPRI);
	}
	(void) splx(s);

	DEBUGF(ttydebug & SHOW_OPEN, printf("asyopen%d: ier=%x, mcr=%x", 
		unit, IOIN(&asyaddr->ier), IOIN(&asyaddr->mcr) ));
	if ( error = (*linesw[tp->t_line].l_open)(dev, tp) ) 
		asyclose(dev);
	DEBUGF(ttydebug & SHOW_OPEN, printf(" ldisc=%x, flags=%x, state=%x\n",
		tp->t_line, tp->t_flags, tp->t_state ));
	return ( error );
}


/*ARGSUSED*/
asyclose(dev)
	dev_t dev;
{
	register int unit = minor(dev);
	register struct tty *tp = &asy[unit];
	register struct asydevice *asyaddr;

	DEBUGF(ttydebug & SHOW_OPEN, printf("In asyclose ..."));
	
	asyaddr = (struct asydevice *)((struct cbuf *)tp->t_addr)->t_addr;
	(*linesw[tp->t_line].l_close)(tp);
	IOOUT(&asyaddr->lcr, IOIN(&asyaddr->lcr) & ~LCR_SETB);  /* clr break */
	if ((tp->t_state&(TS_WOPEN|TS_HUPCLS))||(tp->t_state&TS_ISOPEN)==0) {
		IOOUT(&asyaddr->ier, ~(IER_DR | IER_LSR | IER_MSR) );
		IOOUT(&asyaddr->mcr, MCR_IEN);
		IOOUT(&asyaddr->fcr, ~(FCR_FCR0));
		DEBUGF(ttydebug & SHOW_OPEN, 
			printf("turning off DTR, RTS...disabling ints..."));
	}
	ttyclose(tp);
	DEBUGF(ttydebug & SHOW_OPEN, printf("asyclose end\n"));
}


/*ARGSUSED*/
asyread(dev, uio)
	register dev_t dev;
	register struct uio *uio;
{
	register struct tty *tp = &asy[minor(dev)];
	DEBUGF(ttydebug & SHOW_RDWR, {
	    register struct asydevice *asyaddr;
	    asyaddr = (struct asydevice *)((struct cbuf *)tp->t_addr)->t_addr;
	    printf("In asyread: asyaddr=(%x) lcr=(%x) ier=(%x)\n", 
		    asyaddr, IOIN(&asyaddr->lcr), IOIN(&asyaddr->ier) ); 
	} );
	return ((*linesw[tp->t_line].l_read)(tp, uio));
}


/*ARGSUSED*/
asywrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register struct tty *tp = &asy[minor(dev)];

	DEBUGF(ttydebug & SHOW_RDWR, printf("In asywrite\n"));
	return ((*linesw[tp->t_line].l_write)(tp, uio));
}



/*ARGSUSED*/
/* modeled after dh-11 */
asyioctl(dev, cmd, addr, flag)
	dev_t dev;
	caddr_t addr;
{
	register int unit = minor(dev);
	register struct tty *tp = &asy[unit];
	register struct asydevice *asyaddr;
	register int error, s;
	int trigger_level;
	char trigger_bits;

	asyaddr = (struct asydevice *)((struct cbuf *)tp->t_addr)->t_addr;
	DEBUGF(ttydebug, printf("asyioctl%d: cmd=%x, flag=%x\n",unit,cmd,flag));
	error = (*linesw[tp->t_line].l_ioctl)(tp, cmd, addr, flag);
#if NAP > 0
	if ( cmd == TIOCSETD )
		if ( *(int *)addr == APLDISC )
			tp->t_oproc = asyoutc;
		else 
			tp->t_oproc = asystart;
#endif NAP
	if (error >= 0)
		return (error);
	error = ttioctl(tp, cmd, addr, flag);
	if (error >= 0) {
		if (cmd == TIOCSETP || cmd == TIOCSETN || cmd == TIOCLBIS ||
		    cmd == TIOCLBIC || cmd == TIOCLSET ) {
			asyparam(unit);
		}
		return (error);
	}
	s = spl_asy();
	switch (cmd) {
	case TIOCSBRK:		       /* Set break */
		IOOUT(&asyaddr->lcr, IOIN(&asyaddr->lcr) | LCR_SETB);
		break;
	case TIOCCBRK:		       /* Clear break */
		IOOUT(&asyaddr->lcr, IOIN(&asyaddr->lcr) & ~LCR_SETB);
		break;
	case TIOCSDTR:		       /* Set DTR and RTS */
		IOOUT(&asyaddr->mcr, MCR_IEN | MCR_DTR | MCR_RTS);
		break;
	case TIOCCDTR:		       /* Clear DTR and RTS */
		IOOUT(&asyaddr->mcr, MCR_IEN);
		break;
#if	0
	case TIOCSETT:		       /* Set trigger level */
		trigger_level = *(int *)addr;
		if (trigger_level < 4)  
 			trigger_bits = 0;
		else
			if (trigger_level < 8)
				trigger_bits = FCR_FCR6;
			else
	       		 	if (trigger_level < 14)
					trigger_bits = FCR_FCR7;
				else 
 					trigger_bits = (FCR_FCR7 | FCR_FCR6);
			
		IOOUT(&asyaddr->fcr, (FCR_FCR0 | trigger_bits));
		break;	
#endif	/* 0 */
	default:
		(void) splx(s);
		return (ENOTTY);
	}
	(void) splx(s);
	return (0);
}




asyparam(unit)
	register int unit;
{
	register struct tty *tp = &asy[unit];
	register struct asydevice *asyaddr;
	register char bits;
	register int s;

	asyaddr = (struct asydevice *)((struct cbuf *)tp->t_addr)->t_addr;
	DEBUGF(ttydebug & SHOW_IO, 
 	    printf("asyparam%d: tp->t_flags=(%x)", unit, tp->t_flags));

	s = spl_asy();
	if (tp->t_ispeed == B0) {
		tp->t_state |= TS_HUPCLS;
		IOOUT(&asyaddr->mcr, MCR_IEN); /* hang up line (DTR,RTS =0) */
		(void) splx(s);
		return;
	}

	/* Set the baud rate and initial asy settings */
	ASY_SETBAUD(asyaddr, tp->t_ispeed);
	if (tp->t_flags & (RAW | LITOUT | PASS8))
		bits = LCR_8BIT;
	else
		bits = (LCR_7BIT | LCR_PEN);
	if (tp->t_flags & EVENP)
		bits |= LCR_EPS;
	if (tp->t_ispeed == B110)
		bits |= LCR_STB;
	IOOUT(&asyaddr->lcr, bits);


	DEBUGF(ttydebug & SHOW_REGS, printf("lcr=(%x)\n", IOIN(&asyaddr->lcr)));
	DEBUGF(ttydebug & SHOW_IO, printf(" asyparam end\n"));
	(void) splx(s);
}


/*
 * unit indicates which port on a 4 line ctlr caused the interrupt
 */
asyint(ctlr)
	register int ctlr;
{
	register unsigned char port;
	register int unit = UNIT(ctlr,0);
	register struct asydevice *asyaddr = (struct asydevice *)asydinfo[ctlr]->iod_addr;
	register struct tty *tp;
	register unsigned char saveiir;
	register int i, didit = 0;

	DEBUGF(ttydebug & SHOW_INTR,printf("asyint%d: addr=%x ",unit,asyaddr));
	if (isquad(asyaddr)) {
		port = IOIN(&asyaddr->ppir);
		DEBUGF(ttydebug & SHOW_INTR, printf("port=%x ",port));
		for ( i = 0 ; i < NPORTS ; i++, unit++ ) {
			if (port & PORTMASK(i) ) {
				tp = &asy[unit];
				asyaddr = (struct asydevice *)
					((struct cbuf *)tp->t_addr)->t_addr;
				while (((saveiir=IOIN(&asyaddr->iir))&IIR_PEND)==0) {
					DEBUGF(ttydebug & SHOW_REGS, 
					    printf("saveiir=%x\n", saveiir));
					switch (saveiir & IIR_MASK) {
					case IIR_LSR:
					case IIR_RXB:
						asyrint(unit, tp, asyaddr);
						didit++;
						break;
					case IIR_TXB:
						asyxint(unit, tp, asyaddr);
						didit++;
						break;
					case IIR_MSR:
						asymint(unit, tp, asyaddr);
						didit++;
						break;
					default:
						printf("asy%d:iir=%x\n",unit,saveiir);
						break;
					}
				}
			}
		}
		if ( didit )
			IOOUT(asyreset[asydinfo[ctlr]->iod_irq & 0xf], 0);
	} else 
	{
		tp = &asy[unit];
		asyaddr = (struct asydevice *)
			((struct cbuf *)tp->t_addr)->t_addr;
		while (( (saveiir=IOIN(&asyaddr->iir) ) & IIR_PEND) == 0) {
			switch (saveiir & IIR_MASK) {
			case IIR_LSR:
			case IIR_RXB:
				asyrint(unit, tp, asyaddr);
				didit++;
				break;
			case IIR_TXB:
				asyxint(unit, tp, asyaddr);
				didit++;
			break;
			case IIR_MSR:
				asymint(unit, tp, asyaddr);
				didit++;
				break;
			default:
				printf("asy%d:iir=%x\n",unit,saveiir);
				break;
			}
		}
	}
	DEBUGF(ttydebug & SHOW_INTR, printf("didit=%x\n",didit ));
	return( didit == 0 );
}

/*ARGSUSED*/
asyrint(unit, tp, asyaddr)
	register int unit;
	register struct tty *tp;
	register struct asydevice *asyaddr;
{
	register int c, err;
	register char savelsr;
	int charcount = 0;
	struct cbuf *cbuf = &asycbuf[unit];
	
	DEBUGF(ttydebug & SHOW_INTR, printf("asyrint%d: \n", unit));
	while ((savelsr = IOIN(&asyaddr->lsr)) & LSR_DR) {
		c = IOIN(&asyaddr->rxb);
#if NAP > 0
		if (tp->t_line == APLDISC) {
			if (savelsr & LSR_BI)
				c = APBREAK;
			else if (savelsr & LSR_FE)
				c = APERROR;
			DEBUGF(ttydebug & SHOW_CHARS, 
				printf("AP GOT -> (%x)\n", (char)c));
			(*linesw[tp->t_line].l_rint)(c, tp);
		} else 
#endif NAP
		{
		if (savelsr & (LSR_PE | LSR_FE | LSR_OR | LSR_BI)) {
			if ((savelsr & LSR_PE) && 
			    ((tp->t_flags & (EVENP|ODDP)) != (EVENP|ODDP)) &&
			    ((tp->t_flags & (EVENP|ODDP)) != 0) ) {
				DEBUGF(ttydebug,
					printf("asy%d: parity error\n",unit));
	    			continue;
			}
			/*
			 * A framing error (break) generates
			 * a null (in raw mode, for getty), or
			 * an interrupt (in cooked/cbreak modes).
			 */
			if (savelsr & (LSR_FE | LSR_BI))
				if (tp->t_flags & RAW)
					c = 0;
				else
					c = tp->t_intrc;

			if (savelsr & LSR_OR)
				printf("asy%d: overrun error\n", unit );
			DEBUGF(ttydebug & SHOW_CHARS, 
				printf("GOT -> (%c) (%x)\n", c, c));
		} /* end if errors */
		DEBUGF(ttydebug & SHOW_REGS, printf("TTYBUF\n")); 
		charcount++;
		CPUT(cbuf, c, err );
		if ( err ) {
			printf("asy%d: cbuf full\n", unit ); 
			break;
		}
	        } /* end else */
	} /* end while */
	if ( charcount ) {
		TTY_ENQUEUE( &ttyintrq, tp );
		setsoftTTY();
	}
	DEBUGF(ttydebug & SHOW_INTR, 
		printf("asy%d: charcount=%d\n",unit,charcount));
}

/*ARGSUSED*/
asyxint(unit, tp, asyaddr)
	register int unit;
	register struct tty *tp;
	register struct asydevice *asyaddr;
{
	DEBUGF(ttydebug & SHOW_INTR, printf("asyxint%d: ", unit));
#if NAP > 0
	if (tp->t_line == APLDISC) {
		DEBUGF(ttydebug & SHOW_INTR, printf("calling AP xint \n"));
		(*linesw[tp->t_line].l_start)(tp);
	} else 
#endif NAP
	{
		tp->t_state &= ~TS_BUSY;
		if (tp->t_outq.c_cc <= 0) {
			/* Let asystart turn on interrupts if it has to */
			IOOUT(&asyaddr->ier, IOIN(&asyaddr->ier) & ~IER_THRE);
		}
		if (tp->t_line)
			(*linesw[tp->t_line].l_start)(tp);
		else
			asystart(tp);
	}
	DEBUGF(ttydebug & SHOW_INTR, printf("asyxint end\n"));
}


asystart(tp)
	register struct tty *tp;
{
	register struct asydevice *asyaddr;
	register int c, s, i;
	register int unit = minor(tp -> t_dev);
	register int ctlr = CTLR(unit);       

	asyaddr = (struct asydevice *)((struct cbuf *)tp->t_addr)->t_addr;
	DEBUGF(ttydebug & SHOW_IO, printf("asystart addr=%x\n", asyaddr));
	/*
	 * Must hold interrupts to prevent state of the tp
	 * from changing
	 */
	s = spl_asy();
	/*
	 * if it's currently active, or delaying, no need to do anything.
	 */
	if (tp->t_state & (TS_TIMEOUT | TS_BUSY | TS_TTSTOP)) {
		splx(s);
		return;
	}
	if (asysoftFIFO[ctlr] == fifomode) {
 		DEBUGF(ttydebug & SHOW_CHARS, 
			printf("asystart - fifomode\n")); 
		if (IOIN(&asyaddr->lsr) & LSR_THRE) {
			/* xmit up to 16 chars per interrupt for buffered card */
			for (i = 1; i <= 16; i++) { 
				if ((tp->t_outq.c_cc <= 0) || (c = getc(&tp->t_outq)) == -1) {
					goto out1;
		 		}
				DEBUGF(ttydebug & SHOW_CHARS,
					printf("put (%c) (%x)\n", (char)(c), (char)(c) ));
				if ( tp->t_flags & (RAW | LITOUT) || (c <= 0177) ) {
					IOOUT(&asyaddr->txb, c);
				} else {
					timeout(ttrstrt, (caddr_t)tp, (c & 0177));
					tp->t_state |= TS_TIMEOUT;
					DEBUGF(ttydebug & SHOW_CHARS,printf("asystart timeout\n"));
					goto out1;
				}
        		} /* end of for loop */    
		}  /* end of LSR_THRE */
	}  /* end of FIFO mode */
	else {  /* non-FIFO mode */
		DEBUGF(ttydebug & SHOW_CHARS, 
			printf("asystart - non-fifomode/n"));
		while (IOIN(&asyaddr->lsr) & LSR_THRE) { 
			if ((tp->t_outq.c_cc <= 0) || (c = getc(&tp->t_outq)) == -1) {
				goto out1;
			}
			DEBUGF(ttydebug & SHOW_CHARS,
				printf("put (%c) (%x)\n", (char)(c), (char)(c) ));
			if ( tp->t_flags & (RAW | LITOUT) || (c <= 0177) ) {
				IOOUT(&asyaddr->txb, c);
			} else {
				timeout(ttrstrt, (caddr_t)tp, (c & 0177));
				tp->t_state |= TS_TIMEOUT;
				DEBUGF(ttydebug & SHOW_CHARS,printf("asystart timeout\n"));
				goto out1;
			}
		} /* end of while */
        }  /* end of non-FIFO mode */
	tp->t_state |= TS_BUSY;
	DEBUGF(ttydebug & SHOW_REGS, printf("asystart: enable TX int\n"));
	IOOUT(&asyaddr->ier, IOIN(&asyaddr->ier) | (IER_THRE));
	goto out2;
out1:
	DEBUGF(ttydebug & SHOW_REGS, printf("asystart: disable TX int\n"));
	IOOUT(&asyaddr->ier, IOIN(&asyaddr->ier) & ~(IER_THRE));
	tp->t_state &= ~TS_BUSY;

	/*
	 * if there are sleepers, and output has drained below low
	 * water mark, wake up sleepers.
	 */

	if (tp->t_outq.c_cc <= TTLOWAT(tp)) {
		if (tp->t_state & TS_ASLEEP) {
			tp->t_state &= ~TS_ASLEEP;
			wakeup((caddr_t) & tp->t_outq);
		}
		if (tp->t_wsel) {
			selwakeup(tp->t_wsel, tp->t_state & TS_WCOLL);
			tp->t_wsel = 0;
			tp->t_state &= ~TS_WCOLL;
		}
	}
out2:
	(void) splx(s);
	DEBUGF(ttydebug & SHOW_IO, printf(" asystart: done\n" ));
}


asymint(unit, tp, asyaddr)
	register int unit;
	register struct tty *tp;
	register struct asydevice *asyaddr;
{
	register char savemsr;

	savemsr = IOIN(&asyaddr->msr);
	DEBUGF(ttydebug & SHOW_INTR, 
		printf("asymint%d: savemsr=%x\n", unit, savemsr ) );
#if NAP > 0
	if (tp->t_line == APLDISC)
		return (0);
#endif
	if (savemsr & MSR_DDCD) {     /* if carrier transition */
		if (savemsr & MSR_DCD) { /* carrier on */
			DEBUGF( ttydebug & SHOW_INTR,printf("Carrier detect "));
			if ((tp->t_state & TS_CARR_ON) == 0) {
				wakeup((caddr_t) & tp->t_rawq);
				tp->t_state |= TS_CARR_ON;
			}
		} else {  		  /* carrier dropped */
			DEBUGF( ttydebug & SHOW_INTR,printf("Carrier lost "));	
			if (tp->t_state & TS_CARR_ON) {
				gsignal(tp->t_pgrp, SIGHUP);
				gsignal(tp->t_pgrp, SIGCONT);
				ttyflush(tp, FREAD | FWRITE);
			}
			tp->t_state &= ~TS_CARR_ON;
		}
	}
	DEBUGF(ttydebug & SHOW_INTR, printf("asymint end\n"));
}


asyselect(dev, rw)
	dev_t dev;
	int rw;
{
	register struct tty *tp = &asy[minor(dev)];

	return ((*linesw[tp->t_line].l_select)(dev, rw));
}

/* 
 * This routine is for the support of the 3812 printer. It will perform one
 * of three functions:
 *	    1) set the break bit 
 *	    2) clear the break bit  
 *	or  3) output one character.
 */
#if NAP > 0
asyoutc(tp, c)
	register struct tty *tp;
	register int c;
{
	register struct asydevice *asyaddr;
	register int s = spl_asy();

	asyaddr = (struct asydevice *)((struct cbuf *)tp->t_addr)->t_addr;
	DEBUGF(ttydebug & SHOW_CHARS, printf("asyoutc: PUT (%x)\n", (char) c));
	if (c >= 0)
		IOOUT(&asyaddr->txb, c);
	else {
		switch (c) {
		case APBREAKON:
			IOOUT(&asyaddr->lcr, IOIN(&asyaddr->lcr) | LCR_SETB);
			break;
		case APBREAKOFF:
			IOOUT(&asyaddr->lcr, IOIN(&asyaddr->lcr) & ~LCR_SETB);
			break;
		case APINTMASK:
			IOOUT(&asyaddr->ier, (IER_LSR | IER_THRE | IER_DR));
			break;
		}
	}
	(void) splx(s);
}
#endif

#endif NASY
