/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/**********************************************************************
 * HISTORY
 * 20-Apr-88  Richard Sanzi (sanzi) at Carnegie-Mellon University
 *	Added mswait() from ACIS 4.3 release 2.
 * 
 * 23-Apr-87  Bill Bolosky (bolosky) at Carnegie-Mellon University
 *	hc: turned off optimization.
 *
 * 29-Jan-87  Rich Sanzi (sanzi) at Carnegie-Mellon University
 *	Removed includes of "../h/tbioctl.h" and "../h/tbdefs.h" and
 *	replaced with an include of "../h/tablet.h".
 ***********************************************************************
 */
 
#ifdef	hc
pragma	off (optimize);
#endif	hc

/*
 * 5799-CGZ (C) COPYRIGHT IBM CORPORATION  1986
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/*
 * ibm032 Mouse driver, which works closely with the keyboard
 * code in cons.c
 */
#include "ms.h"
#if NMS > 0
#include "romp_debug.h"
#include "param.h"
#include "conf.h"
#include "dir.h"
#include "user.h"
#include "proc.h"
#include "ioctl.h"
#include "tty.h"
#include "systm.h"
#include "uio.h"
#include "../ca/debug.h"
#include "buf.h"
#include "../cacons/kls.h"
#include "../caio/mouseio.h"
#include "../caio/mousereg.h"
#include "../caio/mousevar.h"
#include "../cacons/consio.h"

#include "tablet.h"

struct ms_softc ms;
struct mouseq	mscmdq[MSPOOLSZ];
struct mouseq	*msfree;
extern struct clist ms_block,ms_uart;
int msdebug;
int msdebug2;
int msdone();
char ms_reset_ack[] = {0xff, 0x08, 0x00, 0x00};

extern int cons_if;
struct tty ms_tty[NUMBER_CONS];

#define MS_UNIT(dev) (minor(dev) & 0x0f)
#define MS_LINE(dev) (minor(dev) >> 4)


/*ARGSUSED*/

msopen(dev, flag)
	dev_t dev;
{
	register struct ms_softc *mssc = &ms;
	register struct tty *tp;
	register int unit = MS_UNIT(dev);
	register int s;
	register int error;
	int generic = 0;
	int ldisc = MS_LINE(dev);

	DEBUGF(msdebug & SHOW_OPEN, printf("In MSOPEN unit=%x\n", unit));

	/* If Generic console minor device then assign this to input focus */
	if (unit == CONS_GEN) {
		unit = cons_if;
		generic++;
	}

	tp = &ms_tty[unit];
	tp->t_oproc = 0;
	if ((tp->t_state & TS_ISOPEN) == 0) {
		ttychars(tp);
		tp->t_ospeed = tp->t_ispeed = ISPEED;
		tp->t_state = TS_ISOPEN | TS_CARR_ON;
		tp->t_flags = ODDP;
		if (mssc->ms_cnt <= 0) {
			s = KLSSPL();
			msparam(unit);
			if (reset_mouse()) {
				tp->t_state = 0;
				splx(s);
				return (EIO);
			}
			splx(s);
		}
		mssc->ms_cnt++;
	} else if (generic)
		return (EBUSY);

	DEBUGF(msdebug & SHOW_OPEN, printf("MSOPEN: call ttyopen (%x)\n",
			linesw[tp->t_line].l_open));
	if (error = (*linesw[tp->t_line].l_open)(unit, tp)) {
		msclose(unit);
		return (error);
	}
	if (tp->t_state & TS_XCLUDE && u.u_uid != 0)
		return (EBUSY);
	if ((ldisc) && (ldisc != tp->t_line))
		return (msioctl(dev, TIOCSETD, (caddr_t) & ldisc, 0));
	return (0);
}

/*ARGSUSED*/
msclose(dev)
	dev_t dev;
{
	register int unit = MS_UNIT(dev);
	register struct ms_softc *mssc = &ms;
	register struct tty *tp;
	register int s;

	DEBUGF(msdebug & SHOW_OPEN, printf("In MSCLOSE\n"));

	/* If Generic console minor device then assign this to input focus */
	if (unit == CONS_GEN)
		unit = cons_if;

	tp = &ms_tty[unit];

	if (--mssc->ms_cnt <= 0) {
		s = KLSSPL();
		ms_cmd(UARTCNT,MSDIS);
		kls_flush(ms_block);
		kls_flush(ms_uart);
		mssc->ms_flags = 0;
		mssc->ms_rate = MS_RATE_100;
		mssc->ms_resl = MS_RES_100;
		mssc->msenabled = 0;
		splx(s);
	}
	(*linesw[tp->t_line].l_close)(tp);
	ttyclose(tp);
	tp->t_state = 0;
	DEBUGF(ttydebug & SHOW_OPEN, printf("MSCLOSE end\n"));
}

/*ARGSUSED*/
msread(dev, uio)
	register dev_t dev;
	register struct uio *uio;
{
	register unit = MS_UNIT(dev);
	register struct tty *tp;

	DEBUGF(msdebug & SHOW_RDWR, printf("In MSREAD\n"));

	/* If Generic console minor device then assign this to input focus */
	if (unit == CONS_GEN)
		unit = cons_if;
	tp = &ms_tty[unit];

	return ((*linesw[tp->t_line].l_read)(tp, uio));
}


/*
 * The Mouse wants to give us a character.
 * Catch the character, and see who it goes to.
 */
/*ARGSUSED*/
msrint()
{
	register int c;
	register struct ms_softc *mssc = &ms;
	register struct tty *tp = &ms_tty[cons_if];

	DEBUGF(msdebug2,printf("In msrint, flags = %x\n",ms.ms_flags););
	/* if MS_QUERY is set someone wants to get at the mouse directly */
	if(mssc->ms_flags & MS_QUERY) {
		mssc->ms_flags &= ~MS_QUERY;
		wakeup( (caddr_t) &mssc->ms_flags);
		DEBUGF(msdebug2,printf("wakeup called\n"););
		return;
	}
	/* if MS_BUSY then someone is still using the input */
	if (mssc->ms_flags & MS_BUSY)
		return;
	/*
	 * Otherwise we aren't interested in ms_block, just get the data and 
	 * send it to the line disipline.
  	 */
	kls_flush(ms_block);
	/* If the mouse is closed, no one is interested */
	if((tp->t_state & TS_ISOPEN) == 0) {
		kls_flush(ms_uart);
		return;
	}
	while((c=kls_read(ms_uart)) != -1) 
		(*linesw[tp->t_line].l_rint)(c,tp);
}

/*ARGSUSED*/
msioctl(dev, cmd, addr, flag)
	dev_t dev;
	caddr_t addr;
{
	register struct ms_softc *mssc = &ms;
	register struct tty *tp; 
	register int unit = MS_UNIT(dev);
	register int error = 0;
	register int *mssemaphore = &ms.ms_sem;
	int	s;

	s = KLSSPL();

	if (unit == CONS_GEN)
		unit = cons_if;

	tp = &ms_tty[unit];

	DEBUGF(msdebug,printf("in MSIOCTL(dev=%x, cmd=%x, addr=%x, flag=%x)\n",
		dev, cmd, addr, flag));
	switch (cmd) {
	case MSIC_STREAM:
		/* Set To STREAM mode (Two byte command) */
		ms_cmd(UARTCNT, MSSTREAM);
		mssc->ms_flags &= ~MS_NOSTREAM;
		break;
	case MSIC_REMOTE:
		/* Set To REMOTE mode (Two byte command) */
		ms_cmd(UARTCNT, MSREMOTE);
		mssc->ms_flags |= MS_NOSTREAM;
		break;
	case MSIC_STATUS:
		/* Give the user the current raw status of the mouse */
		msstatus(addr);
		break;
	case MSIC_READXY:
		ms_swait(mssemaphore);

		/* Force a read request to the mouse and go get the data */
		if (mssc->msenabled)
			ms_cmd(UARTCNT, MSDIS);
		kls_flush(ms_block);
		kls_flush(ms_uart);
		ms_cmd(UARTCMD, MSRDDATA);
		msrint();
		if (mssc->msenabled)
			ms_cmd(UARTCNT, MSTRANS);
		ms_ssignal(mssemaphore);
		break;
	case MSIC_ENABLE:
		/* Disable the mouse from sending data in stream mode */
		ms_cmd(UARTCNT, MSTRANS);
		mssc->msenabled = 1;
		break;
	case MSIC_DISABLE:
		/* Disable the mouse from sending data in stream mode */
		ms_swait(mssemaphore);
		ms_cmd(UARTCNT, MSDIS);
		mssc->msenabled = 0;
		kls_flush(ms_block);
		kls_flush(ms_uart);
		ms_ssignal(mssemaphore);
		break;
	case MSIC_EXP:
		/* Turn on Exponential scalling on the mouse */
		ms_cmd(UARTCNT, MSEXPON);
		mssc->ms_flags |= MS_EXP;
		break;
	case MSIC_LINEAR:
		/* Turn on Linear scalling on the mouse */
		ms_cmd(UARTCNT, MSEXPOFF);
		mssc->ms_flags &= ~MS_EXP;
		break;
	case MSIC_SAMP:
		/* Set the sampling rate used for the mouse */
		DEBUGF(msdebug,
			printf("Set samp (%x)\n", MSSETSAM | *(long *)addr);
		);
		ms_cmd(UARTCNT, MSSETSAM | *(long *)addr);
		mssc->ms_rate = *(long *)addr;
		break;
	case MSIC_RESL:
		/* Set the resolution the mouse uses */
		DEBUGF(msdebug,
			printf("Set resl (%x)\n", MSSETRES | *(long *)addr);
		);
		ms_cmd(UARTCNT, MSSETRES | *(long *)addr);
		mssc->ms_resl = *(long *)addr;
		break;
	default:
		/* TTY ioctls */
		splx(s);
		error = (*linesw[tp->t_line].l_ioctl)(tp, cmd, addr);
		if (error >= 0)
			return (error);
		error = ttioctl(tp, cmd, addr, flag);
		if (error >= 0) {
			if (cmd == TIOCSETP || cmd == TIOCSETN) {
				s = KLSSPL();
				msparam(unit);
				splx(s);
			}
			return (error);
		}
		if (error < 0)
			error = ENOTTY;
	}
	splx(s);
	return (error);
}

msselect (dev, rw)
	dev_t dev;
	int rw;
{
	register int unit = MS_UNIT(dev);
	register struct tty *tp;

	/* If Generic console minor device then assign this to input focus */
	if (unit == CONS_GEN)
		/* Get display number for current input focus */
		unit = cons_if;

	/* Rebuild dev with new minor device number */
	dev = (dev & 0xff00) | unit;
	tp = &ms_tty[unit];

	return ((*linesw[tp->t_line].l_select)(dev, rw));
}

msparam(unit)
	register int unit;
{
	register int x;
	register struct tty *tp = &ms_tty[unit];

	DEBUGF(msdebug & SHOW_IO, {
		printf("In MSPARAM\n");
		printf("tp->t_flags = (%x)\n", tp->t_flags);
	}
	);

	/* Enable UART interface with a clear */
	x = ms_cmd(EXTCMD, ENUART);
	if (x != 0x0000) {
		printf("msparam: uart did not enable (%x)\n", x);
	}
	/* Set the baud rate and initial mouse settings */
	DEBUGF(msdebug & SHOW_IO, printf("MSPARAM: Setting baud (%d)\n",
		MSBAUD(tp->t_ispeed));
	);
	x = ms_cmd(MSBAUDCMD, MSBAUD(tp->t_ispeed));
	if (x != 0x0000) {
		printf("msparam: uart didn't Set baud rate (%x)\n", x);
	}
	/* Set the baud rate and initial mouse settings */
	x = ms_cmd(MSFRAME,(tp->t_flags & ODDP) ? UART_FRM_ODDP : UART_FRM_EVENP);
	if (x != 0x0000) {
		printf("msparam: uart didn't init framing (%x)\n", x);
	}
	DEBUGF(msdebug & SHOW_IO, printf(" msparam end\n"));
}

msstatus(arg)
	register char *arg;
{
	register int x,i;
	register int *mssemaphore = &ms.ms_sem;
	int	s;

	ms_swait(mssemaphore);
	if (ms.msenabled)
		ms_cmd(UARTCNT, MSDIS);

	kls_flush(ms_uart);
	kls_flush(ms_block);
	s=KLSSPL();
	ms.ms_flags |= (MS_QUERY | MS_BUSY);
	ms_cmd(UARTCMD, QYMOUSE);
	while (ms.ms_flags & MS_QUERY)
		sleep((caddr_t) &ms.ms_flags,MSPRI);
	if ((x=kls_read(ms_block)) != 4)
		printf("msstatus: Bad block count %d\n",x);
	for (i=0; (i < 4) && ((x = kls_read(ms_uart)) >= 0); i++)
		*arg++ = (char) x;
	if (i != 4)
		printf("msstatus: Not all status read\n");
	ms.ms_flags &= ~MS_BUSY;
	if (ms.msenabled)
		ms_cmd(UARTCNT, MSTRANS);
	ms_ssignal(mssemaphore);
	splx(s);
}

reset_mouse()
{
	register int i;
	register int retry = 0;
	register char x;
	register int *mssemaphore = &ms.ms_sem;
	int s;

	/* Enable Blocking mode on UART */
	if (ms_cmd(EXTCMD, SETMSBLK)) {
		printf("reset_mouse: uart didn't Set block mode\n");
		return (1);
	}
	ms_swait(mssemaphore);
	/* Reset the MOUSE */
	do {
		kls_flush(ms_uart);
		kls_flush(ms_block);
		s=KLSSPL();
		ms.ms_flags |= (MS_QUERY | MS_BUSY);
		if (ms_cmd(UARTCMD, RSMOUSE)) {
			printf("reset_mouse: mouse didn't accept reset\n");
			splx(s);
			continue;
		}
		while (ms.ms_flags & MS_QUERY)
			sleep( (caddr_t) &ms.ms_flags,MSPRI);
		splx(s);
		if ((x=kls_read(ms_block)) != 4) {
			printf("reset_mouse: Bad block count = %d\n",x);
			ms.ms_flags &= ~MS_BUSY;
			continue;
		}
		for (i = 0; i < 4; i++)
			if ((x = kls_read(ms_uart) != ms_reset_ack[i]))
				break;
		ms.ms_flags &= ~MS_BUSY;
		if (i == 4)
			break;
	} while (retry++ < MS_MAX_RETRY);

	if (retry >= MS_MAX_RETRY) {
		printf("reset_mouse: Mouse didn't reset (%x)\n", x);
		ms_ssignal(mssemaphore);
		return (1);
	}
	/* Disable Blocking mode on UART */
	if (ms_cmd(EXTCMD, CLRMSBLK)) {
		printf("reset_mouse: uart didn't Reset block mode\n");
		ms_ssignal(mssemaphore);
		return (1);
	}
	/* Read Configuration */
	s=KLSSPL();
	ms.ms_flags |= (MS_QUERY | MS_BUSY);
	if (ms_cmd(UARTCMD, MSRDCONF)) {
		printf("reset_mouse: mouse didn't accept read config\n");
		ms_ssignal(mssemaphore);
		return (1);
	}
	while (ms.ms_flags & MS_QUERY)
		sleep( (caddr_t) &ms.ms_flags,MSPRI);
	splx(s);

	if ((x = kls_read(ms_uart)) != MS_CONFIGURED) {
		printf("reset_mouse: Wrong configuration resp (%x)\n", x);
		ms.ms_flags &= ~MS_BUSY;
		ms_ssignal(mssemaphore);
		return (1);
	}
	ms.ms_flags &= ~MS_BUSY;

	/* Enable Blocking mode on UART */
	if (ms_cmd(EXTCMD, SETMSBLK)) {
		printf("reset_mouse: uart didn't Set block mode\n");
		ms_ssignal(mssemaphore);
		return (1);
	}
	/* Enable data transmissions */
	if (ms_cmd(UARTCNT, MSTRANS)) {
		printf("reset_mouse: mouse didn't accept enable cmd\n");
		ms_ssignal(mssemaphore);
		return (1);
	}
	ms.msenabled = 1;
	ms_ssignal(mssemaphore);

	return (0);

}

ms_cmd(dest,cmd)
{
	register struct	mouseq	 *qp;
	register int	s;
	int	 ret;

	s=KLSSPL();
	DEBUGF(msdebug2,printf("In mscmd, dest=0x%x, cmd=0x%x\n",dest,cmd););
	while( !(qp = (struct mouseq *) klsalloc(&msfree)) )
		sleep( (caddr_t) &msfree, MSPRI);
	qp->ms_count = 0;
	qp->cmd1.qp_ret = KLSINVRET;
	DEBUGF(msdebug,printf("Got queue\n"););
	if (cmd & 0xff00) {
		qp->cmd1.qp_dest = dest;
		qp->cmd1.qp_cmd = cmd >> 8;
		qp->cmd1.qp_callback = msdone;
		qp->ms_count = 1;
	}
	qp->cmd2.qp_dest = dest;
	qp->cmd2.qp_cmd = cmd & 0xff;
	qp->cmd2.qp_callback = msdone;
	qp->cmd2.qp_ret = KLSINVRET;
	msstrategy(qp);
	while ((qp->cmd2.qp_ret == KLSINVRET) || (qp->cmd2.qp_ret == MS_BUSY_REJ))
		sleep( (caddr_t) qp,MSPRI);
	ret=qp->cmd2.qp_ret;
	klsfree(msfree,qp);
	wakeup( (caddr_t) &msfree);
	splx(s);
	return(ret);
}

msstrategy(qp)
	register struct	mouseq	*qp;
{
	register struct ms_softc *mssc = &ms;

	qp->qp_next = NULL;
	if (mssc->ms_head == NULL) {
		mssc->ms_head = qp;
		mssc->ms_tail = qp;
		msstart();
	} else {
		mssc->ms_tail->qp_next=qp;
		mssc->ms_tail=qp;
	}
}

msdone(qp)
	register struct klsq *qp;
{
	register struct ms_softc *mssc= &ms;
	register struct mouseq *current;

	DEBUGF(msdebug,printf("In mssdone, cmd=0x%x, data=0x%x, ret=0x%x\n",
		qp->qp_dest,qp->qp_cmd,qp->qp_ret););

	if ((current=mssc->ms_head) == NULL) {
		panic("msdone.");
	}
	/*
	 * User mode machine check encountered. Reset the current command
	 * to the nearest full command and let kbdreset restart
	 */
	if ((qp->qp_ret) == KLS_SOFT_ERROR) {
		if ((current->ms_count == 0) && 
			(current->cmd1.qp_ret != KLSINVRET))
			current->ms_count = 1;
		return;
	}

	if (qp->qp_ret == MS_BUSY_REJ) {
		DEBUGF(msdebug2,printf("msbusy\n"););
		msstart();
		return;
	} 

#if	ROMP_DEBUG
	if (qp->qp_ret)
		printf("mssdone: 0x%x from dest 0x%x, cmd 0x%x.\n",
			qp->qp_ret,qp->qp_dest,qp->qp_cmd);
#endif 	ROMP_DEBUG

	/* find next command */
	if ((--current->ms_count) < 0) {
		mssc->ms_head = current->qp_next;
		wakeup( (caddr_t) current);
		if (mssc->ms_head != NULL)
			msstart();
	} else 
		msstart();
}

msstart()
{
	register struct mouseq *mshead;

	if ((mshead = ms.ms_head) == NULL)
		return;
	if (mshead->ms_count == 1)
		klsstrategy(&mshead->cmd1);
	else if (mshead->ms_count == 0)
		klsstrategy(&mshead->cmd2);
	else
		panic("msstart");
}
	
		
msreset()
{
	register int x;
	register struct ms_softc	*mssc = &ms;
	register struct tty *tp = &ms_tty[cons_if];
	register int i;
	register int retry = 0;
	int iid;

	if (mssc->ms_cnt > 0) {
		/* Enable UART interface with a clear */
		if (x=kls_raw_cmd(EXTCMD, ENUART))
			printf("msreset: uart did not enable (%x)\n", x);
		/* Set baud rate and parity */
		if (x=kls_raw_cmd(MSBAUDCMD, MSBAUD(tp->t_ispeed)))
			printf("msreset: uart didn't Set baud rate (%x)\n", x);
		if (x=kls_raw_cmd(MSFRAME,(tp->t_flags & ODDP) ? UART_FRM_ODDP : UART_FRM_EVENP))
			printf("msreset: uart didn't init framing (%x)\n", x);

		/* Enable Blocking mode on UART */
		if (kls_raw_cmd(EXTCMD, SETMSBLK))
			printf("msreset: uart didn't Set block mode\n");
		/* Reset the MOUSE */
		do {
			kls_pflush(10);
			if (kls_raw_cmd(UARTCMD, RSMOUSE)) {
				printf("msreset: mouse didn't accept reset\n");
				continue;
			}
			if ((x=kls_raw_read(&iid)) != 4) {
				printf("msreset: Bad block count = %d\n",x);
				continue;
			}
			for (i = 0; i < 4; i++)
		 		if ((x = kls_raw_read(&iid) != ms_reset_ack[i]))
					break;
			if (i == 4)
				break;
		} while (retry++ < MS_MAX_RETRY);

		if (retry >= MS_MAX_RETRY)
			printf("msreset: Mouse didn't reset (%x)\n", x);
		/* Disable Blocking mode on UART */
		if (kls_raw_cmd(EXTCMD, CLRMSBLK))
			printf("msreset: uart didn't Reset block mode\n");
		/* Read Configuration */
		if (kls_raw_cmd(UARTCMD, MSRDCONF))
			printf("msreset: mouse didn't accept read config\n");
		if ((x = kls_raw_read(&iid)) != MS_CONFIGURED)
			printf("msreset: Wrong configuration resp (%x)\n", x);
		/* Enable Blocking mode on UART */
		if (kls_raw_cmd(EXTCMD, SETMSBLK))
			printf("msreset: uart didn't Set block mode\n");

		if (mssc->ms_flags & MS_NOSTREAM)
			if (kls_raw_cmd(UARTCNT,MSREMOTE))
				printf("msreset: couldn't set remote mode\n");
		if (mssc->ms_flags & MS_EXP)
			if (kls_raw_cmd(UARTCNT,MSEXPON))
				printf("msreset: couldn't set exponential mode\n");
		if (mssc->ms_rate != MS_RATE_100) {
			if ((x=kls_raw_cmd(UARTCNT,MSSETSAM >> 8)) == 0) 
				while((x=kls_raw_cmd(UARTCNT,mssc->ms_rate)) == MS_BUSY_REJ)
					/* void */;
			if (x)
				printf("msreset: couldn't set sampling rate\n");
		}
		if (mssc->ms_resl != MS_RES_100){
			if ((x=kls_raw_cmd(UARTCNT,MSSETRES >> 8)) == 0) 
				while((x=kls_raw_cmd(UARTCNT,mssc->ms_resl)) == MS_BUSY_REJ)
					/* void */;
			if (x)
				printf("msreset: couldn't set resolution\n");
		}
		
	}
	if (mssc->msenabled)
		klscmd(UARTCNT,MSTRANS,0);
	if (mssc->ms_head != NULL)
		msstart();
}

msinit()
{
	register struct ms_softc	*mssc = &ms;

	msfree = (struct mouseq *) klsminit(mscmdq,sizeof(struct mouseq),MSPOOLSZ);
	mssc->ms_sem = 0;
	mssc->ms_flags = 0;
	mssc->ms_rate = MS_RATE_100;
	mssc->ms_resl = MS_RES_100;
	
}

ms_swait(semaphore)
	int	*semaphore;
{
	int	s=KLSSPL();

	while ((*semaphore) < 0)
		sleep( (caddr_t) semaphore,MSPRI);
	(*semaphore)--;
	splx(s);
}

ms_ssignal(semaphore)
	int	*semaphore;
{
	int	s=KLSSPL();

	(*semaphore)++;
	wakeup( (caddr_t) semaphore);
	splx(s);
}

mswait()
{
	while (ms.ms_head != NULL)
		;
}

#endif NMS
