/*
 *  Mach B008 driver for the IBM RT
 *
 *  Copyright 1989 Cornell University
 *
 *  %W% (Cornell Theory Center) %G%
 *
 *  09/05/89 : Added volatile defs and fixed bugs for Mach 2.4 (MRM)
 *  06/27/89 : Added VICIC_ANLYZ to assert/deassert Analyse. (MRM)
 *  06/26/89 : Cleaned up code a little. (MRM)
 *  01/18/89 : Added code to handle DMA.  Doesn't work yet. (MRM)
 *  12/22/88 : Added read/write code to do polled/interrupt I/O.  Also
 *	       changed debugging code & added several ioctl's. (MRM)
 *  12/18/88 : Added code to handle vicc controller, in prep. for dma
 *             Note that there is only one vic per vicc (MRM)
 *  12/14/88 : added VICIC_RESET ioctl and functional vicreset() code. (MRM)
 *  12/13/88 : created (MRM)
 */

#ifdef  hc
pragma  off (optimize);
#else
#define volatile
#endif  hc


#include "vic.h"
#if NVIC > 0

/* #include "stdio.h" */
#include "../h/param.h"
#include "../h/errno.h"
#include "../h/uio.h"
#include "../h/ioctl.h"
#include "../ca/io.h"
#include "../h/buf.h"
/* #include "apdefs.h" */
#include "../caio/ioccvar.h"
#include "../caio/dmavar.h"
#include "../caio/dmareg.h"
#include "../caio/vicio.h"
#include "../caio/vicreg.h"

/* "helpers"... */
static int t4_bcmnd();
static long vicout(), vicin(), vicdma();

/* version ID */
static char vic_version[] = "2.3";

int vicprobe(), vicattach();
int vicintr(), vicstrategy(), vicminphys();
int vichalttimeout();
int vicslave(), vicdgo(), vicchanrelse();

caddr_t vicstd[] = {	/* standard locations */
	(caddr_t)0xf0000300,
	(caddr_t)0xf0000200,
	(caddr_t)0xf0000150,
	0
};

struct iocc_device *vicdinfo[NVIC];
struct iocc_ctlr *vicminfo[NVIC];

/* fake iocc_ctlr structures for non-controller DMA */
struct iocc_ctlr viccic[NVIC];

struct iocc_driver vicdriver =
{
    vicprobe, vicslave, vicattach, vicdgo,
/*  addrs   dname  dinfo     mname   minfo     intr     csr */
    vicstd, "vic", vicdinfo, "vicc", vicminfo, vicintr, 2,
/*  dmachanrelse */
    vicchanrelse,
/*  flags */
    0
};

int vicdebug = 0x0049;

#define VICUNIT(dev)		(minor(dev) >> 3)
#define VICUDMA(dev)		((minor(dev) >> 2) & 1)
#define VICDEBUG(how,stmt)	if (vicdebug & (how)) stmt
#define VICODBG 	0x0001	/* open/close debugging 	*/
#define VICCDBG 	0x0002	/* config debugging statements 	*/
#define VICIODBG 	0x0004	/* ioctl debugging	 	*/
#define VICDDBG 	0x0008	/* dma debugging		*/
#define VICRWDBG 	0x0010	/* read/write debugging 	*/
#define VICSDBG 	0x0020	/* strategy debugging 		*/
#define VICIDBG 	0x0040	/* interrupt debugging 		*/

#define VICPRI		(PZERO+5)

#define VIC_DMA_CHANNEL	DMA_CHAN3	/* which DMA channel the devices use */
int vic_dma_channel = VIC_DMA_CHANNEL;

struct buf vicrbuf[NVIC];	/* buffers for read */
struct buf vicwbuf[NVIC];	/* buffers for write */

struct vic_softc {
	int sc_state;		/* state; see bits below		*/
	int sc_opencnt;		/* number of times opened		*/
	int sc_r_wait;		/* polled I/O wait channel		*/
	int sc_w_wait;		/* polled I/O wait channel		*/
	int sc_dma_wait;	/* DMA wait channel			*/
} vic_softc[NVIC];

/* bits for state */
#define VIC_OPEN	0x0001		/* Device has been opened */
#define VIC_RESET	0x0002		/* Reset bit; if set bail out */
#define VIC_DODMA	0x0004		/* Doing DMA if 1 else polled*/
#define VIC_ACTIVE	0x0008		/* Polled I/O in progress */
#define VIC_DMAACTIVE	0x0010		/* DMA in progress	  */
#define VIC_RASLEEP	0x0020		/* Waiting on polled I/O read */
#define VIC_WASLEEP	0x0040		/* Waiting on polled I/O write */
#define VIC_DMAASLEEP	0x0080		/* Waiting on DMA	  */

/* Try to make the B008 interrupt.  We do this by resetting it and enabling
 * interrupts on "ready to receive".
 */
vicprobe(reg, ic)
  register caddr_t reg;
  register struct iocc_ctlr *ic;
{
	register struct vicdevice volatile *vicaddr = 
				(struct vicdevice volatile *)reg;
	int i;

	VICDEBUG(VICCDBG,printf("vic: vicprobe(%x)\n", reg));

	vicreset(vicaddr);			/* reset the B008         */
	for (i=0; i<NVIC; i++)
		vic_softc[i].sc_state = 0;	/* initialize */
	/* .1s is probably a little long, but it's nice to be safe */
	DELAY(100000);				/* wait for reset	  */
	IOOUT(&vicaddr->int_ctl, VICINT_OUT);	/* enable RTR intrupt     */
	if ((IOIN(&vicaddr->out_stat) && VICST_IORDY) == 0) {
		printf("vic: device at %x not ready\n",vicaddr);
	}
		
	IOOUT(&vicaddr->out_stat, VICST_INTENA); /* turn on intrpt	  */
	/* B008 should interrupt here */
	PROBE_DELAY(10000);			/* allow time for intr    */
	IOOUT(&vicaddr->int_ctl, 0);		/* turn off interrupts    */
	IOOUT(&vicaddr->out_stat, 0);		/*   "			  */
	return(PROBE_OK);
#if 0
	return(PROBE_NOINT);			/* no interrupts          */
#endif
}

/* Routine to determine whether a slave exists.  We only have
 * controller support in the driver at all because we want to do dma;
 * there can only be one slave per controller since they are really
 * one and the same.
 *
 * Note that with the new code to fake a controller, this never gets
 * called.
 *
 * In general, a slave routine returns 1 if the slave exists, 0 if
 * it does not.
 */
vicslave(iod, vicaddr)
	register struct iocc_device *iod;
	register struct vicdevice volatile *vicaddr;
{
	register int slave = iod->iod_slave;

	VICDEBUG(VICCDBG,printf("vic: vicslave dev=%x slave=%x\n",vicaddr,slave));
	if (slave > 0)
		return 0;
	else
		return 1;
}

vicattach(iod)
	register struct iocc_device *iod;
{
	register struct vicdevice volatile *vicaddr;
	register int unit = (int)(iod->iod_unit);
	register struct iocc_ctlr *ic = &viccic[unit];

	vicaddr = (struct vicdevice volatile *)iod->iod_addr;
	VICDEBUG(VICCDBG,printf("vic: vicattach addr=%x\n",vicaddr));

	vic_softc[unit].sc_opencnt = 0;

	/* initialize fake iocc_ctlr structure.  This is because only
	 * controllers are supposed to do dma; we make structures so
	 * we look like a controller to the dma routines
	 */ 
	ic->ic_driver = &vicdriver; 
	ic->ic_ctlr = unit; 
	ic->ic_alive = 0;
	ic->ic_addr = iod->iod_addr; 
	ic->ic_irq = iod->iod_irq;
	
	printf("vic%d: version %s\n",unit,vic_version);
	vicreset(vicaddr);
	return(1);
}


vicopen(dev, flag)
	dev_t dev;
	int flag;
{
	register int unit = VICUNIT(dev);
	register struct vic_softc *sc = &vic_softc[unit];
	register struct iocc_device *iod = vicdinfo[unit];

	VICDEBUG(VICODBG,printf("vic%1d: vicopen\n",unit));
	if (unit >= NVIC || iod == 0 || iod->iod_alive == 0) {
		return (ENODEV);
	}
#ifdef VICEXCLIO
	if ((!suser()) && (sc->sc_state & VIC_OPEN)) { 
		printf("vic: B008 %d already open\n", unit);
		return(EBUSY);
	}
#endif
	if (!(sc->sc_state & VIC_OPEN)) {
		vicrbuf[unit].b_flags = 0;
		vicwbuf[unit].b_flags = 0;
	};
	sc->sc_state |= VIC_OPEN;
	sc->sc_opencnt++;
	return(0);
}


vicclose(dev, flag)
	dev_t dev;
	int flag;
{
	register int unit = VICUNIT(dev);
	register struct vic_softc *sc = &vic_softc[unit];
	struct iocc_ctlr *ic = &viccic[unit];
	register int s;

	VICDEBUG(VICODBG,printf("vic%1d: vicclose oc=%d\n",unit,sc->sc_opencnt));

	s = VIC_SPL();				/* no funny biz */


#ifdef VICCLDI
	IOOUT(&vicaddr->int_ctl, 0);
	IOOUT(&vicaddr->in_stat, 0);		/* disable interrupts */
	IOOUT(&vicaddr->out_stat, 0);	
#endif

	/* clean up buffers */
	viccleanupbuffer(&vicrbuf[unit]);
	viccleanupbuffer(&vicwbuf[unit]);

	/* release dma channel if it's still being held */
	if (sc->sc_state & VIC_DODMA) {
		dma_done(ic->ic_dmachannel);
		VICDEBUG(VICODBG,printf("vic%1d: vicclose dma active\n",unit));
	}

	/* we have to clear all of the bits in case something
	 * was interrupted and left things in a strange state.
	 */
	sc->sc_state = 0;
	sc->sc_opencnt = 0;

	splx(s); 
	return(0);
}

/*
** we will let physio do all the hard work for us on read
** and write...we can take care of raw vs. DMA type stuff
** in the strategy routines.
*/
vicread(dev, uio)
	register dev_t dev;
	register struct uio *uio;
{
	register int unit = VICUNIT(dev);

	VICDEBUG(VICRWDBG,printf("vic%d: vicread\n",unit));
	return(physio(vicstrategy, &vicrbuf[unit], dev, B_READ,
		vicminphys, uio));
}


vicwrite(dev, uio)
	register dev_t dev;
	register struct uio *uio;
{
	register int unit = VICUNIT(dev);

	VICDEBUG(VICRWDBG,printf("vic%d: vicwrite\n",unit));
	return(physio(vicstrategy, &vicwbuf[unit], dev, B_WRITE,
		vicminphys, uio));
}

/* vic ioctl routine.  See also vicio.h */
vicioctl(dev, cmd, addr, flag)
	dev_t dev;
	int cmd,flag;
	caddr_t addr;
{
	register int unit = VICUNIT(dev);
	register struct vicdevice volatile *vicaddr;
	register struct iocc_device *iod = vicdinfo[unit];
	int s,*iparam;
	u_char uc;
	struct viciio *vparam;

	VICDEBUG(VICIODBG,printf("vic%1d: vicioctl\n",unit));
	if (unit >= NVIC) 
		return (ENODEV);
	vicaddr = (struct vicdevice volatile *)iod->iod_addr;
	
	s = VIC_SPL();
	switch (cmd) {
	case VICIC_RDREG:		/* read device register */
		if (!suser()) { splx(s); return(EPERM); }
		vparam = (struct viciio *)addr;
		vparam->vi_vreg &= 0x1f;	/* only use 5 bits for register */
		VICDEBUG(VICIODBG,printf("vic%1d: register 0x%x\n",unit,(int)vparam->vi_vreg));
		vparam->vi_rgval = IOIN((&vicaddr->in_data)+
					vparam->vi_vreg);
		break;
	case VICIC_WRREG:		/* write device register */
		if (!suser()) { (void)splx(s); return(EPERM); }
		vparam = (struct viciio *)addr;
		vparam->vi_vreg &= 0x1f; /* only use 5 bits for register */
		VICDEBUG(VICIODBG,printf("vic%1d: register 0x%x\n",unit,(int)vparam->vi_vreg));
		IOOUT(((&vicaddr->in_data)+vparam->vi_vreg), vparam->vi_rgval);
		break;
	case VICIC_RESET:		/* initiate T800 reset */
		vicreset(vicaddr);
		break;
	case VICIC_INTRPT:		/* make device interrupt */
		/* note this only works immed. after a VICIC_RESET!! */
		if (!suser()) { (void)splx(s); return(EPERM); }
		IOOUT(&vicaddr->int_ctl, (VICINT_IN | VICINT_OUT));
		IOOUT(&vicaddr->out_stat, VICST_INTENA);/* intr on */
		break;
	case VICIC_ISTAT:		/* read C012 input status register */
		iparam = (int *)addr;	
		*iparam = (int)((u_char)IOIN(&vicaddr->in_stat));
		break;
	case VICIC_OSTAT:		/* read C012 output status register*/
		iparam = (int *)addr;	
		*iparam = (int)((u_char)IOIN(&vicaddr->out_stat));
		break;
	case VICIC_BERROR:		/* read B008 error register */
		iparam = (int *)addr;
		*iparam = (int)vicaddr->rsterr;
		break;
	case VICIC_DEBUG:		/* set new debugging value */
		iparam = (int *)addr;
		vicdebug = *iparam & 0xFFFF;
		break;
	case VICIC_FLUSH:		/* write device register */
		if (!suser()) { (void)splx(s); return(EPERM); }
		vicflush(unit,VICUDMA(dev));
		break;
	case VICIC_ANLYZ:		/* assert(1)/deassert(0) Analyse */
		iparam = (int *)addr;
		uc = (*iparam != 0);
		IOOUT(&vicaddr->analyse, uc);
		break;
	default:
		(void) splx(s);
		VICDEBUG(VICIODBG,printf("vic%d: bad ioctl %x\n",unit,cmd));
		return(ENOTTY);
	}
	(void) splx(s);
	return(0);
}

/* vicintr: interrupt handler */
vicintr(unit, irq)
	int unit, irq;
{
	register struct vicdevice volatile *vicaddr;
	register struct iocc_device *iod = vicdinfo[unit];
	register struct vic_softc *sc = &vic_softc[unit];
	
	VICDEBUG(VICIDBG,printf("vic: vicintr %d\n",irq));
	if (iod == NULL)
		panic("vicintr iod");
	vicaddr = (struct vicdevice volatile *)iod->iod_addr;
	if (vicaddr == NULL)
		panic("vicintr vicaddr");
	if ((irq != 3) && (irq != 5)) {
		printf("vic%d: unknown interrupt %d\n",unit,irq);
		return(INT_NOT_MINE);
	}

	if (sc->sc_state & VIC_DMAACTIVE) {
		/* DMA is active; assume interrupt is from DMA */
		IOOUT(&vicaddr->int_ctl, 0);	/* interrupts off */
		if (sc->sc_state & VIC_DMAASLEEP) {
			sc->sc_state &= ~VIC_DMAASLEEP;
			wakeup((caddr_t)&sc->sc_dma_wait);
		} else {
			printf("vic: spurious int %d\n",irq);
			return(INT_NOT_MINE);
		}
		sc->sc_state &= ~VIC_DMAACTIVE;
	} else {
		/* must be polled I/O */
		if ((IOIN(&vicaddr->in_stat) & (VICST_IORDY | VICST_INTENA))
		    == (VICST_IORDY | VICST_INTENA)) {
			if (sc->sc_state & VIC_RASLEEP) {
				sc->sc_state &= ~VIC_RASLEEP;
				if (sc->sc_state & VIC_WASLEEP) 
					IOOUT(&vicaddr->int_ctl, VICINT_OUT);
				else
					IOOUT(&vicaddr->int_ctl, 0);
				IOOUT(&vicaddr->in_stat, 0);		/* ints off */
				wakeup((caddr_t)&sc->sc_r_wait);
			} else {
				printf("vic: intr: rd int with no wait\n");
			}			
		} else
		if ((IOIN(&vicaddr->out_stat) & (VICST_IORDY | VICST_INTENA))
		    == (VICST_IORDY | VICST_INTENA)) {
			if (sc->sc_state & VIC_WASLEEP) {
				sc->sc_state &= ~VIC_WASLEEP;
				if (sc->sc_state & VIC_RASLEEP) 
					IOOUT(&vicaddr->int_ctl, VICINT_IN);
				else
					IOOUT(&vicaddr->int_ctl, 0);
				IOOUT(&vicaddr->out_stat, 0);		/* ints off */
				wakeup((caddr_t)&sc->sc_w_wait);
			} else {
				printf("vic: intr: wr int with no wait\n");
			}			
		} else {
			printf("vic: intr: spurious int\n");
			return(INT_NOT_MINE);
		}
	}
	return(INT_SERVICED);
}


/* vicreset: initiate a transputer reset.  Note that the reset cycle may
 *		not complete until after this routine exits
 */
vicreset(vicaddr)
  	struct vicdevice volatile *vicaddr;
{
	VICDEBUG(VICCDBG,printf("vic: vicreset\n"));
	IOOUT(&vicaddr->analyse, 0);		/* bring analyse low */
	IOOUT(&vicaddr->rsterr, 1);		/* bring reset high  */
	DELAY(1000);				/* delay 1ms         */
	IOOUT(&vicaddr->rsterr, 0);		/* bring reset low   */
}

vicdgo(ic, len, ioaddr, bp)
	register struct iocc_ctlr	*ic;
	register long			len;
	register u_int			ioaddr;
	register struct buf		*bp;
{
	register struct vic_softc	*sc = &vic_softc[ic->ic_ctlr];
	register struct vicdevice volatile	*vicaddr = 
					(struct vicdevice volatile *)ic->ic_addr;
	/*struct dma_8bit_device *dmaaddr = 
		(struct dma_8bit_device *)DM_CTL1_BASE; */
	int s;

	VICDEBUG(VICDDBG, printf("vic: vicdgo len=%x\n",len));

	/* handle potential error conditions */
	if (len == 0 || ioaddr == DMA_INV_IOADDR) {
		VICDEBUG(VICDDBG, printf("vic: vicdgo error\n"));
		sc->sc_state |= VIC_RESET;
		if (sc->sc_state & VIC_DMAASLEEP)
			wakeup((caddr_t)&sc->sc_dma_wait);
	}

	s = VIC_SPL();
	sc->sc_state |= VIC_DMAACTIVE;
	IOOUT(&vicaddr->int_ctl, VICINT_DMA);	/* turn interrupts on */
/*	dmaaddr->dm_cmd = 0; */
	if (bp->b_flags & B_READ) 		/* initiate operation */
		IOOUT(&vicaddr->dma_req, VICDMA_READ);
	else
		IOOUT(&vicaddr->dma_req, VICDMA_WRITE);
	splx(s);
	
	return(0);
}

/* chanrelse.  Called when dma wants us to release the channel.
 *
 * Returns 0 if channel has been released, non-0 if not.
 */
vicchanrelse(channel)
	short channel;
{
	register int i;
	
	VICDEBUG(VICDDBG, printf("vic: vicchanrelse(%d)\n",(int)channel));

	if (channel != vic_dma_channel) return 1; /* not us! */

	/* check to see if any unit is doing dma; don't give up if so */
	for (i=0; i<NVIC; i++)
		if (vic_softc[i].sc_state & VIC_DODMA) return 0;

	/* we should be OK since noone seems to be doing DMA */
	return 1;	/* release channel */
}

/*
 * Transfer size limit
 */
int vicmaxdma = 8192;

vicminphys(bp)
  struct buf *bp;
{
    if (bp->b_bcount > vicmaxdma)
	bp->b_bcount = vicmaxdma;
}

/* vicstrategy: this gets called from physio to do the actual I/O */
vicstrategy(bp)
	struct buf *bp;
{
	register int unit = VICUNIT(bp->b_dev);
	register struct vicdevice volatile *vicaddr;
	register struct vic_softc *sc = &vic_softc[unit];
	register int dma = VICUDMA(bp->b_dev);
	struct iocc_ctlr *ic = &viccic[unit];
	long resid;
	int error = 0;
	
	vicaddr = (struct vicdevice volatile *)(vicdinfo[unit]->iod_addr);
	VICDEBUG(VICSDBG,printf("vic: vicstrategy %s %x\n",(bp->b_flags&B_READ)? "read" : "write",bp->b_bcount));

	if (dma) {
		/* Use dma.  Call vicdma to do the transfer setup; operation
		 * is complete if vicdma returns 0
		 */
		if (resid = vicdma(ic, bp, sc)) {
			bp->b_flags |= B_ERROR;
			bp->b_resid = resid;
			error = EIO;
		}
	} else {
		/* do it polled.  Enable interrputs and call either
 		 * vicin or vicout.
		 */
		IOOUT(&vicaddr->int_ctl, (VICINT_IN | VICINT_OUT));
		if (resid = 
		    ( (bp->b_flags & B_READ) ?
		      vicin(vicaddr, sc, bp->b_un.b_addr, bp->b_bcount)  :
		      vicout(vicaddr, sc, bp->b_un.b_addr, bp->b_bcount)
		    )
		   )
		{
			bp->b_flags |= B_ERROR;
			bp->b_resid = resid;
			error = EIO;
		}
	} /* if (dma) */
	VICDEBUG(VICSDBG,printf("vic: vicstrategy out, error=%d\n",error));
	bp->b_error = error;
	iodone(bp);
	return;
}


int vic_max_tries = 2;	/* max tries on polling, must be >0 */

/* vicin: gets cnt bytes from B008 into buf via polling.  If the status
 * register shows not ready vic_max_tries times, we enable interrupts
 * and sleep.  Vicintr should wake us up.
 */
static long vicin(vicaddr, sc, buf, cnt)
	register struct vicdevice volatile *vicaddr;
	struct vic_softc *sc;
	register char *buf;
	long cnt;
{
	register int try;
	register int s;	

	VICDEBUG(VICRWDBG,printf("vicin(0x%x): get 0x%x\n",vicaddr,cnt));
	sc->sc_state |= VIC_ACTIVE;	
	try = vic_max_tries;
	while (cnt) {
		if (IOIN(&vicaddr->in_stat) & VICST_IORDY) { 
			*buf++ = IOIN(&vicaddr->in_data);
			cnt--;
			try = vic_max_tries;
		} else if ((--try)==0) {
			VICDEBUG(VICRWDBG,printf("vicin: sleep with 0x%x left\n",cnt));
			s = VIC_SPL();
			IOOUT(&vicaddr->int_ctl, (VICINT_IN | VICINT_OUT));
			IOOUT(&vicaddr->in_stat, VICST_INTENA);	/* intr on */
			sc->sc_state |= VIC_RASLEEP;
			sleep((caddr_t)&sc->sc_r_wait, VICPRI);
			if (sc->sc_state & VIC_RESET) {
				/* oops, something happened while asleep */
				splx(s); 
				printf("vicin: reset encountered\n");
				return(cnt);
			}
			splx(s);
			try = vic_max_tries;
		}
	}
	sc->sc_state &= ~VIC_ACTIVE;
	return(0);
}
	
/* vicout: puts cnt bytes from buf to B008 via polling.  Works like vicin */
static long vicout(vicaddr, sc, buf, cnt)
	register struct vicdevice volatile *vicaddr;
	struct vic_softc *sc;
	register char *buf;
	long cnt;
{
	register int try;
	register int s;	

	VICDEBUG(VICRWDBG,printf("vicout(0x%x): put 0x%x\n",vicaddr,cnt));
	sc->sc_state |= VIC_ACTIVE;
	try = vic_max_tries;
	while (cnt) {
		if (IOIN(&vicaddr->out_stat) & VICST_IORDY) {
			IOOUT(&vicaddr->out_data,*buf++);
			cnt--;
			try = vic_max_tries;
		} else if ((--try)==0) {
			VICDEBUG(VICRWDBG,printf("vicout: sleep with 0x%x left\n",cnt));
			s = VIC_SPL();
			IOOUT(&vicaddr->int_ctl, (VICINT_IN | VICINT_OUT));
			IOOUT(&vicaddr->out_stat, VICST_INTENA);/* intr on */
			sc->sc_state |= VIC_WASLEEP;
			sleep((caddr_t)&sc->sc_w_wait, VICPRI);
			if (sc->sc_state & VIC_RESET) {
				/* oops, something happened while asleep */
				splx(s);
				printf("vicout: reset encountered\n");
				return(cnt);
			}
			splx(s);
			try = vic_max_tries;
		}
	}
	sc->sc_state &= ~VIC_ACTIVE;
	return(0);
}

/* perform DMA to/from the B008.  This routine sets up the DMA transfer,
 * then waits.  When the DMA channel is free, the dma routines call
 * vicdgo to start the DMA transfer.  The B008 will signal the end of the
 * transfer by interrupting; vicintr will wake us up then.
 *
 * Returns 0 on success, number of words not transferred on error
 */
static long vicdma(ic, bp, sc)
	register struct iocc_ctlr *ic;
	register struct buf *bp;
	register struct vic_softc *sc;
{
	register int dmastat;
	/*struct dma_8bit_device *dmaaddr = 
		(struct dma_8bit_device *)DM_CTL1_BASE; */

	register int s;
	
	VICDEBUG(VICDDBG, printf("vicdma: %s %x\n",(bp->b_flags & B_READ) ? "read" : "write", bp->b_bcount));
	ic->ic_dmachannel = vic_dma_channel;	/* B008 Channel */
/*	ic->ic_dmaflags = DMA_DEMAND | DMA_EXCLUSIVE | DMA_CANTINT; */
	ic->ic_dmaflags = DMA_DEMAND | DMA_EXCLUSIVE;
	ic->ic_dmabuf = bp;			

	s = VIC_SPL();
	sc->sc_state |= VIC_DODMA;
	dmastat = dma_setup(ic);		/* set up the transfer   */
	if (dmastat != DMA_OK_RET) {		/* possibly couldn't get */
		splx(s);
		VICDEBUG(VICDDBG, printf("vicdma: error %x in dma_setup\n",
 			dmastat));
		return(bp->b_bcount);
	}

	VICDEBUG(VICDDBG, printf("vicdma: sleeping\n"));
	sc->sc_state |= VIC_DMAASLEEP;
	(void) sleep(&sc->sc_dma_wait, VICPRI);	/* Wait for transfer     */
	sc->sc_state &= ~VIC_DMAASLEEP;
	VICDEBUG(VICDDBG, printf("vicdma: awakened\n"));
	if (sc->sc_state & VIC_RESET) {
		/* bail out */
		sc->sc_state &= ~VIC_DODMA;
		dma_done(ic->ic_dmachannel);
		splx(s);
		printf("vicdma: reset encountered\n");
		return(bp->b_bcount);
	}

/*	IOOUT(&dmaaddr->dm_cmd,DM_CMD_DEFAULT); */
	dma_done(ic->ic_dmachannel);
	sc->sc_state &= ~VIC_DODMA;
	splx(s);
	VICDEBUG(VICDDBG, printf("vicdma: dma success\n"));
	return(0);
}
	
/* attempt to flush driver.  This is not normally called, but may
 * be called via ioctl to get rid of stuck processes when the
 * board goes south, etc.
 */
vicflush(unit,dma)
	register int unit, dma;
{
	register struct vic_softc *sc = &vic_softc[unit];
	register struct vicdevice volatile *vicaddr;
	register struct iocc_device *iod = vicdinfo[unit];
	int flag = 0;

	printf("vic%d: vicflush st=0x%x",unit,sc->sc_state);
	vicaddr = (struct vicdevice volatile *)iod->iod_addr;

	sc->sc_state |= VIC_RESET;
	IOOUT(&vicaddr->int_ctl, 0);	/* interrupts off */
	if (sc->sc_state & VIC_RASLEEP) {
		sc->sc_state &= ~VIC_RASLEEP;
		wakeup((caddr_t)&sc->sc_r_wait);
		flag++;
	}
	if (sc->sc_state & VIC_WASLEEP) {
		sc->sc_state &= ~VIC_WASLEEP;
		wakeup((caddr_t)&sc->sc_w_wait);
		flag++;
	}
	if (sc->sc_state & VIC_DMAASLEEP) {
		sc->sc_state &= ~VIC_DMAASLEEP;
		wakeup((caddr_t)&sc->sc_dma_wait);
		flag++;
	}
	if (!flag) {
		/* if we get here, there's either nothing to do or
		 * we are waking up folks that are waiting on a
		 * buffer.
		 */
		viccleanupbuffer(&vicrbuf[unit]);
		viccleanupbuffer(&vicwbuf[unit]);
	}
	printf("vic%d: vicflush done st=0x%x",unit,sc->sc_state);
}	

viccleanupbuffer(bp)
  struct buf *bp;
{
    if (bp->b_flags & B_BUSY) {
	bp->b_flags &= ~(B_BUSY|B_PHYS|B_READ);
	iodone(bp);
    }
}

#endif
