/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * File:	callout_statistics.c
 * Purpose:
 *	Code for call-out statistics gathering.
 *
 * HISTORY
 * $Log:	callout_statistics.c,v $
 * Revision 2.3  89/01/12  07:43:55  rpd
 * 	Updated for new MACH_DEBUG, MACH_CO_INFO, renamed MACH_CO_STATS options,
 * 	and new location of callout_statistics.h in mach_debug/.
 * 	[89/01/12  04:39:48  rpd]
 * 
 * Revision 2.2  88/09/25  22:08:48  rpd
 * 	Updated vm_move calls.
 * 	[88/09/24  17:41:54  rpd]
 * 	
 * 	Added callout_statistics_fudge variable.
 * 	[88/09/12  23:00:47  rpd]
 * 	
 * 	host_callout_info no longer returns a dummy record at the beginning.
 * 	host_callout_statistics_reset no longer resets cos_current_size.
 * 	[88/09/10  01:18:35  rpd]
 * 	
 * 	Implemented host_callout_info.
 * 	[88/09/09  22:45:58  rpd]
 * 	
 * 	Created.
 * 	[88/09/09  04:50:24  rpd]
 * 
 */

#include <mach_debug.h>
#include <mach_co_info.h>
#include <mach_co_stats.h>

#include <sys/kern_return.h>
#include <sys/task.h>
#include <sys/lock.h>
#include <sys/callout.h>
#include <vm/vm_param.h>
#include <vm/vm_map.h>
#include <kern/ipc_globals.h>
#include <mach_debug/callout_statistics.h>

#if	MACH_DEBUG && MACH_CO_INFO
/*
 *	Routine:	host_callout_info [exported, user]
 *	Purpose:
 *		Return the contents of the call-out queue.
 */
kern_return_t
host_callout_info(target_task, info, infoCnt)
	task_t target_task;
	callout_info_array_t *info;
	unsigned int *infoCnt;
{
	unsigned int actual;	/* this many queue elements */
	unsigned int space;	/* space for this many elements */

	vm_size_t size;
	vm_offset_t addr;

	/* initial guess for amount of memory to allocate */
	size = page_size;

	for (;;) {
		register callout_info_t *info;
		register struct callout *event;
		int s;

		/* allocate memory non-pageable, so don't fault
		   while holding locks */
		(void) vm_allocate(ipc_kernel_map, &addr, size, TRUE);
		(void) vm_map_pageable(ipc_kernel_map,
				       addr, addr + size, FALSE);

		info = (callout_info_t *) addr;
		space = size / sizeof(callout_info_t);
		actual = 0;

		s = splhigh();
		simple_lock(&callout_lock);

		for (event = calltodo.c_next;
		     (actual < space) && (event != 0);
		     actual++, event = event->c_next) {
			info[actual].coi_time = event->c_time;
			info[actual].coi_arg = event->c_arg;
			info[actual].coi_func = event->c_func;
		}

		simple_unlock(&callout_lock);
		splx(s);

		if (event == 0)
			break;

		/* free current memory block */
		(void) kmem_free(ipc_kernel_map, addr, size);

		/* go for another try, allowing for expansion */
		size = round_page(2 * actual * sizeof(callout_info_t));
	}

	if (actual == 0) {
		/* no members, so return null pointer and deallocate memory */
		*info = 0;
		*infoCnt = 0;

		(void) kmem_free(ipc_kernel_map, addr, size);
	} else {
		vm_size_t size_used;

		*infoCnt = actual;

		size_used = round_page(actual * sizeof(callout_info_t));

		/* finished touching it, so make the memory pageable */
		(void) vm_map_pageable(ipc_kernel_map,
				       addr, addr + size_used, TRUE);

		/* the memory needs to be in ipc_soft_map */
		(void) vm_move(ipc_kernel_map, addr,
			       ipc_soft_map, size_used, TRUE,
			       (vm_offset_t *) info);

		/* free any unused memory */
		if (size_used != size)
			(void) kmem_free(ipc_kernel_map,
					 addr + size_used, size - size_used);
	}

	return KERN_SUCCESS;
}
#endif	MACH_DEBUG && MACH_CO_INFO

#if	MACH_CO_STATS
callout_statistics_t callout_statistics;
int callout_statistics_fudge;

#if	MACH_DEBUG
/*
 *	Routine:	host_callout_statistics [exported, user]
 *	Purpose:
 *		Return the accumulated call-out statistics.
 */
kern_return_t
host_callout_statistics(target_task, statistics)
	task_t target_task;
	callout_statistics_t *statistics;
{
	register int s;

	s = splhigh();
	simple_lock(&callout_lock);

	*statistics = callout_statistics;

	simple_unlock(&callout_lock);
	splx(s);

	return KERN_SUCCESS;
}

/*
 *	Routine:	host_callout_statistics_reset [exported, user]
 *	Purpose:
 *		Reset the accumulated call-out statistics.
 */
kern_return_t
host_callout_statistics_reset(target_task)
	task_t target_task;
{
	register int s;

	s = splhigh();
	simple_lock(&callout_lock);

	callout_statistics_fudge = -callout_statistics.cos_current_size;
	callout_statistics.cos_num_timeout = 0;
	callout_statistics.cos_cum_timeout_size = 0;
	callout_statistics.cos_cum_timeout_pos = 0;

	callout_statistics.cos_num_untimeout = 0;
	callout_statistics.cos_num_untimeout_hit = 0;
	callout_statistics.cos_cum_untimeout_size = 0;
	callout_statistics.cos_cum_untimeout_pos = 0;

	callout_statistics.cos_num_softclock = 0;
	callout_statistics.cos_cum_softclock_size = 0;

	simple_unlock(&callout_lock);
	splx(s);

	return KERN_SUCCESS;
}
#endif	MACH_DEBUG
#endif	MACH_CO_STATS
