/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	ipc_cache.h,v $
 * Revision 2.7  88/10/18  03:19:15  mwyoung
 * 	Use <kern/macro_help.h> to avoid lint.
 * 	[88/10/15            mwyoung]
 * 
 * Revision 2.6  88/10/11  10:12:01  rpd
 * 	Changed obj_cache_copyin so that the "barf" parameter
 * 	is syntactically a statement instead of an expression. 
 * 	[88/10/09  15:58:53  rpd]
 * 
 * Revision 2.5  88/09/25  22:09:45  rpd
 * 	Totally revamped the translation cache.
 * 	Fixed the port_copyout_cached (now obj_cache_copyout) bug.
 * 	[88/09/24  17:47:33  rpd]
 * 	
 * 	Moved definitions of PORT_UNCACHED, PORT_RESERVED_CACHE
 * 	out of MACH_IPC_TCACHE conditional.
 * 	[88/09/20  01:55:31  rpd]
 * 	
 * 	Replaced CACHE_PORT_TRANSLATIONS with MACH_IPC_TCACHE.
 * 	Added PORT_RESERVED_CACHE() predicate.
 * 	[88/09/19  23:28:06  rpd]
 * 	
 * 	Changed includes to the new style.
 * 	[88/09/19  16:09:15  rpd]
 * 
 * Revision 2.4  88/08/06  18:12:17  rpd
 * Explictly use macro versions of port_release and port_reference.
 * 
 * Revision 2.3  88/07/20  23:00:21  rpd
 * Use kernel_ipc_space instead of kernel_only.  Some macro clean-up.
 * 
 * Revision 2.2  88/07/20  16:31:22  rpd
 * Created from mach_ipc.c.
 * 
 */ 

#ifndef	_IPC_CACHE_
#define	_IPC_CACHE_	1

/*
 * The per-task Translation Cache
 *
 * We cache mappings between a task's local name for a port and the
 * internal pointer to that port.  Actually, we can cache mappings for
 * general objects, but currently the cache has two lines which
 * are used only for caching ports encounted in message headers.
 *
 * Each cache line has two fields, a local_name (port_name_t) and
 * a global name (kern_obj_t).  Values of PORT_NULL/KERN_OBJ_NULL
 * indicate an invalid cache line.  Both values in a valid cache
 * line are non-null.
 *
 * [There is no advantage to making the PORT_NULL/KERN_OBJ_NULL mapping
 * a valid cache line, because the fast path code has to check for NULL
 * as a special case anyway, to know whether the object should
 * gain/lose a reference.]
 *
 * The cache is part of the task structure, and is accessed under the
 * IPC task lock.  The cache does not hold any references for objects.
 * If there is a valid mapping in the cache, then the task must have
 * a translation record for the object, which holds a reference.
 *
 * Any changes to the task's object/local-name mappings must flush
 * old mappings from the cache.  This can happen when the task loses
 * rights to an object or the task's local name for the object changes.
 *
 * If a valid mapping is found in the cache, then one may assume that
 * the task is ipc_active.  (And so the mapping may be used without
 * checking ipc_active.)  This is because mappings are only entered
 * into the cache when the task is active, and when the task is terminated
 * all cached mappings are immediately flushed.
 *
 * If a valid mapping is found in the cache, then one may assume that
 * the task is not a kernel_ipc_space task.  (And so the mapping may be
 * used without checking kernel_ipc_space.)  This is because mappings
 * are never entered into the cache of kernel_ipc_space tasks.
 *
 * The obj_cache macros which manipulate the cache generally require their
 * caller to hold the task's IPC task lock and no other locks.
 */

#include <mach_ipc_tcache.h>

#include <sys/boolean.h>
#include <sys/kern_obj.h>
#include <sys/task.h>
#include <kern/macro_help.h>

#if	MACH_IPC_TCACHE

/*
 * extern void
 * obj_cache_set(task, index, name, object)
 *	task_t task;
 *	int index;
 *	port_name_t name;
 *	kern_obj_t object;
 *
 * Stores a mapping in the specified cache line.
 * The task must be locked.
 */

#define	obj_cache_set(task, index, _name, _object)			\
	MACRO_BEGIN							\
	(task)->obj_cache[index].object = (_object);			\
	(task)->obj_cache[index].name = (_name);			\
	MACRO_END


/*
 * extern void
 * obj_cache_clear(task, index)
 *	task_t task;
 *	int index;
 *
 * Resets the specified cache line.
 * The task must be locked.
 */

#define obj_cache_clear(task, index)					\
	MACRO_BEGIN							\
	obj_cache_set((task), (index), PORT_NULL, KERN_OBJ_NULL);	\
	MACRO_END


/*
 * extern void
 * obj_cache_flush(task, object)
 *	task_t task;
 *	kern_obj_t object;
 *
 * Flushes all mappings for the object from the task's cache.
 * Both the task and object must be locked.
 */

#define obj_cache_flush(task, _object) 					\
	MACRO_BEGIN							\
	if ((task)->obj_cache[OBJ_CACHE_LOCAL].object == (_object))	\
		obj_cache_clear((task), OBJ_CACHE_LOCAL);		\
									\
	if ((task)->obj_cache[OBJ_CACHE_REMOTE].object == (_object))	\
		obj_cache_clear((task), OBJ_CACHE_REMOTE);		\
	MACRO_END


/*
 * extern void
 * obj_cache_copyout(task, index, object, name)
 *	task_t task;
 *	int index;
 *	kern_obj_t object;
 *	port_name_t &name;
 *
 * Copies out the object, returning a local name.
 * The caller loses a reference to the object.
 * The task must be locked.
 */

#define obj_cache_copyout(task, index, _object, _name)			\
	MACRO_BEGIN							\
	assert((task) != TASK_NULL);					\
	assert((_object) != KERN_OBJ_NULL);				\
									\
	if ((task)->obj_cache[index].object == (_object)) {		\
		obj_release(_object);					\
		(_name) = (task)->obj_cache[index].name;		\
		ipc_event(port_copyout_hits);				\
	} else {							\
		object_copyout_cache((task), (index),			\
				     (_object), &(_name));		\
	} 								\
	MACRO_END


/*
 * extern void
 * obj_cache_copyin(task, index, name, object, barf)
 *	task_t task;
 *	int index;
 *	port_name_t name;
 *	kern_obj_t &object;
 *	code barf;
 *
 * Copies in the local name, returning an object.
 * The caller gains a reference to the object.
 * If the copyin fails, executes the "barf" code,
 * which syntactically must be a complete statement
 * and which should return/goto.  The task must be locked.
 */

#define obj_cache_copyin(task, index, _name, _object, barf)		\
	MACRO_BEGIN							\
	if ((task)->obj_cache[index].name == (_name)) {			\
		(_object) = (task)->obj_cache[index].object;		\
		obj_reference(_object);					\
		ipc_event(port_copyin_hits);				\
	} else {							\
		if (!object_copyin_cache((task), (index),		\
					 (_name), &(_object)))		\
			barf						\
	} 								\
	MACRO_END


/*
 * extern void
 * obj_cache_init(task)
 *	task_t task;
 *
 * Initializes the task's cache.
 * The task doesn't have to be locked if nobody can get at it.
 */

#define obj_cache_init(task)						\
	MACRO_BEGIN							\
	obj_cache_clear((task), OBJ_CACHE_LOCAL);			\
	obj_cache_clear((task), OBJ_CACHE_REMOTE);			\
	MACRO_END


/*
 * extern void
 * obj_cache_terminate(task)
 *	task_t task;
 *
 * Finalizes the task's cache.
 * The task must be locked.
 */

#define	obj_cache_terminate(task)

#else	MACH_IPC_TCACHE

#define obj_cache_set(task, index, name, object)
#define obj_cache_clear(task, index)
#define obj_cache_flush(task, object)
#define obj_cache_init(task)
#define obj_cache_terminate(task)

#define obj_cache_copyout(task, index, object, name) \
	MACRO_BEGIN							\
	object_copyout_cache((task), (index), (object), &(name));	\
	MACRO_END

#define obj_cache_copyin(task, index, name, object, barf) \
	MACRO_BEGIN							\
	if (!object_copyin_cache((task), (index), (name), &(object)))	\
		barf							\
	MACRO_END

#endif	MACH_IPC_TCACHE

#endif	_IPC_CACHE_
