/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * File:	ipc_globals.c
 * Purpose:
 *	Define & initialize Mach IPC global variables.
 *
 * HISTORY
 * $Log:	ipc_globals.c,v $
 * Revision 2.10  89/01/15  16:22:06  rpd
 * 	Removed the hits fields in the TP/TL hash buckets.
 * 	[89/01/15  14:57:08  rpd]
 * 
 * Revision 2.9  89/01/10  23:29:01  rpd
 * 	Changed MACH_IPCSTATS to MACH_IPC_STATS.
 * 	[89/01/10  22:59:16  rpd]
 * 
 * Revision 2.8  88/12/21  14:21:44  mja
 * 	Changed size pararameter to zinit for port_hash_zone
 * 	to be a function of PORT_MAX.
 * 	[88/12/20  15:26:30  mrt]
 * 
 * Revision 2.7  88/11/23  16:38:12  rpd
 * 	Changed mach_ipc_debug to ipc_debug.
 * 	[88/11/23  10:39:04  rpd]
 * 
 * Revision 2.6  88/10/18  03:19:33  mwyoung
 * 	Use <kern/macro_help.h> to avoid lint.
 * 	[88/10/15            mwyoung]
 * 
 * Revision 2.5  88/10/11  10:14:00  rpd
 * 	Added complex_notification_template.
 * 	[88/10/11  07:57:46  rpd]
 * 
 * Revision 2.4  88/08/25  18:14:53  mwyoung
 * 	Corrected include file references.
 * 	[88/08/22            mwyoung]
 * 	
 * 	Eliminate the ipc_soft_map's pmap.  Inadvertent pmap operations
 * 	during paging IPC operations can cause deadlocks.
 * 	[88/08/11  19:11:01  mwyoung]
 * 
 * Revision 2.3  88/08/06  18:14:44  rpd
 * Replaced ipc_basics.h with ipc_copyin.h/ipc_copyout.h.
 * 
 * Revision 2.2  88/07/22  07:23:14  rpd
 * Created for Mach IPC global variables and initialization.
 * 
 */

#include <mach_ipc_stats.h>

#include <sys/boolean.h>
#include <sys/port.h>
#include <sys/task.h>
#include <sys/zalloc.h>
#include <sys/notify.h>
#include <sys/mach_param.h>
#include <sys/kern_port.h>
#include <sys/kern_set.h>
#include <vm/vm_param.h>
#include <vm/vm_map.h>
#include <vm/vm_kern.h>
#include <kern/ipc_hash.h>
#include <kern/ipc_copyin.h>
#include <kern/ipc_copyout.h>
#include <kern/ipc_globals.h>

zone_t kmsg_zone;
zone_t kmsg_zone_large;

zone_t port_zone;
zone_t port_zone_reserved;

zone_t set_zone;

task_t ipc_soft_task;
vm_map_t ipc_soft_map;
vm_map_t ipc_kernel_map;

unsigned int ipc_debug = 0;

zone_t port_hash_zone;

port_hash_bucket_t *TP_table;
port_hash_bucket_t *TL_table;

notification_t notification_template;
notification_t complex_notification_template;

object_copyout_table_t object_copyout_table[MSG_TYPE_LAST];

object_copyin_table_t object_copyin_table[MSG_TYPE_LAST][PORT_TYPE_LAST];

unsigned int timeout_minimum = 1;
unsigned int timeout_scaling_factor;

/*
 *	Routine:	object_copyout_init [internal]
 *	Purpose:
 *		Called to initialize object_copyout_table.
 */
void
object_copyout_init()
{
	int mt;
	port_type_t pt;

#define init(mt, _destroy, _func, _nomerge) \
	MACRO_BEGIN					\
	object_copyout_table[mt].destroy = (_destroy);	\
	object_copyout_table[mt].func = (_func);	\
	object_copyout_table[mt].nomerge = (_nomerge);	\
	MACRO_END

	for (mt = 0; mt < MSG_TYPE_LAST; mt++)
		init(mt, 0, 0, FALSE);

	init(MSG_TYPE_PORT, 0, 0, FALSE);
	init(MSG_TYPE_PORT_RECEIVE,
	     port_destroy_receive, port_copyout_receive, FALSE);
	init(MSG_TYPE_PORT_OWNERSHIP,
	     port_destroy_own, port_copyout_own, FALSE);
	init(MSG_TYPE_PORT_ALL,
	     port_destroy_receive_own, port_copyout_receive_own, FALSE);

#undef	init

#define init(mt, pt, _result) \
	object_copyout_table[mt].result[pt] = (_result)	\

	for (mt = 0; mt < MSG_TYPE_LAST; mt++)
		for (pt = 0; pt < PORT_TYPE_LAST; pt++)
			init(mt, pt, PORT_TYPE_NONE);

	init(MSG_TYPE_PORT, PORT_TYPE_NONE, PORT_TYPE_SEND);
	init(MSG_TYPE_PORT, PORT_TYPE_SEND, PORT_TYPE_SEND);
	init(MSG_TYPE_PORT, PORT_TYPE_RECEIVE, PORT_TYPE_RECEIVE);
	init(MSG_TYPE_PORT, PORT_TYPE_OWN, PORT_TYPE_OWN);
	init(MSG_TYPE_PORT, PORT_TYPE_RECEIVE_OWN, PORT_TYPE_RECEIVE_OWN);

	init(MSG_TYPE_PORT_RECEIVE, PORT_TYPE_NONE, PORT_TYPE_RECEIVE);
	init(MSG_TYPE_PORT_RECEIVE, PORT_TYPE_SEND, PORT_TYPE_RECEIVE);
	init(MSG_TYPE_PORT_RECEIVE, PORT_TYPE_OWN, PORT_TYPE_RECEIVE_OWN);

	init(MSG_TYPE_PORT_OWNERSHIP, PORT_TYPE_NONE, PORT_TYPE_OWN);
	init(MSG_TYPE_PORT_OWNERSHIP, PORT_TYPE_SEND, PORT_TYPE_OWN);
	init(MSG_TYPE_PORT_OWNERSHIP, PORT_TYPE_RECEIVE, PORT_TYPE_RECEIVE_OWN);

	init(MSG_TYPE_PORT_ALL, PORT_TYPE_NONE, PORT_TYPE_RECEIVE_OWN);
	init(MSG_TYPE_PORT_ALL, PORT_TYPE_SEND, PORT_TYPE_RECEIVE_OWN);

#undef	init
}

/*
 *	Routine:	object_copyin_init [internal]
 *	Purpose:
 *		Called to initialize object_copyin_table.
 */
void
object_copyin_init()
{
	int mt;
	port_type_t pt;

#define init(mt, pt, _illegal, _nodealloc, _dodealloc, _result, _func) \
	MACRO_BEGIN						\
	object_copyin_table[mt][pt].illegal = (_illegal);	\
	object_copyin_table[mt][pt].nodealloc = (_nodealloc);	\
	object_copyin_table[mt][pt].dodealloc = (_dodealloc);	\
	object_copyin_table[mt][pt].result = (_result);		\
	object_copyin_table[mt][pt].func = (_func);		\
	MACRO_END

	for (mt = 0; mt < MSG_TYPE_LAST; mt++)
		for (pt = 0; pt < PORT_TYPE_LAST; pt++)
			init(mt, pt, TRUE, FALSE, FALSE, PORT_TYPE_NONE, 0);

	init(MSG_TYPE_PORT, PORT_TYPE_SEND,
	     FALSE, FALSE, FALSE,
	     PORT_TYPE_SEND, 0);

	init(MSG_TYPE_PORT, PORT_TYPE_RECEIVE,
	     FALSE, TRUE, FALSE,
	     PORT_TYPE_RECEIVE, 0);

	init(MSG_TYPE_PORT, PORT_TYPE_OWN,
	     FALSE, TRUE, FALSE,
	     PORT_TYPE_OWN, 0);

	init(MSG_TYPE_PORT, PORT_TYPE_RECEIVE_OWN,
	     FALSE, TRUE, FALSE,
	     PORT_TYPE_RECEIVE_OWN, 0);

	init(MSG_TYPE_PORT_RECEIVE, PORT_TYPE_RECEIVE,
	     FALSE, FALSE, FALSE,
	     PORT_TYPE_SEND, port_copyin_receive);

	init(MSG_TYPE_PORT_RECEIVE, PORT_TYPE_RECEIVE_OWN,
	     FALSE, TRUE, FALSE,
	     PORT_TYPE_OWN, port_copyin_receive);

	init(MSG_TYPE_PORT_OWNERSHIP, PORT_TYPE_OWN,
	     FALSE, FALSE, FALSE,
	     PORT_TYPE_SEND, port_copyin_own);

	init(MSG_TYPE_PORT_OWNERSHIP, PORT_TYPE_RECEIVE_OWN,
	     FALSE, TRUE, FALSE,
	     PORT_TYPE_RECEIVE, port_copyin_own);

	init(MSG_TYPE_PORT_ALL, PORT_TYPE_RECEIVE_OWN,
	     FALSE, FALSE, FALSE,
	     PORT_TYPE_SEND, port_copyin_receive_own);

#undef	init
}

/*
 *	Routine:	ipc_bootstrap [exported]
 *	Purpose:
 *		Initialize IPC structures needed even before
 *		the "kernel task" can be initialized
 */
void
ipc_bootstrap()
{
	int i;

	kmsg_zone_large = zinit(MSG_SIZE_MAX,
				128 * MSG_SIZE_MAX,
				MSG_SIZE_MAX, FALSE,
				"large messages");

	kmsg_zone = zinit(KERN_MSG_SMALL_SIZE,
			  2048 * KERN_MSG_SMALL_SIZE,
			  32 * KERN_MSG_SMALL_SIZE, FALSE,
			  "messages");

	port_hash_zone = zinit(sizeof(struct port_hash),
			       10*PORT_MAX*sizeof(struct port_hash),
			       page_size, FALSE,
			       "port translations");

	port_zone = zinit(sizeof(struct kern_port),
			  PORT_MAX * sizeof(struct kern_port),
			  4 * page_size, FALSE,
			  "ports");
	zchange(port_zone, FALSE, FALSE, TRUE);

	set_zone = zinit(sizeof(struct kern_set),
			 SET_MAX * sizeof(struct kern_set),
			 4 * page_size, FALSE,
			 "sets");
	zchange(set_zone, FALSE, FALSE, TRUE);

	port_zone_reserved = zinit(sizeof(struct kern_port),
				   PORT_MAX * sizeof(struct kern_port),
				   4 * page_size, FALSE,
				   "ports (reserved)");

	TP_table = (port_hash_bucket_t *) kmem_alloc(kernel_map,
		   (vm_size_t) (PORT_HASH_COUNT * sizeof(port_hash_bucket_t)));
	if (TP_table == (port_hash_bucket_t *) 0)
		panic("ipc_bootstrap: cannot create TP_table");

	for (i = 0; i < PORT_HASH_COUNT; i++) {
		queue_init(&TP_table[i].head);
		bucket_lock_init(&TP_table[i]);
	}

	TL_table = (port_hash_bucket_t *) kmem_alloc(kernel_map,
		   (vm_size_t) (PORT_HASH_COUNT * sizeof(port_hash_bucket_t)));
	if (TL_table == (port_hash_bucket_t *) 0)
		panic("ipc_bootstrap: cannot create TL_table");

	for (i = 0; i < PORT_HASH_COUNT; i++) {
		queue_init(&TL_table[i].head);
		bucket_lock_init(&TL_table[i]);
	}

#if	MACH_IPC_STATS
	ipc_stats_init();
#endif	MACH_IPC_STATS

	object_copyin_init();
	object_copyout_init();
}

/*
 *	Routine:	ipc_init [exported]
 *	Purpose:
 *		Called to initialize remaining data structures before
 *		any user traps are handled.
 */
void
ipc_init()
{
	register notification_t *n;
	vm_offset_t min, max;
	extern int hz;

	/* Create a template for notification messages. */

	n = &notification_template;
	n->notify_header.msg_local_port = PORT_NULL;
	n->notify_header.msg_remote_port = PORT_NULL;
	n->notify_header.msg_simple = TRUE;
	n->notify_header.msg_type = MSG_TYPE_EMERGENCY;
	n->notify_header.msg_id = 0;
	n->notify_header.msg_size = sizeof(notification_t);

	n->notify_type.msg_type_name = MSG_TYPE_PORT_NAME;
	n->notify_type.msg_type_inline = TRUE;
	n->notify_type.msg_type_deallocate = FALSE;
	n->notify_type.msg_type_longform = FALSE;
	n->notify_type.msg_type_number = 1;
	n->notify_type.msg_type_size = 32;

	/* Create a template for complex_notification messages. */

	n = &complex_notification_template;
	n->notify_header.msg_local_port = PORT_NULL;
	n->notify_header.msg_remote_port = PORT_NULL;
	n->notify_header.msg_simple = FALSE;
	n->notify_header.msg_type = MSG_TYPE_EMERGENCY;
	n->notify_header.msg_id = 0;
	n->notify_header.msg_size = sizeof(notification_t);

	n->notify_type.msg_type_name = 0;
	n->notify_type.msg_type_inline = TRUE;
	n->notify_type.msg_type_deallocate = FALSE;
	n->notify_type.msg_type_longform = FALSE;
	n->notify_type.msg_type_number = 1;
	n->notify_type.msg_type_size = 32;

	/* Compute the timeout scaling factor. */

	timeout_scaling_factor = (1000 / hz);

	/* Create a task used to hold rights and data in transit. */

	if (task_create(TASK_NULL /* kernel_task */, FALSE, &ipc_soft_task)
					!= KERN_SUCCESS)
		panic("ipc_init");
	ipc_soft_map = ipc_soft_task->map;

	ipc_soft_map->pmap = PMAP_NULL;

	ipc_kernel_map = kmem_suballoc(kernel_map, &min, &max,
				       1024 * 1024, TRUE);

	kernel_task->ipc_privilege = TRUE;
	kernel_task->kernel_ipc_space = TRUE;
}
