#
#  Source version control 
#
######################################################################
# HISTORY
# 12-May-88  Mike Accetta (mja) at Carnegie-Mellon University
#	Added "compat", "inline", and "uxkern" directories.
#	[ V5.1(XF24) ]
#
# 24-Mar-88  Mike Accetta (mja) at Carnegie-Mellon University
#	Changed to make check-ins relative to $CIBASE environment
#	variable.
#	[ V5.1(XF22) ]
#
# 20-Feb-88  Mike Accetta (mja) at Carnegie-Mellon University
#	Don't delete after check-in.
#	[ V5.1(XF22) ]
#
# 22-Jan-88  Mike Accetta (mja) at Carnegie-Mellon University
#	Updated to eliminate need for individual Makefile's which
#	define NEWSRC in each source directory;  corrected to do an
#	implicit open when altering version files so that they can be
#	checked out before being altered;  added necessary version file
#	dependencies when archiving sources and/or changing versions so
#	that these operations work correctly when the files are checked
#	in;  moved version stamp file out of ../conf source area since
#	its components are there instead.
#	[ V5.1(XF21) ]
#
# 16-Oct-87  Mike Accetta (mja) at Carnegie-Mellon University
#	Correct edit rule to use ${VERSION} variable.
#	[ V5.1(XF19) ]
#
# 09-Oct-87  Mike Accetta (mja) at Carnegie-Mellon University
#	Reset .SUFFIXES to prevent single suffix rules from triggering
#	on some directory names; redirect input from </dev/tty before
#	archiving until RCS has better luck determining the right login
#	name.
#	[ V5.1(XF18) ]
#
# 07-Jul-87  Mike Accetta (mja) at Carnegie-Mellon University
#	Changed ${ARCHIVE} from ../ARCHIVE to ./ARCH_TIMES; added
#	"src/V" directory.
#	[ V5.1(XF13) ]
#
# 09-May-87  Mike Accetta (mja) at Carnegie-Mellon University
#	Moved specific source file dependency lists into individual
#	directory Makefile's to simplify procedure and treat all
#	directories identically; fixed to automatically create
#	${ARCHIVE} directory (and moved from ./FILES to ../ARCHIVE).
#	[ V5.1(XF11) ]
#
# 16-Apr-87  Mike Accetta (mja) at Carnegie-Mellon University
#	Added "stand" directory.
#	[ V5.1(F8) ]
#
# 13-Mar-87  Mike Accetta (mja) at Carnegie-Mellon University
#	Made "archive" the default rule.
#	[ V5.1(F7) ]
#
# 11-Mar-87  Mike Accetta (mja) at Carnegie-Mellon University
#	Renamed "sys" directory to "bsd".
#	[ V5.1(F6) ]
#
# 06-Mar-87  Mike Accetta (mja) at Carnegie-Mellon University
#	Remove "accent" directory from archive list, add "vice".
#	[ V5.1(F5) ]
#
# 31-Jan-87  Mike Accetta (mja) at Carnegie-Mellon University
#	Created.
#	[ V5.1(F1) ]
#
######################################################################
#



#
#  These rules are used to maintain a master kernel source area with
#  simple version control.  They trigger the history, check-in and
#  version update facilities of the version control makefile which can
#  be applied only when making changes to a master source area.
#  A master source area is defined by creating a control directory
#  (normally "V") and creating a symbolic "Makefile" link to this
#  file in that directory.
#
#
#  A master source area contains:
#
#  - The set of directories required to build an entire kernel for
#    one or more machine types.
#  - The control directory that records the last check-in times of
#    modules in these directories such that when any file changes,
#    it can be detected and checked-in automatically before beginning
#    changes that would comprise a new version of the source; and
#    the version components which comprise the version identifier that
#    tags the current source state.
#
#  The version identifier tag for the sources is comprised of the
#  following four components and has the form:
#  <major>.<minor>(<variant><edit>) where:
#
#  1) <major> is the major version number which identifies the base
#     level of the sources.  This number will change very infrequently
#     and have so far progressed from (1=3BSD, 2=4BSD, 3=4.1BSD,
#     4=4.2BSD, and 5=4.3BSD)
#  2) <minor> is the minor version number which identifies a
#     significant state change from a previous version that is tracked
#     by all the parallel master development source trees at a site.
#     This number begins at 1 for each new major version and is
#     incremented only when all diverging master source areas are
#     synchronized to a common state.
#  3) <variant> is the version variant string which serves to
#     distinuish between divergent master source areas.  This string is
#     normally a constant which is selected once when the master source
#     area is initially established and never changes thereafter.
#  4) <edit> is the edit version number which distinguishes between
#     successive sets of changes to a given master source area.  This
#     number is incremented whenever any change is made to the source
#     area that will cause binaries built from that area to differ from
#     current binaries already deployed.
#
#  Any master source area is organized such that none of the
#  directories or files propagated from it need be writeable on the
#  target system in order to build new configurations.  The version
#  identifier for a particular source area applies to all kernel
#  binaries built from the associated sources independent of the
#  machine type.  Changes to the sources are made only in the master
#  source area and upgraded if necessary to client machines for
#  configuration generation, compilation and linking.  Only the
#  configuration and build directories are ever changed on client
#  machines (unless the machines also contain variant master source
#  areas into which the read-only upgrades might be merged to
#  created a different version identifier).
#
#
#  The rules described below are used to perform the common editing
#  functions that are involved with building new versions of the
#  kernel.
#
#  close   - log, archive and unlock any source files modified since
#	     the last archive;  prevent any accidental further changes
#	     to the source by write-protecting all source directories
#	     (the individual sources files are write-protected by the
#	     archive and unlock RCS procedure).
#  open    - write-enable the source directories allowing changes (although
#	     individual files must still be checked-out with a lock to
#            open them if necessary)
#  archive - log, archive and unlock as above
#  edit    - log, archive and unlock everything; increment edit version; create
#	     new $VERSION; re-open
#  minor   - log, archive and unlock everything; increment minor and edit
#	     version; create new $VERSION; re-open
#  major   - log, archive and unlock everything; increment major and edit,
#	     reset minor version; create new $VERSION; re-open
#  version - log, archive and unlock everything; create new $VERSION; re-open
#
#  newvers - short for "close edit open"
#


#
#  Version file pathname definitions
#

.SUFFIXES:
.SUFFIXES: .dummy

ARCHIVE=ARCH_TIMES
VERSION_STAMP=version_stamp
VERSION_PREFIX=${CIBASE}/conf/version.
MAJOR  =${VERSION_PREFIX}major
MINOR  =${VERSION_PREFIX}minor
EDIT   =${VERSION_PREFIX}edit
VARIANT=${VERSION_PREFIX}variant

VERSION_FILES=${MAJOR} ${MINOR} ${VARIANT} ${EDIT}


#
#  Directories which contain sources.
#
DIRS= \
	${CIBASE}/bsd \
	${CIBASE}/ca \
	${CIBASE}/ca/kdb \
	${CIBASE}/cacons \
	${CIBASE}/caif \
	${CIBASE}/caio \
	${CIBASE}/compat \
	${CIBASE}/conf \
	${CIBASE}/emul \
	${CIBASE}/h \
	${CIBASE}/inline \
	${CIBASE}/kern \
	${CIBASE}/mon \
	${CIBASE}/net \
	${CIBASE}/netimp \
	${CIBASE}/netinet \
	${CIBASE}/netns \
	${CIBASE}/pixrect \
	${CIBASE}/src \
	${CIBASE}/src/config \
	${CIBASE}/src/V \
	${CIBASE}/stand \
	${CIBASE}/sun \
	${CIBASE}/sun/kdb \
	${CIBASE}/sun3 \
	${CIBASE}/sundev \
	${CIBASE}/sunif \
	${CIBASE}/sunstand \
	${CIBASE}/sunwindow \
	${CIBASE}/sunwindowdev \
	${CIBASE}/sysV \
	${CIBASE}/uxkern \
	${CIBASE}/vax \
	${CIBASE}/vax/kdb \
	${CIBASE}/vax/inline \
	${CIBASE}/vaxif \
	${CIBASE}/vaxmba \
	${CIBASE}/vaxuba \
	${CIBASE}/vice \
	${CIBASE}/vm 



#
#  The default is to archive anything that has changed (keyed on
#  directory modification times, though).
#
default: archive



#
#  Insure that the archive time directory is created
#

mkARCHIVE: ${ARCHIVE}

${ARCHIVE}:
	mkdir ${ARCHIVE}



#
#  Open the sources for modification
#

open:	.open
	@echo "[ version `cat ${VERSION_STAMP}` open ]"

.open:
	@chmod +w ${DIRS}


#
#  Close the sources to protect against accidental modification.  This
#  path forces a full archive every time (by not creating the target
#  file) and is used when it is necessary guarantee that everything has
#  been archived (such as before creation of a new version begins).
#

close:	mkARCHIVE .archive_always .close
	@echo "[ version `cat ${VERSION_STAMP}` closed ]"

.close:
	@chmod -w ${DIRS}


#
#  Archive the sources against any accidental modification.  To speed
#  processing, with this rule only those directories which have changed since
#  the last archive are considered candidates since the wild-card
#  expansion of all files can take considerable time.
#

archive: mkARCHIVE .archive
	@touch .archive


newvers: close .open edit stamp open

edit: close .open edit.incr stamp open

minor: close .open minor.incr edit.incr stamp open

major: close .open major.incr minor.reset edit.incr stamp open



#
#  Create set of directories to archive.  The second target is
#  never created and will always generate a complete list, while
#  the first is based only on the previous archive.  This technique
#  depends on the fact that an RCS checkout into one of the source
#  directories will cause it to be modified and insure that it
#  is newer than any prior archive so that the directory will be
#  considered when searching for modified files.
#
#  To avoid having to generate explicit rules for each source
#  directory, we invoke "make" individually on each directory with the
#  NEWSRC variable set to the appropiate target to generate the changed
#  file list consisting of all non-directory files in the target area
#  which are newer than the target time.  N.B.  The rule uses the "./*"
#  form to prevent the generated names from conflicting with other
#  common target names in this makefile (e.g. "version").
#
.archive .archive_always: ${DIRS}
	@CWD=`pwd`; \
	 for i in $?; do \
	 eval `echo $$i | sed -n -e 's,^'"${CIBASE}"'/,D=,p' -e 's,/,_,g' -e 's,^D=,S=,p' `; \
	 NEWSRC=$$CWD/${ARCHIVE}/$$S; \
	 echo "[[[ archiving ${CIBASE}/$$D ]]]" ; \
	 (F="`make D=${CIBASE}/$$D NEWSRC=$$NEWSRC NEWSRC`"; \
	  make F="$$F" ARCH=$$NEWSRC ARCH); \
	 done

NEWSRC = .

NEWSRC:	${NEWSRC}

${NEWSRC}: ${D}/*
	@ls -dF $? | sed \
	    -e '/\/lex.yy.c$$/d' \
	    -e '/\/y.tab.[ch]$$/d' \
	    -e '/\.o$$/d' \
	    -e '/\/$$/d'

ARCH=.

ARCH: ${ARCH}

${ARCH}: ${F}
	@arch </dev/tty -version `cat ${VERSION_STAMP}` $?
	@touch $@



#
#  Increment version number in file by 1.
#
edit.incr minor.incr major.incr: ${EDIT} ${MINOR} ${MAJOR}
	@F=${VERSION_PREFIX}`basename $@ .incr`; \
	 E=`cat $$F`; \
	 chmod -f +w $$F; \
	 expr $$E + 1 >$$F; \
	 arch -nowhist $$F

#
#  Reset version number in file to 1.
#
minor.reset: ${MINOR}
	@F=${VERSION_PREFIX}`basename $@ .reset`; \
	 chmod -f +w $$F; \
	 echo 1 >$$F; \
	 arch -nowhist $$F


#
#  Build version identfier file from components
#

${VERSION_STAMP} stamp: ${VERSION_FILES}
	@awk '{n[NR]=$$1;} END{printf "%d.%d(%s%d)\n",n[1],n[2],n[3],n[4];}' \
	    ${VERSION_FILES} >.tmp
	@chmod -f +w ${VERSION_STAMP}; cp .tmp ${VERSION_STAMP}; rm -f .tmp
	@echo "[ version `cat ${VERSION_STAMP}` stamped ]"
