/* 
 * Mach Operating System
 * Copyright (c) 1988 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * File:	apa8ctty.c
 *
 * HISTORY
 * $Log:	apa8ctty.c,v $
 * Revision 2.2  88/08/06  18:35:15  rpd
 * Created.
 * 
 */

/*
 * 5799-CGZ (C) COPYRIGHT IBM CORPORATION 1986,1987
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */

/* $Header: apa8ctty.c,v 2.2 88/08/06 18:35:15 rpd Exp $ */
/* $ACIS:apa8ctty.c 9.0$ */
/* $Source: /afs/cs.cmu.edu/source_mach/rcs/kernel/standca/apa8ctty.c,v $ */


#ifndef lint
static char *rcsid = "$Header: apa8ctty.c,v 2.2 88/08/06 18:35:15 rpd Exp $";
#endif

#include "apaeightc.h"
#if NAPAEIGHTC > 0
 
extern int beep();
extern unsigned short *apa_fontblt();

#include "types.h"
 
#include "sa.h"

#undef LOCAL
#define LOCAL static

typedef struct {
	unsigned short *rwaddr;
} screen_stuff;
 
#include "apa8ctty.h"
#include "screen_conf.h"

#define APA8C_FONT	1	/* apa_fontblt id */
/*
 * For now use the same font as the APA-8.
 */
#include "apaeight.h"
#if NAPAEIGHT <= 0
#include "apa8tty_font.h"
#else
extern struct font_head apa8fonthead;
extern unsigned short apa8fontBits[];
extern Blt_Bitmap apa8fontBM;
#endif

/*
 * Define macros to check, set and reset the cursor bit.
 */
LOCAL short apa8Ccursorflag;
#define CURSOR_OFF (apa8Ccursorflag == FALSE)
#define CURSOR_ON (apa8Ccursorflag == TRUE)
#define SET_CURSOR  apa8Ccursorflag = TRUE
#define RESET_CURSOR apa8Ccursorflag = FALSE

/*
 * Pointer to the bitmap cursor spot.
 */
LOCAL unsigned short *apa8Cscr;

/*
 * This is an array of masks to be used when trying to determine where the
 * character is in a 16 bit word on the bitmap.  Note it covers all 9 bits.
 */
LOCAL unsigned short char_mask[] = {
	0x007f,		/* 0 */ 
	0xfe00,		/* 7 */
	0xfc01,		/* 6 */
	0xf803,		/* 5 */
	0xf007,		/* 4 */
	0xe00f,		/* 3 */
	0xc01f,		/* 2 */
	0x803f,		/* 1 */
};

/*
 * This is an array of masks to be used when rotating a character and
 * writing in the y direction. (Thats why byte 1 and byte 2 in the mask
 * are the same.)
 */
LOCAL unsigned short y1_rotatemasks[] = {
	0x0000,		/* 8 */ 
	0xfefe,		/* 1 */
	0xfcfc,		/* 2 */
	0xf8f8,		/* 3 */
	0xf0f0,		/* 4 */
	0xe0e0,		/* 5 */
	0xc0c0,		/* 6 */
	0x8080,		/* 7 */
};

/*
 * This array of rotate masks is used on the second pass of writing a
 * character in apa8Cwrite_char. The problem is that we want to mask out
 * the "9th" bit (characters are 8 bits, the 9th bit is the space between
 * characters which may contain garbage.) Note each of these values is 
 * equal to:
 * 		((~y1_rotatemasks[ii]) ^ (1 << ((ii - 1) & 0x7)))
 */
LOCAL unsigned short y2_rotatemasks[] = {
	0x7f7f,		/* 1 */ 
	0x0000,		/* 0 */
	0x0101,		/* 7 */
	0x0303,		/* 6 */
	0x0707,		/* 5 */
	0x0f0f,		/* 4 */
	0x1f1f,		/* 3 */
	0x3f3f,		/* 2 */
};

#ifdef notdef
/*
 * This is an array of data masks to be used on the second pass of writing
 * a character on the screen.  Even though we allow the 9th bit to be written,
 * we must make it blank.
 */
LOCAL unsigned short y_datamasks[] = {
	0x007f,		/* no bit 7 */
	0x00fe,		/* no bit 0 */
	0x00fd,		/* no bit 1 */
	0x00fb,		/* no bit 2 */
	0x00f7,		/* no bit 3 */
	0x00ef,		/* no bit 4 */
	0x00df,		/* no bit 5 */
	0x00bf,		/* no bit 6 */
};
#endif

/*
 * This is an array of masks to be used to pick out the character from 
 * the rotated bytes in apa8Cwrite_char.
 */
LOCAL unsigned short x_rotatemasks[] = {
	0x00ff,		/* 0 */ 
	0xfe01,		/* 7 */
	0xfc03,		/* 6 */
	0xf807,		/* 5 */
	0xf00f,		/* 4 */
	0xe01f,		/* 3 */
	0xc03f,		/* 2 */
	0x807f,		/* 1 */
};


/*
 * The amount to rotate a character where the cursor is.
 */
LOCAL unsigned short rotate_amt;

/*
 * Font information.
 */
#define PADCOL_WD	1
#define PADCOL_HT	2

#define MAX_COLUMNS	80
#define MAX_LINES	35
char apa8c_buffer[MAX_COLUMNS*MAX_LINES];
int apa8c_buf_pointer;

LOCAL unsigned short  font_wd,	/* Maximum width of the given font */
		       font_ht,	/* Maximum height of the given font */
		       col_wd,	/* Width of the spot in which a character
				   is placed */
		       col_ht,	/* Height of the spot in which a character
				   is placed */
		       col_MAX,	/* Number of the last column in one line.
				   Equals the number of characters on one
				   line minus one */
		       line_MAX;/* Number of the last line.  Equals the 
				   number of lines on the screen minus one */

/*
 * Array of short masks used to compute the masks to mask out the edges
 * of the destination that should not be affected.  A one means that bit
 * in the destination can be changed.  The values in left_mask_values are
 * set up to be used with the left edge and the values in right_mask_values
 * are to be used in right edge. 
LOCAL unsigned short left_mask_values[] = {
		0xffff,0x7fff,0x3fff,0x1fff,0x0fff,0x07ff,0x03ff,0x01ff,
		0x00ff,0x007f,0x003f,0x001f,0x000f,0x0007,0x0003,0x0001,
		0x0000
};

LOCAL unsigned short right_mask_values[] = {
		0xffff,0x8000,0xc000,0xe000,0xf000,0xf800,0xfc00,0xfe00,
		0xff00,0xff80,0xffc0,0xffe0,0xfff0,0xfff8,0xfffc,0xfffe,
		0x0000
};
 */

#define FONT_WD 8

#define NULL ((unsigned short *)0)

/*
 * Each element points to the bitmap address of the character used to index
 * the array.
 */
LOCAL unsigned short *bitmap_addr[SIZEOF_DPLIST];
LOCAL unsigned long line_addr[MAX_LINES];
LOCAL unsigned long col_addr[MAX_COLUMNS];

apa8Cset_fontinfo()
{
	register int i;

	/*
	 * Set up the variables for the font width and height and the width
	 * and height for each space (or "cell") where the character will be
	 * placed.
	 */
	font_wd = apa8fonthead.header.maxx;
	font_ht = apa8fonthead.header.maxy;
	col_wd = font_wd + PADCOL_WD;

	/*
	 * An odd col_ht does not work in this driver because everything has
	 * been optimized for a 9x16 bit font.
	 */
	if ((col_ht = font_ht + PADCOL_HT) & 0x1)
		printf("WARNING: THE COLUMN HEIGHT IS ODD!!\n");

	/*
	 * Figure out the max number of columns.  Center the columns on the
	 * screen.  
	 */
	if ((col_MAX = (SCREEN_WD/col_wd) - 1) > MAX_COLUMNS) {
		col_MAX = MAX_COLUMNS - 1;
	}

	/*
	 * Figure out the max number of lines and center them on the screen.
	 */
	if ((line_MAX = (SCREEN_HT/col_ht) - 1) > MAX_LINES) {
		line_MAX = MAX_LINES - 1;
	}

	/*
	 * Foreach line on the screen calculate its starting bus address.
	 */
	for (i = 0; i < (line_MAX + 1); i++) {
		line_addr[i] = APA8CBASE + SL_TO_BUS(LINE_TO_SL(i));
	}

	for (i = 0; i < (col_MAX + 1); i++) {
		col_addr[i] = BM_TO_BUS(COL_TO_BM(i));
	}

}

/*
 * Set up the fontbltstruct so that it contains the font bitmap as the source,
 * the screen as the destination, the combination rule as source copy, a
 * clipping rectangle as the whole screen and the destination rectangle
 * as the first spot for the font.
 */ 
apa8Cinit_fontblt(fontblt,fontBM)
register fontbltstruct *fontblt;
register Blt_Bitmap *fontBM;
{
	/*
	 * The source bitmap is the font bitmap.
	 */
	fontblt->src_bitmap = *fontBM;

	/*
	 * Set up the screen as the destination bitmap.
	 */
	fontblt->dst_bitmap.nshorts = DIV_BPW(BITMAP_WD);
	SETRECT(&fontblt->dst_bitmap.rect,0,0,BITMAP_WD,SCREEN_HT);

	/*
	 * Point the base at the beginning of the APA-8C's bitmap.
	 */
	fontblt->dst_bitmap.base = (unsigned short *)APA8CBASE;

	SETRECT(&fontblt->dst_rect,SCREEN_WD,0,SCREEN_WD+FONT_WD,font_ht);
}

/*
 * Increment the destination rectangle to the next spot for the next character.
 * Returns TRUE if the font will overflow into the queue area.
 */
apa8Cincr_dst_rect(dst_rect)
Blt_Rectangle *dst_rect;
{
	if ((dst_rect->corner_x + (2* FONT_WD)) > BITMAP_WD) {
		if ((dst_rect->corner_y+col_ht) > SCREEN_HT) {
			return(TRUE);
		}
		else {
			/*
			 * Go to the next line.
			 */
			SETRECT(dst_rect,
				SCREEN_WD,
				dst_rect->origin_y + col_ht,
				SCREEN_WD + FONT_WD,
				dst_rect->corner_y + col_ht);
		}
	}
	else {
		dst_rect->corner_x += FONT_WD;
		dst_rect->origin_x += FONT_WD;
	}
	return(FALSE);
}

/*
 * Loads the given font into the hidden screen area.  Keeps track of where
 * the font is put into the font area of the hidden bitmap.
 * If the font cannot be read then it returns FALSE (0) otherwise it
 * returns TRUE (1).
 */
apa8Cfont_load(font,fontdata,fontBM)
register struct font_head *font;
register unsigned short *fontdata;
register Blt_Bitmap *fontBM;
{
	register i,j;
	fontbltstruct fontblt;
	int no_room;            /* indicates whether or not there is room
				   for the next character in the font area */
	unsigned short addrused[SIZEOF_DPLIST]; /* Used to keep track of the
						  addresses visited when
						  adding font characters */

	/*
	 * Set up the write and data mask.
	 */
	SET_APA8C_WRITEMASK(NOMASK);
	SET_APA8C_DATAMASK(HIDDEN_DM);

	/*
	 * Set up the APA-8C control register to use its system read/write
	 * calls in the X direction.
	 */
	SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_SWR | DCR_X | HIDDEN_FUNC);

	/*
	 * Initialize the structure that will be passed to fontblt.
	 */
	apa8Cinit_fontblt(&fontblt,fontBM);

	/*
	 * Read in each character using the dispatch information and
	 * place it in the hidden bitmap area.
	 */
	no_room = FALSE;
	for (i = 0;(i < SIZEOF_DPLIST) && (no_room == FALSE); i++) {
		/*
		 * Keep this address to indicate that we have already dealt
		 * with the character image at this address.  This address
		 * will be compared with later addresses if a match is found
		 * then that later character will just take the information
		 * generated from the earlier character and the character
		 * will not have to be loaded into the hidden area of the
		 * bitmap.  This saves hidden bitmap space.
		 */
		addrused[i] = font->dplist[i].addr;

		/*
		 * Have we been to this address before??  If so don't
		 * load the character.
		 */
		for (j=0;  j < i && font->dplist[i].addr != addrused[j]; j++);
	
		if (j < i && font->dplist[i].addr == addrused[j]) {
			/*
	 		 * we've already loaded this font into the 
			 * bitmap go get the info from this other character.
	 		 */
			bitmap_addr[i] = bitmap_addr[j];
		}
	 	else if (font->dplist[i].addr >= font->header.size || 
			 (font->dplist[i].nbytes == 0)) {
			/*
	 		 * There is no character for this dispatch 
			 * area, NULL indicates this.
	 		 */
			if (font->dplist[i].nbytes == 0)
				addrused[i] = font->header.size;
			bitmap_addr[i] = NULL;
		}
	 	else	{
			/*
			 * Set up the source rectangle to encompass the
			 * characters bit image in the source bitmap.
			 */
			SETRECT(&fontblt.src_rect,
				0,
				font->dplist[i].addr/BTOW(font_wd),
				font_wd,
				(font->dplist[i].addr/BTOW(font_wd))+font_ht);

			/*
			 * Call font bltter to put up the character image.
			 */
			bitmap_addr[i] = apa_fontblt(&fontblt,APA8C_FONT);
			fontblt.dst_rect.origin_x += FONT_WD;
			fontblt.dst_rect.corner_x += FONT_WD;

			/*
			 * Put the same font image out there again.
			 */
			apa_fontblt(&fontblt,APA8C_FONT);

			/*
			 * Set up the destination rectangle for the next blt.
			 */
			no_room = apa8Cincr_dst_rect(&fontblt.dst_rect);
		}
	}

        if ((i != SIZEOF_DPLIST) && (no_room == TRUE))
                return(FALSE);
        else
                return(TRUE);
}

/*
 * Clear all of the APA-8C. (Including the hidden area.)
 */
apa8Cclear()
{
	register unsigned short *apa8Cscr;
	register i,j;

	apa8Cscr = (unsigned short *)APA8CBASE;
	SET_APA8C_DATAMASK(CLEAR_DM);
	SET_APA8C_WRITEMASK(NOMASK);

	/*
	 * Set up for adaptive write in x direction.
	 */
	SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | DCR_X | DEFAULT_FUNC);

	/*
	 * Clear the visible screen.
	 */
	for (i = SCREEN_HT + 1; --i ; apa8Cscr += MUL_2(HIDDEN_WORD_WD))
		for (j = SCREEN_WORD_WD + 1; --j; apa8Cscr += 2)
			*apa8Cscr = 0x0000;

	/*
	 * Clear hidden bitmap area.  NOTE: DEFAULT_FUNC may not be DCR_FCPB.
	 */
	apa8Cscr = (unsigned short *)APA8CBASE +
				    MUL_2(SCREEN_WORD_WD);
	SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | DCR_X | DCR_FCPB);

	for (i = SCREEN_HT + 1; --i ; apa8Cscr += MUL_2(SCREEN_WORD_WD))
		for (j = HIDDEN_WORD_WD + 1; --j; apa8Cscr += 2)
			*apa8Cscr = 0x0000;
	bzero(apa8c_buffer,(line_MAX+1)*(col_MAX+1));
}

apa8C_set_colors()
{
	register i;

	/*
	 * For now we only need to set index 0 (the background color which
	 * is the 1's in the bitmap.) and index 15 (the foreground color
	 * which is the 0's in the bitmap.)
	 */
	for (i = 0; i < VLT_SIZE; i++) {
		if (i == 15) {
			*(unsigned short *)VLT_REG = (unsigned short)
						     (i | FG_COLOR);
		}
		else {
			*(unsigned short *)VLT_REG = (unsigned short)
						     (i | BG_COLOR);
		}
	}
}

apa8C_init_hdwr()
{
	/*
	 * Set up the write and data mask.
	 */
	SET_APA8C_WRITEMASK(NOMASK);
	SET_APA8C_DATAMASK(HIDDEN_DM);

	/*
	 * Set up the APA-8C control register to use its system read/write
	 * calls in the X direction.
	 */
	SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_SWR | DCR_X | HIDDEN_FUNC);

	/*
	 * Set up the color plane select register and foreground/background
	 * registers.
	 */
	*(unsigned short *)CPS_REG = DEFAULT_CPS;
	*(unsigned short *)FGBG_REG = DEFAULT_FGBG;

	/*
	 * Clear the Video Look-up Table.
	 */
	apa8C_set_colors();
}

/*
 * Returns TRUE (1) if the APA-8C screen is found.
 */
apa8C_probe(addr)
unsigned short *addr;
{
	apa8C_init_hdwr();

	/*
	 * Test for the APA-8C screen.
	 */
	*(unsigned short *)addr = 0x0F0F;
	if (*(unsigned short *)addr != 0x0F0F) {
		return(FALSE);
	}
	return(TRUE);
}

/*
 * Initializes the APA-8C screen.
 */
apa8C_screen_init()
{
	RESET_CURSOR;

	/*
	 * Initialize the APA-8C.
	 */
	apa8C_init_hdwr();

	/*
	 * Clear the visible screen and the hidden memory area.
	 */
	apa8Cclear();

        /*
         * Calculate the number of columns and lines on the screen based on
         * the maximum width and height of this font.  (NOTE: Assume that
         * the font is fixed width.)
         */
        apa8Cset_fontinfo();

        /*
         * Initialize the font. (Put it in the hidden bitmap area.)
         */
        if (apa8Cfont_load(&apa8fonthead,apa8fontBits,&apa8fontBM) == FALSE) {
		printf("Can't create APA-8C font! Abort.\r\n");
                return;
        }

	/*
	 * set up where to start frame buffer (screen bitmap).
	 */
	apa8Cscr = (unsigned short *)APA8CBASE;
	apa8c_buf_pointer = 0;
	rotate_amt = 0;

	APA8CWRITE_CURSOR();
}

/*
 * NOTE: The font_ht MUST be an even number.
 */
apa8Cwrite_char(fromapa8Cscr,toapa8Cscr,func)
register unsigned short *fromapa8Cscr, *toapa8Cscr;
register unsigned short func;  /* function used to write the character
				  to the screen. */
{
	/*
	 * Set up the writemask and datamask.
	 */
	SET_APA8C_WRITEMASK(x_rotatemasks[rotate_amt]);
	SET_APA8C_DATAMASK(DEFAULT_DM);

	/*
	 * We may be in the debugger..
	 */
	*(unsigned short *)CPS_REG = DEFAULT_CPS;
	*(unsigned short *)FGBG_REG = DEFAULT_FGBG;

	/*
	 * Set up the control register to do adaptive read and write.
	 * in X direction.
	 */
	SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | DCR_X | func);
	
	/*
	 * Write out the character.
	 */
	CHAR_LOOP(toapa8Cscr,fromapa8Cscr,NEXT_SL);
}

apa8C_screen_putc(ch,attr)
register char ch;
register int attr;
{
	register unsigned short *addr;
	register unsigned short func;

	addr = bitmap_addr[ch];

	if (addr != NULL) {
		/*
		 * Put the character onto the screen where the cursor is.
		 */
		APA8CREMOVE_CURSOR();
		apa8c_buffer[apa8c_buf_pointer] = ch;

		if (attr & REVERSE_VIDEO) {
			register unsigned short *scr = (apa8Cscr + 1);

			/*
			 * Clear the 9th bit.
			 * 
			 * Set up the writemask and datamask.
			 */
			SET_APA8C_WRITEMASK(y2_rotatemasks[rotate_amt]);
			SET_APA8C_DATAMASK(CLEAR_DM);

			/*
			 * Set up the control register to do adaptive read
			 * and write in Y direction.
			 */
			func = NOTFC(DEFAULT_FUNC) | rotate_amt;
			SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | func);

			CHAR_BLANK_LOOP(scr,func,MUL_2(NEXT_SL));
		} else {
			register unsigned short *scr = (apa8Cscr + 1);

			/*
			 * Clear the 9th bit.
			 * 
			 * Set up the writemask and datamask.
			 */
			SET_APA8C_WRITEMASK(y2_rotatemasks[rotate_amt]);
			SET_APA8C_DATAMASK(CLEAR_DM);

			/*
			 * Set up the control register to do adaptive read
			 * and write in Y direction.
			 */
			func = DEFAULT_FUNC | rotate_amt;
			SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | func);

			CHAR_BLANK_LOOP(scr,func,MUL_2(NEXT_SL));
		}

		apa8Cwrite_char(addr,apa8Cscr,func);

		/*
		 * Don't bother writing the cursor back out.  The emulator
		 * is going to call pos_cursor anyway.
		 * APA8CWRITE_CURSOR();
		 */
	}
}

apa8C_pos_cursor(column_number,line_number)
register int column_number;
register int line_number;
{
	APA8CREMOVE_CURSOR();
	apa8Cscr = (unsigned short *)(LINE_TO_BUS(line_number) +
						 COL_TO_BUS(column_number));
	apa8c_buf_pointer = (line_number*(col_MAX+1)) + column_number;

	rotate_amt = COL_TO_ROTATE(column_number);

	/*
	 * removed the cursor now write it back out.
	 */
	APA8CWRITE_CURSOR();
}

/*
 * On a given line (logical line usually numbering 0 to 28) clear all the
 * scan lines between the first and last bytes (using func) inclusive.
 */
apa8Cclearline(first,last,line,func)
register int first;	/* first byte to be cleared */
int last;	 	/* last byte to be cleared */
int line;		/* line at which to start */
unsigned short func; /* the function used to clear the screen */
{
	register unsigned short *tmp_apa8Cscr;
	register i,j,width;

	tmp_apa8Cscr = (unsigned short *)(LINE_TO_BUS(line) + BM_TO_BUS(first));
	width = last - first + 1;

	/*
	 * Set up the control register to do adaptive write in Y direction.
	 * Because we are going in the Y direction, we want to move 2
	 * scanlines down (hence we use MUL_2).
	 */
	SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | func);
	for (i = DIV_2(col_ht) + 1; --i; ) {
		for (j = width + 1; --j; tmp_apa8Cscr++ ) {
			*tmp_apa8Cscr = i;
		}
		tmp_apa8Cscr += (MUL_2(NEXT_SL) - width);
	}
}


apa8C_screen_blank(attr,sline,scol,eline,ecol)
int attr;   /* attribute of how to clear the screen (i.e. reverse video, etc) */int sline,scol; /* The starting line and column */
int eline,ecol; /* The ending line and column */
{
	register unsigned short *tmp_apa8Cscr;
	register i;
	register sbyte,ebyte;
	unsigned short func;

	APA8CREMOVE_CURSOR();
	lp_screen_blank(apa8c_buffer,sline,scol,eline,ecol,CONS_APA8C);

	if (attr & NORMAL_VIDEO) {
		func = DEFAULT_FUNC;
	}
	else if (attr & REVERSE_VIDEO) {
		func = NOTFC(DEFAULT_FUNC);
	}

	/*
	 * Set up the datamask to clear parts of the screen.
	 */
	SET_APA8C_DATAMASK(CLEAR_DM);

	/*
	 * Special case the easy loop where scol is the first column and
	 * ecol is the last column.
	 */
	if (scol == FIRSTCOL && ecol == col_MAX) {
		SET_APA8C_WRITEMASK(NOMASK);
		/*
		 * Clear from the start line to the end line.
		 */
		tmp_apa8Cscr = (unsigned short *)LINE_TO_BUS(sline);

		/*
		 * Set up the control register to do adaptive write 
		 * in X direction.
		 */
		SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | DCR_X | func);
		for (i = ((eline - sline + 1) * col_ht) + 1; --i; ) {
			BLANK_LOOP(tmp_apa8Cscr,i,2);
			tmp_apa8Cscr += NEXT_SL;
		}

		return;
	}

	/*
	 * Right now we are dealing with logical columns which will span
	 * 2 bytes.  Clear the first partial byte and the last partial byte
	 * in the first and last line.  Then deal with clearing the screen
	 * in terms of bytes.
	 */

	/*
	 * Calculate the starting byte.
	 */
	sbyte = COL_TO_BM(scol);

	/*
	 * If the starting column is not the first column and the start column
	 * does not fall on a byte boundary then clear part of of the first
	 * byte.
	 */
	if (scol > FIRSTCOL && MOD_BPB(scol*col_wd)) {
		/*
		 * Set up the writemask with the mask for the first half of
		 * the character.
		 */
		SET_APA8C_WRITEMASK(y1_rotatemasks[COL_TO_ROTATE(scol)]);
		tmp_apa8Cscr = (unsigned short *)(LINE_TO_BUS(sline) +
						 COL_TO_BUS(scol));

		/*
		 * Set up the control register to do adaptive write 
		 * in Y direction.  Because we are going in the Y direction,
		 * we want to move 2 scanlines down (hence we use MUL_2).
		 */
		SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | func);
		for (i = (col_ht >> 1) + 1; --i ; ) {
			*tmp_apa8Cscr = i;
			tmp_apa8Cscr += MUL_2(NEXT_SL);
		}

		/*
		 * Finished with this byte move forward to the next one.
		 */
		sbyte += 1;
	}

	/*
	 * Calculate the byte boundrary for the last bit in the end column.
	 */
	ebyte = COL_TO_BM(ecol) + 1;

	/*
	 * If the end columns given is not the last column and the next
	 * column after the end column doesn't start on a NEW byte then
	 * only part of that byte should be cleared so we clear it here.
	 */
	if (ecol < col_MAX && MOD_BPB((ecol+1)*col_wd)) {
		/*
		 * Set up the writemask with the mask for the second half of
		 * the character. (The first half will be cleared below.)
		 */
		SET_APA8C_WRITEMASK(y2_rotatemasks[COL_TO_ROTATE(ecol)]);
		tmp_apa8Cscr = (unsigned short *)(LINE_TO_BUS(eline) +
						 COL_TO_BUS(ecol) + 2);

		/*
		 * Set up the control register to do adaptive write 
		 * in Y direction.  Because we are going in the Y direction,
		 * we want to move 2 scanlines down (hence we use MUL_2).
		 */
		SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | func);
		for (i = (col_ht >> 1) + 1; --i ; ) {
			*tmp_apa8Cscr = i;
			tmp_apa8Cscr += MUL_2(NEXT_SL);
		}

		/*
		 * Finished with this byte move back.
		 */
		ebyte -= 1;
	}

	SET_APA8C_WRITEMASK(NOMASK);

	if ((scol > FIRSTCOL) && (sline < eline)) {
		/*
		 * Clear the first partial line which is from start 
		 * byte figured above, to the end of the line.
		 */
		apa8Cclearline(sbyte,SCREEN_BYTE_WD,sline,func);

		/*
		 * Cleared this line.
		 */
		sbyte = 0;
		scol = FIRSTCOL;
		sline += 1;
	}

	if ((ecol < col_MAX) && (sline < eline)) {
		/*
		 * clear from first byte in a scanline to the ebyte figured
		 * above.
		 */
		apa8Cclearline(0,ebyte,eline,func);

		/*
		 * Cleared this line.
		 */
		ebyte = SCREEN_BYTE_WD;
		ecol = col_MAX;
		eline -= 1;
	}

	/*
	 * Clear the area inbetween sline and eline.
	 */
	if (sline == eline) {
		if (ebyte >= sbyte)
			apa8Cclearline(sbyte,ebyte,sline,func);
	}
	else if (ecol == col_MAX && scol == FIRSTCOL) {
		/*
		 * Clear from the start line to the end line.
		 */
		tmp_apa8Cscr = (unsigned short *)LINE_TO_BUS(sline);

		/*
		 * Set up the control register to do adaptive write 
		 * in X direction.
		 */
		SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | DCR_X | func);
		for (i = ((eline - sline + 1) * col_ht) + 1; --i; ) {
			BLANK_LOOP(tmp_apa8Cscr,i,2);
			tmp_apa8Cscr += NEXT_SL;
		}
	}

	/*
	 * Don't bother writing the cursor out because the emulator will be
	 * calling pos_cursor anyway.
	 * APA8CWRITE_CURSOR();
	 */
}

/*
 * scroll_down takes the block of lines starting at start_line and ending
 * at the bottom of the screen minus n_scrolls number of lines, and moves them 
 * down to the end of the screen.
 * For example, 
 *	Before:	1. XYZ
 *		2. ABC
 *		3. FGH
 *
 *	After scrolling 2 lines down:
 *		~
 *		~
 *		1. XYZ
 */

apa8Cscroll_down(src_line,dst_line,nlines)
int src_line;	/* Line number to copy from */
int dst_line;	/* Line number to copy to */
int nlines;	/* Number of lines to be copied */
{
	register unsigned short *tmp_to;   /* bitmap pointer to destination. */
	register unsigned short *tmp_from; /* bitmap pointer to source. */
	register i;	   /* loop counter for scanlines and bytes per
			      scanline */

	/*
	 * Note the difference between SCAN lines (0 - 512)
	 * and lines. (0 - LINE_MAX)
	 */
	tmp_from = (unsigned short *)(LINE_TO_BUS(src_line + nlines) - 
				      MUL_2(NEXT_SL));

	/*
	 * Set up the destination bitmap pointer to be at the last line,
	 * col_ht number of SCAN lines down.
	 */
	tmp_to = (unsigned short *)(LINE_TO_BUS(dst_line + nlines) -
				    MUL_2(NEXT_SL));

	SET_APA8C_WRITEMASK(NOMASK);
	SET_APA8C_DATAMASK(DEFAULT_DM);

	/*
	 * set up the control register to adaptive read/write in the X
	 * direction using the default function so as not to change anything.
	 */
	SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_X | DCR_ADWR | DCR_FCPB); 

	/*
	 * Set up for the number of scanlines to be moved. (Plus 2 because:
	 * the while loop immediately subtracts 1 and tmp_from INCLUDES the
	 * the line to be copied.)
	 */
	i = (nlines * col_ht) + 1;

	/*
	 * foreach scan line do...
	 */
	while (--i) {
		/*
		 * foreach word in the width of the visible screen do...
		 */ 
		/*
		 * SCREEN_LOOP is 45 of these instructions in a row, a
		 * totally unrolled loop.
		 */
		SCREEN_LOOP(tmp_to,tmp_from,2);

		/*
		 * Go to the next scan line above.
		 */
		tmp_to -= NEXT_SL;
		tmp_from -= NEXT_SL;
	}
}

/*
 * scroll_up takes the block of lines starting n_scrolls number of lines below
 * start_line and ending at the bottom of the screen, and moves them up to
 * start_line.
 * For example, 
 *	Before:	1. XYZ
 *		2. ABC
 *		3. FGH
 *
 *	After scrolling 2 lines up:
 *		3. FGH
 *		~
 *		~
 */

apa8Cscroll_up(src_line,dst_line,nlines)
int src_line;	/* Line number to copy from */
int dst_line;	/* Line number to copy to */
int nlines;	/* Number of lines to be copied */
{
	register unsigned short *tmp_to;   /* bitmap pointer to destination. */
	register unsigned short *tmp_from; /* bitmap pointer to source. */
	register i;	   /* loop counter for scanlines and bytes per
			      scanline */

	/*
	 * Set up the destination bitmap pointer to be the starting scanline.
	 */
	tmp_to = (unsigned short *)LINE_TO_BUS(dst_line);

	/*
	 * Set up the source bitmap pointer to be n_scrolls LINEs in front
	 * of the  destination bitmap pointer.
	 * Note:  The difference between SCAN lines (0 - 512) and LINEs.
	 * (0 - line_MAX)
	 */
	tmp_from = (unsigned short *)LINE_TO_BUS(src_line);

	/*
	 * Set up the data and writemask.
	 */
	SET_APA8C_WRITEMASK(NOMASK);
	SET_APA8C_DATAMASK(DEFAULT_DM);

	/*
	 * Set up for the number of scanlines to be moved.
	 * NOTE: +1 because the while loop immediately subtracts 1.
	 */
	i =  (nlines * col_ht) + 1;

	/*
	 * set up the control register to adaptive read/write in the X
	 * direction using the default function so as not to change anything.
	 */
	SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_X | DCR_ADWR | DCR_FCPB); 

	/*
	 * foreach scan line do...
	 */
	while (--i) {
		/*
		 * foreach word in each scanline do...
		 */
		/*
		 * SCREEN_LOOP is 45 of these instructions in a row, a
		 * totally unrolled loop.
		 */
		SCREEN_LOOP(tmp_to,tmp_from,2);
		/*
		 * Go to the next scan line below.
		 */
		tmp_to += NEXT_SL;
		tmp_from += NEXT_SL;
	}
}

apa8C_screen_move(line_top,line_bottom,line_dest)
long line_top,line_bottom,line_dest;
{
	/*
	 * Remove the cursor before doing any scrolling.
	 */
	APA8CREMOVE_CURSOR();
	lp_screen_move(apa8c_buffer,line_top,line_bottom,line_dest,CONS_APA8C);

	/*
	 * Decide which way we are scrolling.
	 */
	*(unsigned short *)CPS_REG = MOVE_CPS;
	if (line_top < line_dest) {
		apa8Cscroll_down(line_top,line_dest,line_bottom-line_top+1);
	}
	else if (line_top > line_dest) {
		apa8Cscroll_up(line_top,line_dest,line_bottom-line_top+1);
	}
	*(unsigned short *)CPS_REG = DEFAULT_CPS;

	/*
	 * Don't bother writing the cursor out because the emulator will
	 * be calling pos_cursor anyway.
	 * APA8CWRITE_CURSOR();
	 */
}

apa8C_screen_print(si,flags)
	register SCREEN_INFO *si;
{
	lp_screen_print(apa8c_buffer,si,flags,CONS_APA8C);
}

#if NAPAEIGHT <= 0
/*
 * The follow routines used to be unrolled macros for speed. However, space
 * is the limitting factor in standalone, so the are now subroutines.
 */
/*
 * NOTE: the value of "reg" does not matter because all we want to do is
 *       call the to[] address.
 */
CHAR_BLANK_LOOP(to,reg,incr)
	unsigned short *to,reg;
	int	incr;
{
	int i;

	for (i=0; i < 8; i++, to +=incr) {
		*to = reg;
	}
}

CHAR_LOOP(to,from,incr) 
	unsigned short *to,*from;
	int incr;
{
	int i;
	for (i=0; i < 16; i++, to +=incr, from +=incr) {
		*to = *from;
	}
}

/*
 * Unrolled loop for across the APA-8 visible screen.
 */
SCREEN_LOOP(to,from,incr) 
	unsigned short *to, *from;
	int incr;
{
	int i;

	for (i=0; i < 45 ;i++, to += incr, from += incr) {
		*to = *from;
	}
}

BLANK_LOOP(to,reg,incr)
	unsigned short *to,reg;
	int incr;
{
	for (i=0; i < 45; i++, to +=incr) {
		*to = reg;
	}
}
#endif NAPAETTY

APA8C_XOR_CURSOR(mask,scr_ptr) 
	register unsigned short *scr_ptr;
	short mask;
{
	int i;
	SET_APA8C_WRITEMASK(mask);
	SET_APA8C_DATAMASK(DEFAULT_DM);
	SET_APA8C_FC(DCR_VEN | DCR_SEN | DCR_ADWR | DCR_X | DCR_FCPNB);
	for (i=0; i < 16; i++, scr_ptr+=NEXT_SL) {
		*scr_ptr = *scr_ptr;
	}
}

/*
 * Removing and writing the cursor functions are now macros.
 */
APA8CREMOVE_CURSOR()
{
	if (CURSOR_ON) {
		RESET_CURSOR;
		APA8C_XOR_CURSOR(char_mask[rotate_amt],apa8Cscr);
	}
}

APA8CWRITE_CURSOR()	{
	if (CURSOR_OFF) {
		SET_CURSOR;
		APA8C_XOR_CURSOR(char_mask[rotate_amt],apa8Cscr);
	}
}
#endif NAPAETTYC
