/* 
 * Mach Operating System
 * Copyright (c) 1988 Carnegie-Mellon University
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 *	File:	vm_object.h
 *	Author:	Avadis Tevanian, Jr., Michael Wayne Young
 *
 *	Copyright (C) 1985, Avadis Tevanian, Jr., Michael Wayne Young
 *
 *	Virtual memory object module definitions.
 *
 * HISTORY
 * $Log:	vm_object.h,v $
 * Revision 2.7  89/01/15  16:42:44  rpd
 * 	Use decl_simple_lock_data, simple_lock_addr.
 * 	[89/01/15  15:33:00  rpd]
 * 
 * Revision 2.6  88/12/19  03:00:38  mwyoung
 * 	Use MACRO_BEGIN, MACRO_END.  Remove lint.
 * 	[88/12/17            mwyoung]
 * 
 * Revision 2.5  88/10/18  03:45:42  mwyoung
 * 	The object cache lock is not exported either.
 * 	[88/10/15            mwyoung]
 * 	
 * 	Remove dependency on wait_result field in vm_object_paging_wait.
 * 	The interruptible case is now bogus.
 * 	[88/10/15            mwyoung]
 * 	
 * 	Documented the vm_object fields.  Moved the object hash
 * 	structures to vm/vm_object.c, as they're non-XP
 * 	implementation-specific.  Move the object list and cache
 * 	list queues, as they're also implementation-specific.
 * 	Declare the remaining variables as external, to avoid
 * 	wasteful "common" symbol references.
 * 	[88/09/22            mwyoung]
 * 	
 * 	Add macros for beginning and ending paging operations.
 * 	[88/09/22            mwyoung]
 * 	
 * 	Add debugging field to verify that termination proceeds
 * 	correctly.  Remove more old history.
 * 	[88/09/16            mwyoung]
 * 
 * Revision 2.4  88/08/25  18:29:38  mwyoung
 * 	Add vm_object_t->pager_initialized.  Distinguish between the
 * 	internal initialization (now indicated by "initialized") and the
 * 	memory manager's willingness to handle request (still "ready").
 * 	[88/08/16  04:23:04  mwyoung]
 * 	
 * 	Split out "temporary" attribute from "internal" object attribute.
 * 	[88/08/11  18:58:58  mwyoung]
 * 
 * Revision 2.2.1.3  88/07/11  13:35:25  mwyoung
 * See below.
 * 
 * Revision 2.2.1.2  88/07/04  16:48:58  mwyoung
 * Added vm_object_t->copy_strategy field.  Use <vm/memory_object.h>
 * definitions.
 * 
 * Revision 2.2.1.1  88/06/28  21:10:19  mwyoung
 * Added vm_object_t->wanted field.
 * 
 */

#ifndef	_VM_OBJECT_
#define	_VM_OBJECT_

#ifdef	KERNEL
#include <mach_xp.h>
#endif	KERNEL

#include <sys/queue.h>
#include <sys/lock.h>
#include <sys/boolean.h>
#include <vm/memory_object.h>
#include <sys/port.h>
#include <machine/vm_types.h>

#include <sys/assert.h>

/*
 *	Types defined:
 *
 *	vm_object_t		Virtual memory object.
 */

struct vm_object {
	queue_chain_t		memq;		/* Resident memory */
	queue_chain_t		object_list;	/* list of all objects */
	decl_simple_lock_data(,	Lock)		/* Synchronization */
	int			LockHolder;	/* Thread holding Lock */

	int			ref_count;	/* Number of references */
	vm_size_t		size;		/* Object size (only valid
						 * if internal)
						 */

	int			resident_page_count;
						/* number of resident pages */

	struct vm_object	*copy;		/* Object that should receive
						 * a copy of my changed pages
						 */
	struct vm_object	*shadow;	/* My shadow */
	vm_offset_t		shadow_offset;	/* Offset into shadow */

	memory_object_t		pager;		/* Where to get data */
	vm_offset_t		paging_offset;	/* Offset into memory object */
	memory_object_control_t	pager_request;	/* Where data comes back */
	memory_object_name_t	pager_name;	/* How to identify region */

	unsigned int
	/* boolean_t */		pager_created:1,/* Has pager ever been created? */
	/* boolean_t */		pager_initialized:1,/* Are fields ready to use? */
	/* boolean_t */		pager_ready:1;	/* Will manager take requests? */

	memory_object_copy_strategy_t
				copy_strategy;	/* How to handle data copy */

	unsigned int
				paging_in_progress:16,
						/* The memory object ports are
						 * being used (e.g., for pagein
						 * or pageout) -- don't change any
						 * of these fields (i.e., don't
						 * collapse, destroy or terminate)
						 */
	/* boolean_t */		wanted:1,	/* Some thread is waiting for
						 * (paging_in_progress == 0)?
						 */
	/* boolean_t */		can_persist:1,	/* The kernel may keep the data
						 * for this object (and rights to
						 * the memory object) after all
						 * address map references are
						 * deallocated?
						 */
	/* boolean_t */		internal:1,	/* Created by the kernel (and
						 * therefore, managed by the
						 * default memory manger)
						 */
	/* boolean_t */		temporary:1,	/* Permanent objects may be changed
						 * externally by the memory manager,
						 * and changes made in memory must
						 * be reflected back to the memory
						 * manager.  Temporary objects lack
						 * both of these characteristics.
						 */
	/* boolean_t */		alive:1,	/* Not yet terminated (debug) */
	/* boolean_t */		single_use:1;	/* Does only one address map
						 * have access to this object?
						 * [reserved for future use]
						 */

	queue_chain_t		cached_list;	/* Attachment point for the list
						 * of objects cached as a result
						 * of their can_persist value
						 */
};

typedef struct vm_object	*vm_object_t;
#define	VM_OBJECT_NULL		((vm_object_t) 0)

#ifdef	KERNEL
extern
vm_object_t	kernel_object;		/* the single kernel object */

/*
 *	Declare procedures that operate on VM objects.
 */

void		vm_object_init ();
void		vm_object_terminate();
vm_object_t	vm_object_allocate();
void		vm_object_reference();
void		vm_object_deallocate();
void		vm_object_pmap_copy();
void		vm_object_pmap_remove();
void		vm_object_page_remove();
void		vm_object_shadow();
void		vm_object_copy();
void		vm_object_collapse();
vm_object_t	vm_object_lookup();
port_t		vm_object_name();
void		vm_object_remove();
#if	MACH_XP
vm_object_t	vm_object_enter();
void		vm_object_pager_create();
void		vm_object_destroy();
#else	MACH_XP
void		vm_object_enter();
void		vm_object_setpager();
#endif	MACH_XP

void		vm_object_print();

port_t		vm_object_request_port();
vm_object_t	vm_object_request_object();

/*
 *	Routines implemented as macros
 */

#define		vm_object_paging_begin(object) 					\
	(object)->paging_in_progress++

#define		vm_object_paging_end(object) 				\
	MACRO_BEGIN							\
	assert((object)->paging_in_progress != 0);			\
	if ((--(object)->paging_in_progress == 0) && (object)->wanted) {\
		thread_wakeup((int) (object));				\
		(object)->wanted = FALSE;				\
	}								\
	MACRO_END

#define		vm_object_paging_wait(object, interruptible)		\
	MACRO_BEGIN							\
	while ((object)->paging_in_progress != 0) {			\
		(object)->wanted = TRUE;				\
		vm_object_sleep((int) (object), (object), (interruptible));\
		vm_object_lock(object);					\
									\
		/*XXX if ((interruptible) &&	*/			\
		    /*XXX (current_thread()->wait_result != THREAD_AWAKENED))*/ \
			/*XXX break; */					\
	}								\
	MACRO_END


/*
 *	Object locking macros (with and without debugging)
 */

#if	VM_OBJECT_DEBUG
#define	vm_object_lock_init(object)	MACRO_BEGIN simple_lock_init(&(object)->Lock); (object)->LockHolder = 0; MACRO_END
#define	vm_object_lock(object)		MACRO_BEGIN simple_lock(&(object)->Lock); (object)->LockHolder = (int) current_thread(); MACRO_END
#define	vm_object_unlock(object)	MACRO_BEGIN (object)->LockHolder = 0; simple_unlock(&(object)->Lock); MACRO_END
#define	vm_object_lock_try(object)	(simple_lock_try(&(object)->Lock) ? ( ((object)->LockHolder = (int) current_thread()) , TRUE) : FALSE)
#define	vm_object_sleep(event, object, interruptible) \
					MACRO_BEGIN (object)->LockHolder = 0; thread_sleep((event), simple_lock_addr((object)->Lock), (interruptible)); MACRO_END
#else	VM_OBJECT_DEBUG
#define	vm_object_lock_init(object)	simple_lock_init(&(object)->Lock)
#define	vm_object_lock(object)		simple_lock(&(object)->Lock)
#define	vm_object_unlock(object)	simple_unlock(&(object)->Lock)
#define	vm_object_lock_try(object)	simple_lock_try(&(object)->Lock)
#define	vm_object_sleep(event, object, interruptible) \
					thread_sleep((event), simple_lock_addr((object)->Lock), (interruptible))
#endif	VM_OBJECT_DEBUG
#endif	KERNEL

#endif	_VM_OBJECT_
