/* 
 * Mach Operating System
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 *	File:	vm/vm_pageout.c
 *	Author:	Avadis Tevanian, Jr., Michael Wayne Young
 *
 *	Copyright (C) 1985, Avadis Tevanian, Jr., Michael Wayne Young
 *
 *	The proverbial page-out daemon.
 *
 * HISTORY
 * $Log:	vm_pageout.c,v $
 * Revision 2.9  89/01/15  16:43:25  rpd
 * 	Updated includes for the new mach/ directory.
 * 	Use simple_lock_addr.
 * 	[89/01/15  15:34:22  rpd]
 * 
 * Revision 2.8  88/12/19  03:01:11  mwyoung
 * 	Wake up anyone waiting for a page being cleaned.
 * 	[88/12/09            mwyoung]
 * 	
 * 	Use dirty bits, not clean words.
 * 	[88/12/09            mwyoung]
 * 	
 * 	Use symbolic name for XPR tag.
 * 	[88/11/22            mwyoung]
 * 
 * Revision 2.7  88/08/25  18:30:06  mwyoung
 * 	Corrected include file references.
 * 	[88/08/22            mwyoung]
 * 	
 * 	Must assert_wait before using thread_set_timeout.
 * 	[88/08/18            mwyoung]
 * 
 * Revision 2.4.1.1  88/07/24  14:38:06  mwyoung
 * Don't port_copyout the memory_manager_default port to the kernel_task.
 * 
 * Revision 2.6  88/08/06  19:26:38  rpd
 * Eliminated use of kern/mach_ipc_defs.h.
 * 
 * Revision 2.5  88/07/29  03:21:51  rpd
 * Removed erroneous port_copyout of memory_manager_default.
 * 
 * Revision 2.4  88/07/17  19:30:44  mwyoung
 * *** empty log message ***
 * 
 * Revision 2.3.1.3  88/07/07  10:37:27  mwyoung
 * Prevent flooding of the default memory manager.
 * 
 * Revision 2.3.1.2  88/07/04  16:53:19  mwyoung
 * Use new memory_object types.
 * 
 * Revision 2.3.1.1  88/06/28  21:13:22  mwyoung
 * Split up use of kernel_only field.
 * 
 * Use vm_object_t->wanted flag to trigger wakeup when
 * paging_in_progress gets to zero.
 * 
 * New pageout daemon for MACH_XP.  Old one went to "vm/vm_pageout.c".
 * 
 *
 * 21-Jun-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Split up use of kernel_only field.
 *
 * 12-Jun-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Use vm_object_t->wanted flag to trigger wakeup when
 *	paging_in_progress gets to zero.
 *
 * 11-Apr-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	MACH_XP: Artificially wire down pages sent to the default pager.
 *	Make sure that vm_pageout_page is called with the page removed
 *	from pageout queues.
 *
 * 29-Mar-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Prevent double pageout of pages belonging to internal objects.
 *
 *  8-Mar-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	MACH_XP: Add "initial" argument to vm_pageout_page().
 *
 * 29-Feb-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Use vm_object_pager_create() to create and bind a pager to an
 *	internal object.
 *
 * 24-Feb-88  David Golub (dbg) at Carnegie-Mellon University
 *	Handle IO errors on paging (non-XP only).
 *
 * 18-Jan-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Removed "max_flush" argument from vm_pageout_page().
 *	Eliminated old history.
 *
 *  6-Jan-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Allow more parameters to be set before boot.
 *
 * 18-Nov-87  Michael Young (mwyoung) at Carnegie-Mellon University
 *	MACH_XP: Changed error message handling in vm_pageout_page().
 *
 */

#include <sys/xpr.h>

#include <mach_xp.h>

#if	MACH_XP
#include <vm/vm_page.h>
#include <vm/pmap.h>
#include <vm/vm_object.h>
#include <vm/vm_pageout.h>
#include <vm/vm_statistics.h>
#include <vm/vm_param.h>
#include <sys/thread_swap.h>
#include <sys/kern_port.h>		/* For PORT_OBJECT_PAGING_REQUEST */
#include <kern/ipc_pobj.h>
#include <mach/mach_types.h>
#include <mach/memory_object.h>
#include <mach/memory_object_default.h>

int	vm_pageout_debug = 0;
int	vm_pageout_double_laundry_count = 0;

int	vm_pageout_internal_only = 0;

int	vm_pageout_burst_max = 10;
int	vm_pageout_burst_wait = 5;

/*
 *	Routine:	vm_pageout_page
 *	Purpose:
 *		Causes the specified page to be written back to
 *		its pager.  The page in question is not freed here,
 *		but will be freed by the pager when it is done.
 *		The page will be written regardless whether it
 *		is marked as clean.
 *
 *		The "initial" argument specifies whether this
 *		data is an initialization only, and should use
 *		memory_object_data_initialize instead of memory_object_data_write.
 *
 *	In/out conditions:
 *		The page in question must not be on any pageout queues.
 *		The object to which it belongs must be locked.
 *	Implementation:
 *		Move this page to a completely new object.
 */
void		vm_pageout_page(m, initial)
	vm_page_t	m;
	boolean_t	initial;
{
	vm_object_t	new_object;
	vm_object_t	old_object = m->object;
	vm_offset_t	paging_offset = m->offset + old_object->paging_offset;
	kern_return_t	rc;
	auto
	struct vm_page	holding_page;

	XPR(XPR_VM_PAGEOUT, ("vm_pageout_page: begin %x %x", m, initial));
	if (m->busy)
		panic("vm_pageout_page: busy page");

	/*
	 *	In order to create a new object, we must unlock
	 *	the old object; before doing so, we mark this page
	 *	busy (to prevent pagein) and increment paging_in_progress
	 *	(to prevent object termination).
	 */

	old_object->paging_in_progress++;

	/*
	 *	Create a place-holder page where the old one was, to prevent
	 *	anyone from attempting to page in this page while we've unlocked.
	 */

	vm_page_remove(m);
	PAGE_WAKEUP(m);

	vm_page_init(&holding_page, old_object, m->offset, (vm_offset_t) 0);
	holding_page.fictitious = TRUE;

	vm_object_unlock(old_object);

	/*
	 *	Allocate a new object into which we can put the page.
	 *
	 *	If the old object was internal, then we shouldn't
	 *	ever page this out.
	 */

	new_object = vm_object_allocate(PAGE_SIZE);
	vm_object_lock(new_object);

	/*
	 *	Move this page into to the new object
	 */

	vm_page_insert(m, new_object, 0);
	m->clean = FALSE;
	m->dirty = TRUE;

	/*
	 *	Put the old page on the pageout queues; if a bogus
	 *	user-supplied pager fails to release the page, it will
	 *	get paged out again to the default pager.
	 */

	vm_page_lock_queues();
	vm_stat.pageouts++;
	if (old_object->internal)
		m->wire_count++;
	 else
		vm_page_activate(m);
	vm_page_unlock_queues();

	/*
	 *	Mark the page as in the laundry, so we can detect
	 *	pages which aren't released, and so we can decide
	 *	when to stop the pageout scan.
	 */

	if (m->laundry) {
		if (vm_pageout_debug)
			printf("vm_pageout_page: double laundry, object 0x%x offset 0x%x page 0x%x physical 0x%x!\n",
			       old_object, paging_offset, m, VM_PAGE_TO_PHYS(m));
		vm_pageout_double_laundry_count++;
	}
	 else {
		vm_page_laundry_count++;
		m->laundry = TRUE;
	}
	
	/*
	 *	Since IPC operations may block, we drop locks now.
	 *	[The placeholder page is busy, and we still have paging_in_progress
	 *	incremented.]
	 */

	vm_object_unlock(new_object);

	/*
	 *	Write the data to its pager.
	 *	Note that the data is passed by naming the new object,
	 *	not a virtual address; the pager interface has been
	 *	manipulated to use the "internal memory" data type.
	 *	[The object reference from its allocation is donated
	 *	to the eventual recipient.]
	 */

	if ((rc = (initial ? memory_object_data_initialize : memory_object_data_write)
			(old_object->pager,
			 old_object->pager_request,
			 paging_offset, (pointer_t) new_object, PAGE_SIZE))
					!= KERN_SUCCESS) {
		if (vm_pageout_debug)
			printf("vm_pageout_page: memory_object_data_write failed, %d, page 0x%x\n", rc, m);
		vm_object_deallocate(new_object);
	}

	/*
	 *	Pick back up the old object lock to clean up
	 *	and to meet exit conditions.
	 */

	vm_object_lock(old_object);
	
	/*
	 *	Free the placeholder page to permit pageins to continue
	 *	[Don't need to hold queues lock, since this page should
	 *	never get on the pageout queues!]
	 */
	vm_page_free(&holding_page);

	if ((--old_object->paging_in_progress == 0) && old_object->wanted) {
		thread_wakeup((int) old_object);
		old_object->wanted = FALSE;
	}
	XPR(XPR_VM_PAGEOUT, ("vm_pageout_page: done %x %x", m, initial));
}

/*
 *	vm_pageout_scan does the dirty work for the pageout daemon.
 */
void		vm_pageout_scan()
{
	register vm_page_t	m;
	int		page_shortage;
	int		pages_moved;
	int		burst_count;
	register
	int		s;
	int		free;

	/*
	 *	Only continue when we want more pages to be "free"
	 */

	s = splimp();
	simple_lock(&vm_page_queue_free_lock);
	free = vm_page_free_count;
	simple_unlock(&vm_page_queue_free_lock);
	splx(s);

	if (free < vm_page_free_target) {
		swapout_threads();

		/*
		 *	Be sure the pmap system is updated so
		 *	we can scan the inactive queue.
		 */

		pmap_update();
	}

	/*
	 *	Start scanning the inactive queue for pages we can free.
	 *	We keep scanning until we have enough free pages or
	 *	we have scanned through the entire queue.  If we
	 *	encounter dirty pages, we start cleaning them.
	 *
	 *	NOTE:	The page queues lock is not held at loop entry,
	 *		but *is* held upon loop exit.
	 */

	pages_moved = burst_count = 0;

	while (TRUE) {
		register
		vm_object_t	object;

		vm_page_lock_queues();

		m = (vm_page_t) queue_first(&vm_page_queue_inactive);
		if (queue_end(&vm_page_queue_inactive, (queue_entry_t) m))
			break;

		/*
		 *	If there are enough free pages, stop anyway.
		 */

		s = splimp();
		simple_lock(&vm_page_queue_free_lock);
		free = vm_page_free_count;
		simple_unlock(&vm_page_queue_free_lock);
		splx(s);

		if (free >= vm_page_free_target)
			break;

		/*
		 *	If we've moved a bunch of pages, pause to let
		 *	the memory manager(s) catch up.  This should
		 *	be dynamically tuned, but isn't yet.
		 */

		if (burst_count >= vm_pageout_burst_max) {
			assert_wait(0, TRUE);
			thread_set_timeout(vm_pageout_burst_wait);
			vm_page_unlock_queues();
			thread_block();
			burst_count = 0;
			continue;
		}

		/*
		 */

		object = m->object;

		/*
		 *	If we're extremely tight on pages, we can only
		 *	send pages to the (trusted) default pager.
		 */

		if (free < vm_pageout_internal_only) {
			if (!object->internal) {
				/*
				 *	[Don't count this as a reactivation.]
				 */

				vm_page_activate(m);
				vm_page_unlock_queues();
				continue;
			}

			/*
			 *	Even then, only work until we get
			 *	to the reserved count.
			 */

			if (free < vm_page_free_reserved) {
				/* XXX If the burst_wait is computed
				 * dynamically, it would probably be
				 * wise to change this one to a fixed
				 * timeout
				 */
				assert_wait(0, TRUE);
				thread_set_timeout(vm_pageout_burst_wait);
				vm_page_unlock_queues();
				thread_block();
				continue;
			}
		}

		/*
		 *	If it's being used, merely reactivate.
		 */

		if (pmap_is_referenced(VM_PAGE_TO_PHYS(m))) {
			vm_page_activate(m);
			vm_stat.reactivations++;
			vm_page_unlock_queues();
			continue;
		}

		/*
		 *	Try to lock object; since we've got the
		 *	page queues lock, we can only try for this one.
		 */
		if (!vm_object_lock_try(object)) {
			/*
			 *	Move page to end and continue.
			 */
			queue_remove(&vm_page_queue_inactive, m, vm_page_t, pageq);
			queue_enter(&vm_page_queue_inactive, m,	vm_page_t, pageq);
			vm_page_unlock_queues();
			continue;
		}

		/*
		 *	Remove the page from the inactive list.
		 */

		queue_remove(&vm_page_queue_inactive, m, vm_page_t, pageq);
		vm_page_inactive_count--;
		m->inactive = FALSE;
		m->busy = TRUE;

		/*
		 *	Eliminate all mappings
		 */

		pmap_remove_all(VM_PAGE_TO_PHYS(m));
		if (!m->dirty)
			m->dirty = pmap_is_modified(VM_PAGE_TO_PHYS(m));

		/*
		 *	If it's clean, we can merely free the page.
		 */

		if (!m->dirty) {
			vm_page_free(m);
			pages_moved++;
			vm_object_unlock(object);
			vm_page_unlock_queues();
			continue;
		}

		vm_page_unlock_queues();

		/*
		 *	If there is no pager for the page, create
		 *	one and hand it to the default pager.
		 *	[First try to collapse, so we don't make
		 *	a pager unnecessarily.]
		 */

		vm_object_collapse(object);

		if (object->pager == MEMORY_OBJECT_NULL) {
			vm_object_pager_create(object);
			vm_page_lock_queues();
			vm_page_activate(m);
			vm_page_unlock_queues();

			PAGE_WAKEUP(m);
			vm_object_unlock(object);
			continue;
		}

		PAGE_WAKEUP(m);

		vm_pageout_page(m, FALSE);
		vm_object_unlock(object);
		pages_moved++;
		burst_count++;
	}

	/*
	 *	Compute the page shortage.  If we are still very low on memory
	 *	be sure that we will move a minimal amount of pages from active
	 *	to inactive.
	 */

	page_shortage = vm_page_inactive_target - vm_page_inactive_count;

	if ((page_shortage <= 0) && (pages_moved == 0))
		page_shortage = 1;

	while (page_shortage > 0) {
		/*
		 *	Move some more pages from active to inactive.
		 */

		if (queue_empty(&vm_page_queue_active)) {
			break;
		}
		m = (vm_page_t) queue_first(&vm_page_queue_active);
		vm_page_deactivate(m);
		page_shortage--;
	}

	vm_page_unlock_queues();
}

task_t	pageout_task;

/*
 *	vm_pageout is the high level pageout daemon.
 */

void vm_pageout()
{
	pageout_task = current_task();
	pageout_task->kernel_vm_space = TRUE;
	pageout_task->kernel_ipc_space = TRUE;
	pageout_task->ipc_privilege = TRUE;
	current_thread()->vm_privilege = TRUE;

	(void) spl0();

	/*
	 *	Initialize some paging parameters.
	 */

	if (vm_page_free_min == 0)
		if ((vm_page_free_min = vm_page_free_count / 20) < 3)
			vm_page_free_min = 3;

	if (vm_page_free_reserved == 0)
		if ((vm_page_free_reserved = vm_page_free_min / 2) < 10)
			vm_page_free_reserved = 10;

	if (vm_pageout_internal_only == 0)
		if ((vm_pageout_internal_only = 2 * vm_page_free_reserved / 3) < 5)
			vm_pageout_internal_only = 5;

	if (vm_page_free_target == 0)
		vm_page_free_target = (vm_page_free_min * 4) / 3;

	if (vm_page_inactive_target == 0)
		vm_page_inactive_target = vm_page_free_min * 2;

	if (vm_page_free_target <= vm_page_free_min)
		vm_page_free_target = vm_page_free_min + 1;

	if (vm_page_inactive_target <= vm_page_free_target)
		vm_page_inactive_target = vm_page_free_target + 1;

	/*
	 *	The pageout daemon is never done, so loop
	 *	forever.
	 */

	simple_lock(&vm_page_queue_free_lock);
	while (TRUE) {
		if (vm_page_free_wanted)
			simple_unlock(&vm_page_queue_free_lock);
		 else
			thread_sleep((int) &vm_page_free_wanted,
				     simple_lock_addr(vm_page_queue_free_lock),
				     FALSE);
		vm_pageout_scan();
		simple_lock(&vm_page_queue_free_lock);
	}
}
#else	MACH_XP

#include <vm/old_pageout.c>
#endif	MACH_XP
