/* 
 * Mach Operating System
 * Copyright (c) 1988 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 *	File:	vm/vm_pager.c
 *	Author:	Avadis Tevanian, Jr., Michael Wayne Young
 *
 *	Copyright (C) 1986, Avadis Tevanian, Jr., Michael Wayne Young
 *	Copyright (C) 1985, Avadis Tevanian, Jr., Michael Wayne Young
 *
 *	Paging space routine stubs.  Emulates a matchmaker-like interface
 *	for builtin pagers.
 *
 * HISTORY
 * $Log:	vm_pager.c,v $
 * Revision 2.6  88/12/19  03:01:25  mwyoung
 * 	Corrected include file references.
 * 	[88/11/22            mwyoung]
 * 	
 * 	Add some XPR debugging.
 * 	[88/11/22  01:50:39  mwyoung]
 * 
 * Revision 2.5  88/10/18  03:46:16  mwyoung
 * 	The object cache lock is not required to change persistence.
 * 	[88/10/15            mwyoung]
 * 
 * Revision 2.4  88/07/17  19:32:14  mwyoung
 * *** empty log message ***
 * 
 * Revision 2.3.1.2  88/07/04  16:56:05  mwyoung
 * Fix up types for compatibility.
 * 
 * Moved the external memory management interface implementation to
 * file "vm/memory_object.c".
 * 
 * Add memory_object_set_attributes().
 * 
 * Revision 2.3.1.1  88/06/28  21:17:08  mwyoung
 * Added pager_data_error.
 * 
 * Account for wired-down pages during pager_lock_request.
 * 
 * Correct handling of modified bits in pager_lock_request.  It's still
 * not optimal, but is more correct.
 * 
 * Added pager_attributes.
 * 
 *
 * 16-May-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Added pager_data_error.
 *
 * 16-May-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Account for wired-down pages during pager_lock_request.
 *
 * 17-Apr-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Correct handling of modified bits in pager_lock_request.  It's still
 *	not optimal, but is more correct.
 *
 * 14-Mar-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Updated to use the latest vm_pageout_page() calling sequence.
 *
 *  8-Mar-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Correct handling of tasks that use the kernel's address space
 *	when using the FAST_PAGER_DATA optimization.
 *	
 *	When flushing a range of pages for which a request is pending,
 *	throw away the request (so that it is made again).  [This isn't
 *	necessary for the kernel, but probably is easier on the memory manager.]
 *
 * 24-Feb-88  David Golub (dbg) at Carnegie-Mellon University
 *	Handle IO errors on paging (non-XP only).
 *
 *  4-Feb-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Remove old routines for cleaning/flushing/locking.
 *	Avoid absent pages during pager_lock_request() calls.
 *
 * 11-Jan-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Moved copy_user_to_physical to vm/vm_kern.c.
 *
 * 11-Jan-88  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Make PAGE_ASSERT_WAIT's non-interruptible (?).
 *
 *  4-Dec-87  Michael Young (mwyoung) at Carnegie-Mellon University
 *	Fixed to conform to current interface.
 *	
 *	Condensed old history.  Relevant contributions:
 *		Take data directly from user space in
 *		 pager_data_provided [rfr].
 *		Basic external paging interface
 *		 implementation [mwyoung, bolosky].
 *		Handle device mapping [dbg].
 *		Original work [avie, dbg].
 */

#include <mach_xp.h>

#if	MACH_XP
#else	MACH_XP
#include <sys/kern_return.h>
#include <vm/vm_pager.h>
#include <vm/vm_page.h>
#include <vm/vm_prot.h>
#include <sys/boolean.h>
#include <vm/inode_pager.h>
#include <sys/xpr.h>

#include <vm/pmap.h>

extern
boolean_t	vm_page_zero_fill();

kern_return_t pager_cache(object, should_cache)
	vm_object_t	object;
	boolean_t	should_cache;
{
	if (object == VM_OBJECT_NULL)
		return(KERN_INVALID_ARGUMENT);

	vm_object_lock(object);
	object->can_persist = should_cache;
	vm_object_unlock(object);

	vm_object_deallocate(object);

	return(KERN_SUCCESS);
}

pager_return_t vm_pager_get(pager, m)
	memory_object_t	pager;
	vm_page_t	m;
{
	pager_return_t	result;

	XPR(XPR_MEMORY_OBJECT, ("vm_pager_get: pager %x, page %x",
				pager, m));
	if (pager == MEMORY_OBJECT_NULL) {
		(void) vm_page_zero_fill(m);
		return(PAGER_SUCCESS);
	}
	if (((struct pager_struct *)pager)->is_device)
		return(device_pagein(m));
	result = inode_pagein(m);
	XPR(XPR_MEMORY_OBJECT, ("vm_pager_get: pager %x, page %x, result %x",
				pager, m, result));
	return(result);
}

pager_return_t vm_pager_put(pager, m)
	memory_object_t	pager;
	vm_page_t	m;
{
	pager_return_t	result;

	XPR(XPR_MEMORY_OBJECT, ("vm_pager_put: pager %x, page %x",
				pager, m));
	if (pager == MEMORY_OBJECT_NULL)
		panic("vm_pager_put: null pager");
	if (((struct pager_struct *)pager)->is_device)
		return(device_pageout(m));
	result = inode_pageout(m);
	XPR(XPR_MEMORY_OBJECT, ("vm_pager_put: pager %x, page %x, result %x",
				pager, m, result));
	return(result);
}

boolean_t vm_pager_deallocate(pager)
	memory_object_t	pager;
{
	if (pager == MEMORY_OBJECT_NULL)
		panic("vm_pager_deallocate: null pager");
	if (((struct pager_struct *)pager)->is_device)
		return(device_dealloc(pager));
	return(inode_dealloc(pager));
}

memory_object_t vm_pager_allocate(size)
	vm_size_t	size;
{
	return(inode_alloc(size));
}

boolean_t vm_pager_has_page(pager, offset)
	memory_object_t	pager;
	vm_offset_t	offset;
{
	if ((pager == MEMORY_OBJECT_NULL) || (((struct pager_struct *)pager)->is_device))
		panic("vm_pager_has_page");
	return(inode_has_page(pager,offset));
}

#endif	MACH_XP
