#include <sccs.h>
SCCSID(kernel/i386/tkr.c, bos, 130	AIX)	/* Modified: 23:41:14 4/21/93 */
/* 
 * (C) Copyright IBM Corp. 1989 
 * All Rights Reserved 
 * Licensed Materials - Property of IBM 
 */

/*
 * Token-Ring Adapter
 */

#include <param.h>
#include <systm.h>
#include <errno.h>
#include <tkr.h>
#include <debugflag.h>
#include <conf.h>
#include <machinfo.h>
#include <user.h>
#include <ioctl.h>
#include <vmalloc.h>

#include <mmu386.h>
#include <pos.h>
#include <intr86.h>

#include <mbuf.h>
#include <protosw.h>
#include <socket.h>
#include <if.h>
#include <netisr.h>
#include <route.h>
#include <in.h>
#include <in_systm.h>
#include <ip.h>
#include <in_var.h>
#include <if_ieee802.h>

#ifndef TOKENDBG
#define TOKENDBG	(1L << ('T' - 'A'))
#endif /* TOKENDBG */

#define TKRPSID		0xE000	/* PS/2 pos cardid */
#define TKR16PSID	0xE001	/* PS/2 4/16 token ring pos cardid */
#define TKR_RAMSIZE 16384	/* Currently we only support 16k shared ram */

#define	spltoken()	splimp()


/*
 * The following defines and data structures are for AT bus machines.
 */
#define NUM_TKDEV 2  /* Maximum number of WD8003 boards supported */
struct tk_data {
   char           tk_cardid[MAX_DRV_TAG];
   unsigned long  tk_unitnum;
   unsigned long  tk_ioaddr;     /* I/O port address */
   unsigned long  tk_memaddr;    /* shared memory base address */
};
static struct tk_data tk_data[NUM_TKDEV];

/*
 * base port for each possible device.  All other programmed i/o instructions
 * for a particular board are accessed as these base values + a fixed constant
 */
int port_addr[]	= {
	0xa20,
	0xa24,
};

struct tk_softc {
	struct arpcom es_ac;		/* common LAN structures */
};
#define	es_if	es_ac.ac_if		/* network-visible interface  */
#define es_addr	es_ac.ac_lanaddr	/* hardware LAN address */


struct board {
	char far *netdata;		/* pointer to data in receive buffers */
	struct tk_softc softc;		/* IPC interface structure */
	struct tkdevice far *mmio_addr; /* the mmio region where the rom and 
					 * interrupt registers, etc., live. */
	char far *sram;			/* shared ram virtual address */
	paddr_t ram_paddr;		/* ram physical address */
	u_short dhb_addr;		/* location of the xmit buffer */
	int unit;			/* unit this struct is for */
	int type;			/* say which kind of adapter we have */
	u_char comm_correlate;		/* id of outstanding xmit command */
	u_short srb;			/* system request buffer */
	u_short arb;			/* adapter request buffer */
	u_short ssb;			/* system status buffer */
	u_short asb;			/* adapter status buffer */
	int active;			/* busy processing a command? */
	int need_trans;			/* need to setup a transmit command? */
	int open_retries;		/* flag for retrying the open */
	int init_done;			/* is board ready ? */
	int oc_in_progress;		/* Open/Close in progress */
	int asb_busy;			/* asb being used? */
	struct asb_q asb_req[ASB_QUEUE_SIZE];/* queue for saving asb requests */
	int asb_qhead;			 /* that can't immediately be serviced*/
	int asb_qtail;			/* head and tail pointers */
	int oint;			/* count for token output interrupts; 
                                         * set in tk_ostart or tkintr, and
                                         * incremented in tk_watch. It is
                                         * cleared in trans_complete and
                                         * tk_reset.
					 */
	char rbuf[TKR_RAMSIZE];		/* internal receive buffer */
};

struct board *tk_primary = NULL;	/* primary adapter handle */
struct board *tk_alt = NULL;		/* alternate adapter handle */

tkr_reopen_timeout = FALSE;		/* tk_reopen timeout() active */
int have_tkr = 0;			/* do we have token ring hardware? */
long tkr_softerr = 0;			/* count of soft error MAC frames */
u_short tk_sid = DIRECT_STATION;	/* default station id */

/* these variables are really only useful for on the fly patching */
int tkrdebug = 0;

int num_recv8 = NUM_RECV8;	/* number of buffers if 8k of ram */
int recv_len8 = RECV_LEN8;	/* length of receive buffer */

int num_recv16 = NUM_RECV16;	/* number of buffers if 16k of ram */
int recv_len16 = RECV_LEN16;	/* length of receive buffer */

int xmit_len = XMIT_LEN;	/* length of transmit buffer */

/*
 * ADAPTER I/II defines of the recommended physical addresses of shared 
 * ram in the primary and alternate adapters.  ADAPTER A discovers its
 * own address.  It's nice to have these for easy patching.  This
 * particular alternate ram address was chosen because the reccomended
 * values suggested in the token ring hardware book conflict with the
 * U-B board.  So far chooing 0xce000L as the rom address seems to allow
 * two boards to work with a U-B board.
 */
paddr_t primary_ram = 0xd8000L;
paddr_t alt_ram = 0xdc000L;

/* Function declarations */
int tk_watch();
int tk_reopen();
int tkintr();
int tkrexist();

extern struct devdata devdata[];

extern struct tty *notty();

extern if_down();

int tkr_ramtbl[4] = {
	8192,
	16384,
	32768,
	65536
};

#ifdef DEBUGVAR
static char *adapter_name[] = {
	"unknown type",
	"adapter/I",
	"adapter/II",
	"adapter/A"
};
#endif /* DEBUGVAR */


/* 
 * tk_init
 * ------
 * Determine what kind of a token ring(s) we have (if any), set inital memory
 * locations, map data structure templates to the board's memory, and 
 * reset the board to a power on condition.
 */
tkinit(dev)
dev_t dev;
{
	int i;
	struct tkdevice far *tk_addr;	/* temp for mmio region */
	paddr_t mmio_paddr;		/* phyical address of mmio region */
	paddr_t ram_paddr;		/* ram physical address */
	int tk_ramsize;			/* size of shared ram */
	int unit;			/* index for which adapter we're using*/
	int type;			/* type of board we're using */
	u_long card_id;			/* hard-wired value card identifier */
	int irq;			/* interrupt request level */
	int slot;			/* card slot we are examining */
	short maj;			/* device major number */
	unsigned char posa;		/* pos info for the PS/2 */
	unsigned char posb;

	debug(TOKENDBG,("tkinit: BEGIN\n"));

	if (have_tkr) {
		printf("911-360 tkinit: tried to re-initialize board\n");
		return;
	} 

	/* 
	 * If we're on a 386 machine, then see if we're a PS/2.  If so, we
	 * can use devexist to return the slot positions of the cards installed.
	 * Based on the pos data obtained we can determine whether we're
	 * a primary, or an alternate adapter, as well as our irq level.
	 */
	if (HAS_MICRO_CHAN()) {
		maj = major(dev);

		/* return the next slot for the token ring */
		for (slot = 0; (slot = tkrexist(maj, 1, slot)) >= 0;
			slot++) {

			posa = devdata[slot].pd_pos3;
			posb = devdata[slot].pd_pos4;

			/* interpret the pos data and produce an interrupt 
			 * level */
			if ((posa & 0x80) == 0)
				irq = (posb & 1) ? 10 : 2;
			else
				irq = (posb & 1) ? 11 : 3;

			unit = ((posa & 1) == 0) ? ADAPTER0 : ADAPTER1;

			tk_ramsize = tkr_ramtbl[(posa >> 2) & 3];
			if (tk_ramsize != TKR_RAMSIZE) {
				printf("911-361 %s token ring has invalid RAM size\n",
					unit == ADAPTER0 ? "Primary" : "Alternate");
				printf("911-362 Use the Reference Diskette to select");
				printf("911-363  a 16 KB RAM size.");
				continue;
			}

			/* adapter_A code */

			/*
			 * bit 0 of SETUP_2 is bit 19 of mmio physical address
			 * bits 7-2 of SETUP_1 are bits 18-13 of mmio physical 
			 * address.  bits 0 and 1 are the interrupt switch
			 * settings, and are masked off.
			 */
			mmio_paddr = (u_char) READ_SETUP1(port_addr[unit]);
			mmio_paddr = ((mmio_paddr & ~3) << 11) | 
			   (u_long) ((READ_SETUP2(port_addr[unit]) & 1) << 19);

			/*
			 * bits 7-1 of SETUP_2 are bits 19-13 of ram physical
			 *  address
			 */
			ram_paddr = (u_long)
			     (READ_SETUP2(port_addr[unit]) & ~MMIO_BIT19) << 12;

			/*
			 * it is necessary each time to clear a pointer for
			 * MAPINIT to work properly
			 */
			tk_addr = NULL;

			/* map memory to the ROM address */
			MAPINIT((caddr_t) tk_addr);

			MAPIN((caddr_t) tk_addr, mmio_paddr,
				sizeof(struct tkdevice));

			tk_devinstall(dev,irq);

			tk_doinit(unit, ADAPTER_A, ram_paddr,
				tk_addr, tk_ramsize);
		}
	}
	else {
		int tried_force = 0;

		/*
		 * For non-PS/2 machines, there isn't any pos data available.
		 * Thus it is necessary to probe each adapter, map it into
		 * our memory space, and read an area of rom to see if
		 * certain hard-coded values are there.  The irq level
		 * is obtained from reading the switches.  Unit 0 is always
		 * the primary adapter, and unit 1 is the alternate.
		 */
		for (unit = 0; unit <= NUM_TKDEV; unit++) {
			if (isa_devexist("token",unit,&tk_data[unit],sizeof(struct tk_data)) ==
				-1)
            	break;
		forceone:

			/* update port_addr to reflect what the user actually requested */
			port_addr[unit]=(int) tk_data[unit].tk_ioaddr;

			/* adapter_I/adapter_II code */
			mmio_paddr = (u_char) (READ_SETUP1(port_addr[unit]));

			/*
			 * save interrupt switch values (stored in the low two 
			 * bits), for later attachment 
			 */
			switch (mmio_paddr & 3) {
			case 0:
				irq = 2;
				break;

			case 1:
				irq = 3;
				break;

			case 2:
				irq = 6;
				break;

			case 3:
				irq = 7;
				break;
			}

			/*
			 * convert result obtained from reading the switch to
			 * an address where the board thinks that it lives. The
			 * result is a 20 bit address with the high bit always 1
			 * (hence the OR'ing).  The low two bits of the offset 
			 * are the interrupt switches, and are masked off when
			 * calculating the mmio physical address.
			 */
			mmio_paddr = ((mmio_paddr & ~(3)) << 11) | 0x80000L;

			ram_paddr = (paddr_t) tk_data[unit].tk_memaddr;

			/*
			 * it is necessary each time to clear these values for 
			 * MAPINIT to work properly
			 */
			tk_addr = NULL;

			/*
			 * map memory to the ROM address so we can see
			 * if there is really a board out there
			 */
			MAPINIT((caddr_t) tk_addr);

			MAPIN((caddr_t) tk_addr, mmio_paddr, 
				sizeof(struct tkdevice));

			/*
			 * we need to get read the card id to make sure that
			 * there really is a token board out there
			 */
			card_id = 0;
			for (i = 0; i < sizeof(tk_addr->tk_id); i += 2) {
				card_id = (u_long) card_id << 4;
				card_id += tk_addr->tk_id[i] & 0x0f;
			}

			debug(TOKENDBG,
			       ("card id = 0x%lx. Expected 0x%lx|0x%lx|0x%lx\n",
				card_id, CARD_ID_I, CARD_ID_II, CARD_ID_A));

			/* check to make sure that we have a valid card id */
			if (card_id == CARD_ID_I) {
				type = ADAPTER_I;
			}
			else if (card_id == CARD_ID_II) {
				type = ADAPTER_II;
			}
			else {
				type = ADAPTER_NONE;
				continue;
			}

			debug(TOKENDBG,("card is %s\n",	adapter_name[type]));

			/* only 386's can dynamically attach themselves */
			tk_devinstall(dev, irq);
			if (tk_doinit(unit, type, ram_paddr,
					tk_addr, TKR_RAMSIZE) < 0)
				continue;
		} 

		/*
		 * Didn't find through isa_devexist(), try to detect a
		 * a card at factory default settings
		 */
		if (!have_tkr && !tried_force) {
			tk_data[unit].tk_ioaddr = 0xA20;
			tk_data[unit].tk_memaddr = primary_ram;
			tried_force = 1;
			goto forceone;
		}
	}
        /* have_tkr is set in tk_doinit */
        /* only start a watch if we have a board */
        if (have_tkr)
                timeout(tk_watch, (caddr_t) OUTPUT_WATCH, (time_t) slow * hz);
        else
		printf("911-364 tkinit: No token ring card.  Option not enabled.\n");
}

/* 
 * tkreset
 * ------
 * Disable any/all token adapters for shutdown.  Simply set
 * them to the reset state, since all of the processes are dead.
 */
tkreset()
{
	spl_t oldspl;

	oldspl = splhi();
	if (have_tkr & PRIMARY) {
		A_RESET(port_addr[ADAPTER0]);
	}
	if (have_tkr & ALTERNATE) {
		A_RESET(port_addr[ADAPTER1]);
	}
	splx(oldspl);
}

/*
 * tkrexist
 * -----------
 * Perform the same function as devexist but look for either of the types of
 * token ring cards that we know about ( both 4 and 16 mb cards).
 */
tkrexist(maj, flag, slot)
unsigned short maj;
unsigned short flag;
int slot;
{
	int tmp_slot;
	if ((tmp_slot = devexist(TKRPSID, maj, flag, slot)) < 0)
		return(devexist(TKR16PSID, maj, flag, slot));
	return(tmp_slot);
}


/* 
 * tk_devinstall
 * --------------
 * Install the device into the dev switch.  Since this routine may
 * be called multiple times, it is necessary to guard against multiple 
 * device installs and intrattaches at the same level.
 */
tk_devinstall(dev, irq)
dev_t dev;
int irq;
{
	static int old_irq = -1;
	short maj;

	maj = major(dev);

	/* this routine hasn't been called before, so install the device */
	if (old_irq == -1) {
		DEV_INSTALL(maj, tkinit, tkreset, nulldev, nulldev, tkintr, 0);
		CDEV_INSTALL(maj, nulldev, nulldev, nulldev, nulldev, notty);
	}

	/* if this device has a new interrupt level, attach it */
	if (old_irq != irq)
		/* Or in SPL_CLIST in case network tty's are used. */
		if (intrattach(tkintr, irq, SPL_IMP
#ifdef NTY
					| SPL_CLIST
#endif /* NTY */
		) == -1) {
			/*
			 * This interrupt is already in use
			 */
			printf("%s %d %s\n",
				"Token ring driver initialization failed: INT",
				irq,
				"is already in use.");
		}

	/*
	 * save the last interrupt request, so we will know whether or
	 * not the next board will need to be attached
	 */
	old_irq = irq;
}

/*
 * tk_doinit
 * ---------
 * Common code for the token ring's initialization.  Allocate memory
 * for a board structure, and initialize the board.  We set pointers here
 * to the ram and rom areas of the ring as well as initializing the
 * tokennet port information.  have_tkr is set here if the init succeeds.
 */
tk_doinit(unit, type, ram_paddr, tk_addr, ram_size)
int unit;
int type;
paddr_t ram_paddr;
struct tkdevice *tk_addr;
int ram_size;
{
	char *tk_sram = NULL;	/* temporary ram pointer */
	struct board *tk_board; /* temp board pointer */
	int adapter_addr;

	 /* allocate space for the handle */
	if (unit == ADAPTER0) {
		tk_primary = (struct board *)
			kmemalloc(sizeof(struct board), MA_DBLWD | MA_LONGTERM);
		if (tk_primary == NULL) {
			printf("911-365 tk_doinit: no memory for primary ring\n");
			return -1;
		}

		tk_board = tk_primary;
	}
	else {
		tk_alt = (struct board *) kmemalloc(sizeof(struct board),
						MA_DBLWD | MA_LONGTERM);
		if (tk_alt == (struct board *)NULL) {
			printf("911-366 tk_doinit:no memory for alternate ring\n");
			return -1;
		}

		tk_board = tk_alt;
	}

	bzero(tk_board, sizeof(struct board));

	MAPINIT(tk_sram);
	MAPIN(tk_sram, ram_paddr, ram_size);

	debug(TOKENDBG, ("tk_doinit: tk_addr %p ram 0x%lx (tk_sram %p)\n",
		tk_addr, ram_size, tk_sram));

	/* store the rom and ram regions */
	tk_board->mmio_addr = tk_addr;
	tk_board->sram = tk_sram;
	tk_board->ram_paddr = ram_paddr;

	/* remember the board type */
	tk_board->type = type;

	/* remember the unit */
	tk_board->unit = unit;

	/* put the board in a power on state */
	if (tk_reset(tk_board, unit) < 0)
		return -1;

	adapter_addr = FLIP(RAM_WORD(tk_sram, tk_board->srb + 8));

	bcopy((caddr_t) &tk_sram[adapter_addr], tk_board->softc.es_addr,
			LAN_ADDR_SIZE);

	tkattach(unit);

	/* set mask to indicate which board we have */
	if (unit == ADAPTER0)
		have_tkr |= PRIMARY;
	else
		have_tkr |= ALTERNATE;
}

/* 
 * tk_reset
 * ------
 * Reads the adapter switches to determine where the adapter should reside in 
 * mapped memory and issues programmed i/o instructions to initialize the board.
 * It may be necessary to busy wait in this routine for the initialization to
 * complete since when called out of the cdevsw, the interrupt routine isn't 
 * setup.  If the ring is open, it will be closed and the initial bring up
 * diagnostics will be run again.
 *
 * NOTE:  Should always be called at spltoken()
 */

tk_reset(tk_board, unit)
register struct board *tk_board;/* handle for token ring rom and ram area */
int unit;			/* index for prorammed i/o */
{
	struct tkdevice *tk_addr = tk_board->mmio_addr;
	char far *tk_sram = tk_board->sram;
	register int srb;
	int retry = 0;
	u_short bring_up_code;
	u_char init_code;

	/*
	 * Reset the open/close in progress indicator, since a reset
	 * will cancel those operations.
	 */
	tk_board->oc_in_progress = FALSE;

	/* initialize the adapter */
	A_RESET(port_addr[unit]);

	/* wait LATCH_TIME ticks before releasing the adapter */
	delayticks(LATCH_TIME);

	A_RELEASE(port_addr[unit]);

	/*
	 * ADAPTER/A and maybe ADAPTER/I start diagnostics only when
	 * interrupts are enabled.
	 */
	tk_addr->set_isrp_high = (INTR_ENABLE | NMI_INTRCTRL);

	/* we may be resetting the board so, re-initialize all the crucial
	 * variables here.
	 */
	tk_board->srb = tk_board->arb = tk_board->ssb = tk_board->asb = 0;

	tk_board->comm_correlate = tk_board->dhb_addr = 0;

	tk_board->oint = tk_board->asb_busy = tk_board->active = 0;


	while ((retry != INIT_WAITS)
	&&     (! (tk_addr->isrp_low & SRB_RESPONSE))) {
		delayticks(LATCH_TIME);
		++retry;
	}

	debug(TOKENDBG, ("tk_reset: delayed %d ticks\n", LATCH_TIME * retry));

	/* disable adapter interrupts */
	tk_addr->reset_isrp_high = ~INTR_ENABLE;

	if (! (tk_addr->isrp_low & SRB_RESPONSE)) {
		printf("911-367 tk_reset: can't init unit %d's hardware\n", unit);
		if (tkrdebug)
			debugger((int *) NULL);
		return -1;
	}

	/* clear the interrupt just given us by the adapter */
	tk_addr->reset_isrp_low = ~SRB_RESPONSE;

	if (tk_board->type != ADAPTER_A) 
		tk_addr->rrr_high =  tk_board->ram_paddr >> 12;

	srb = tk_board->srb = FLIP(tk_addr->wrb);

	debug(TOKENDBG,("tk_reset: rrr_high = 0x%x wrb = 0x%x srb = 0x%x\n",
			tk_addr->rrr_high, tk_addr->wrb, srb));
	
	init_code = tk_sram[srb];
	bring_up_code = FLIP(RAM_WORD(tk_sram, srb + 6));

	if ((init_code != INIT_COMPLETE) || (bring_up_code)) {
		printf("911-368 tk_reset: ring init unit %d failed - init_code 0x%x\n",
			unit, init_code);
		init_error("tk_reset: ", bring_up_code);
		return -1;
	}

	return 1;
}

/* 
 * tkintr:
 * -------
 * As long a interrupts are pending, service them by dispatching the appropriate
 * handler routine.  The order in which interrupts are handled is critical.
 * ASB_FREE and SSB_RESPONSE's are handled first so that important adapter
 * resources may be freed.  RECEIVED_DATA and TRANSMIT_DATA_REQUEST commands
 * are of the next highest importance to handle since they free the arb for
 * further use and try to make use of the asb.  If we receive an unknown srb
 * response or an unknown arb command the interrupt is ignored and cleared.
 * If the interrupt is completely unrecognized, we panic.
 */
unsigned tkr_heat;

tkintr(vec)
int vec;
{
	register struct board *tk_board;
	register char *tk_sram;
	register u_char	isrp_low;
	register u_short srb;
	register u_short arb;
	int unit;

	if (tkr_heat) {
		if (tkrdebug) {
			printf("911-369 tkintr: entered twice\n");	
			debugger((int *)0);
		}
		else {
			ncprintf("tkintr: entered twice\n");	
		}
	}
	tkr_heat++;

	debug(TOKENDBG,("TKINTR: just got an interrupt\n"));

	/* Loop until we determine that neither board has any pending 
	 * interrupts. Re-read the interrupt registers each time to check 
	 * for new interrupts that may have arrived.  Also note that if
	 * there isn't a token ring board, or there isn't an interrupt
	 * for us, we break out of the loop and exit the interrupt handler.
	 */
	for (;;) {
		if ((have_tkr & PRIMARY)
		&&  (isrp_low = tk_primary->mmio_addr->isrp_low)) {
			tk_board = tk_primary;
			unit = ADAPTER0;

		}
		else if ((have_tkr & ALTERNATE)
		&&       (isrp_low = tk_alt->mmio_addr->isrp_low)) {
			tk_board = tk_alt;
			unit = ADAPTER1;

		}
		else	/* all done handling interrupts */
			break;

		debug(TOKENDBG,
			("tkintr: isrp_low = %b\n",isrp_low,ISRP_LOW_BITS));

		if (isrp_low & ASB_FREE) {
			debug(TOKENDBG, ("got an ASB free intr\n"));
			/* De-queue any pending asb requests */
			tk_dqasb(tk_board);
			continue;
		}

		if (isrp_low & SSB_RESPONSE) {
			debug(TOKENDBG, ("got a TRANSMIT_RESULT intr\n"));
			trans_complete(tk_board);
			continue;
		}

		/* setup convenient temporary pointers */
		tk_sram = tk_board->sram;

		if (isrp_low & SRB_RESPONSE) {
			srb = tk_board->srb;

			switch ((u_char) tk_sram[srb]) {
			case DIR_OPEN_ADAPTER:
				debug(TOKENDBG,("got an OPEN_COMPLETE intr\n"));
				tk_open_complete(tk_board,tk_sram,srb,unit);
				break;
			case TRANSMIT_DIR_FRAME:
				debug(TOKENDBG, ("got an TRANSMIT_RET intr\n"));
				trans_ret(tk_board,tk_sram,srb,unit);
				break;
			case DIR_CLOSE_ADAPTER:
				tk_board->srb = FLIP(tk_board->mmio_addr->wrb);
				tk_board->mmio_addr->reset_isrp_low =
							  (u_char)~SRB_RESPONSE;
				tk_board->oc_in_progress = FALSE;

				/* If the interface should be IFF_UP, then
				 * go issue an adapter open.
				 */
				if (tk_board->softc.es_if.if_flags & IFF_UP)
					tk_open_adapter(tk_board);
				break;
			case DIR_READ_LOG:
				/*
				 * if we want to record the card error counters
				 * this is the place.
				 */
				tk_board->mmio_addr->reset_isrp_low =
							(u_char)~SRB_RESPONSE;
				break;
			default:
				/* these don't seem to be very interesting */
				ncprintf("tkintr: UNRECOGNIZED srb response ");
				ncprintf("in unit %d: srb = 0x%x\n", 
					unit, (u_char) tk_sram[srb]);
				/* clear interrupt */
				tk_board->mmio_addr->reset_isrp_low =
							  (u_char)~SRB_RESPONSE;
			}
			continue;
		}

		if (isrp_low & ARB_COMMAND) {
			arb = tk_board->arb;

			debug(TOKENDBG,("arb = 0x%x tk_sram[arb] = 0x%x\n",
				arb, tk_sram[arb] & 0xff));

			if ((u_char)tk_sram[arb] == TRANSMIT_DATA_REQUEST) {

				debug(TOKENDBG, ("got a TRANSMIT_CMD intr\n"));

				trans_cmd(tk_board, tk_sram, arb);
			}
			else if ((u_char) tk_sram[arb] == RECEIVED_DATA) {

				debug(TOKENDBG,("got a RECEIVE_CMD intr\n"));

				tk_recv_intr(tk_board, tk_sram, arb);
			}
			else if ((u_char) tk_sram[arb] == RING_STATUS_CHANGE) {

				debug(TOKENDBG, ("got a RING_CMD intr\n"));

				tk_ring_cmd(tk_board, tk_sram, arb);
			}
			else {
				/* these don't seem to be very interesting */
				ncprintf("tkintr: UNRECOGNIZED arb response ");
				ncprintf("in unit %d: arb = 0x%x\n",
					unit, (u_char) tk_sram[arb]);
	
				/* Clear interrupt and free the adapter's arb
				 * for further command processing
				 */
				tk_board->mmio_addr->reset_isrp_low =
								~ARB_COMMAND;
				tk_board->mmio_addr->set_isra_low = ARB_FREE;
			}
			continue;
		}

		if (isrp_low & ADAPTER_CHK) {
			debug(TOKENDBG, ("got an ADAPTER_CHK_RET intr\n"));
			tk_adapter_chk(tk_board);
			continue;
		}

		printf("911-370 tkintr: tk_board = %p unit = %d\n", tk_board, unit);
		printf("911-371 isrp_low = %b, interrupt vector = 0x%x\n",
			isrp_low, ISRP_LOW_BITS, vec);
		panic("911-372 tkintr: UNRECOGNIZED interrupt.\n");
	}

out:
	debug(TOKENDBG,("TKINTR: end isrp_low = %b\n", isrp_low,ISRP_LOW_BITS));

	/* release interrupts on both boards, since at this point we know
	 * that there aren't any pending interrupts 
	 */
	A_IRELEASE(port_addr[ADAPTER0]);
	A_IRELEASE(port_addr[ADAPTER1]);
	
	tkr_heat = 0;
}

/*
 * tk_open_adapter()
 * -----------------
 * This routine sets up an open command for the adapter.  The adapter isn't
 * open until we receive an OPEN_COMPLETE interrupt which is handled in 
 * tk_open_complete()
 */
tk_open_adapter(tk_board)
struct board *tk_board;
{

	register int i;
	register char *tk_sram = tk_board->sram;
	register int srb;

	/* Indicate that we are in the process of opening the adapter.
	 * Once the adapter has completed opening, it will interrupt us
	 * and tk_open_complete() will be called which then clear the
	 * tk_board->oc_in_progress flag.
	 */
	tk_board->oc_in_progress = TRUE;
	srb = tk_board->srb = FLIP(tk_board->mmio_addr->wrb);

	 /* Interrupts will be enabled as long as the the adapter is open.
	  */
	tk_board->mmio_addr->set_isrp_high = (INTR_ENABLE | NMI_INTRCTRL);

	/* prepare the srb with the open command */
	tk_sram[srb] = 	DIR_OPEN_ADAPTER;
	tk_sram[srb + 8] = OPEN_OPTS_HIGH;
	tk_sram[srb + 9] = OPEN_OPTS_LOW;

	/* don't specify NODE, GROUP, and FUNCTIONAL addresses */
	for (i = 10; i < 24; ++i)
		tk_sram[srb + i] = 0;

	/*
	 * Based on the location of the SRB, (which is near the edge of RAM)
	 * determine if we have 16k or 8k of ram and set the number of 
	 * buffers accordingly.
	 */
	if (srb > (8 * 1024)) {	/* 16k of ram */
		/* words are stored in reverse order */
		RAM_WORD(tk_sram, srb + 24) = FLIP(num_recv16);
		RAM_WORD(tk_sram, srb + 26) = FLIP(recv_len16);

	}
	else {	/* 8k of ram */
		RAM_WORD(tk_sram, srb + 24) = FLIP(num_recv8);
		RAM_WORD(tk_sram, srb + 26) = FLIP(recv_len8);
	}

	/* the size of the transmit buffer is independent of the ram size */
	RAM_WORD(tk_sram, srb + 28) = FLIP(xmit_len);
	tk_sram[srb + 30] = NUM_XMIT;

	for (i = 31; i < 60; i++)
		tk_sram[srb + i] = 0;

	/* interrupt the adapter */
	tk_board->mmio_addr->set_isra_low = SRB_COMMAND;
}

/* 
 * tk_open_complete
 * ---------------
 * Interrupt handler called when the adapter open has completed.  Sets the 
 * values of the asb, arb, ssb, and srb.  Calls the ethernet initialization
 * routines.
 */
tk_open_complete(tk_board, tk_sram, srb, unit)
register struct board *tk_board;
register char *tk_sram;
register int srb;
int unit;
{
	register struct ifnet *ifp;
	int was_running;

	ifp = &tk_board->softc.es_if;

	/* clear the interrupt */
	tk_board->mmio_addr->reset_isrp_low = ~SRB_RESPONSE;

	if (tk_sram[srb + 2] != SUCCESS) {

		printf("911-373 tk_open_complete: ring open of unit %d failed\n", unit);
		open_error(tk_board, "tk_open_complete: ", 
					(u_char) tk_sram[srb + 2],
					FLIP(RAM_WORD(tk_sram, srb + 6)));
		/* If the token ring seizes, keep trying every 30 seconds
		/* If the administrator types "ifconfig tkname down",
		/* Give up.
                /* This fix means that the adminstrator won't have to 	
		/* re-ifconfig every machine. 
		/* The interface will come back after the ring comes back.
		*/
		if (have_tkr) {
			if ( (ifp->if_flags&IFF_UP) &&
			     (tk_board->open_retries) ) {
				printf("911-374 Retry adapter open %d\n",
					   tk_board->open_retries);
				if (! tkr_reopen_timeout) {
					tkr_reopen_timeout = TRUE;
					timeout(tk_reopen,
						(caddr_t) REOPEN_ADAPTER,
						(time_t) 30 * hz);
				}
				tk_board->oc_in_progress = FALSE;
				return;
			}
			printf("911-416 tkr interface down: giving up\n");
		}

		was_running = ((ifp->if_flags & IFF_RUNNING) != 0);
		ifp->if_flags &= ~(IFF_UP|IFF_RUNNING) ;
		timeout(if_down, (caddr_t)ifp, (time_t) 0);
		while (ifp->if_snd.ifq_head) {
			register struct mbuf *m;

			IF_DEQUEUE(&ifp->if_snd, m);
			IF_DROP(&ifp->if_snd);
			m_freem(m);
		}

		tk_board->oc_in_progress = FALSE;
		if (was_running)
			tk_close_adapter(tk_board);
		return;
	}

	/* the open went fine.  Assign the command control blocks */
	tk_board->open_retries = 0;
	tk_board->asb = FLIP(RAM_WORD(tk_sram, srb + 8));
	tk_board->arb = FLIP(RAM_WORD(tk_sram, srb + 12));
	tk_board->ssb = FLIP(RAM_WORD(tk_sram, srb + 14));
	tk_board->srb = FLIP(RAM_WORD(tk_sram, srb + 10));

	debug(TOKENDBG,("asb = 0x%x arb = 0x%x ssb = 0x%x srb = 0x%x\n",
			tk_board->asb,tk_board->arb,
			tk_board->ssb,tk_board->srb));


	/*
	 * If while we were waiting for the open complete interrupt,
	 * someone did an 'ifconfig tk0 down', then go close the
	 * adapter.  If only people would make up their mind...
	 */
	if (! (ifp->if_flags & IFF_UP)) {
		tk_close_adapter(tk_board);
		while (ifp->if_snd.ifq_head) {
			register struct mbuf *m;

			IF_DEQUEUE(&ifp->if_snd, m);
			IF_DROP(&ifp->if_snd);
			m_freem(m);
		}
		return;
	}

	/*
	 * setup a transmit command ahead of time so that one will always
	 * be ready when ostart is called
	 */
	tk_setup_trans(tk_board,tk_sram,tk_board->srb);
	ifp->if_flags |= IFF_RUNNING;
	/*
	 * The init is not done till we reach this point. Previously this flag
	 * was being set in tk_open_adapter() and that is incorrect. What was
	 * happening is that if a transmit occured before the open complete
	 * interrupt was received the adapter was being reprogrammed
	 * for the transmit and the open was left in an uncompleted and 
	 * undefined state. Consequently receives were not working and we were
	 * not being interrupted by received packets.
	 * Setting this "semaphore" in the correct place
	 * protects the card from doing any other work till the open(init) is
	 * indeed done. Fixes PTM26033.
	 */
	tk_board->init_done = 1;
	tk_board->oc_in_progress = 0;
}

/*
 * tk_close_adapter()
 * -----------------
 * This routine sets up an close command for the adapter.
 * tk_close_complete()
 */
tk_close_adapter(tk_board)
struct board *tk_board;
{

	/* Indicate that we are in the process of closing the adapter.
	 * Once the adapter has completed closing, it will interrupt us
	 * and tkintr() will be called which then clear the 
	 * tk_board->oc_in_progress flag.
	 */
	tk_board->oc_in_progress = 1;

	/* prepare the srb with the close command */
	tk_board->sram[tk_board->srb] = DIR_CLOSE_ADAPTER;
	tk_board->need_trans = 0;
	tk_board->init_done = 0;

	/* interrupt the adapter */
	tk_board->mmio_addr->set_isra_low = SRB_COMMAND;
	tk_board->softc.es_if.if_flags &= ~IFF_RUNNING;
}

/*
 * tk_setup_trans
 * --------------
 * This routine is called so that we can always try and have a transmit
 * command outstanding when ostart is called.   That allows us to not have
 * to wait for the adapter to tell us that it is ready to transmit before 
 * data can actually be moved, hopefully speeding up response time.
 */
tk_setup_trans(tk_board, tk_sram, srb)
register struct board *tk_board;
register char *tk_sram;
register int srb;
{
	/* set up transmit command. The adapter will interrupt us when it
	 * is ready to for us to place our data in shared ram.
	 */
	tk_sram[srb] = TRANSMIT_DIR_FRAME;
	RAM_WORD(tk_sram, srb + 4)  = FLIP(tk_sid); /* station id = DIRECT */ 

	/* interrupt the adapter */
	tk_board->mmio_addr->set_isra_low = SRB_COMMAND;

	/*
	 * various interrupt handlers will take care of actually sending
	 * the packet.  Return for now.
	 */

	debug(TOKENDBG, ("tk_setup_trans: transmit command sent\n"));
}

/* 
 * trans_ret
 * ---------
 * If a transmit command cannot be sent for some reason, print the error
 * and return after clearing the interrupt 
 */
trans_ret(tk_board, tk_sram, srb, unit)
struct board *tk_board;
char *tk_sram;
int srb;
int unit;
{
	u_char return_code;

	return_code = (u_char) tk_sram[srb + 2];

	printf("911-375 bad return from TRANSMIT_DIR_COMMAND in unit %d: 0x%x\n", 
		unit, return_code);

	/* clear interrupt */
	tk_board->mmio_addr->reset_isrp_low = ~SRB_RESPONSE;
}

/* 
 * trans_cmd
 * ---------
 * Handle the adapter transmit command by freeing the ARB and always
 * calling ostart.  It is the driver's responsibility to kick
 * the network code so that it can discover that there is no work to do.
 */
trans_cmd(tk_board, tk_sram, arb)
register struct board *tk_board;
register char *tk_sram;
register int arb;
{
	tk_board->need_trans = 0;

	tk_board->comm_correlate = (u_char) tk_sram[arb + 1];

	tk_board->dhb_addr = FLIP(RAM_WORD(tk_sram, arb + 6));

	debug(TOKENDBG, ("dhb_addr = 0x%x, comm_correlate = 0x%x\n",
		tk_board->dhb_addr, tk_board->comm_correlate));

	/* Clear the interrupt. */
	tk_board->mmio_addr->reset_isrp_low = ~ARB_COMMAND;

	/* Free the adapter's arb for further command processing */
	tk_board->mmio_addr->set_isra_low = ARB_FREE;
	
	tk_ipc_ostart(&tk_board->softc.es_if);
}

/* 
 * trans_complete
 * --------------
 * Since the upper layer protocols aren't interested in the return code,
 * errors aren't reported.  The ostart active flag is reset here to allow
 * subsequent calls to that routine.
 */
trans_complete(tk_board)
register struct board *tk_board;
{
#ifdef DEBUGVAR
	u_char ret_code;

	ret_code = (u_char) tk_board->sram[tk_board->ssb + 2];

	debug(TOKENDBG,("command 0x%x correlate 0x%x returned 0x%x\n",
		(u_char) tk_board->sram[tk_board->ssb + 0], 
		(u_char) tk_board->sram[tk_board->ssb + 1],
		ret_code));

	if ((debug_var & TOKENDBG) && (ret_code == 0x22))
		printf("returned FS byte = 0x%x\n",
			(u_char) tk_board->sram[tk_board->ssb + 6]);
#endif /* DEBUGVAR */
		
	tk_board->oint = 0;
	tk_board->active--;

	/* let the adapter know that the ssb is free and clear the interrupt*/
	tk_board->mmio_addr->reset_isrp_low = ~SSB_RESPONSE;
	tk_board->mmio_addr->set_isra_low = SSB_FREE;
}

/* 
 * copy_to_tkrbuf
 * --------------
 * copy received packet into our local buffer 
 */
copy_to_tkrbuf(tk_sram, tk_rbuf, recv_buffer_addr, len)
register char *tk_sram;
char *tk_rbuf;
u_short recv_buffer_addr;
int len;	/* total frame size */
{
	int rbuf;		/* offset of next token ring receive buffer */
	register char *dest;	/* address to copy next group of bytes to */
	char *src;		/* address to copy next group of bytes from */
	int clen;		/* number of characters to copy from this 
				   buffer */
	int tlen = 0;		/* number of bytes copied into receive buf 
				   so far */

	rbuf = recv_buffer_addr + 2;	/* addr of next ptr */
	dest = tk_rbuf;
	clen = FLIP(RAM_WORD(tk_sram, recv_buffer_addr + 6));
	src = &tk_sram[rbuf + 6];
	do {
		if ((tlen += clen) > TKR_RAMSIZE) {
			ncprintf("copy_to_tkrbuf: bad size %d\n",tlen);
			return(-1);
		}
		bcopy(src, dest, clen);
		dest += clen;

		rbuf = FLIP(RAM_WORD(tk_sram, rbuf));

		if (rbuf) {
			src = &tk_sram[rbuf + 6];
			clen = FLIP(RAM_WORD(tk_sram, rbuf + 4));
		}
	} while(rbuf);
	return(0);
}

unsigned copy_to_hits;

/* 
 * tk_recv_intr
 * ------------
 * Process the adapter's receive interrupt command.  It is important to 
 * note that the interrupt needs to be cleared before the ARB_FREE bit 
 * is set.  This protects against an another ARB_COMMAND being presented
 * and unintentionaly cleared.  
 *
 * Processing consists of finding the beginning of the frame data and handing
 * it off to an upper layer, which will copy it into its private data space.
 * Once this has been done, the adapter is told via tk_response where to
 * begin freeing buffer space.
 */
tk_recv_intr(tk_board, tk_sram, arb)
struct board *tk_board;
char *tk_sram;
int arb;
{
	register int len;
	spl_t	s;
	int	lan_len, data_len;
	struct ie2_llc_hdr *llcp;
	struct mbuf *m = NULL;
	char *kludgep;
	extern struct mbuf *tkget();
	u_short recv_buffer_addr = 0;		/* pointer to beginning of 
						 * receive buffers in shared 
						 * ram */

	/*
	 *  determine the length of the received packet.
	 */
	
	len = FLIP(RAM_WORD(tk_sram, arb + 10));
	lan_len = (u_char) (tk_sram[arb + 8]);		/* length of mac hdr*/
	data_len = len - lan_len;
	recv_buffer_addr = FLIP(RAM_WORD(tk_sram, arb + 6));
	if (copy_to_tkrbuf(tk_sram, tk_board->rbuf, recv_buffer_addr, len)) {
		copy_to_hits++;
		if (tk_board->need_trans)
			tk_board->need_trans = 0x00;
		s = spltoken();
		/* reset the board and re-open the adapter */
		if (tk_reset(tk_board,  tk_board->unit) < 0)
			panic("911-376 unable to reset token ring board");

		tk_open_adapter(tk_board);
		splx(s);
		return;
	}

	/* clear the interrupt */
	tk_board->mmio_addr->reset_isrp_low = ~ARB_COMMAND;

	/* Free the adapter's arb for further command processing */
	tk_board->mmio_addr->set_isra_low = ARB_FREE;

	/* let the adapter know that these receive buffers are free */
	tk_qasb(tk_board, RECEIVED_DATA, recv_buffer_addr);

	/* NEEDSWORK: receiver should be shut off */
	if (! (tk_board->softc.es_if.if_flags & IFF_UP))
		return;

	/* Check for minimum length packet */
	if (data_len < sizeof(struct ie2_llc_hdr)) {
		ncprintf("tkr_recv_intr: short packet - %d bytes\n", len);
		return;
	}
	llcp = mac_to_llc((struct ie5_mac_hdr *)tk_board->rbuf);

	switch (llcp->ctrl & ~CTRL_P_OR_F) {
	case TEST_CONTROL:
		m = tkget((char *) llcp, data_len, &tk_board->softc.es_if);
		if (m == NULL)
			goto out;
		/*
		 * this is a different format packet that ethernet would
		 * send. I don't know the function of this call, but it
		 * currently does not do anything anyway. Just beware of
		 * the difference between token and ether when finishing
		 * this function.
		 */
		lan_test_recv(&tk_board->softc.es_if, m);
		goto out;
	case XID_CONTROL:
		m = tkget((char *) llcp, data_len, &tk_board->softc.es_if);
		if (m == NULL)
			goto out;
		/*
		 * this is a different format packet that ethernet would
		 * send. I don't know the function of this call, but it
		 * currently does not do anything anyway. Just beware of
		 * the difference between token and ether when finishing
		 * this function.
		 */
		lan_xid_recv(&tk_board->softc.es_if, m);
		goto out;
	case UI_CONTROL:
		break;
	default:
		debug(TOKENDBG,("tk_recv_intr: unknown LLC ctrl field %d\n", 
			llcp->ctrl));
		goto out;
	}
	if (llcp->dsap != SNAP_SAP) {
		debug(TOKENDBG, ("tk_rev_intr: dsap field %d expected %d\n", 
			llcp->dsap, SNAP_SAP));
		goto out;
	}
	if (llcp->ssap != SNAP_SAP) {
		debug(TOKENDBG, ("tk_rev_intr: ssap field %d expected %d\n",
			llcp->ssap, SNAP_SAP));
		goto out;
	}

	switch (ntohs(llcp->type)) {
	case LANTYPE_IP:
		debug(TOKENDBG, ("tk_recv_intr: LANTYPE_IP traffic\n"));

		m = tkget((char *)llcp + sizeof(struct ie2_llc_hdr), 
			  data_len - sizeof(struct ie2_llc_hdr),
			  &tk_board->softc.es_if);
		if (m == NULL)
			break;
		if (IF_QFULL(&ipintrq)) {
			IF_DROP(&ipintrq);
			m_freem(m);
			break;
		}
		IF_ENQUEUE(&ipintrq, m);
		schednetisr(NETISR_IP);
		break;
	case LANTYPE_ARP:
		debug(TOKENDBG, ("tk_recv_intr: LANTYPE_ARP traffic\n"));

		ie5_arpinput(&tk_board->softc.es_ac, tk_board->rbuf);
		break;
	}
out: 	;
}

/*
 * tk_qasb
 * -----------
 * Anytime either a transmit or receive command needs to inform the adapter
 * of some information via the asb, we queue the request here in this routine
 * for processing when the resource becomes available.  If the asb is already
 * free, then mark that there is one outstanding request (via asb_busy++),
 * and call tk_response directly.  If further requests arrive before this
 * one can be serviced, then they will be queued instead.
 */
tk_qasb(tk_board, cmd, response_data)
register struct board *tk_board;
u_char cmd;
u_short response_data;
{
	register struct asb_q *asb_req = tk_board->asb_req;
	register int asb_qtail = tk_board->asb_qtail;

	/* if no one is on the queue, respond immediately to the request */
	if (tk_board->asb_busy++ == 0) {
		tk_response(tk_board, cmd, response_data);
		return;
	}
#ifndef NOCHECKS
	else if (tk_board->asb_busy > ASB_QUEUE_SIZE) {
		printf("911-377 tk_qasb: number of asb queue elements =%d\n",
		       tk_board->asb_busy);
		panic("911-378 tk_qasb: asb queue overflow");
	}
#endif /* NOCHECKS */

	asb_req[asb_qtail].cmd = cmd;
	asb_req[asb_qtail].response_data = response_data;

	/* handle wrapping of tail pointer */
	tk_board->asb_qtail = (asb_qtail + 1) % ASB_QUEUE_SIZE;

	/* last ditch try to see if the asb is free */
	/* actually this ends up succeeding much of the time */
	if (tk_board->mmio_addr->isrp_low & ASB_FREE) {
		/* De-queue any pending asb requests */
		tk_dqasb(tk_board);
		return;
	}
}


/*
 * tk_dqasb
 * -----------
 * Called in response to an ASB_FREE interrupt.  First we check to see if
 * there is anyone on the queue to remove.  The first request isn't queued,
 * so --asb_busy gives a true indication of how many elements are remaining
 * to be serviced.  If there are any elements to dequeue, then call 
 * tk_response() to respond with the delayed adapter command.
 */
tk_dqasb(tk_board)
register struct board *tk_board;
{
	register struct asb_q *asb_req = tk_board->asb_req;
	register int asb_qhead = tk_board->asb_qhead;
	u_char cmd;
	u_short response_data;

	/* clear the interrupt */
	tk_board->mmio_addr->reset_isrp_low = ~ASB_FREE;

	/* if no one is on the queue, return immediately */
	if (--tk_board->asb_busy == 0) {
		return;
	}
#ifndef NOCHECKS
	else if (tk_board->asb_busy < 0) {
		printf("911-379 tk_dqasb: number of asb queue elements =%d\n",
			tk_board->asb_busy);
		panic("911-380 tk_dqasb: asb queue inconsistent");
	}
#endif /* NOCHECKS */

	cmd = asb_req[asb_qhead].cmd;
	response_data = asb_req[asb_qhead].response_data;

	/* handle head pointer wrapping properly */
	tk_board->asb_qhead = (asb_qhead + 1) % ASB_QUEUE_SIZE;

	tk_response(tk_board,cmd,response_data);
}


/*
 * tk_response
 * -----------
 * Respond to an adapter command by setting codes in the asb.  When this
 * routine is called, we know that the asb is free, so there is no need for
 * checking any status.
 * 
 * If respond to a transmit command then flag that we are now expecting an 
 * output interrupt, and setup another transmit ahead of time.
 */
tk_response(tk_board,cmd,response_data)
register struct board *tk_board;
u_char cmd;
u_short response_data;
{
	register char *tk_sram = tk_board->sram;
	register int asb = tk_board->asb;

	debug(TOKENDBG, ("tk_response: responding to command 0x%x\n",cmd));

	/* it is only possible to have one tranmit command outstanding at
	 * a time, thus only one comm_correlate is necessary
	 */
	if (cmd == TRANSMIT_DIR_FRAME)
		tk_sram[asb + 1] = tk_board->comm_correlate;

	tk_sram[asb] = cmd;
	tk_sram[asb + 2] = SUCCESS;			/* return code */

	RAM_WORD(tk_sram, asb + 4) = FLIP(tk_sid);	/* station id */
	RAM_WORD(tk_sram, asb + 6) = FLIP(response_data);

	/* interrupt the adapter.  Tell it to let us know when the asb
	 * is free so we can clear the busy flag
	 */
	tk_board->mmio_addr->set_isra_low = ASB_RESPONSE | ASB_FREE_REQ;

	if (cmd == TRANSMIT_DIR_FRAME) {
		tk_board->oint = 1;		/* we are expecting a completion
						 * interrupt */
		tk_board->need_trans = 1;	/* we are going to need */
						/* a transmit command now */
		tk_setup_trans(tk_board,tk_sram,tk_board->srb);
	}
}

/*
 * tk_adapter_chk
 * --------------
 * This is a serious error that implies that the ring will be closed.  There
 * are 14 bytes of data that could be interrogated to discover what the problem
 * was.  I'll do it later.
 */

tk_adapter_chk(tk_board)
struct board *tk_board;
{
	/* clear the adapter check interrupt */
	tk_board->mmio_addr->reset_isrp_low = ~ADAPTER_CHK;
}

/* 
 * tk_ring_cmd
 * -------------
 * Reports ring status changes on the console
 */
tk_ring_cmd(tk_board, tk_sram, arb)
struct board *tk_board;
char *tk_sram;
int arb;
{
	register u_short ring_status;

	ring_status = FLIP(RAM_WORD(tk_sram,arb + 6));

	debug(TOKENDBG, ("ring_status = 0x%x\n",ring_status));

	if (ring_status & SIGNAL_LOSS)
		tk_err(tk_board, "SIGNAL_LOSS",	"can't detect any signal", 0);

	if (ring_status & HARD_ERROR)
		tk_err(tk_board, "HARD_ERROR",
			"Beacon frames are being received or transmitted", 0);

	if (ring_status & SOFT_ERROR)
		tkr_softerr++;

	if (ring_status & TRANSMIT_BEACON)
		tk_err(tk_board, "TRANSMIT_BEACON",
			"Adapter is transmitting beacon frames", 0);

	if (ring_status & LOBE_WIRE_FAULT)
		tk_err(tk_board, "LOBE_WIRE_FAULT",
			"Open or short circuit detected.", 1);

	if (ring_status & AUTO_REMOVAL)
		tk_err(tk_board, "AUTO_REMOVAL",
			"Adapter hardware error detected", 1);

	if (ring_status & REMOVE_RECEIVED)
		tk_err(tk_board, "REMOVE_RECEIVED",
			"Remove MAC frame received.", 1);

	if (ring_status & COUNTER_OVERFLOW) {

		/* prepare the srb with the read log command */
		tk_board->sram[tk_board->srb] = DIR_READ_LOG;

		/* interrupt the adapter */
		tk_board->mmio_addr->set_isra_low = SRB_COMMAND;
	}

	if (ring_status & SINGLE_STATION)
		tk_err(tk_board, "SINGLE_STATION",
			"No other sites detected", 0);

	if (ring_status & RING_RECOVERY)
		tk_err(tk_board, "RING_RECOVERY",
			"Adapter is receiving or transmitting contintion MAC frames", 0);

	/* clear and release the interrupt */
	tk_board->mmio_addr->reset_isrp_low = ~ARB_COMMAND;
	
	/* Free the adapter's arb for further command processing */
	tk_board->mmio_addr->set_isra_low = ARB_FREE;
}

tk_err(tk_board, error, text, closed)
struct board *tk_board;
char *error, *text;
int closed;
{
	char *name;
	register struct ifnet *ifp = &tk_board->softc.es_if;

	name = (tk_board == tk_primary) ? "PRIMARY" : "ALTERNATE";
	if (!closed) {
		ncprintf("token_ring: %s error in %s adapter:\n", error, name);
		ncprintf("	- %s\n", text);
	} else if (ifp->if_flags & IFF_RUNNING) {
		printf("911-381 token_ring: %s error in %s adapter:\n", error, name);
		printf("911-382 	- %s\n", text);
		printf("911-383 	- Adapter is closed\n");
		tk_close_adapter(tk_board);
		ifp->if_flags &= ~(IFF_UP|IFF_RUNNING) ;
		timeout(if_down, (caddr_t)ifp, (time_t) 0);
		tk_board->oc_in_progress = FALSE;
		/* drop all wating pkts	*/
		while (ifp->if_snd.ifq_head) {
			register struct mbuf *m;

			IF_DEQUEUE(&ifp->if_snd, m);
			IF_DROP(&ifp->if_snd);
			m_freem(m);
		}
	}
}

/*
 * tk_watch
 * -------- 
 * Detect lost output interrupts and reset the board appropriately.
 * May also be use to unconditionally reset and re-open all adapters.
 */
#define TK_OINT_MAXTIME 20	

unsigned oint_hits;
unsigned need_trans_hits;

tk_watch(watch_status)
int watch_status;
{
	register spl_t s;
	register int unit;
	register struct board *tk_board;


	s = spltoken();

	for (unit = ADAPTER0; unit <= ADAPTER1; unit++) {

		if (unit == ADAPTER0) {
			if (have_tkr & PRIMARY)
				tk_board = tk_primary;
			else 
				continue;
		} else {
			if (have_tkr & ALTERNATE)
				tk_board = tk_alt;
			else 
				continue;
		}

		/* output is pending; */
		if ((tk_board->oint) && (tk_board->oint++ == TK_OINT_MAXTIME)) {
			printf("911-384 tk_watch: LOST TOKEN OUTPUT ");
			printf("911-385 INTERRUPT in unit %d\n\t: waited %d+ secs\n",
				unit, TK_OINT_MAXTIME * slow);
			printf("911-386 \t: resetting board\n");
			oint_hits++;

		} else if ((tk_board->need_trans) &&
			   (tk_board->need_trans++ == TK_OINT_MAXTIME)) {

			ncprintf("tk_watch: NEED TRANS RESET %x\n",
 				tk_board->need_trans);
			ncprintf("INTERRUPT in unit %d\n\t: waited %d+ secs\n",
				unit, TK_OINT_MAXTIME * slow);
			ncprintf("\t: resetting board\n");
			need_trans_hits++;
		}
		else if (watch_status == REOPEN_ADAPTER) {
			ncprintf("tk_watch: Retry Open\n");	
		}
		/* if we don't want an unconditional reopen, continue */
		else if (watch_status != REOPEN_ADAPTER)
			continue;

		/*
		 * If there is an open or close in progress, let it
		 * finish of its own accord.
		 */
		if (tk_board->oc_in_progress) 
			continue;

		/* reset the board and re-open the adapter */
		if (tk_reset(tk_board,unit) < 0)
			continue;

		tk_open_adapter(tk_board);
	}
	splx(s);

	/* only start a watch if we have a board. This avoids a problem if
	 * someone tries to reset a non-existent token ring board
	 */
	if (have_tkr)
		timeout(tk_watch, (caddr_t) OUTPUT_WATCH, (time_t) slow * hz);
}

tk_reopen(watch_status)
int watch_status;
{
	register spl_t s;
	register int unit;
	register struct board *tk_board;

	s = spltoken();

	/* Mark timeout as no longer active */
	tkr_reopen_timeout = FALSE;

	for (unit = ADAPTER0; unit <= ADAPTER1; unit++) {

		if (unit == ADAPTER0) {
			if (have_tkr & PRIMARY)
				tk_board = tk_primary;
			else 
				continue;
		} else {
			if (have_tkr & ALTERNATE)
				tk_board = tk_alt;
			else 
				continue;
		}

		/*
		 * If this board is doing open retries, AND there is no
		 * open or close command in progress, AND the interface
		 * is still marked up, THEN go retry the open.
		 */
		if ( tk_board->open_retries &&
		     (! (tk_board->oc_in_progress)) &&
		     (tk_board->softc.es_if.if_flags & IFF_UP) ) {
			printf("911-387 tk_reopen: Retry Open unit %d\n",unit);	
			/* reset the board and re-open the adapter */
			if (tk_reset(tk_board,unit) < 0)
				continue;

			tk_open_adapter(tk_board);
		}
	}
	splx(s);
}

/* 
 * init_error
 * ---------
 * Print out any bring up error messages.
 */
init_error(owner_msg, bring_up_code)
char *owner_msg;
u_short bring_up_code;
{
	printf("911-388 %s", owner_msg);
	switch (bring_up_code) {
			/* should never happen */
	case SUCCESS:
	 	printf("911-389 Initialization successful\n");
		break;

	case PROCESSOR_FAIL:
		printf("911-390 Processor initialization failed\n");
		break;

	case ROM_FAIL:
		printf("911-391 ROM test diagnostic failed\n");
		break;

	case RAM_FAIL:
		printf("911-392 RAM test diagnostic failed\n");
		break;

	case INSTRUCTION_FAIL:
		printf("911-393 Instruction test diagnostic failed\n");
		break;

	case INTERRUPT_FAIL:
		printf("911-394 Interrupt test diagnostic failed\n");
		break;

	case MEMORY_FAIL:
		printf("911-395 Memory interface hardware diagnostic failed\n");
		break;

	case PROTOCOL_FAIL:
		printf("911-396 Protocol handler diagnostic failed\n");
		break;

	case TIMER_FAIL:
		printf("911-397 Adapter's programmable timer failed\n");
		break;

	case WRITE_RAM_FAIL:
		printf("911-398 Cannot write to shared ram\n");
		break;

	case READ_RAM_FAIL:
		printf("911-399 Reading from shared ram caused an error\n");
		break;

	case RO_RAM_FAIL:
		printf("911-400 Writing into shared ram read-only ");
		printf("911-401 area did not cause an error\n");
		break;

	case INIT_TIMEOUT:
		printf("911-402 Initialzation timed out\n");
		break;

	default:
		printf("911-403 Unknown failure during bring up 0x%x\n", bring_up_code);
		break;
	}
}

/*
 * open_error
 * ----------
 * If an error occurred while trying to open the adapter, then print
 * out the information here.
 */

static char *phaserr[] = {
	"Phase error 0",
	"Lobe media test",
	"Physical insertion",
	"Address verification",
	"Roll call poll",
	"Request parameters"
};
static char *openerr[] = {
	"Open error 0",
	"Function failure",
	"Signal loss",
	"Wire fault",
	"Frequency error",
	"Timeout",
	"Ring failure",
	"Ring Beaconing",
	"Duplicate node address",
	"Parameter request",
	"Remove received",
	"ReservedB",
	"ReservedC"
};
	
open_error(tk_board, owner_msg, ret_code, open_errcode)
struct board *tk_board;
char *owner_msg;
unsigned short open_errcode;
u_char ret_code;
{

	int phnib,errnib;

	printf("911-404 %s ", owner_msg);
	switch(ret_code) {
	case INVALID_FAIL:
		printf("911-405 Invalid command code\n");
		break;

	case OPEN_FAIL:
		printf("911-406 Adapter open - should be closed\n");
		break;

	case PARAM_FAIL:
		printf("911-407 Required parameter(s) not provided\n");
		break;

	case CANCEL_FAIL:
		phnib = (open_errcode & 0xf0) >> 4;
		errnib = open_errcode & 0x0f;
		printf("911-408 Command canceled\n");
		printf("911-409 \t - unrecoverable failure: 0x%x%x\n", phnib, errnib);
		printf("911-410 \t - (%s : %s)\n", phaserr[phnib], openerr[errnib]);
		switch (open_errcode) {
			case 0x11:
			case 0x26:
			case 0x27:
			case 0x32:
			case 0x35:
			case 0x36:
			case 0x37:
			case 0x42:
			case 0x45:
			case 0x46:
			case 0x47:
			case 0x55:
			case 0x56:
			case 0x57:
			case 0x59:
				/* should retry the open in 30 seconds */
				tk_board->open_retries++;
				break;
		}
		break;

	case RBUF_FAIL:
		printf("911-411 Inadequate receive buffers for open\n");
		break;

	case ADDR_FAIL:
		printf("911-412 Invalid NODE_ADDRESS\n");
		break;

	case RBUF_LEN_FAIL:
		printf("911-413 Invalid adapter receive buffer len\n");
		break;

	case XBUF_LEN_FAIL:
		printf("911-414 Invalid adapter transmit buffer len\n");
		break;

	default:
		printf("911-415 Unknown failure\n");
		break;
	}
}


/*
 * The following functions form the bulk of IP support routines for tkr.c:
 * 	tkattach(),
 *	tkget(), 
 *	tk_ipc_output(),
 *	tk_ipc_ostart(), 
 *	tk_ipc_ioctl()
 * These were taken from eth_un.c and revamped to make them function properly
 * within tkr.c.
 */

int tk_ipc_output(), tk_ipc_ioctl();

tkattach(unit)
{
	register struct tk_softc *tk_softc;
	register struct ifnet *ifp;

	tk_softc = (unit == ADAPTER0) ? &tk_primary->softc : &tk_alt->softc;

	ifp = &tk_softc->es_if;
	ifp->if_unit = unit;
	ifp->if_name = "tk";
	ifp->if_mtu = TOKEN_MTU;

	ifp->if_ioctl = tk_ipc_ioctl;
	ifp->if_output = tk_ipc_output;
	ifp->if_flags = IFF_BROADCAST | IFF_NOTRAILERS;
	if_attach(ifp);
}

struct mbuf *
tkget(addr, totlen, ifp)
register u_char *addr;
register int totlen;
struct ifnet *ifp;
{
	register int len;
	register struct mbuf *m;
	struct mbuf *top = NULL, **mp = &top;
	u_char *mcp;

	++ifp->if_ipackets;

	while (totlen > 0) {
		MGET(m, M_DONTWAIT, MT_DATA);
		if (m == NULL)
			goto bad;
		len = totlen;
		if (ifp != NULL)
			len += sizeof(ifp);
  		if (len >= mincluster) {
			MCLGET(m);
			if (m->m_len == CLBYTES)
				m->m_len = len = MIN(CLBYTES, len);
			else
				m->m_len = len = MIN(MLEN, len);
		}
		else {
			m->m_len = len = MIN(MLEN, len);
			m->m_off = MMINOFF;
		}
		mcp = mtod(m, u_char *);
		if (ifp != NULL) {
			* (mtod(m, struct ifnet **)) = ifp;
			mcp += sizeof(ifp);
			len -= sizeof(ifp);
			ifp = NULL;
		}
		if (len > TKR_RAMSIZE) {
			panic("911-416 tkget: size to large");
		}
		bcopy(addr, mcp, len);
		addr += len;
		*mp = m;
		mp = &m->m_next;
		totlen -= len;
	}
	return top;
bad:
	m_freem(top);
	return NULL;
}


tk_ipc_output(ifp, m0, dst)
struct ifnet *ifp;
struct mbuf *m0;
struct sockaddr *dst;
{
	register struct lan_llc_header *lh;
	register off;
	int type;
	struct lan_arp *ah;
	struct mbuf *m = m0;
	struct sockaddr_802_5 sa_tr;
	struct sockaddr_802_5 *sap = &sa_tr;
	struct in_addr idst;
	struct tk_softc *tk_softc;
	int error, usetrailers;
	short snap_type;
	int hdr_len, mac_len, llc_len;
	struct ie2_llc_hdr *llcp;
	struct ie5_mac_hdr *macp;
	spl_t s;

	/* Make sure that the net is up and running */
	if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING)) {
		error = ENETDOWN;
		goto bad;
	}

	/* Determine which adapter to use, primary or secondary */
	tk_softc = (ifp->if_unit == ADAPTER0) ?
			&tk_primary->softc : &tk_alt->softc;
		
	/* Figure out the MAC destination address */
	switch (dst->sa_family) {
	case AF_INET:
		debug(TOKENDBG, ("tk_ipc_output: AF_INET\n"));

		idst = ((struct sockaddr_in *) dst)->sin_addr;
		if (!ie5_arpresolve(&tk_softc->es_ac,m,&idst,sap,&usetrailers))
			return 0;
		off = ntohs((u_short)mtod(m, struct ip *)->ip_len) - m->m_len;
		if (usetrailers && off > 0 && (off & 0x1ff) == 0 &&
		    m->m_off >= MMINOFF + 2 * sizeof (u_short)) {
			type = ETHERTYPE_TRAIL + (off>>9);
			m->m_off -= 2 * sizeof (u_short);
			m->m_len += 2 * sizeof (u_short);
			*mtod(m, u_short *) = htons((u_short)LANTYPE_IP);
			*(mtod(m, u_short *) + 1) = htons((u_short)m->m_len);
			goto gottrailertype;
		}
		off = 0;
		goto gottype;
	case AF_UNSPEC:
		debug(TOKENDBG, ("tk_ipc_output: AF_UNSPEC\n"));

		sap = (struct sockaddr_802_5 *) dst;
		goto gottype;
	default:
		debug(TOKENDBG, ("tk_ipc_ostart: cannot handle address family 0x%x\n", dst->sa_family));
		error = EAFNOSUPPORT;
		goto bad;
	}

gottrailertype:

	/*
	 * Packet to be sent as trailer: move first packet
	 * (control information) to end of chain.
	 */
	while (m->m_next)
		m = m->m_next;
	m->m_next = m0;
	m = m0->m_next;
	m0->m_next = 0;
	m0 = m;

gottype:
	/*
	 * Add local net header.
	 *
	 * Calculate the hdr length, ie5_mac_hdr + ie2_llc_hdr.
	 */
	macp = &sap->sa_mac ;
	mac_len = mac_size( macp ) ;
	llc_len = sizeof( struct ie2_llc_hdr ) ;
	hdr_len = mac_len + llc_len ;

	/*
	 * Find enough room for the headers.
	 */
	if ( (m0->m_off > MMAXOFF) || (MMINOFF + hdr_len > m0->m_off) ) {
		m = m_get(M_DONTWAIT, MT_HEADER);
		if (m == 0) {
			error = ENOBUFS;
			goto bad ;
		}
		m->m_next = m0;
		m0 = m ;
		m0->m_off = MMINOFF;
		m0->m_len = hdr_len ;
	} else {
		m0->m_off -= hdr_len ;
		m0->m_len += hdr_len ;
	}

	/*
	 * Fill in the mac header.
	 */
	macp = mtod(m, struct ie5_mac_hdr *);
	bcopy( (caddr_t)&sap->sa_mac, (caddr_t)macp, mac_len ) ;

	/*
	 * Fill in the llc header.
	 */
	llcp = mac_to_llc( macp ) ;
	bcopy( (caddr_t)&sap->sa_llc, (caddr_t)llcp, llc_len ) ;

	s = splimp();
	if (IF_QFULL(&ifp->if_snd)) {
		IF_DROP(&ifp->if_snd);
		error = ENOBUFS;
		splx(s);
		goto qfull;
	}
	IF_ENQUEUE(&ifp->if_snd, m);
	tk_ipc_ostart(ifp);
	splx(s);
	return 0;
qfull:
	m0 = m;
bad:
	m_freem(m0);
	return error;
}

long	bks_hits;

tk_ipc_ostart(ifp)
struct ifnet *ifp;
{
	register int dhb_addr;
	register char *tk_sram;
	register u_long offset;
	int unit;
	struct board *tk_board;
	struct mbuf *m, *m0;
	long old_ifp_opackets;

	/* check to make sure that we have a token ring board */
	if (! have_tkr)
		return;

	debug(TOKENDBG, ("tk_ipc_ostart: entered\n"));

	unit = ifp->if_unit;
	tk_board = (unit == ADAPTER0) ? tk_primary : tk_alt;
		
	/* The adapter isn't ready for us to xmit yet */
	if (!tk_board->init_done) {
		debug(TOKENDBG,
			("tk_ipc_ostart: INIT not done\n"));
		return;
	}
	if (tk_board->need_trans) {
		debug(TOKENDBG,
			("tk_ipc_ostart: NEED to setup transmit\n"));
		return;
	}

	if (tk_board->active) {
		debug(TOKENDBG,("tk_ipc_ostart: token ring is busy\n"));
		return;
	}

	++tk_board->active;			/* mark as busy */

	/* setup temp variables */
	dhb_addr = tk_board->dhb_addr;
	tk_sram = tk_board->sram;

	old_ifp_opackets = ifp->if_opackets;
	
	IF_DEQUEUE(&ifp->if_snd, m);

	if (m == NULL) {
		tk_board->active--;
		return;
	}

	offset = 0;
	for (m0 = m; m0 != NULL; m0 = m0->m_next) {
		bcopy(mtod(m0, char *), &tk_sram[dhb_addr + offset],
			 m0->m_len);
		offset += m0->m_len;
	}
 	if (offset > 32767) {
 		bks_hits += 1;
 		ncprintf("tk_ipc_ostart: offset > 32767\n");
 	}

	m_freem(m);

	++ifp->if_opackets;

	/* setup a response to the adapter */
	tk_qasb(tk_board, TRANSMIT_DIR_FRAME, offset);
}

tk_ipc_ioctl(ifp, cmd, data)
struct ifnet *ifp;
int cmd;
caddr_t data;
{
	register struct ifaddr *ifa = (struct ifaddr *) data;
	struct board *tk_board;
	int error = 0;

	if (ifp->if_unit == 0)
		tk_board = tk_primary;
	else
		tk_board = tk_alt;

	switch (cmd) {
	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;

		switch (ifa->ifa_addr.sa_family) {
		case AF_INET:
			((struct arpcom *)ifp)->ac_ipaddr=IA_SIN(ifa)->sin_addr;
			arpwhohas((struct arpcom *)ifp, &IA_SIN(ifa)->sin_addr);
			break;
		}
		/* falls through */

	case SIOCSIFFLAGS:
		/*
		 * If we are in the process of opening or closing, store
		 * the last request (OPEN/CLOSE) in final_state and indicate
		 * that we have open/close request pending.  When
		 * the close/open is done, the adapter will interrupt
		 * us, an tkintr() will be called.  It is there, that
		 * we look at the final_state and close/open the adapter,
		 * accordingly.  
		 */
		if (tk_board->oc_in_progress) 
			break;

		if (ifp->if_flags & IFF_UP) {
			if (tk_board->softc.es_if.if_flags & IFF_RUNNING)
				break;
			else
				tk_open_adapter(tk_board);
		}
		else {
			if (tk_board->softc.es_if.if_flags & IFF_RUNNING)
				tk_close_adapter(tk_board);
		}
		break;

	default:
		error = EINVAL;
		break;
	}
	return error;
}
