/*
 * Copyright (c) 1982, 1986, 1989 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution is only permitted until one year after the first shipment
 * of 4.4BSD by the Regents.  Otherwise, redistribution and use in source and
 * binary forms are permitted provided that: (1) source distributions retain
 * this entire copyright notice and comment, and (2) distributions including
 * binaries display the following acknowledgement:  This product includes
 * software developed by the University of California, Berkeley and its
 * contributors'' in the documentation or other materials provided with the
 * distribution and in all advertising materials mentioning features or use
 * of this software.  Neither the name of the University nor the names of
 * its contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *	@(#)kern_time.c	7.13 (Berkeley) 6/28/90
 */

#include "param.h"
#include "user.h"
#include "kernel.h"
#include "proc.h"

#include "machine/reg.h"
#include "machine/cpu.h"

/* 
 * Time of day, system time and interval timer support.
 *
 * These routines provide the kernel entry points to get and set
 * the time-of-day and per-process interval timers.  Subroutines
 * here provide support for adding and subtracting timeval structures
 * and decrementing interval timers, optionally reloading the interval
 * timers when they expire.
 */

/* ARGSUSED */
gettimeofday(p, uap, retval)
	struct proc *p;
	register struct args {
		struct	timeval *tp;
		struct	timezone *tzp;
	} *uap;
	int *retval;
{
	struct timeval atv;
	int error = 0;

	if (uap->tp) {
		microtime(&atv);
		if (error = copyout((caddr_t)&atv, (caddr_t)uap->tp,
		    sizeof (atv)))
			return (error);
	}
	if (uap->tzp)
		error = copyout((caddr_t)&tz, (caddr_t)uap->tzp,
		    sizeof (tz));
	return (error);
}

settimeofday(p, uap, retval)
	struct proc *p;
	struct args {
		struct	timeval *tv;
		struct	timezone *tzp;
	} *uap;
	int *retval;
{
	struct timeval atv;
	struct timezone atz;
	int error, s;

	if (error = suser(u.u_cred, &u.u_acflag))
		return (error);
	if (uap->tv) {
		if (error = copyin((caddr_t)uap->tv, (caddr_t)&atv,
		    sizeof (struct timeval)))
			return (error);

		while (atv.tv_usec >= 1000000) {
			atv.tv_usec -= 1000000;
			atv.tv_sec++;
		}

		/*
		 * Update the time and the boottime.  The change to
		 * the boottime is computed by taking the difference
		 * between the current time and the time being set.
		 * We don't try too hard to be accurate (or we would
		 * call microtime() to get the current time).  We
		 * do try to avoid negative intermediate results so
		 * that this will work portably even if struct timeval's
		 * become unsigned.
		 */
		s = splclock();
		if (time.tv_sec > atv.tv_sec)
			boottime.tv_sec -= time.tv_sec - atv.tv_sec;
		else
			boottime.tv_sec += atv.tv_sec - time.tv_sec;

		if (time.tv_usec > atv.tv_usec) {
			boottime.tv_usec += 1000000
			    + atv.tv_usec - time.tv_usec;
			if (boottime.tv_usec >= 1000000)
				boottime.tv_usec -= 1000000;
			else
				boottime.tv_sec--;
		} else {
			boottime.tv_usec += atv.tv_usec - time.tv_usec;
			if (boottime.tv_usec >= 1000000) {
				boottime.tv_usec -= 1000000;
				boottime.tv_sec++;
			}
		}
		time = atv;
		splx(s);
		resettodr();
	}
	if (uap->tzp && (error = copyin((caddr_t)uap->tzp, (caddr_t)&atz,
	    sizeof (atz))) == 0)
		tz = atz;
	return (error);
}


/* ARGSUSED */
getsystimes(p, uap, retval)
	struct proc *p;
	register struct args {
		struct	timeval *utp;
		struct	timeval *btp;
	} *uap;
	int *retval;
{
	struct timeval utv;
	struct timeval btv;
	int error = 0;
	int s;

	if (uap->btp && !uap->utp) {
		/* only wants the boottime.  Easy optimization. */
		s = splclock(); btv = boottime; splx(s);
	} else if (uap->utp) {
		microsystime(&utv, &btv);
		if (error = copyout((caddr_t)&utv, (caddr_t)uap->utp,
		    sizeof (utv)))
			return (error);
	}

	if (uap->btp)
		error = copyout((caddr_t)&btv, (caddr_t)uap->btp, sizeof (btv));
	return (error);
}



int	tickdelta;			/* current clock skew, us. per tick */
struct	timeval timedelta;		/* adjustment yet to be applied */

/* ARGSUSED */
adjclock(p, uap, retval)
	struct proc *p;
	register struct args {
		struct timeadj *newadj;
		struct timeadj *oldadj;
	} *uap;
	int *retval;
{
	struct timeadj adj, oadj;
	int muststep, ntickdelta;
	int s, error;

	if (error = suser(u.u_cred, &u.u_acflag))
		return (error);

	/*
	 * Do the trivial case, a query for the current state
	 */
	if (!uap->newadj) {
		if (uap->oldadj) {
			s = splclock();
			oadj.ta_time = timedelta;
			oadj.ta_rate.tv_sec = 0;
			if (tickdelta < 0) {
				oadj.ta_flags = ADJ_SLOW;
				oadj.ta_rate.tv_usec = (-tickdelta) * hz;
			} else {
				oadj.ta_flags = 0;
				oadj.ta_rate.tv_usec = tickdelta * hz;
			}
			splx(s);
			error = copyout((caddr_t)&oadj, (caddr_t)uap->oldadj,
			    sizeof (struct timeadj));
		}
		return (error);
	}

	error = copyin((caddr_t)uap->newadj, (caddr_t)&adj, sizeof (adj));
	if (error != 0)
		return (error);

	/*
	 * Limit slew rate to 50%.  Step if not in range.  Steps
	 * are done by adding the adjustment to boottime and then
	 * recomputing time.  Slews are done by fixing up tickdelta
	 * and timedelta.  All of this must be done with the
	 * interrupts held off, unfortunately, but we do what we
	 * can beforehand.
	 */
	if (adj.ta_rate.tv_sec != 0 || adj.ta_rate.tv_usec == 0 ||
	    adj.ta_rate.tv_usec > 500000) {
		if (adj.ta_flags & ADJ_SLEWONLY) {
			if (uap->oldadj) {
				oadj.ta_time.tv_sec = oadj.ta_time.tv_usec = 0;
				oadj.ta_rate.tv_sec = 0;
				oadj.ta_rate.tv_usec = 500000;
				oadj.ta_flags = 0;
				error = copyout((caddr_t)&oadj,
				    (caddr_t)uap->oldadj,
				    sizeof (struct timeadj));
				if (error)
					return (error);
			}
			return (ERANGE);
		}
		muststep = 1;
	} else {
		ntickdelta = (int)((adj.ta_rate.tv_usec + (long)hz) / (long)hz);
		muststep = 0;
	}
	oadj.ta_time.tv_sec = oadj.ta_time.tv_usec = 0;

	s = splclock();
	if (tickdelta < 0) {
		oadj.ta_flags = ADJ_SLOW;
		oadj.ta_rate.tv_usec = (time_t)(-tickdelta);
	} else {
		oadj.ta_flags = 0;
		oadj.ta_rate.tv_usec = (time_t)(tickdelta);
	}
	if (tickdelta && uap->oldadj)
		oadj.ta_time = timedelta;

	/*
	 * If the adjustment is cumulative we need to add this adjustment
	 * to the remainder of the previous one.
	 */
	if (tickdelta != 0 && (adj.ta_flags & ADJ_CUMULATIVE)) {
		if ((adj.ta_flags ^ oadj.ta_flags) & ADJ_SLOW) {
			if (timercmp(&(adj.ta_time), &timedelta, >)) {
				timedelta.tv_sec += adj.ta_time.tv_sec
				    - timedelta.tv_sec;
				timedelta.tv_usec = adj.ta_time.tv_usec
				    + 1000000 - timedelta.tv_usec;
			} else {
				adj.ta_flags ^= ADJ_SLOW;
				timedelta.tv_sec -= adj.ta_time.tv_sec;
				timedelta.tv_usec += 1000000 -
				    adj.ta_time.tv_usec;
			}
			if (timedelta.tv_usec >= 1000000)
				timedelta.tv_usec -= 1000000;
			else
				timedelta.tv_sec--;
		} else {
			timedelta.tv_sec += adj.ta_time.tv_sec;
			timedelta.tv_usec += adj.ta_time.tv_usec;
			if (timedelta.tv_usec >= 1000000) {
				timedelta.tv_usec -= 1000000;
				timedelta.tv_sec++;
			}
		}
	} else {
		timedelta = adj.ta_time;
	}

	/*
	 * Okay, timedelta holds the total adjustment to be done.  If we're
	 * stepping, add this to boottime and recompute time.  Otherwise,
	 * set tickdelta from the slew rate.
	 */
	if (muststep) {
		if (adj.ta_flags & ADJ_SLOW) {
			boottime.tv_sec -= timedelta.tv_sec;
			boottime.tv_usec += 1000000 - timedelta.tv_usec;
			if (boottime.tv_usec >= 1000000)
				boottime.tv_usec -= 1000000;
			else
				boottime.tv_sec--;
		} else {
			boottime.tv_sec += timedelta.tv_sec;
			boottime.tv_usec += timedelta.tv_usec;
			if (boottime.tv_usec > 1000000) {
				boottime.tv_usec -= 1000000;
				boottime.tv_sec++;
			}
		}
		timedelta.tv_sec = timedelta.tv_usec = 0;
		tickdelta = 0;

		/* XXX: should eliminate the time variable completely */
		time.tv_sec = uptime.tv_sec + boottime.tv_sec;
		time.tv_usec = uptime.tv_usec + boottime.tv_usec;
		if (time.tv_usec > 1000000) {
			time.tv_usec -= 1000000;
			time.tv_sec++;
		}
	} else {
		if (timedelta.tv_sec == 0 && timedelta.tv_usec == 0)
			tickdelta = 0;
		else if (adj.ta_flags & ADJ_SLOW)
			tickdelta = -ntickdelta;
		else
			tickdelta = ntickdelta;
	}
	splx(s);

	/*
	 * Finally, if we need to return the old value, clean it up
	 * and copy it out.
	 */
	if (uap->oldadj) {
		oadj.ta_rate.tv_sec = 0;
		oadj.ta_rate.tv_usec *= (time_t)hz;
		error = copyout((caddr_t)&oadj, (caddr_t)uap->oldadj,
		    sizeof (struct timeadj));
	}
	return (error);
}

/* XXX: this is obsolete */
extern	int tickadj;			/* "standard" clock skew, us./tick */
long	bigadj = 1000000;		/* use 10x skew above bigadj us. */

/* ARGSUSED */
adjtime(p, uap, retval)
	struct proc *p;
	register struct args {
		struct timeval *delta;
		struct timeval *olddelta;
	} *uap;
	int *retval;
{
	struct timeval atv, oatv;
	register long ndelta;
	register int ntickdelta;
	register int otickdelta;
	int s, error;

	if (error = suser(u.u_cred, &u.u_acflag))
		return (error);
	if (error =
	    copyin((caddr_t)uap->delta, (caddr_t)&atv, sizeof (struct timeval)))
		return (error);
	ndelta = atv.tv_sec * 1000000 + atv.tv_usec;
	if (ndelta > bigadj || ndelta < (-bigadj))
		ntickdelta = 10 * tickadj;
	else if (ndelta)
		ntickdelta = tickadj;
	else
		ntickdelta = 0;

	if (ndelta < 0) {
		ntickdelta = (-ntickdelta);
		ndelta = (-ndelta);
	}

	atv.tv_sec = ndelta / 1000000;
	atv.tv_usec = ndelta % 1000000;

	s = splclock();
	if (uap->olddelta)
		oatv = timedelta;
	timedelta = atv;
	otickdelta = tickdelta;
	tickdelta = ntickdelta;
	splx(s);

	if (uap->olddelta) {
		if (otickdelta < 0) {
			oatv.tv_sec = (-oatv.tv_sec);
			oatv.tv_usec = (-oatv.tv_usec);
		}
		(void) copyout((caddr_t)&oatv, (caddr_t)uap->olddelta,
			sizeof (struct timeval));
	}
	return (0);
}

/*
 * Get value of an interval timer.  The process virtual and
 * profiling virtual time timers are kept in the u. area, since
 * they can be swapped out.  These are kept internally in the
 * way they are specified externally: in time until they expire.
 *
 * The real time interval timer is kept in the process table slot
 * for the process, and its value (it_value) is kept as an
 * absolute system time rather than as a delta, so that it is easy
 * to keep periodic real-time signals from drifting.  System uptime
 * is used as the time to immunize the timers against time-of-day
 * changes.
 *
 * Virtual time timers are processed in the hardclock() routine of
 * kern_clock.c.  The real time timer is processed by a timeout
 * routine, called from the softclock() routine.  Since a callout
 * may be delayed in real time due to interrupt processing in the system,
 * it is possible for the real time timeout routine (realitexpire, given below),
 * to be delayed in real time past when it is supposed to occur.  It
 * does not suffice, therefore, to reload the real timer .it_value from the
 * real time timers .it_interval.  Rather, we compute the next time in
 * absolute time the timer should go off.
 */
/* ARGSUSED */
getitimer(p, uap, retval)
	struct proc *p;
	register struct args {
		u_int	which;
		struct	itimerval *itv;
	} *uap;
	int *retval;
{
	struct itimerval aitv;
	int s;

	if (uap->which > ITIMER_PROF)
		return (EINVAL);
	s = splclock();
	if (uap->which == ITIMER_REAL) {
		/*
		 * Convert from absoulte to relative time in .it_value
		 * part of real time timer.  If time for real time timer
		 * has passed return 0, else return difference between
		 * current time and time for the timer to go off.
		 */
		aitv = p->p_realtimer;
		if (timerisset(&aitv.it_value))
			if (timercmp(&aitv.it_value, &uptime, <))
				timerclear(&aitv.it_value);
			else
				timevalsub(&aitv.it_value, &uptime);
	} else
		aitv = u.u_timer[uap->which];
	splx(s);
	return (copyout((caddr_t)&aitv, (caddr_t)uap->itv,
	    sizeof (struct itimerval)));
}

/* ARGSUSED */
setitimer(p, uap, retval)
	struct proc *p;
	register struct args {
		u_int	which;
		struct	itimerval *itv, *oitv;
	} *uap;
	int *retval;
{
	struct itimerval aitv;
	register struct itimerval *itvp;
	int s, error;

	if (uap->which > ITIMER_PROF)
		return (EINVAL);
	itvp = uap->itv;
	if (itvp && (error = copyin((caddr_t)itvp, (caddr_t)&aitv,
	    sizeof(struct itimerval))))
		return (error);
	if ((uap->itv = uap->oitv) && (error = getitimer(p, uap, retval)))
		return (error);
	if (itvp == 0)
		return (0);
	if (itimerfix(&aitv.it_value) || itimerfix(&aitv.it_interval))
		return (EINVAL);
	s = splclock();
	if (uap->which == ITIMER_REAL) {
		untimeout(realitexpire, (caddr_t)p);
		if (timerisset(&aitv.it_value)) {
			timevaladd(&aitv.it_value, &uptime);
			timeout(realitexpire, (caddr_t)p, hzto(&aitv.it_value));
		}
		p->p_realtimer = aitv;
	} else
		u.u_timer[uap->which] = aitv;
	splx(s);
	return (0);
}

/*
 * Real interval timer expired:
 * send process whose timer expired an alarm signal.
 * If time is not set up to reload, then just return.
 * Else compute next time timer should go off which is > current time.
 * This is where delay in processing this timeout causes multiple
 * SIGALRM calls to be compressed into one.
 */
realitexpire(p)
	register struct proc *p;
{
	int s;

	psignal(p, SIGALRM);
	if (!timerisset(&p->p_realtimer.it_interval)) {
		timerclear(&p->p_realtimer.it_value);
		return;
	}
	for (;;) {
		s = splclock();
		timevaladd(&p->p_realtimer.it_value,
		    &p->p_realtimer.it_interval);
		if (timercmp(&p->p_realtimer.it_value, &uptime, >)) {
			timeout(realitexpire, (caddr_t)p,
			    hzto(&p->p_realtimer.it_value));
			splx(s);
			return;
		}
		splx(s);
	}
}

/*
 * Check that a proposed value to load into the .it_value or
 * .it_interval part of an interval timer is acceptable, and
 * fix it to have at least minimal value (i.e. if it is less
 * than the resolution of the clock, round it up.)
 */
itimerfix(tv)
	struct timeval *tv;
{

	if (tv->tv_sec < 0 || tv->tv_sec > 100000000 ||
	    tv->tv_usec < 0 || tv->tv_usec >= 1000000)
		return (EINVAL);
	if (tv->tv_sec == 0 && tv->tv_usec != 0 && tv->tv_usec < tick)
		tv->tv_usec = tick;
	return (0);
}

/*
 * Decrement an interval timer by a specified number
 * of microseconds, which must be less than a second,
 * i.e. < 1000000.  If the timer expires, then reload
 * it.  In this case, carry over (usec - old value) to
 * reducint the value reloaded into the timer so that
 * the timer does not drift.  This routine assumes
 * that it is called in a context where the timers
 * on which it is operating cannot change in value.
 */
itimerdecr(itp, usec)
	register struct itimerval *itp;
	int usec;
{

	if (itp->it_value.tv_usec < usec) {
		if (itp->it_value.tv_sec == 0) {
			/* expired, and already in next interval */
			usec -= itp->it_value.tv_usec;
			goto expire;
		}
		itp->it_value.tv_usec += 1000000;
		itp->it_value.tv_sec--;
	}
	itp->it_value.tv_usec -= usec;
	usec = 0;
	if (timerisset(&itp->it_value))
		return (1);
	/* expired, exactly at end of interval */
expire:
	if (timerisset(&itp->it_interval)) {
		itp->it_value = itp->it_interval;
		itp->it_value.tv_usec -= usec;
		if (itp->it_value.tv_usec < 0) {
			itp->it_value.tv_usec += 1000000;
			itp->it_value.tv_sec--;
		}
	} else
		itp->it_value.tv_usec = 0;		/* sec is already 0 */
	return (0);
}

/*
 * Add and subtract routines for timevals.
 * N.B.: subtract routine doesn't deal with
 * results which are before the beginning,
 * it just gets very confused in this case.
 * Caveat emptor.
 */
timevaladd(t1, t2)
	struct timeval *t1, *t2;
{

	t1->tv_sec += t2->tv_sec;
	t1->tv_usec += t2->tv_usec;
	timevalfix(t1);
}

timevalsub(t1, t2)
	struct timeval *t1, *t2;
{

	t1->tv_sec -= t2->tv_sec;
	t1->tv_usec -= t2->tv_usec;
	timevalfix(t1);
}

timevalfix(t1)
	struct timeval *t1;
{

	if (t1->tv_usec < 0) {
		t1->tv_sec--;
		t1->tv_usec += 1000000;
	}
	if (t1->tv_usec >= 1000000) {
		t1->tv_sec++;
		t1->tv_usec -= 1000000;
	}
}
