/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)tty_tb.c	7.5 (Berkeley) 5/9/89
 */

#include "tb.h"
#if NTB > 0

/*
 * Line discipline for RS232 tablets;
 * supplies binary coordinate data.
 */
#include "param.h"
#include "user.h"
#include "tablet.h"
#include "tty.h"

#include <machine/screen_conf.h>

static	int tbdecode(), gtcodecode(), poldecode();
static	int tblresdecode(), tbhresdecode();
static	int mspcdecode(), msplan2decode(), msplan3decode();

/*
 * Tablet configuration table.
 */
struct	tbconf {
	short	tbc_recsize;	/* input record size in bytes */
	short	tbc_uiosize;	/* size of data record returned user */
	int	tbc_sync;	/* mask for finding sync byte/bit */
	int	tbc_data_type;	/* 1 = ASCII  0 = BINARY */
	int	(*tbc_decode)();/* decoding routine */
	char	*tbc_run;	/* enter run mode sequence */
	char	*tbc_point;	/* enter point mode sequence */
	char	*tbc_stop;	/* stop sequence */
	char	*tbc_start;	/* start/restart sequence */
	int	tbc_flags;
#define	TBF_POL		0x1	/* polhemus hack */
#define	TBF_INPROX	0x2	/* tablet has proximity info */
};

#define NTBS	16
struct tb tb[NTBS];

struct	tbconf tbconf[TBTYPE] = {
{ 0 },
{ 5, sizeof (struct tbmspos), 0200, 1, tbdecode, "6", "4" },
{ 5, sizeof (struct tbmspos), 0200, 1, tbdecode, "\1CN", "\1RT", "\2", "\4" },
{ 8, sizeof (struct gtcopos), 0200, 1, gtcodecode },
{17, sizeof (struct polpos), 0200, 1, poldecode, 0, 0, "\21", "\5\22\2\23",
  TBF_POL },
{ 5, sizeof (struct tbmspos), 0100, 1, tblresdecode, "\1CN", "\1PT", "\2", "\4",
  TBF_INPROX },
{ 6, sizeof (struct tbmspos), 0200, 1, tbhresdecode, "\1CN", "\1PT", "\2", "\4",
  TBF_INPROX },
{ 5, sizeof (struct tbmspos), 0100, 1, tblresdecode, "\1CL\33", "\1PT\33", 0, 0},
{ 6, sizeof (struct tbmspos), 0200, 1, tbhresdecode, "\1CL\33", "\1PT\33", 0, 0},
{ 5, sizeof (struct tbmspos), 0x80, 0, mspcdecode, 0, 0, 0, 0},
{ 4, sizeof (struct tbmspos), 0x0B, 0, msplan3decode, 0, 0, 0, 0},
{ 4, sizeof (struct tbmspos), 0x0B, 0, msplan2decode, 0, 0, 0, 0},
};

/*
 * Open as tablet discipline; called on discipline change.
 */
/*ARGSUSED*/
tbopen(dev, tp)
	dev_t dev;
	register struct tty *tp;
{
	register struct tb *tbp;

	if (tp->t_line == TABLDISC)
		return (EINVAL);
	ttywflush(tp);
	for (tbp = tb; tbp < &tb[NTBS]; tbp++)
		if (tbp->tbflags == 0)
			break;
	if (tbp >= &tb[NTBS])
		return (EBUSY);
	tbp->tbflags = TBTIGER|TBPOINT;		/* default */
	tbp->last_char = tbp->cbuf;
	tbp->num_chars = 0;
	bzero((caddr_t)&tbp->rets, sizeof (tbp->rets));
	tp->T_LINEP = (caddr_t)tbp;
	tp->t_flags |= LITOUT;
	return (0);
}

/*
 * Line discipline change or last device close.
 */
tbclose(tp)
	register struct tty *tp;
{
	register int s;
	int modebits = TBPOINT|TBSTOP;

	tbioctl(tp, BIOSMODE, &modebits, 0);
	s = spltty();
	((struct tb *)tp->T_LINEP)->tbflags = 0;
	tp->T_LINEP = 0;
	tp->t_rawq.c_cc = 0;		/* clear queues -- paranoid */
	tp->t_canq.c_cc = 0;
	tp->t_line = 0;			/* paranoid: avoid races */
	splx(s);
}

/*
 * Read from a tablet line.
 * Characters have been buffered in a buffer and decoded.
 */
tbread(tp, uio, flag)
	register struct tty *tp;
	struct uio *uio;
	int flag;
{
	register struct tb *tbp = (struct tb *)tp->T_LINEP;
	register struct tbconf *tc = &tbconf[tbp->tbflags & TBTYPE];
	int ret;

	if ((tp->t_state&TS_CARR_ON) == 0)
		return (EIO);
	ret = uiomove(&tbp->rets, tc->tbc_uiosize, uio);
	if (tc->tbc_flags&TBF_POL)
		tbp->rets.polpos.p_key = ' ';
	return (ret);
}

/* for emulator support */
extern int		cons_if;
extern SCREEN_INFO	cons_info[];
#define ms_info (cons_info[cons_if])

/*
 * Low level character input routine.
 * Stuff the character in the buffer, and decode
 * if all the chars are there.
 *
 * This routine could be expanded in-line in the receiver
 * interrupt routine to make it run as fast as possible.
 */
tbinput(c, tp)
	register int c;
	register struct tty *tp;
{
	register struct tb *tbp = (struct tb *)tp->T_LINEP;
	register struct tbconf *tc = &tbconf[tbp->tbflags & TBTYPE];
	register SCREEN_INFO *si = &ms_info;

	if (tc->tbc_recsize == 0 || tc->tbc_decode == 0)	/* paranoid? */
		return;

	/*
	 * Don't start until we locate sync bit/byte.
	 */
	if ((tbp->num_chars == 0) && !(c&tc->tbc_sync))
		return;

	if (tc->tbc_data_type)
		*tbp->last_char++ = c & 0177;
	else
		*tbp->last_char++ = c;

	/* 
	 * Call decode routine only if a full record has been collected, and
	 * then reset the input buffer. The return check on decode is for the
	 * 3 button emulation on a 3 button mouse. We don't call record_pos()
	 * while we wait to see if the other button will be pressed/released.
	 */
	if (++tbp->num_chars == tc->tbc_recsize) {
		if ((*tc->tbc_decode)(tc, tbp->cbuf, &tbp->rets, tp, si))
			record_pos(tbp, tp, si);
		tbp->last_char = tbp->cbuf;
		tbp->num_chars = 0;
	
	}
}


/*
 * Give the new event to the emulator.
 */
static
record_pos(tbp, tp, si)
	struct tb *tbp;
	struct tty *tp;
	SCREEN_INFO *si;
{
	register struct tbmspos *tbmspos = &(tbp->rets.tbmspos);
	register int s;

	wakeup ((caddr_t)tbmspos);

	/* Give the emulator the data */
	s = spl5();
	si->ine.intr_reason = E_INT_MOUSE;
	(*emulsw[si->ine.emulator].e_rint)(0, tp, si);
	splx(s);
}


/*
 * Decode GTCO 8 byte format (high res, tilt, and pressure).
 */
static
gtcodecode(tc, cp, tbpos, tp, si)
	struct tbconf *tc;
	register char *cp;
	register struct gtcopos *tbpos;
	struct tty *tp;
	SCREEN_INFO *si;
{

	tbpos->pressure = *cp >> 2;
	tbpos->status = (tbpos->pressure > 16) | TBINPROX; /* half way down */
	tbpos->xpos = (*cp++ & 03) << 14;
	tbpos->xpos |= *cp++ << 7;
	tbpos->xpos |= *cp++;
	tbpos->ypos = (*cp++ & 03) << 14;
	tbpos->ypos |= *cp++ << 7;
	tbpos->ypos |= *cp++;
	tbpos->xtilt = *cp++;
	tbpos->ytilt = *cp++;
	tbpos->scount++;

	return(1);
}

/*
 * Decode old Hitachi 5 byte format (low res).
 */
static
tbdecode(tc, cp, tbpos, tp, si)
	struct tbconf *tc;
	register char *cp;
	register struct tbmspos *tbpos;
	struct tty *tp;
	SCREEN_INFO *si;
{
	register char byte;

	byte = *cp++;
	tbpos->status = (byte&0100) ? TBINPROX : 0;
	byte &= ~0100;
	if (byte > 036)
		tbpos->status |= 1 << ((byte-040)/2);
	tbpos->xpos = *cp++ << 7;
	tbpos->xpos |= *cp++;
	if (tbpos->xpos < 256)			/* tablet wraps around at 256 */
		tbpos->status &= ~TBINPROX;	/* make it out of proximity */
	tbpos->ypos = *cp++ << 7;
	tbpos->ypos |= *cp++;
	tbpos->scount++;

	return(1);
}

/*
 * Decode new Hitach 5-byte format (low res).
 */
static
tblresdecode(tc, cp, tbpos, tp, si)
	struct tbconf *tc;
	register char *cp;
	register struct tbmspos *tbpos;
	struct tty *tp;
	SCREEN_INFO *si;
{

	*cp &= ~tc->tbc_sync;		/* mask sync bit */
	tbpos->status = (*cp++ >> 2) | TBINPROX;
	if (tc->tbc_flags&TBF_INPROX && tbpos->status&020)
		tbpos->status &= ~(020|TBINPROX);
	tbpos->xpos = *cp++;
	tbpos->xpos |= *cp++ << 6;
	tbpos->ypos = *cp++;
	tbpos->ypos |= *cp++ << 6;
	tbpos->scount++;

	return(1);
}

/*
 * Decode new Hitach 6-byte format (high res).
 */
static
tbhresdecode(tc, cp, tbpos, tp, si)
	struct tbconf *tc;
	register char *cp;
	register struct tbmspos *tbpos;
	struct tty *tp;
	SCREEN_INFO *si;
{
	char byte;

	byte = *cp++;
	tbpos->xpos = (byte & 03) << 14;
	tbpos->xpos |= *cp++ << 7;
	tbpos->xpos |= *cp++;
	tbpos->ypos = *cp++ << 14;
	tbpos->ypos |= *cp++ << 7;
	tbpos->ypos |= *cp++;
	tbpos->status = (byte >> 2) | TBINPROX;
	if (tc->tbc_flags&TBF_INPROX && tbpos->status&020)
		tbpos->status &= ~(020|TBINPROX);
	tbpos->scount++;

	return(1);
}

/*
 * Polhemus decode.
 */
static
poldecode(tc, cp, polpos, tp, si)
	struct tbconf *tc;
	register char *cp;
	register struct polpos *polpos;
	struct tty *tp;
	SCREEN_INFO *si;
{

	polpos->p_x = cp[4] | cp[3]<<7 | (cp[9] & 0x03) << 14;
	polpos->p_y = cp[6] | cp[5]<<7 | (cp[9] & 0x0c) << 12;
	polpos->p_z = cp[8] | cp[7]<<7 | (cp[9] & 0x30) << 10;
	polpos->p_azi = cp[11] | cp[10]<<7 | (cp[16] & 0x03) << 14;
	polpos->p_pit = cp[13] | cp[12]<<7 | (cp[16] & 0x0c) << 12;
	polpos->p_rol = cp[15] | cp[14]<<7 | (cp[16] & 0x30) << 10;
	polpos->p_stat = cp[1] | cp[0]<<7;
	if (cp[2] != ' ')
		polpos->p_key = cp[2];

	return(1);
}

#ifndef SIGNEXTEND
#undef SIGNEXTEND
#endif

#ifdef  ibm032
#define SIGNEXTEND(c)  ((int)(c)>=128 ? (int)(c)-256 : (int)(c))
#else
#define SIGNEXTEND(n)  (n)
#endif

/* Generic button definitions */
#define RIGHT_BUT		0x04
#define MIDDLE_BUT		0x02
#define LEFT_BUT		0x01
#define NO_BUT			0x00


/*
 *       SUPPORTS:
 *              PC MOUSE by Mouse Systems
 * 
 *      DATA FORMAT:
 *              8-BIT 3-BYTE BINARY DELTA
 *
 *      Stop   7   6   5   4   3   2   1   0   Start
 *      Bit                                     Bit
 *      --------------------------------------------
 *      1      1   0   0   0   0   L   M   R     0
 *      1      X7  X6  X5  X4  X3  X2  X1  X0    0
 *      1      Y7  Y6  Y5  Y4  Y3  Y2  Y1  Y0    0
 *      1      X7  X6  X5  X4  X3  X2  X1  X0    0
 *      1      Y7  Y6  Y5  Y4  Y3  Y2  Y1  Y0    0
 *      --------------------------------------------
 *
 *      key:
 *
 *      Bits 4-7  = Sync bits
 *                1000 -  identifies first byte in the format;
 *
 *      L      = Left   Button
 *      M      = middle Button
 *      R      = Right  Button
 *      L,M,R  = Action Key indicators: 0 = key pressed
 *                                      1 = key not pressed
 *
 *      X0-X7  = X Bits (2's complement)
 *      Y0-Y7  = Y Bits (2's complement)
 *      X0,Y0  = Least Significant Bit
 *
 */
#define MCS_BUTTONS_MASK	0x07
#define MCS_RIGHT_BUT		0x01
#define MCS_MIDDLE_BUT		0x02
#define MCS_LEFT_BUT		0x04

#define MCS_BIGSTEP		64

static
mspcdecode(tc, cp, mspos, tp, si)
	struct tbconf *tc;
        register char *cp;
        register struct tbmspos *mspos;
	struct tty *tp;
	SCREEN_INFO *si;
{
	register short status;
	register short *pstatus;
	register int xpos = 0;
	register int ypos = 0;
	register    incr;
	int i;

	*(pstatus = &mspos->status) = 0;
	status =  (~(*cp++)) & 7;

	if (status & MCS_LEFT_BUT)
		*pstatus |= LEFT_BUT;
	if (status & MCS_MIDDLE_BUT)
		*pstatus |= MIDDLE_BUT;
	if (status & MCS_RIGHT_BUT)
		*pstatus |= RIGHT_BUT;

	for (i = 1; i <= 4; i++) {
#if 0
		if ((*cp > MCS_BIGSTEP) && (*cp < (256-MCS_BIGSTEP))) {
			mspos->xpos = 0;
			mspos->ypos = 0;
			return (0);
		}
#endif
		incr = *cp++;
		incr = SIGNEXTEND(incr);

		switch (i) {
		case 1: 
		case 3: 
			xpos += incr;
			break;
		case 2: 
		case 4: 
			ypos -= incr;
			break;
		}
	}
	mspos->xpos = xpos;
	mspos->ypos = ypos;
	mspos->scount++;

	return(1);
}


/*
 *
 *	SUPPORTS:                                                              
 *		2 BUTTON PLANAR MOUSE
 *
 *	DATA FORMAT:
 *		4-BYTE BINARY
 *
 *
 *	BYTE 1 - Sync	0x0B
 *
 *	BYTE 2 - Status
 *
 *		Bit 0	
 *		Bit 1	Y Data Sign		1 = Negative
 *		Bit 2	
 *		Bit 3	X Data Sign		1 = Negative
 *		Bit 4	
 *		Bit 5	Left button status	1 = Depressed
 *		Bit 6	
 *		Bit 7	Right button status	1 = Depressed
 *
 *	BYTE 3 - X Data Byte  (Two's complement, Bit 0 = LSB)
 *
 *	BYTE 4 - Y Data Byte  (Two's complement, Bit 0 = LSB)
 *
 */

#define XNEG		0x04
#define YNEG		0x02

#define RT_BUT_MASK		0xE0
#define RT_RIGHT_BUT		0x80
#define RT_MIDDLE_BUT		0x40
#define RT_LEFT_BUT		0x20

extern int mstype;		/* XXX: temp: from rtio/mouse.c */
				/* XXX: NEED TO ENABLE mouse acceleration */
				/*	in x_emul.c */

/* 2 Button reporting for Planar Mouse */
static
msplan2decode(tc, cp, mspos, tp, si)
	struct tbconf *tc;
        register char *cp;
        register struct tbmspos *mspos;
	struct tty *tp;
	SCREEN_INFO *si;
{
	register short status;
	register short *pstatus;
	register int xpos, ypos;

	status = *++cp;
	*(pstatus = &mspos->status) = 0;
	if (status & RT_LEFT_BUT)
		*pstatus |= LEFT_BUT;
	if (status & RT_MIDDLE_BUT)
		*pstatus |= MIDDLE_BUT;
	if (status & RT_RIGHT_BUT)
		*pstatus |= RIGHT_BUT;

	/* Figure x delta */
	xpos = (*++cp)>>1;
        xpos = (status & XNEG)? (xpos) | 0xffffffC0 : (xpos);

	/* Figure y delta (negative since downward movement of mouse
	 * causes a negative delta and most applications want that to be
	 * a possitive delta)
	 */
	ypos = (*++cp)>>1;
        ypos = -((status & YNEG)? (ypos) | 0xffffffC0 : (ypos));

	if (mstype) {	/* XXX: speed up the optical mouse */
	    mspos->ypos = 2*ypos;
	    mspos->xpos = 2*xpos;
	}
	else {
	    mspos->ypos = ypos;
	    mspos->xpos = xpos;
	}
	mspos->scount++;
	
	return(1);
}


static int single_but = 0;
static int lastbut = 0;
static struct tty *ptp;
static SCREEN_INFO *psi;
#define SETTIMEOUT(a, b, c, tp, si) {\
				ptp = tp; psi = si;\
				timeout(a, b, c);\
			}

/* 3 Button Emulation for Planar Mouse */
static
msplan3decode(tc, cp, mspos, tp, si)
	struct tbconf *tc;
        register char *cp;
        register struct tbmspos *mspos;
	struct tty *tp;
	SCREEN_INFO *si;
{
	int pollms();

	/* use msplan2decode() to do the decoding, since it's the same mouse */
	msplan2decode(tc, cp, mspos, tp, si);
	if (mstype)	/* XXX: dont do 3-emulation with optical mouse */
	    return (1);

	/* Check if both buttons are down and simulate a middle button */
	if (mspos->status == ( RIGHT_BUT | LEFT_BUT ))
		mspos->status = MIDDLE_BUT;

	if (!single_but && (lastbut != mspos->status)) {
		/* were holding down 2 buttons and have released one */
		if ((lastbut == MIDDLE_BUT) && (mspos->status != NO_BUT)) {
			mspos->status = MIDDLE_BUT;
			single_but = 0;

		/* 1 button pressed, wait for 2nd button */
		}else if ((lastbut == NO_BUT)&&(mspos->status != MIDDLE_BUT)) {
			single_but++;
			SETTIMEOUT(pollms, mspos, 4, tp, si);
		}

	/* no change yet, should we continue to wait for other button? */
	} else if ((single_but) && (lastbut == mspos->status)) {
		untimeout(pollms, mspos);

		/* waited enough, call it 1 button */
		if (++single_but > 2)
			single_but = 0;
		else
			SETTIMEOUT(pollms, mspos, 4, tp, si);

	/* either released button or pressed 2nd button, so lets record it */
	} else if (single_but) {
		untimeout(pollms, mspos);
		single_but = 0;
		
		/* released the 1 button so record up and down events */
		if (mspos->status == NO_BUT) {
			struct tb *tbp;
			
			/* Record the Previous down button */
			mspos->status = lastbut;
			tbp = (struct tb *) (tp->T_LINEP);
			record_pos(tbp, tp, si);

			/* Record the up motion */
			mspos->xpos = mspos->ypos = mspos->status = 0;
		}
	}
	lastbut = mspos->status;
	return(!single_but);
}

static
pollms (mspos)
	struct tbmspos *mspos;
{
	register struct tb *tbp;

	tbp = (struct tb *) (ptp->T_LINEP);

	/* waited enough, call it 1 button and record it */
	if (++single_but > 2) {
		single_but = 0;
		record_pos(tbp, ptp, psi);
	} else
		SETTIMEOUT(pollms, mspos, 2, ptp, psi);
}


/*ARGSUSED*/
tbioctl(tp, cmd, data, flag)
	struct tty *tp;
	caddr_t data;
{
	register struct tb *tbp = (struct tb *)tp->T_LINEP;

	switch (cmd) {

	case BIOGMODE:
		*(int *)data = tbp->tbflags & TBMODE;
		break;

	case BIOSTYPE:
		if (tbconf[*(int *)data & TBTYPE].tbc_recsize == 0 ||
		    tbconf[*(int *)data & TBTYPE].tbc_decode == 0)
			return (EINVAL);
		tbp->tbflags &= ~TBTYPE;
		tbp->tbflags |= *(int *)data & TBTYPE;
		/* fall thru... to set mode bits */

	case BIOSMODE: {
		register struct tbconf *tc;

		tbp->tbflags &= ~TBMODE;
		tbp->tbflags |= *(int *)data & TBMODE;
		tc = &tbconf[tbp->tbflags & TBTYPE];
		if (tbp->tbflags&TBSTOP) {
			if (tc->tbc_stop)
				ttyoutput(tc->tbc_stop, tp);
		} else if (tc->tbc_start)
			ttyoutput(tc->tbc_start, tp);
		if (tbp->tbflags&TBPOINT) {
			if (tc->tbc_point)
				ttyoutput(tc->tbc_point, tp);
		} else if (tc->tbc_run)
			ttyoutput(tc->tbc_run, tp);
		ttstart(tp);
		break;
	}

	case BIOGTYPE:
		*(int *)data = tbp->tbflags & TBTYPE;
		break;

	case TIOCSETD:
	case TIOCGETD:
	case TIOCGETP:
	case TIOCGETC:
		return (-1);		/* pass thru... */

	default:
		return (ENOTTY);
	}
	return (0);
}

/* 
 * Since the read always returns whatever is in tbc->rets regardless of
 * whether it is new old or has even been writen to at all, we just return 1.
 *
 * This is called explicitly (since there isn't an entry in linesw for
 * select) from msselect in mouse.c if tp->t_line is TABLDISC.
 */
tbselect(dev, rw)
	dev_t dev;
	int rw;
{
	return(1);
}

#endif /* NTB */
