#include "aha.h"
#if NAHA > 0

/*
 * Copyright (C) 1988, 1989 Christer Bernerus, 
 *		 Chalmers University of Technology, Gothenburg, SWEDEN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms as is or as a source
 * for derivative work are permitted provided that the above copyright 
 * notice and this paragraph are  duplicated in all such forms 
 * and that any documentation, advertising materials,
 * and other materials related to such distribution
 * and use acknowledge that the software was developed
 * at Chalmers University of Technology, SWEDEN.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Adaptec AHA-154XC SCSI adapter driver 
 * 
 * $Source: /usr/src/sys/rt/dev/RCS/aha.c,v $ 
 * $Revision: 1.6 $ $Date: 1994/08/12 23:51:35 $ 
 * $Author: md $ $State: Exp $
 *
 * $Log: aha.c,v $
 * Revision 1.6  1994/08/12  23:51:35  md
 * Add support for 1542C model controllers.
 *
 * Revision 1.5  1994/05/22  12:33:57  roger
 * header file changes.
 *
 * Revision 1.4  1994/05/07  14:23:32  md
 * Change #include statements to use system search path instead of absolute.
 *
 * Revision 1.3  1994/03/17  15:58:02  md
 * Make the panic message due to and invalid scsi command a little more
 * informative.
 *
 * Revision 1.2  1992/12/09  10:20:42  md
 * Add back in debugging hooks.
 *
 * Revision 1.1  1992/11/05  20:36:40  md
 * Initial revision
 *
 * Revision 3.1  90/03/05  15:08:52  bernerus
 * The debug flags ahanotice, ahausage and ahatoutdebug were shut off
 * in the distributed version since they tend to fill up log files.
 * The ahalinkflag variable were removed.
 * Bug fix: Clearing of interrupt port moved to after MBI's are scanned
 * since any calls to printf() otherwise may cause ahascanmbi to be reentered.
 * More debug printouts enabled with ahaerrdebug.
 * 
 * Revision 3.0  90/02/08  09:45:01  bernerus
 * February 1990 release.
 * 
 * Revision 2.10  90/02/06  16:46:15  bernerus
 * The PARANOID checks have been diversified, if PARANOID is defined, all
 * of the checks are enabled.
 * The dma code used really needed to be enclosed in spl_dma() sections
 * to work correctly. This wasn't documented in the manual but was found after
 * careful study of dma.c. The lack of spl_dma caused a slow deterioration which
 * in the end made the file server hang occasionally, since the TCW's sometimes
 * weren't returned correctly. This might also explain some file system incon-
 * sistencies which seems to have disappeared now.
 * Ahascanmbi now keeps track on where next mbi will probably be found. 
 * Ahaustart and its descendants may now be run with interrupts off by defining
 * SPL_MUCH. This is done if PARANOID is defined.
 * The statistics collected may be shut off by #undef'ing AHASTATISTICS.
 * Ahacsbtoddcode no longer needs the ctlr parameter.
 * Usage of command link id's may be shut off by #undef'ing AHACLID.
 * All ccb's now also saves a pointer to its corresponding MBO which speeds up
 * the freeing later.
 * The routines ahainc... has been renamed ahanext...
 * The DEBUG stmts in ahatout() can be turned on using ahatoutdebug.
 * The practice of calling ahaustart instead of queueing devices and then
 * starting all devices didn't work since the TCW's are shared between all 
 * devices on the channel. The ahaustart calling code has therefore 
 * been removed.
 * 
 * Revision 2.9  90/01/11  09:19:42  bernerus
 * The BUFMODE possibility is withdrawn, didn't work.
 * Ahaustart: Counting the buffer Q is now done in the for loop header.
 *            The checking of mbop before ahazapmbo is now done within ahazapmbo.
 *            Testing: The whole body of ahaustart is now running with interrupts
 *                     disabled (splbio).
 *            xhaiodone is only written once in the XHU_NEXT part.
 *            The variable prevccbp is renamed ccbp.
 *            The mbo is now picked outside the for loop.
 *            The XHU_DODATA seciton now first set up a ccb, then explicitly
 *            starts the mbo.
 * Ahaxsetup: No longer takes the "action" parameter. Starting of chains should be
 *            done explicitly.
 * Ahascanmbi:Mbi rescanning is now considered to be PARANOID.
 *            Some of the checks of the xs pointers etc are also
 *            considered PARANOID.
 *            Ahasccom no longer supports the sending of Bus device reset. This
 *            is now done by the new function ahabdr().
 *            The "ahanodma" test has been withdrawn.
 *            A ccb without a data pointer is no longer considered an error.
 *            Another test: the tail of ahascanmbi no longer queue devices but
 *            instead call ahaustart directly for the unit in question. This
 *            means that a fast disk may starve other disks, but may increase
 *            throughput.
 * Ahadoscsi: The routine now first set up a ccb, then explicitly starts the mbo.
 * Ahasccom:  Is now renamed ahabdr since its only function is to send bus device
 *            reset.
 * Ahastartmbo: Is now a (void) function, since it will always succeed.
 *              It no longer tries to pick an mbo but expects to get one via
 *              the mbop parameter.
 * Ahagetccb: The check of admp->freeccb is now considered PARANOID.
 * Ahagetmbo: The check of admp->freeccb is now considered PARANOID.
 * Ahazapmbo: Now checks mbop, if 0, nothing is done.
 * 
 * Revision 2.8  89/11/27  09:04:16  bernerus
 * Now more ANSI-fied, function prototypes.
 * A lot of fixes from Hakan Winbom entered.
 * Bug fix:  Interrupts are now shut off in ahaustart until the buffer is
 * removed from the queue, else the queue can be linked into the free list.
 * 
 * Revision 2.7  89/11/15  14:49:13  bernerus
 * Removed stuff earlier disabled by #ifdef NOXHA, the stuff is now in xha.c.
 * Also reboved some debugging uprint... functions.
 * 
 * Revision 2.6  89/11/13  09:16:25  bernerus
 * Ahatur() not used, deleted.
 * Ahaadm struct moved to ahareg.h.
 * Ahazapmbo clears a specified mbo and keeps track of the MBO statistics.
 * All functions that previously used the xha_device pointer (xi) for
 * pointing out a specific device now uses the xha_slave pointer (xs) instead.
 * The function ahacstart is renamed xhastartdevices and moved to xha.c.
 * The function ahatstart is renamed ahaustart.
 * New function ahaiodone is used wherever iodone was used.
 * Ahaxsetup now takes an optional pointer to a preloaded SCSI CDB which
 * then will be used instead of asking the device driver's start() routine.
 * Ahasccom now only takes 4 arguments instead of previously 8.
 * The function ahascanmbi now doesn't use the host adapter
 *  status for determining whether the completed command returned
 *  intermediate status, since this seemed to be broken in
 *  the AHA1542.
 * The function ahadoscsi now takes 5 args instead of 7. The
 *  deleted ones are extracted from the xha_slave structure. The
 *  function now sets B_HEAD in the special buffer in order to make
 *  it possible to disable any requeueing of the special buffer. A
 *  once only retry function has been incorporated in ahadoscsi to
 * be able to cope with the UNIT ATTENTION problem.
 * The function ahasccom now takes 4 arguments instead of 7. The
 *  deleted ones are extracted from the xha_slave struct.
 * The function ahasetupccb now takes 4 arguments instead of 7.
 *  The deleted ones are extracted from the xha_slave structure.
 * The function ahafillccb now takes 6 arguments instead of 8.
 *  The deleted ones are extracted from the xha_slave structure.
 * The uprint* functions in the debug section have been deleted.
 * The function ahafreembo now uses ahazapmbo instead of doing
 *  the job by itself.
 * SCSI CDB's are now uniformly pointed to by using a
 *  (union scsi_cdb *) type.
 * The fuctions ahacstart, ahaqdevice, ahastartdevices,
 * aharequeue, ahadgo, xcstr, xhdump and the dspl functions
 *   has been moved to xha.c and been renamed xha....  .
 * 
 * Revision 2.5  89/08/22  08:27:53  bernerus
 * Added spl trace on the display and a spl trace history area.
 * Added loop count checks on all loops.
 * Ahadmaerrdebug is now on.
 * Added NO-DMA test capability for the FBQ kernel.
 * 
 * Revision 2.4  89/05/23  12:55:27  bernerus
 * Mostly a cleanup of the code. The copyright notice has changed.
 * A bug fix has been installed that fixed a bug in the device queuing
 * routines.
 * 
 * Revision 2.3  89/02/08  13:01:03  bernerus
 * Ahatur now does real things, operates as a device probe routine.
 * Ahacstart is now split into two routines: ahaqdevice and ahastartdevices
 * A new ahacstart is written that calls the two routines in sequence. However
 * the ahacstart call at the end of ahascanmbi was split into two parts so
 * that ahastartdevices is only called once per interrupt.
 * 
 * Revision 2.2  89/02/06  08:13:57  bernerus
 * Hords of changes, mostly due to restructuring of the source.
 * Copyright notice has changed a bit.
 * A new function ahadoscsi() permits scsi calls during autoconfig.
 * Adapter initialization on attach nor tries 10 times before giving up.
 * Ahatur() is now a dummy function, probing should be done by the device's
 * probe or attach routine by using SCSI inquiry thru the ahadoscsi() function.
 * More error messages now have adapter tag: "aha0: This message should never occur"
 * 
 * Revision 2.1  89/01/30  07:59:34  bernerus
 * Performed name changes of some variables.
 * 
 * Revision 2.0  89/01/16  10:50:40  bernerus
 * Changed Copyright notice
 * Changed names cu to cs, cs to csc, xu to xt, xt to xtc.
 * Reformatted function predeclarations.
 * Moved some variables into the ahaadm structure.
 * Removed display() calls and last CDF check in ahakick().
 * Renames ahaxstart() to ahaxsetup(), xi_xhanum to xi_ctlr, 
 * ahaclearmbo() to ahafreembo().
 * Ahatstart now reserves an mbo b4 building link chain.
 * Shut off MBO interrupt.
 * 
 */

/*********************** STATIC DATA SECTION **********************************/
#define PARANOID
#ifdef PARANOID
#define PARTIALFILLCHECK
#undef NOMBIHOLES
#define CCBCHECK
#define CCBPCHECK
#define XSCHECK
#define SPL_MUCH
#else
#define NOMBIHOLES
#endif /* PARANOID */

#include <sys/param.h>
#include <sys/dkstat.h>
#include <sys/vm.h>
#include <sys/buf.h>
#include <sys/types.h>

#include "rt/include/ioccvar.h"
#include "rt/include/scsi.h"
#include "rt/include/xhavar.h"
#include "rt/dev/ahareg.h"
#include "rt/dev/dmareg.h"
#include "rt/dev/dmavar.h"
#include "rt/rt/debug.h"

int aha_bus_on  =       AHABUS_ON ; /* HW for debugger use */
int aha_bus_off =       AHABUS_OFF ;
int aha_speed   =       AHASPEED8 ;

#ifdef DEBUG
extern int      dmadebug;
int             ahadebug = 	 0;
int		ahatrace =	 0;
int             ahanotice = 	 0;
int             ahacmddebug = 	 0;
int             ahakickdebug = 	 0;
int             ahaintdebug = 	 0;
int             ahascandebug = 	 0;
int             ahainitdebug = 	 0;
int             ahacstartdebug = 0;
int             ahatstartdebug = 0;
int             ahaxsetupdebug = 0;
int		aharetrydebug =  0;
int             ahaerrdebug = 	 0;
int             ahadmaerrdebug = 0;
int             ahausage = 	 0;
int             ahasleepdebug =  0;
int		ahambdebug = 	 0;
int		ahadmadebug =	 0;
int		aharequeuedebug =0;
int		ahadoscsidebug = 0;
int		ahafillccbdebug =0;
int		ahatoutdebug =   0;
#include	"rt/dev/xxc.h"
#endif /* DEBUG */

/* #define DSPL		/* show status on LED's */
#ifndef DSPL
#define dsplbio(x) splbio()
#define dsplx(x,y) splx(x)
#define dsplhigh(x) splhigh()
#endif /* DSPL */

struct iocc_ctlr *ahainfo[NAHA];

#ifdef FBQ
int		ahatocheck = 0;		/* Disable timeout code */
int		ahanodma   = 0;		/* Testing, disable dma mapping */
#else
int		ahatocheck = 0;
#endif /* FBQ */
/* Standard I/O addresses for the controller */
caddr_t         ahastd[] = 
{
    (caddr_t) 0xf0000330,
    (caddr_t) 0xf0000334,
    (caddr_t) 0xf0000230,
    (caddr_t) 0xf0000234,
    (caddr_t) 0xf0000130,
    (caddr_t) 0xf0000134,
    (caddr_t) 0
};


struct xhac_driver ahacdriver = 
{
     ahadoscsi,  	/* Send specific SCSI command */
     xhacstart,		/* channel start routine */
     ahaustart,		/* unit start routine */
};

struct iocc_driver ahadriver = 
{
    			    ahaprobe,	/* probe     */
    			    0,		/* slave     */
    (int (*) ())	    ahaattach,	/* attach    */
    (int (*) ())	    xhadgo,	/* dgo       */
    			    ahastd,	/* addr      */
    (char *) 		    0,	    	/* dname     */
    (struct iocc_device **) 0,		/* dinfo     */
    			    "aha",	/* mname     */
    			    ahainfo,	/* minfo     */
    			    ahaint,	/* intr      */
    			    0x0,	/* csr       */
    			    0,		/* chanrelse */
    			    0		/* flags     */
};
#ifdef HZ
#undef HZ	/* XXX */
#endif
#define HZ 100
#define TOTIME 4*HZ

#ifdef DEBUG
int             aha_nmbox = NMBOX;	/* Info point for debugger */
int             aha_nccb = NCCB;	/* Info point for debugger */
#endif /* DEBUG */

struct ahaadm ahaadm[NAHA];

/*
/*************** CONFIGURATION FUNCTIONS SECTION *********************/
ahaprobe( register caddr_t reg, register struct iocc_ctlr *ic)
{
    register struct aha_regs *addr = (struct aha_regs *) reg;
    extern int      int_level;

    /* First, issue a reset command that also clears the interrupt port */

    addr->aha_csr = AHCSR_HRST | AHCSR_SCRST | AHCSR_IRST | AHCSR_SRST;

    DELAY(10000);

    addr->aha_csr = AHCSR_IRST ;

    /*
     * Now, the command port should be empty. So we can issue a NOP command
     * that will give us an interrupt 
     */

    ahacmd(addr, AHACMD_NOP,ic->ic_ctlr, 1);
    PROBE_DELAY(10000);
    if (int_level < 0)
    {
	printf("Ahaprobe: no interrupt from aha at %x! \n", addr);
	return PROBE_BAD;
    }


    if(aharesetha(addr))
    {
	return PROBE_BAD;
    }
    return PROBE_OK;
}

void
ahaattach( register struct iocc_ctlr *ic)
{
    int             		i,s;
    struct buf     	       *bp;
    int             		dmasetuprv;
    ioaddr_t        		ioaddr;
    register int		ctlr = ic->ic_ctlr;
    struct ahaadm	       *admp = &ahaadm[ic->ic_ctlr];
    int				initcnt = 0;

    /* 
     * Clear adm area, this will as a side effect mark all mailboxes as free 
     * The CCB's though, should be set free with special code.
     */
    bzero(admp, sizeof ahaadm[0]);

    while(ahainit(ic))
    {
	if(aharesetha((struct aha_regs *) ic->ic_addr))
	{
		ic->ic_alive = 0;
		printf("aha%d: attach reset failed\n",ic->ic_ctlr);
		return;
	}
        if(++initcnt>10)
	{
		ic->ic_alive = 0;
		printf("aha%d: attach init failed\n",ic->ic_ctlr);
		return;
	}
    }


    /* Init next-free pointers */
    admp->freembi = admp->mbi;
    admp->freembo = admp->mbo;
    admp->freeccb = admp->ccb;
    admp->nokick = 1;  /* Don't start anything yet */
    for (i = 0; i < NMBOX; i++)
	admp->mbi[i].aha_ccbp = 1; /* An illegal value */
    for (i = 0; i < NCCB; i++)
	admp->ccb[i].aha_op = AHAOP_CCB_FREE;

    /* Configure DMA stuff */
    ic->ic_dmaflags = 
	(DMA_CASCADE | DMA_PAGE | DMA_PHYSICAL | DMA_EXCLUSIVE | DMA_CANTWAIT);
    ic->ic_dmabuf = 0;

    /* 
     * Allocate and set up an io buffer for mailbox usage 
     */
    bp = &(admp->iobuf);
    bp->b_flags |= B_LOCKED;			    /* We keep it forever     */
    bp->b_dev = (dev_t) NODEV;			    /* No device associated   */
    bp->b_error = 0;				    /* No errors yet 	      */
    bp->b_un.b_addr=(caddr_t) admp; 		    /* Data space address     */
    bp->b_bcount = sizeof(ahaadm[ctlr]);     	    /* Byte count	      */
    DEBUGF(ahadmadebug,dmadebug=1;);
    if ((dmasetuprv = dma_setup(ic)) != DMA_OK_RET)
    {
	DEBUGF(ahadmadebug,dmadebug=0;);
	printf("Ahaattach: DMA setup failed, rv=%d\n", dmasetuprv);
	return;
    }

    s = spl_dma();
    ioaddr = (ioaddr_t) dma_map(ic->ic_dmachannel, bp);
    splx(s);
    if (ioaddr == (ioaddr_t) DMA_INV_IOADDR)
    {
	DEBUGF(ahadmadebug,dmadebug=0;);
	printf("Ahaattach: dma_map failed\n");
	return;
    }
    printf("aha%d: adm area at 0x%x (%x) %d bytes\n", ic->ic_ctlr, admp,
	ioaddr, bp->b_bcount);
    admp->ioaddr = ioaddr;
    dma_go(ic->ic_dmachannel);
    DEBUGF(ahadmadebug,dmadebug=0;);
    ahainitmbox(NMBOX, (struct ahambox *) admp->mbo, ctlr); /* Init Mailboxes */

    /* DELAY(100);  HW ? */
    admp->nokick = 0;		/* Allow kicks */
}

/*
/* 
 * Unit start routine, this will attempt to start all the available requests
 * for the unit indicated by xs
 */
ahaustart(register struct xha_slave *xs)
{
    register struct buf	*bp;
    register int	 s;
    register int	 bufq;
    struct ahaccb	*ccbp = 0;
    struct ahaccb	*firstccbp = 0;
    struct ahambox	*mbop = 0;

    DEBUGF(ahatrace,printf("aha%d: Ahaustart: xs=%x\n",xs->xs_ctlr,xs););
    /* 
     * We first check if there is anything to do or if someone else is
     * already working on it, if so we have nothing to do
     */
    s = dsplbio(9);
    if((bp = xs->xs_tab.b_actf) == NULL) /* Anything in Q ? */
    {
	dsplx(s,10);			 /* NO, bye */
	return AHS_DONE;
    } 
    if(xs->xs_tab.b_active )		/* Somebody already doing this ? */
    {					
	dsplx(s,11);			/* Yes, bye */
	DEBUGF(ahanotice,printf("%s%d:Ahaustart: already active\n",
			xs->xs_driver->xd_dname,xs->xs_unit););
	return AHS_RETRY;
    }
    /* Yes, there is something to do */
    xs->xs_tab.b_active = 1;    /* Note to others that I'm working on it */
#ifndef SPL_MUCH
    dsplx(s,12);
#endif

    DEBUGF(ahatstartdebug,printf("%s%d:Ahaustart: starting unit \n",
			xs->xs_driver->xd_dname,xs->xs_unit););

    mbop = ahagetmbo(xs->xs_ctlr);

    for(bufq=1;;bufq++)
    {
	switch ((*xs->xs_driver->xd_ustart) (xs))
	{
	case XHU_NEXT:
#ifndef SPL_MUCH
	    s = dsplbio(13); 
#endif
	    if((xs->xs_tab.b_actf = bp->av_forw) != 0) /* Remove it from Q */
	    {
		xhaiodone(bp,"1");
		bp = xs->xs_tab.b_actf;
#ifndef SPL_MUCH
		dsplx(s,14); 
#endif
		continue;
	     }
	     xhaiodone(bp,"1");
	     xs->xs_tab.b_active = 0;
	     ahazapmbo(xs->xs_ctlr,mbop);
	     dsplx(s,15);
	     return XHS_DONE;
	case XHU_DODATA:
	    DEBUGF(ahatstartdebug,printf("Ahaustart: starting xfer %x\n",bp););
	    if((mbop == 0) || (ccbp=ahaxsetup(xs,bp,0,0)) == 0)
	    {
	        ahazapmbo(xs->xs_ctlr,mbop);
		xs->xs_tab.b_active = 0;
#ifdef SPL_MUCH
		dsplx(s,19);
#endif
		return AHS_RETRY;
            }
	    /* Flush the request from the queue, if it fails and should be
	     * retried, it will be requeued by the interrupt routine */
#ifndef SPL_MUCH
	    s = dsplbio(49); 
#endif
	    ahastartmbo(xs->xs_ctlr,ccbp,mbop);
	    xs->xs_tab.b_actf = bp->av_forw;
	    dsplx(s,50);
	    return XHS_DONE;
	case XHU_DOLINKS:
	    DEBUGF(ahatstartdebug,printf("Ahaustart: starting link %x\n",bp););
	    /*
	     * First, see if we have an MBO reserved so that we're sure to 
	     * have one for the chain since backing off a chain too difficult
	     */
	    if(mbop == 0)
	    {
		xs->xs_tab.b_active = 0;
#ifdef SPL_MUCH
		dsplx(s,20);
#endif
		return AHS_RETRY;
	    }
	    /* 
             * Here we set up CCB's as long as possible
	     */
	    if((ccbp = ahaxsetup(xs,bp,ccbp,0)) == 0)
	    {
		/* 
		 * We get here if ahaxsetup fails. If we were unable to
		 * set up anything, we'll have to defer all actions to a
		 * later time
		 */
		if(firstccbp == 0) 
		{
		    DEBUGF(aharetrydebug,ahaintdebug=1;);
		    /*
		     * If we've reserved an MBO, we must free it here
		     */
		    ahazapmbo(xs->xs_ctlr,mbop);
		    xs->xs_tab.b_active = 0;
#ifdef SPL_MUCH
		    dsplx(s,21);
#endif
		    return AHS_RETRY;
		}
		/* 
		 * Else at least one CCB were set up.
		 * Start the chain built so far, return that the Q isn't empty 
		 */
		ahastartmbo(xs->xs_ctlr,firstccbp,mbop);
#ifdef SPL_MUCH
		dsplx(s,22);
#endif
		return AHS_RETRY;
            }
	    else /* I.e. No problems setting up CCB */
	    {
		if(firstccbp == 0) 
		{
			firstccbp = ccbp;
		}
		/* 
		 * Shut off interrupts here so that a request can't be entered
		 * without getting processed, only requeued requests could
		 * be entered while running here as a process, these requests
		 * are always entered at the end.
		 */
		DEBUGF(ahatstartdebug,printf("Ahaustart: xfer %x DONE\n",bp););
		/* Get new bp, unqueue old one and see if we hit the end */
#ifndef SPL_MUCH
		s = dsplbio(16); 
#endif
		if((bp = xs->xs_tab.b_actf = bp->av_forw) != 0)
		{
#ifndef SPL_MUCH
		    dsplx(s,17); 
#endif
#ifdef AHASTATISTICS
		    if(bufq > ahaadm[xs->xs_ctlr].maxbufq)
		    {
			DEBUGF(ahanotice,printf("Ahaustart: Max Link length = %d\n",bufq););
			ahaadm[xs->xs_ctlr].maxbufq = bufq;
		    }
#endif /* AHASTATISTICS */
		    continue;
		}
		ahastartmbo(xs->xs_ctlr,firstccbp,mbop);
		dsplx(s,18);
		return XHS_DONE;
	    }
	}
    }
}
/*	
/* This routine sets up ccb's for a xfer.
 * If prevccbp is nonzero, a link will be set up to prevccbp.
 * return value is ccb set up or 0 if xfer couldn't be started */
struct ahaccb *
ahaxsetup(  register struct xha_slave      *xs,
	    register struct buf *bp,
	    register struct ahaccb *prevccbp,
	    register union scsi_cdb   *com)
{
    register struct iocc_ctlr   *ic = xs->xs_xi->xi_hd->xh_iocc;
    register ioaddr_t            ioaddr=0;
    union scsi_cdb		 combuf;
    register struct ahaccb 	*ccbp;
    int				s;

    DEBUGF(ahaxsetupdebug | ahatrace,printf("aha%d: Ahaxsetup: ENTRY %s%d, bp=%x, combuf=%x,prio=%d\n",xs->xs_ctlr,
	xs->xs_driver->xd_dname,xs->xs_unit,bp,&combuf,getspl()););

    /* If no SCSI command were given to us, we ask the device driver's "start"
     * routine for a command. If we were given a command, it's probably a 
     * special command entered via ahadoscsi()
     */
    if(com == 0)
    {
	bzero(&combuf,sizeof combuf);	/* Must be cleared first */

	if (xs->xs_driver->xd_start == 0 ||
	    (com = (*xs->xs_driver->xd_start) (xs,bp,&combuf)) == 0)
	{
	    printf("%s%d: SCSI command pointer zero or no driver start routine\n",
		    xs->xs_driver->xd_dname, XHAUNIT(bp->b_dev));
	    return 0;
	}
    }
    /*
     * This is the place where things really happen
     * First we set up the DMA mapping, next we get a CCB and fill it in.
     * Then we get a free MBO, fills it with the address of the CCB, and
     * lastly we kick the host adapter telling it that there is work to do.
     */
    DEBUGF(ahaxsetupdebug,printf("Ahaxsetup: %s%d, bp=%x, com=%x,prio=%d\n",
	xs->xs_driver->xd_dname,XHAUNIT(bp->b_dev),bp,com,getspl()););
    
    if(bp->b_bcount > 0)
    {
        /* Set up DMA mapping */
        s = spl_dma();
	ioaddr = (ioaddr_t) dma_map(ic->ic_dmachannel, bp); 
        splx(s);
	if (ioaddr == (ioaddr_t) DMA_INV_IOADDR)
	{
	    DEBUGF(ahadmaerrdebug,printf("%s%d: Ahaxsetup: dma_map failed\n",
		xs->xs_driver->xd_dname, XHAUNIT(bp->b_dev)););
            return 0;
	}
    }

    ccbp = ahasetupccb(xs,com,ioaddr,bp);
    if(ccbp && (prevccbp != 0) )
	ahalinkccb(xs->xs_ctlr,prevccbp,ccbp);

    if((ccbp == 0) && (ioaddr != 0))
    {
        s = spl_dma();
	dma_free_map(ic->ic_dmachannel, ioaddr,bp->b_bcount);
        splx(s);
    }
    return ccbp;
}

/*
/*************** INTERRUPT FUNCTIONS SECTION *********************/
/*
 * Interrupt routine, most of it's work is done in
 * the ahascanmbi() function which takes care of all MBI's
 */
ahaint(int ctlr)
{
    register struct iocc_ctlr *ic = ahainfo[ctlr];
    u_char          iflgs,
                    csr;
    struct ahaadm  *admp = &ahaadm[ctlr];
    register struct aha_regs *addr = (struct aha_regs *) ic->ic_addr;

    csr   = addr->aha_csr;			/* Read status register           */
    iflgs = addr->aha_iflgs;			/* Read the interrupt port  	  */
    if(!(iflgs & AHIFLGS_INT))			/* If no interrupt, skip the rest */
    {
	DEBUGF(ahanotice,printf("Ahaint, stray interrupt ... iflgs=%b\n",
		iflgs,AHIFLGS_BITS););
	return 1;
    }

    admp->move = 1;		/* Note to ahatout that int occurred */

    if ((csr & AHCSR_INIT) && (csr & AHCSR_INVDCMD)) /* Need mailbox init */
    {
	ahainitmbox(NMBOX, (struct ahambox *) admp->mbo,ctlr);
	DEBUGF(ahanotice,printf("Ahaint: reinitialized mboxes\n"););
	DELAY(1000);
    }
    if (iflgs & AHIFLGS_HACC)		/* If command complete, set indicator */
    {
	admp->cmdcmpl = CMPL_CC;
    }

    DEBUGF(ahaintdebug, printf("Ahaint: csr=%b, iflgs=%b, prio=%d\n",
		                csr, AHCSR_BITS, iflgs, AHIFLGS_BITS,getspl()););

    if ((iflgs & AHIFLGS_MBIF) || (iflgs & AHIFLGS_SCRD))    /* In mailbox filled */
    {
#ifdef AHASTATISTICS
	admp->mbiusage=
#endif /* AHASTATISTICS */
	  ahascanmbi(ctlr,iflgs & AHIFLGS_SCRD);
#ifdef AHASTATISTICS
	if(admp->mbiusage > admp->maxmbiusage)
	{
	    admp->maxmbiusage = admp->mbiusage;
	    DEBUGF(ahausage, printf("Ahaint: Max MBI usage = %d\n", admp->maxmbiusage););
	}
#endif /* AHASTATISTICS */
    }

    addr->aha_csr = AHCSR_IRST;	 	/* Clear the interrupt port */

    if(!(csr & AHCSR_INIT) && (!admp->nokick))
    {
	ahakick(ctlr);
    }
    return 0;
}

/*
 * Scan all MBI's and take appropriate actions
 */
ahascanmbi(int ctlr,int scsireset)
{
    /* The variables below are  ordered in # of occurrances in the routine    */
    register struct xha_slave  *xs;		    /* Ptr to unit 	      */
    register struct ahaccb     *ccbp;		    /* Ptr to CCB involved    */
    register struct buf        *bp;		    /* Ptr to buffer involved */
    register struct ahambox    *mp;		    /* Ptr to mbox in question*/
    register u_long             ioccbp;		    /* CCB part of MBI	      */
    register int    		dccode;    /* Code sent to driver int routine */
    register u_char 		csb;		    /* Status byte of MBI     */
    	     struct iocc_ctlr   *ic = ahainfo[ctlr];
    	     struct scsi_esense	*sdp;		    /* Ptr to sense info      */
    	     u_char 		tarstat;	    /* Target status          */
    	     u_char 		hastat;		    /* Host adapter status    */
#if defined(AHASTATISTICS) || defined(NOMBIHOLES)
    	     int    		nmbi = 0;	    /* Count used mbi's	      */
#endif /* AHASTATISTICS || NOMBIHOLES */
	     struct ahaadm     *admp = &ahaadm[ctlr];
	     struct xha_hd     *xhp = 0;	    /* Ptr to xha_hd struct   */
	     unsigned		dptr;		    /* Data pointer in ccb    */
	     unsigned		lptr;		    /* Link ptr in ccb        */
	     struct ahambox	*mpinit;	    /* Initial mbox ptr */
#ifdef PARTIALFILLCHECK
	     int		mbirescan = 1;	    /* Times to go thru MBI's */
#endif /* PARTIALFILLCHECK */
	     int		wrap;	/* 1 if we've wrapped around the edge */
	     int		s;

    /* Scan the IN mailboxes and see what should be done to them */

    DEBUGF(ahascandebug | ahatrace,
	 printf("Ahascanmbi: Scanning in mailboxes  ctlr=%d\n", ctlr););
    mpinit = admp->freembi;

#ifdef PARANOID
    if((mpinit < admp->mbi) || (mpinit >= &admp->mbi[NMBOX]))
    {
	printf("ahascanmbimbo: mpinit out of range: %x set to %x\n",
		mpinit,admp->mbi);
	mpinit = admp->mbi;
    }
#endif /* PARANOID */

#ifdef PARTIALFILLCHECK
    while(mbirescan--)
    {
#endif /* PARTIALFILLCHECK */
	for(wrap=0,mp=mpinit;(wrap==0)||(mp != mpinit);
	    mp=ahanextmbip(admp,mp,&wrap))
	{
		/* Don't change the order of the following three statements */
	    if((csb = mp->aha_csb) == MBI_MAILBOX_FREE )
	    {
#ifdef AHASTATISTICS
		if((mp==mpinit)&&(wrap==0))
		{
		    admp->mbimisses++;
		}
#endif /*AHASTATISTICS*/
#ifdef NOMBIHOLES
		if(nmbi > 0)
		{
		    admp->freembi = mp;
		    break;
		}
#endif /*NOMBIHOLES*/
		continue;
	    }
#ifdef AHASTATISTICS
		if((mp==mpinit)&&(wrap==0))
		{
		    admp->mbihits++;
		}
#endif /*AHASTATISTICS*/

	    ioccbp = mp->aha_ccbp;
#ifdef PARTIALFILLCHECK
	    if(ioccbp & 0x01)	/* Mailbox being filled ?	   */
	    {
		    printf("Ahascanmbi: partially filled MBI, enqueued\n");
		    mbirescan = 1;
		    continue;
	    }
#endif /* PARTIALFILLCHECK */
#if defined(AHASTATISTICS) || defined(NOMBIHOLES)
	    nmbi++;
#endif /* AHASTATISTICS || NOMBIHOLES */

#ifndef NOMBIHOLES
	    /* It's difficult to set a correct freembi if we're going thru
	       all mbi's so instead we set it to the next mbi every time
	       we pick a filled one */
	    admp->freembi = ahanextmbip(admp,mp,0);
#endif /* ! NOMBIHOLES */
	   

#ifdef CCBPCHECK
	    /*
	     * Validate the ccb pointers
	     */
	    if (ioccbp < 0xfe0000)
	    {
		printf("Ahascanmbi: Strange! ccb pointer = %x\n", ioccbp);
		DELAY(10000000);
		ahafreembi(mp);
		continue;
	    }
#endif /* CCBPCHECK */

	    /* 
	     *  Translate the ioccb pointer to a virtual address we can use.
	     */
	    ccbp = (struct ahaccb *) ahvaddr(ioccbp, ctlr);

#ifdef CCBPCHECK
	    if((ccbp < admp->ccb) || (ccbp >= &admp->ccb[NCCB]))
	    {
		    printf("Invalid CCB pointer = %x ioccbp=%x\n",ccbp, ioccbp);
		    DELAY(10000000);
		    ahafreembi(mp);
		    continue;
	    }
	    if((((int)ccbp - (int)admp->ccb) % 
		((int)&admp->ccb[1] - (int) &admp->ccb[0])) != 0)
	    {
		    printf("CCB pointer points between ccb's, ccbp = %x, ioccbp=%x\n",
			   ccbp, ioccbp);
		    DELAY(10000000);
		    ahafreembi(mp);
		    continue;
	    }
#endif /* CCBPCHECK */
	    /*
	     * Zero out corresponding MBO so that sane things could be
	     * done if the adapter hangs up, I.E. we here mark that the
	     * corresponding MBO isn't active anymore within the host adapter.
	     */
	    ahafreembo(ctlr, ccbp);
	    /* Most of the MBI's scanned are OK, so we make a special test here to avoid most of the calls to ahacsbtodccode. 
             */
	    if(csb == MBI_NOERROR)
	    {
		dccode = XHI_CMPL;
	    }
	    else
	    {
		/*
		 * Let's have a closer look at the status byte
		 */
		if((dccode = ahacsbtodccode(csb,mp)) == XHI_NEXT)
		{
		    continue;
		}
#ifdef DEBUG
		if (csb != MBI_NOERROR)
		{
		    DEBUGF(ahaerrdebug, printf("Ahascanmbi: MBI ERROR csb=%d\n",csb););
		    DEBUGF(ahaerrdebug, printmbox(mp,1,1););
		    DEBUGF(ahaerrdebug, printccb(ccbp););
		}
#endif /* DEBUG */
	    }
	    DEBUGF(ahascandebug,printf("Ahascanmbi: ccbp=%x\n",ccbp););
	    DEBUGF(ahascandebug,printccb(ccbp););
	    sdp  = (struct scsi_esense *)		/* Sense info 	      */
		(((char *) &ccbp->aha_cdb) + ccbp->aha_cmdlen);
	    tarstat = ccbp->aha_tarstat;	        /* Target status      */
	    hastat  = ccbp->aha_hastat;			/* Host status	      */
	    xs      = ccbp->aha_xs;			/* Device pointer     */
	    bp      = ccbp->aha_bp;			/* Buffer pointer     */
	    dptr    = xhaget3(ccbp->aha_dptr);		/* Data pointer	      */
	    lptr    = xhaget3(ccbp->aha_lptr);		/* Link pointer       */

#ifdef XSCHECK
	    /* 
	     * The if stmt below is a guard against zero pointers
	     */
	    if (xs && 
		xs->xs_driver &&
	        xs->xs_driver->xd_int)
	    {
#endif	/* XSCHECK */


		/*
		 * Check Host status and scsireset to set dccode accordingly
		 */
		dccode = ahastattodccode(dccode, hastat, scsireset);

		if(bp)
		{
		    /* Let the target driver's int. routine decide what to do */
		    switch ((*xs->xs_driver->xd_int) (xs, bp, dccode, sdp, tarstat))
		    {
		    case XHD_RESETRETRY:
			DEBUGF(ahaerrdebug || ahanotice, 
			    printf("Ahascanmbi: int routine returned RESETRETRY\n"););
			ahabdr(xs); /* Send Bus dev reset*/
			xharequeue(xs,bp);
			break;
		    case XHD_SCRESETRETRY:
			DEBUGF(ahaerrdebug || ahanotice,
			    printf("Ahascanmbi: int routine returned SCRESETRETRY\n"););
			ahascreset((struct aha_regs *) xs->xs_xi->xi_xha);
			xharequeue(xs,bp);
			break;
		    case XHD_DONE:
			DEBUGF(ahascandebug ||
			       (ahaerrdebug && (csb != MBI_NOERROR)), 
				printf("Ahascanmbi: int routine returned DONE\n"););

			xs->xs_tab.b_errcnt = 0;

			/* IN CASE THE BUFFER IS ALREADY DONE,
			 * THERE ARE PROBABLY NO TCW'S MAPPED SO WE 
			 * DISABLE THE FREEING BELOW BY CLEARING 
			 * THE dptr VARIABLE 
			 */
			 
			if(xhaiodone(bp,"3"))
			    dptr=0;
			break;

		    case XHD_RETRY:
			DEBUGF(ahaerrdebug, 
			    printf("Ahascanmbi: int routine returned RETRY\n"););
			xharequeue(xs,bp);
			break;
		    default:
			panic("ahaint: device int. routine return code error");
		    } /* End switch(xs-int()) */
		} /* End if(bp) */
		else
		{
			DEBUGF(ahascandebug,
				printf("Ahascanmbi: ccbp=%x, bp==NULL\n"););
	        }
#ifdef XSCHECK
	    } /* End if(xs ... ) */
	    else
	    {
		DEBUGF(ahascandebug,printf("Ahascanmbi: ccbp=%x, xs==NULL\n",ccbp););
	    }
#endif /*XSCHECK*/

	    /*
	     * Free the DMA maps
	     */
	    if (dptr)
	    {
		s = spl_dma();
		dma_free_map(ic->ic_dmachannel, dptr, ccbp->aha_dlen);
		splx(s);
	    }

	    ahafreeccb(ctlr, ccbp); /* Release the ccb */
	    ahafreembi(mp);	    /* Free the MBI    */
	    /* 
	     * If not an intermediate interrupt, we 
	     * mark the device inactive and call the channel start routine 
	     */

	    if (
#ifdef XSCHECK
	        xs && 
		xs->xs_driver && 
		xs->xs_alive && 
#endif /* XSCHECK */
		tarstat != SCTS_INT_GOOD)
	    {
		if(lptr)
		{
		    DEBUGF(ahanotice,
			printf("Ahascanmbi: hastat=0x%x, tarstat=0x%x, lptr=0x%x\n",
			    hastat, tarstat, lptr););
		}
		else
		{
		    xs->xs_tab.b_active = 0; /* We ain't active anymore */

		    if(xs->xs_tab.b_actf)    /* Sth more in q ? */
		    {
			xhaqdevice(xs);	/* Q device for later start */
			xhp = xs->xs_xi->xi_hd;	/* Remember ptr to xha_hd */
		    }
		}
	    } /* End if(xs ... ) */
	} /* End for() ... */
#ifdef PARTIALFILLCHECK
    } /* End while(mbirescan) */
#endif /* PARTIALFILLCHECK */
    if(xhp)
    {
	xhastartdevices(xhp);		/* Start all queued devices */
    }
#ifdef AHASTATISTICS
    return nmbi;
#else
    return 0;
#endif /* AHASTATISTISCS */
}

ahacsbtodccode(int csb, struct ahambox *mp)
{
	DEBUGF(ahatrace,
		printf("Ahacsbtodccode: csb=%d, mp=%x\n",csb,mp););
	switch (csb)
	{
	case MBI_HOST_TIMEOUT:
	    printf("Ahaint:  CCB timeout aborted: %x\n", mp->aha_ccbp);
	    return XHI_TIMEOUT;
	case MBI_HOST_ABORTED:
	    printf("Ahaint:  CCB aborted: %x\n", mp->aha_ccbp);
	    return XHI_ERR;
	case MBI_NO_CCB:
	    printf("Ahaint:  CCB not found: %x\n", mp->aha_ccbp);
	    ahafreembi(mp);
	    return XHI_NEXT;
	case MBI_MAILBOX_FREE:
	    return XHI_NEXT;
	case MBI_NOERROR:
	    DEBUGF(ahascandebug, printf("Ahacsbtodccode: In mailbox:\n"););
	    DEBUGF(ahascandebug, printmbox(mp, 1, 1););
	    return XHI_CMPL;
	case MBI_ERROR:
	    DEBUGF(ahascandebug, printf("Ahacsbtodccode: In mailbox:\n"););
	    DEBUGF(ahascandebug, printmbox(mp, 1, 1););
            return XHI_ERR;
	default:

	    printf("Ahacsbtodccode: invalid mailbox-in status byte code %d at %x\n",
		   csb, mp);
	    return XHI_ERR;
	}
}

/*
/* Decode host adapter status to a device completion code */
ahastattodccode(int dccode, int hastat, int scsireset)
{
    DEBUGF(ahatrace,
	printf("Ahastattodccode: dccode=%d, hastat=%d\n",dccode,hastat););
    if(scsireset)
    {
	return XHI_SCSIRESET;
    }
    switch(hastat)
    {
    case AHAST_NOERROR:		/* No error */
    case AHAST_LKCPL:		/* Linked cmd cmpl, no error */
    case AHAST_LKCPL_INT:	/* Linked cmd cmpl, no error, intrpt */
	return dccode;
    case AHAST_SELTO:		/* Selection Timeout */
	return XHI_SELTO;
    case AHAST_OUVR:		/* Data over/under run */
	return XHI_OUVR;
    case AHAST_BUSFREE:		/* Unexpected bus free */
    case AHAST_TGT_FAIL:	/* Target bus phase sequence failure */
	return XHI_BUSFAIL;
    case AHAST_MBO_CMD:		/* MBO command was invalid */
    case AHAST_CCB_OP:		/* CCB opcode was invalid */
    case AHAST_INVD_LINK:	/* Linked CCB have uneq status bytes */
	return XHI_HASWFAIL;
    }
}

/*
/************** BOTTOM I/O DEVICE DRIVER ROUTINES SECTION ********************/

ahalinkccb(int ctlr, struct ahaccb  *fromccb, struct ahaccb  *toccb)
{
	sccb	*ccp;
	DEBUGF(ahatrace,
	    printf("aha%d: Ahalinkccb: fromccb=%x, toccb=%x\n",fromccb,toccb););
	
	ccp = (sccb *)(((char *)&(fromccb->aha_cdb))+(fromccb->aha_cmdlen - 1));
	
	ccp -> sc_link = 1;
	ccp -> sc_flag = 1;

	asgn3(fromccb->aha_lptr,ahioaddr((caddr_t) toccb,ctlr));

	fromccb->aha_fwd = toccb;
	toccb->aha_bwd = fromccb;
}


u_short
ahadoscsi( struct xha_slave *xs, union scsi_cdb  *command,
	   caddr_t addr, int bcount, int read)
{
    struct buf *bp;
    volatile    struct  ahaccb *ccbp; /* HW at least one of the fields it is
                                         pointing at is volatile */
    int rv;
    short	ctlr,target,lun;
    struct	ahambox	*mbop;

    ctlr=xs->xs_ctlr;target=xs->xs_target;lun=xs->xs_lun;

    DEBUGF(ahadoscsidebug | ahatrace,
	printf("aha%d:Ahadoscsi ENTRY: target=%d, lun=%d, addr=%x, bcount=%d\n",
		ctlr,target,lun,addr,bcount););

    bp = &ahaadm[xs->xs_ctlr].stbuf;

    if(bp->b_flags & B_BUSY)
    {
	printf("Ahadoscsi: buffer is busy\n");
	return AHAST_BUFBUSY;
    }

    /* Guard against bcount >0 while addr == 0 */
    bp->b_bcount = (addr != 0) ? bcount : 0;
    bp->b_un.b_addr = addr;
    bp->b_flags = B_BUSY | B_HEAD | (read ? B_READ:0);
	
    if((read == B_READ) && (bp->b_un.b_addr != 0) )
	bzero(bp->b_un.b_addr,bp->b_bcount);
	
    mbop = ahagetmbo(xs->xs_ctlr);
    if(mbop == 0)
	return SCTS_DRV_FAIL;

    ccbp = ahaxsetup(xs,bp,0,command);
    if(ccbp == 0)
    {
	bp->b_flags &= ~B_BUSY;
	ahazapmbo(xs->xs_ctlr,mbop);
	return SCTS_DRV_FAIL;
    }

    ahastartmbo(xs->xs_ctlr,ccbp,mbop);

    DEBUGF(ahadoscsidebug,printf("aha%d:Ahadoscsi: waiting for completion\n",
	    ctlr););

    while((bp->b_flags & B_DONE) == 0)
    {
	DEBUGF(ahadoscsidebug,printf("aha%d:Ahadoscsi: op=%d,hastat=%d\n",
	    ctlr,ccbp->aha_op,ccbp->aha_hastat););
    }

    DELAY(100);
    DEBUGF(ahadoscsidebug,printf("aha%d:Ahadoscsi: completed: op=%d,hastat=%d\n",
	ctlr,ccbp->aha_op,ccbp->aha_hastat););


#ifdef DEBUG
    if(ahadoscsidebug)
    {
	if(ccbp->aha_tarstat == SCTS_CHECK_CONDITION)
	{
		printccb(ccbp);
	}
    }
#endif /* DEBUG */

    if(ccbp->aha_tarstat == SCTS_CHECK_CONDITION)
    {
	struct scsi_esense *sdp;

	sdp  = (struct scsi_esense *)		/* Sense info 	      */
	    (((char *) &ccbp->aha_cdb) + ccbp->aha_cmdlen);

	if(sdp->es_key == SCES_KEY_UNIT_ATTENTION)
	{
	    DEBUGF(ahadoscsidebug | ahatrace,
		printf("aha%d: Retrying\n",ctlr););

	    bp->b_bcount = (addr != 0) ? bcount : 0;
	    bp->b_un.b_addr = addr;
	    bp->b_flags = B_BUSY | B_HEAD | (read ? B_READ:0);

	    if((read == B_READ) && (bp->b_un.b_addr != 0) )
	        bzero(bp->b_un.b_addr,bp->b_bcount);

	    mbop = ahagetmbo(xs->xs_ctlr);
	    if(mbop == 0)
		return SCTS_DRV_FAIL;


            ccbp = ahaxsetup(xs,bp,0,command);
    	    if(ccbp == 0)
	    {
		bp->b_flags &= ~B_BUSY;
		ahazapmbo(xs->xs_ctlr,mbop);
		return SCTS_DRV_FAIL;
	    }
	    ahastartmbo(xs->xs_ctlr,ccbp,mbop);
	    DEBUGF(ahadoscsidebug,printf("aha%d:Ahadoscsi: waiting for completion\n",
		ctlr););

	    while((bp->b_flags & B_DONE) == 0)
	    {
		DEBUGF(ahadoscsidebug,printf("aha%d:Ahadoscsi: op=%d,hastat=%d\n",
		    ctlr,ccbp->aha_op,ccbp->aha_hastat););
	    }
	}
    }

    rv = (ccbp->aha_tarstat + 
	    (ahastattodccode(XHI_CMPL, ccbp->aha_hastat, 0) << 8));
    DEBUGF(ahadoscsidebug,printf("aha%d: Ahadoscsi: tarstat=%d, hastat=%d, rv=%d\n",
		ctlr,ccbp->aha_tarstat,ccbp->aha_hastat,rv););

    DEBUGF(ahadoscsidebug,printf("aha%d:Ahadoscsi: completed: op=%d,hastat=%d\n"
,
        ctlr,ccbp->aha_op,ccbp->aha_hastat););

    bp->b_flags &= ~B_BUSY;
    return rv;
}

struct ahaccb *
ahabdr(struct xha_slave *xs)
	 
{
    struct ahaccb  *ccbp;
    struct ahambox *mbop;

    DEBUGF(ahatrace,printf("aha%d:Ahabdr:xs=%x,\n",
			xs->xs_ctlr,xs););
#ifdef PARANOID
    if(!xs)
    {
	printf("Ahabdr: xs==NULL\n command aborted\n.");
	return 0;
    }
#endif /* PARANOID */

    if((mbop = ahagetmbo(xs->xs_ctlr)) == 0)
    {
	return 0;
    }

    if((ccbp = ahasetupccb(xs, 0, 0, 0)) == 0)
    {
	ahazapmbo(xs->xs_ctlr,mbop);
	return 0;
    }

    ahastartmbo(xs->xs_ctlr,ccbp,mbop);

    return ccbp;
}

/*
/* Set up a CCB */
struct ahaccb *
ahasetupccb( struct xha_slave *xs, union scsi_cdb  *com, 
	     ioaddr_t ioaddr, struct buf *bp)
{
    struct ahaccb  *ccbp;
    DEBUGF(ahatrace,printf("aha%d:Ahasetupccb:xs=%x,com=%x,ioaddr=%x,bp=%x\n",
			xs->xs_ctlr,xs,com,ioaddr,bp););

    if((ccbp = ahagetccb(xs->xs_ctlr)) == 0)
    {
	return 0;
    }

    if (com)
    {
	ahafillccb(ccbp, com, ioaddr, bp ? bp->b_bcount:0, xs, bp);
    }
    else
    {
	/* We assume a bus device reset */
	ccbp->aha_op = AHAOP_BDR;
	ccbp->aha_id = xs->xs_target;
	ccbp->aha_lun = xs->xs_lun;
    }
    return ccbp;
}

void
ahastartmbo(int ctlr, struct ahaccb  *ccbp, register struct ahambox *mbop)
{
    register struct ahaadm  *admp = &ahaadm[ctlr];
#ifdef AHACLID
    register struct ahaccb  *clccbp;
    register int    clid;
    register int    loopcnt;
#endif /* AHACLID */

    DEBUGF(ahatrace,printf("aha%d:Ahastartmbo:ccbp=%x,mbop=%x\n",
			ctlr,ccbp,mbop););

#ifdef AHACLID
    clid = mbop - admp->mbo; /* Get MBO index */
	
    /* Now fill in command link ID's of all CCB's */
    for(clccbp = ccbp, loopcnt=0; clccbp; loopcnt++ )
    {
	if(loopcnt > 1000000)
	{
		printf("Ahastartmbo: loop count exceeded\n");
		break;
	}
	clccbp->aha_clid = clid;
	if((clccbp = (struct ahaccb *) xhaget3(clccbp->aha_lptr)) == 0)
	{
	    break;
	}
	clccbp = (struct ahaccb *) ahvaddr((ioaddr_t) clccbp, ctlr);
    }
#endif /* AHACLID */


    ahafillmbo(ctlr, mbop, ccbp);	/* Fill the MBO and	*/

    if (!admp->nokick) 
    {
	ahakick(ctlr);			/* start the controller	*/
    }

    if(admp->topending == 0)		/* Start a timeout 	*/
    {
	admp->move = 0;
	timeout(ahatout,ctlr,TOTIME);
	admp->topending = 1;
    }
}

void
ahafillmbo(int ctlr, struct ahambox *mbop, struct ahaccb  *ccbp)
{
    DEBUGF(ahatrace,printf("aha%d:Ahafillmbo:ccbp=%x,mbop=%x\n",
			ctlr,ccbp,mbop););
#ifdef DEBUG
    /* Check to see if the mbo was properly gotten from ahagetmbo */
    if(mbop->aha_ccbp != 1)
    {
	DEBUGF(ahanotice,
		printf("Ahafillmbo: mbop=%x, *Mbop=%x\n",mbop,*mbop););
    }
#endif /* DEBUG */
    ccbp->aha_mbo = mbop;   /* Save reference to MBO */
    /* Things MUST happen in the order below */
    mbop->aha_ccbp = ahioaddr((caddr_t) ccbp, ctlr);
    mbop->aha_csb = MBO_START_SCSI;
}



void
ahafillccb( struct ahaccb *ccbp, union scsi_cdb *com, ioaddr_t ioaddr,
	    int dlen, struct xha_slave *xs, struct buf *bp)
{
    DEBUGF(ahatrace,
	printf("aha%d:Ahafillccb:ccbp=%x,com=%x,ioaddr=%x,dlen=%d, xs=%x,bp=%x\n", 
		xs->xs_ctlr,ccbp,com,ioaddr,dlen,xs,bp););
    ccbp->aha_op = AHAOP_SCSI;
    ccbp->aha_id = xs->xs_target;
    switch (com->cdb_6.sc_op)
    {
    case SC0_READ:
    case SC0_MODE_SENSE:
    case SC0_REQUEST_SENSE:
    case SC0_READ_REVERSE:
    case SC1_READ:
	ccbp->aha_din = 1;
	ccbp->aha_dout = 0;
	break;
    case SC0_WRITE:
    case SC0_MODE_SELECT:
    case SC1_WRITE:
	ccbp->aha_dout = 1;
	ccbp->aha_din = 0;
	break;
    default:
	ccbp->aha_dout = 0;
	ccbp->aha_din = 0;
    }
    ccbp->aha_lun = xs->xs_lun;
    ccbp->aha_dlen = dlen;
    asgn3(ccbp->aha_dptr, ioaddr);
    asgn3(ccbp->aha_lptr, 0);
    ccbp->aha_clid = 0;
    ccbp->aha_hastat = AHAST_STARTED;
    ccbp->aha_tarstat = 0;
    ccbp->aha_cmdlen = SCSICMDLEN(com->cdb_6.sc_op);
    if(ccbp->aha_cmdlen == 0) {
	printf("ahafillccb: op code = 0x%02x\n", com->cdb_6.sc_op);
	panic("ahafillccb: invalid SCSI op code\n");
    }
    ccbp->aha_senselen = sizeof (struct scsi_esense);
    bcopy((char *) com, (char *) &ccbp->aha_cdb, ccbp->aha_cmdlen);
    ccbp->aha_xs = xs;
    ccbp->aha_bp = bp;
    if(xs && 
       xs->xs_driver &&
       xs->xs_driver->xd_ttl )
    {
	ccbp->aha_ttl = (*xs->xs_driver->xd_ttl)(com);
    }
    DEBUGF(ahafillccbdebug,
    {
	printf("Ahafillccb: ccb at %x,\n\ttarget=%d, lun=%d\n",ccbp,
		xs->xs_target,xs->xs_lun);
	printccb(ccbp);	/* as suggested by HW */
    }
    );
}
/*
/*
 * Find a free MBI for simulated usage.
 */
struct ahambox *
ahagetmbi(int ctlr)
{
    struct ahambox *mbip;
    DEBUGF(ahatrace,printf("aha%d:Ahagetmbi:\n", ctlr););

    for (mbip = ahaadm[ctlr].mbi; mbip->aha_csb != MBI_MAILBOX_FREE; mbip++)
    {
	if (mbip >= &ahaadm[ctlr].mbi[NMBOX])
	{
	    return 0;
	}
    }
    return mbip;
}

/*
 * Round robin increment of a pointer to a CCB
 * *wrap is increased if wrapping occurred
 */
struct ahaccb *
ahanextccbp(struct ahaadm *admp, struct ahaccb *ccbp, int *wrap)
{
	if(++ccbp >= &admp->ccb[NCCB]) /* Hit the edge yet ? */
	{
		(*wrap)++; /* Tell caller that we wrapped around the edge */
		return admp->ccb; /* Return start of array */
	}
	return ccbp;	/* Just return next one */
}

/*
 * Round robin increment of a pointer to a MBO
 * *wrap is increased if wrapping occurred
 */
struct ahambox *
ahanextmbop(register struct ahaadm *admp, 
	   register struct ahambox *mbop, register int *wrap)
{
	if(++mbop >= &admp->mbo[NMBOX]) /* Hit the edge yet ? */
	{
		(*wrap)++; /* Tell caller that we wrapped around the edge */
		return admp->mbo; /* Return start of array */
	}
	return mbop;	/* Just return next one */
}

/*
 * Round robin increment of a pointer to a MBI
 * If wrap != 0, *wrap is increased if wrapping occurred
 */
struct ahambox *
ahanextmbip(register struct ahaadm *admp, 
	   register struct ahambox *mbip, register int *wrap)
{
	if(++mbip >= &admp->mbi[NMBOX]) /* Hit the edge yet ? */
	{
		if(wrap != 0) 
		    (*wrap)++; /* Tell caller that we wrapped around the edge */
		return admp->mbi; /* Return start of array */
	}
	return mbip;	/* Just return next one */
}
/*
/* Pick a free CCB */
struct ahaccb *
ahagetccb(int ctlr)
{
    register struct ahaccb	*ccbp;
    register struct ahaadm	*admp = &ahaadm[ctlr];
    register struct ahaccb	*ccbpinit;
             int		 wrap = 0;
	     int                 s;

agloop:
    ccbpinit = admp->freeccb;

#ifdef PARANOID
    if((ccbpinit < admp->ccb) || (ccbpinit >= &admp->ccb[NCCB]))
    {
	printf("ahagetccb: ccbpinit out of range: %x set to %x\n",
		ccbpinit,admp->ccb);
	ccbpinit = admp->ccb;
    }
#endif

    s = dsplbio(24);
    for (ccbp = ccbpinit;
	(ccbp->aha_op != AHAOP_CCB_FREE) ; 
	 ccbp=ahanextccbp(admp,ccbp,&wrap))
    {
	if (wrap && (ccbp == ccbpinit))
	{
	    DEBUGF(ahasleepdebug, printf("Ahagetccb: No CCB available.\n"););
	    dsplx(s,25);
	    return 0;
	}
    }

    bzero(ccbp, sizeof *ccbp ); 	/* Zero out the ccb */
    ccbp->aha_op = AHAOP_CCB_RESERVED;	/* Mark it as reserved */
    dsplx(s,26);

#ifdef AHASTATISTICS
    if (++admp->ccbusage > admp->maxccbusage)
    {
	admp->maxccbusage = admp->ccbusage;
	DEBUGF(ahausage, printf("Ahagetccb: Max CCB usage = %d\n", admp->maxccbusage););
    }
#endif /* AHASTATISTICS */

    admp->freeccb = ahanextccbp(admp,ccbp,&wrap);

    return ccbp;
}

/* Get a free MBO */
struct ahambox *
ahagetmbo(int ctlr)
{
    register struct ahaadm	*admp     = &ahaadm[ctlr];
    register struct ahambox	*mbopinit = admp->freembo ;
    register struct ahambox	*mbop;
             int		 wrap     = 0;
	     int		 s;

#ifdef PARANOID
    if((mbopinit < admp->mbo) || (mbopinit >= &admp->mbo[NMBOX]))
    {
	printf("ahagetmbo: mbopinit out of range: %x set to %x\n",
		mbopinit,admp->mbo);
	mbopinit = admp->mbo;
    }
#endif

    s = dsplbio(27);
    for (mbop = mbopinit;
	(mbop->aha_csb != MBO_MAILBOX_FREE) || (mbop->aha_ccbp) ; 
	 mbop=ahanextmbop(admp,mbop,&wrap))
    {
	if (wrap && (mbop == mbopinit))
	{
	    DEBUGF(ahasleepdebug, printf("Ahagetmbo: No mbo avail.\n"););
	    dsplx(s,28);
	    return 0;
	}
    }
    mbop->aha_ccbp = 1;	/* Reserve the Box */

#ifdef AHASTATISTICS
    if (++admp->mbousage > admp->maxmbousage)
    {
	admp->maxmbousage = admp->mbousage;
	DEBUGF(ahausage, printf("Ahagetmbo: Max MBO usage = %d\n", admp->maxmbousage););
    }
#endif /* AHASTATISTICS */

    dsplx(s,29);

    admp->freembo = ahanextmbop(admp,mbop,&wrap);
    return mbop;
}

/*
/*
 * Free a CCB
 */
void
ahafreeccb(int ctlr, struct ahaccb  *ccbp)
{
    struct ahaadm        *admp = &ahaadm[ctlr];
    if (ccbp)
    {
	ccbp->aha_op = AHAOP_CCB_FREE;
	admp->freeccb = ccbp;
#ifdef AHASTATISTICS
	if (--admp->ccbusage < 0)
	    admp->ccbusage = 0;
#endif AHASTATISTICS
    }
#ifdef PARANOID
    else
    {
	printf("ahafreeccb: ccbp=0\n");
    }
#endif /* PARANOID */
}

void
ahafreembi(struct ahambox *mp)
{
      /* We set it to an obviously illegal value so that we can be sure
	that the last byte is really written by the AHA */
      mp->aha_ccbp = 1;	/* Don't change order of these 2 stmts */
      mp->aha_csb = MBI_MAILBOX_FREE;
}


/* 
  * Free and clear MBO entries so that the timeout routine can 
  * recover outstanding requests.
  */
void
ahafreembo(int ctlr, struct ahaccb *ccbp)
{
    register struct ahambox *mbop;


    DEBUGF(ahascandebug | ahatrace,
	    printf("Ahafreembo: ctlr=%d, ccbp=%x\n", ctlr, ccbp););
    if((mbop = ccbp->aha_mbo) != 0)
    {
    DEBUGF(ahascandebug | ahatrace,
	    printf("Ahafreembo: mbop=%x aha_csb=%x aha_ccbp=%x\n", 
		mbop, mbop->aha_csb, mbop->aha_ccbp););
	if(mbop->aha_csb == MBO_MAILBOX_FREE && mbop->aha_ccbp == ahioaddr((caddr_t) ccbp,ctlr))
	{
	    ahazapmbo(ctlr,mbop);
	}
    }
}

void
ahazapmbo(int ctlr, struct ahambox *mbop)
{
    struct ahaadm  *admp = &ahaadm[ctlr];

    DEBUGF(ahatrace,
	printf("Ahazapmbo: ctlr=%d, mbop=%x\n",ctlr,mbop););
    if(mbop)
    {
	bzero(mbop, sizeof *mbop);
	admp->freembo = mbop;
#ifdef AHASTATISTICS
	if (--admp->mbousage < 0)
	    admp->mbousage = 0;
#endif /* AHASTATISTICS */
    }
    return;
}

/*
/***************** HARDWARE CONTROL ROUTINES SECTION *************************/
/*
 * These routines have the purpose of initailizing the hardware and checking
 * hardware operation.
 */
aharesetha(register struct aha_regs   *addr)
{
	register char dummy;
	int	s;
	register char csr;

    /* Empty Status and Data-In port */
    s = dsplhigh(30);

    /* 
     * In the stmt's below the register reading is the important thing, the
     * rest is there in order to fool the optimizer
     */
    if((dummy = addr->aha_csr) != 0)
    {
	dummy++; 
    }
    if((dummy = addr->aha_cd) != 0)
    {
	dummy--;
    }
    if((dummy = addr->aha_iflgs) != 0)
    {
	dummy++;
    }

    DELAY(1000);

    /* Reset everything in it */
    DEBUGF(ahainitdebug, printf("Aharesetha: AHA general reset\n"););
    addr->aha_csr = AHCSR_HRST | AHCSR_SRST | AHCSR_IRST | AHCSR_SCRST;

    DELAY(100000);

    /* Await completion of the self test */
    printf("AHA at 0x%x reset ...", &addr->aha_csr);
    while (addr->aha_csr & (AHCSR_STST))
	DELAY(1000);
    DEBUGF(ahainitdebug, printf("Aharesetha: AHA self test done \n"););

    /* Wait for either DIAGF or INIT to be set */
    while (!((csr = addr->aha_csr) & (AHCSR_DIAGF | AHCSR_INIT)))
	DELAY(100);
    printf(" done.\n");

    DEBUGF(ahainitdebug,printf("Aharesetha: AHA csr=%d\n",(short) csr););
    if (csr & AHCSR_DIAGF)
    {
	printf("Aharesetha: internal diagnostics failed\n");
	dsplx(s,31);
	return 1;
    }
    dsplx(s,32);
    return 0;
}

ahareinit(register struct iocc_ctlr *ic)
{
    register struct aha_regs   *addr = (struct aha_regs *) ic->ic_addr;

    if(aharesetha(addr))
    {
	printf("Ahareinit: Aha reset failed (Internal diag failure)\n");
	return 1;
    }
    if(ahainit(ic))
    {
	return 1;
    }
    ahainitmbox(NMBOX,(struct ahambox *) ahaadm[ic->ic_ctlr].mbo,ic->ic_ctlr);
    return 0;
}

/*
/* Initialize the host adapter */
ahainit(register struct iocc_ctlr *ic)
{
    int             		i;
    int         		rv;
    u_int           		intchan;
    register struct aha_regs   *addr = (struct aha_regs *) ic->ic_addr;
    char			swvers[5];
    char		       *board_id;
    char		       *bios_ver="";
    int				c_model=0;
    struct aha_extbios {
	u_char  flags;		/* Bit 3 == 1 extended bios enabled */
	u_char  mailboxlock;	/* mail box lock code to unlock it */
    } extbios;


    /* Get SW revision */
    ahacmd(addr, AHACMD_INQ,ic->ic_ctlr, 1);
    DELAY(10000);
    for(i=0; i < 4; i++)
    {
	if ((rv = ahaget(addr,ic->ic_ctlr)) < 0)
	{
	    printf("Ahainit: init failed(INQ, i = %d)\n",i);
	    return 1;
	}
	swvers[i] = rv;
    }
    swvers[4] = '\0';
    switch(swvers[0]) {
	case 0x20:	/* BusLogic 545 */
		board_id="BusLogic 545";
		break;
	case 0x31:	/* AHA-1540 */
		board_id="AHA-1540";
		break;
	case 0x41:	/* AHA-1540A/1542A/1542B */
		board_id="AHA-1542B";
		break;
	case 0x42:	/* AHA-1640 */
		board_id="AHA-1640";
		break;
	case 0x43:	/* AHA-1542C */
		board_id="AHA-1542C";
		c_model=1;
		break;
	case 0x45:	/* AHA-1542CF, BIOS v2.01 */
		bios_ver=" (BIOS v2.01)";
	case 0x44:	/* AHA-1542CF */
		board_id="AHA-1542CF";
		c_model=1;
		break;
	default:
		board_id="";
    }
    printf("aha%d: %s%s Firmware Revision '%s'\n",
	ic->ic_ctlr, board_id, bios_ver, swvers+2);

    /* Get the configuration data */
    ahacmd(addr, AHACMD_RCD, ic->ic_ctlr, 1);
    if ((rv = ahaget(addr,ic->ic_ctlr)) < 0)
    {
	printf("Ahainit: Init failed\n");
	return 1;
    }
    ic->ic_dmachannel -= 1;
    if (rv)
    {
	for (ic->ic_dmachannel = 0; !(rv & 1); ic->ic_dmachannel++)
	{
	    rv >>= 1;
	}
    }
    DEBUGF(ahainitdebug,
	printf("aha%d: Ahainit: Dma channel %d\n", ic->ic_ctlr, ic->ic_dmachannel););

    if ((rv = ahaget(addr,ic->ic_ctlr)) < 0)
    {
	printf("Ahainit: Init failed\n");
	return 1;
    }
    intchan = 0;
    if (rv)
    {
	for (intchan = 9; !(rv & 1); intchan++)
	{
	    rv >>= 1;
	}
    }
    DEBUGF(ahainitdebug,
	printf("aha%d: Ahainit: Interrupt channel %d\n", ic->ic_ctlr, intchan););

    if ((rv = ahaget(addr,ic->ic_ctlr)) < 0)
    {
	printf("Ahainit: Init failed\n");
	return 1;
    }
    DEBUGF(ahainitdebug,printf("aha%d: Ahainit: SCSI ID=%d\n", ic->ic_ctlr, rv & 7););


    ahasbontime(AHABUS_ON, ic->ic_ctlr);	/* Set Max BUSON time       */
    ahasbofftime(AHABUS_OFF, ic->ic_ctlr);	/* Set BUS OFF time         */
    if (!c_model) {
	/* XXX Must deal with configuring the DRQ channel if we do this */
	ahasetspeed(AHASPEED8, ic->ic_ctlr);	/* Set Transfer speed       */
    }
    ahassto(250, ic->ic_ctlr);			/* Selection timeout 250 mS */
    /* ahasetmboint(AHAMBOINT_ENABLE,ic->ic_ctlr); /* Enable mailbox-out int*/

   /* If we are a 1542C or 1542CF disable the extended bios so that the
    * mailbox interface is unlocked.
    * No need to check the extended bios flags as some of the
    * extensions that cause us problems are not flagged in that byte.
    */
    if (c_model) {
	printf("aha%d: %s detected, unlocking mailbox\n",ic->ic_ctlr, board_id);
	ahacmd(addr, AHACMD_EXT_BIOS, ic->ic_ctlr, 1);
	DELAY(10000);
	if ((extbios.flags = (u_char)ahaget(addr, ic->ic_ctlr)) < 0) {
	    printf("Ahainit: EXT_BIOS get flags failed\n");
	    return 1;
	}
	if ((extbios.mailboxlock = (u_char)ahaget(addr, ic->ic_ctlr)) < 0) {
	    printf("Ahainit: EXT_BIOS get mailboxlock failed\n");
	    return 1;
	}
	ahacmd(addr, AHACMD_MBX_ENABLE, ic->ic_ctlr, 1);
	ahacmd(addr, 0, ic->ic_ctlr, 0);
	ahacmd(addr, extbios.mailboxlock, ic->ic_ctlr, 0);
    }
    return 0;
}

/*
/*
 * Timeout routine
 */

void
ahatout(int ctlr)
{
    struct aha_regs *addr = (struct aha_regs *) ahainfo[ctlr]->ic_addr;
    struct ahambox *mbop;
    struct ahambox *mbip;
    struct ahaccb  *ccbp;
    struct ahaadm  *admp;
    u_char          iflgs;
    int	i,s,nmboxes;

    /* First note to driver that no timeout is pending */
    s = dsplbio(33);
    admp = &ahaadm[ctlr];
    admp->topending = 0;
    /* See if a kick was skipped */
    if(admp->skippedkick)
    {
	ahakick(ctlr);
	if(admp->skippedkick)
	{
	    admp->topending = 1;
	    timeout(ahatout,ctlr,TOTIME);
	}
	dsplx(s,34);
	return;
    }

    /* Are there currently any active MBO's ? */

#ifdef AHASTATISTICS
    if(admp->mbousage == 0)
    {
	dsplx(s,36);
	return;
    }
#endif /* AHASTATISTICS */

    /* OK, there are active MBO's, are we hung ? */
    /* See if there has been any interrupts since last Timeout activation */
    if(admp->move != 0)
    {
	char	dummy;
	/* Yes, interrupts has been taken, clear the move bit and issue 
	 * another timeout to see if the ctrlr REALLY have been waiting
	 * for a whole timeout period (with MBO's active).
	 */
	admp->move = 0;
	admp->topending = 1;
	dummy = addr->aha_cd;
	timeout(ahatout,ctlr,TOTIME);
	dsplx(s,37);
	return;
    }
    /* We get here when there are outstanding MBO's and there have been no
     * interrupt for a whole timeout period
     */

    DEBUGF(ahatoutdebug | ahaerrdebug,
	printf("Ahatout: csr=%b, iflgs=%b, prio=%d\n",
	    addr->aha_csr,AHCSR_BITS,addr->aha_iflgs,AHIFLGS_BITS, getspl());
    );

    iflgs = addr->aha_iflgs;		/* Read the interrupt port  */
    if(iflgs & AHIFLGS_INT)
    {
	(void) spl3();
	DEBUGF(ahatoutdebug | ahanotice,printf("Ahatout: Lost interrupts, Simulated one\n"););
	ahaint(ctlr);
	dsplx(s,38);
	return;
    }

    if(!ahatocheck)
    {
	dsplx(s,35);
	return;
    }
#ifdef DEBUG
    if(ahatoutdebug | ahaerrdebug)
    {
	printmbox(admp->mbo,0,NMBOX);
	DELAY(3000000);
	printmbox(admp->mbi,1,NMBOX);
	DELAY(3000000);
    }
#endif /* DEBUG */
    if (addr->aha_csr & AHCSR_IDLE)
    {
	int kickit=0;
	for(i=0;i<NMBOX;i++)
	{
	    mbop = &admp->mbo[i];

	    if(mbop->aha_ccbp)
	    {
		ccbp = (struct ahaccb *)
		    ahvaddr(mbop->aha_ccbp, ctlr); 
		if(ccbp < admp->ccb || 
		   ccbp >= &admp->ccb[NCCB])
		{
		    printf("Ahatout: Invalid CCB pointer = %x,ioccbp=%x\n",ccbp,mbop->aha_ccbp);
		    DELAY(1000000);
		    continue;
		}
		printf("Ahatout: forgotten mbox=%x\n",mbop);
		if(ccbp->aha_hastat == AHAST_STARTED)
		{
			printf("Ahatout: Hastat == STARTED, mbox restarted\n");
			mbop->aha_csb = MBO_START_SCSI;
			kickit++;
		}
		else
		{
			printf("Ahatout:Hastat = %d, Tarstat=%d\n",ccbp->aha_hastat,
				ccbp->aha_tarstat);
			mbip = ahagetmbi(ctlr);
			if(ccbp->aha_hastat || ccbp->aha_tarstat)
			{
			    mbip->aha_csb = MBI_ERROR;
			}
			else
			{
			    mbip->aha_csb = MBI_NOERROR;
			}

			mbip->aha_ccbp = mbop->aha_ccbp;
			printf("Simulated MBI intr for MBI=%x, ccbp=%x\n",
			    mbip, mbop->aha_ccbp);
			ahascanmbi(ctlr,0);
		}
	    }
	}
	if(kickit)
	{
	    ahakick(ctlr);
	}
	dsplx(s,39);
	return;
    }
    /* See if there is any CCB that isn't timed out yet */
    for(i=0;i<NMBOX;i++)
    {
	mbop = &admp->mbo[i];

	if(mbop->aha_ccbp)
	{
	    ccbp = (struct ahaccb *)
		ahvaddr(mbop->aha_ccbp, ctlr); 
	    if(ccbp < admp->ccb || 
		   ccbp >= &admp->ccb[NCCB])
	    {
		printf("Ahatout: Invalid CCB pointer = %x,ioccbp=%x\n",ccbp,mbop->aha_ccbp);
		DELAY(1000000);
		continue;
	    }
	    ccbp->aha_ttl -= TOTIME/HZ;
	    if(ccbp->aha_ttl >= 0)
	    /* This one didnt time out, so there is really no controller hang */	    /* we just restart another timeout period */
	    {
		admp->move = 0;
		admp->topending = 1;
		DEBUGF(ahatoutdebug | ahaerrdebug,printf("Ahatout: No timeout yet \n"); );
		timeout(ahatout,ctlr,TOTIME);
		dsplx(s,40);
		return;
	    }
	}
    }
    printf("aha%d is hung\n", ctlr);
    /* ahatstartdebug=1; */
    /* ahacstartdebug=1; */
    dsplx(s,41);
    ahareinit(xha_hd[ctlr].xh_iocc);
    admp->nokick = 1;	    /* Don't start anything yet */

    nmboxes=0;
    for (mbop = admp->mbo;
	 mbop < &admp->mbo[NMBOX]; mbop++)
    {
	if (mbop->aha_ccbp)
	{
	    mbip = ahagetmbi(ctlr);
	    mbip->aha_csb = MBI_HOST_TIMEOUT;
	    mbip->aha_ccbp = mbop->aha_ccbp;
	    nmboxes++;
	    DEBUGF(ahatoutdebug | ahanotice,printf("Ahatout: Simulated MBI:%x=%x.%x, for mbo:%x=%x.%x\n",
		    mbip, mbip->aha_csb, mbip->aha_ccbp,
		    mbop, mbop->aha_csb, mbip->aha_ccbp););
	}
    }
    if(nmboxes)
    {
	ahascanmbi(ctlr,0);
    }
    admp->nokick = 0;		    /* Let's go again */
    /* Kick the controller */
    ahakick(ctlr);
}

ahaget(struct aha_regs *addr, int ctlr)
{
    int             waittime = 0;
    int             rv;
#ifdef DEBUG
    short           csr = -1;
#endif /* DEBUG */
    u_char          ahcsr;

    for (ahcsr = addr->aha_csr; !(ahcsr & AHCSR_DF); ahcsr = addr->aha_csr)
    {
	if ((ahaadm[ctlr].cmdcmpl != CMPL_NOTYET) && (ahaadm[ctlr].cmdcmpl != CMPL_CC))
	{
	    printf("aha%d: AHACMD ABORTED cc=%d\n",ctlr,ahaadm[ctlr].cmdcmpl);
	    ahaadm[ctlr].cmdcmpl = CMPL_ABORT;
	    return -1;
	}
	if (ahcsr & AHCSR_INVDCMD)
	{
	    printf("aha%d: AHACMD INVALID\n",ctlr);
	    ahaadm[ctlr].cmdcmpl = CMPL_INVD;
	    return -1;
	}
#ifdef DEBUG
	if (ahadebug)
	{
	    if (ahcsr != csr)
	    {
		if (csr != -1)
		    printf("\n");
		printf("aha%d: Ahaget: csr=%b", ctlr, (short) ahcsr, AHCSR_BITS);
		csr = ahcsr;
	    }
	    else
	    {
		printf(".");
	    }
	}
#endif /* DEBUG */
	DELAY(100000);
	if (++waittime > 40)
	{
	    printf("aha%d: Ahaget: timeout\n",ctlr);
	    return -1;
	}
    }
#ifdef DEBUG
    if (ahadebug)
    {
	if (csr != -1)
	{
	    printf("\n");
	}
    }
#endif /* DEBUG */
    rv = addr->aha_cd;
    DEBUGF(ahadebug, printf("Ahaget: returning 0x%x\n", rv););
    return rv;
}

void
ahascreset(struct aha_regs  *addr)
{
	addr->aha_csr = AHCSR_SCRST;
}

/* Send START SCSI command to the HA, Don't have to wait for IDLE */
void
ahakick(int ctlr)
{
    u_char          cmd;
    u_char          csr;
    u_char	    dummy,dummyread=0;
    int		    loops=0;
    register int	s;
    struct aha_regs *addr=(struct aha_regs *)ahainfo[ctlr]->ic_addr;
    register struct ahaadm *admp=&ahaadm[ctlr];

#ifdef FBQ
	if(ctlr != 0 )
	{
		printf("Ahakick: ctlr=%d, skipped\n");
		return;
	}
#endif

    cmd = AHACMD_START_SCSI;

    s = dsplbio(42);
    if(admp->kicking)
    {
    	dsplx(s,43);
	return;
    }
    admp->kicking = 1;

again:
    if( (csr = addr->aha_csr) & (AHCSR_CDF))
    {
	admp->skippedkick = 1;
	admp->kicking=0;
	dsplx(s,44);
	return;
    }
	
    for (csr = addr->aha_csr; (csr & (AHCSR_STST | AHCSR_CDF))
	 ; csr = addr->aha_csr)
    {
	DEBUGF(ahakickdebug,printf("Ahakick: csr=%b loops=%d\n",csr,AHCSR_BITS,loops++););
	loops++;
	if(loops > 5000000)
	{
		if(dummyread==0)
		{
		    printf("Ahakick: Almost infinite loop\n");
		    dummy = addr->aha_cd;
		    dummyread++;
		    printf("ahakick: Read dummy data = %d\n",dummy);
		    loops = 0;
		    goto again;
		}
		break;
	}
	DELAY(1);
    }
    if((csr & AHCSR_DF) && (dummyread == 0))
    {
	dummy = addr->aha_cd;
	printf("ahakick: Read dummy data = %d\n",dummy);
	goto again;
    }
    addr->aha_cd = cmd;
    admp->kicking=0;
    admp->skippedkick = 0;
    dsplx(s,45);
    return;
}

ahacmd(struct aha_regs *addr, u_char cmd, int ctlr, int cmd_op)
{
    int             waittime = 0;
    int             waitmax = 300;
    u_short         csr;
    u_short         ir;
    int		    s; 

    DEBUGF(ahacmddebug, printf("Ahacmd: Addr=%x, cmd=%x\n", addr, cmd););
    s = dsplbio(46);
    for (csr = addr->aha_csr;
	 (csr & (AHCSR_STST | AHCSR_CDF)) ||
	 (cmd_op && !(csr & AHCSR_IDLE));
	  csr = addr->aha_csr)
    {
	ir = addr->aha_iflgs;
	DEBUGF(ahacmddebug, printf("Ahacmd: Aha_csr = %b aha_iflgs=%b\n",
		                   csr, AHCSR_BITS, ir, AHIFLGS_BITS););
	DELAY(1000);
	if (++waittime > waitmax)
	{
            printf("Ahacmd: Addr=%x, cmd=%x timeout\n", addr, cmd);
	    /*
	     * Primitive recovery action.
	     */
	    if (waitmax >= 300) /* HW == changed to >= */
	    {
		if (csr & AHCSR_CDF)
		{
		    printf("Warning... aha reset due to too many timeouts\n");
		    addr->aha_csr = AHCSR_IRST | AHCSR_SRST;
		}

		waitmax = 600;
	    }
	    /* We get here if reset didn't work */

	    dsplx(s,47);
	    return -1;
	}
    }
    ahaadm[ctlr].cmdcmpl = CMPL_NOTYET;
    DEBUGF(ahacmddebug, printf("Ahacmd: Command=0x%x\n", (short) cmd););
    addr->aha_cd = cmd;
    dsplx(s,48);
    return 0;
}

#define ahaop(xcmd) ahacmd((struct aha_regs *) ahainfo[ctlr]->ic_addr,(xcmd),ctlr,1);
#define ahaopdata(xdata) ahacmd((struct aha_regs *) ahainfo[ctlr]->ic_addr,(xdata),ctlr,0)

/* Initialize the mailbox pointer */
void
ahainitmbox(int n, struct ahambox *amo, int ctlr)
{
    ioaddr_t        mbp = ahioaddr((caddr_t) amo, ctlr);

    ahaop(AHACMD_MBINIT);
    ahaopdata(n);			    /* # of mailboxes */
    ahaopdata((long) mbp >> 16 & 0xFF);/* MSB  */
    ahaopdata((long) mbp >> 8 & 0xFF); /* MISB */
    ahaopdata((long) mbp & 0xFF);	    /* LSB  */
}

void
ahasetspeed(int i, int ctlr) 
{
    ahaop(AHACMD_SXFER_SPEED);
    ahaopdata(i);
}

void
ahawritefifo(char *cp, int ctlr) 
{
    ioaddr_t        iop = ahioaddr(cp, ctlr);

    ahaop(AHACMD_WFIFO);
    ahaopdata(iop >> 16 & 0xFF);	    /* MSB  */
    ahaopdata(iop >> 8 & 0xFF);	            /* MISB */
    ahaopdata(iop & 0xFF);		    /* LSB  */
}

void
ahareadfifo(caddr_t cp, int ctlr) 
{
    ioaddr_t        iop = ahioaddr(cp, ctlr);

    ahaop(AHACMD_RFIFO);
    ahaopdata(iop >> 16 & 0xFF);	    /* MSB  */
    ahaopdata(iop >> 8 & 0xFF);	    /* MISB */
    ahaopdata(iop & 0xFF);		    /* LSB  */
}

void
ahassto(int time, int ctlr)
{
    ahaop(AHACMD_SSTO);
    ahaopdata(time == 0 ? AHASSTO_DISABLE : AHASSTO_ENABLE);
    ahaopdata(0);
    ahaopdata(time >> 16 & 0xFF);
    ahaopdata(time & 0xFF);
}

void
ahasetmboint(int ena, int ctlr)
{
    ahaop(AHACMD_MBOINT);
    ahaopdata(ena);
}

void
ahasbontime(int time, int ctlr)
{
    ahaop(AHACMD_SBON_TIME);
    ahaopdata(time & 0xFF);
}

ahasbofftime(int time, int ctlr)
{
    ahaop(AHACMD_SBOFF_TIME);
    ahaopdata(time & 0xFF);
}
/*
/******************** DMA INTERFACE ROUTINES **********************************/
/*
 * Convert I/O address to virtual memory address, 
 */
caddr_t
ahvaddr(ioaddr_t ioaddr, int ctlr)
{
	return (caddr_t) ioaddr -
	    (caddr_t) ahaadm[ctlr].ioaddr +
	    (caddr_t) &ahaadm[ctlr];
}

/*
 * Convert vitrual memory address to an I/O address.
 */
ioaddr_t
ahioaddr(caddr_t addr, int ctlr)
{
	return (ioaddr_t) ((long) ahaadm[ctlr].ioaddr +
			   (addr - (caddr_t) &ahaadm[ctlr]));
}

/*
/***************** DEBUGGING ROUTINES *****************************************/

#ifdef DEBUG
/* Some debugging aids */
void printccb(struct ahaccb  *cp)
{
    struct scsi_esense	*esp;

    printf("CCB    at %x:", cp);
    if (!cp)
    {
	return;
    }
    xhdump((u_char *)cp, 16);
    printf("CDB    at %x:", &cp->aha_cdb);
    xhdump((u_char *)&cp->aha_cdb, cp->aha_cmdlen);
    esp = (struct scsi_esense *) (((char *) (&cp->aha_cdb)) + cp->aha_cmdlen);
    printf("Esense at %x:", esp);
    xhdump((u_char *) esp, 8 + esp->es_addlen);
    printf("Xs & bp  are:");
    xhdump((u_char *)&cp->aha_xs, 8);
    printf("\n\tOp\t= 0x%x", cp->aha_op);
    printf("\tTarget\t= %d", cp->aha_id);
    printf("\tDout\t= %d, Din= %d", cp->aha_dout, cp->aha_din);
    printf("\tLun\t= %d", cp->aha_lun);
    printf("\tCmdlen\t= %d\n", cp->aha_cmdlen);
    printf("\tDlen\t= %d", cp->aha_dlen);
    printf("\tDptr\t= 0x%x%x%x", cp->aha_dptr[0], cp->aha_dptr[1], cp->aha_dptr[2]);
    printf("\tLptr\t= 0x%x%x%x", cp->aha_lptr[0], cp->aha_lptr[1], cp->aha_lptr[2]);
    printf("\tClid\t= %d", cp->aha_clid);
    printf("\tHastat\t= %d", cp->aha_hastat);
    printf("\tTarstat\t= %d\n", cp->aha_tarstat);
    printcdb(&cp->aha_cdb,cp->aha_xs->xs_xi->xi_type);
    printesense(esp);
}

void printcdb(union scsi_cdb *cdbp, int type)
{
	struct	scsi_cmdinfo	*cip;
	char	**carr;
	int	cmax;
	int	i;

	if(type > scsi_cmdinfomax)
		type = 0;
	cip = &scsi_cmdinfo[type];
	carr = cip->cmdarr;
	cmax = cip->arrmax;

	if(cdbp->cdb_6.sc_op <= cmax )
		printf("CDB op=0x%x(%s)\n",cdbp->cdb_6.sc_op,
			carr[cdbp->cdb_6.sc_op]);
	else
		printf("CDB op=0x%x(???)\n",cdbp->cdb_6.sc_op);
}

void printesense(struct scsi_esense *esp)
{
	printf("Sense key=0x%x(%s)\n",esp->es_key,scsi_esense_keys[esp->es_key]);
	if(esp->es_addlen > 5)
		printf("Sense code=0x%x\n",esp->es_scode);
	if(esp->es_addlen > 6)
		printf("Subsense code=0x%x\n",esp->es_subscode);
	if(esp->es_addlen > 10)
	{
		if(esp->es_fpv)
			printf("%s error at byte %d",esp->es_cd ? "CDB":"Data",
				esp->es_fp);
		if(esp->es_bpv)
			printf(" bit %d", esp->es_bp);

		printf("\n");
	}
}

void printmbox(struct ahambox *box, int io, int n)
{
    if (!box)
    {
	return;
    }
    if(n<=0)
    {
	return;
    }

    printf("%s Mbox%s at %x:\n", io ? "In " : "Out", n>1 ? "es":"", box);
	xhdump((u_char *)box,4*n); 
}

#endif /* DEBUG */
#endif /* NAHA */
