#include "cd.h"
#if NCD > 0 
/*
 * Copyright (C) 1988, 1989 Christer Bernerus, 
 *     		 Chalmers University of Technology, Gothenburg, SWEDEN
 * Copyright (C) 1993 Mark Dapoz, 
 *     		 Bergen Scientific Centre, Bergen, NORWAY
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms as is or as a source
 * for derivative work are permitted provided that the above copyright 
 * notice and this paragraph are  duplicated in all such forms 
 * and that any documentation, advertising materials,
 * and other materials related to such distribution
 * and use acknowledge that the software was developed
 * at Chalmers University of Technology, SWEDEN.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *
 * SCSI CD-ROM driver
 *
 * $Source: /tmp/u/sys/rt/dev/RCS/cd.c,v $
 * $Revision: 1.14 $ $Date: 1994/06/11 20:26:19 $
 * $Author: roger $ $State: Exp $
 *
 * $Log: cd.c,v $
 * Revision 1.14  1994/06/11  20:26:19  roger
 * changes for the '44 fcntl.h.
 *
 * Revision 1.13  1994/05/22  12:33:57  roger
 * header file changes.
 *
 * Revision 1.12  1994/05/13  15:11:25  roger
 * srreg.h moved to rtio.
 *
 * Revision 1.11  1994/05/07  14:12:23  md
 * Change #include statements to use system search path instead of absolute.
 *
 * Revision 1.10  1994/04/28  22:17:12  md
 * Limit the amount of data rwritten to the d_typename member.
 *
 * Revision 1.9  1994/04/28  20:27:10  md
 * Return proper name for d_typename when asked for disk information
 *
 * Revision 1.8  1994/04/27  21:53:13  md
 * Set the partition types to be ISO 9660 when the disk label is requested.
 *
 * Revision 1.7  1994/04/20  19:14:44  md
 * Add support for immutable and append only files.
 *
 * Revision 1.6  1994/04/05  12:18:07  md
 * Allow zero length buffers for uscsi commands.
 * Set uscsi_status to reflect result of uscsi command just issued.
 *
 * Revision 1.5  1994/03/26  09:26:37  md
 * Changes to support multisession PhotoCD discs.
 *
 * Revision 1.4  1994/02/07  15:18:49  md
 * Fix firmware rev and date string being printed (was being cut short).
 *
 * Revision 1.3  1994/02/01  13:45:59  md
 * Many changes to support the Toshiba XM-3401B CD-ROM drive.
 *
 * Revision 1.2  1993/05/12  21:15:23  md
 * changes needed for cd-rom support for iso filesystem
 *
 * Revision 1.1  1993/04/20  19:44:41  md
 * Initial revision
 *
 *
 */

#include <sys/param.h>
#include <sys/vm.h>
#include <sys/buf.h>
#include <sys/time.h>
#include <sys/proc.h>
#include <sys/errno.h>
#include <machine/io.h>
#include <sys/file.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/systm.h>
#include <sys/cmap.h>
#include <sys/dkstat.h>
#include <sys/ioctl.h>
#include <sys/disklabel.h>
#include <machine/mmu.h>
#include <machine/dkio.h>

#include "rt/include/ioccvar.h"
#include "rt/include/scsi.h"
#include "rt/dev/scsi_cd.h"
#include "rt/dev/srreg.h"
#include "rt/include/xhavar.h"
#include "rt/dev/cdio.h"
#include "rt/rt/debug.h"
#include "rt/dev/scsiconfig.h"
#include <ufs/fs.h>

#define MAXNPART 8
#define MAXQPERUNIT 2
#define MAXNBPS	        4096     
/* #define DSPL		/* show status on LED's */
#ifndef DSPL
#define dsplbio(x) splbio()
#define dsplx(x,y) splx(x)
#define dsplhigh(x) splhigh()
#endif /* DSPL */

#define PARANOID

#define	TOSHIBA		/* support for Toshiba multisession discs */

#define	CD_BSIZE 2048	/* mode 1 sector size */
#define	BLOCK_SIZE_2048	/* enable performance enhancements */

/*#define	CD_BSIZE 2336	/* mode 2 sector size? */


/* THIS SHOULD BE READ OFF THE PACK, PER DRIVE */
struct size {
	struct partentry {
	    daddr_t nblocks;
	    int     pblkoff;
	}
	partition[MAXNPART];
};

struct size
cd_def_sizes = {
        15884,  0,              /* A=cyl 0 thru 29 */
        33480,  16200,          /* B=cyl 30 thru 91 */
        621180,   0,            /* C=cyl 0 thru 1217 */
        16740,  49680,          /* D=cyl 92 thru 122 */
	0,	0,		/* E = nothing */
	0,	0,		/* F = nothing */
	82620, 538380,		/* G = cyl 997 thru 1149 */
        471960,  66420,		/* H = cyl 123 thru  996 */
};

struct size
cd_sizes = {
	0,	0,		/* A = nothing */
	0,	0,		/* B = nothing */
        72479,  0,		/* C = whole CD */
	0,	0,		/* D = nothing */
	0,	0,		/* E = nothing */
	0,	0,		/* F = nothing */
	0,	0,		/* G = nothing */
	0,	0,		/* H = nothing */
};

struct size
cd_ibm_sizes = {
	0,	0,		/* A = nothing */
	0,	0,		/* B = nothing */
        72479,  0,		/* C = whole CD */
	0,	0,		/* D = nothing */
	0,	0,		/* E = nothing */
	0,	0,		/* F = nothing */
	0,	0,		/* G = nothing */
	0,	0,		/* H = nothing */
};

/* The structure below is copied into the softc structure on attach */
struct  cdst {
	char	inqname[17];
        short   nsect;
        short   ntrak;
        short   nspc;
        short   ncyl;
	short	nbps;
        struct  size *defsizes;
	char	typename[8];
	short	flags;
#define CDFL_NO_LINK	0	/* SCSI Command linking not supported */
#define CDFL_LINK	1	/* SCSI Command linking supported */
} cdst[] = {
/* inqname,nsect,ntrak, nspc, ncyl, nbps, def.sizes, typename    */
{"Default ", 34, 1, 34*15, 1218, 512,  &cd_def_sizes,"default", CDFL_NO_LINK},
{"CD-ROM DRIVE:XM",  0, 1, 0, 1, CD_BSIZE, &cd_ibm_sizes, "XM-3301B",CDFL_LINK},
{"CD-ROM XM-3401TA", 0, 1, 0, 1, CD_BSIZE, &cd_sizes, "XM-3401B", CDFL_LINK},
};

struct  cd_softc {
	int	sc_busy:1;		/* Set when BUSY is detected, reset otherwise; */
	int	sc_nomap:1;	        /* Don't map or check partitions */
	int	sc_cinited:1;		/* True if c part'n is initialized */
	int	sc_spundown:1;		/* Drive has been spun down */
	int	sc_autospinup:1;	/* Spin up automatically	*/
	int	sc_noretries:1;		/* Don't do retries on SCSI pass thru */
	int	sc_busyretries;		/* No of retries for busy drive */
	int	sc_queuedepth;		/* Current queue depth */
	struct	partsec
	{
	    char	ps_phys:1;	/* Xfer had PHYS flag set in buffer */
	    struct buf *ps_bp;		/* Associated buffer zero if unused */
	    caddr_t	ps_addr;	/* Original b_addr in buf */
	    long	ps_bcount;	/* Original bcount in buf */
	    long	ps_nbps;	/* Sector size of the disk */
	    char	ps_space[MAXNBPS];	/* Data space */
	} sc_part[MAXQPERUNIT];
	struct	scsi_capdata	sc_cdata;	/* Capacity data */
	struct	cdst sc_cdst;
	struct	size sc_size;		/* Actual partitioning */
} cd_softc[NCD];

#ifdef	TOSHIBA
#define	CD_AUDIO_TYPE	0x00		/* CD Audio or CD-ROM disc */
#define	CD_I_TYPE	0x10		/* CD-I disc */
#define	CD_XA_TYPE	0x20		/* CD-ROM XA disc */

static struct tosh_cd_id {		/* Toshiba disc information */
	u_char	disc_id;
	u_char	minutes;
	u_char	seconds;
	u_char	frames;
} toshiba_disc_id = {0, 0, 0, 0};
#endif


/* Functions predeclarations */
#include "rt/dev/xxc.h"

int cdprobe(caddr_t ,struct xha_device *);
int cdattach(struct xha_slave *);
int cdttl(union scsi_cdb *);
int cdmodesense(struct xha_slave *, struct scsi_msiobuf *);
int cdstartstop(struct xha_slave *,short);
int cdreadcapacity(struct xha_slave *, struct scsi_capdata *);
int cdslave(struct xha_slave *,caddr_t);
int cdgettype(struct xha_slave *, char **, char *);
int cdopen(dev_t,int);
int cdclose(dev_t,int);
int cdxstrat(struct xha_slave *,int , struct buf *);
int cdstrategy(struct buf *);
int cdustart(struct xha_slave *);
int cdfillpart(struct partsec *);
int cdint(struct xha_slave *, struct buf *, int, struct scsi_esense *, u_char);
int cdpartdone(struct xha_slave *, struct buf *);
int cdemptypart(struct partsec *);
int cdread(dev_t, struct uio *);
int cdwrite(dev_t, struct uio *);
int cdphys(dev_t dev);
int cdioctl(dev_t, int, caddr_t, int);
int cdminphys(struct buf *);
int cdtryretry(struct xha_slave *, struct buf *, int,int);
int cdidcpy(u_char *,char *,int);
void cdpartsetup(struct xha_slave *, struct buf *, int, int);
union scsi_cdb *cdstart(struct xha_slave *,struct buf *,union scsi_cdb *);
struct partsec * cdgetps(struct xha_slave *, struct buf *);
unsigned cdget3(char *);

struct partsec *
cdgetps(struct xha_slave *xs, struct buf *bp)
{
        struct cd_softc *sc = &cd_softc[xs->xs_unit];
	
	struct partsec *ps;

	for(ps = sc->sc_part; ps < &(sc->sc_part[MAXQPERUNIT]); ps++)
		if(ps->ps_bp == bp) return ps;
	return NULL;
}
#define PARTIAL(bp)	(((bp)->b_un.b_addr > (caddr_t) cd_softc) && ((bp)->b_un.b_addr < (caddr_t) &cd_softc[NCD]))

/*
 * Since we want to have more than one transfer on each unit going on
 * "simultaneously", we have to track retry counts in each buffer instead
 * of doing it in the device's buffer queue header. This means that a suitable
 * variable must be found that isn't needed during I/O. The only one we cound
 * find is b_error. Unfortunately this variable isn't cleared for transfers
 * coming from the swap system so we clear it here in the device driver since
 * we didn't want to touch the swap system's code.
 * In ordinary device drivers the retry counting is done in the b_resid variable
 * in the buffer queue head (which is just a header). The definition for this is
 * in ../h/buf.h which defines b_errcnt to be b_resid. The b_resid variable
 * can't be used in the transfer buffers since we use it for indicating cylinder
 * number to disksort(), and thus can't easily be changed.
 * When a transfer is completed, b_error will regain its usual function.
 */ 
#undef b_cylin
#define b_cylin b_resid
#undef b_errcnt
#define b_errcnt b_error

#define dkblock(bp)	((bp)->b_blkno/(cd_block_size/DEV_BSIZE))

int cd_block_size=CD_BSIZE;	/* initial default */
struct size cdsizes[NCD];


/* Bits in minor device */
/*	xx xx xxxx
	++ ++ +--+
	 |  |   \- partition number
	 |  \----- unit number
	 \-------- mode type (mode1, mode2/form1, mode2/form2)
*/
#define CDUNIT(dev)	((minor(dev) >> 4) & 0x3)	/* System's unit no */
#define CDMODE(dev)	((minor(dev) >> 6) & 0x3)	/* Sector mode */
#define CDMIN_PART(dev)	(minor(dev) & 0xf)		/* Partition # */

caddr_t real_buf_addr();

struct xha_device *cdcinfo[NCDC];
struct xha_slave *cdinfo[NCD];
struct buf rcdbuf[NCD];

#ifdef DEBUG
int     cddebug		= 0;
int	cdtrace		= 0;
int     cdustartdebug	= 0;
int     cdstartdebug	= 0;
int     cdintdebug	= 0;
int     cderrdebug	= 0;
int	cdstrategydebug = 0;
int	cdminphysdebug  = 0;
int	cdpartial	= 0;
int	cdgetpartdebug  = 0;
int	cdioctldebug    = 0;
int	cdopendebug     = 0;
#endif /* DEBUG */
int	cddisksort 	= 1;	/* Use disksort */
int	cddolinks	= 1;	/* Use SCSI linked commands */


struct xha_driver cdcdriver =
{
	cdprobe, cdslave, 
	(int (*)()) cdattach, 
	cdustart,
	cdstart,
	cdint,
	"cdc","cd",cdcinfo,cdinfo,
	cdttl,
	EARLY_INT,
};

struct scsi_msiobuf cdmsiobuf;
struct	scsi_ms cdmsbuf;

cdprobe(caddr_t reg,struct xha_device *xi)
{
    return xxprobe(xi, &cdcdriver, SCPDT_RODA);
}
char *cd_responses[] =
{
	"SCSI-1",
	"CCS",
	"SCSI-2",
	"","","","","",
	"","","","","","","",""
};

cdattach(struct xha_slave *xs)
{
    struct scsi_inqblk 	iqbuf;
    struct scsi_inqblk 	*iq = &iqbuf;
    struct cd_softc	*sc = &cd_softc[xs->xs_unit];
    static char		vid[sizeof(iq->sinq_vid)+1];
    static char		pid[sizeof(iq->sinq_pid)+1];
    static char		fwrev[sizeof(iq->sinq_fwrev)+1];
    static char		fwdate[sizeof(iq->sinq_fwdate)+1];
    char		*typep;
    int			type;
    struct scsi_ms	*msp = &cdmsbuf;
    unsigned		lblen;

    DEBUGF(cdtrace,printf("Cdattach: xs=%x\n",xs););

    if(xs->xs_unit > NCD)
    {
	printf("cd%d: Not configured\n",xs->xs_unit);
    }

    if(xxinquiry(xs,iq) == 0)
    {

	cdidcpy(iq->sinq_vid,vid,sizeof vid );
	cdidcpy(iq->sinq_pid,pid,sizeof pid );
	cdidcpy(iq->sinq_fwrev,fwrev,sizeof fwrev );
	cdidcpy(iq->sinq_fwdate,fwdate,sizeof fwdate );

	type = cdgettype(xs,&typep,pid);

	if(type < 0)
	{
	    type = 0;
	    printf("%s%d: Unknown cd-rom type: %s using %s\n", 
		   xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,typep,   			cdst[type].typename);  			 
	}

	xs->xs_xi->xi_type = iq->sinq_pdt;
	/*
	printf("%s%d: cd-rom type %s name=%s\n",
	    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,
	    cdst[xs->xs_xi->xi_type].typename, typep); 
        */
	/* 
	 * Fill in default values for disk info. 
	 */
	bcopy(&(cdst[xs->xs_xi->xi_type]),&(sc->sc_cdst),sizeof(struct cdst));
	/* 
	 * Clear size structure.
	 */
	bzero(&(sc->sc_size),sizeof(struct size));
		    
	printf("%s%d:%s %s Firmware Rev. %s (%s) ANSI V.%d(%s)\n",
	    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,
	    vid,pid,fwrev,fwdate,iq->sinq_ansiversion,cd_responses[iq->sinq_rdd]);

	if(cdmodesense(xs,&cdmsiobuf) == 0)
	{
	    bzero(msp,sizeof (*msp));

	    xxfillmspages(cdmsiobuf.msio,msp);

	    strncpy(sc->sc_cdst.inqname,pid, sizeof sc->sc_cdst.inqname);

	    /* Get true no of sectors (redetermined later) */
	    sc->sc_cdst.nsect = msp->sms_daf.daf_spt;

	    /* Only ever one head on a cd-rom */
	    sc->sc_cdst.ntrak = 1;

	    /* We always look like one huge track */
	    sc->sc_cdst.ncyl  = 1;

#ifdef	DEAD_CODE
	    if(cdreadcapacity(xs,&sc->sc_cdata) == 0)
	    {
		DEBUGF(cdtrace,printf("Cdslave: capacity read blklen=%d, nblocks=%d\n",
		    sc->sc_cdata.cd_blklen,sc->sc_cdata.cd_blocks););
		sc->sc_cdst.nbps = sc->sc_cdata.cd_blklen;
		sc->sc_size.partition[2].nblocks = sc->sc_cdata.cd_blocks;
	    }
	    else
	    {
		DEBUGF(cdtrace,printf("Cdslave: capacity read failed\n"););
		    sc->sc_cdst.nbps  = msp->sms_daf.daf_nbps;
		sc->sc_size.partition[2].nblocks = 
		sc->sc_cdst.ncyl * sc->sc_cdst.nspc;
	    }
#else
	    sc->sc_cdst.nbps = msp->sms_daf.daf_nbps;
	    sc->sc_cdst.nspc = 0;
	    sc->sc_size.partition[2].nblocks = 0;

#endif
	    lblen = msp->sms_hd.sm_bdarr[0].sbd_blklen;
	    if(lblen && sc->sc_cdst.nbps && (sc->sc_cdst.nbps != lblen))
	    {
		printf("%s%d: WARNING! logical block length and sector length differs\n!!", 
	    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit);
		if(((sc->sc_cdst.nbps % 256) != 0) && (lblen % 256 == 0))
		{
		    sc->sc_cdst.nbps = lblen;
		    printf("    block length set to %d\n",lblen);
		}
		else 
		{
			if((sc->sc_cdst.nbps % 256 == 0) && (lblen % 256 != 0))
			{
			    printf("    block length set to %d\n",sc->sc_cdst.nbps);
			}
		}
	    }
		
	    sc->sc_size.partition[2].pblkoff = 0;
	    sc->sc_cinited = 1;

#ifdef	CD_PARAM_PRINT
	    printf("nsect=%d, ntrak=%d, nspc=%d, ncyl=%d, nbps=%d, csize=%d",
		   sc->sc_cdst.nsect, sc->sc_cdst.ntrak, sc->sc_cdst.nspc,
		   sc->sc_cdst.ncyl, sc->sc_cdst.nbps,
		   sc->sc_size.partition[2].nblocks);
	    if(msp)
		printf(", tpz=%d, altspz=%d, alttpv=%d",
		       msp->sms_daf.daf_tpz, msp->sms_daf.daf_altspz,
		       msp->sms_daf.daf_alttpv);
	    if(strcmp(vid,"CDC") == 0)
		printf("\nWren Caching %sabled : \n",
		       msp->sms_csh.csh_ce ? "en" : "dis" );
#endif
	}
	else
	{
	    printf("%s%d: Mode sense failed\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit);
	    return 0;
	}
    }
    else
    {
	printf("%s%d: Inquiry failed\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit);
	return 0;
    }
    if (xs->xs_dk >= 0) {
	dk_wpms[xs->xs_dk] = 1000;  /* XXX arbitrary words per second */
#ifdef	CD_PARAM_PRINT
	printf(", dk_wpms=%d\n", dk_wpms[xs->xs_dk]);
#endif
    } else {
#ifdef	CD_PARAM_PRINT
	printf(", dk_wpms=0\n");
#endif
   }
    return 1;
}


/* Tell controller routine expected max lifetime of */
/* a command (time in seconds)			    */
cdttl(union scsi_cdb *com)
{
	DEBUGF(cdtrace,printf("Cdtl: com=%x\n",com););
	switch(com->cdb_6.sc_op)
	{
	    case SC0_MODE_SENSE:
	    case SC0_MODE_SELECT:
	    case SC1_READ_CAPACITY:
	    case SC2_PLAY_AUDIO_MSF:
	    case SC2_PLAY_AUDIO:
	    case SC2_READ_SUBCHANNEL:
	    case SC2_READ_TOC:
	    case SC0_START_STOP_UNIT:
	    case SC2_PLAY_TRACK_INDEX:
			return 10;
	    case SC2_PAUSE_RESUME_UNIT:
	    default:
			return 1;
	}
}

cdmodesense(struct xha_slave *xs, struct scsi_msiobuf *msp)
{
	register union scsi_cdb *scb;

	DEBUGF(cdtrace, printf("Cdmodesense: xs=%x, msp=%x\n", xs,msp););
	scb = xxgetscbuf(xs);
	scb->cdb_6.sc_lbarest[0] = SCMS_CURRENT | SCMS_ALL;
	return xxdoscsi(xs, scb, SC0_MODE_SENSE,(caddr_t) msp,
		sizeof *msp,B_READ);
}

cdmodeselect(struct xha_slave *xs, struct scsi_modesense * mscp)
{
        static union scsi_cdb *scb;

	DEBUGF(cdtrace, printf("Cdmodeselect: xs=%x, msp=%x\n", xs,mscp););
	scb = xxgetscbuf(xs);
        /* scb->cdb_6.sc_lbaMSB = 0x1 ; /* turn on SMP Save Mode Page bit */
	return xxdoscsi(xs,scb,SC0_MODE_SELECT,(caddr_t) mscp,
			sizeof *mscp,B_WRITE);
}

cdstartstop(struct xha_slave *xs,short start)
{
	return xxscsicmd(xs,SC0_START_STOP_UNIT,0,start,B_READ);
}

cdpauseresume(struct xha_slave *xs,short pause)
{
	return xxscsicmd(xs,SC2_PAUSE_RESUME_UNIT,0,pause,B_READ);
}

cd_get_mode(struct xha_slave *xs, struct cd_mode_data *msp, int page)
{
	register union scsi_cdb *scb;
	register int ret;

	DEBUGF(cdtrace, printf("cd_get_mode: xs=%x, msp=%x, size=%d\n",
		xs, msp, sizeof(*msp)););
	scb = xxgetscbuf(xs);
	scb->cdb_6.sc_lbarest[0] = page;
	ret = xxdoscsi(xs, scb, SC0_MODE_SENSE,(caddr_t) msp,
		sizeof(*msp),B_READ);
	DEBUGF(cdtrace, printf("cd_get_mode: xxdoscsi=%d\n", ret););
	return(ret);
}

cd_set_mode(struct xha_slave *xs, struct cd_mode_data *mscp)
{
        register union scsi_cdb *scb;
	register int ret;

	DEBUGF(cdtrace, printf("cd_set_mode: xs=%x, msp=%x, size=%d\n", 
		xs, mscp, sizeof(*mscp)););
	scb = xxgetscbuf(xs);
        scb->cdb_6.sc_lbaMSB = 0x10; /* turn on PF Page Format bit (SCSI-2) */
	mscp->header.data_length = 0;
	ret = xxdoscsi(xs,scb,SC0_MODE_SELECT,(caddr_t) mscp,
			sizeof(*mscp),B_WRITE);
	DEBUGF(cdtrace, printf("cd_set_mode: xxdoscsi=%d\n", ret););
	return(ret);
}

cd_play_tracks(struct xha_slave *xs,int strack,int sindex,int etrack,int eindex)
{
	register union scsi_cdb *scb;

	DEBUGF(cdtrace, printf("cd_play_tracks: xs=%x\n", xs););
	scb = xxgetscbuf(xs);
        scb->cdb_10.sc_lba[2] = strack;
        scb->cdb_10.sc_lba[3] = sindex;
        scb->cdb_10.sc_xflen[0] = etrack;
        scb->cdb_10.sc_xflen[1] = eindex;
	return(cddoscsi(xs,scb,SC2_PLAY_TRACK_INDEX,0,0,B_READ));
}

cd_play_msf(struct xha_slave *xs, int smin, int ssec, int sframe,
	    int emin, int esec, int eframe)
{
	register union scsi_cdb *scb;

	DEBUGF(cdtrace, printf("cd_play_msf: xs=%x\n", xs););
	scb = xxgetscbuf(xs);
        scb->cdb_10.sc_lba[1] = smin;
        scb->cdb_10.sc_lba[2] = ssec;
        scb->cdb_10.sc_lba[3] = sframe;
        scb->cdb_10.sc_resvd2 = emin;
        scb->cdb_10.sc_xflen[0] = esec;
        scb->cdb_10.sc_xflen[1] = eframe;
	return(cddoscsi(xs,scb,SC2_PLAY_AUDIO_MSF,0,0,B_READ));
}

cd_play(struct xha_slave *xs, int blk, int len)
{
	register union scsi_cdb *scb;

	DEBUGF(cdtrace, printf("cd_play: xs=%x blk=%d len=%d\n", xs,blk,len););
	scb = xxgetscbuf(xs);
	scb->cdb_10.sc_lba[0] = (blk >> 24) & 0xff;
	scb->cdb_10.sc_lba[1] = (blk >> 16) & 0xff;
	scb->cdb_10.sc_lba[2] = (blk >> 8) & 0xff;
	scb->cdb_10.sc_lba[3] = blk & 0xff;
	return(xxdoscsi(xs,scb,SC2_PLAY_AUDIO,0,len,B_READ));
}

cd_read_subchannel(struct xha_slave *xs, int mode, int format, int track,
		   struct cd_sub_channel_info *data, int len)
{
	register union scsi_cdb *scb;

	DEBUGF(cdtrace, printf("cd_read_subchannel: xs=%x mode=%d format=%d track=%d data=%x, len=%d\n", xs, mode, format, track, data, len););
	scb = xxgetscbuf(xs);

        if (mode == CD_MSF_FORMAT || mode == CDROM_MSF)
		scb->cdb_10.sc_resvd1 = 1;	/* msf on */
	scb->cdb_10.sc_lba[0]=0x40;		/* SubQ on */
	scb->cdb_10.sc_lba[1]=format;
	scb->cdb_10.sc_resvd2=track;
	return(xxdoscsi(xs,scb,SC2_READ_SUBCHANNEL,(caddr_t)data,len,B_READ));
}

cd_read_toc(struct xha_slave *xs, int mode, int start,
	    struct cd_toc_entry *data, int len)
{
	register union scsi_cdb *scb;

	DEBUGF(cdtrace, printf("cd_read_toc: xs=%x mode=%d start=%d data=%x, len=%d\n", xs, mode, start, data, len););
	scb = xxgetscbuf(xs);

        if (mode == CD_MSF_FORMAT || mode == CDROM_MSF)
		scb->cdb_10.sc_resvd1 = 1;	/* msf on */
	scb->cdb_10.sc_resvd2=start;		/* starting track */
	return(xxdoscsi(xs,scb,SC2_READ_TOC,(caddr_t)data,len,B_READ));
}

#ifdef	TOSHIBA
cd_toshiba_read_disc_info(struct xha_slave *xs, int type, int track,
	    caddr_t data, int len)
{
	register union scsi_cdb *scb;

	DEBUGF(cdtrace, printf("cd_toshiba_read_disc_info: xs=%x type=%d track=%d data=%x, len=%d\n", xs, type, track, data, len););
	scb = xxgetscbuf(xs);

	scb->cdb_10.sc_resvd1 = (type>>1)&0x01;	/* type of info to return */
	scb->cdb_10.sc_reladr = type&0x01;

	scb->cdb_10.sc_lba[0]=track;		/* track to examine */
	return(xxdoscsi(xs,scb,SC2_TOSHIBA_READ_DISC_INFORMATION,data,len,B_READ));
}
#endif

cddoscsi(struct xha_slave *xs, union scsi_cdb *scb, int op, caddr_t addr,
         long length, int read)
{
    /*
     * Do a SCSI command with a provided and eventually prefilled CDB;
     */
    DEBUGF(cdtrace,
        printf("Cddoscsi: xs=%x, scb=%x, cmd=%d, addr=%x, length=%d, read=%d\n",
            xs,scb,op,addr,length,read););
    cdfillscbuf(scb, xs->xs_lun, op);
    if(((*xs->xs_xi->xi_hadriver->xhc_doscsi) (xs, scb, addr, length, read)) ==
         ((XHI_CMPL << 8) + SCTS_GOOD))
    {
        return(0);
    }
    return(EIO);
}

#define SCSICMDGRP(op) (op>>5&7)

cdfillscbuf(union scsi_cdb *scp,int lun,int op)
{

    switch(SCSICMDGRP(op))
    {
    case 0:
        scp->cdb_6.sc_op = op;
        scp->cdb_6.sc_lun = lun;
        break;
    case 1:
    case 2:
    case 6:	/* Toshiba specific command group */
        scp->cdb_10.sc_op = op;
        scp->cdb_10.sc_lun = lun;
        break;
    case 5:
        scp->cdb_12.sc_op = op;
        scp->cdb_12.sc_lun = lun;
        break;
    default:
        printf("Unsupported scsi command group %d, op=%x\n", SCSICMDGRP(op),op);
/*      panic("SCSI op code"); */
        break;
    }
}

cdreadcapacity(struct xha_slave *xs, struct scsi_capdata *cd)
{
    register union scsi_cdb *scb;
    DEBUGF(cdtrace, printf("Cdreadcapacity: xs=%x, cd=%x\n", xs,cd););
    scb = xxgetscbuf(xs);
    xxfillscbuf(scb,xs->xs_lun,SC1_READ_CAPACITY,0);
    if(((*xs->xs_xi->xi_hadriver->xhc_doscsi) (xs, scb, (caddr_t) cd,
	8, B_READ)) == ((XHI_CMPL << 8) + SCTS_GOOD))
    {
	return 0;
    }
    return -1;
}

cdslave(struct xha_slave *xs,caddr_t addr)
{
			/* See if something is there */
    return(xxslave(xs,&cdcdriver,SCPDT_RODA));
}

cdgettype(struct xha_slave *xs, char **typep, char *inqname)
{
    struct cdst *st;

    DEBUGF(cdtrace,
	    printf("Cdgettype: xs=%x, typep=%x, inqname=%s\n", xs,typep,inqname););
    *typep = inqname;

    for (st = cdst; st < cdst + (sizeof cdst/sizeof (struct cdst)); st++)
    {
	if(strncmp(st->inqname,inqname,sizeof (st->inqname)) == 0)
	{
	    return (st - cdst);
	}
    }
    return -1;
}

cdsize(dev_t dev)
{
	register int			un = CDUNIT(dev);
	register int			pa = CDMIN_PART(dev);
	register struct xha_slave	*xs;
	struct cd_softc 		*sc;

	DEBUGF(cdtrace,
	    printf("Cdsize: dev=%x\n", dev););
	if (un >= NCD || pa < 0 || pa >= MAXNPART ||
	    (xs = cdinfo[un]) == 0 || xs->xs_alive == 0)
		return(-1);
        sc = &cd_softc[xs->xs_unit];
	return(sc->sc_size.partition[pa].nblocks);
}


cdopen(dev_t dev,int flag)
{

	struct xha_slave *xs;
        register int cdunit = CDUNIT(dev);
        struct cd_softc *sc;
	int mode1_sectors(struct xha_slave *);
	int mode2form1_sectors(struct xha_slave *);
	int mode2form2_sectors(struct xha_slave *);
	DEBUGF(cdtrace,
	    printf("Cdopen: dev=%x, flag=%x\n", dev, flag););

        if (cdunit >= NCD || (xs = cdinfo[cdunit]) == 0 || xs->xs_alive == 0)
                return (ENXIO);

	if(flag & O_EXCL) {
	    if(xs->xs_ocnt > 0) {
		return(EBUSY);
	    }
	    xs->xs_flags |= XSFL_EXOPEN;
	}
	xs->xs_ocnt++;

        sc = &cd_softc[xs->xs_unit];
	if(sc->sc_spundown && sc->sc_autospinup) {
	    cdstartstop(xs,1);
	}

#ifdef	TOSHIBA
				/* get info about multisession discs */
	cd_toshiba_read_disc_info(xs, 0x03, 0, (caddr_t)&toshiba_disc_id,
		sizeof(toshiba_disc_id));

	DEBUGF(cdtrace, printf("cdopen: Toshiba info: disc_id=0x%02x min=0x%02x sec=0x%02x frame=0x%02x\n", toshiba_disc_id.disc_id, toshiba_disc_id.minutes, toshiba_disc_id.seconds, toshiba_disc_id.frames););
#endif

	switch (CDMODE(dev)) {
		case 0:			/* mode 1 sectors (2048 bytes) */
			if (mode1_sectors(xs))
			    return (ENXIO);	/* error? */
			break;
		case 1:			/* mode 2 form 1 sectors (2048 bytes) */
			if (mode2form1_sectors(xs))
			    return (ENXIO);	/* error? */
			break;
		case 2:			/* mode 2 form 2 sectors (2324 bytes) */
			if (mode2form2_sectors(xs))
			    return (ENXIO);
			break;
		case 3:			/* use drive settings */
		default:
			break;
	}
	if(cdreadcapacity(xs,&sc->sc_cdata) == 0) {
	    DEBUGF(cdopendebug,
		printf("Cdopen: capacity read blklen=%d, nblocks=%d\n",
		sc->sc_cdata.cd_blklen,sc->sc_cdata.cd_blocks););
	    sc->sc_cdst.nbps = sc->sc_cdata.cd_blklen;
	    sc->sc_cdst.nspc = 100;
	    sc->sc_cdst.ncyl = (sc->sc_cdata.cd_blocks/100)+1;
	    sc->sc_size.partition[0].nblocks = 
	    sc->sc_size.partition[2].nblocks = sc->sc_cdata.cd_blocks;
	} else {
	    DEBUGF(cdopendebug,
		printf("Cdopen: capacity read failed\n"););
	    sc->sc_cdst.nbps = 0;
	    sc->sc_cdst.nspc = 0;
	    sc->sc_cdst.ncyl = 0;
	    sc->sc_size.partition[0].nblocks =
	    sc->sc_size.partition[2].nblocks = 0;
	    /*
	    if (!minor(dev)&0x10)
                return(ENXIO);
	    */
	}
        return(0);
}

cdclose(dev_t dev,int flag)
{
	struct xha_slave *xs;
        register int cdunit = CDUNIT(dev);

        if (cdunit >= NCD || (xs = cdinfo[cdunit]) == 0 || xs->xs_alive == 0)
                return (ENXIO);
	DEBUGF(cdtrace,
	    printf("Cdclose: dev=%x, flag=%x\n", dev, flag););

	xs->xs_ocnt=0;
	xs->xs_flags &= ~(XSFL_SPECSCSI | XSFL_EXOPEN);
}

cdstrategy(struct buf *bp)
{
    register struct xha_slave *xs;
    int pa = CDMIN_PART(bp->b_dev);
    register int cdunit = CDUNIT(bp->b_dev);
#ifdef DEBUG
    long sz;
#endif /* DEBUG */

    DEBUGF(cdtrace,
	printf("Cdstrategy: bp=%x\n", bp););

    DEBUGF(cdstrategydebug, sz=(bp->b_bcount+cd_block_size-1)/cd_block_size; );

    if (cdunit >= NCD) 
    {
	bp->b_error = ENXIO;
	DEBUGF(cdstrategydebug,
	    printf("Cdstrategy(BAD): bp=0x%x, sz=%d, blkno=%d\n",
		bp,sz,bp->b_blkno););
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
    }

    xs = cdinfo[cdunit];
    if (xs == 0 || xs->xs_alive == 0) 
    {
	bp->b_error = ENXIO;
	DEBUGF(cdstrategydebug,
	    printf("Cdstrategy(BAD): bp=0x%x, sz=%d, blkno=%d\n",
		bp,sz,bp->b_blkno););
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
    }

    cdxstrat(xs, pa, bp);
}

cdxstrat(struct xha_slave *xs,int pa, register struct buf *bp)
{
    register struct cdst *st;
    register struct cd_softc *sc;
    long bn, sz;
    int s;
    register struct xhac_driver *xc;
    struct buf *dp;

#ifdef	BLOCK_SIZE_2048
		/* XXX  >>11 is same as /CD_BSIZE but faster (and inflexible)*/
    sz = (bp->b_bcount+CD_BSIZE-1) >> 11;  
#else
    sz = (bp->b_bcount+cd_block_size-1) / cd_block_size;  
#endif

    xc  = xs->xs_xi->xi_hadriver;
    sc = &cd_softc[xs->xs_unit];
    st = &(sc->sc_cdst);
    if(sc->sc_spundown && sc->sc_autospinup)
    {
	cdstartstop(xs,1);
    }
    DEBUGF(cdstrategydebug,printf("Cdstrategy: bp=0x%x, sz=%d, blkno=%d, xs=%x\n",
	    bp,sz,bp->b_blkno,xs););
    bn = dkblock(bp);

#ifdef	TOSHIBA
		/* multisession disc support replaces iso 9660 blk number 16 */
    if ((toshiba_disc_id.disc_id == CD_XA_TYPE) && bn == 16) {
	int new_iso_blk=(bcd2i(toshiba_disc_id.minutes)*60 +
			 bcd2i(toshiba_disc_id.seconds))*75 +
			 bcd2i(toshiba_disc_id.frames);
	DEBUGF(cdtrace, printf("cdxstrat: trying to read block 16 of a multisession disc, new block=%d\n", new_iso_blk););
	if (new_iso_blk >= 150) {	/* check to see if valid */
	    new_iso_blk-=150;		/* offset to start of data */
	    bn=new_iso_blk+bn;		/* relocate it */
	    bp->b_blkno=(cd_block_size/DEV_BSIZE)*bn;
	}
    }
#endif

    if ((bn < 0) || ((sc->sc_nomap == 0) && 
		     (bn+sz > sc->sc_size.partition[pa].nblocks)))
    {
	if (bn == sc->sc_size.partition[pa].nblocks)
	{
	    bp->b_resid = bp->b_bcount;
	    iodone(bp);
	    return;
	}
	bp->b_error = EINVAL;
	DEBUGF(cdstrategydebug,
	    printf("Cdstrategy(BAD): bp=0x%x, sz=%d, blkno=%d\n",
		bp,sz,bp->b_blkno););
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
    }

    s = dsplbio(50);
#ifdef PARANOID
    {
	int loopcount;
	for(dp=&xs->xs_tab,loopcount=0;dp;dp = dp->b_actf)
	{
	    if(++loopcount >= 100000)
	    {
		printf("%s%d:Cdstrategy: request list loop detected\n",
		    xs->xs_xi->xi_driver->xd_sname, xs->xs_unit);

		printf("xs_tab=%x, tab.actf=%x, dp=%x dp->b_actf=%x\n",
		    &xs->xs_tab,xs->xs_tab.b_actf,dp,dp->b_actf);
		break;
	    }
	    if(dp == bp)
	    {
		printf("%s%d:Cdstrategy: bp=%x already in Q\n",
		    xs->xs_xi->xi_driver->xd_sname, xs->xs_unit,bp);
		dsplx(s,51);
		(*xc->xhc_cstart)(xs);
		return;
	    }
	}
    }
#endif /* PARANOID */

    if(!cddisksort)
    {
	bp->av_forw = NULL;
	dp = &xs->xs_tab;
	if (dp->b_actf == NULL)
	    dp->b_actf = bp;
	else
	    dp->b_actl->av_forw = bp;
	dp->b_actl = bp;
	bp->b_errcnt = 0;
    }
    else
    {
	bp->b_cylin = bn;

	if(sc->sc_nomap == 0)
		bp->b_cylin += sc->sc_size.partition[pa].pblkoff;

/*
 * We really doesn't have to sort by cylinders since we can sort by block
 * numbers. If you really want to sort by cylinder numbers. define CYLSORT
 */
#ifdef CYLSORT
	if(st->nspc)
		bp->b_cylin /= st->nspc;
#endif /* CYLSORT */

	bp->b_errcnt = 0;
	disksort(&xs->xs_tab, bp);
    }
    dsplx(s,51);
    if(xs->xs_tab.b_active == 0)
	    (*xc->xhc_cstart)(xs);
}

cdustart(struct xha_slave *xs)
{
    struct cd_softc *sc = &cd_softc[xs->xs_unit];

    DEBUGF(cdustartdebug | cdtrace ,printf("Cdustart: xs=0x%x\n",xs););

    if(cddolinks && (sc->sc_cdst.flags & CDFL_LINK) )
	return XHU_DOLINKS;
    else
	return XHU_DODATA;
}

void
cdpartsetup(register struct xha_slave *xs, register struct buf *bp, 
	    int	bn,int sz)
{
	register struct partsec *ps;
	int	s;
	struct	cd_softc *sc = &cd_softc[xs->xs_unit];

	DEBUGF(cdpartial | cdtrace,
		printf("Cdstart(Partial): xs=0x%x, bn=%d, sz=%d, bcount=%d\n",
		    xs,bn,sz,bp->b_bcount););
	s = dsplbio(52);
	ps = cdgetps(xs,NULL); /* NULL will get us a free entry */
	if(ps == NULL)
	    panic("Cdstart: partial sector init failed");
	
	/* Save interesting facts in buf structure */
	ps->ps_bp = bp;
	ps->ps_addr = bp->b_un.b_addr;
	ps->ps_bcount = bp->b_bcount;
	ps->ps_phys = (bp->b_flags & B_PHYS) ? 1 : 0;
	ps->ps_nbps = sc->sc_cdst.nbps;
	dsplx(s,53);
	/* Now let's modify the buf struct a bit */
	bp->b_un.b_addr = ps->ps_space;	/* link i/o from buffer to this space */
	bp->b_bcount = sc->sc_cdst.nbps;
	bp->b_flags &= ~B_PHYS;
	
	if(!(bp->b_flags & B_READ))
	    cdfillpart(ps);
}

union scsi_cdb *
cdstart(struct xha_slave *xs, struct buf *bp, union scsi_cdb *combuf)
{
    register union scsi_cdb *scbuf;
    register int bn;
    register int sz;
    register struct cd_softc *sc = &cd_softc[xs->xs_unit];
    dev_t 	dev;

    DEBUGF(cdtrace,
		printf("Cdstart: xs=%x,bp=%x,combuf=%x\n",xs,bp,combuf););
    dev = bp->b_dev;
    bn = dkblock(bp);
    if(!sc->sc_nomap)
    {
	bn += sc->sc_size.partition[CDMIN_PART(dev)].pblkoff;
    }
    sz = (bp->b_bcount + (sc->sc_cdst.nbps-1)) / sc->sc_cdst.nbps;	/* Compute no of blks */
    if(sz == 0)
    {
	/* This should never happen, but if it does, and the command is 
	   a write, really bad things might occur, since a transfer length
	   of 0 means that 256 blocks will be transferred */
	printf("Cdstart: bp=0x%x, b_flags=0x%x, b_forw=0x%x, b_back=0x%x\n",
		bp,bp->b_flags,bp->b_forw,bp->b_back);
	printf("         av_forw=0x%x, av_back=0x%x, b_bcount=%d\n",
		bp->av_forw,bp->av_back,bp->b_bcount);
	printf("         b_bufsize=0x%x,b_error=0x%x,b_dev=0x%x\n",
		bp->b_bufsize,bp->b_error,bp->b_dev);
	printf("         b_addr=0x%x,b_blkno=%d,b_resid=%d,b_proc=0x%x\n",
		bp->b_un.b_addr,bp->b_blkno,bp->b_resid,bp->b_proc);
	/* printf("         b_iodone=0x%x,b_pfcent=%d,b_bitmap=0x%x\n",
		bp->b_iodone,bp->b_pfcent,bp->b_bitmap); */
	printf("         b_iodone=0x%x,b_pfcent=%d\n",
		bp->b_iodone,bp->b_pfcent);
    }

    DEBUGF(cdstartdebug,printf("Cdstart: xs=0x%x, bn=%d, sz=%d\n",xs,bn,sz););
    if(combuf)
	scbuf =  combuf;
    else
	scbuf = xxgetscbuf(xs);

    if(xs->xs_flags & XSFL_SPECSCSI)
    {
	scbuf = &xs->xs_cdb;
	scbuf->cdb_6.sc_lun = xs->xs_lun;
	return scbuf;
    }

    /*
     * Code here to handle partial sector transfers 
     */

    if(bp->b_bcount < sc->sc_cdst.nbps)
    {
	cdpartsetup(xs,bp,bn,sz);
    }

    if((bn < 0x1FFFFF) && (sz != 0) && (sz < 256) ) 
    {

	xxfillscbuf(scbuf,xs->xs_lun,bp->b_flags & B_READ ? SC0_READ:SC0_WRITE,sz);
	scbuf->cdb_6.sc_lbaMSB     = bn >> 16 & 0x1F;
	scbuf->cdb_6.sc_lbarest[0] = bn >> 8  & 0xFF;
	scbuf->cdb_6.sc_lbarest[1] = bn       & 0xFF;
	DEBUGF(cdstartdebug,printf("Cdstart: Return SC0: bn=%d, sz=%d\n",bn,sz););
    }
    else
    {
	xxfillscbuf(scbuf,xs->xs_lun,bp->b_flags & B_READ ? SC1_READ:SC1_WRITE,sz);
	scbuf->cdb_10.sc_lba[0]    = bn >> 24 & 0xFF;
	scbuf->cdb_10.sc_lba[1]    = bn >> 16 & 0xFF;
	scbuf->cdb_10.sc_lba[2]    = bn >>  8 & 0xFF;
	scbuf->cdb_10.sc_lba[3]    = bn       & 0xFF;
	DEBUGF(cdstartdebug,printf("Cdstart: Return SC1: bn=%d, bno=%d\n",bn,sz););
    }
    if (xs->xs_dk >= 0) 
    {
	dk_busy |= 1<<xs->xs_dk;
	dk_seek[xs->xs_dk] ++;
	dk_xfer[xs->xs_dk]++;
	dk_wds[xs->xs_dk] += bp->b_bcount >> 6; 
    }

    return (scbuf);
}

cdfillpart(struct partsec *ps)
{
    register int	fillcount = ps->ps_bcount % ps->ps_nbps;
    register caddr_t	fillfrom  = ps->ps_addr + ps->ps_bcount - fillcount;
    register caddr_t        from, from2;
    register int            s, v;
    register u_int          n;

    DEBUGF(cdtrace | cdpartial,
	printf("Cdfillpart: filling %d bytes\n",ps->ps_bcount););

    if(!ps->ps_phys)
    {
	DEBUGF(cdpartial,
	    printf("bcopy(%x,%x,%d) (COOKED)\n",
	        fillfrom,ps->ps_space,fillcount););
	bcopy(fillfrom,ps->ps_space,fillcount);
    }
    else
    {
	ps->ps_bp->b_flags |= B_PHYS;
	from = real_buf_addr(ps->ps_bp, fillfrom);
	n = PGOFSET + 1 - ((u_int)from & PGOFSET);
	if (n >= fillcount) 
	{
	    DEBUGF(cdpartial,
		printf("bcopy(%x,%x,%d) (RAW1)\n",
		    from,ps->ps_space,fillcount););
	    s = dsplhigh(54);
	    GET_VR0(v);
	    bcopy(from, ps->ps_space, fillcount);
	} 
	else 
	{
	    from2 = real_buf_addr(ps->ps_bp, fillfrom + n);
	    DEBUGF(cdpartial,
		printf("bcopy(%x,%x,%d) (RAW2)\n",
		    from,ps->ps_space,n););
	    DEBUGF(cdpartial,
		printf("bcopy(%x,%x,%d) (RAW2)\n",
		    from2,ps->ps_space + n, fillcount - n););
	    s = dsplhigh(55);
	    GET_VR0(v);
	    bcopy(from, ps->ps_space, n);
	    bcopy(from2, ps->ps_space + n, fillcount - n);
	}
	SET_VR(v);
	dsplx(s,56);
	ps->ps_bp->b_flags &= ~B_PHYS;
    }
    DEBUGF(cdtrace | cdpartial,
	  printf("bzero(%x,%d)\n",
	  ps->ps_space + fillcount, ps->ps_nbps - fillcount););
    bzero(ps->ps_space + fillcount, ps->ps_nbps - fillcount);
}

cdint(struct xha_slave *xs, struct buf *bp, int ccode, 
      struct scsi_esense *sdp, u_char tarstat)
{
    register struct cd_softc *sc = &cd_softc[xs->xs_unit];
    register int	bn;
    register int	sz;

    DEBUGF(cdintdebug | cdtrace,
	printf("Cdint: xs=0x%x, bp=0x%x, ccode=%d, sdp=0x%x\n",
        xs,bp,ccode,sdp););
    if (xs->xs_dk >= 0)
	dk_busy &= ~(1<<xs->xs_dk);
    if(bp && (bp->b_flags & B_HEAD))
    {
	xs->xs_flags &= ~XSFL_SPECSCSI;
	return XHD_DONE;
    }
    if(tarstat == SCTS_BUSY)
    {
	if(sc->sc_busy)	/* I.e. we were here last time too */
	{
	    if(++sc->sc_busyretries > 5)
	    {
		bp->b_error=EIO;
		bp->b_flags |= B_ERROR;
		sc->sc_busy = 0;	/* Next time we'll find out ... */
		cdpartdone(xs,bp);
		xs->xs_flags &= ~XSFL_SPECSCSI;
		return XHD_DONE;
	    }
	    return XHD_RETRY;
	}
	sc->sc_busy = 1;
	sc->sc_busyretries=1;
	return XHD_RETRY;
    }
    sc->sc_busy = 0;
    switch(ccode)
    {
    case XHI_OUVR:
    case XHI_CMPL:
	if(bp && (bp->b_flags & B_HEAD))
        {
	    xs->xs_flags &= ~XSFL_SPECSCSI;
	    return XHD_DONE;
        }
        bp->b_resid = 0;
	break;
    case XHI_TIMEOUT:
	if(bp && (bp->b_flags & B_HEAD))
	{
	    xs->xs_flags &= ~XSFL_SPECSCSI;
	    return XHD_DONE;
	} 
	return cdtryretry(xs,bp,1,EIO); 
    case XHI_SELTO: 
        /* B_INVAL is used here to disable any retries during autoconfig or any other special SCSI commands */ 
	if(bp && (bp->b_flags & B_HEAD)) 
	{ 
		xs->xs_flags &= ~XSFL_SPECSCSI; 
		return XHD_DONE;
	}
	if(++bp->b_errcnt <= 1)
	    return XHD_RESETRETRY;
	if(bp->b_errcnt <= 2 )
	    return XHD_SCRESETRETRY;
	bp->b_bcount = bp->b_resid;
	bp->b_resid = 0;
	bp->b_error = EIO;;
	bp->b_flags |= B_ERROR;
	cdpartdone(xs,bp);
	xs->xs_flags &= ~XSFL_SPECSCSI;
	return XHD_DONE;
    case XHI_SCSIRESET:
	if(bp && (bp->b_flags & B_HEAD))
	{
	    xs->xs_flags &= ~XSFL_SPECSCSI;
	    return XHD_DONE;
	}
	return XHD_RETRY;
    case XHI_ERR:
        /* Analyze the error */
        if(sdp)
	{
	    bcopy(sdp, &(xs->xs_esense), sizeof(xs->xs_esense));
	    xs->xs_flags |= XSFL_HASSENSE;

            switch (sdp->es_key)
            {
	        case SCES_KEY_RECOVERED_ERROR:
                    DEBUGF(cderrdebug,printf("Cdint: RECOVERED ERROR\n"););
                    break;
                case SCES_KEY_NOT_READY:
                    DEBUGF(cderrdebug,printf("Cdint: NOT READY\n"););
		    printf("%s%d: not ready\n", xs->xs_xi->xi_driver->xd_sname,xs->xs_unit);
		    bp->b_error = EIO;
		    bp->b_flags |= B_ERROR;
		    break;
                case SCES_KEY_MEDIUM_ERROR:
                    DEBUGF(cderrdebug,printf("%s%d: Cdint: MEDIUM ERROR\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		    sz = (bp->b_bcount + (sc->sc_cdst.nbps-1)) / 
			  sc->sc_cdst.nbps;	/* compute no of blks */
		    bn = dkblock(bp) + (sc->sc_nomap ? 0 : 
			(sc->sc_size.partition[CDMIN_PART(bp->b_dev)].pblkoff));
                                      printf("%s%d: Cdint: MEDIUM ERROR abs starting blk=%d, rel=%d, size=%d blocks\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,bn,dkblock(bp),sz);
		    return cdtryretry(xs,bp,5,EIO);
                case SCES_KEY_HARDWARE_ERROR:
                    DEBUGF(cderrdebug,printf("%s%d: Cdint: HARDWARE ERROR\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		    return cdtryretry(xs,bp,5,EIO);
                case SCES_KEY_ILLEGAL_REQUEST:
                    DEBUGF(cderrdebug,printf("%s%d: Cdint: ILLEGAL REQUEST\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		    return cdtryretry(xs,bp,1,EIO);
                case SCES_KEY_UNIT_ATTENTION:
                    DEBUGF(cderrdebug,printf("%s%d: Cdint: UNIT ATTENTION\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		    return cdtryretry(xs,bp,2,EIO);
                case SCES_KEY_ABORTED_COMMAND:
                    DEBUGF(cderrdebug,printf("%s%d: Cdint: ABORTED COMMAND\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		    return cdtryretry(xs,bp,1,EIO);
                default:
		    DEBUGF(cderrdebug,printf("%s%d: Cdint: Unknown KEY=%d\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,
			sdp->es_key););
                    break;
            }
	}
	else
	    printf("Cdint: Sense data pointer is zero\n");
	if(bp && (bp->b_flags & B_HEAD))
	{
	    xs->xs_flags &= ~XSFL_SPECSCSI;
	    return XHD_DONE;
	}
	break;
    }

    /* Make sure the byte count is reasonable.                           */
    if(bp->b_bcount < 0)
	bp->b_bcount = bp->b_resid;

    cdpartdone(xs,bp);
    xs->xs_flags &= ~XSFL_SPECSCSI;
    return XHD_DONE;
}

cdpartdone(struct xha_slave *xs, struct buf *bp)
{
	register struct partsec *ps;

	if(!PARTIAL(bp)) 
		return;
	DEBUGF(cdtrace | cdpartial,
	    printf("Cdpartdone: xs=%x, bp=%x\n",xs,bp););
	ps = cdgetps(xs,bp);
	/* Now restore buf header variables */
	if(ps->ps_phys)
		bp->b_flags |= B_PHYS;
	bp->b_un.b_addr = ps->ps_addr;
	/* If we were reading, copy out the stuff */
 	if(bp->b_flags & B_READ)
		cdemptypart(ps);
	ps->ps_addr = NULL;	
	ps->ps_bcount = 0;
	ps->ps_bp = NULL;	/* Free this entry */
}

cdemptypart(struct partsec *ps)
{
    register int	emptycount = ps->ps_bcount % ps->ps_nbps;
    register caddr_t	emptyto  = ps->ps_addr + ps->ps_bcount - emptycount;
    register caddr_t        to, to2;
    register int            s, v;
    register u_int          n;

    DEBUGF(cdtrace | cdpartial,
	printf("Cdemptypart(%x): copying %d bytes\n",ps,ps->ps_bcount););
    if (!ps->ps_phys) 
    {
	DEBUGF(cdpartial,
	    printf("bcopy(%x,%x,%d) (COOKED)\n",
		ps->ps_space,emptyto,emptycount););
	bcopy(ps->ps_space, emptyto, emptycount);
    }
    else 
    {
	to = real_buf_addr(ps->ps_bp, emptyto);
	n = PGOFSET + 1 - ((u_int)to & PGOFSET);
	if (n >= emptycount) 
	{
	    DEBUGF(cdpartial,
		printf("bcopy(%x,%x,%d) (RAW1)\n",
		    ps->ps_space, to, emptycount););
	    s = dsplhigh(57);
	    GET_VR0(v);
	    bcopy(ps->ps_space, to, emptycount);
	} 
	else 
	{
	    to2 = real_buf_addr(ps->ps_bp, emptyto + n);
	    DEBUGF(cdpartial,
		printf("bcopy(%x,%x,%d) (RAW2)\n", ps->ps_space, to, n););
	    DEBUGF(cdpartial,
		printf("bcopy(%x,%x,%d) (RAW2)\n",
		    ps->ps_space + n, to2, emptycount - n););
	    s = dsplhigh(58);
	    GET_VR0(v);
	    bcopy(ps->ps_space, to, n);
	    bcopy(ps->ps_space + n, to2, emptycount - n);
	}
	SET_VR(v);
	dsplx(s,59);
    }
    DEBUGF(cdtrace | cdpartial,printf("Cdemptypart(%x) DONE\n",ps););
}

#ifdef	BSD_43	/* BSD Reno doesn't need these, done via rawread/rawwrite */

cdread(dev_t dev, struct uio *uio)
{
 int errno;

    DEBUGF(cdtrace,
	printf("Cdread: dev=0x%x,uio=0x%x\n",dev,uio););
    errno = cdphys(dev);
    DEBUGF(cdtrace,
	printf("Cdread: errno=%d\n",errno););
    if (errno)
        return (errno);
    return (physio(cdstrategy, &rcdbuf[CDUNIT(dev)], dev, B_READ, cdminphys, uio))
;
}

cdwrite(dev_t dev, struct uio *uio)
{
    int errno;

    DEBUGF(cdtrace,
    	printf("Cdwrite: dev=0x%x,uio=0x%x\n",dev,uio););
    errno = cdphys(dev);
    DEBUGF(cdtrace,
    	printf("Cdwrite: errno=%d\n",errno););
    if (errno)
        return (errno);
    return (physio(cdstrategy, &rcdbuf[CDUNIT(dev)], dev, B_WRITE, cdminphys, uio)
);
}
#endif

cdphys(dev_t dev)
{
        register int unit;
        struct xha_slave *xs;

        DEBUGF(cdtrace,
		printf("Cdphys: dev=0x%x\n",dev););

        unit = CDUNIT(dev);
	if( unit > NCD)
		return ENXIO;
        if (((xs = cdinfo[unit]) == 0 )
            || (xs->xs_alive == 0) )
                return (ENXIO);
        return (0);
}

cdioctl(dev_t dev, int cmd, caddr_t data, int flag)
{
    register int cdunit = CDUNIT(dev);
    register struct xha_slave *xs;
    register struct cd_softc *sc;
    register struct dkpart *dk = (struct dkpart *) data;
    register struct partentry *pep;
    register int pa;
    union scsi_cdb *scp;
    extern int parse_sun_uscsi(register struct xha_slave *, caddr_t);

					/* defines from hd.c -md */
    register struct disklabel       *dl;

    if(cdunit >= NCD || (xs = cdinfo[cdunit]) == 0 || xs->xs_alive == 0)
    {
	DEBUGF(cdioctldebug,printf("Cdioctl: bad device %d/%d\n",major(dev),minor(dev)););
	return ENODEV;
    }

    DEBUGF(cdioctldebug | cdtrace,
	printf("%s%d: Cdioctl: cmd = %u, data=%x\n",
	xs->xs_xi->xi_driver->xd_sname, xs->xs_unit, cmd, data););
    sc = &cd_softc[xs->xs_unit];

    switch(cmd)
    {
    case CDIOCEJECT:	/* 386bsd */
    case CDROMEJECT:	/* SunOS */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCEJECT entered\n"););
	if(cdstartstop(xs,0x02) < 0)		/* Stop & eject unit */
		return(EIO);
	sc->sc_spundown = 1;
	return(0);

    case CDIOCSTART:	/* 386bsd */
    case CDROMSTART:	/* SunOS */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCSTART entered\n"););
	if(cdstartstop(xs,1) < 0)		/* Start unit */
		return(EIO);
	sc->sc_spundown = 0;
	return(0);

    case CDIOCSTOP:	/* 386bsd */
    case CDROMSTOP:	/* SunOS */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCSTOP entered\n"););
	if(cdstartstop(xs,0) < 0)		/* Stop unit */
		return(EIO);
	sc->sc_spundown = 1;
	return(0);

    case CDIOCPAUSE:	/* 386bsd */
    case CDROMPAUSE:	/* SunOS */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCPAUSE entered\n"););
	if(cdpauseresume(xs,0) < 0)		/* pause unit */
		return(EIO);
	return(0);

    case CDIOCRESUME:	/* 386bsd */
    case CDROMRESUME:	/* SunOS */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCRESUME entered\n"););
	if(cdpauseresume(xs,1) < 0)		/* resume unit */
		return(EIO);
	return(0);

    case CDROMPLAYMSF:	/* SunOS */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDROMPLAYMSF entered\n"););
	{
	struct	cdrom_msf *args = (struct  cdrom_msf *)data;
	struct	cd_mode_data cd_data;

	if(cd_get_mode(xs, &cd_data, AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.sotc = 0;
	cd_data.page.audio.immed = 1;
	if(cd_set_mode(xs, &cd_data))
		return(EIO);
	return(cd_play_msf(xs,
		args->cdmsf_min0, args->cdmsf_sec0, args->cdmsf_frame0,
		args->cdmsf_min1 ,args->cdmsf_sec1, args->cdmsf_frame1));
	}

    case CDROMPLAYTRKIND:	/* SunOS */
    case CDIOCPLAYTRACKS:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCPLAYTRACKS entered\n"););
	{
	struct	ioc_play_track *args = (struct  ioc_play_track *)data;
	struct	cd_mode_data cd_data;

	DEBUGF(cdioctldebug,printf("strack=%d, sindex=%d, etrack=%d, eindex=%d\n", args->start_track ,args->start_index ,args->end_track ,args->end_index););

	if(cd_get_mode(xs, &cd_data, AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.sotc = 0;
	cd_data.page.audio.immed = 1;
	if(cd_set_mode(xs, &cd_data))
		return(EIO);
	return(cd_play_tracks(xs ,args->start_track ,args->start_index,
			      args->end_track ,args->end_index));
	}

    case CDIOCPLAYBLOCKS:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCPLAYBLOCKS entered\n"););
	{
	struct	ioc_play_blocks *args = (struct  ioc_play_blocks *)data;
	struct	cd_mode_data cd_data;

	DEBUGF(cdioctldebug,printf("block=%d, length=%d\n", args->blk,
		args->len););

	if(cd_get_mode(xs, &cd_data, AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.sotc = 0;
	cd_data.page.audio.immed = 1;
	if(cd_set_mode(xs, &cd_data))
		return(EIO);
	return(cd_play(xs, args->blk,args->len));
	}

    case CDROMSUBCHNL:		/* SunOS */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDROMSUBCHNL entered\n"););
	{
	struct cdrom_subchnl *args
			= (struct cdrom_subchnl *)data;
	struct cd_sub_channel_info cd_data;

	if (cd_read_subchannel(xs, args->cdsc_format,
			CD_CURRENT_POSITION,0,&cd_data,sizeof(cd_data)))
		return(EIO);
	args->cdsc_audiostatus=cd_data.header.audio_status;
	args->cdsc_adr=cd_data.what.position.addr_type;
	args->cdsc_ctrl=cd_data.what.position.control;
	args->cdsc_trk=cd_data.what.position.track_number;
	args->cdsc_ind=cd_data.what.position.index_number;
	if (args->cdsc_format == CDROM_MSF) {
	    args->cdsc_absaddr.msf.minute=cd_data.what.position.absaddr[1];
	    args->cdsc_absaddr.msf.second=cd_data.what.position.absaddr[2];
	    args->cdsc_absaddr.msf.frame =cd_data.what.position.absaddr[3];

	    args->cdsc_reladdr.msf.minute=cd_data.what.position.reladdr[1];
	    args->cdsc_reladdr.msf.second=cd_data.what.position.reladdr[2];
	    args->cdsc_reladdr.msf.frame =cd_data.what.position.reladdr[3];
	} else {
	    args->cdsc_absaddr.lba=(cd_data.what.position.absaddr[0]<<24)+
				   (cd_data.what.position.absaddr[1]<<16)+
				   (cd_data.what.position.absaddr[2]<<8)+
				    cd_data.what.position.absaddr[3];
	    args->cdsc_reladdr.lba=(cd_data.what.position.reladdr[0]<<24)+
				   (cd_data.what.position.reladdr[1]<<16)+
				   (cd_data.what.position.reladdr[2]<<8)+
				    cd_data.what.position.reladdr[3];
	}
	args->cdsc_format=cd_data.what.position.data_format;
	return(0);
	}

    case CDIOCREADSUBCHANNEL:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCREADSUBCHANNEL entered\n"););
	{
	struct ioc_read_subchannel *args
			= (struct ioc_read_subchannel *)data;
	struct cd_sub_channel_info cd_data;
	int len=args->data_len;

	if(len>sizeof(cd_data)||
	   len<sizeof(struct cd_sub_channel_header))
		return(EINVAL);
	if(cd_read_subchannel(xs,args->address_format,
			args->data_format,args->track,&cd_data,len))
		return(EIO);
	len=MIN(len,((cd_data.header.data_len[0]<<8)+cd_data.header.data_len[1]+
			sizeof(struct cd_sub_channel_header)));
	if(copyout(&cd_data,args->data,len)!=0)
		return(EFAULT);
	return(0);
	}

    case CDIOCGETVOL:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCGETVOL entered\n"););
	{
	struct ioc_vol *arg = (struct ioc_vol *)data;
	struct cd_mode_data cd_data;

	if(cd_get_mode(xs,&cd_data,AUDIO_PAGE))
		return(EIO);
	arg->vol[LEFT_PORT] = cd_data.page.audio.port[LEFT_PORT].volume;
	arg->vol[RIGHT_PORT] = cd_data.page.audio.port[RIGHT_PORT].volume;
	arg->vol[2] = cd_data.page.audio.port[2].volume;
	arg->vol[3] = cd_data.page.audio.port[3].volume;
	}
	return(0);

    case CDROMVOLCTRL:	/* SunOS */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDROMVOLCTRL entered\n"););
	{
	struct cdrom_volctrl *arg = (struct cdrom_volctrl *)data;
	struct cd_mode_data cd_data;

	if(cd_get_mode(xs,&cd_data,AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.port[LEFT_PORT].volume = arg->channel0;
	cd_data.page.audio.port[RIGHT_PORT].volume = arg->channel1;
	cd_data.page.audio.port[2].volume = arg->channel2;
	cd_data.page.audio.port[3].volume = arg->channel3;
	if(cd_set_mode(xs,&cd_data))
		return(EIO);
	}
	return(0);

    case CDIOCSETVOL:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCSETVOL entered\n"););
	{
	struct ioc_vol *arg = (struct ioc_vol *)data;
	struct cd_mode_data cd_data;

	if(cd_get_mode(xs,&cd_data,AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.port[LEFT_PORT].volume = arg->vol[LEFT_PORT];
	cd_data.page.audio.port[RIGHT_PORT].volume = arg->vol[RIGHT_PORT];
	cd_data.page.audio.port[2].volume = arg->vol[2];
	cd_data.page.audio.port[3].volume = arg->vol[3];
	if(cd_set_mode(xs,&cd_data))
		return(EIO);
	}
	return(0);

    case CDIOCSETMONO:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCSETMONO entered\n"););
	{
	struct ioc_vol *arg = (struct ioc_vol *)data;
	struct cd_mode_data cd_data;

	if(cd_get_mode(xs,&cd_data,AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.port[LEFT_PORT].channels =
		LEFT_CHANNEL|RIGHT_CHANNEL|4|8;
	cd_data.page.audio.port[RIGHT_PORT].channels =
		LEFT_CHANNEL|RIGHT_CHANNEL;
	cd_data.page.audio.port[2].channels = 0;
	cd_data.page.audio.port[3].channels = 0;
	if(cd_set_mode(xs,&cd_data))
		return(EIO);
	}
	return(0);

    case CDIOCSETSTERIO:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCSETSTERIO entered\n"););
	{
	struct ioc_vol *arg = (struct ioc_vol *)data;
	struct cd_mode_data cd_data;

	if(cd_get_mode(xs,&cd_data,AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.port[LEFT_PORT].channels = LEFT_CHANNEL;
	cd_data.page.audio.port[RIGHT_PORT].channels = RIGHT_CHANNEL;
	cd_data.page.audio.port[2].channels = 0;
	cd_data.page.audio.port[3].channels = 0;
	if(cd_set_mode(xs,&cd_data))
		return(EIO);
	}
	return(0);

    case CDIOCSETMUTE:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCSETMUTE entered\n"););
	{
	struct ioc_vol *arg = (struct ioc_vol *)data;
	struct cd_mode_data cd_data;

	if(cd_get_mode(xs,&cd_data,AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.port[LEFT_PORT].channels = 0;
	cd_data.page.audio.port[RIGHT_PORT].channels = 0;
	cd_data.page.audio.port[2].channels = 0;
	cd_data.page.audio.port[3].channels = 0;
	if(cd_set_mode(xs,&cd_data))
		return(EIO);
	}
	return(0);

    case CDIOCSETLEFT:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCSETLEFT entered\n"););
	{
	struct ioc_vol *arg = (struct ioc_vol *)data;
	struct cd_mode_data cd_data;

	if(cd_get_mode(xs,&cd_data,AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.port[LEFT_PORT].channels = 15;
	cd_data.page.audio.port[RIGHT_PORT].channels = 15;
	cd_data.page.audio.port[2].channels = 15;
	cd_data.page.audio.port[3].channels = 15;
	if(cd_set_mode(xs,&cd_data))
		return(EIO);
	}
	return(0);

    case CDIOCSETRIGHT:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCSETRIGHT entered\n"););
	{
	struct ioc_vol *arg = (struct ioc_vol *)data;
	struct cd_mode_data cd_data;

	if(cd_get_mode(xs,&cd_data,AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.port[LEFT_PORT].channels = RIGHT_CHANNEL;
	cd_data.page.audio.port[RIGHT_PORT].channels = RIGHT_CHANNEL;
	cd_data.page.audio.port[2].channels = 0;
	cd_data.page.audio.port[3].channels = 0;
	if(cd_set_mode(xs,&cd_data))
		return(EIO);
	}
	return(0);

    case CDIOCSETPATCH:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOCSETPATCH entered\n"););
	{
	struct ioc_patch *arg = (struct ioc_patch *)data;
	struct cd_mode_data cd_data;

	if(cd_get_mode(xs,&cd_data,AUDIO_PAGE))
		return(EIO);
	cd_data.page.audio.port[LEFT_PORT].channels = arg->patch[0];
	cd_data.page.audio.port[RIGHT_PORT].channels = arg->patch[1];
	cd_data.page.audio.port[2].channels = arg->patch[2];
	cd_data.page.audio.port[3].channels = arg->patch[3];
	if(cd_set_mode(xs,&cd_data))
		return(EIO);
	}
	return(0);

    case CDROMREADTOCHDR:	/* SunOS */
    case CDIOREADTOCHEADER:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOREADTOCHEADER entered\n"););
	{
	struct ioc_toc_header th;

	if(cd_read_toc(xs,0,0,(struct cd_toc_entry *)(&th),sizeof(th)))
		return(EIO);
	th.len=(th.len&0xff)<<8+((th.len>>8)&0xff);
	bcopy(&th,data,sizeof(th));
	}
	return(0);

    case CDROMREADTOCENTRY:	/* SunOS */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDROMREADTOCENTRY entered\n"););
	{
	struct cdrom_tocentry *te=(struct cdrom_tocentry *)data;
	struct _toc_data {
	    struct ioc_toc_header hdr;
	    struct cd_toc_entry data;
	} toc_data;

	if(cd_read_toc(xs,te->cdte_format,te->cdte_track,(struct cd_toc_entry *)&toc_data,sizeof(toc_data)))
		return(EIO);
	te->cdte_track=toc_data.data.track;
	te->cdte_adr=toc_data.data.addr_type;
	te->cdte_ctrl=toc_data.data.control;
	if (te->cdte_format == CDROM_MSF) {
	    te->cdte_addr.msf.minute=toc_data.data.addr[1];
	    te->cdte_addr.msf.second=toc_data.data.addr[2];
	    te->cdte_addr.msf.frame=toc_data.data.addr[3];
	} else {
	te->cdte_addr.lba=(toc_data.data.addr[0]<<24)+
			  (toc_data.data.addr[1]<<16)+
			  (toc_data.data.addr[2]<<8)+
			   toc_data.data.addr[3];
	}
	}
	return(0);

    case CDIOREADTOCENTRYS:	/* 386bsd */
	DEBUGF(cdioctldebug,printf("Cdioctl: CDIOREADTOCENTRYS entered\n"););
	{
	struct ioc_read_toc_entry *te=(struct ioc_read_toc_entry *)data;
	struct cd_toc_entry cd_data[101];
	struct ioc_toc_header *th;
	int len=te->data_len;

	th=(struct ioc_toc_header *)cd_data;

	if(len>sizeof(cd_data) || len<sizeof(struct cd_toc_entry))
		return(EINVAL);
	if(cd_read_toc(xs,te->address_format, te->starting_track, cd_data, len))
		return(EIO);
	len=MIN(len-4,((((th->len&0xff)<<8)+((th->len>>8)))+sizeof(*th))-4);
	if (len < 0)
		len=0;
	if(copyout(th+1,te->data,len)!=0)
		return(EFAULT);
	}
	return(0);

    case DKIOCSAUTOSPIN:
	sc->sc_autospinup = 1;
	return 0;

    case DKIOCSNOAUTOSPIN:
	sc->sc_autospinup = 0;
	return 0;
    case DKIOCSMODESELECT:
    case CDIOCSMODESELECT:
        if(cdmodeselect(xs,(struct scsi_modesense *)data) < 0)    /* SCSI Mode Select */
        {
                return EIO;
        }
        return 0;


    case DKIOCSMODESENSE:
    case CDIOCSMODESENSE:
        if(cdmodesense(xs,(struct scsi_msiobuf *) data) < 0)    /* SCSI Mode Sense */
        {
                return EIO;
        }
        return 0;

    case DKIOCSSCSI:
	if (securelevel >= 2)	/* not allowed in high security state */
		return EPERM;
	if(xxioctl(xs, cmd, data, flag) < 0)	/* SCSI pass through */
	{
		return EIO;
	}
	return 0;
    case DKIOCGSCSISENSE:
	DEBUGF(cdioctldebug,
		printf("Get SCSI sense:"););
	if(xs->xs_flags & XSFL_HASSENSE)
	{
		bcopy(&xs->xs_esense,data,sizeof(xs->xs_esense));
		DEBUGF(cdioctldebug,
			printf("Sense data delivered:"););
		DEBUGF(cdioctldebug,
			xxhdump(data,sizeof(xs->xs_esense)););
		return 0;
	}
	DEBUGF(cdioctldebug,printf("Nothing stored\n"););
	return -1;
    case DKIOCDSR:
	sc->sc_noretries = 1;
	return 0;
    case DKIOCESR:
	sc->sc_noretries = 0;
	return 0;
    case DIOCGDINFO: {
	int i;

	dl = (struct disklabel *) data;
	bzero(dl,sizeof(struct disklabel));
	dl->d_magic = DISKMAGIC;
	dl->d_type = DTYPE_SCSI;
	dl->d_subtype = 0;
	/* be careful when copying inqname since it's larger than d_typename */
	i=sizeof(dl->d_typename)-1;
	strncpy(dl->d_typename, sc->sc_cdst.inqname, i);
	dl->d_typename[i]='\0';	/* just in case */
		/* disk geometry: */
	dl->d_secsize = sc->sc_cdst.nbps;
	dl->d_nsectors = 100;
	dl->d_ntracks = 1;
	dl->d_ncylinders = (sc->sc_cdata.cd_blocks/100)+1;
	dl->d_secpercyl = 100;
	dl->d_secperunit = sc->sc_cdata.cd_blocks;
	dl->d_sparespertrack = 0;
	dl->d_sparespercyl = 0;
	dl->d_acylinders = 0;
	dl->d_magic = DISKMAGIC;
	dl->d_magic2 = DISKMAGIC;
	dl->d_flags = D_REMOVABLE;
	dl->d_rpm = 300;
	dl->d_interleave = 1;
	dl->d_checksum = dkcksum(dl);
	DEBUGF(cdioctldebug, 
	    printf("CGDINFO: secsize=%d nsectors=%d ntracks=%d ncyl=%d secpcyl=%d secpunit=%d\n",
	    dl->d_secsize, dl->d_nsectors, dl->d_ntracks, dl->d_ncylinders,
	    dl->d_secpercyl, dl->d_secperunit););

			/* filesystem and partition information: */
	dl->d_npartitions = 3;
	dl->d_partitions[0].p_offset = 0;
	dl->d_partitions[0].p_size = sc->sc_cdata.cd_blocks;
	dl->d_partitions[0].p_fstype = FS_ISO9660;	/* most likely */
	dl->d_partitions[0].p_fsize = 0;
			/* Fill in "c" partition just in case */
	dl->d_partitions[2].p_offset = 0;
	dl->d_partitions[2].p_size = sc->sc_cdata.cd_blocks;
	dl->d_partitions[2].p_fstype = FS_ISO9660;	/* most likely */
	dl->d_partitions[2].p_fsize = 0;
	return 0;
	}

    case DKIOCGPART:
	pep = &sc->sc_size.partition[2];
	dk->dk_size = pep->nblocks;		/* Fill in size */
	dk->dk_start = 0;			/* Fill in starting block */
	dk->dk_blocksize = sc->sc_cdst.nbps;	/* Fill in blocksize */
	dk->dk_ntrack = sc->sc_cdst.ntrak;	/* Fill in track count */
	dk->dk_nsector = sc->sc_cdst.nsect;	/* Fill in sector count */
	dk->dk_ncyl = sc->sc_cdst.ncyl;		/* Fill in cyl count */
	strcpy(dk->dk_name,"cd");		/* Fill in name */
	    DEBUGF(cdioctldebug,printf("%s%d: Cdioctl: DKIOCGPART done\n",
		xs->xs_xi->xi_driver->xd_sname, xs->xs_unit););
	return 0;

    case DIOCWDINFO:
    case DIOCSDINFO:
#ifdef	NOT_READY_YET
	{
	int error=0;

	if ((flag & FWRITE) == 0)
		return EBADF;
	else
		error = setdisklabel(&cd->disklabel, (struct disklabel *)data,
                         /*(cd->flags & DKFL_BSDLABEL) ? cd->openparts : */ 0 );
		if (error == 0)
			cd->flags |= CDHAVELABEL;
	return error;
	}
#else
	return ENOTTY;
#endif

    case USCSICMD:
	if (securelevel >= 2)	/* not allowed in high security state */
		return EPERM;
	return(parse_sun_uscsi(xs, data));

    case DIOCWLABEL:
	return EBADF;

    case CDROMREADMODE1:
    case CDROMREADMODE2:
    default:
	return ENOTTY;
    }
}

cdminphys(struct buf *bp)
{
    register int cdunit = CDUNIT(bp->b_dev);
    register struct iocc_ctlr     *ic;
    register struct xha_slave    *xs;
    register struct cd_softc	  *sc;
    extern unsigned                 (*dma_get_minphys())();
#ifdef DEBUG
    long	bcount;
#endif /* DEBUG */
    DEBUGF(cdtrace,
	printf("Cdminphys: bp=%x\n",bp););
    xs = cdinfo[cdunit];
    sc = &cd_softc[xs->xs_unit];

#if 0
    if(cdphys(bp->b_dev)) return;	/* Check existence and aliveness */
#endif
#ifdef DEBUG
    bcount = bp->b_bcount;
    if(cdminphysdebug && 		/* Debugging here ? */
       sc->sc_cdst.nbps && 		/* Nbps initalized ? */
       (bp->b_bcount & (sc->sc_cdst.nbps-1)))
    {
	printf("Cdminphys: bp=%x, bp->bcount=%d not modulo %d on entry\n",
		bp,bp->b_bcount,sc->sc_cdst.nbps);
    }
#endif /* DEBUG */

    minphys(bp);		/* Enforce kernel-wide b_bcount restriction */
#ifdef DEBUG
	if(cdminphysdebug && (bp->b_bcount != bcount))
	{
	    printf("Cdminphys bp=%x,bcount=%d changed by minphys() to %d\n",
		    bp,bcount,bp->b_bcount);
	    bcount = bp->b_bcount;
	}
#endif /* DEBUG */

    if(bp->b_bcount > 65536)
	bp->b_bcount = 65536 ; 
#ifdef DEBUG
	if(cdminphysdebug && (bp->b_bcount != bcount))
	{
	    printf("Cdminphys bp=%x,bcount=%d changed by cdminphys() to %d\n",
		    bp,bcount,bp->b_bcount);
	    bcount = bp->b_bcount;
	}
#endif /* DEBUG */


    if(xs && xs->xs_xi && xs->xs_xi->xi_hd &&  xs->xs_xi->xi_hd->xh_iocc )
    {
	ic = xs->xs_xi->xi_hd->xh_iocc;
	(*dma_get_minphys(ic))(bp);     /* dma's bcount restrictions */
    }
    else
    {
	if(xs)
	    printf("%s%d: Cdminphys: cdunit=%d, xs=%x\n", 
		xs->xs_xi->xi_driver->xd_sname, xs->xs_unit, cdunit, xs);
	else
	    printf("cd%d: Cdminphys: xs = NULL\n",cdunit);

	if(xs)
	    printf("      Cdminphys: xs->xs_xi=%x\n", xs->xs_xi);

	if(xs && xs->xs_xi)
	    printf("      Cdminphys: xs->xs_xi->xi_hd=%x\n", xs->xs_xi->xi_hd);

	if(xs && xs->xs_xi && xs->xs_xi->xi_hd)
	    printf("      Cdminphys: xs->xs_xi->xi_hd->xh_iocc=%x\n", 
		xs->xs_xi->xi_hd->xh_iocc);
	dma_minphyspage(bp);
    }
#ifdef DEBUG
	if(cdminphysdebug && (bp->b_bcount != bcount))
	{
	    printf("Cdminphys bp=%x,bcount=%d changed by dma_get_minphys() to %d\n",
		    bp,bcount,bp->b_bcount);
	    bcount = bp->b_bcount;
	}
#endif /* DEBUG */
    /*
     * Dma minphys routines can leave us w/ a partial
     * xfer *and* leftovers.  We can't handle that so
     * we truncate any partial (pushing it into leftovers).
     */
    if (sc->sc_cdst.nbps && 	/* Nbps initialized ? */
	(bp->b_bcount >= sc->sc_cdst.nbps))
	bp->b_bcount &= ~(sc->sc_cdst.nbps - 1);
#ifdef DEBUG
	if(cdminphysdebug && (bp->b_bcount != bcount))
	{
	    printf("Cdminphys bp=%x,bcount=%d corrected by cdminphys() to %d\n",
		    bp,bcount,bp->b_bcount);
	    bcount = bp->b_bcount;
	}
    if(cdminphysdebug && 
       sc->sc_cdst.nbps &&
       (bp->b_bcount & (sc->sc_cdst.nbps-1)))
    {
	printf("Cdminphys: bp=%x, bp->bcount=%d not modulo %d on exit.\n",
		bp,bp->b_bcount,sc->sc_cdst.nbps);
	bcount = bp->b_bcount;
    }
#endif /* DEBUG */

}

cdtryretry(struct xha_slave *xs, struct buf *bp, int maxretries, int errcode)
{
    register struct cd_softc *sc = &cd_softc[xs->xs_unit];
    DEBUGF(cdtrace,
	printf("Cdtryretry: xs=%x,bp=%x,maxretries=%d,errcode=%d\n",
		xs,bp,maxretries,errcode););
    if(!((xs->xs_flags & XSFL_SPECSCSI) && sc->sc_noretries) &&
	  ++bp->b_errcnt <= maxretries)
	return XHD_RETRY;
    else
    {
	bp->b_bcount = bp->b_resid;
	bp->b_resid = 0;
	bp->b_error = errcode;
	bp->b_flags |= B_ERROR;
	cdpartdone(xs,bp);
	xs->xs_flags &= ~XSFL_SPECSCSI;
	return XHD_DONE;
    }
}

/* 
 * Combine three bytes into an unsigned
 */
unsigned
cdget3(char *p)
{
    return ((*p & 0xFF) << 16) | ((p[1] & 0xFF) << 8) | (p[2] & 0xFF);
}

cdidcpy(u_char *f,char *t,int size)
{
    register int	i;

    bcopy((char *)f,t,size-1);
    t[size-1] = '\0';

    for(i=size-1;i>=0;i--)
    {
	if((t[i] != ' ') && (t[i] != '\0'))
	{
	    break;
	}
	t[i] = '\0';
    }
}

do_sun_uscsi(xs, uscsi_cdb, uscsi_cdblen, uscsi_bufaddr, uscsi_buflen,
	      uscsi_status, uscsi_flags)
struct xha_slave *xs;
caddr_t uscsi_cdb;
int     uscsi_cdblen;
caddr_t uscsi_bufaddr;
int     uscsi_buflen;
unsigned char   *uscsi_status;
int     uscsi_flags;
{
    register union scsi_cdb *scb;
    int rc;
    caddr_t buf;

    DEBUGF(cdtrace, printf("do_sun_uscsi: xs=%x uscsi_cdb=%x uscsi_cdblen=%d uscsi_bufaddr=%x uscsi_buflen=%d uscsi_status=%d, uscsi_flags=%x\n", xs, uscsi_cdb, uscsi_cdblen, uscsi_bufaddr, uscsi_buflen, *uscsi_status, uscsi_flags););

    scb = xxgetscbuf(xs);

    if (uscsi_cdblen < 0 || uscsi_cdblen > 12) {	/* cdb size valid? */
	printf("cd%d: cdb size of %d for USCSI not valid\n",
		xs->xs_unit, uscsi_cdblen);
	return(EINVAL);
    }
    if (uscsi_buflen < 0) {				/* buf size valid? */
	printf("cd%d: buf size of %d for USCSI not valid\n",
		xs->xs_unit, uscsi_buflen);
	return(EINVAL);
    }
			/* get some memory for the i/o operation */
    if ((buf = (caddr_t)malloc(uscsi_buflen, M_DEVBUF, M_NOWAIT)) <= 0)
	return(EINVAL);
    bzero(buf, uscsi_buflen);
					/* get the scsi command */
    if (copyin(uscsi_cdb, scb, uscsi_cdblen) != 0)
	return(EFAULT);

					/* get the data if we're writing */
    if ((uscsi_flags == B_WRITE) && uscsi_buflen) {
	if (copyin(uscsi_bufaddr, buf, uscsi_buflen) != 0)
	    return(EFAULT);
    }

	/* call the scsi driver to execute the command */
    if (((*xs->xs_xi->xi_hadriver->xhc_doscsi)
	 (xs, scb, buf, uscsi_buflen, uscsi_flags)) ==
         ((XHI_CMPL << 8) + SCTS_GOOD)) {
	rc=0;
	*uscsi_status=0x00;	/* USCSI_STATUS_GOOD in scsi/impl/uscsi.h */
    } else {
	rc=-1;
	*uscsi_status=0x02;	/* USCSI_STATUS_CHECK in scsi/impl/uscsi.h */
    }

					/* save the data if we just read it */
    if ((uscsi_flags == B_READ) && uscsi_buflen && !rc) {
	if (copyout(buf, uscsi_bufaddr, uscsi_buflen) != 0)
	    return(EFAULT);
    }

    free(buf, M_DEVBUF);	/* free any memory allocated */

    return(rc);
}

/*
 * Set the density and block size on the Toshiba XM-3401B drive
 */
#define	MAX_DENSITY_ATTEMPTS	3

SetXM3401Density(xs, density, sec_size)
struct xha_slave *xs;
int density, sec_size;
{
    int attempts;
    struct scsi_modesense *p;

    DEBUGF(cdtrace, printf("cd%d SetXM4101Density: density=%d size=%d\n", xs->xs_unit, density, sec_size););

    p=(struct scsi_modesense *)&cdmsiobuf;

	/* first get all the mode sense pages and make our changes */
    for (attempts=1; attempts <= MAX_DENSITY_ATTEMPTS; attempts++) {
	if (cdmodesense(xs, &cdmsiobuf) == 0) {
	    p->sm_bdl=8;
	    p->sm_length=0;
	    p->sm_wp=0;
	    p->sm_speed=0;
	    p->sm_bdarr[0].sbd_density=density;
	    p->sm_bdarr[0].sbd_blklen=sec_size;
	    break;
	} else {
	    if (attempts = MAX_DENSITY_ATTEMPTS) {
		printf("cd%d SetXM4101Density: Failed mode sense\n",
			xs->xs_unit);
		return(1);
	    }
	}
    }
	/* now write it back to the device */
    for (attempts=1; attempts <= MAX_DENSITY_ATTEMPTS; attempts++) {
	if (cdmodeselect(xs, p) < 0) {
	    if (attempts = MAX_DENSITY_ATTEMPTS) {
		printf("cd%d SetXM4101Density: Failed mode select\n",
			xs->xs_unit);
		return(1);
	    }
	} else {
	    break;
	}
    }
    return(0);
}

mode1_sectors(xs)
struct xha_slave *xs;
{
    return(SetXM3401Density(xs, 0, 2048));
}

mode2form1_sectors(xs)
struct xha_slave *xs;
{
    return(SetXM3401Density(xs, 129, 2048));
}

mode2form2_sectors(xs)
struct xha_slave *xs;
{
    return(1);		/* cause an error, we currently can't handle sizes
			   greater than 2048 bytes per block */
/*  return(SetXM3401Density(xs, 129, 2324)); */
}

cd_da_sectors(xs)
struct xha_slave *xs;
{
    return(1);		/* cause an error, we currently can't handle sizes
			   greater than 2048 bytes per block */
/*  return(SetXM3401Density(xs, 130, 2352)); */
}

static bcd2i(val)	/* convert a BCD coded byte to an integer */
int val;
{
    return(((val&0xF0)>>4)*10+(val&0x0F));
}
#endif /* NCD  > 0 */
