#include "cs.h"
#if NCS > 0

/*
 * Copyright (C) 1988, 1989 Christer Bernerus,
 *	         Chalmers University of Technology, Gothenburg, SWEDEN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms as is or as a source
 * for derivative work are permitted provided that the above copyright 
 * notice and this paragraph are  duplicated in all such forms 
 * and that any documentation, advertising materials,
 * and other materials related to such distribution
 * and use acknowledge that the software was developed
 * at Chalmers University of Technology, SWEDEN.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Pertec CS240 SCSI Tape Streamer driver 
 *
 * $Source: /usr/src/sys/rt/dev/RCS/cs.c,v $
 * $Revision: 1.12 $ $Date: 1995/03/16 14:41:34 $
 * $Author: md $ $State: Exp $
 *
 * $Log: cs.c,v $
 * Revision 1.12  1995/03/16  14:41:34  md
 * Use block mode writes if the buffer is a multiple of the current
 * block size.  Some drives (eg. 1/4" tape) can't handle variable
 * size records which are bigger than the maximum block size.
 *
 * Revision 1.11  1995/03/15  16:10:23  md
 * Need to return partial blocks when either a file mark or end of tape
 * is detected.  Also cleaned up file mark handling code.  It's now good
 * enough for things like tar and dump but it may get confused if a
 * program tries to seek() after hitting a tape mark.  You also need to
 * reopen the tape device if reading past the tape mark is desired.
 *
 * Revision 1.10  1995/03/13  15:24:47  md
 * Reading of blocks should be handled based on the block size set, not
 * the density of the drive.
 *
 * Revision 1.9  1995/03/12  21:21:12  md
 * Major changes to support transfers which are larger than a single
 * block.  Block size is also now setable via the "mt" command.  Performance
 * is now improved significantly for tape which have been written with
 * small blocks (eg. 1K).
 *
 * Revision 1.9  1995/03/10  21:54:09  md
 * Support variable length blocks as the default mode.
 *
 * Revision 1.8  1994/06/11  20:26:19  roger
 * changes for the '44 fcntl.h.
 *
 * Revision 1.7  1994/05/22  12:33:57  roger
 * header file changes.
 *
 * Revision 1.6  1994/05/12  20:21:40  roger
 * removed dir.h.
 *
 * Revision 1.5  1994/05/07  14:52:00  md
 * Change #include statements to use system search path instead of absolute.
 *
 * Revision 1.4  1993/04/20  17:43:30  md
 * more changes to cleanup the 4mm tape support
 *
 * Revision 1.3  1993/02/24  23:28:14  md
 * lots of little changes to support 4mm DAT drives and fix some compatibility
 * problems with reno.
 *
 * Revision 1.2  1992/12/09  08:40:43  md
 * Add back in debugging hooks.
 *
 * Revision 1.1  1992/11/05  18:55:41  md
 * Initial revision
 *
 * Revision 3.0  90/02/08  09:45:56  bernerus
 * February 1990 release.
 * 
 * Revision 2.6  90/02/06  17:07:33  bernerus
 * Now uses xxscsicmd instead of csmodesense.
 * uprintf's changed to printf's.
 * Now supports MODE SELECT and MODE SENSE in ioctl.
 * 
 * Revision 2.5  89/11/15  14:51:56  bernerus
 * The macro CSCUNIT is not used anymore and has been deleted.
 * The usage of the macro CSUNIT has been reduced. Instead the info in the
 * xha_slave structure is used.
 * Some functions used during autoconfig now uses the generalized functions
 * in xxc.c instead.
 * All functions that previously used the xha_device pointer (xi) for pointing
 * out a specific device now uses the xha_slave pointer (xs) instead.
 * SCSI CDB's are now uniformly pointed to by using a (union scsi_cdb *) type.
 * 
 * Revision 2.4  89/08/22  08:36:56  bernerus
 * Probing is now done through the SCSI Inquiry command to make sure
 * that the target is of the correct type.
 * Where possible, the use of the xha_device structure is replaced
 * by using the xha_slave structure. A major revision of the triver will
 * contain this change thoroughly and enable LUN's different from zero.
 * 
 * Revision 2.3  89/02/24  13:10:06  bernerus
 * Changed the copyright notice.
 * 
 * Revision 2.2  89/02/08  13:09:36  bernerus
 * Made some ANSI-C constructions.
 * Now we do a SCSI Inquiry to get the tape type.
 * The probe routine now really does something.
 * 
 * Revision 2.1  89/01/30  08:01:57  bernerus
 * Removed some display() calls.
 * Added initial flag value in xha_driver structure.
 * 
 * Revision 2.0  89/01/16  11:16:50  bernerus
 * Added copyright notice.
 * Changed names cu->cs, cs->csc.
 * Added ioctl routine.
 * 
 *
 */

#include <machine/pte.h>

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/vm.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/user.h>
#include <sys/ioctl.h>
#include <sys/cmap.h>
#include <sys/uio.h>
#include <sys/tty.h>
#include <sys/mtio.h>
#include <sys/proc.h>
#include <machine/dkio.h>
#include "rt/include/ioccvar.h"
#include "rt/include/scsi.h"
#include "rt/include/xhavar.h"
#include "rt/rt/debug.h"

#define INF             10000000L		    /* a block number that
						     * won't exist */

/* Bits in minor device */

#define CSUNIT(dev)	XHAUNIT(dev)
#define CSNOREW(dev)	XHAMIN_NOREW(dev)

struct xha_device *cscinfo[NCSC];
struct xha_slave *csinfo[NCS];

#ifdef DEBUG
int             csdebug = 0;
int		cstrace = 0;
int		csprobedebug = 0;
int             csintdebug = 0;
int             csmodeselectdebug = 0;
int		cssccomdebug = 0;
int		cssleepdebug = 0;
int		cserrdebug = 0;
#endif /* DEBUG */

/* Per unit status information */

struct cs_softc
{
    char               scf_open:1;	/* unit is open 		      */
    char               scf_written:1;	/* last operation was a write 	      */
    char               scf_eot:1;	/* end of tape encountered 	      */
    char               scf_ieot:1;	/* ignore EOT condition 	      */
    char               scf_fm:1;	/* file mark encountered 	      */
    char               scf_selectmode:1;/* Mode select needed b4 1:st op      */
    char               scf_busy:1;	/* Unit busy detected 		      */
    char               scf_blkmode:1;	/* reading in block mode	      */
    int                sc_busyretries;	/* Busy retry count	 	      */
    daddr_t            sc_blkno;	/* current physical block number      */
    daddr_t            sc_nxrec;	/* firewall input block number 	      */
    short              sc_resid;	/* residual function count for ioctl  */
    int                sc_slave;	/* slave number for unit 	      */
    struct tty        *sc_ttyp;		/* record user's tty for errors       */
} cs_softc[NCS];

struct cs_hardc {
    int                hc_density;	/* Tape drive density code	      */
    int                hc_blklen;	/* Tape drive block length	      */
    int                hc_maxblklen;	/* Maximum supported block length     */
    int                hc_minblklen;	/* Minimum supported block length     */
} cs_hardc[NCS];

char sensebuf[NCS][SIZEOF_MODESENSE];	/* MODE SENSE buffer		      */
char selectbuf[NCS][SIZEOF_MODESELECT];	/* MODE SELECT buffer		      */

struct buf      rcsbuf[NCS];	    /* data transfer buffer structures */
struct buf      ccsbuf[NCS];	    /* tape command buffer structures */

#define b_command b_iodone
#define b_repcnt  b_bcount

/* Functions predefinitions */
#include "rt/dev/xxc.h"

int csprobe(caddr_t, struct xha_device *); 
int csattach(struct xha_slave *); 
int csslave(struct xha_slave *, caddr_t);
int csidcpy(u_char *,char *,int);
int csttl(union scsi_cdb *);
int csopen(dev_t, int);
int csmodeselect(struct xha_slave *, struct scsi_modeselect *);
int csmodesense(struct xha_slave *, struct scsi_modesense *);
int csclose(dev_t, int);
int cssccom(struct xha_slave *, union scsi_cdb *, int, caddr_t, int);
int csustart(struct xha_slave *);
int csint(struct xha_slave *, struct buf *, int, struct scsi_esense *, u_char);
int csread(dev_t, struct uio *);
int cswrite(dev_t, struct uio *);
int csphys(dev_t, struct uio *);
int csstrategy(struct buf *);
int csxstrat(struct xha_slave *, struct buf *);
void csminphys(struct buf *);
union scsi_cdb *csstart(struct xha_slave *,struct buf *,union scsi_cdb *);

#define spl_tape() _spl4()	     /* cpu level 4 */

struct xha_driver cscdriver =
{
    csprobe, 
    csslave,  
    csattach,  
    csustart,
    csstart,
    csint,
    "csc", 
    "cs", 
    cscinfo, 
    csinfo,
    csttl,
    EARLY_INT,
};
struct densinfo 
{
	int	density;
	float	widthmm, widthin;
	int	tracks, bpmm, bpi, blksize;
	char	*code, *type, *std;
} denstab[] =
{
{  1, 12.7,  0.5,   9,   32,   800,     0, "NRZI", "Reel",	"X3.22-1983"  },
{  2, 12.7,  0.5,   9,   63,  1600,     0, "PE",   "Reel",	"X3.39-1986"  },
{  3, 12.7,  0.5,   9,  256,  6250,     0, "GCR",  "Reel",	"X3.54-1986"  },
{  4,  6.3,  0.25,  4,  315,  8000,   512, "GCR",  "Cartridge",	"QIC-11"      },
{  5,  6.3,  0.25,  4,  315,  8000,   512, "GCR",  "Cartridge", "X3.136-1986" },
{  6, 12.7,  0.5,   9,  126,  3200,     0, "PE",   "Reel",	"X3.157-1987" },
{  7,  6.3,  0.25,  4,  252,  6400,   512, "IMFM", "Cartridge", "X3.116-1986" },
{  8,  3.81, 0.15,  4,  315,  8000,     0, "GCR",  "Cassette",  "X3.158-1987" },
{  9, 12.7,  0.5,  18, 1491, 37871,     0, "GCR",  "Cartridge", "X3B5/87-099" },
{ 10, 12.7,  0.5,  22,  262,  6667,     0, "MFM",  "Cartridge", "X3B5/86-199" },
{ 11,  6.3,  0.25,  4,   63,  1600,   512, "PE",   "Cartridge", "X3.56-1986"  },
{ 12, 12.7,  0.5,  24,  500, 12690,     0, "GCR",  "Cartridge", "HI-TC1"      },
{ 13, 12.7,  0.5,  24,  999, 25380,     0, "GCR",  "Cartridge", "HI-TC2"      },
{ 14,  0.0,  0.0,   1,    0,     0,     0, "Unkn", "Unknown",   "Reserved"    },
{ 15,  6.3,  0.25, 15,  394, 10000,   512, "GCR",  "Cartridge", "QIC-120"     },
{ 16,  6.3,  0.25, 18,  394, 10000,   512, "GCR",  "Cartridge", "QIC-150"     },
{ 17,  6.3,  0.25, 26,  630, 16000,   512, "GCR",  "Cartridge", "QIC-320"     },
{ 18,  6.3,  0.25, 30, 2034, 51667,   512, "RLL",  "Cartridge", "QIC-1350"    },
{ 19,  3.81, 0.15,  1, 2400, 61000, 10240, "DDS",  "Cassette",  "X3B5/88-185A"},
{ 20,  8.0,  0.315, 1, 1703, 43245,  1024, "RLL",  "Cassette",  "X3.202-1991" },
{ 21,  8.0,  0.315, 1, 1789, 45434,  1024, "RLL",  "Cassette",  "ECMA TC17"   },
{ 22, 12.7,  0.5,  48,  394, 10000,     0, "MFM",  "Cartridge", "X3.193-1990" },
{ 23, 12.7,  0.5,  48, 1673, 42500,     0, "MFM",  "Cartridge", "X3B5/91-174" },
{140,  8.0,  0.315, 1, 1789, 45434, 10240, "RLL",  "Cassette",  "X3.202-1991" },
{  0,  0.0,  0.0,   1,    0,     0,     0, "Def",  "Def",	"None"	      },
};


csprobe(caddr_t reg,struct xha_device *xi)
{
    return xxprobe(xi,&cscdriver,SCPDT_SASD);
}

csattach(struct xha_slave *xs)
{
    struct scsi_inqblk  iqbuf;
    struct scsi_inqblk  *iq = &iqbuf;
    static char         vid[sizeof(iq->sinq_vid)+1];
    static char         pid[sizeof(iq->sinq_pid)+1];
    static char         fwrev[sizeof(iq->sinq_fwrev)+1];
    struct scsi_modesense *msenp;
    union  scsi_cdb	  *scb;
    int			dcode;
    struct densinfo	*dp;
    struct cs_hardc	*hc = &cs_hardc[xs->xs_unit];
    struct {
		 char	reserved;
	unsigned char	max_h;
	unsigned char	max_m;
	unsigned char	max_l;
	unsigned char	min_h;
	unsigned char	min_l;
    } limits;

    if (xs->xs_unit > NCS)
    {
	printf("cs%d: Not configured\n", xs->xs_unit);
	return 0;
    }
    if(xxinquiry(xs,iq) == 0)
    {
	if(iq->sinq_pdt != SCPDT_SASD)
	{
	    printf("cs%d: Not a tape\n",xs->xs_unit);
	    return 0;
	}

	xs->xs_xi->xi_type = iq->sinq_pdt; /* Save device type here */

	csidcpy(iq->sinq_vid,vid,sizeof vid -1);
	csidcpy(iq->sinq_pid,pid,sizeof pid -1);
	csidcpy(iq->sinq_fwrev,fwrev,sizeof fwrev -1);

	msenp = (struct scsi_modesense *) sensebuf[xs->xs_unit];
	
	if (xxscsicmd(xs, SC0_MODE_SENSE,(caddr_t) msenp,SIZEOF_MODESENSE,B_READ) != 0)
	{
	    printf("cs%d: mode sense failed\n", xs->xs_unit);
	    return 0;
	}

	printf("%s%d:%s %s Rev. %s ANSI V.%d\n",
	    xs->xs_driver->xd_sname,xs->xs_unit,
	    vid,pid,fwrev,iq->sinq_ansiversion);

	dcode=msenp->sm_bdarr[0].sbd_density;

#ifdef	PRINT_BLOCK_DIAG
	printf("%s%d: nblocks=%d, blklen=%d\n",
	    xs->xs_driver->xd_sname,xs->xs_unit,
	    msenp->sm_bdarr[0].sbd_nblocks,
	    msenp->sm_bdarr[0].sbd_blklen);
#endif
	for (dp=denstab; dp->density; dp++) {
	    if (dp->density == dcode) {
		printf("%s%d:%s tape (%s), %d track(s), density=%dbpmm blksize=%d",
			xs->xs_driver->xd_sname,xs->xs_unit,
			dp->type, dp->code, dp->tracks, dp->bpmm, dp->blksize);
		break;
	    }
	}
	if (!dp->density && dcode)
	    printf(", density code=%d",dcode);
	printf("\n");
	hc->hc_density=dp->density;
	hc->hc_blklen=dp->blksize;

	scb = xxgetscbuf(xs);
	xxfillscbuf(scb, xs->xs_lun, SC0_READ_BLOCK_LIMITS, 0);
	if (((*xs->xs_xi->xi_hadriver->xhc_doscsi) (xs, scb, (caddr_t)&limits, 
		sizeof(limits), B_READ)) != ((XHI_CMPL << 8) + SCTS_GOOD)) {
	    printf("%s%d: read block limits failed\n",
		xs->xs_driver->xd_sname,xs->xs_unit);
			/* reasonable defaults (from EXB-8505) */
	    hc->hc_maxblklen=50331648;	/* 48MB */
	    hc->hc_minblklen=0;
	} else {
	    hc->hc_maxblklen=limits.max_h<<16 + limits.max_m<<8 + limits.max_l;
	    hc->hc_minblklen=limits.min_h<<8 + limits.min_l;
	    printf("%s%d:block size limits [max=%d, min=%d]\n",
		xs->xs_driver->xd_sname,xs->xs_unit,
		hc->hc_maxblklen, hc->hc_minblklen);
	}
    }
    else
    {
	printf("cs%s: inquiry failed\n", xs->xs_unit);
	return 0;
    }

    bzero(&ccsbuf[xs->xs_unit], sizeof ccsbuf[xs->xs_unit]);
    bzero(&rcsbuf[xs->xs_unit], sizeof rcsbuf[xs->xs_unit]);
    bzero(&cs_softc[xs->xs_unit], sizeof cs_softc[xs->xs_unit]);
    bzero(sensebuf[xs->xs_unit], SIZEOF_MODESENSE);
    bzero(selectbuf[xs->xs_unit], SIZEOF_MODESELECT);

    return 1;
}

csidcpy(u_char *f,char *t,int size)
{
    register int        i;

    bcopy((char *)f,t,size-1);
    t[size-1] = '\0';

    for(i=size-1;i>=0;i--)
    {
        if((t[i] != ' ') && (t[i] != '\0'))
        {
            break;
        }
        t[i] = '\0';
    }
}

csslave(struct xha_slave *xs, caddr_t addr)
{
    return xxslave(xs,&cscdriver,SCPDT_SASD);
}

/* Tell controller routine expected max lifetime of */
/* a command (time in seconds)			    */
csttl(union scsi_cdb *com)
{
	switch(com->cdb_6.sc_op)
	{
	    case SC0_REWIND:		return 1000/24;
	    case SC0_SPACE:		return 1000;
	    case SC0_TRACK_SELECT:	return 5;
	    case SC0_LOAD_UNLOAD:	return 1000/24+30;
	    case SC0_ERASE:		return 1000;
	    default:			return 1;
	}
}

csopen(dev_t dev, int flag)				    /* Open routine */
{
    struct xha_slave *xs;
    struct cs_softc *sc;
    struct scsi_modesense *msenp;
    struct proc *p = u.u_procp;

    /* Validate unit and aliveness */
    xs = csinfo[CSUNIT(dev)];

    if ((CSUNIT(dev) >= NCS)
	|| (xs == 0)
	|| (xs->xs_alive == 0))
    {
	return (ENXIO);
    }


    /*
     * Check if already open, no spl's needed since we don't voluntarily give
     * up the cpu and the system will nor reschedule while in kernel 
     */
    if ((sc = &cs_softc[xs->xs_unit])->scf_open)
	return (EBUSY);
    bzero(sc, sizeof *sc);
    sc->scf_open = 1;

    /* OK, now get mode status from the unit */

    msenp = (struct scsi_modesense *) sensebuf[xs->xs_unit];
    if (xxscsicmd(xs, SC0_MODE_SENSE,(caddr_t) msenp, SIZEOF_MODESENSE, 
		B_READ) != 0) {
	printf("cs%s: not online ?\n", xs->xs_unit);
	sc->scf_open = 0;
	return EIO;
    }

    if ((flag & FWRITE) && (msenp->sm_wp))
    {
	printf("cs%s: write protected\n", xs->xs_unit);
	sc->scf_open = 0;
	return EIO;
    }

    sc->scf_selectmode = 1;

    sc->sc_blkno = (daddr_t) 0;

    /* Since cooked I/O may do a read-ahead before a write, trash   */
    /* on a tape can make the first write fail.  Suppress the first */
    /* read-ahead unless definitely doing read-write                */

    sc->sc_nxrec = ((flag & (O_TRUNC | FWRITE)) == (O_TRUNC | FWRITE))
	? (daddr_t) 0
	: (daddr_t) INF;

    sc->scf_fm = 0;	/* ignore any previously encountered file marks */

    sc->sc_ttyp = p->p_session->s_ttyp; /* in aos was u.u_ttyp -md */
    return 0;
}

/*
 * Code to execute mode select command 
 */
csmodeselect(struct xha_slave *xs, struct scsi_modeselect *mselp)
{
    register union scsi_cdb *scp;

    /* Get a SCSI command buffer  and fill it in */
    scp = xxgetscbuf(xs);
    xxfillscbuf(scp, xs->xs_lun, SC0_MODE_SELECT, SIZEOF_MODESELECT);

    return (cssccom(xs,scp,1,(caddr_t) mselp,SIZEOF_MODESELECT));
}

/*
 * Code to execute SCSI Mode sense command 
 */
csmodesense(struct xha_slave *xs, struct scsi_modesense *msenp)
{
    register union scsi_cdb *scp;

    /* Get a SCSI command buffer  and fill it in */
    scp = xxgetscbuf(xs);
    xxfillscbuf(scp, xs->xs_lun, SC0_MODE_SENSE, SIZEOF_MODESENSE);

    return (cssccom(xs,scp,1,(caddr_t) msenp,SIZEOF_MODESENSE));
}

csclose(dev_t dev, int flag)
{
    struct xha_slave *xs = csinfo[CSUNIT(dev)];
    struct cs_softc *sc = &cs_softc[xs->xs_unit];
    union scsi_cdb *scp;

    /*
     * Now we must shut off selectmode if still set, or else the 
     * process will hang up since csstrategy will call cssccom while
     * csstrategy is called from cssccom ....
     */
    sc->scf_selectmode = 0;

    if (((flag & (FREAD | FWRITE)) == FWRITE)
	|| ((flag & FWRITE) && (sc->scf_written)))
    {
	scp = xxgetscbuf(xs);
		/* Write 2 filemarks if we're really done (ie. rewinding)
		   otherwise only 1 since we're probably going to do more
		   writing. */
	if (CSNOREW(dev))
	    xxfillscbuf(scp, xs->xs_lun, SC0_WRITE_FILEMARKS, 1);
	else
	    xxfillscbuf(scp, xs->xs_lun, SC0_WRITE_FILEMARKS, 2);
	cssccom(xs,scp,1,0,0);		    /* Write filemark(s) */
    }

    /*
     * If we should rewind the tape, issue that command, else we just close it 
     */

    if (!CSNOREW(dev))
    {
	scp = xxgetscbuf(xs);
	xxfillscbuf(scp, xs->xs_lun, SC0_REWIND, 0);
#ifdef	WAIT_FOR_REWIND
	scp->cdb_6.sc_lbaMSB |= 0;	    /* IMMEDIATE bit not set */
#else
	scp->cdb_6.sc_lbaMSB |= 1;	    /* IMMEDIATE bit set */
#endif
	cssccom(xs,scp,1,0,0);		    /* Rewind the tape */
    }
    sc->scf_open = 0;
    return 0;
}

cssccom(struct xha_slave *xs, union scsi_cdb *scp, 
	int wait, caddr_t spacep, int size)
{
    register struct buf *bp;
    int             s;

    /* Get command buffer */
    bp = &ccsbuf[xs->xs_unit];
    s = spl5();
    while (bp->b_flags & B_BUSY)
    {
	if ((bp->b_repcnt == 0) && (bp->b_flags & B_DONE))
	    break;
	bp->b_flags |= B_WANTED;
	DEBUGF(cssleepdebug,
	    printf("Waiting for command buffer %x *scp=%x. zzzz\n",
		bp,scp););
	sleep((caddr_t) bp, PRIBIO);
	DEBUGF(cssleepdebug,
	    printf("Aaah got command buffer %x, *scp=%x ?\n",bp,scp););
    }
    bp->b_flags = B_BUSY | B_READ;
    splx(s);

    /* Fill in the buf variables */
    bp->b_bufsize = size;
    bp->b_command = (int (*) ()) scp;
    bp->b_bcount = size;
    bp->b_blkno = 0;
    bp->b_error = 0;
    bp->b_errcnt = 0;
    bp->b_un.b_addr = spacep;
    DEBUGF(cssccomdebug, printf("cssccom: buf=%x,spacep=0x%x, size=%d, op=%x,wait=%d\n",
				      bp,
				      bp->b_un.b_addr,bp->b_bufsize,
				      scp->cdb_6.sc_op,wait);); 

    /* Call strategy routine */
    csxstrat(xs, bp);

    /* Await completion */
    if(wait)
    {
	DEBUGF(cssleepdebug,
	    printf("Waiting for command bfr %x op=%x to complete\n",
		     bp,scp->cdb_6.sc_op););
	iowait(bp);
	DEBUGF(cssleepdebug,
	    printf("Command bfr %x, op=%x done\n",bp,scp->cdb_6.sc_op););
    }

    /* If anyone wants this buffer, wake them up */
    if(bp->b_flags & B_WANTED)
	wakeup((caddr_t) bp);

    bp->b_flags &= ~(B_BUSY | B_WANTED | B_PHYS);
    /* return geterror(bp);	FIX geterror missing -md */
    return;
}

csustart(struct xha_slave *xs)
{
    register struct buf *bp = xs->xs_tab.b_actf;
    register struct cs_softc *sc = &cs_softc[xs->xs_unit];

    DEBUGF(csdebug, printf("csustart: xs=0x%x\n", xs););
    if (sc->scf_open == 0)
    {
	bp->b_flags |= B_ERROR;
	return (XHU_NEXT);
    }
    if (bp != &ccsbuf[xs->xs_unit])
    {

	if ((sc->scf_eot && !sc->scf_ieot) &&
	    ((bp->b_flags & B_READ) == 0))
	{
	    bp->b_flags |= B_ERROR;
	    bp->b_error = ENOSPC;
	    return (XHU_NEXT);
	}

	/* special case tests for cooked mode */

	if (bp != &rcsbuf[xs->xs_unit])
	{

	    DEBUGF(csdebug,printf("csustart: b_blkno=%d(%d)  sc->sc_nxrec=%d\n",
		    bp->b_blkno, bdbtofsb(bp->b_blkno), sc->sc_nxrec););

		/* if file mark detected then just return no data */
	    if ((bp->b_flags & B_READ) && sc->scf_fm) {
		bp->b_resid = bp->b_bcount;
		return (XHU_NEXT);
	    }

	    /* seek beyond end of file */

	    if (bdbtofsb(bp->b_blkno) > sc->sc_nxrec)
	    {
		DEBUGF(csdebug, printf("csustart: seek beyond end of file (%d > %d)\n", bdbtofsb(bp->b_blkno), sc->sc_nxrec););
		bp->b_flags |= B_ERROR;
		bp->b_error = EIO;
		return (XHU_NEXT);
	    }

	    /* This should be end of file, but the buffer      */
	    /* system wants a one-block look-ahead.  Humor it. */

	    if ((bdbtofsb(bp->b_blkno) == sc->sc_nxrec)
		&& (bp->b_flags & B_READ))
	    {
		sc->sc_resid = bp->b_bcount;
		clrbuf(bp);
		bp->b_resid = sc->sc_resid;
		return (XHU_NEXT);
	    }

	    /* If writing, mark the next block invalid. */

	    if ((bp->b_flags & B_READ) == 0) {
		sc->sc_nxrec = bdbtofsb(bp->b_blkno) + 
			       bdbtofsb(bp->b_bcount/DEV_BSIZE) + 1;
		DEBUGF(csdebug, printf("csustart: set sc->sc_nxrec to %d  b_blkno=%d, b_bcount=%d\n", sc->sc_nxrec, bp->b_blkno, bp->b_bcount););
	    }
	}
    }
    return (XHU_DODATA);
}

union scsi_cdb *
csstart(struct xha_slave *xs, struct buf *bp, union scsi_cdb *combuf)
{
    register union scsi_cdb *scbuf;
    register int tx_count;
    struct cs_softc *sc = &cs_softc[xs->xs_unit];
    struct cs_hardc *hc = &cs_hardc[xs->xs_unit];

    if (bp == &ccsbuf[xs->xs_unit])
	return (union scsi_cdb *) bp->b_command;


    if(combuf)
	scbuf = combuf;
    else
	scbuf = xxgetscbuf(xs);

    xxfillscbuf(scbuf, xs->xs_lun, bp->b_flags & B_READ ? SC0_READ : SC0_WRITE, 1);

#ifdef	OLD_CODE
    if (bp->b_flags & B_PHYS)
    {
	scbuf->cdb_6.sc_lbarest[0] = bp->b_bcount >> 16 & 0xFF;
	scbuf->cdb_6.sc_lbarest[1] = bp->b_bcount >> 8 & 0xFF;
	scbuf->cdb_6.sc_xflen[0] = bp->b_bcount & 0xFF;
#ifdef	DO_SILI
	if (bp->b_flags & B_READ)
	    scbuf->cdb_6.sc_lbaMSB = 2;		    /* SILI = 1 */
#endif
    }
    else
	scbuf->cdb_6.sc_lbaMSB = 1;		    /* ie. fixed mode */
#endif

    tx_count=bp->b_bcount;
    sc->scf_blkmode=0;

	/* If we have a block size set and the size of the buffer we've 
	   been provided is a multiple of that block size, then issue a 
	   SCSI READ/WRITE for exactly those number of blocks.  Otherwise 
	   we have to assume it's a variable sized record and let the drive
	   try and handle it.  When reading this usually ends up reading 
	   just one block and performance will suffer severely.  Note
	   that a MODE SELECT has already been issused by csxstrat()
	   to set the block size.
	*/
    if (hc->hc_blklen) {
	if (!(bp->b_bcount % hc->hc_blklen)) {
	    sc->scf_blkmode=1;
	    scbuf->cdb_6.sc_lbaMSB = 1;	/* fixed blocks */
	    tx_count=bp->b_bcount/hc->hc_blklen;
	}
    }

    scbuf->cdb_6.sc_lbarest[0] = tx_count >> 16 & 0xFF;
    scbuf->cdb_6.sc_lbarest[1] = tx_count >> 8 & 0xFF;
    scbuf->cdb_6.sc_xflen[0] = tx_count & 0xFF;
#ifdef	DO_SILI
    if (bp->b_flags & B_READ)
	scbuf->cdb_6.sc_lbaMSB |= 2;		    /* SILI = 1 */
#endif
    return scbuf;
}

csint(struct xha_slave *xs, struct buf *bp, 
      int ccode, struct scsi_esense *sdp, u_char tarstat)
{
    struct cs_softc *sc = &cs_softc[xs->xs_unit];
    struct cs_hardc *hc = &cs_hardc[xs->xs_unit];

    DEBUGF(csintdebug, printf("csint: xs=0x%x, bp=0x%x, ccode=%d, sdp=0x%x, tarstat=%d\n",
		               xs, bp, ccode, sdp, tarstat););

    if(bp && (bp->b_flags & B_HEAD))
	    return XHD_DONE;

    if (tarstat == SCTS_BUSY)
    {
	if (sc->scf_busy)
	{
	    if (++sc->sc_busyretries > 10)
	    {
		bp->b_error = EIO;
		bp->b_flags |= B_ERROR;
		sc->scf_busy = 0;
		return XHD_DONE;
	    }
	    return XHD_RETRY;
	}
	sc->scf_busy = 1;
	sc->sc_busyretries = 1;
	return XHD_RETRY;
    }
    sc->scf_busy = 0;


    switch (ccode)
    {
    case XHI_CMPL:
	sc->scf_written = !(bp->b_flags & B_READ);
	sc->sc_blkno++;
	bp->b_resid = 0;
	return XHD_DONE;
    case XHI_TIMEOUT:
	if (++bp->b_errcnt <= 1)
	    return XHD_RETRY;
	bp->b_error = EIO;
	bp->b_flags |= B_ERROR;
	return XHD_DONE;

    case XHI_SELTO:
    /* B_INVAL is used here to disable any retries during autoconfig or
       any other special SCSI commands
     */
        if(bp->b_flags & B_INVAL)
            return XHD_DONE;
        bp->b_bcount = bp->b_resid;
        bp->b_resid = 0;
        bp->b_error = EIO;;
        bp->b_flags |= B_ERROR;
        return XHD_DONE;

    case XHI_ERR:
	/* Analyze the error */
	if (sdp) {

	    bcopy(sdp, &(xs->xs_esense), sizeof(xs->xs_esense));
	    xs->xs_flags |= XSFL_HASSENSE;

	    switch (sdp->es_key)
	    {
	    case SCES_KEY_NO_SENSE:
		DEBUGF(cserrdebug, printf("csint: NO SENSE KEY\n"););
		/* Check FMK, ILI and EOM bits here */
		sc->sc_blkno++;
		if (sdp->es_fmark)
		{
		    int res = (sdp->es_info[0] << 24) +
			      (sdp->es_info[1] << 16) +
			      (sdp->es_info[2] << 8) +
			       sdp->es_info[3];

		    DEBUGF(cserrdebug, printf("csint: File mark detected\n"););
		    sc->scf_fm = 1;
		    bp->b_bcount -= sc->scf_blkmode ? res*hc->hc_blklen : res;
		    bp->b_resid = 0;
		    if (bp != &rcsbuf[xs->xs_unit]) {
			sc->sc_nxrec = bdbtofsb(bp->b_blkno);
			if (sc->scf_blkmode)
			    sc->sc_nxrec += bdbtofsb(btodb(bp->b_bcount));
		    }
		}
		if (sdp->es_ili) /* illegal length indicated */
		{
		    /*
		     * Get residual count, hope this will do expected things
		     * for most compilers 
		     */
		    int res = (sdp->es_info[0] << 24) +
			      (sdp->es_info[1] << 16) +
			      (sdp->es_info[2] << 8) +
			       sdp->es_info[3];

		    DEBUGF(cserrdebug, 
			   printf("csint: Invalid length indicated, res=%d\n",
			   res););
		    if (res < 0) {	/* record too long for buffer */
			bp->b_resid = bp->b_bcount;
			bp->b_error = EINVAL;
			bp->b_flags |= B_ERROR;
		    } else {		/* record is shorter than buffer */
					/* truncate to record size */
			bp->b_bcount -= sc->scf_blkmode ? res*hc->hc_blklen:res;
			bp->b_resid = 0;
		    }
		}
		if (sdp->es_eom)
		{
		    int res = (sdp->es_info[0] << 24) +
			      (sdp->es_info[1] << 16) +
			      (sdp->es_info[2] << 8) +
			       sdp->es_info[3];

		    DEBUGF(cserrdebug, 
		   	   printf("csint: End of medium detected\n"););
		    sc->scf_eot = 1;
		    bp->b_bcount -= sc->scf_blkmode ? res*hc->hc_blklen : res;
		    bp->b_resid = 0;
		}
		return XHD_DONE;
	    case SCES_KEY_RECOVERED_ERROR:
		DEBUGF(csintdebug, printf("csint: RECOVERED ERROR KEY\n"););
		sc->scf_written = !(bp->b_flags & B_READ);
		sc->sc_blkno++;
		bp->b_resid = 0;
		return XHD_DONE;
	    case SCES_KEY_NOT_READY:
		DEBUGF(cserrdebug, printf("csint: NOT READY KEY\n"););
		bp->b_error = EIO;
		bp->b_flags |= B_ERROR;
		break;
	    case SCES_KEY_MEDIUM_ERROR:
		DEBUGF(cserrdebug, printf("csint: MEDIUM ERROR KEY\n"););
		bp->b_error = EIO;
		bp->b_flags |= B_ERROR;
		break;
	    case SCES_KEY_HARDWARE_ERROR:
		DEBUGF(cserrdebug, printf("csint: HARDWARE ERROR KEY\n"););
		bp->b_error = EIO;
		bp->b_flags |= B_ERROR;
		break;
	    case SCES_KEY_ILLEGAL_REQUEST:
		DEBUGF(cserrdebug, printf("csint: ILLEGAL REQUEST KEY\n"););
		if (++bp->b_errcnt <= 1)
		    return XHD_RETRY;
		else
		{
		    bp->b_bcount = bp->b_resid;
		    bp->b_resid = 0;
		    bp->b_error = EIO;
		    bp->b_flags |= B_ERROR;
		    break;
		}
	    case SCES_KEY_UNIT_ATTENTION:
		DEBUGF(cserrdebug, printf("csint: UNIT ATTENTION KEY\n"););
		if (++bp->b_errcnt <= 1)
		    return XHD_RETRY;
		else
		{
		    bp->b_bcount = bp->b_resid;
		    bp->b_resid = 0;
		    bp->b_error = EIO;
		    bp->b_flags |= B_ERROR;
		    break;
		}
	    case SCES_KEY_DATA_PROTECT:
		DEBUGF(cserrdebug, printf("csint: DATA PROTECT KEY\n"););
		bp->b_error = EIO;
		bp->b_flags |= B_ERROR;
		break;
	    case SCES_KEY_BLANK_CHECK:
		DEBUGF(cserrdebug, printf("csint: BLANK CHECK KEY\n"););
		bp->b_flags |= B_ERROR;
		break;
	    case SCES_KEY_COPY_ABORTED:
		DEBUGF(csintdebug, printf("csint: COPY ABORTED KEY\n"););
		bp->b_flags |= B_ERROR;
		break;
	    case SCES_KEY_ABORTED_COMMAND:
		DEBUGF(csintdebug, printf("csint: ABORTED COMMAND KEY\n"););
		bp->b_flags |= B_ERROR;
		break;
	    case SCES_KEY_VOLUME_OVERFLOW:
		DEBUGF(cserrdebug, printf("csint: VOLUME OVERFLOW KEY\n"););
		bp->b_error = ENOSPC;
		bp->b_flags |= B_ERROR;
		break;
	    default:
		DEBUGF(cserrdebug, 
		       printf("csint: Unknown KEY=%d\n", sdp->es_key););
		break;
	    }
	} else
	    printf("Csint: Sense data pointer is zero\n");
	break;
    }

    /* Make sure the byte count is reasonable.                           */
    if (bp->b_bcount < 0)
	bp->b_bcount = bp->b_resid;

    return XHD_DONE;
}

csread(dev_t dev, struct uio *uio)
{
    int             errno;

    errno = csphys(dev, uio);
    if (errno)
	return (errno);
    return (physio(csstrategy, &rcsbuf[CSUNIT(dev)], dev, B_READ, csminphys, uio));
}

cswrite(dev_t dev, struct uio *uio)
{
    int             errno;

    DEBUGF(csdebug, printf("cswrite: dev=0x%x,uio=0x%x\n", dev, uio););
    errno = csphys(dev, uio);
    if (errno)
	return (errno);
    return (physio(csstrategy, &rcsbuf[CSUNIT(dev)], dev, B_WRITE, csminphys, uio));

}

csphys(dev_t dev, struct uio *uio)
{
    register int    unit;
    struct xha_slave *xs;
    register int    bsize = uio->uio_iov->iov_len;

    DEBUGF(csdebug, printf("csphys: dev=0x%x,uio=0x%x\n", dev, uio););

    unit = CSUNIT(dev);
    if (((xs = csinfo[unit]) == 0)
	|| (xs->xs_alive == 0))
	return (ENXIO);
    if ((bsize > 0xffff)			    /* blocksize limit */
	|| (bsize <= 0))
	return (EINVAL);
    return (0);
}

csstrategy(struct buf *bp)
{
    register struct xha_slave *xs = csinfo[CSUNIT(bp->b_dev)];
    csxstrat(xs,bp);
}

csxstrat(struct xha_slave *xs, struct buf *bp)
{
    register struct buf *dp;
    register int    s;
    register struct xhac_driver *xc;
    register struct cs_softc *sc;
    register struct cs_hardc *hc;
    struct scsi_modeselect *mselp;
    struct scsi_modesense  *msenp;

    DEBUGF(csdebug, printf("csxstrat: bp=0x%x\n", bp););
    sc = &cs_softc[xs->xs_unit];
    hc = &cs_hardc[xs->xs_unit];
    xc = xs->xs_xi->xi_hadriver;
    if (sc->scf_selectmode)
    {
	mselp = (struct scsi_modeselect *) selectbuf[xs->xs_unit];
	bzero(mselp, SIZEOF_MODESELECT);
	mselp->sms_bdl = sizeof mselp->sms_bdarr[0];
#ifdef	FIXED_BLOCKS
	if (!(bp->b_flags & B_PHYS))
	{
	    mselp->sms_bufmode = 1;
	    mselp->sms_bdarr[0].sbd_blklen = bp->b_bcount;
	}
#endif
	mselp->sms_bdarr[0].sbd_blklen  = hc->hc_blklen;
	mselp->sms_bdarr[0].sbd_density  = hc->hc_density;
	mselp->sms_bdarr[0].sbd_nblocks = 0;
	mselp->sms_bufmode = 1;
	sc->scf_selectmode = 0;		    /* have to clear this bit
					     * first, or else an
					     * infinite loop will start */
	if (csmodeselect(xs, mselp) != 0)
	{
	    printf("%s%d: Warning: Mode Select failed\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit);
	}

#ifdef	CSXSTRAT_VERIFY_MODE_SELECT
	msenp = (struct scsi_modesense *) sensebuf[xs->xs_unit];
	if (xxscsicmd(xs, SC0_MODE_SENSE,(caddr_t) msenp,
		SIZEOF_MODESENSE, B_READ) != 0) {
	    printf("cs%d: mode sense failed\n", xs->xs_unit);
	} else {
	    printf("%s%d: nblocks=%d, blklen=%d density=%d bufmode=%d\n",
		xs->xs_driver->xd_sname,xs->xs_unit,
		msenp->sm_bdarr[0].sbd_nblocks,
		msenp->sm_bdarr[0].sbd_blklen,
		msenp->sm_bdarr[0].sbd_density,
		msenp->sm_bufmode);
	}
#endif
    }

    bp->av_forw = NULL;
    dp = &xs->xs_tab;
    s = spl_tape();
    if (dp->b_actf == NULL)
	dp->b_actf = bp;
    else
	dp->b_actl->av_forw = bp;
    dp->b_actl = bp;
	(*xc->xhc_cstart) (xs);
    splx(s);
}

csioctl(dev_t dev, int cmd, caddr_t data, int flag)
{
	register struct cs_softc *sc;
	register struct cs_hardc *hc;
	register struct xha_slave *xs;
	int	csunit = CSUNIT(dev);
	int fcount;
	struct mtop *mtop;
	struct mtget *mtget;
	union scsi_cdb *scp;
	int	smode,rv;
	static u_char csops[] =
	{
		SC0_WRITE_FILEMARKS, /* MTWEOF write an end-of-file record */
		SC0_SPACE,	/* MTFSF forward space file */
		SC0_SPACE,	/* MTBSF backward space file */
		SC0_SPACE,	/* MTFSR forward space record */
		SC0_SPACE,	/* MTBSR backward space record */
		SC0_REWIND,	/* MTREW rewind */
		SC0_LOAD_UNLOAD,/* MTOFFL rewind and put the drive offline */
		SC0_TEST_UNIT_READY, /* MTNOP no operation, sets status only */
		0xFF,		/* MTCACHE enable controller cache */
		0xFF,		/* MTNOCACHE disable controller cache */
		SC0_ERASE,	/* MTERASE erase tape */
		SC0_LOAD_UNLOAD,/* MTRETENSION retension tape */
		SC0_SPACE	/* MTEOM go to end of data */
	};	
	DEBUGF( (csdebug),
		printf("csioctl: entered\n"));

        if(csunit >= NCS || (xs = csinfo[csunit]) == 0 || xs->xs_alive == 0)
        {
	    DEBUGF(csdebug,
		printf("csioctl: bad device %d/%d\n",major(dev),minor(dev)););
	    return ENODEV;
	}

	sc = &cs_softc[xs->xs_unit];
	hc = &cs_hardc[xs->xs_unit];

	switch (cmd) {

	case MTIOCTOP:	/* tape operation */
		mtop = (struct mtop *)data;
		switch (mtop->mt_op) {

		case MTWEOF:
			fcount = mtop->mt_count;
			break;
		case MTNOCACHE:
		case MTCACHE:
			return (ENODEV);
		case MTFSR:
		case MTFSF:
			fcount = mtop->mt_count;
			break;
		case MTBSR:
		case MTBSF:
			fcount = -mtop->mt_count;
			break;
		case MTERASE:
			fcount = 1;
			break;
		case MTREW: 
			fcount = 0;
			break;
		case MTOFFL: 
			fcount = 0;  /* Becomes !Re-Ten & !Load */
			break;
		case MTNOP: 
			fcount = 0;
			break;
		case MTEOM: 
			fcount = 0;
			break;
		case MTRETENSION:
			fcount = 0x3; /* Becomes Re-Ten | Load */
			break;
		case MTBLK:
			if (mtop->mt_count < 0 ||
				mtop->mt_count < hc->hc_minblklen ||
				mtop->mt_count > hc->hc_maxblklen)
			    return (EINVAL);
			else
			    hc->hc_blklen = mtop->mt_count;
			return (0);
			break;
		default:
			return (ENODEV);
		}
		DEBUGF(csdebug,
			printf("cmd=0x%x fcount=%d ", csops[mtop->mt_op], fcount));
		scp = xxgetscbuf(xs);
		xxfillscbuf(scp,xs->xs_lun,csops[mtop->mt_op], fcount);
		switch(csops[mtop->mt_op])
		{
			case SC0_SPACE:
				if(fcount < 0)
				{
					scp->cdb_6.sc_lbarest[0] = 0xff;
					scp->cdb_6.sc_lbarest[1] = 0xff;
				}
		/* For scsi-2 the "scp->cdb_6.sc_lbaMSB" field is 
		   defined as follows:

			Code         Description          Support
		      ---------  --------------------    ---------
			000b     Blocks                  Mandatory
			001b     Filemarks               Mandatory
			010b     Sequential Filemarks    Optional
			011b     End-of-Data             Optional
			100b     Setmarks                Optional
			101b     Sequential Setmarks     Optional
		      110b-111b  Reserved
		*/
				switch (mtop->mt_op) {
					case MTFSR:  /* Blocks */
					case MTBSR:
						scp->cdb_6.sc_lbaMSB |= 0x0;
						break;
					case MTBSF:  /* Filemarks */
					case MTFSF:
						scp->cdb_6.sc_lbaMSB |= 0x1;
						break;
					case MTEOM:  /* End-of-Data */
						scp->cdb_6.sc_lbaMSB |= 0x3;
						break;
					default:
						panic("cs: unknown mt command for scsi SPACE command");
				}
				break;
			case SC0_REWIND:
#ifdef	dont_wait_for_rewind
				scp->cdb_6.sc_lbaMSB |= 1; /*IMMEDIATE bit set*/
#else
				scp->cdb_6.sc_lbaMSB |= 0;
#endif
				break;
			case SC0_WRITE_FILEMARKS:
			case SC0_LOAD_UNLOAD:
			case SC0_TEST_UNIT_READY:
				break;
			case SC0_ERASE:
				scp->cdb_6.sc_lbaMSB |= 0x1; /* Long erase */
				break;
			default:
				break;
		}
		smode = sc->scf_selectmode;
		sc->scf_selectmode = 0; /* Disable mode select */
		rv = cssccom(xs,scp,1,0,0);
		sc->scf_selectmode = smode;
		return rv;

	case MTIOCGET:
		mtget = (struct mtget *)data;
		mtget->mt_dsreg = 0;
		/*
		mtget->mt_erreg = sc->sc_data[0] + (sc->sc_data[1] << 8);
		*/
		mtget->mt_resid = sc->sc_resid;
		mtget->mt_blkno = hc->hc_blklen;
		mtget->mt_type = MT_ISCS;
		break;
	case DKIOCSMODESELECT:
		smode = sc->scf_selectmode;
		sc->scf_selectmode = 0; /* Disable (automatic) mode select */
		if(csmodeselect(xs,(struct scsi_modeselect *)data) < 0)
		{
		    sc->scf_selectmode = smode;
		    return -1;
		}
		sc->scf_selectmode = smode;
		return 0;

	case DKIOCSMODESENSE:
		smode = sc->scf_selectmode;
		sc->scf_selectmode = 0; /* Disable mode select */
		if(csmodesense(xs,(struct scsi_modesense *) data) < 0)
		{
		    sc->scf_selectmode = smode;
		    return -1;
		}
		sc->scf_selectmode = smode;
		return 0;

	case DKIOCGSCSISENSE:
		if(xs->xs_flags & XSFL_HASSENSE) {
		    bcopy(&xs->xs_esense,data,sizeof(xs->xs_esense));
		    xs->xs_flags ^= XSFL_HASSENSE;
		} else {
		    bzero(data, sizeof(xs->xs_esense));
		}
		return 0;

	default:
		return (ENOTTY);
	}
	return (0);
}

void
csminphys(struct buf *bp)
{
    register struct iocc_ctlr *ic;
    register struct xha_slave *xs;
    extern unsigned (*dma_get_minphys()) ();

    minphys(bp);				    /* Enforce kernel-wide
						     * b_bcount restriction */

    if (bp->b_bcount > 65536)
	bp->b_bcount = 65536;

    xs = csinfo[CSUNIT(bp->b_dev)];

    ic = xs->xs_xi->xi_hd->xh_iocc;
    (*dma_get_minphys(ic)) (bp);		    /* dma's bcount
						     * restrictions */

}

#endif /* NCS > 0 */
