/*-
 * Copyright (c) 1992, 1993 Erik Forsberg.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 * NO EVENT SHALL I BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This driver is based on NetBSD lms.c, version:
 *	_Id: lms.c,v 1.7 1993/12/20 09:06:17 mycroft Exp _
 */
#if !defined(lint) && !defined(NO_RCS_HDRS)
static char *rcsid = "$Header: /sys/rt/dev/RCS/lms.c,v 1.3 1994/05/22 12:26:24 roger Exp $";
#endif

#include "lms.h"
#if NLMS > 0

#include <sys/param.h>
#include <sys/conf.h>
#include <sys/time.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/malloc.h>
#include <sys/ioctl.h>
#include <sys/tty.h>
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/proc.h>
#include <sys/vnode.h>
#include <sys/errno.h>

#include <machine/io.h>
#include "rt/include/ioccvar.h"
#include "rt/rt/debug.h"

/* #include "../rtio/mouse.h" */

#define BUTSTATMASK	0x07	/* Any mouse button down if any bit set */

struct lmsdevice {
        u_char data;	/* Offset for data port, read-only */
        u_char sign;	/* Offset for signature port, read-write */
        u_char cntrl;	/* Offset for control port, write-only */
        u_char config;	/* for configuration port, read-write */
};

#define LMSUNIT(dev) (minor(dev) & 0x0f)
#define LMSLINE(dev) (minor(dev) >> 4)

static struct lms_softc {	/* Driver status information */
	struct lmsdevice *lmsaddr;/* hardware address */
	u_char state;		/* Mouse driver state */
	u_char status;		/* Mouse button status */
	u_char button;		/* Previous mouse button status bits */
	int x, y;		/* accumulated motion in the X,Y axis */
} lms_softc[NLMS];

#define OPEN	1		/* Device is open */
#define ASLP	2		/* Waiting for mouse data */

/* struct isa_driver lmsdriver = { lmsprobe, lmsattach, "lms" }; */

int lmsprobe(), lmsattach(), lmsintr();

caddr_t lmsstd[] = { (caddr_t)0xf000023c, (caddr_t)0xf0000238, 0 };

struct iocc_device *lmsinfo[NLMS];

struct iocc_driver lmsdriver = {
	/* probe, slave, attach, dgo, addr, dname, dinfo, mname, csr, intr */
	lmsprobe, 0, lmsattach, 0, lmsstd, "lms", lmsinfo, 0, 0, lmsintr 
};

struct tty lms_tty[NLMS];

#define outb IOOUT
#define inb IOIN

int 
lmsprobe(addr)
	register caddr_t addr;
{
	int val;
        register struct lmsdevice *lmsp = (struct lmsdevice *)addr;

	/* Configure and check for port present */

	outb(&lmsp->config, 0x91);
	DELAY(10);
	outb(&lmsp->sign, 0x0C);
	DELAY(10);
	val = inb(&lmsp->sign);
	DELAY(10);
	outb(&lmsp->sign, 0x50);

	/* Check if something is out there */

	if (val == 0x0C && inb(&lmsp->sign) == 0x50) {

		/* Enable Bus Mouse interrupts */
		outb(&lmsp->cntrl, 0);

		/* wait around for an interrupt.... */
		PROBE_DELAY(10000);

		/* Disable mouse interrupts */
		outb(&lmsp->cntrl, 0x10);

		return(PROBE_OK);
	}

	/* Not present */

	return(PROBE_BAD);
}

int
lmsattach(iod)
        register struct iocc_device *iod;
{
	int unit = iod->iod_unit;
	struct lms_softc *sc = &lms_softc[unit];

	/* Save I/O base address */
        sc->lmsaddr = (struct lmsdevice *)(iod->iod_addr);

	/* Disable mouse interrupts */

	outb(&sc->lmsaddr->cntrl, 0x10);

	/* Setup initial state */

	sc->state = 0;

	/* Done */

	return(0);
}

int 
lmsopen(dev, flag)
        dev_t dev;
        int flag;
{
	register struct iocc_device *iod;
	register struct tty *tp;
	int unit = LMSUNIT(dev);
	struct lms_softc *sc;
	int error;

	/* Validate unit number */

	if (unit >= NLMS)
		return(ENXIO);

	/* Get device data */

	sc = &lms_softc[unit];
	iod = lmsinfo[unit];

	/* If device does not exist */

	if (iod == 0 || iod->iod_alive == 0)
		return (ENXIO);

	/* Disallow multiple opens */

        tp = &lms_tty[unit];
        tp->t_oproc = tp->t_param = 0;
        if ((tp->t_state & TS_ISOPEN) == 0) {
                ttychars(tp);
                tp->t_state = TS_ISOPEN | TS_CARR_ON;
		tp->t_dev = dev;

	    /* Initialize state */

	    sc->state |= OPEN;
	    sc->status = 0;
	    sc->button = 0;
	    sc->x = 0;
	    sc->y = 0;

	    /* Enable Bus Mouse interrupts */

	    outb(&sc->lmsaddr->cntrl, 0);

        } else 
		return (EBUSY);

        if (error = (*linesw[tp->t_line].l_open)(unit, tp)) {
                lmsclose(dev);
                return (error);
        }

	/* Successful open */

	return(0);
}

int 
lmsclose(dev, flag)
        dev_t dev;
        int flag;
{
	int unit = LMSUNIT(dev);
	struct lms_softc *sc = &lms_softc[unit];
        register struct tty *tp = &lms_tty[unit];
	int s = spltty();

	/* Disable further mouse interrupts */

	outb(&sc->lmsaddr->cntrl, 0x10);

	/* Complete the close */

	sc->state &= ~OPEN;

        (*linesw[tp->t_line].l_close)(tp);
        ttyclose(tp);
        tp->t_state = 0;

	/* close is almost always successful */

	(void) splx(s);
	return(0);
}

/*ARGSUSED*/
lmsread(dev, uio, flag)
        register dev_t dev;
        register struct uio *uio;
        int flag;
{
        register unit = LMSUNIT(dev);
        register struct tty *tp;

        tp = &lms_tty[unit];

        return ((*linesw[tp->t_line].l_read)(tp, uio, flag));
}

int 
lmsioctl(dev, cmd, addr, flag)
        dev_t dev;
	int cmd;
	caddr_t addr;
	int flag;
{
/* 	struct mouseinfo info; */
	struct lms_softc *sc;
	int s, error = 0, unit = LMSUNIT(dev);
        register struct tty *tp;

	/* Get device information */

	sc = &lms_softc[unit];
        tp = &lms_tty[unit];

	/* Perform IOCTL command */

	switch (cmd) {

#if 0	/* We dont care about this right now... */
	case MOUSEIOCREAD:

		/* Don't modify info while calculating */

		s = spltty();

		/* Build mouse status octet */

		info.status = sc->status;
		if (sc->x || sc->y)
			info.status |= MOVEMENT;

		/* Encode X and Y motion as good as we can */

		if (sc->x > 127)
			info.xmotion = 127;
		else if (sc->x < -127)
			info.xmotion = -127;
		else
			info.xmotion = sc->x;

		if (sc->y > 127)
			info.ymotion = 127;
		else if (sc->y < -127)
			info.ymotion = -127;
		else
			info.ymotion = sc->y;

		/* Reset historical information */

		sc->x = 0;
		sc->y = 0;
		sc->status &= ~BUTCHNGMASK;

		/* Allow interrupts and copy result buffer */

		splx(s);
		error = copyout(&info, addr, sizeof(struct mouseinfo));
		break;
#endif

	default:
                /* TTY ioctls */
                error = (*linesw[tp->t_line].l_ioctl)(tp, cmd, addr, flag);
                if (error >= 0)
                        return (error);
                error = ttioctl(tp, cmd, addr, flag);
                if (error < 0)
                        error = ENOTTY;
        }

	/* Return error code */

	return(error);
}

int 
lmsintr(unit)
	int unit;
{
	struct lms_softc *sc = &lms_softc[unit];
	char hi, lo, dx, dy, buttons, changed;
        register struct tty *tp = &lms_tty[unit];
	register struct iocc_device *iod = lmsinfo[unit];
        register struct lmsdevice *lmsp = sc->lmsaddr;
	extern int cold;

	/* XXX: (temp?) hack to guarantee that this card gets shutdown... */
	if (!cold && !(sc->state & OPEN)) {
		/* Disable further mouse interrupts */
		outb(&lmsp->cntrl, 0x10);
		return (0);
	}

	outb(&lmsp->cntrl, 0xAB);
	hi = inb(&lmsp->data) & 15;
	outb(&lmsp->cntrl, 0x90);
	lo = inb(&lmsp->data) & 15;
	dx = (hi << 4) | lo;
	dx = (dx == -128) ? -127 : dx;

	outb(&lmsp->cntrl, 0xF0);
	hi = inb(&lmsp->data);
	outb(&lmsp->cntrl, 0xD0);
	lo = inb(&lmsp->data);
	outb(&lmsp->cntrl, 0);
	dy = ((hi & 15) << 4) | (lo & 15);
	dy = (dy == -128) ? 127 : -dy;

	buttons = (~hi >> 5) & 7;
	changed = buttons ^ sc->button;
	sc->button = buttons;
	sc->status = buttons | (sc->status & ~BUTSTATMASK) | (changed << 3);

	/* Update accumulated movements */

	sc->x += dx;
	sc->y += dy;

	/* If device in use and a change occurred... */

	if ( ! (dx || dy || changed) )	/* nothing changed, But... */
		return (0);		/* we cant share interrupts... */

	if (sc->state & OPEN) {
		/* send a 5 byte stream to the line discipline */
                (*linesw[tp->t_line].l_rint)((0x80|(buttons ^ BUTSTATMASK)),tp);
                (*linesw[tp->t_line].l_rint)(dx,tp);
                (*linesw[tp->t_line].l_rint)(dy,tp);
                (*linesw[tp->t_line].l_rint)(0,tp);
                (*linesw[tp->t_line].l_rint)(0,tp);
	}
	return (0);
}

int lmsselect(dev, rw)
	dev_t dev;
	int rw;
{
        register int unit = LMSUNIT(dev);
	struct lms_softc *sc = &lms_softc[unit];
        register struct tty *tp = &lms_tty[unit];
	int s, ret;

	/* Silly to select for output */
	if (rw == FWRITE)
		return(0);

        /*
         * Tablets don't queue things in the same way as normal tty stuff
         * so need to call its specific routine if it is a tablet. Must call
         * explicitly since there isn't a select entry in linesw.
         */
        tp = &lms_tty[unit];
        if(tp->t_line == TABLDISC)
                return(tbselect(dev, rw));
	return ttselect(dev, rw);
}
#endif
