#include "xt.h"
#if NXT > 0 
/*
 * Copyright (C) 1988, 1989 Christer Bernerus, 
 *     		 Chalmers University of Technology, Gothenburg, SWEDEN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms as is or as a source
 * for derivative work are permitted provided that the above copyright 
 * notice and this paragraph are  duplicated in all such forms 
 * and that any documentation, advertising materials,
 * and other materials related to such distribution
 * and use acknowledge that the software was developed
 * at Chalmers University of Technology, SWEDEN.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *
 * Maxtor XT-4380S SCSI Disk driver
 *
 * $Source: /usr/src/sys/rt/dev/RCS/xt.c,v $
 * $Revision: 1.11 $ $Date: 1994/07/11 20:17:52 $
 * $Author: md $ $State: Exp $
 *
 * $Log: xt.c,v $
 * Revision 1.11  1994/07/11  20:17:52  md
 * Add an entry fo a Seagate ST41600N 1.6GB drive.
 * Print info about synchronous modes supported by devices.
 *
 * Revision 1.10  1994/06/11  20:26:19  roger
 * changes for the '44 fcntl.h.
 *
 * Revision 1.9  1994/05/22  12:24:51  roger
 * header file changes.
 *
 * Revision 1.8  1994/05/07  17:45:20  md
 * Add entry for Segate WREN 7 drive.
 * Set the number of sectors per cylinder to the amount usable, not the
 * actual (physical) number.
 *
 * Revision 1.7  1994/04/28  16:32:12  md
 * Return proper name for d_typename when asked for disk information.
 * Return correct block offsets for partitions.
 *
 * Revision 1.6  1994/04/20  19:04:17  md
 * Add support for immutable and append only files.
 *
 * Revision 1.5  1994/02/07  15:06:16  md
 * Fix firmware rev and date string being printed (was being cut short).
 *
 * Revision 1.4  1992/12/09  10:57:41  md
 * change reference of machineio to rtio.
 *
 * Revision 1.3  1992/12/09  09:41:52  md
 * Add back in debugging hooks.
 *
 * Revision 1.2  1992/12/05  20:47:10  md
 * dk_wpms is now initialized correctly based on drive geometry.
 *
 * Revision 1.1  1992/11/05  21:11:45  md
 * Initial revision
 *
 * Revision 3.1  90/03/05  15:22:14  bernerus
 * The Adaptec ACB-4525 does NOT support SCSI command linking. The
 * device table is updated accordingly.
 * A new debug variable xterrdebug enables debugging printouts on IO errors.
 * The stuff dealing with drive starting is moved to xtslave so that the
 * results can be properly returned to autoconf.
 * A minor bug in xtclose caused xs_flags not to be reset upon close. Won't
 * bite you unless you use SCSI pass-through.
 * SCSI-pass through mode will now be turned off if a timeout occurs.
 * Medium errors and hardware errors are now retried 5 times before giving up.
 * 
 * Revision 2.8  89/11/15  14:54:43  bernerus
 * Now uses the SCSI command READ_CAPACITY for determining the size
 * of the "c" partition and the logical block size.
 * 
 * The dkblock() macro is now used for getting the disk block number
 * from the buffer.
 * 
 * Some functions used during autoconfig have been somewhat generalized
 * and moved to xxc.c.
 * 
 * All functions that previously used the xha_device pointer (xi) for
 * pointing out a specific device now uses the xha_slave pointer (xs)
 * instead.
 * 
 * When computing the associated cylinder, the driver avoids dividing
 * by zero if nspc should be zero. Instead the block number will be
 * given to disksort, but this won't cause any more trouble than bad
 * efficiency.
 * 
 * When a buffer with bcount==0 arrives at xstart, a SCSI READ or
 * WRITE with block count == 0 was emitted. Unfortunately the SCSI
 * standard has a quirk here and 256 blocks will be read or - even
 * worse- written. The last case will result in a corrupted disk.
 * Now, xtstart will in this case emit a SCSI READ LONG or WRITE LONG
 * which doesn't have this quirk. Since a buffer with bcount==0 really
 * shouldn't appear, and the cause has not been investigated, a (quite
 * verbose) printout will be done on the console to aid in the search
 * of this irritating bug.
 * 
 * If a buffer given to xtint, no retries will be performed.
 * 
 * The debug printouts now prints which function they belong to, this
 * makes debugging a little easier. Also a variable "xttrace" may be
 * set which enables the entry debug printouts.
 * 
 * Xtstrategy checks that a given buffer isn't already in the queue,
 * when this happens, a printout is done, that printout is now fixed
 * so that the buffer address will really be written.
 * 
 * The macro XTCUNIT is not used anymore and has been deleted.
 * 
 * The usage of the macro XTUNIT has been reduced. Instead the info
 * in the xha_slave structure is used.
 * 
 * Severe checking of the backpointers from the xha_slave structure
 * is now done in xtminphys in order to reveal any bugs in the autoconfig
 * process.
 * 
 * SCSI CDB's are now uniformly pointed to by using a (union scsi_cdb *) type.
 * 
 * Revision 2.7  89/08/22  08:32:45  bernerus
 * SCSI command linking capability is now a feature that is variable
 * per drive type.
 * Added display trace of the spl's.
 * 
 * Revision 2.6  89/05/23  12:52:08  bernerus
 * The code for starting disks didn't work as expected. Now it does.
 * 
 * Revision 2.5  89/05/23  09:26:57  bernerus
 * Now handles sector sizes different from 512 (untested).
 * Spin up/down and automatic spin up capability added.
 * Probe routine now uses SCSI Inquiry instead of Test Unit Ready
 * to determine existence of a disk. It also only acctpts DASD types.
 * The slave routine now attempts to start a drive that isn't ready
 * before doing mode sensing.
 * 
 * Revision 2.4  89/02/24  13:09:42  bernerus
 * Changed the copyright notice.
 * 
 * Revision 2.3  89/02/08  13:07:11  bernerus
 * Partial sectors queue was horribly large, now a lot smaller.
 * We now do a SCSI Inquiry to get the disk type, and a Mode sense to
 * get the disk's geometries, thus making the driver less device dependent.
 * 
 * Revision 2.2  89/01/30  08:01:10  bernerus
 * Added minidisk support.
 * 
 * Revision 2.1  89/01/18  09:18:31  bernerus
 * Added some minidisk support.
 * 
 * Revision 2.0  89/01/16  11:11:58  bernerus
 * Added copyright notice.
 * Changed names xu->xt, xt->xtc.
 * 
 *
 */
#include <sys/param.h>
#include <sys/vm.h>
#include <sys/buf.h>
#include <sys/time.h>
#include <sys/proc.h>
#include <sys/errno.h>
#include <machine/io.h>
#include <sys/file.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/cmap.h>
#include <sys/dkstat.h>
#include <sys/ioctl.h>
#include <sys/disklabel.h>
#include <machine/mmu.h>
#include <machine/dkio.h>

#include "rt/include/ioccvar.h"
#include "rt/include/scsi.h"
#include "rt/include/xhavar.h"
#include "rt/rt/debug.h"
#include "rt/dev/scsiconfig.h"
#include <ufs/fs.h>

#define MAXNPART 8
#define MAXQPERUNIT 2
#define MAXNBPS	        4096     
/* #define DSPL		/* show status on LED's */
#ifndef DSPL
#define dsplbio(x) splbio()
#define dsplx(x,y) splx(x)
#define dsplhigh(x) splhigh()
#endif /* DSPL */

#define PARANOID

/* THIS SHOULD BE READ OFF THE PACK, PER DRIVE */
/* 4380S Geometries :
	36	Sectors/Track 512 bytes/sector
	18360   tracks totally and
	540     sectors/cylinder
*/
struct size {
	struct partentry {
	    daddr_t nblocks;
	    int     pblkoff;
	}
	partition[MAXNPART];
};

struct size
def_sizes = {
        15884,  0,              /* A=cyl 0 thru 29 */
        33480,  16200,          /* B=cyl 30 thru 91 */
        621180,   0,            /* C=cyl 0 thru 1217 */
        16740,  49680,          /* D=cyl 92 thru 122 */
	0,	0,		/* E = nothing */
	0,	0,		/* F = nothing */
	82620, 538380,		/* G = cyl 997 thru 1149 */
        471960,  66420,		/* H = cyl 123 thru  996 */
};

struct size
xt43_sizes = {
        15884,  0,              /* A=cyl 0 thru 29 */
        33480,  16200,          /* B=cyl 30 thru 91 */
        621180,   0,            /* C=cyl 0 thru 1217 */
        16740,  49680,          /* D=cyl 92 thru 122 */
	0,	0,		/* E = nothing */
	0,	0,		/* F = nothing */
	82620, 538380,		/* G = cyl 997 thru 1149 */
        471960,  66420,		/* H = cyl 123 thru  996 */
};
struct size
xt42_sizes = {
        16236,  0,              /* A=cyl 0 thru 40 */
        33660,  16236,          /* B=cyl 41 thru 125 */
        484704,   0,            /* C=cyl 0 thru 1223 */
        16236,  49896,          /* D=cyl 125 thru 165 */
	0,	0,		/* E = nothing */
	0,	0,		/* F = nothing */
	82368, 373032,		/* G = cyl 941 thru 1148 */
};
struct size
xt41_sizes = {
        16128,  0,              /* A=cyl 0 thru 63 */
        33516,  16128,          /* B=cyl 64 thru 196 */
        308448,   0,            /* C=cyl 0 thru 1223 */
        16128,  49680,          /* D=cyl 197 thru 260 */
	0,	0,		/* E = nothing */
	0,	0,		/* F = nothing */
	82404, 538380,		/* G = cyl 821 thru 1147 */
        141120,  66420,		/* H = cyl 261 thru  820 */
};
/* HW added Wren default partitions for wren (minidisk will override ) */
/* Note these partition sizes are not clever, just somewhat compatible
   with history .... */
/* Wren V Geometries :
        1541    Cylinders
        15      Tracks
        51      Sectors/Track 512 bytes/sector
        23115   tracks totally and
        765     sectors/cylinder
*/

struct size
wv15_sizes = {
#if     CS_GENERIC
        (  55-  1)*51*15,   1*15*51,  /* A=cyl   1 thru   54 */
                      0,    0*15*51,  /* B */
        (1540-  0)*51*15,   0*15*51,  /* C=cyl   0 thru 1540 */
        ( 162- 55)*51*15,  55*15*51,  /* D=cyl  55 thru  161 */
        ( 265-162)*51*15, 162*15*51,  /* E=cyl 162 thru  264 */
                      0,    0*15*51,  /* F */
        (1534-  1)*51*15,   1*15*51,  /* G=cyl   1 thru 1534 */
        (1534-265)*51*15, 265*15*51,  /* H=cyl 265 thru 1534 */
#else   CS_GENERIC
/*
------------------------------------------------------------------------------
| a |  b  | d |    h     |                      g                            |
------------------------------------------------------------------------------
|                                c                                           |
------------------------------------------------------------------------------
*/
        (  29-  1)*51*15,   1*15*51, /* A=cyl 1 thru 29 */
        (91-   30)*51*15,  30*15*51, /* B=cyl 30 thru 91 */
        (1540-  1)*51*15,   1*15*51, /* C=cyl 1 thru 1540 */
        ( 122- 92)*51*15,  55*15*51, /* D=cyl 92 thru 122 */
                       0,   0*15*51, /* E = nothing */
                       0,   0*15*51, /* F = nothing */
        (1540-997)*51*15, 997*15*51, /* G = cyl 997 thru 1540 */
        (966- 123)*51*15, 123*15*51, /* H = cyl 123 thru  996 */
#endif  CS_GENERIC
};

struct size
wvi15_sizes = {
/*
------------------------------------------------------------------------------
| a |  b  | d |    h     |                      g                            |
------------------------------------------------------------------------------
|                                c                                           |
------------------------------------------------------------------------------
*/
        (  29-  1)*53*15,   1*15*53, /* A=cyl 1 thru 29 */
        (91-   30)*53*15,  30*15*53, /* B=cyl 30 thru 91 */
        (1627-  1)*53*15,   1*15*53, /* C=cyl 1 thru 1627 */
        ( 122- 92)*53*15,  55*15*53, /* D=cyl 92 thru 122 */
                       0,   0*15*53, /* E = nothing */
                       0,   0*15*53, /* F = nothing */
        (1627-997)*53*15, 997*15*53, /* G = cyl 997 thru 1627 */
        (966- 123)*53*15, 123*15*53, /* H = cyl 123 thru  996 */
};
struct size
fj63_sizes = {
/*
------------------------------------------------------------------------------
| a |  b  | d |    h     |                      g                            |
------------------------------------------------------------------------------
|                                c                                           |
------------------------------------------------------------------------------
*/
        (  29-  1)*53*15,   1*15*53, /* A=cyl 1 thru 29 */
        (91-   30)*53*15,  30*15*53, /* B=cyl 30 thru 91 */
        (1654-  1)*53*15,   1*15*53, /* C=cyl 1 thru 1627 */
        ( 122- 92)*53*15,  55*15*53, /* D=cyl 92 thru 122 */
                       0,   0*15*53, /* E = nothing */
                       0,   0*15*53, /* F = nothing */
        (1654-997)*53*15, 997*15*53, /* G = cyl 997 thru 1627 */
        (966- 123)*53*15, 123*15*53, /* H = cyl 123 thru  996 */
};

struct size
wren7_sizes = {
        1050000,	0,	/* A=cyl 0 thru 1000 */
	0,		0,	/* B = nothing */
        2026500,	0,	/* C=cyl 0 thru 1930 */
        976500,	  1050000,	/* D=cyl 1000 thru 1930 */
	0,		0,	/* E = nothing */
	0,		0,	/* F = nothing */
	0,		0,	/* G = nothing */
	0,		0,	/* H = nothing */
};

struct size
st41600n_sizes = {
        1275000,	0,	/* A=cyl 0 thru 1000 */
	0,		0,	/* B = nothing */
        2674950,	0,	/* C=cyl 0 thru 2098 */
        1399950,  1275000,	/* D=cyl 1000 thru 2098 */
	0,		0,	/* E = nothing */
	0,		0,	/* F = nothing */
	0,		0,	/* G = nothing */
	0,		0,	/* H = nothing */
};
/* The structure below is copied into the softc structure on attach */
struct  xtst {
	char	inqname[17];
        short   nsect;
        short   ntrak;
        short   nspc;
        short   ncyl;
	short	nbps;
        struct  size *defsizes;
	char	typename[8];
	short	flags;
#define XTFL_NO_LINK	0	/* SCSI Command linking not supported */
#define XTFL_LINK	1	/* SCSI Command linking supported */
} xtst[] = {
/* inqname,nsect,ntrak, nspc,ncyl, nbps, def.sizes, typename    */
{"Default ",  34,  15, 34*15,1218, 512, &def_sizes,  "default", XTFL_NO_LINK},
{"8760S",     54,  15, 34*15,1629, 512, &def_sizes,  "xt8760s", XTFL_LINK},
{"XT-4380S",  34,  15, 34*15,1218, 512, &xt43_sizes, "xt4380s", XTFL_LINK},
{"XT-4280S",  34,  11, 34*11,1218, 512, &xt42_sizes, "xt4280s", XTFL_LINK},
{"XT-4170S",  34,   7, 34* 7,1218, 512, &xt41_sizes, "xt4170s", XTFL_LINK},
{"ST41600N",  75,  17, 75*17,2098, 512, &st41600n_sizes, "st41600n", XTFL_LINK},
{"94181-15",  51,  15, 51*15,1541, 512, &wv15_sizes, "wv15", XTFL_NO_LINK},
{"94191-15",  53,  15, 53*15,1627, 512, &wvi15_sizes,"wvi15",XTFL_LINK},
{"94601-15",  70,  15, 70*15,1930, 512, &wren7_sizes,"Wren7",XTFL_LINK},
{"M2263S-512",53,  15, 53*15,1654, 512, &fj63_sizes,"m2263S",XTFL_LINK},
{"ACB-4525",  51,  15, 51*15,1627, 512, &def_sizes,"acb4525",XTFL_NO_LINK},
{"ACB-4525Z",  51,  15, 51*15,1627, 512, &def_sizes,"acb4525z",XTFL_LINK},
};

struct  xt_softc {
	int	sc_busy:1;		/* Set when BUSY is detected, reset otherwise; */
	int	sc_nomap:1;	        /* Don't map or check partitions */
	int	sc_cinited:1;		/* True if c part'n is initialized */
	int	sc_spundown:1;		/* Drive has been spun down */
	int	sc_autospinup:1;	/* Spin up automatically	*/
	int	sc_noretries:1;		/* Don't do retries on SCSI pass thru */
	int	sc_busyretries;		/* No of retries for busy drive */
	int	sc_queuedepth;		/* Current queue depth */
	struct	partsec
	{
	    char	ps_phys:1;	/* Xfer had PHYS flag set in buffer */
	    struct buf *ps_bp;		/* Associated buffer zero if unused */
	    caddr_t	ps_addr;	/* Original b_addr in buf */
	    long	ps_bcount;	/* Original bcount in buf */
	    long	ps_nbps;	/* Sector size of the disk */
	    char	ps_space[MAXNBPS];	/* Data space */
	} sc_part[MAXQPERUNIT];
	struct	scsi_capdata	sc_cdata;	/* Capacity data */
	struct	xtst sc_xtst;
	struct	size sc_size;		/* Actual partitioning */
} xt_softc[NXT];


/* Functions predeclarations */
#include "rt/dev/xxc.h"

int xtprobe(caddr_t ,struct xha_device *);
int xtattach(struct xha_slave *);
int xtttl(union scsi_cdb *);
int xtmodesense(struct xha_slave *, struct scsi_msiobuf *);
int xtstartstop(struct xha_slave *,short);
int xtreadcapacity(struct xha_slave *, struct scsi_capdata *);
int xtslave(struct xha_slave *,caddr_t);
int xtgettype(struct xha_slave *, char **, char *);
int xtopen(dev_t,int);
int xtclose(dev_t,int);
int xtpartitioninit(struct xha_slave *, struct xt_softc *);
int xtxstrat(struct xha_slave *,int , struct buf *);
int xtstrategy(struct buf *);
int xtustart(struct xha_slave *);
int xtfillpart(struct partsec *);
int xtint(struct xha_slave *, struct buf *, int, struct scsi_esense *, u_char);
int xtpartdone(struct xha_slave *, struct buf *);
int xtemptypart(struct partsec *);
int xtread(dev_t, struct uio *);
int xtwrite(dev_t, struct uio *);
int xtphys(dev_t dev);
int xtioctl(dev_t, int, caddr_t, int);
int xtminphys(struct buf *);
int xttryretry(struct xha_slave *, struct buf *, int,int);
int xtgetpartitions(struct xha_slave *);
int xtidcpy(u_char *,char *,int);
void xtpartsetup(struct xha_slave *, struct buf *, int, int);
union scsi_cdb *xtstart(struct xha_slave *,struct buf *,union scsi_cdb *);
struct partsec * xtgetps(struct xha_slave *, struct buf *);
unsigned xtget3(char *);

struct partsec *
xtgetps(struct xha_slave *xs, struct buf *bp)
{
        struct xt_softc *sc = &xt_softc[xs->xs_unit];
	
	struct partsec *ps;

	for(ps = sc->sc_part; ps < &(sc->sc_part[MAXQPERUNIT]); ps++)
		if(ps->ps_bp == bp) return ps;
	return NULL;
}
#define PARTIAL(bp)	(((bp)->b_un.b_addr > (caddr_t) xt_softc) && ((bp)->b_un.b_addr < (caddr_t) &xt_softc[NXT]))

/*
 * Since we want to have more than one transfer on each unit going on
 * "simultaneously", we have to track retry counts in each buffer instead
 * of doing it in the device's buffer queue header. This means that a suitable
 * variable must be found that isn't needed during I/O. The only one we cound
 * find is b_error. Unfortunately this variable isn't cleared for transfers
 * coming from the swap system so we clear it here in the device driver since
 * we didn't want to touch the swap system's code.
 * In ordinary device drivers the retry counting is done in the b_resid variable
 * in the buffer queue head (which is just a header). The definition for this is
 * in ../h/buf.h which defines b_errcnt to be b_resid. The b_resid variable
 * can't be used in the transfer buffers since we use it for indicating cylinder
 * number to disksort(), and thus can't easily be changed.
 * When a transfer is completed, b_error will regain its usual function.
 */ 
#define b_cylin b_resid
#undef b_errcnt
#define b_errcnt b_error

#define dkblock(bp)	((bp)->b_blkno)


struct size xtsizes[NXT];


/* Bits in minor device */

#define XTUNIT(dev)	XHAUNIT(dev)
caddr_t real_buf_addr();

struct xha_device *xtcinfo[NXTC];
struct xha_slave *xtinfo[NXT];
struct buf rxtbuf[NXT];

#ifdef DEBUG
int     xtdebug		= 0;
int	xttrace		= 0;
int     xtustartdebug	= 0;
int     xtstartdebug	= 0;
int     xtintdebug	= 0;
int     xterrdebug	= 0;
int	xtstrategydebug = 0;
int	xtminphysdebug  = 0;
int	xtpartial	= 0;
int	xtgetpartdebug  = 0;
int	xtioctldebug    = 0;
#endif /* DEBUG */
int	xtdisksort 	= 1;	/* Use disksort */
int	xtdolinks	= 1;	/* Use SCSI linked commands */


struct xha_driver xtcdriver =
{
	xtprobe, xtslave, 
	(int (*)()) xtattach, 
	xtustart,
	xtstart,
	xtint,
	"xtc","xt",xtcinfo,xtinfo,
	xtttl,
	EARLY_INT,
};

struct scsi_msiobuf xtmsiobuf;
struct	scsi_ms xtmsbuf;

xtprobe(caddr_t reg,struct xha_device *xi)
{
    DEBUGF(xttrace, printf("Xxprobe: xi=%x\n",xi););
    return xxprobe(xi, &xtcdriver, SCPDT_DASD);
}
char *responses[] =
{
	"SCSI-1",
	"CCS",
	"SCSI-2",
	"","","","","",
	"","","","","","","",""
};
int bus_sizes[] = { 8,16,32,32};

xtattach(struct xha_slave *xs)
{
    struct scsi_inqblk 	iqbuf;
    struct scsi_inqblk 	*iq = &iqbuf;
    struct xt_softc	*sc = &xt_softc[xs->xs_unit];
    static char		vid[sizeof(iq->sinq_vid)+1];
    static char		pid[sizeof(iq->sinq_pid)+1];
    static char		fwrev[sizeof(iq->sinq_fwrev)+1];
    static char		fwdate[sizeof(iq->sinq_fwdate)+1];
    char		*typep;
    int			type;
    struct scsi_ms	*msp = &xtmsbuf;
    unsigned		lblen;

    DEBUGF(xttrace,printf("Xtattach: xs=%x\n",xs););

    if(xs->xs_unit > NXT)
    {
	printf("xt%d: Not configured\n",xs->xs_unit);
    }

    if(xxinquiry(xs,iq) == 0)
    {

	xtidcpy(iq->sinq_vid,vid,sizeof vid );
	xtidcpy(iq->sinq_pid,pid,sizeof pid );
	xtidcpy(iq->sinq_fwrev,fwrev,sizeof fwrev );
	xtidcpy(iq->sinq_fwdate,fwdate,sizeof fwdate );

	type = xtgettype(xs,&typep,pid);

	if(type < 0)
	{
	    type = 0;
	    printf("%s%d: Unknown disk type: %s using %s\n", 
		   xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,typep,   			xtst[type].typename);  			 
	}

	xs->xs_xi->xi_type = iq->sinq_pdt;
	/*
	printf("%s%d: Disk type %s name=%s\n",
	    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,
	    xtst[xs->xs_xi->xi_type].typename, typep); 
        */
	/* 
	 * Fill in default values for disk info. 
	 */
	bcopy(&(xtst[xs->xs_xi->xi_type]),&(sc->sc_xtst),sizeof(struct xtst));
	/* 
	 * Clear size structure.
	 */
	bzero(&(sc->sc_size),sizeof(struct size));
		    
	printf("%s%d:%s %s Firmware Rev. %s (%s) ANSI V.%d(%s)\n",
	    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,
	    vid,pid,fwrev,fwdate,iq->sinq_ansiversion,responses[iq->sinq_rdd]);
	if (iq->sinq_sync)
	    printf("%s%d:unit supports synchronous data transfers using %d bit wide bus\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit, 
		bus_sizes[(iq->sinq_wbus32<<1) + iq->sinq_wbus16]);
	if(xtmodesense(xs,&xtmsiobuf) == 0)
	{
	    bzero(msp,sizeof (*msp));

	    xxfillmspages(xtmsiobuf.msio,msp);

	    strncpy(sc->sc_xtst.inqname,pid, sizeof sc->sc_xtst.inqname);
	    /* Now let's calculate the disk geometry. SCSI disks are quite tricky. We want to report the usable geometry excluding slip sectore or alternate tracks. Therefore, if the disk uses slip sectoring on a per track basis, nsect=daf_spt - daf_altspz. Otherwise if the disk uses a # of alternate sectors per cylinder, we report the true number of sectors per track. However the number of sectors per cylinder will NOT be the same as nspc*ntracks. The number of cylinders is decremented with the number of spare tracks / tracks per cyl rounded down to the next cylinder boundary */

	    /* We don't beleive in no tracks per zone, maybe we should, but then division by zero might catch us later */
	    if(msp->sms_daf.daf_tpz == 0)
		msp->sms_daf.daf_tpz = 1;

	    /* Get true no of sectors */
	    sc->sc_xtst.nsect = msp->sms_daf.daf_spt;
	    /* If no of tracks per zone == 1, we report the actual usable no of sectors by decreasing with the number of alternate sectors per track */
	    if(msp->sms_daf.daf_tpz == 1)
		sc->sc_xtst.nsect -= msp->sms_daf.daf_altspz;

	    /* Assume we are using all heads ;-) */
	    sc->sc_xtst.ntrak = msp->sms_rdg.rdg_hmaxheads;

	    /* We also don't beleive in zero heads which will avoid any division by zero later */
	    if(sc->sc_xtst.ntrak == 0)
		sc->sc_xtst.ntrak = 1;

#ifdef	notdef
	    /* Sectors per cylinder = No of (true) sectors * no of tracks, then 
	     If we have zonesize 1, we loose altspz*ntrak sectors. If we have zonesize==no of tracks we loose altspz sectors. Other values of the zone size will make this calculation give more or less crazy results if the zone size isn't an integral part of the track number, but I havn't seen any disk allowing this (yet) */
	    sc->sc_xtst.nspc  = msp->sms_daf.daf_spt * sc->sc_xtst.ntrak
	      - ((sc->sc_xtst.ntrak/ msp->sms_daf.daf_tpz)*msp->sms_daf.daf_altspz);
#endif
	   /* XXX  WARNING!! tabe above really seems to screw up mkfs since
	   it really expects to see nspc equal to nsect*ntrak (reported
	   number of secotors times reported number of tracks) otherwise it
	   complains.  I really don't understand the above logic so I'm just
	   going to set it as mkfs want's and hope for the best.  It should
	   be ok since the scsi drive maps logical blocks to real blocks
	   anyways.  -md */
	    sc->sc_xtst.nspc = sc->sc_xtst.nsect * sc->sc_xtst.ntrak;

	    /* Now get the disk's cylinder count. First pick the maximum cylinder count */
	    sc->sc_xtst.ncyl  = 
	        xtget3((char *)msp->sms_rdg.rdg_maxcyl);
	    /* Then, if there are alternate tracks, we decrease the cylinder count with one for each cylinder used up rounding this figure up */
	    if(msp->sms_daf.daf_alttpv > 0)
		sc->sc_xtst.ncyl -= ((msp->sms_daf.daf_alttpv -1) / sc->sc_xtst.ntrak) + 1 ; 

	    if(xtreadcapacity(xs,&sc->sc_cdata) == 0)
	    {
		DEBUGF(xttrace,printf("Xtslave: capacity read blklen=%d, nblocks=%d\n",
		    sc->sc_cdata.cd_blklen,sc->sc_cdata.cd_blocks););
		sc->sc_xtst.nbps = sc->sc_cdata.cd_blklen;
		sc->sc_size.partition[2].nblocks = sc->sc_cdata.cd_blocks;
	    }
	    else
	    {
		DEBUGF(xttrace,printf("Xtslave: capacity read failed\n"););
		    sc->sc_xtst.nbps  = msp->sms_daf.daf_nbps;
		sc->sc_size.partition[2].nblocks = 
		sc->sc_xtst.ncyl * sc->sc_xtst.nspc;
	    }
	    lblen = msp->sms_hd.sm_bdarr[0].sbd_blklen;
	    if(lblen && sc->sc_xtst.nbps && (sc->sc_xtst.nbps != lblen))
	    {
		printf("%s%d: WARNING! logical block length and sector length differs\n!!", 
	    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit);
		if(((sc->sc_xtst.nbps % 256) != 0) && (lblen % 256 == 0))
		{
		    sc->sc_xtst.nbps = lblen;
		    printf("    block length set to %d\n",lblen);
		}
		else 
		{
			if((sc->sc_xtst.nbps % 256 == 0) && (lblen % 256 != 0))
			{
			    printf("    block length set to %d\n",sc->sc_xtst.nbps);
			}
		}
	    }
		
	    sc->sc_size.partition[2].pblkoff = 0;
	    sc->sc_cinited = 1;

	    printf("nsect=%d, ntrak=%d, nspc=%d, ncyl=%d, nbps=%d, csize=%d",
		   sc->sc_xtst.nsect, sc->sc_xtst.ntrak, sc->sc_xtst.nspc,
		   sc->sc_xtst.ncyl, sc->sc_xtst.nbps,
		   sc->sc_size.partition[2].nblocks);
	    if(msp)
		printf(", tpz=%d, altspz=%d, alttpv=%d",
		       msp->sms_daf.daf_tpz, msp->sms_daf.daf_altspz,
		       msp->sms_daf.daf_alttpv);
	    if(strcmp(vid,"CDC") == 0)
		printf("\nWren Caching %sabled : \n",
		       msp->sms_csh.csh_ce ? "en" : "dis" );

	    xtpartitioninit(xs,sc); /* Initialize partitions */
	}
	else
	{
	    printf("%s%d: Mode sense failed\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit);
	    return 0;
	}
    }
    else
    {
	printf("%s%d: Inquiry failed\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit);
	return 0;
    }
    if (xs->xs_dk >= 0) {
#ifdef	STATIC_WPMS
	/*				pre-Reno systems -md
	static float mspw = .0000016097;

	dk_mspw[xs->xs_dk] = mspw;
	*/
					/* 1/mspw for Reno */
	printf(", dk_wpms=%d\n", dk_wpms[xs->xs_dk] = 621234);	
#else
	/*
	 * Seconds per word = (60 / rpm) / (nsectors * secsize/2)
	 */
#define	XT_RPM	3600	/* no dynamic way of getting it */
	printf(", dk_wpms=%d\n", dk_wpms[xs->xs_dk] =
	    (XT_RPM * sc->sc_xtst.nsect * sc->sc_xtst.nbps) / 120);
#endif
    } else
	printf(", dk_wpms=0\n");
    return 1;
}


/* Tell controller routine expected max lifetime of */
/* a command (time in seconds)			    */
xtttl(union scsi_cdb *com)
{
	DEBUGF(xttrace,printf("Xttl: com=%x\n",com););
	switch(com->cdb_6.sc_op)
	{
	    case SC0_START_STOP_UNIT:		return 60;
	    default:			return 1;
	}
}

xtmodesense(struct xha_slave *xs, struct scsi_msiobuf *msp)
{
	register union scsi_cdb *scb;

	DEBUGF(xttrace, printf("Xtmodesense: xs=%x, msp=%x\n", xs,msp););
	scb = xxgetscbuf(xs);
	scb->cdb_6.sc_lbarest[0] = SCMS_CURRENT | SCMS_ALL;
	return xxdoscsi(xs, scb, SC0_MODE_SENSE,(caddr_t) msp,
		sizeof *msp,B_READ);
}

xtmodeselect(struct xha_slave *xs, struct scsi_msc * mscp)
{
        static union scsi_cdb *scb;

	DEBUGF(xttrace, printf("Xtmodeselect: xs=%x, msp=%x\n", xs,mscp););
	scb = xxgetscbuf(xs);
        scb->cdb_6.sc_lbaMSB = 0x1 ; /* turn on SMP Save Mode Page bit */
	return xxdoscsi(xs,scb,SC0_MODE_SELECT,(caddr_t) mscp,
			sizeof *mscp,B_WRITE);
}

xtstartstop(struct xha_slave *xs,short start)
{
	return xxscsicmd(xs,SC0_START_STOP_UNIT,0,start,B_READ);
}

xtreadcapacity(struct xha_slave *xs, struct scsi_capdata *cd)
{
    register union scsi_cdb *scb;
    DEBUGF(xttrace, printf("Xtreadcapacity: xs=%x, cd=%x\n", xs,cd););
    scb = xxgetscbuf(xs);
    xxfillscbuf(scb,xs->xs_lun,SC1_READ_CAPACITY,0);
    if(((*xs->xs_xi->xi_hadriver->xhc_doscsi) (xs, scb, (caddr_t) cd,
	8, B_READ)) == ((XHI_CMPL << 8) + SCTS_GOOD))
    {
	return 0;
    }
    return -1;
}

xtslave(struct xha_slave *xs,caddr_t addr)
{
    DEBUGF(xttrace, printf("Xtslave: xs=%x, addr=%x\n",xs, addr););
    /* See if something is there */
    if(xxslave(xs,&xtcdriver,SCPDT_DASD) == 0)
    	return 0;

    if(xxtur(xs) < 0 ) /* Isn't ready ? */
    {
	/* NO, start it first */
	DEBUGF(xttrace,
		printf("%s%d: Attempting drive start\n",
		    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
	if(xtstartstop(xs,1) < 0)
	{
	    DEBUGF(xttrace,
		printf("%s%d: Couldn't start drive\n",
		    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
	    return 0;
	}
	if(xxtur(xs) != 0)
	    return 0;
    }
    if(xtmodesense(xs,&xtmsiobuf) == 0)
	return 1;
    return 0;
}

xtgettype(struct xha_slave *xs, char **typep, char *inqname)
{
    struct xtst *st;

    DEBUGF(xttrace,
	    printf("Xtgettype: xs=%x, typep=%x, inqname=%s\n", xs,typep,inqname););
    *typep = inqname;

    for (st = xtst; st < xtst + (sizeof xtst/sizeof (struct xtst)); st++)
    {
	if(strncmp(st->inqname,inqname,sizeof (st->inqname)) == 0)
	{
	    return (st - xtst);
	}
    }
    return -1;
}

xtsize(dev_t dev)
{
	register int			un = XTUNIT(dev);
	register int			pa = XHAMIN_PART(dev);
	register struct xha_slave	*xs;
	struct xt_softc 		*sc;

	DEBUGF(xttrace,
	    printf("Xtsize: dev=%x\n", dev););
	if (un >= NXT || pa < 0 || pa >= MAXNPART ||
	    (xs = xtinfo[un]) == 0 || xs->xs_alive == 0)
		return(-1);
        sc = &xt_softc[xs->xs_unit];
	return(sc->sc_size.partition[pa].nblocks);
}


xtopen(dev_t dev,int flag)
{

	struct xha_slave *xs;
        register int xtunit = XTUNIT(dev);
        struct xt_softc *sc;

	DEBUGF(xttrace,
	    printf("Xtopen: dev=%x, flag=%x\n", dev, flag););

        if (xtunit >= NXT || (xs = xtinfo[xtunit]) == 0 || xs->xs_alive == 0)
                return (ENXIO);

	if(flag & O_EXCL)
	{
		if(xs->xs_ocnt > 0)
		{
			return(EBUSY);
		}
		xs->xs_flags |= XSFL_EXOPEN;
	}
	xs->xs_ocnt++;

        sc = &xt_softc[xs->xs_unit];
	if(sc->sc_spundown && sc->sc_autospinup)
	{
		xtstartstop(xs,1);
	}
        return (0);

}

xtclose(dev_t dev,int flag)
{
	struct xha_slave *xs;
        register int xtunit = XTUNIT(dev);

        if (xtunit >= NXT || (xs = xtinfo[xtunit]) == 0 || xs->xs_alive == 0)
                return (ENXIO);
	DEBUGF(xttrace,
	    printf("Xtclose: dev=%x, flag=%x\n", dev, flag););

	xs->xs_ocnt=0;
	xs->xs_flags &= ~(XSFL_SPECSCSI | XSFL_EXOPEN);
}

xtpartitioninit(struct xha_slave *xs, struct xt_softc *sc)
{
    daddr_t csize;
    int	cblkoff;

    DEBUGF(xttrace,
	    printf("Xtpartitioninit: xs=%x, sc=%x\n", xs,sc););

    sc->sc_nomap = 1;      /* Open whole disk, no mapping */

    if(xtgetpartitions(xs) == 0)
    {
	/* No minidisk info, use default */
	/* Save C partition values in case it is already initialized */
	csize = sc->sc_size.partition[2].nblocks;
	cblkoff = sc->sc_size.partition[2].pblkoff;

	bcopy(sc->sc_xtst.defsizes, &sc->sc_size,sizeof( struct size ));

	/* If the C partition is already initalized in the attach
	 routine, we restore the values for the c partition here */
	if(sc->sc_cinited)
	{
	    sc->sc_size.partition[2].nblocks = csize;
	    sc->sc_size.partition[2].pblkoff = cblkoff;
	}
	sc->sc_cinited = 1;
    }
    else
    {
	/* 
	 * NOTE! The minidisk utilities doesn't set up the c partition
         * If this isn't initializes to sane things, lots of other things
	 * will break, so we always copy the default c partition from
	 * the standard table
	 */
	if(sc->sc_cinited == 0)
	{
	    bcopy(&(sc->sc_xtst.defsizes->partition[2]),
		  &(sc->sc_size.partition[2]),
		  sizeof (struct partentry));
	    sc->sc_cinited = 1;
	}
    }
    sc->sc_nomap = 0;      /* Map again */
}

xtstrategy(struct buf *bp)
{
    register struct xha_slave *xs;
    int pa = XHAMIN_PART(bp->b_dev);
    register int xtunit = XTUNIT(bp->b_dev);
#ifdef DEBUG
    long sz;
#endif /* DEBUG */

    DEBUGF(xttrace,
	printf("Xtstrategy: bp=%x\n", bp););

    DEBUGF(xtstrategydebug, sz = (bp->b_bcount+511) >> 9; );

    if (xtunit >= NXT) 
    {
	bp->b_error = ENXIO;
	DEBUGF(xtstrategydebug,
	    printf("Xtstrategy(BAD): bp=0x%x, sz=%d, blkno=%d\n",
		bp,sz,bp->b_blkno););
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
    }

    xs = xtinfo[xtunit];
    if (xs == 0 || xs->xs_alive == 0) 
    {
	bp->b_error = ENXIO;
	DEBUGF(xtstrategydebug,
	    printf("Xtstrategy(BAD): bp=0x%x, sz=%d, blkno=%d\n",
		bp,sz,bp->b_blkno););
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
    }

    xtxstrat(xs, pa, bp);
}

xtxstrat(struct xha_slave *xs,int pa, register struct buf *bp)
{
    register struct xtst *st;
    register struct xt_softc *sc;
    long bn, sz;
    int s;
    register struct xhac_driver *xc;
    struct buf *dp;

    sz = (bp->b_bcount+511) >> 9;

    xc  = xs->xs_xi->xi_hadriver;
    sc = &xt_softc[xs->xs_unit];
    st = &(sc->sc_xtst);
    if(sc->sc_spundown && sc->sc_autospinup)
    {
	xtstartstop(xs,1);
    }
    DEBUGF(xtstrategydebug,printf("Xtstrategy: bp=0x%x, sz=%d, blkno=%d, xs=%x\n",
	    bp,sz,bp->b_blkno,xs););
    bn = dkblock(bp);
    if ((bn < 0) || ((sc->sc_nomap == 0) && 
		     (bn+sz > sc->sc_size.partition[pa].nblocks)))
    {
	if (bn == sc->sc_size.partition[pa].nblocks)
	{
	    bp->b_resid = bp->b_bcount;
	    iodone(bp);
	    return;
	}
	bp->b_error = EINVAL;
	DEBUGF(xtstrategydebug,
	    printf("Xtstrategy(BAD): bp=0x%x, sz=%d, blkno=%d\n",
		bp,sz,bp->b_blkno););
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
    }

    s = dsplbio(50);
#ifdef PARANOID
    {
	int loopcount;
	for(dp=&xs->xs_tab,loopcount=0;dp;dp = dp->b_actf)
	{
	    if(++loopcount >= 100000)
	    {
		printf("%s%d:Xtstrategy: request list loop detected\n",
		    xs->xs_xi->xi_driver->xd_sname, xs->xs_unit);

		printf("xs_tab=%x, tab.actf=%x, dp=%x dp->b_actf=%x\n",
		    &xs->xs_tab,xs->xs_tab.b_actf,dp,dp->b_actf);
		break;
	    }
	    if(dp == bp)
	    {
		printf("%s%d:Xtstrategy: bp=%x already in Q\n",
		    xs->xs_xi->xi_driver->xd_sname, xs->xs_unit,bp);
		dsplx(s,51);
		(*xc->xhc_cstart)(xs);
		return;
	    }
	}
    }
#endif /* PARANOID */

    if(!xtdisksort)
    {
	bp->av_forw = NULL;
	dp = &xs->xs_tab;
	if (dp->b_actf == NULL)
	    dp->b_actf = bp;
	else
	    dp->b_actl->av_forw = bp;
	dp->b_actl = bp;
	bp->b_errcnt = 0;
    }
    else
    {
	bp->b_cylin = bn;

	if(sc->sc_nomap == 0)
		bp->b_cylin += sc->sc_size.partition[pa].pblkoff;

/*
 * We really doesn't have to sort by cylinders since we can sort by block
 * numbers. If you really want to sort by cylinder numbers. define CYLSORT
 */
#ifdef CYLSORT
	if(st->nspc)
		bp->b_cylin /= st->nspc;
#endif /* CYLSORT */

	bp->b_errcnt = 0;
	disksort(&xs->xs_tab, bp);
    }
    dsplx(s,51);
    if(xs->xs_tab.b_active == 0)
	    (*xc->xhc_cstart)(xs);
}

xtustart(struct xha_slave *xs)
{
    struct xt_softc *sc = &xt_softc[xs->xs_unit];

    DEBUGF(xtustartdebug | xttrace ,printf("Xtustart: xs=0x%x\n",xs););

    if(xtdolinks && (sc->sc_xtst.flags & XTFL_LINK) )
	return XHU_DOLINKS;
    else
	return XHU_DODATA;
}

void
xtpartsetup(register struct xha_slave *xs, register struct buf *bp, 
	    int	bn,int sz)
{
	register struct partsec *ps;
	int	s;
	struct	xt_softc *sc = &xt_softc[xs->xs_unit];

	DEBUGF(xtpartial | xttrace,
		printf("Xtstart(Partial): xs=0x%x, bn=%d, sz=%d, bcount=%d\n",
		    xs,bn,sz,bp->b_bcount););
	s = dsplbio(52);
	ps = xtgetps(xs,NULL); /* NULL will get us a free entry */
	if(ps == NULL)
	    panic("Xtstart: partial sector init failed");
	
	/* Save interesting facts in buf structure */
	ps->ps_bp = bp;
	ps->ps_addr = bp->b_un.b_addr;
	ps->ps_bcount = bp->b_bcount;
	ps->ps_phys = (bp->b_flags & B_PHYS) ? 1 : 0;
	ps->ps_nbps = sc->sc_xtst.nbps;
	dsplx(s,53);
	/* Now let's modify the buf struct a bit */
	bp->b_un.b_addr = ps->ps_space;	/* link i/o from buffer to this space */
	bp->b_bcount = sc->sc_xtst.nbps;
	bp->b_flags &= ~B_PHYS;
	
	if(!(bp->b_flags & B_READ))
	    xtfillpart(ps);
}

union scsi_cdb *
xtstart(struct xha_slave *xs, struct buf *bp, union scsi_cdb *combuf)
{
    register union scsi_cdb *scbuf;
    register int bn;
    register int sz;
    register struct xt_softc *sc = &xt_softc[xs->xs_unit];
    dev_t 	dev;

    DEBUGF(xttrace,
		printf("Xtstart: xs=%x,bp=%x,combuf=%x\n",xs,bp,combuf););
    dev = bp->b_dev;
    bn = dkblock(bp);
    if(!sc->sc_nomap)
    {
	bn += sc->sc_size.partition[XHAMIN_PART(dev)].pblkoff;
    }
    sz = (bp->b_bcount + (sc->sc_xtst.nbps-1)) / sc->sc_xtst.nbps;	/* Compute no of blks */
    if(sz == 0)
    {
	/* This should never happen, but if it does, and the command is 
	   a write, really bad things might occur, since a transfer length
	   of 0 means that 256 blocks will be transferred */
	printf("Xtstart: bp=0x%x, b_flags=0x%x, b_forw=0x%x, b_back=0x%x\n",
		bp,bp->b_flags,bp->b_forw,bp->b_back);
	printf("         av_forw=0x%x, av_back=0x%x, b_bcount=%d\n",
		bp->av_forw,bp->av_back,bp->b_bcount);
	printf("         b_bufsize=0x%x,b_error=0x%x,b_dev=0x%x\n",
		bp->b_bufsize,bp->b_error,bp->b_dev);
	printf("         b_addr=0x%x,b_blkno=%d,b_resid=%d,b_proc=0x%x\n",
		bp->b_un.b_addr,bp->b_blkno,bp->b_resid,bp->b_proc);
	/* printf("         b_iodone=0x%x,b_pfcent=%d,b_bitmap=0x%x\n",
		bp->b_iodone,bp->b_pfcent,bp->b_bitmap); */
	printf("         b_iodone=0x%x,b_pfcent=%d\n",
		bp->b_iodone,bp->b_pfcent);
    }

    DEBUGF(xtstartdebug,printf("Xtstart: xs=0x%x, bn=%d, sz=%d\n",xs,bn,sz););
    if(combuf)
	scbuf =  combuf;
    else
	scbuf = xxgetscbuf(xs);

    if(xs->xs_flags & XSFL_SPECSCSI)
    {
	scbuf = &xs->xs_cdb;
	scbuf->cdb_6.sc_lun = xs->xs_lun;
	return scbuf;
    }

    /*
     * Code here to handle partial sector transfers 
     */

    if(bp->b_bcount < sc->sc_xtst.nbps)
    {
	xtpartsetup(xs,bp,bn,sz);
    }

    if((bn < 0x1FFFFF) && (sz != 0) && (sz < 256) ) 
    {

	xxfillscbuf(scbuf,xs->xs_lun,bp->b_flags & B_READ ? SC0_READ:SC0_WRITE,sz);
	scbuf->cdb_6.sc_lbaMSB     = bn >> 16 & 0x1F;
	scbuf->cdb_6.sc_lbarest[0] = bn >> 8  & 0xFF;
	scbuf->cdb_6.sc_lbarest[1] = bn       & 0xFF;
	DEBUGF(xtstartdebug,printf("Xtstart: Return SC0: bn=%d, sz=%d\n",bn,sz););
    }
    else
    {
	xxfillscbuf(scbuf,xs->xs_lun,bp->b_flags & B_READ ? SC1_READ:SC1_WRITE,sz);
	scbuf->cdb_10.sc_lba[0]    = bn >> 24 & 0xFF;
	scbuf->cdb_10.sc_lba[1]    = bn >> 16 & 0xFF;
	scbuf->cdb_10.sc_lba[2]    = bn >>  8 & 0xFF;
	scbuf->cdb_10.sc_lba[3]    = bn       & 0xFF;
	DEBUGF(xtstartdebug,printf("Xtstart: Return SC1: bn=%d, bno=%d\n",bn,sz););
    }
    if (xs->xs_dk >= 0) 
    {
	dk_busy |= 1<<xs->xs_dk;
	dk_seek[xs->xs_dk] ++;
	dk_xfer[xs->xs_dk]++;
	dk_wds[xs->xs_dk] += bp->b_bcount >> 6; 
    }

    return (scbuf);
}

xtfillpart(struct partsec *ps)
{
    register int	fillcount = ps->ps_bcount % ps->ps_nbps;
    register caddr_t	fillfrom  = ps->ps_addr + ps->ps_bcount - fillcount;
    register caddr_t        from, from2;
    register int            s, v;
    register u_int          n;

    DEBUGF(xttrace | xtpartial,
	printf("Xtfillpart: filling %d bytes\n",ps->ps_bcount););

    if(!ps->ps_phys)
    {
	DEBUGF(xtpartial,
	    printf("bcopy(%x,%x,%d) (COOKED)\n",
	        fillfrom,ps->ps_space,fillcount););
	bcopy(fillfrom,ps->ps_space,fillcount);
    }
    else
    {
	ps->ps_bp->b_flags |= B_PHYS;
	from = real_buf_addr(ps->ps_bp, fillfrom);
	n = PGOFSET + 1 - ((u_int)from & PGOFSET);
	if (n >= fillcount) 
	{
	    DEBUGF(xtpartial,
		printf("bcopy(%x,%x,%d) (RAW1)\n",
		    from,ps->ps_space,fillcount););
	    s = dsplhigh(54);
	    GET_VR0(v);
	    bcopy(from, ps->ps_space, fillcount);
	} 
	else 
	{
	    from2 = real_buf_addr(ps->ps_bp, fillfrom + n);
	    DEBUGF(xtpartial,
		printf("bcopy(%x,%x,%d) (RAW2)\n",
		    from,ps->ps_space,n););
	    DEBUGF(xtpartial,
		printf("bcopy(%x,%x,%d) (RAW2)\n",
		    from2,ps->ps_space + n, fillcount - n););
	    s = dsplhigh(55);
	    GET_VR0(v);
	    bcopy(from, ps->ps_space, n);
	    bcopy(from2, ps->ps_space + n, fillcount - n);
	}
	SET_VR(v);
	dsplx(s,56);
	ps->ps_bp->b_flags &= ~B_PHYS;
    }
    DEBUGF(xttrace | xtpartial,
	  printf("bzero(%x,%d)\n",
	  ps->ps_space + fillcount, ps->ps_nbps - fillcount););
    bzero(ps->ps_space + fillcount, ps->ps_nbps - fillcount);
}

xtint(struct xha_slave *xs, struct buf *bp, int ccode, 
      struct scsi_esense *sdp, u_char tarstat)
{
    register struct xt_softc *sc = &xt_softc[xs->xs_unit];
    register int	bn;
    register int	sz;

    DEBUGF(xtintdebug | xttrace,
	printf("Xtint: xs=0x%x, bp=0x%x, ccode=%d, sdp=0x%x\n",
        xs,bp,ccode,sdp););
    if (xs->xs_dk >= 0)
	dk_busy &= ~(1<<xs->xs_dk);
    if(bp && (bp->b_flags & B_HEAD))
    {
	xs->xs_flags &= ~XSFL_SPECSCSI;
	return XHD_DONE;
    }
    if(tarstat == SCTS_BUSY)
    {
	if(sc->sc_busy)	/* I.e. we were here last time too */
	{
	    if(++sc->sc_busyretries > 5)
	    {
		bp->b_error=EIO;
		bp->b_flags |= B_ERROR;
		sc->sc_busy = 0;	/* Next time we'll find out ... */
		xtpartdone(xs,bp);
		xs->xs_flags &= ~XSFL_SPECSCSI;
		return XHD_DONE;
	    }
	    return XHD_RETRY;
	}
	sc->sc_busy = 1;
	sc->sc_busyretries=1;
	return XHD_RETRY;
    }
    sc->sc_busy = 0;
    switch(ccode)
    {
    case XHI_OUVR:
    case XHI_CMPL:
	if(bp && (bp->b_flags & B_HEAD))
        {
	    xs->xs_flags &= ~XSFL_SPECSCSI;
	    return XHD_DONE;
        }
        bp->b_resid = 0;
	break;
    case XHI_TIMEOUT:
	if(bp && (bp->b_flags & B_HEAD))
	{
	    xs->xs_flags &= ~XSFL_SPECSCSI;
	    return XHD_DONE;
	} 
	return xttryretry(xs,bp,1,EIO); 
    case XHI_SELTO: 
        /* B_INVAL is used here to disable any retries during autoconfig or any other special SCSI commands */ 
	if(bp && (bp->b_flags & B_HEAD)) 
	{ 
		xs->xs_flags &= ~XSFL_SPECSCSI; 
		return XHD_DONE;
	}
	if(++bp->b_errcnt <= 1)
	    return XHD_RESETRETRY;
	if(bp->b_errcnt <= 2 )
	    return XHD_SCRESETRETRY;
	bp->b_bcount = bp->b_resid;
	bp->b_resid = 0;
	bp->b_error = EIO;;
	bp->b_flags |= B_ERROR;
	xtpartdone(xs,bp);
	xs->xs_flags &= ~XSFL_SPECSCSI;
	return XHD_DONE;
    case XHI_SCSIRESET:
	if(bp && (bp->b_flags & B_HEAD))
	{
	    xs->xs_flags &= ~XSFL_SPECSCSI;
	    return XHD_DONE;
	}
	return XHD_RETRY;
    case XHI_ERR:
        /* Analyze the error */
        if(sdp)
	{
	    bcopy(sdp, &(xs->xs_esense), sizeof(xs->xs_esense));
	    xs->xs_flags |= XSFL_HASSENSE;

            switch (sdp->es_key)
            {
	        case SCES_KEY_RECOVERED_ERROR:
                    DEBUGF(xterrdebug,printf("Xtint: RECOVERED ERROR\n"););
                    break;
                case SCES_KEY_NOT_READY:
                    DEBUGF(xterrdebug,printf("Xtint: NOT READY\n"););
		    printf("%s%d: not ready\n", xs->xs_xi->xi_driver->xd_sname,xs->xs_unit);
		    bp->b_error = EIO;
		    bp->b_flags |= B_ERROR;
		    break;
                case SCES_KEY_MEDIUM_ERROR:
		    sz = (bp->b_bcount + (sc->sc_xtst.nbps-1)) / 
			  sc->sc_xtst.nbps;	/* compute no of blks */
		    bn = dkblock(bp) + (sc->sc_nomap ? 0 : 
			(sc->sc_size.partition[XHAMIN_PART(bp->b_dev)].pblkoff));
                                      printf("%s%d: Xtint: MEDIUM ERROR abs startuing blk=%d, rel=%d, size=%d blocks\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,bn,dkblock(bp),sz);
		    return xttryretry(xs,bp,5,EIO);
                case SCES_KEY_HARDWARE_ERROR:
                    DEBUGF(xterrdebug,printf("%s%d: Xtint: HARDWARE ERROR\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		    return xttryretry(xs,bp,5,EIO);
                case SCES_KEY_ILLEGAL_REQUEST:
                    DEBUGF(xterrdebug,printf("%s%d: Xtint: ILLEGAL REQUEST\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		    return xttryretry(xs,bp,1,EIO);
                case SCES_KEY_UNIT_ATTENTION:
                    DEBUGF(xterrdebug,printf("%s%d: Xtint: UNIT ATTENTION\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		    return xttryretry(xs,bp,2,EIO);
                case SCES_KEY_ABORTED_COMMAND:
                    DEBUGF(xterrdebug,printf("%s%d: Xtint: ABORTED COMMAND\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		    return xttryretry(xs,bp,1,EIO);
                default:
		    DEBUGF(xterrdebug,printf("%s%d: Xtint: Unknown KEY=%d\n",
				xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,
			sdp->es_key););
                    break;
            }
	}
	else
	    printf("Xtint: Sense data pointer is zero\n");
	if(bp && (bp->b_flags & B_HEAD))
	{
	    xs->xs_flags &= ~XSFL_SPECSCSI;
	    return XHD_DONE;
	}
	break;
    }

    /* Make sure the byte count is reasonable.                           */
    if(bp->b_bcount < 0)
	bp->b_bcount = bp->b_resid;

    xtpartdone(xs,bp);
    xs->xs_flags &= ~XSFL_SPECSCSI;
    return XHD_DONE;
}

xtpartdone(struct xha_slave *xs, struct buf *bp)
{
	register struct partsec *ps;

	if(!PARTIAL(bp)) 
		return;
	DEBUGF(xttrace | xtpartial,
	    printf("Xtpartdone: xs=%x, bp=%x\n",xs,bp););
	ps = xtgetps(xs,bp);
	/* Now restore buf header variables */
	if(ps->ps_phys)
		bp->b_flags |= B_PHYS;
	bp->b_un.b_addr = ps->ps_addr;
	/* If we were reading, copy out the stuff */
 	if(bp->b_flags & B_READ)
		xtemptypart(ps);
	ps->ps_addr = NULL;	
	ps->ps_bcount = 0;
	ps->ps_bp = NULL;	/* Free this entry */
}

xtemptypart(struct partsec *ps)
{
    register int	emptycount = ps->ps_bcount % ps->ps_nbps;
    register caddr_t	emptyto  = ps->ps_addr + ps->ps_bcount - emptycount;
    register caddr_t        to, to2;
    register int            s, v;
    register u_int          n;

    DEBUGF(xttrace | xtpartial,
	printf("Xtemptypart(%x): copying %d bytes\n",ps,ps->ps_bcount););
    if (!ps->ps_phys) 
    {
	DEBUGF(xtpartial,
	    printf("bcopy(%x,%x,%d) (COOKED)\n",
		ps->ps_space,emptyto,emptycount););
	bcopy(ps->ps_space, emptyto, emptycount);
    }
    else 
    {
	to = real_buf_addr(ps->ps_bp, emptyto);
	n = PGOFSET + 1 - ((u_int)to & PGOFSET);
	if (n >= emptycount) 
	{
	    DEBUGF(xtpartial,
		printf("bcopy(%x,%x,%d) (RAW1)\n",
		    ps->ps_space, to, emptycount););
	    s = dsplhigh(57);
	    GET_VR0(v);
	    bcopy(ps->ps_space, to, emptycount);
	} 
	else 
	{
	    to2 = real_buf_addr(ps->ps_bp, emptyto + n);
	    DEBUGF(xtpartial,
		printf("bcopy(%x,%x,%d) (RAW2)\n", ps->ps_space, to, n););
	    DEBUGF(xtpartial,
		printf("bcopy(%x,%x,%d) (RAW2)\n",
		    ps->ps_space + n, to2, emptycount - n););
	    s = dsplhigh(58);
	    GET_VR0(v);
	    bcopy(ps->ps_space, to, n);
	    bcopy(ps->ps_space + n, to2, emptycount - n);
	}
	SET_VR(v);
	dsplx(s,59);
    }
    DEBUGF(xttrace | xtpartial,printf("Xtemptypart(%x) DONE\n",ps););
}

#ifdef	BSD_43	/* BSD Reno doesn't need these, done via rawread/rawwrite */

xtread(dev_t dev, struct uio *uio)
{
 int errno;

    DEBUGF(xttrace,
	printf("Xtread: dev=0x%x,uio=0x%x\n",dev,uio););
    errno = xtphys(dev);
    DEBUGF(xttrace,
	printf("Xtread: errno=%d\n",errno););
    if (errno)
        return (errno);
    return (physio(xtstrategy, &rxtbuf[XTUNIT(dev)], dev, B_READ, xtminphys, uio))
;
}

xtwrite(dev_t dev, struct uio *uio)
{
    int errno;

    DEBUGF(xttrace,
    	printf("Xtwrite: dev=0x%x,uio=0x%x\n",dev,uio););
    errno = xtphys(dev);
    DEBUGF(xttrace,
    	printf("Xtwrite: errno=%d\n",errno););
    if (errno)
        return (errno);
    return (physio(xtstrategy, &rxtbuf[XTUNIT(dev)], dev, B_WRITE, xtminphys, uio)
);
}
#endif

xtphys(dev_t dev)
{
        register int unit;
        struct xha_slave *xs;

        DEBUGF(xttrace,
		printf("Xtphys: dev=0x%x\n",dev););

        unit = XTUNIT(dev);
	if( unit > NXT)
		return ENXIO;
        if (((xs = xtinfo[unit]) == 0 )
            || (xs->xs_alive == 0) )
                return (ENXIO);
        return (0);
}

xtioctl(dev_t dev, int cmd, caddr_t data, int flag)
{
    register int xtunit = XTUNIT(dev);
    register struct xha_slave *xs;
    register struct xt_softc *sc;
    register struct dkpart *dk = (struct dkpart *) data;
    register struct partentry *pep;
    register int pa;
					/* defines from hd.c -md */
    register struct disklabel       *dl;
#define	NPART		8	/* number of partitions */
#define	MAXBADBLKS	1000	/* max number of BBT entries */


    if(xtunit >= NXT || (xs = xtinfo[xtunit]) == 0 || xs->xs_alive == 0)
    {
	DEBUGF(xtioctldebug,printf("Xtioctl: bad device %d/%d\n",major(dev),minor(dev)););
	return ENODEV;
    }


    DEBUGF(xtioctldebug | xttrace,
	printf("%s%d: Xtioctl: cmd = %d\n, data=%x\n",
	xs->xs_xi->xi_driver->xd_sname, xs->xs_unit, cmd););
    sc = &xt_softc[xs->xs_unit];

    switch(cmd)
    {
    case DKIOCGPART:
	int i;
	
	xtpartitioninit(xs,sc);

	pa = XHAMIN_PART(dev);
	pep = &sc->sc_size.partition[pa];
	dk->dk_size = pep->nblocks;		/* Fill in size */
	dk->dk_start = pep->pblkoff;		/* Fill in starting block */
	dk->dk_blocksize = sc->sc_xtst.nbps;	/* Fill in blocksize */
	dk->dk_ntrack = sc->sc_xtst.ntrak;	/* Fill in track count */
	dk->dk_nsector = sc->sc_xtst.nsect;	/* Fill in sector count */
	dk->dk_ncyl = pep->nblocks / sc->sc_xtst.nspc;	/* Fill in cyl count */
	strcpy(dk->dk_name,"xt");		/* Fill in name */
	    DEBUGF(xtioctldebug,printf("%s%d: Xtioctl: DKIOCGPART done\n",
		xs->xs_xi->xi_driver->xd_sname, xs->xs_unit););
	return 0;

#ifdef	FROM_HD_DRIVER
    case DKIOCGPART:
	dk->dk_size = off->len;			/* size */
	if (off->start == 0 && off->len != 0)
		dk->dk_size -= st->nspc + MAXBADBLKS; /* grotty - make the
					/* proper available size for newfs */
	dk->dk_start = off->start * st->nspc;	/* start */
	dk->dk_blocksize = NBPS(st);		/* device blocksize */
	dk->dk_ntrack = st->ntpc;		/* tracks */
	dk->dk_nsector = st->nspt;		/* sectors */
	dk->dk_ncyl = off->len / st->nspc;	/* number of cylinders */
	strcpy(dk->dk_name, hdtypes[iod->iod_type]); /* copy name */
	return 0;
#endif

    case DIOCGDINFO:	/* from hd.c -md */
	dl = (struct disklabel *) data;
	xtpartitioninit(xs,sc);

	pa = XHAMIN_PART(dev);
	pep = &sc->sc_size.partition[pa];

	dl->d_magic = DISKMAGIC;
	dl->d_type = DTYPE_SCSI;
	dl->d_subtype = 0;
	/* be careful when copying inqname since it's larger than d_typename */
	i=sizeof(dl->d_typename)-1;
	strncpy(dl->d_typename, sc->sc_xtst.inqname, i);
	dl->d_typename[i]='\0';	/* just in case */
		/* disk geometry: */
	dl->d_secsize = sc->sc_xtst.nbps;
	dl->d_nsectors = sc->sc_xtst.nsect;
	dl->d_ntracks = sc->sc_xtst.ntrak;
	dl->d_ncylinders = sc->sc_xtst.ncyl;
	dl->d_secpercyl = sc->sc_xtst.nspc;
	dl->d_secperunit = dl->d_ncylinders * dl->d_ntracks *  dl->d_nsectors;
	dl->d_sparespertrack = 0;
	dl->d_sparespercyl = 0;
	dl->d_acylinders = 0;
	dl->d_magic2 = DISKMAGIC;
	dl->d_rpm = 3600;
	dl->d_interleave = 1;
	DEBUGF(xtioctldebug, 
	    printf("CGDINFO: secsize=%d nsectors=%d ntracks=%d ncyl=%d secpcyl=%d secpunit=%d\n",
	    dl->d_secsize, dl->d_nsectors, dl->d_ntracks, dl->d_ncylinders,
	    dl->d_secpercyl, dl->d_secperunit););

			/* filesystem and partition information: */
	dl->d_npartitions = MAXPARTITIONS;
	for (i = 0; i < NPART && i < MAXPARTITIONS; ++i) {
		dl->d_partitions[i].p_size = sc->sc_size.partition[i].nblocks;
		dl->d_partitions[i].p_offset = sc->sc_size.partition[i].pblkoff;
		if (dl->d_partitions[i].p_offset == 0
		    && dl->d_partitions[i].p_size != 0)
			dl->d_partitions[i].p_size -= dl->d_secpercyl + MAXBADBLKS;
		dl->d_partitions[i].p_fsize = 1024;
		dl->d_partitions[i].p_fstype = (i == 1 ? FS_SWAP : FS_BSDFFS);
		dl->d_partitions[i].p_frag = 8;
		dl->d_partitions[i].p_cpg = 16;
		}
	dl->d_bbsize = BBSIZE;
	dl->d_sbsize = SBSIZE;
	return 0;

    case DKIOCSTART:
	if(xtstartstop(xs,1) < 0)			/* Start unit */
	{
		return -1;
	}
	sc->sc_spundown = 0;
	return 0;

    case DKIOCSTOP:
	if(xtstartstop(xs,0) < 0)			/* Stop unit */
	{
		return -1;
	}
	sc->sc_spundown = 1;
	return 0;

    case DKIOCSAUTOSPIN:
	sc->sc_autospinup = 1;
	return 0;

    case DKIOCSNOAUTOSPIN:
	sc->sc_autospinup = 0;
	return 0;
    case DKIOCSMODESELECT:
	if (securelevel >= 2)	/* not allowed in high security states */
		return -1;
        if(xtmodeselect(xs,(struct scsi_msc *)data) < 0)    /* SCSI Mode Select */
        {
                return -1;
        }
        return 0;


    case DKIOCSMODESENSE:
        if(xtmodesense(xs,(struct scsi_msiobuf *) data) < 0)    /* SCSI Mode Sense */
        {
                return -1;
        }
        return 0;

    case DKIOCSSCSI:
	if (securelevel >= 2)	/* not allowed in high security states */
		return -1;
	if(xxioctl(xs, cmd, data, flag) < 0)	/* SCSI pass through */
	{
		return -1;
	}
	return 0;
    case DKIOCGSCSISENSE:
	DEBUGF(xtioctldebug,
		printf("Get SCSI sense:"););
	if(xs->xs_flags & XSFL_HASSENSE)
	{
		bcopy(&xs->xs_esense,data,sizeof(xs->xs_esense));
		DEBUGF(xtioctldebug,
			printf("Sense data delivered:"););
		DEBUGF(xtioctldebug,
			xxhdump(data,sizeof(xs->xs_esense)););
		return 0;
	}
	DEBUGF(xtioctldebug,printf("Nothing stored\n"););
	return -1;
    case DKIOCDSR:
	sc->sc_noretries = 1;
	return 0;
    case DKIOCESR:
	sc->sc_noretries = 0;
	return 0;
    default:
	return ENOTTY;
    }
}

xtminphys(struct buf *bp)
{
    register int xtunit = XTUNIT(bp->b_dev);
    register struct iocc_ctlr     *ic;
    register struct xha_slave    *xs;
    register struct xt_softc	  *sc;
    extern unsigned                 (*dma_get_minphys())();
#ifdef DEBUG
    long	bcount;
#endif /* DEBUG */
    DEBUGF(xttrace,
	printf("Xtminphys: bp=%x\n",bp););
    xs = xtinfo[xtunit];
    sc = &xt_softc[xs->xs_unit];

#if 0
    if(xtphys(bp->b_dev)) return;	/* Check existence and aliveness */
#endif
#ifdef DEBUG
    bcount = bp->b_bcount;
    if(xtminphysdebug && 		/* Debugging here ? */
       sc->sc_xtst.nbps && 		/* Nbps initalized ? */
       (bp->b_bcount & (sc->sc_xtst.nbps-1)))
    {
	printf("Xtminphys: bp=%x, bp->bcount=%d not modulo %d on entry\n",
		bp,bp->b_bcount,sc->sc_xtst.nbps);
    }
#endif /* DEBUG */

    minphys(bp);		/* Enforce kernel-wide b_bcount restriction */
#ifdef DEBUG
	if(xtminphysdebug && (bp->b_bcount != bcount))
	{
	    printf("Xtminphys bp=%x,bcount=%d changed by minphys() to %d\n",
		    bp,bcount,bp->b_bcount);
	    bcount = bp->b_bcount;
	}
#endif /* DEBUG */

    if(bp->b_bcount > 65536)
	bp->b_bcount = 65536 ; 
#ifdef DEBUG
	if(xtminphysdebug && (bp->b_bcount != bcount))
	{
	    printf("Xtminphys bp=%x,bcount=%d changed by xtminphys() to %d\n",
		    bp,bcount,bp->b_bcount);
	    bcount = bp->b_bcount;
	}
#endif /* DEBUG */


    if(xs && xs->xs_xi && xs->xs_xi->xi_hd &&  xs->xs_xi->xi_hd->xh_iocc )
    {
	ic = xs->xs_xi->xi_hd->xh_iocc;
	(*dma_get_minphys(ic))(bp);     /* dma's bcount restrictions */
    }
    else
    {
	if(xs)
	    printf("%s%d: Xtminphys: xtunit=%d, xs=%x\n", 
		xs->xs_xi->xi_driver->xd_sname, xs->xs_unit, xtunit, xs);
	else
	    printf("xt%d: Xtminphys: xs = NULL\n",xtunit);

	if(xs)
	    printf("      Xtminphys: xs->xs_xi=%x\n", xs->xs_xi);

	if(xs && xs->xs_xi)
	    printf("      Xtminphys: xs->xs_xi->xi_hd=%x\n", xs->xs_xi->xi_hd);

	if(xs && xs->xs_xi && xs->xs_xi->xi_hd)
	    printf("      Xtminphys: xs->xs_xi->xi_hd->xh_iocc=%x\n", 
		xs->xs_xi->xi_hd->xh_iocc);
	dma_minphyspage(bp);
    }
#ifdef DEBUG
	if(xtminphysdebug && (bp->b_bcount != bcount))
	{
	    printf("Xtminphys bp=%x,bcount=%d changed by dma_get_minphys() to %d\n",
		    bp,bcount,bp->b_bcount);
	    bcount = bp->b_bcount;
	}
#endif /* DEBUG */
    /*
     * Dma minphys routines can leave us w/ a partial
     * xfer *and* leftovers.  We can't handle that so
     * we truncate any partial (pushing it into leftovers).
     */
    if (sc->sc_xtst.nbps && 	/* Nbps initialized ? */
	(bp->b_bcount >= sc->sc_xtst.nbps))
	bp->b_bcount &= ~(sc->sc_xtst.nbps - 1);
#ifdef DEBUG
	if(xtminphysdebug && (bp->b_bcount != bcount))
	{
	    printf("Xtminphys bp=%x,bcount=%d corrected by xtminphys() to %d\n",
		    bp,bcount,bp->b_bcount);
	    bcount = bp->b_bcount;
	}
    if(xtminphysdebug && 
       sc->sc_xtst.nbps &&
       (bp->b_bcount & (sc->sc_xtst.nbps-1)))
    {
	printf("Xtminphys: bp=%x, bp->bcount=%d not modulo %d on exit.\n",
		bp,bp->b_bcount,sc->sc_xtst.nbps);
	bcount = bp->b_bcount;
    }
#endif /* DEBUG */

}

xttryretry(struct xha_slave *xs, struct buf *bp, int maxretries, int errcode)
{
    register struct xt_softc *sc = &xt_softc[xs->xs_unit];
    DEBUGF(xttrace,
	printf("Xttryretry: xs=%x,bp=%x,maxretries=%d,errcode=%d\n",
		xs,bp,maxretries,errcode););
    if(!((xs->xs_flags & XSFL_SPECSCSI) && sc->sc_noretries) &&
	  ++bp->b_errcnt <= maxretries)
	return XHD_RETRY;
    else
    {
	bp->b_bcount = bp->b_resid;
	bp->b_resid = 0;
	bp->b_error = errcode;
	bp->b_flags |= B_ERROR;
	xtpartdone(xs,bp);
	xs->xs_flags &= ~XSFL_SPECSCSI;
	return XHD_DONE;
    }
}

xtgetpartitions(struct xha_slave *xs)
{
	struct	buf	minibuf;
	char	mmbuf[1024];
	struct	minidirectory	*mdp= (struct minidirectory *) &mmbuf ;
	register struct minidisk *mdiskp;
	register int m,nxt;
	int	i=0;
	register int pa,nparts = 0;
	register struct xt_softc *sc = &xt_softc[xs->xs_unit];

	DEBUGF(xttrace,
	    printf("Xtgetpartitions: xs=%x\n", xs););
	/* Set up a xfer buffer */
	
	minibuf.b_un.b_addr =  (caddr_t)mdp;
	minibuf.b_bcount = sizeof mmbuf;
        minibuf.b_flags = B_BUSY | B_READ;
	minibuf.b_blkno = MINIDISK_BLOCK;

	/* Start the xfer */
	DEBUGF(xtgetpartdebug,printf("%s%d: Starting minidisk read\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););

	xtxstrat(xs,2,&minibuf);

	/* 
	 * Await completion, need busy wait here in order to aviod
         * reentrance 
         */
	DEBUGF(xtgetpartdebug,printf("%s%d: Waiting for minidisk read\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););

	while((minibuf.b_flags & B_DONE) == 0)
		if (++i > 10000000) break;
	if(minibuf.b_flags & B_DONE == 0)
	{
	
	    DEBUGF(xtgetpartdebug,printf("%s%d: Minidisk read timed out\n",
		    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
	    return 0;
	}
	DEBUGF(xtgetpartdebug,printf("%s%d: Minidisk read done\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););

	if (minibuf.b_flags & B_ERROR)
	{
	    DEBUGF(xtgetpartdebug,printf("%s%d: Minidisk read failed\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
	    return 0;
	}
	if ( mdp->header.number <= 0 ||
            mdp->header.number >= MAXDISKS)
	{
	    DEBUGF(xtgetpartdebug,printf("%s%d: Minidisk contained bad data\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
	    return 0;
	}

	DEBUGF(xtgetpartdebug,printf("%s%d: Minidisk read OK\n",
		xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););

	m = mdp->header.number;
	
	for(nxt = mdp->header.first; nxt >= 0; nxt = mdiskp->next)
	{
	    int nlen = strlen(xs->xs_xi->xi_driver->xd_sname);
	    if(--m < 0 || nxt > MAXDISKS)
	    {
		DEBUGF(xtgetpartdebug,printf("%s%d: Minidisk corrupted\n",
		    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit););
		return 0;
	    }
	    mdiskp = &(mdp->minidisk[nxt]);
	    if (ISFREE(mdiskp))
	    {
		continue;
	    }
	    pa = mdiskp->name[nlen+1] - 'a';
	    if((strncmp(xs->xs_xi->xi_driver->xd_sname, mdiskp->name,nlen) == 0 &&
		pa >= 0 && pa < MAXNPART ) ||
	       (mdiskp->type == TYPE_PAGE && (pa == 1) ) )
	    {
		nparts++;
		DEBUGF(xtgetpartdebug,
		    printf("%s%d: Partition %s start at %d, size = %d\n",
			    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,
			    mdiskp->name,mdiskp->start,mdiskp->size););
		sc->sc_size.partition[pa].pblkoff = mdiskp->start;
		sc->sc_size.partition[pa].nblocks = mdiskp->size;
	    }
	}
	DEBUGF(xtgetpartdebug,printf("%s%d: xtgetpart done, nparts=%d\n",
	    xs->xs_xi->xi_driver->xd_sname,xs->xs_unit,nparts););
	return nparts;
}
/* 
 * Combine three bytes into an unsigned
 */
unsigned
xtget3(char *p)
{
    return ((*p & 0xFF) << 16) | ((p[1] & 0xFF) << 8) | (p[2] & 0xFF);
}

xtidcpy(u_char *f,char *t,int size)
{
    register int	i;

    bcopy((char *)f,t,size-1);
    t[size-1] = '\0';

    for(i=size-1;i>=0;i--)
    {
	if((t[i] != ' ') && (t[i] != '\0'))
	{
	    break;
	}
	t[i] = '\0';
    }
}
#endif /* NXT  > 0 */
