/*
 * 5799-WZQ (C) COPYRIGHT IBM CORPORATION  1986,1987,1988
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/* $Header: /sys/rt/rt/RCS/vm_machdep.c,v 1.11 1994/05/22 12:41:21 roger Exp $ */
/* $ACIS:vm_machdep.c 12.0$ */
/* $Source: /sys/rt/rt/RCS/vm_machdep.c,v $ */

#if !defined(lint) && !defined(NO_RCS_HDRS)
static char *rcsid = "$Header: /sys/rt/rt/RCS/vm_machdep.c,v 1.11 1994/05/22 12:41:21 roger Exp $";
#endif

/*     vm_machdep.c    6.1     83/07/29        */
/* $Header: /sys/rt/rt/RCS/vm_machdep.c,v 1.11 1994/05/22 12:41:21 roger Exp $ */
/* $ACIS:vm_machdep.c 12.0$ */
/* $Source: /sys/rt/rt/RCS/vm_machdep.c,v $ */

#include "rt/rt/debug.h"
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/cmap.h>
#include <sys/vm.h>
#include <sys/text.h>
#include <sys/buf.h>

#include <machine/pte.h>
#include <machine/mmu.h>
#include <machine/reg.h>
#include "rt/include/io.h"
#include "rt/include/frame.h"


/*
 * Set a red zone in the kernel stack after the u. area.
 */
/* ARGSUSED */
setredzone(pte, vaddr)
	register struct pte *pte;
	caddr_t vaddr;
{
	DEBUGF (vmdebug,
		printf("setredzone( pte=%x )\n", pte));
	*(int *)(pte - 1) = PG_URKR;	  /* fake pte in memory of VAX */
}


/*
 * Map a page into the HATIPT.  Make sure the hash chains are always valid,
 * as MMU may saunter down one at any time.
 */

mapin(pte, vpage, model)
	struct pte *pte;
	register u_int vpage;
	register u_int model;		  /* THIS IS REALLY A STRUCT PTE */
{
	register struct hatipt_entry *ipte, *head;
	register struct cmap *c;
	u_int sid, key;

	DEBUGF(vmdebug,
		printf("mapin(pte=0x%x,vpage=0x%x,model=0x%x)",
		    pte, vpage, model));

	/* point to the ipte for this page frame */
	ipte = &MMU_HATIPT[model & PG_PFNUM];
	if (ipte->ipt_ptr != MMU_UNDEF_PTR) { /* debug */
		panic("dup mapin");
	}

	/* get the segment id (placed there by memall or setsid) */
	sid = (ipte->key_addrtag & MMU_ADDRTAG_MASK) >> MMU_VPAGE_BITS;

	/* determine the vpage and segment from the virt addr */
	vpage &= btop(0x0fffffff);

	/* determine the key, write, tid, lockbits */
	key = rtakey(model);	  /* non-special segments only */

	/* fill in the fields to reflect the new virt addr and prot */
	ipte->key_addrtag = (key << MMU_KEY_SHIFT)
		|(sid << MMU_VPAGE_BITS)
		|vpage;
	ipte->w = 1;		  /* special segments only */
	ipte->tid = 0;		  /* special segments only */
	ipte->lockbits = -1;	  /* special segments only */

	/* hash to find the head of the ipte chain */
	head = &MMU_HATIPT[MMU_HASH(sid, vpage)];

	/* link in the new ipte (first on the chain) */
	ipte->ipt_ptr = head->hat_ptr;
	set_hatptr(head, model & PG_PFNUM);

	/* paranoia */
	invalidate_tlb(vpage);

	/* set the mod bit to reflect prot; reset the referenced bit */
	set_refmod_bits(model & PG_PFNUM, 0, ((model & PG_M) != 0));

	/* remember that this page is mapped in if in cmap */
	if ((c = &cmap[pgtocm(model&PG_PFNUM)]) > cmap && c < ecmap)
		c->c_mapped = 1;

	/* set up the vax page table entry */
	*(unsigned *)pte = model;
}


rtakey(apte)
	register unsigned apte;		  /* THIS IS REALLY A STRUCT PTE */
{
	switch (apte & PG_PROT) {
	case PG_KW:
		return (MMU_KEY_KW);
	case PG_URKW:
		return (MMU_KEY_URKW);
	case PG_UW:
#ifdef SIM_CHG_BIT
		/* 
		 * if we are simulating the change bit then if M bit is on
		 * allow read-write access; otherwise make it read-only,
		 * so that we can determine when it is modified.
		 */
		return ((apte & PG_M) ? MMU_KEY_UW : MMU_KEY_URKR);
#else
		return (MMU_KEY_UW);
#endif
	case PG_URKR:
		return (MMU_KEY_URKR);
	default:
		printf("\nvax pte = 0x%x\n", apte);
		panic("rtakey");
	}
	/*NOTREACHED*/
}


/* set the segment register for the specified virtual address */

alias(pte, vpage)
	register struct pte *pte;
	register u_int vpage;
{
	register struct hatipt_entry *ipte;
	register u_int sid, seg;

	DEBUGF (vmdebug,
		printf("alias( pte 0x%x -> 0x%x, vpage=0x%x ) ",
		    pte, *(unsigned *)pte, vpage));

	/* point to the ipte for this page frame */
	ipte = &MMU_HATIPT[pte->pg_pfnum];

	/* get the segment id (placed there by memall or setsid) */
	sid = (ipte->key_addrtag & MMU_ADDRTAG_MASK) >> MMU_VPAGE_BITS;

	/* determine the vpage and segment from the virt addr */
	seg = (vpage & btop(0xf0000000)) >> MMU_VPAGE_BITS;

	DEBUGF (vmdebug,
		printf("sid=%x,seg=%x\n", sid, seg));

	/* load the segment register for this seg */
	set_segreg(seg, sid);
}

/*
 * reset the segment register after kernel is finished with it.
 * We determine the segment number from the virtual address given.
 */

unalias(addr)
caddr_t addr;
{
	register u_int seg;

	seg = btop(addr) >> MMU_VPAGE_BITS;
	set_segreg(seg, MMU_SID_UNUSED);
}
	

/*
 * Unmap a page from the HATIPT, and keep those hash chains valid !
 */

mapout(pte)
	struct pte *pte;
{
	register struct hatipt_entry *ipte, *this, *next;
	register u_int t, sid, vpage;
	register struct cmap *c;

	DEBUGF (vmdebug,
		printf("mapout:  pte=%x,", pte));

	/* point to the ipte for this page frame */
	ipte = &MMU_HATIPT[pte->pg_pfnum];

	/* look up the sid and vpage fields */
	t = ipte->key_addrtag & MMU_ADDRTAG_MASK;
	sid = t >> MMU_VPAGE_BITS;
	vpage = t & MMU_VPAGE_MASK;

	/* hash to find the head of the ipt chain */
	this = &MMU_HATIPT[MMU_HASH(sid, vpage)];

	/* check the chain for empty */
	if (MMU_ENDCHAIN(t = get_hatptr(this)))
		panic("mapout: empty hash chain");
	next = &MMU_HATIPT[t];

	/* if first on the hash chain, simply point hash anchor table at the
	   successor of the entry being unmapped. */
	if (next == ipte) {	  /* was first, unlink */
		this->hat_ptr = ipte->ipt_ptr;
	}
	/* end case this one first */

	else {			  /* not first, chase down the chain */
		while (next != ipte) {
			/* go on */
			this = next;
			/* not found? */
			if (MMU_ENDCHAIN(t = get_iptptr(this)))
				panic("mapout: ipte not on chain");
			next = &MMU_HATIPT[t];
		}
		/* point predecessor of entry being unmapped to successor of *
		   entry being unmapped. */
		this->ipt_ptr = ipte->ipt_ptr;
	}
	/* end case not first */

	ipte->ipt_ptr = MMU_UNDEF_PTR; /* debug */
	invalidate_tlb(vpage);

	/* remember that this page is unmapped in if in cmap */
	if ((c = &cmap[pgtocm(pte->pg_pfnum)]) > cmap && c < ecmap)
		c->c_mapped = 0;

	/* don't clear the vax pte */
	/* *(int *)pte++ = 0;          */
}


/*
 * Stat counters
 */
int rtbufmapped = 0;
int rtbufunmapped = 0;

/*
 * Do a temporary mapping of the I/O buffer pages from a process into
 * the kernel's virtual address space.  Sometimes useful if we are doing
 * physical I/O from a "device" which requires copying from kernel space.
 *
 * RT's are awkward in that you can't have pages mapped into two address
 * spaces at once.  The code checks to see if the pages are already mapped
 * somewhere and, if so, unmaps them from whereever this is first.  It
 * returns enough information to undo this damage.  In practice, however,
 * it appears that the file system code which uses this (mostly for paging
 * and swapping) always calls with a set of unmapped pages.
 */
long
rtbufmapin(ppte, kpte, kvpage, npg, model)
	register struct pte *ppte;	/* the proc's ptes */
	register struct pte *kpte;	/* the kernel's ptes */
	u_int kvpage;			/* kernel virtual page number */
	int npg;			/* number of pages to remap */
	u_int model;			/* THIS IS REALLY A STRUCT PTE */
{
	register struct hatipt_entry *ipte, *this, *next;
	register int i;
	register u_int t;
	u_int sid, vpage;
	long p_key_addrtag, k_key_addrtag;
	int mapped;

	/*
	 * Make up a key_addrtag field for the first kernel page.
	 * We probably shouldn't assume the system segment, but do
	 * anyway.
	 */
	k_key_addrtag = (rtakey(model) << MMU_KEY_SHIFT)
			| (MMU_SID_SYSTEM << MMU_VPAGE_BITS)
			| (kvpage & MMU_VPAGE_MASK);

	/*
	 * See if these pages are mapped or not.  If so we'll
	 * need to map it out first.  We look at the cmap for
	 * this.  If the page isn't there, assume it is mapped.
	 */
	ipte = &MMU_HATIPT[ppte->pg_pfnum];
	p_key_addrtag = ipte->key_addrtag;
	if (ipte->ipt_ptr == MMU_UNDEF_PTR) {
		rtbufunmapped++;
		mapped = 0;
	} else {
		rtbufmapped++;
		mapped = 1;
	}

	for (i = 0; i < npg; i++, ppte++, kpte++, k_key_addrtag++) {
		if (mapped) {
			if (i == 0) {
				/* ipte init'd outside loop */
				sid = (p_key_addrtag & MMU_SID_MASK)
				    >> MMU_VPAGE_BITS;
				vpage = p_key_addrtag & MMU_VPAGE_MASK;
			} else {
				ipte = &MMU_HATIPT[ppte->pg_pfnum];
				if (ipte->key_addrtag != (p_key_addrtag + i))
					panic("rtbufmapin discontiguous pages");
				vpage++;
			}

			/* hash to find the head of the ipt chain */
			this = &MMU_HATIPT[MMU_HASH(sid, vpage)];

			/* check the chain for empty */
			if (MMU_ENDCHAIN(t = get_hatptr(this)))
				panic("rtbufmapin empty hash chain");
			next = &MMU_HATIPT[t];

			/* if first on the hash chain, simply point hash anchor
			  table at the successor of the entry being unmapped. */
			if (next == ipte) {	  /* was first, unlink */
				this->hat_ptr = ipte->ipt_ptr;
			} else {	  /* not first, chase down the chain */
				while (next != ipte) {
					/* go on */
					this = next;
					/* not found? */
					if (MMU_ENDCHAIN(t = get_iptptr(this)))
					  panic("rtbufmapin ipte not on chain");
					next = &MMU_HATIPT[t];
				}
				/* point predecessor of entry being unmapped to
				   successor of entry being unmapped. */
				this->ipt_ptr = ipte->ipt_ptr;
			}
		} else if (i > 0) {
			ipte = &MMU_HATIPT[ppte->pg_pfnum];
			if (ipte->key_addrtag != p_key_addrtag)
				printf("rtbufmapin CONFUSED\n");
		}

		/*
		 * Now the ipt entry is disconnected from the old hash
		 * chain.  Initialize to the system sid and address and
		 * add it back to a hash chain.
		 */
		ipte->key_addrtag = k_key_addrtag;
		this = &MMU_HATIPT[MMU_HASH(MMU_SID_SYSTEM,
				(k_key_addrtag & MMU_VPAGE_MASK))];

		/* link in the new ipte (first on the chain) */
		ipte->ipt_ptr = this->hat_ptr;
		set_hatptr(this, ppte->pg_pfnum);

		/* paranoia, invalidate kernel virtual page */
		invalidate_tlb(k_key_addrtag & MMU_VPAGE_MASK);

		/* set up the vax page table entry */
		*(u_int *)kpte = model | ppte->pg_pfnum | (mapped?PG_FOD:0);

		/* I hope this does the right thing with the dirty bits */
	}

	/*
	 * Return the process' ipte key_addrtag word to use for restoring
	 * the pages.
	 */
	return (p_key_addrtag);
}


/*
 * Undo the damage done by rtbufmapin
 */
rtbufmapout(ppte, p_key_addrtag, kpte, npg)
	register struct pte *ppte;	/* the proc's ptes */
	long p_key_addrtag;		/* for the proc's first ipt entry */
	register struct pte *kpte;	/* the kernel's ptes */
	int npg;			/* number of pages to remap */
{
	register struct hatipt_entry *ipte, *this, *next;
	register int i;
	register u_int t;
	u_int psid;
	long k_key_addrtag;

	/*
	 * Get the sid for the process
	 */
	psid = (p_key_addrtag & MMU_SID_MASK) >> MMU_VPAGE_BITS;

	for (i = 0; i < npg; i++, ppte++, kpte++) {

		/*
		 * Sanity.  Check to see that the process and the kernel
		 * point to the same page.
		 */
		if (ppte->pg_pfnum != kpte->pg_pfnum)
			panic("rtbufmapout pages not same");

		/* point to the ipte for this page frame */
		ipte = &MMU_HATIPT[kpte->pg_pfnum];

		/*
		 * Get the key_addrtag word.  Sanity check to make sure
		 * it belongs to the kernel.
		 */
		k_key_addrtag = ipte->key_addrtag;
		if (MMU_SID_SYSTEM !=
		    (k_key_addrtag & MMU_SID_MASK) >> MMU_VPAGE_BITS)
			panic("rtbufmapout not system sid");

		/* hash to find the head of the ipt chain */
		this = &MMU_HATIPT[MMU_HASH(MMU_SID_SYSTEM,
				k_key_addrtag&MMU_VPAGE_MASK)];

		/* check the chain for empty */
		if (MMU_ENDCHAIN(t = get_hatptr(this)))
			panic("rtbufmapout empty hash chain");
		next = &MMU_HATIPT[t];

		/* if first on the hash chain, simply point hash anchor
		   table at the successor of the entry being unmapped. */
		if (next == ipte) {	  /* was first, unlink */
			this->hat_ptr = ipte->ipt_ptr;
		} else {	  /* not first, chase down the chain */
			while (next != ipte) {
				/* go on */
				this = next;
				/* not found? */
				if (MMU_ENDCHAIN(t = get_iptptr(this)))
					panic("rtbufmapout ipte not on chain");
				next = &MMU_HATIPT[t];
			}
			/* point predecessor of entry being unmapped to
			   successor of entry being unmapped. */
			this->ipt_ptr = ipte->ipt_ptr;
		}


		/*
		 * Now the ipt entry is disconnected from the old hash
		 * chain.  If it was mapped previously, map it back to
		 * the old address.
		 */
		if (kpte->pg_fod) {
			ipte->key_addrtag = p_key_addrtag + i;
			this = &MMU_HATIPT[MMU_HASH(psid,
					((p_key_addrtag+i) & MMU_VPAGE_MASK))];

			/* link in the new ipte (first on the chain) */
			ipte->ipt_ptr = this->hat_ptr;
			set_hatptr(this, kpte->pg_pfnum);
		} else {
			ipte->key_addrtag = p_key_addrtag;
			ipte->ipt_ptr = MMU_UNDEF_PTR;
		}
	}
}


/*
 * Check for valid program size
 * NB - Check data and data growth separately as they may overflow 
 * when summed together.
 */
chksize(ts, ids, uds, ss)
	unsigned ts, ids, uds, ss;
{
	extern unsigned maxtsize;

	if (ctob(ts) > maxtsize ||
	    ctob(ids) > u.u_rlimit[RLIMIT_DATA].rlim_cur ||
	    ctob(uds) > u.u_rlimit[RLIMIT_DATA].rlim_cur ||
	    ctob(ids + uds) > u.u_rlimit[RLIMIT_DATA].rlim_cur ||
	    ctob(ss) > u.u_rlimit[RLIMIT_STACK].rlim_cur) {
		return (ENOMEM);
	}
	return (0);
}


/*ARGSUSED*/
newptes(pte, v, size)
	register struct pte *pte;
	register u_int v;
	register int size;
{
	while (--size >= 0) {
	DEBUGF (vmdebug,
		printf("newptes: v=0x%x, pte 0x%x -> 0x%x\n", v, pte, *(unsigned *) pte));
		if (pte->pg_v)
			mapin(pte, v, *(unsigned *)pte);
		invalidate_tlb(v);
		pte++;
		v++;
	}
}


rtsetsid(p, pf, type)
	register struct proc *p;
	u_int pf;
	register int type;
{
	register struct hatipt_entry *ipte;
	unsigned sid;

	switch (type) {
	case CSYS:
		sid = MMU_SID_SYSTEM;	/* kernel data */
		break;
	case CTEXT:
		sid = make410sid(p->p_textp);
		break;
	case CDATA:
		sid = p->p_tsize ?
		    makeUsid(p) :	/* 410/413 data in P1 */
		    make407sid(p);	/* 407 data in P0 */
		break;
	case CSTACK:
		sid = makeUsid(p);	/* user stack in P1 */
		break;
	case CUSTRUCT:
		sid = makeUsid(p);	/* upages in P1 */
		break;
	default:
		printf("setsid: unknown c_type 0x%x\n", type);
		panic("setsid");
	}
	ipte = &MMU_HATIPT[pf];
	(*ipte).key_addrtag = sid << MMU_VPAGE_BITS;
}


/*
 * getpage() fetches a page into memory if it can.  It returns
 * 1 if it does, zero otherwise.
 *
 * There are two possibilities here.  Either the page is part of
 * the process, in which case we page it in, or the page is just
 * off the start of the stack, in which case we grow the stack.
 * Otherwise we return an error.
 */
getpage(vaddr, dlyu, sp)
	unsigned vaddr;
	int dlyu;
	int sp;
{
	register unsigned v = clbase(btop(vaddr));
	register struct proc *p = u.u_procp;

	if (isanysv(p, v)) {
		/* okay, page it in */
		pagein(vaddr, dlyu);
	} else if (sp == 0 || !GROWSP(sp)) {
		/* no luck */
		return (0);
	}
	return (1);
}


#ifndef NEWPORT
 /* TODO: pass more info to this proc so that non VTL_RELOCATE hardware
  * can selectively purge TLB entries as required.  Better yet, make is
  * proc a preprocessor macro on MMU harwdare.
 */
invalidate_tlb(vpage)
	register unsigned vpage;
{
	register unsigned i;

	i = (vpage & (MMU_NTLBS - 1)) * MMU_TLBSTEP;
	iow(MMU_TLBAW1 + i, MMU_TLBUNUSED);
	iow(MMU_TLBBW1 + i, MMU_TLBUNUSED);
}


#else

/* TODO: pass more info to this proc so that non VTL_RELOCATE hardware
 * can selectively purge TLB entries as required.  Better yet, make is
 * proc a preprocessor macro on MMU harwdare:
 * #define invalidate_tlb(vpage) iow(MMU_INV_ADDR, ptob(vpage))
 */
invalidate_tlb(vpage)
	register unsigned vpage;
{

	iow(MMU_INV_ADDR, ptob(vpage));
}


#endif

/*
 * Change protection codes of text segment.
 * Have to flush translation buffer since this
 * affect virtual memory mapping of current process.
 */
chgprot(addr, tprot)
	caddr_t addr;
	long tprot;
{
	unsigned v;
	int tp;
	register struct pte *pte;
	register struct cmap *c;
	register struct hatipt_entry *ipte;

	DEBUGF (vmdebug,
		printf("chgprot(addr=%x,tprot=%x)\n", addr, tprot));

	v = clbase(btop(addr));
	if (!isatsv(u.u_procp, v))
		return (EFAULT);
	tp = vtotp(u.u_procp, v);
	pte = tptopte(u.u_procp, tp);
	if (pte->pg_fod == 0 && pte->pg_pfnum) {
		c = &cmap[pgtocm(pte->pg_pfnum)];
		if (c->c_blkno)
			munhash(c->c_vp, (daddr_t)(u_long)c->c_blkno);
	}
	*(int *)pte &= ~PG_PROT;
	*(int *)pte |= tprot;

	/* fix ipte for non-special segments only -- added for ibm032 6/84 */
	if (pte->pg_fod == 0 && pte->pg_pfnum) {
		ipte = &MMU_HATIPT[pte->pg_pfnum];
		ipte->key_addrtag = (ipte->key_addrtag & (~MMU_KEY_BITS))
			|(rtakey(*(int *)pte) << MMU_KEY_SHIFT);
		invalidate_tlb(v);
	}
	distcl(pte);
	return (0);
}


settprot(tprot)
	register long tprot;
{
	register unsigned *pteadr, i;
	register struct hatipt_entry *ipte;

	DEBUGF (vmdebug,
		printf("settprot(tprot=%x,pid=%d)\n", tprot, u.u_procp->p_pid));

	pteadr = (unsigned *)u.u_pcb.pcb_p0br;
	for (i = 0; i < u.u_tsize; i++, pteadr++) {
		*pteadr &= ~PG_PROT;
		*pteadr |= tprot;

		/* fix ipte for non-special segments only */
		if (!(*pteadr & PG_FOD) && (*pteadr & PG_PFNUM)) {
			ipte = &MMU_HATIPT[*pteadr & PG_PFNUM];
			ipte->key_addrtag = (ipte->key_addrtag &(~MMU_KEY_BITS))
				|(rtakey(*pteadr) << MMU_KEY_SHIFT);
		}
	}
}


/*
 * Rest are machine-dependent
 */

getmemc(addr)
	caddr_t addr;
{
#ifdef notdef
	register int c;
	struct pte savemap;

	savemap = mmap[0];
	*(int *)mmap = PG_V | PG_KR | btop(addr);
	mtpr(TBIS, vmmap);
	c = *(char *) & vmmap[(int)addr & PGOFSET];
	mmap[0] = savemap;
	mtpr(TBIS, vmmap);
	return (c & 0377);
#endif
	printf("getmemc(addr=%x)\n", addr);
	return (0);
}


putmemc(addr, val)
	caddr_t addr;
{
#ifdef notdef
	struct pte savemap;

	savemap = mmap[0];
	*(int *)mmap = PG_V | PG_KW | btop(addr);
	mtpr(TBIS, vmmap);
	*(char *) & vmmap[(int)addr & PGOFSET] = val;
	mmap[0] = savemap;
	mtpr(TBIS, vmmap);
#endif
	printf("putmemc(addr=%x,val=%x)\n", addr, val);
}


/*
 * Move pages from one kernel virtual address to another.
 * Both addresses are assumed to reside in the Sysmap,
 * and size must be a multiple of CLSIZE.
 */
pagemove(from, to, size)
	register caddr_t from, to;
	register int size;
{

	register caddr_t ra;
	struct pte temp_pte;

	DEBUGF (vmdebug,
		printf("pagemove(from=0x%x,to=0x%x,size0x%x)\n", from, to, size));

	if (size % CLBYTES)
		panic("pagemove");
	while ((size -= NBPG) >= 0) {
		if ((ra = (caddr_t)vtop(from)) == (caddr_t) - 1)
			panic("pagemove vtop");
		temp_pte.pg_pfnum = btop(ra);
		mapout(&temp_pte);
		mapin(&temp_pte, btop(to), PG_V | PG_KW | (btop(ra)));
		from += NBPG;
		to += NBPG;
	}
}


/*
 * Copy one page (NBPG bytes) from the running process' virtual
 * memory address "from", to the same virtual memory address in
 * the process owning "pte".
 */
cloneseg(from, pte)
	register from;
	register struct pte *pte;
{
	register to;
	extern copybase;

	DEBUGF (vmdebug,
		printf("cloneseg( from=0x%x, to pte 0x%x -> 0x%x)\n",
		    from, pte, *(unsigned *)pte));

	mapin(pte, btop(from), PG_V | PG_M | PG_UW | (*(unsigned *)pte));
	to = ((int) & copybase) | (from & SEGMENT_OFFSET);
	alias(pte, btop(to));
#ifdef notdef
	bcopy(from, to, NBPG);
#else
	*(int *)to = *(int *)from;		/* force pagein to occur */
	copypage(vtop(from), vtop(to));
	unalias((caddr_t) to);			/* get rid of alias */
#endif
}


/*
 * Clear one relocation unit (NBPG bytes)
 */
#ifdef notdef
clearseg(where)				  /* === see locore.ruasm === */
{
	printf("clearseg: where=%x,ptov(where)=%x\n", where, ptov(ctob(where)));
	mfill(ptov(ctob(where)), ctob(1), 0);
}


#endif
 /*
  * map virtual address(bytes) to physical address(bytes)
  */

#ifdef notdef	/* now done in loutil.s */
vtop(where)
	register int *where;
{
	register caddr_t paddr;
	register u_int x;

	x = splimp();
	iow(MMU_VTOP, where);
	paddr = (caddr_t)ior(MMU_REALADDR);
	splx(x);
	return ((int)paddr < 0 ? -1 : (int)paddr);
}


#endif

/*
 * ptov - map physical address(bytes) to virtual address(bytes)
 */
ptov(where)
	register unsigned int where;
{
	register struct hatipt_entry *ipte;
	register unsigned int rpage, addrtag, vpage, sid;
	rpage = btop(where);		  /* shift by log2(pagesize) */
	ipte = &MMU_HATIPT[rpage];
	addrtag = ipte->key_addrtag & 0x1fffffff;
/*printf("ptov:  where=%x,addrtag=%x,ipte=%x  ",where,addrtag,ipte);*/
	vpage = addrtag & MMU_VPAGE_MASK;
	sid = addrtag >> MMU_VPAGE_BITS;
/*printf("  rpage=%x,addrtag=%x,vpage=%x,sid=%x - ",rpage,addrtag,vpage,sid);*/
	where = (where & PGOFSET) | ctob(vpage);
	if (sid == MMU_SID_SYSTEM) {
/*printf("system:%x\n",where+KERNBASE);*/
		return (where | KERNBASE);
	} else if (sid == MMU_SID_UNUSED) {
/*printf("unused\n");*/
		return (-1);
	} else if (sid < MMU_SID_SYSTEM) { /* user text */
/*printf("user text:%x\n",where);*/
		return (where);
	} else {
/*printf("user data:%x\n",where | DATABASE);*/
		return (where | DATABASE);
	}
}


 /* end of ptov */

/*
 * check the validity of an access to a range of virtual addresses
 */
isitok(vaddr, length, access)
	caddr_t vaddr;
	unsigned length;
	register unsigned access;
{
	register unsigned v;
	register unsigned l;
	l = btop(((unsigned)vaddr)+length-1);	/* last page */
	v = btop(((unsigned)vaddr));		/* first page */

	if (l < v)			  /* Negative or really huge values */
		return(0);		  /* are no good. */

	if (v < btop(ENDOFP1)-UPAGES) {		  /* user virtual address */
		register struct pte *pte;
		{
/* calculate pte and v_max (largest legal virtual address in segment) */
			register struct proc *p;
			register unsigned v_max;

			p = u.u_procp;

			if (isatsv(p, v)) {
				pte = tptopte(p, vtotp(p, v));
				v_max = (p)->p_tsize;
			} else if (ISADSV(p, v)) {	/* already done isatsv */
				pte = dptopte(p, vtodp(p, v));
				v_max = (p)->p_dsize + ((p)->p_tsize ? BTOPDATABASE : 0);
			} else if (isassv(p, v)) {
				pte = sptopte(p, vtosp(p, v));
				v_max = BTOPUSRSTACK;
			} else
				return (0);
			if (v >= v_max || l >= v_max)
				return (0);
		}
		for (;v <= l;++v) {
			register int i;
			i = *(int *)pte++ & PG_PROT;
			switch (access) {
			case UWRITE_OK:
				if (i == PG_UW)
					break;
				return (0);
			case UREAD_OK:
				if (i >= PG_UW)
					break;
				return (0);
			case KWRITE_OK:
				if (i == PG_UW || i == PG_URKW || i == PG_KW)
					break;
				return (0);
			case KREAD_OK:
				if (i != PG_NOACC)
					break;
				return (0);
			}
			/* end switch */
		}

		return (1);
	}

	if (access != KREAD_OK && access != KWRITE_OK)
		return (0);

	/* kernel virtual address -- used only by kernel memory device driver */

	{
		register unsigned vtmp = v >> IOPAGESHIFT;

		if (vtmp == (IOSPACE >> IOSPACESHIFT) ||
				vtmp == (IOSPACE2 >> IOSPACESHIFT))
			return ((vtmp == (l >> IOPAGESHIFT)) ? 1 : 0);
	}
	for (;v <= l;++v) {
		register int i;
		if ((i = vtop(ptob(v)) >> PGSHIFT) == -1)
			return (0);
		if (!(MMU_KEYBIT(MMU_HATIPT[i].key_addrtag) & access))
			return (0);
	}

	return (1);
}


#ifdef notdef
 /*
  * check the validity of a future access to a virtual addresses
  */
willbeok(vaddr, access)
	register caddr_t vaddr;
	register unsigned access;
{
	register struct pte *pte;

	/* don't ever use this, vtopte changed definitions */
	return ((pte = vtopte(u.u_procp, btop(((unsigned)vaddr))))
			&&pte->pg_fod
	    &&(access & MMU_KEY_OK(rtakey(*(unsigned *)pte)))
		);
}
#endif

/*
 * translate a virtual address into a real memory address.
 * The virtual address need not be mapped into an address space.
 */
caddr_t
real_buf_addr(bp, bufaddr)
	register struct buf *bp;
	register char *bufaddr;
{
	register v, o;
	register struct pte *pte;
	register struct proc *p;

	if ((bp->b_flags & B_PHYS) == 0)
		return ((caddr_t)vtop(bufaddr));

	v = btop(bufaddr);
	o = (int)bufaddr & PGOFSET;
	p = bp->b_flags & B_DIRTY ? &proc[2] : bp->b_proc;

	if (bp->b_flags & B_UAREA)
		pte = &p->p_addr[v];
	else if (bp->b_flags & B_PAGET)
		pte = &Usrptmap[btokmx((struct pte *)bufaddr)];
	else if (isanysv(p, v))
		pte = vtopte(p, v);
	else
		panic("real_buf_addr");

	return ((caddr_t)((pte->pg_pfnum << PGSHIFT) | o));
}

/*
 * make the kernel text read-only
 * also protect the POST against modification.
 * we do this by setting the the key bits to '11' which is
 * public read-only. The user can be prevented from reading the
 * kernel text by setting C=1 in the segment register, but
 * we don't do this because the user must be able to access
 * the floating point emulator code.
 * stext is the start of the kernel code to be protected.
 * we assume that we can protect the page that stext is in
 */
#define POST	0x800		/* POST area to protect */

set_kernel_read_only()
{
	extern char stext, etext;
	register int startpg = btop(&stext - KERNBASE);
	register int endpg = btop(&etext + NBPG - 1 - KERNBASE);
	register int i;
	register struct hatipt_entry *h;

	h = &MMU_HATIPT[btop(0)];
	h->key_addrtag |= MMU_KEY_URKR << MMU_KEY_SHIFT;
	h = &MMU_HATIPT[btop(POST)];
	h->key_addrtag |= MMU_KEY_URKR << MMU_KEY_SHIFT;

	for (i=startpg, h = &MMU_HATIPT[startpg]; i<endpg; ++i, ++h)
		h->key_addrtag |= MMU_KEY_URKR << MMU_KEY_SHIFT;
}



/*
 * Copy a null terminated string from the user address space into
 * the kernel address space.
 *
 * copyinstr(fromaddr, toaddr, maxlength, &lencopied)
 */
copyinstr(fromaddr, toaddr, maxlength, lencopied)
register char *fromaddr, *toaddr;
register int maxlength;
register int *lencopied;
{
	register char *to = toaddr;
	register int len;
	int length;
	register int error;
	for (;;) {
		len = NBPG - ((int) fromaddr & (NBPG-1));	/* bytes left on page */
		if (len > maxlength)
			len = maxlength;
		if  (!isitok((caddr_t)fromaddr, len, UREAD_OK)) {
			error = EFAULT;
			break;
		}
		error = copystr(fromaddr, to, len, &length);
		fromaddr += length;
		to += length;
		maxlength -= length;
		if (error == 0 || maxlength == 0)
			break;		/* found the null byte or maxlength */
	}
	if (lencopied)
		*lencopied = to - toaddr;
	return(error);	/* copied it */
}


/*
 * Copy a null terminated string from the kernel
 * address space to the user address space.
 *
 * copyoutstr(fromaddr, toaddr, maxlength, &lencopied)
 */
copyoutstr(fromaddr, toaddr, maxlength, lencopied)
register char *fromaddr, *toaddr;
register int maxlength;
register int *lencopied;
{
	register char *to = toaddr;
	register int len;
	int length;
	register int error;
	for (;;) {
		len = NBPG - ((int) to & (NBPG-1));	/* bytes left on page */
		if (len > maxlength)
			len = maxlength;
		if  (!isitok((caddr_t)to, len, UWRITE_OK)) {
			error = EFAULT;
			break;
		}
		error = copystr(fromaddr, to, len, &length);
		fromaddr += length;
		to += length;
		maxlength -= length;
		if (error == 0 || maxlength == 0)
			break;		/* found the null byte or maxlength */
	}
	if (lencopied)
		*lencopied = to - toaddr;
	return(error);	/* copied it */
}

/*
 * Copy a null terminated string from one point to another in
 * the kernel address space.
 *
 * copystr(fromaddr, toaddr, maxlength, &lencopied)
 * return 0 if found null terminating byte, ENOENT if not.
 */
copystr(fromaddr, toaddr, maxlength, lencopied)
register char *fromaddr, *toaddr;
register int maxlength;
register int *lencopied;
{
	register char *to = toaddr;
	register char *end = toaddr + maxlength;
	register int error = ENOENT;

	while (to < end)
		if ((*to++ = *fromaddr++) == 0) {
			error = 0;
			break;
		}
	if (lencopied)
		*lencopied = to - toaddr;
	return(error);
}

#ifdef SIM_CHG_BIT

/*
 * take a protection exception and determine if we are simulating the 
 * modified bit. If not then the caller will deliver the appropriate
 * signal. If the access in the pte was UW and the access in the HAT/IPT
 * is URKR, then set M=1 and change the protection to UW.
 */

protection(mcs_pcs,info,locr0)
	unsigned info;
	int *locr0;
{
	register struct proc *p;
	register struct pte *pte;
	register unsigned v;
	register struct hatipt_entry *ipte;
	int type;

	DEBUGF(padebug & SHOW_PROTSTATE,
		prstate("protection", mcs_pcs, info, locr0[IAR], locr0[ICSCS],locr0));

	/*
	 * Classify faulted page into a segment and get a pte
	 * for the faulted page.
	 */
	v = clbase(btop(info));
	p = u.u_procp;
	if (isatsv(p, v)) {
		type = CTEXT;
		pte = tptopte(p, vtotp(p, v));
	} else if (isadsv(p, v)) {
		type = CDATA;
		pte = dptopte(p, vtodp(p, v));
	} else if (isassv(p, v)) {
		type = CSTACK;
		pte = sptopte(p, vtosp(p, v));
	} else return (-1);		/* couldn't find a pte for it! */
#ifdef DEBUG
	if (p->p_p0br == 0)
		panic("panic: p0br not set!");
#endif DEBUG
	DEBUGF ((padebug & SHOW_PROT),
		printf("protection: pid=%d, vaddr=0x%x, %s pte 0x%x -> 0x%x ", p->p_pid, info, (type == CTEXT ? "text" :
		    (type == CDATA ? "data" : "stack")),
		    pte, (*(unsigned *)pte)));
	if (pte->pg_v == 0 || pte->pg_m) {
		DEBUGF ((padebug & SHOW_PROT), printf("v=%d m=%d\n",
			pte->pg_v,pte->pg_m));
		return(-1);		/* cannot be our protection violation */
	}
	
	/*
	 * test to make sure that the proper protection is PG_UW, and that
	 * the actual (HAT/IPT) protect is read-only.
	 */
	ipte = &MMU_HATIPT[pte->pg_pfnum];
	if (((* (int *) pte) & PG_PROT) != PG_UW ||
			((unsigned) ipte->key_addrtag >> MMU_KEY_SHIFT) != MMU_KEY_URKR) {
		DEBUGF ((padebug & SHOW_PROT), printf("addrtag=%x\n",ipte->key_addrtag));
		return(-1);	/* its a real protection exception */
	}
	/*
	 * at this point we know that we have taken an informational protection
	 * exception and can set m=1, and change the HAT/IPT protection to the
	 * proper value and restart execution.
	 */
	ipte->key_addrtag ^= (MMU_KEY_URKR^MMU_KEY_UW) << MMU_KEY_SHIFT; /* new prot */
	pte->pg_m = 1;
	invalidate_tlb(v);	/* flush translate buffer */
	cnt.v_pdma++;		/* a lie, but at least its counted */
	DEBUGF ((padebug & SHOW_PROT), printf("done\n"));
	return(0);		/* re-execute the packets */
}

/*
 * we suspect that the modified bit has been changed, so we adjust
 * HAT/IPT key bits if they aren't correct.
 */
chg_mod_bit(pte,v)
struct pte *pte;
unsigned v;
{
	register struct hatipt_entry *ipte;
	int apte = * (int *) pte;

	if ((apte & PG_V) == 0)
		return;		/* not valid so not in HAT/IPT */
	if ((apte & PG_PROT) != PG_UW)
		return;		/* not user writable */
	ipte = &MMU_HATIPT[pte->pg_pfnum];
	if (apte&PG_M) {
		if (((unsigned) ipte->key_addrtag >> MMU_KEY_SHIFT) == MMU_KEY_UW)
			return;		/* is writeable */
	} else if (((unsigned) ipte->key_addrtag >> MMU_KEY_SHIFT) == MMU_KEY_URKR)
		return;			/* is read-only */
	DEBUGF ((padebug & SHOW_CHG_PROT),
		printf("chg_mod_bit: pid=%d, vaddr=0x%x, pte 0x%x -> 0x%x addrtag=%x\n", u.u_procp->p_pid, v, pte, * (int *) pte, ipte->key_addrtag));
	/* we know that we have to invert the protections */
	ipte->key_addrtag ^= (MMU_KEY_URKR^MMU_KEY_UW) << MMU_KEY_SHIFT; /* new prot */
	invalidate_tlb(v);	/* flush translate buffer */
}

/*
 * test to see if a store to addr would cause a protection exception
 * if the page isn't mapped in then it couldn't cause a protection
 * exception.
 */
test_protection(addr)
unsigned addr;
{
	unsigned page = vtop(addr);
	register struct hatipt_entry *ipte;

	if (page&0x80000000)
		return(0);		/* not mapped in */
	ipte = &MMU_HATIPT[btop(page)];
	return (((unsigned) ipte->key_addrtag >> MMU_KEY_SHIFT) == MMU_KEY_URKR);
}
#endif SIM_CHG_BIT
