/*
 * Copyright (c) 1980, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) 1980, 1993\n\
	The Regents of the University of California.  All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "@(#)mt.c	8.1 (Berkeley) 6/6/93";
#endif /* not lint */

/*
 * mt --
 *   magnetic tape manipulation program
 */
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

struct commands {
	char *c_name;
	int c_code;
	int c_ronly;
} com[] = {
	{ "bsf",	MTBSF,	1 },
	{ "bsr",	MTBSR,	1 },
	{ "eof",	MTWEOF,	0 },
	{ "fsf",	MTFSF,	1 },
	{ "fsr",	MTFSR,	1 },
	{ "offline",	MTOFFL,	1 },
	{ "rewind",	MTREW,	1 },
	{ "rewoffl",	MTOFFL,	1 },
	{ "status",	MTNOP,	1 },
	{ "weof",	MTWEOF,	0 },
	{ "eom",	MTEOM,	0 },
	{ "blksize",	MTBLK,	1 },
	{ NULL }
};

void err __P((const char *, ...));
void printreg __P((char *, u_int, char *));
void status __P((struct mtget *));
void usage __P((void));

int mtfd;

int
main(argc, argv)
	int argc;
	char *argv[];
{
	register struct commands *comp;
	struct mtget mt_status;
	struct mtop mt_com;
	int ch, len;
	char *p, *tape;

	if ((tape = getenv("TAPE")) == NULL)
		tape = DEFTAPE;

	while ((ch = getopt(argc, argv, "f:t:")) != EOF)
		switch(ch) {
		case 'f':
		case 't':
			tape = optarg;
			break;
		case '?':
		default:
			usage();
		}
	argc -= optind;
	argv += optind;

	if (argc < 1 || argc > 2)
		usage();

	len = strlen(p = *argv++);
	for (comp = com;; comp++) {
		if (comp->c_name == NULL)
			err("%s: unknown command", p);
		if (strncmp(p, comp->c_name, len) == 0)
			break;
	}
	if ((mtfd = open(tape, comp->c_ronly ? O_RDONLY : O_RDWR)) < 0)
		err("%s: %s", tape, strerror(errno));
	if (comp->c_code != MTNOP) {
		mt_com.mt_op = comp->c_code;
		if (*argv) {
			mt_com.mt_count = strtol(*argv, &p, 10);
			if (mt_com.mt_count < 0 || *p)
				err("%s: illegal count", *argv);
		}
		else
			mt_com.mt_count = 1;
		if (ioctl(mtfd, MTIOCTOP, &mt_com) < 0)
			err("%s: %s: %s", tape, comp->c_name, strerror(errno));
	} else {
		if (ioctl(mtfd, MTIOCGET, &mt_status) < 0)
			err("%s", strerror(errno));
		status(&mt_status);
	}
	exit (0);
	/* NOTREACHED */
}

#ifdef vax
#include <vax/mba/mtreg.h>
#include <vax/mba/htreg.h>

#include <vax/uba/utreg.h>
#include <vax/uba/tmreg.h>
#undef b_repcnt		/* argh */
#include <vax/uba/tsreg.h>
#endif

#ifdef sun
#include <sundev/tmreg.h>
#include <sundev/arreg.h>
#endif

#ifdef tahoe
#include <tahoe/vba/cyreg.h>
#endif

#ifdef ibmrt
#include <machine/streg.h>
#include <machine/scsi.h>
#include <machine/dkio.h>
#endif

struct tape_desc {
	short	t_type;		/* type of magtape device */
	char	*t_name;	/* printing name */
	char	*t_dsbits;	/* "drive status" register */
	char	*t_erbits;	/* "error" register */
} tapes[] = {
#ifdef vax
	{ MT_ISTS,	"ts11",		0,		TSXS0_BITS },
	{ MT_ISHT,	"tm03",		HTDS_BITS,	HTER_BITS },
	{ MT_ISTM,	"tm11",		0,		TMER_BITS },
	{ MT_ISMT,	"tu78",		MTDS_BITS,	0 },
	{ MT_ISUT,	"tu45",		UTDS_BITS,	UTER_BITS },
#endif
#ifdef sun
	{ MT_ISCPC,	"TapeMaster",	TMS_BITS,	0 },
	{ MT_ISAR,	"Archive",	ARCH_CTRL_BITS,	ARCH_BITS },
#endif
#ifdef tahoe
	{ MT_ISCY,	"cipher",	CYS_BITS,	CYCW_BITS },
#endif
#ifdef ibmrt
	{ MT_ISST,	"st11",		ST_CTLR_BITS,	ST_ERR_BITS },
	{ MT_ISCS,	"scsi",		0,		0 },
#endif
	{ 0 }
};

/*
 * Interpret the status buffer returned
 */
void
status(bp)
	register struct mtget *bp;
{
	register struct tape_desc *mt;

	for (mt = tapes;; mt++) {
		if (mt->t_type == 0) {
			(void)printf("%d: unknown tape drive type\n",
			    bp->mt_type);
			return;
		}
		if (mt->t_type == bp->mt_type)
			break;
	}
#ifdef	ibmrt
	if (mt->t_type == MT_ISCS) {
		char msbuf[127];
		struct scsi_ms ms;
		struct scsi_esense esense;

		printf("%s tape drive\n", mt->t_name);
		if (ioctl(mtfd, DKIOCSMODESENSE, &msbuf))
			perror("mt");
		xxfillmspages(&msbuf, &ms);
#ifdef	SCSI_DIAG
		print_modesense_pages(&ms);
#else
		if(ms.sms_hd.sm_length >0) 
			write_protect_status(&ms.sms_hd);
#endif
		if (ioctl(mtfd, DKIOCGSCSISENSE, &esense))
			perror("mt");
#ifdef	SCSI_DIAG
		xxhdump(&esense,sizeof esense);
		esense_dump(&esense);
#else
		interpret_sense_key(&esense);
#endif
		printf("   block size=%d\n", bp->mt_blkno);
	} else 
#endif
	{
	(void)printf("%s tape drive, residual=%d\n", mt->t_name, bp->mt_resid);
	printreg("ds", bp->mt_dsreg, mt->t_dsbits);
	printreg("\ner", bp->mt_erreg, mt->t_erbits);
	(void)putchar('\n');
	}
}

/*
 * Print a register a la the %b format of the kernel's printf.
 */
void
printreg(s, v, bits)
	char *s;
	register u_int v;
	register char *bits;
{
	register int i, any = 0;
	register char c;

	if (bits && *bits == 8)
		printf("%s=%o", s, v);
	else
		printf("%s=%x", s, v);
	bits++;
	if (v && bits) {
		putchar('<');
		while (i = *bits++) {
			if (v & (1 << (i-1))) {
				if (any)
					putchar(',');
				any = 1;
				for (; (c = *bits) > 32; bits++)
					putchar(c);
			} else
				for (; *bits > 32; bits++)
					;
		}
		putchar('>');
	}
}

void
usage()
{
	(void)fprintf(stderr, "usage: mt [-f device] command [ count ]\n");
	exit(1);
}

#ifdef	ibmrt

#ifndef	MIN
#define MIN(x,y) ((x) < (y) ? (x) : (y))
#endif

xxfillmspages(u_char *from, struct scsi_ms *to)
{
        u_char *fromstart = from;
        struct scsi_mspghd *pghdp;
        int bavail = 0;

        bcopy(from,(char *) to, 4); /* Copy header, believed to be constant */
        from += 4;

        /* Copy one Block descriptor */
        bcopy(from, (char *) to->sms_hd.sm_bdarr,
                MIN(to->sms_hd.sm_bdl,sizeof(struct scsi_bd)));

        from += to->sms_hd.sm_bdl;

        while(from < (fromstart + to->sms_hd.sm_length))
        {
                pghdp = (struct scsi_mspghd *) from;
                bavail = pghdp->pghd_plen + sizeof (struct scsi_mspghd);

                switch(pghdp->pghd_pcode)
                {
                case 1:         /* errpage */
                        bcopy(from, &to->sms_erp,
                                MIN(sizeof(struct scsi_mserppage), bavail));
                        from += bavail;
                        continue;
                case 2:         /* connpage */
                        bcopy(from, &to->sms_cop,
                                MIN(sizeof(struct scsi_msconnpage), bavail));
                        from += bavail;
                        continue;
                case 3:         /* dadfpage */
                        bcopy(from, &to->sms_daf,
                                MIN(sizeof(struct scsi_msdadfpage), bavail));
                        from += bavail;
                        continue;
                case 4:         /* rigidpage */
                        bcopy(from, &to->sms_rdg,
                                MIN(sizeof(struct scsi_msrigidpage), bavail));
                        from += bavail;
                        continue;
                case 8:         /* Ra cache page */
                        bcopy(from, &to->sms_rac,
                                MIN(sizeof(struct scsi_msracache), bavail));
                        from += bavail;
                        continue;
                case 56:        /* CDC Wren cache page */
                        bcopy(from, &to->sms_csh,
                                MIN(sizeof(struct scsi_mscachepage), bavail));
                        from += bavail;
                        continue;
                case 34:        /* reconnection timing page */
                        bcopy(from, &to->sms_rtc,
                                MIN(sizeof(struct scsi_rtcpage), bavail));
                        from += bavail;
                        continue;
                case 33:        /* additional error recovery page */
                        bcopy(from, &to->sms_aer,
                                MIN(sizeof(struct scsi_msaerpage), bavail));
                        from += bavail;
                        continue;
                case 32:        /* Adaptec dis/reconecct page  */
                        bcopy(from, &to->sms_atc,
                                MIN(sizeof(struct scsi_msatconnpage), bavail));
                        from += bavail;
                        continue;
                case 7:         /* Verify error recovery page */
                        bcopy(from, &to->sms_ver,
                                MIN(sizeof(struct scsi_msverpage), bavail));
                        from += bavail;
                        continue;
                default:
			if (pghdp->pghd_plen)
			    printf("Unknown mode sense page code 0x%x, length=%d bytes\n",
                                pghdp->pghd_pcode, pghdp->pghd_plen);
                        from += bavail;
                        continue;
                }
        }

}

print_modesense_pages(struct scsi_ms *p)
{
	if(p->sms_hd.sm_length >0) print_modesense(&p->sms_hd);
	if(p->sms_erp.erp_plen >0) print_mserppage(&p->sms_erp);
	if(p->sms_cop.cop_plen >0) print_msconnpage(&p->sms_cop);
	if(p->sms_daf.daf_plen >0) print_msdadfpage(&p->sms_daf);
	if(p->sms_rdg.rdg_plen >0) print_msrigidpage(&p->sms_rdg);
	if(p->sms_csh.csh_plen >0) print_mscachepage(&p->sms_csh);
	if(p->sms_rac.rac_plen >0) print_msracache(&p->sms_rac);
	if(p->sms_rtc.rtc_plen >0) print_msrtcpage(&p->sms_rtc);
	if(p->sms_aer.aer_plen >0) print_msaerpage(&p->sms_aer);
	if(p->sms_atc.atc_plen >0) print_msatcpage(&p->sms_atc);
	if(p->sms_ver.ver_plen >0) print_msverpage(&p->sms_ver);
}

print_modesense(struct scsi_modesense *p) 
{
printf("Mode Sense, Sense data length 		%d\n",p->sm_length);  
printf("Mode Sense, Medium type 		%d\n",p->sm_mtype);   
printf("Mode Sense, Write-protect 		%x\n",p->sm_wp);    
printf("Mode Sense, Maxtor Buffered mode 	%x\n",p->sm_bufmode);
printf("Mode Sense, Maxtor Speed code 		%x\n",p->sm_speed); 
printf("Mode Sense, Block descriptor length 	%d\n",p->sm_bdl);     
printf("Mode Sense Block descriptor, density code 	%d\n",
	p->sm_bdarr[0].sbd_density);     
printf("Mode Sense Block descriptor, Number of blocks 	%d\n",
	p->sm_bdarr[0].sbd_nblocks);     
printf("Mode Sense Block descriptor, Block length 	%d\n\n",
	p->sm_bdarr[0].sbd_blklen);     

}

print_mserppage(struct scsi_mserppage *p) 
{ 
	char pname[] = "Error recovery page,";

	if( p->erp_plen > 0 ) printf("%s parameter savable		%x\n", pname, p->erp_ps);
	if( p->erp_plen > 0 ) printf("%s Page code			%x\n", pname, p->erp_pcode);
	if( p->erp_plen > 0 ) printf("%s Page length		%d\n", pname, p->erp_plen);
	if( p->erp_plen > 0 ) printf("%s Automatic write reallocation enable	%x\n", pname, p->erp_awre);
	if( p->erp_plen > 0 ) printf("%s Automatic read reallocation enable		%x\n", pname, p->erp_arre);
	if( p->erp_plen > 0 ) printf("%s Transfer block with error	%x\n", pname, p->erp_tb);
	if( p->erp_plen > 0 ) printf("%s Read continous		%x\n", pname, p->erp_rc);
	if( p->erp_plen > 0 ) printf("%s Enable early correction	%x\n", pname, p->erp_eec);
	if( p->erp_plen > 0 ) printf("%s Post error			%x\n", pname, p->erp_per);
	if( p->erp_plen > 0 ) printf("%s Disable transfer on error	%x\n", pname, p->erp_dte);
	if( p->erp_plen > 0 ) printf("%s Disable correction		%x\n", pname, p->erp_dcr);
	if( p->erp_plen > 1 ) printf("%s Read retry count		%d\n", pname, p->erp_rrtcnt);
	if( p->erp_plen > 2 ) printf("%s Correction span		%d\n", pname, p->erp_cspan);
	if( p->erp_plen > 3 ) printf("%s Head offset count		%d\n", pname, p->erp_hoffset);
	if( p->erp_plen > 4 ) printf("%s Data strobe offset count	%d\n", pname, p->erp_dsoffset);
	if( p->erp_plen > 5 ) printf("%s Recovery time limit	%d\n", pname, p->erp_rtl);
	if( p->erp_plen > 6 ) printf("%s Write retry count	%d\n", pname, p->erp_wrtcnt);
	if( p->erp_plen > 0 ) printf("\n");

};
print_msconnpage(struct scsi_msconnpage *p) 
{
	if( p->cop_plen > 0) printf("Connect Page , parameter savable	%x\n", p->cop_ps);
	if( p->cop_plen > 0) printf("Connect Page , Page code		%x\n", p->cop_pcode);
	if( p->cop_plen > 0) printf("Connect Page , Page length		%d\n", p->cop_plen);
	if( p->cop_plen > 0) printf("Connect Page , Buffer full ratio	%d\n", p->cop_bfr);
	if( p->cop_plen > 1) printf("Connect Page , Buffer empty ratio	%d\n", p->cop_ber);
	if( p->cop_plen > 3) printf("Connect Page , Bus inactivity limit	%d\n", p->cop_bil);
	if( p->cop_plen > 5) printf("Connect Page , Disconnect time limit	%d\n", p->cop_dtl);
	if( p->cop_plen > 7) printf("Connect Page , Connect time limit	%d\n\n", p->cop_ctl);

};
print_msdadfpage(struct scsi_msdadfpage *p) 
{ 
   	if( p->daf_plen > 0) printf("Format Page, Parameter Savable		%x \n",p->daf_ps); 
   	if( p->daf_plen > 0) printf("Format Page, Page code			%x \n",p->daf_pcode); 
   	if( p->daf_plen > 0) printf("Format Page, Page length			%d \n",p->daf_plen);    
   	if( p->daf_plen > 1) printf("Format Page, Tracks per zone			%d \n",p->daf_tpz);     
   	if( p->daf_plen > 3) printf("Format Page, Alternate sectors per zone	%d \n",p->daf_altspz);  
   	if( p->daf_plen > 5) printf("Format Page, Alternate tracks per zone	%d \n",p->daf_atttpz);  
   	if( p->daf_plen > 7) printf("Format Page, Alternate tracks per volume	%d \n",p->daf_alttpv);  
   	if( p->daf_plen > 9) printf("Format Page, Sectors per track		%d \n",p->daf_spt);     
   	if( p->daf_plen > 11) printf("Format Page, No of bytes per sector		%d \n",p->daf_nbps);    
   	if( p->daf_plen > 13) printf("Format Page, Interleave factor		%d \n",p->daf_intrlve); 
   	if( p->daf_plen > 15) printf("Format Page, Track skew			%d \n",p->daf_tskew);   
   	if( p->daf_plen > 17) printf("Format Page, Cylinder skew			%d \n",p->daf_cskew);   
   	if( p->daf_plen > 18) printf("Format Page, Soft sector format		%x \n",p->daf_ssec);  
   	if( p->daf_plen > 18) printf("Format Page, Hard sector format		%x \n",p->daf_hsec);  
   	if( p->daf_plen > 18) printf("Format Page, Removable media			%x \n",p->daf_rmb);   
   	if( p->daf_plen > 18) printf("Format Page, Format by surface		%x \n",p->daf_surf);  
   	if( p->daf_plen > 18) printf("Format Page, Inhibit Save			%x \n\n",p->daf_ins);   

};
print_msrigidpage(struct scsi_msrigidpage *p) 
{ 
 printf("Rigid page,  parameter savable            %x \n",p->rdg_ps);      
 printf("Rigid page,  Page code                    %x \n",p->rdg_pcode);   
 printf("Rigid page,  Page length                  %d \n",p->rdg_plen);      
 printf("Rigid page,  Maximum no of cylinders      %d \n",
	(p->rdg_maxcyl[0]<<16) |
	(p->rdg_maxcyl[1]<<8 ) |
	(p->rdg_maxcyl[2] )
	); 
 printf("Rigid page,  Maximum no of heads          %d \n\n",p->rdg_hmaxheads); 
};
print_mscachepage(struct scsi_mscachepage *p) 
{
	printf("Wren Cache Parameter Savable:	%s\n",
	       p->csh_ps ? "Yes" : "No" );
	printf("Wren Cache page code:		%x\n", p->csh_pcode);
	printf("Wren Cache page length:		%d\n",p->csh_plen);
	printf("Wren Write Index enabled :	%s\n",
	       p->csh_wie ? "Yes" : "No" );
	printf("Wren Caching enabled :		%s\n",
	       p->csh_ce ? "Yes" : "No" );
	printf("Wren Cache table size:		%d\n",p->csh_cts);
	printf("Wren Cache Prefetch Threshold:	%x\n\n",
                        p->csh_pt);
}
print_msracache(struct scsi_msracache *p) 
{
	printf("Cache Parameter Savable:	%s\n",
	       p->rac_ps ? "Yes" : "No" );
	printf("Cache page code:		%x\n", p->rac_pcode);
	printf("Cache page length:		%d\n",p->rac_plen);
	printf("Cache Multiple:	       		%s\n", p->rac_ms ? "Yes":"No" );
	printf("Read Cache disabled :		%s\n", p->rac_rcd ? "Yes":"No");
	printf("Cache Suppression :		%x\n", p->rac_suppr );
	printf("Cache Min prefetch :		%d\n", p->rac_min_pre );
	printf("Cache Max prefetch :		%d\n", p->rac_max_pre );
	printf("Cache Max prefetch restriction :%d\n", p->rac_max_pre_restr );
}

print_msrtcpage(struct scsi_rtcpage *p) 
{
	printf("RC timing Parameter Savable:	%s\n",
	       p->rtc_ps ? "Yes" : "No" );
	printf("RC timing page code:		%x\n", p->rtc_pcode);
	printf("RC timing page length:		%d\n",p->rtc_plen);
	printf("RC timing reserved byte 1:	%d\n",p->rtc_resv1);
	printf("RC timing reserved byte 2:	%d\n",p->rtc_resv2);
}

print_msaerpage(struct scsi_msaerpage *p) 
{
	printf("Addl err rec Parameter Savable:	%s\n",
	       p->aer_ps ? "Yes" : "No" );
	printf("Addl err rec page code:		%x\n", p->aer_pcode);
	printf("Addl err rec page length:		%d\n",p->aer_plen);
	printf("Addl err rec cmd exec delay:	%sabled\n",p->aer_dced?"Dis":"En");
	printf("Addl err rec Post SCSI err:	%sabled\n",p->aer_pser?"En":"Dis");
	printf("Addl err rec Seek retry cnt:	%d\n",p->aer_srtc);
}

print_msverpage(struct scsi_msverpage *p) 
{
	printf("Vfy err rec Parameter Savable:	%s\n",
	       p->ver_ps ? "Yes" : "No" );
	printf("Vfy err rec page code:		%x\n", p->ver_pcode);
	printf("Vfy err rec page length:		%d\n",p->ver_plen);
	printf("Vfy err rec Early recovery:	%sabled\n",p->ver_eer?"En":"Dis");
	printf("Vfy err rec Post SCSI err:	%sabled\n",p->ver_per?"En":"Dis");
	printf("Vfy err rec Xfer on error:	%sabled\n",p->ver_dte?"Dis":"En");
	printf("Vfy err rec Error correction:	%sabled\n",p->ver_dcr?"Dis":"En");
	printf("Vfy err rec Retry count:	%d\n",p->ver_vrtc);
	printf("Vfy err rec Correctable bit len:%d\n",p->ver_cbl);
}

print_msatcpage(struct scsi_msatconnpage *p) 
{
	printf("Adaptec conn Parameter Savable:	%s\n",
	       p->atc_ps ? "Yes" : "No" );
	printf("Adaptec conn page code:		%x\n", p->atc_pcode);
	printf("Adaptec conn page length:		%d\n",p->atc_plen);
	printf("Adaptec conn Host rec overhead:	%d uS\n",p->atc_hroh*100);
	printf("Adaptec conn Buffer pre-fill:	%d bytes\n",p->atc_bpfill*256);
	printf("Adaptec conn Modify data ptr:	%sabled\n",p->atc_mdp?"En":"Dis");
}

char *
xxcstr(c)
u_char c;
{
    static char     outstr[3];
    char           *convstr = "0123456789abcdef";

    outstr[0] = convstr[(c >> 4) & 0xf];
    outstr[1] = convstr[c & 0xf];
    outstr[2] = '\0';

    return outstr;
}


xxhdump(cp, n)
u_char         *cp;
int             n;
{
    int             i = 0;

    while (n--)
    {
        printf("%s", xxcstr(*cp++));
        ++i;
        if ((i % 32) == 0)
        {
            printf("\n");
        }
        else
        {
            if ((i % 4) == 0)
            {
                printf(" ");
            }
        }
    }
    if (i % 32)
    {
        printf("\n");
    }
}

esense_dump(esense)
struct scsi_esense *esense;
{
	printf("Data valid =		%02x\n", esense->es_valid);
	printf("Error class=		%02x\n", esense->es_eclass);
	printf("Error code=		%02x\n", esense->es_ecode);
	printf("Segment number=		%02x\n", esense->es_segno);
	printf("File mark=		%02x\n", esense->es_fmark);
	printf("End of medium=		%02x\n", esense->es_eom);
	printf("Illegal length ind=	%02x\n", esense->es_ili);
	printf("Sense key=		%02x\n", esense->es_key);
        printf("Info bytes=		%02x %02x %02x %02x \n", 
		esense->es_info[0], esense->es_info[1], esense->es_info[2], 
		esense->es_info[3]);
        printf("Additional sense len=	%02x\n", esense->es_addlen);
        printf("Sense code=		%02x\n", esense->es_scode);
        printf("Subsense code=		%02x\n", esense->es_subscode);
        printf("Field replaceable unit=	%02x\n", esense->es_fru);
        printf("Sense key inherent info valid=%02x\n", esense->es_fpv);
        printf("CDB or data=		%02x\n", esense->es_cd);
        printf("Reserved1=		%02x\n", esense->es_resv1);
        printf("Reserved2=		%02x\n", esense->es_resv2);
        printf("Bit pointer valid=	%02x\n", esense->es_bpv);
        printf("Bit pointer=		%02x\n", esense->es_bp);
        printf("Field pointer=		%02x\n", esense->es_fp);
        printf("Command inherent info=	%02x %02x %02x %02x \n", 
		esense->es_cii[0], esense->es_cii[1], esense->es_cii[2], 
		esense->es_cii[3]);
	printf("SCSI target=		%02x\n", esense->es_target);
	printf("SCSI lun=		%02x\n", esense->es_lun);
	printf("CDB op code=		%02x\n", esense->es_op);

	interpret_sense_key(esense);
}

interpret_sense_key(esense)
struct scsi_esense *esense;
{
	int res = (esense->es_info[0] << 24) +	/* residual count */
		  (esense->es_info[1] << 16) +
		  (esense->es_info[2] << 8)  +
		   esense->es_info[3];

	printf("   sense key(0x%x)= ", esense->es_key);
	switch (esense->es_key) {
	    case SCES_KEY_NO_SENSE:
		printf("no sense");
		/* Check FMK, ILI and EOM bits here */
		if (esense->es_fmark)
		    printf(", File mark detected, res=%d", res);
		if (esense->es_ili)
		    printf(", invalid length indicated, res=%d", res);
		if (esense->es_eom)
		    printf(", end of medium detected, res=%d", res);
		break;
	    case SCES_KEY_RECOVERED_ERROR:
		printf("recovered error");
		break;
	    case SCES_KEY_NOT_READY:
		printf("not ready");
		break;
	    case SCES_KEY_MEDIUM_ERROR:
		printf("medium error");
		break;
	    case SCES_KEY_HARDWARE_ERROR:
		printf("hardware error");
		break;
	    case SCES_KEY_ILLEGAL_REQUEST:
		printf("illegal request, res=%d", res);
		break;
	    case SCES_KEY_UNIT_ATTENTION:
		printf("unit attention");
		break;
	    case SCES_KEY_DATA_PROTECT:
		printf("data protect");
		break;
	    case SCES_KEY_BLANK_CHECK:
		printf("blank check");
		break;
	    case SCES_KEY_COPY_ABORTED:
		printf("copy aborted, res=%d", res);
		break;
	    case SCES_KEY_ABORTED_COMMAND:
		printf("aborted command, res=%d", res);
		break;
	    case SCES_KEY_VOLUME_OVERFLOW:
		printf("volume overflow");
		break;
	    default:
		printf("unknown key, res=%d", res);
		break;
	}
	printf("\n");
}

write_protect_status(struct scsi_modesense *p) 
{
	if (p->sm_wp)
	    printf("   write protected\n");
}
#endif /* ibmrt */

#if __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

void
#if __STDC__
err(const char *fmt, ...)
#else
err(fmt, va_alist)
	char *fmt;
        va_dcl
#endif
{
	va_list ap;
#if __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	(void)fprintf(stderr, "mt: ");
	(void)vfprintf(stderr, fmt, ap);
	va_end(ap);
	(void)fprintf(stderr, "\n");
	exit(1);
	/* NOTREACHED */
}
