/*
 * Copyright (c) 1983 Eric P. Allman
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted provided
 * that: (1) source distributions retain this entire copyright notice and
 * comment, and (2) distributions including binaries display the following
 * acknowledgement:  ``This product includes software developed by the
 * University of California, Berkeley and its contributors'' in the
 * documentation or other materials provided with the distribution and in
 * all advertising materials mentioning features or use of this software.
 * Neither the name of the University nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

# include "sendmail.h"

#ifndef lint
# ifdef SMTP
static char sccsid[] = "@(#)usersmtp.c	5.15 (Berkeley) 6/1/90 (with SMTP)";
static char  rcsid[] = "@(#)$Id: usersmtp.c,v 5.15.0.11 1991/06/24 20:31:15 paul Exp $ (with SMTP)";
# else /* ! SMTP */
static char sccsid[] = "@(#)usersmtp.c	5.15 (Berkeley) 6/1/90 (without SMTP)";
static char  rcsid[] = "@(#)$Id: usersmtp.c,v 5.15.0.11 1991/06/24 20:31:15 paul Exp $ (without SMTP)";
# endif /* SMTP */
#endif /* ! lint */

#include <sysexits.h>
#include <errno.h>

#ifdef SMTP

# ifdef __STDC__
static int reply(MAILER *);
static void smtpmessage(const char *, MAILER *, ...);
# else /* !__STDC__ */
static int reply();
static void smtpmessage();
# endif /* __STDC__  */

/*
**  USERSMTP -- run SMTP protocol from the user end.
**
**	This protocol is described in RFC821.
*/

# define REPLYTYPE(r)	((r) / 100)		/* first digit of reply code */
# define REPLYCLASS(r)	(((r) / 10) % 10)	/* second digit of reply code */
# define SMTPGOODREPLY	250			/* positive SMTP response */
# define SMTPCLOSING	421			/* "Service Shutting Down" */

static char	SmtpMsgBuffer[MAXLINE];	/* buffer for commands */
static char	SmtpReplyBuffer[MAXLINE]; /* buffer for replies */
char		SmtpError[MAXLINE] = ""; /* save failure error messages */
static bool	SmtpNeedIntro;		/* set before first error */
static FILE	*SmtpOut;		/* output file */
static FILE	*SmtpIn;		/* input file */
static int	SmtpPid;		/* pid of mailer */

/* following represents the state of the SMTP connection */
static int	SmtpState;		/* connection state, see below */

# define SMTP_CLOSED	0		/* connection is closed */
# define SMTP_OPEN	1		/* connection is open for business */
# define SMTP_SSD	2		/* service shutting down */
/*
**  SMTPINIT -- initialize SMTP.
**
**	Opens the connection and sends the initial protocol.
**
**	Parameters:
**		m -- mailer to create connection to.
**		pvp -- pointer to parameter vector to pass to
**			the mailer.
**		e -- the envelope to deliver (#ifdef MAIL11V3)
**
**	Returns:
**		appropriate exit status -- EX_OK on success.
**		If not EX_OK, it should close the connection.
**
**	Side Effects:
**		creates connection and sends initial protocol.
*/

int
# ifdef MAIL11V3
smtpinit(m, pvp, e)
	register ENVELOPE *e;
# else /* ! MAIL11V3 */
smtpinit(m, pvp)
# endif /* MAIL11V3 */
	MAILER *m;
	char **pvp;
{
	register int r;
	TIME_TYPE SavedReadTimeout;
	char buf[MAXNAME];

	/*
	**  Open the connection to the mailer.
	*/

	if (SmtpState == SMTP_OPEN)
		syserr("smtpinit: already open");

	SmtpIn = SmtpOut = NULL;
	SmtpState = SMTP_CLOSED;
	SmtpError[0] = '\0';
	SmtpNeedIntro = TRUE;
	SmtpPhase = "user open";
	setproctitle("%s %s: %s", CurEnv->e_id, pvp[1], SmtpPhase);
	SmtpPid = openmailer(m, pvp, (ADDRESS *) NULL, TRUE, &SmtpOut, &SmtpIn);
	if (SmtpPid < 0)
	{
		if (tTd(18, 1))
			printf("smtpinit: cannot open %s: stat %d errno %d\n",
			   pvp[0], ExitStat, errno);
		if (CurEnv->e_xfp != NULL)
		{
			register char *p;

			if (errno == 0)
			{
				p = statstring(ExitStat);
				fprintf(CurEnv->e_xfp,
					"%.3s %s (%s)... %s\n",
					p, pvp[1], m->m_name, p);
			}
			else
			{
				r = errno;
				fprintf(CurEnv->e_xfp,
					"421 %s (%s)... Deferred: %s\n",
					pvp[1], m->m_name, errstring(errno));
				errno = r;
			}
		}
		return (ExitStat);
	}
	SmtpState = SMTP_OPEN;

	/*
	**  Get the greeting message.
	**	This should appear spontaneously.  Give it five minutes to
	**	happen.
	*/

	SmtpPhase = "greeting wait";
	setproctitle("%s %s: %s", CurEnv->e_id, CurHostName, SmtpPhase);
	SavedReadTimeout = ReadTimeout;
	if (ReadTimeout > 300)
		ReadTimeout = 300;
	r = reply(m);
	ReadTimeout = SavedReadTimeout;
	if (r < 0 || REPLYTYPE(r) != 2)
		goto tempfail;

	/*
	**  Send the HELO command.
	**	My mother taught me to always introduce myself.
	*/

	smtpmessage("HELO %s", m, MyHostName);
	SmtpPhase = "HELO wait";
	setproctitle("%s %s: %s", CurEnv->e_id, CurHostName, SmtpPhase);
	r = reply(m);
	if (r < 0)
		goto tempfail;
	else if (REPLYTYPE(r) == 5)
		goto unavailable;
	else if (REPLYTYPE(r) != 2)
		goto tempfail;

	/*
	**  If this is expected to be another sendmail, send some internal
	**  commands.
	*/

	if (bitnset(M_INTERNAL, m->m_flags))
	{
		/* tell it to be verbose */
		smtpmessage("VERB", m);
		r = reply(m);
		if (r < 0)
			goto tempfail;

		/* tell it we will be sending one transaction only */
		smtpmessage("ONEX", m);
		r = reply(m);
		if (r < 0)
			goto tempfail;
	}

# ifdef MAIL11V3
	/*
	**  If this mailer can do multiple status returns after DATA command,
	**  ask if it will do so.
	*/
	if (bitnset(M_MANYSTATUS, m->m_flags))
	{
		smtpmessage("MULT", m);
		r = reply(m);
		if (r < 0)
			goto tempfail;
		else if (r == 250)
			SmtpManyStatus = TRUE;
	}
	else
		SmtpManyStatus = FALSE;

	/*
	**  If this mailer wants to see the headers and body early, ask if
	**  now is OK.
	*/

	if (bitnset(M_PREHEAD, m->m_flags))
	{
		smtpmessage("HEAD", m);
		r = reply(m);
		if (r < 0)
			goto tempfail;
		if (REPLYTYPE(r) == 2 || REPLYTYPE(r) == 3)
		{
			/* Send the header and message... */
			(*e->e_puthdr)(SmtpOut, m, CurEnv);
			putline("", SmtpOut, m);
			(*e->e_putbody)(SmtpOut, m, CurEnv);

			/* followed by the proper termination. */
			fprintf(SmtpOut, ".%s", m->m_eol);
			r = reply(m);
			if (r < 0)
				goto tempfail;
		}
	}
# endif /* MAIL11V3 */

	/*
	**  Send the MAIL command.
	**	Designates the sender.
	*/

	expand("\001g", buf, &buf[sizeof buf - 1], CurEnv);
	if (CurEnv->e_from.q_mailer == LocalMailer ||
	    !bitnset(M_FROMPATH, m->m_flags))
	{
		smtpmessage("MAIL From:<%s>", m, buf);
	}
	else
	{
		smtpmessage("MAIL From:<@%s%c%s>", m, MyHostName,
			buf[0] == '@' ? ',' : ':', buf);
	}
	SmtpPhase = "MAIL wait";
	setproctitle("%s %s: %s", CurEnv->e_id, CurHostName, SmtpPhase);
	r = reply(m);
	if (r < 0 || REPLYTYPE(r) == 4)
		goto tempfail;
	else if (r == 250)
		return (EX_OK);
	else if (r == 552)
		goto unavailable;
#ifdef MAIL11V3
	else if (r == 559)
	{
		smtpquit(m);
		return (EX_NOHOST);
	}
#endif /* MAIL11V3 */

	/* protocol error -- close up */
	smtpquit(m);
	return (EX_PROTOCOL);

	/* signal a temporary failure */
  tempfail:
	smtpquit(m);
	return (EX_TEMPFAIL);

	/* signal service unavailable */
  unavailable:
	smtpquit(m);
	return (EX_UNAVAILABLE);
}
/*
**  SMTPRCPT -- designate recipient.
**
**	Parameters:
**		to -- address of recipient.
**		m -- the mailer we are sending to.
**
**	Returns:
**		exit status corresponding to recipient status.
**
**	Side Effects:
**		Sends the mail via SMTP.
*/

int
smtprcpt(to, m)
	ADDRESS *to;
	register MAILER *m;
{
	register int r;

	smtpmessage("RCPT To:<%s>", m, to->q_user);

	SmtpPhase = "RCPT wait";
	setproctitle("%s %s: %s", CurEnv->e_id, CurHostName, SmtpPhase);
	r = reply(m);
	if (r < 0 || REPLYTYPE(r) == 4)
		return (EX_TEMPFAIL);
	else if (REPLYTYPE(r) == 2)
		return (EX_OK);
	else if (r == 550 || r == 551 || r == 553)
		return (EX_NOUSER);
	else if (r == 552 || r == 554)
		return (EX_UNAVAILABLE);
	return (EX_PROTOCOL);
}
/*
**  SMTPDATA -- send the data and clean up the transaction.
**
**	Parameters:
**		m -- mailer being sent to.
**		e -- the envelope for this message.
**
**	Returns:
**		exit status corresponding to DATA command.
**
**	Side Effects:
**		none.
*/

int
smtpdata(m, e)
	MAILER *m;
	register ENVELOPE *e;
{
	register int r;

	/*
	**  Send the data.
	**	First send the command and check that it is ok.
	**	Then send the data.
	**	Follow it up with a dot to terminate.
	**	Finally get the results of the transaction.
	*/

	/* send the command and check ok to proceed */
	smtpmessage("DATA", m);
	SmtpPhase = "DATA wait";
	setproctitle("%s %s: %s", CurEnv->e_id, CurHostName, SmtpPhase);
	r = reply(m);
	if (r < 0 || REPLYTYPE(r) == 4)
		return (EX_TEMPFAIL);
	else if (r == 554)
		return (EX_UNAVAILABLE);
	else if (r != 354 && r != 250)
		return (EX_PROTOCOL);

	/* now output the actual message */
	(*e->e_puthdr)(SmtpOut, m, CurEnv);
	putline("", SmtpOut, m);
	(*e->e_putbody)(SmtpOut, m, CurEnv);

	/* terminate the message */
	fprintf(SmtpOut, ".%s", m->m_eol);
	if (Verbose && !HoldErrs)
		nmessage(Arpa_Info, ">>> .");
#ifdef MAIL11V3
	/*
	** If we are running with mail11v3 support, status is collected in
	** in deliver().
	*/
	return (EX_OK);
#else /* ! MAIL11V3 */

	/* check for the results of the transaction */
	SmtpPhase = "result wait";
	setproctitle("%s %s: %s", CurEnv->e_id, CurHostName, SmtpPhase);
	r = reply(m);
	if (r < 0 || REPLYTYPE(r) == 4)
		return (EX_TEMPFAIL);
	else if (r == 250)
		return (EX_OK);
	else if (r == 552 || r == 554)
		return (EX_UNAVAILABLE);
	return (EX_PROTOCOL);
#endif /* MAIL11V3 */
}
/*
**  SMTPQUIT -- close the SMTP connection.
**
**	Parameters:
**		m -- a pointer to the mailer.
**
**	Returns:
**		none.
**
**	Side Effects:
**		sends the final protocol and closes the connection.
*/

void
smtpquit(m)
	register MAILER *m;
{
	int i = errno;

	/* if the connection is already closed, don't bother */
	if (SmtpIn == NULL)
		return;

	/* send the quit message if not a forced quit */
	if (SmtpState == SMTP_OPEN || SmtpState == SMTP_SSD)
	{
		smtpmessage("QUIT", m);
		(void) reply(m);
		if (SmtpState == SMTP_CLOSED)
			return;
	}

	/* now actually close the connection, but without trashing errno */
	(void) fclose(SmtpIn);
	(void) fclose(SmtpOut);
	errno = i;
	SmtpIn = SmtpOut = NULL;
	SmtpState = SMTP_CLOSED;

	/* and pick up the zombie */
	i = endmailer(SmtpPid, m->m_argv[0]);
	if (i != EX_OK)
		syserr("smtpquit %s: stat %d", m->m_argv[0], i);
}
/*
**  REPLY -- read arpanet reply
**
**	Parameters:
**		m -- the mailer we are reading the reply from.
**
**	Returns:
**		reply code it reads.
**
**	Side Effects:
**		flushes the mail file.
*/

static int
reply(m)
	MAILER *m;
{
	if (SmtpOut != NULL)
		(void) fflush(SmtpOut);

	if (tTd(18, 1))
		printf("reply\n");

	if (bitnset(M_BSMTP, m->m_flags))
		return (SMTPGOODREPLY);

	/*
	**  Read the input line, being careful not to hang.
	*/

	for (;;)
	{
		register int r;
		register char *p;

		/* actually do the read */
		if (CurEnv->e_xfp != NULL)
			(void) fflush(CurEnv->e_xfp);	/* for debugging */

		/* if we are in the process of closing just give the code */
		if (SmtpState == SMTP_CLOSED)
			return (SMTPCLOSING);

		/* get the line from the other side */
		p = sfgets(SmtpReplyBuffer, sizeof SmtpReplyBuffer, SmtpIn);
		if (p == NULL)
		{
			extern char MsgBuf[];		/* err.c */

			/* if the remote end closed early, fake an error */
			if (errno == 0)
# ifdef ECONNRESET
				errno = ECONNRESET;
# else /* ! ECONNRESET */
				errno = EPIPE;
# endif /* ECONNRESET */

			/* Report that connection ended prematurely */
			if (CurEnv->e_xfp != NULL)
				fprintf(CurEnv->e_xfp,
					"421 %s (%s)... Deferred: %s\n",
					CurHostName, m->m_name,
					errstring(errno));

			/* if debugging, pause so we can see state */
			if (tTd(18, 100))
				pause();
# ifdef LOG
			syslog(LOG_INFO, "%s", &MsgBuf[4]);
# endif /* LOG */
			SmtpState = SMTP_CLOSED;
			smtpquit(m);
			return (-1);
		}
		fixcrlf(SmtpReplyBuffer, TRUE);

		if (CurEnv->e_xfp != NULL && index("45", SmtpReplyBuffer[0]) != NULL)
		{
			/* serious error -- log the previous command */
			/* also record who we were talking before first error */
			if (SmtpNeedIntro)
				fprintf(CurEnv->e_xfp,
					"While talking to %s:\n", CurHostName);
			SmtpNeedIntro = FALSE;
			if (SmtpMsgBuffer[0] != '\0')
				fprintf(CurEnv->e_xfp, ">>> %s\n", SmtpMsgBuffer);
			SmtpMsgBuffer[0] = '\0';

			/* now log the message as from the other side */
			fprintf(CurEnv->e_xfp, "<<< %s\n", SmtpReplyBuffer);
		}

		/* display the input for verbose mode */
		if (Verbose && !HoldErrs)
			nmessage(Arpa_Info, "%s", SmtpReplyBuffer);

		/* if continuation is required, we can go on */
		if (SmtpReplyBuffer[3] == '-' || !isdigit(SmtpReplyBuffer[0]))
			continue;

		/* decode the reply code */
		r = atoi(SmtpReplyBuffer);

		/* extra semantics: 0xx codes are "informational" */
		if (r < 100)
			continue;

		/* save temporary failure messages for posterity */
		if (SmtpReplyBuffer[0] == '4' && SmtpError[0] == '\0')
			(void) strcpy(SmtpError, &SmtpReplyBuffer[4]);

		/* reply code 421 is "Service Shutting Down" */
		if (r == SMTPCLOSING && SmtpState != SMTP_SSD)
		{
			/* send the quit protocol */
			SmtpState = SMTP_SSD;
			smtpquit(m);
		}
		errno = 0;
		return (r);
	}
}
/*
**  SMTPMESSAGE -- send message to server
**
**	Parameters:
**		f -- format
**		m -- the mailer to control formatting.
**		a, b, c -- parameters
**
**	Returns:
**		none.
**
**	Side Effects:
**		writes message to SmtpOut.
*/

/*VARARGS1*/
static void
#ifdef __STDC__
smtpmessage(const char *f, MAILER *m, ...)
{
	va_list	ap;
	va_start(ap, m);
#else /* !__STDC__ */
smtpmessage(f, m, va_alist)
	const char *f;
	MAILER *m;
va_dcl
{
	va_list	ap;
	va_start(ap);
#endif /* __STDC__ */

	(void) vsprintf(SmtpMsgBuffer, f, ap);
	if (tTd(18, 1) || (Verbose && !HoldErrs))
		nmessage(Arpa_Info, ">>> %s", SmtpMsgBuffer);
	if (SmtpOut != NULL)
		fprintf(SmtpOut, "%s%s", SmtpMsgBuffer,
			m == 0 ? "\r\n" : m->m_eol);
	va_end(ap);
}

# ifdef MAIL11V3
/*
**  SMTPSTAT -- collect status from DATA command
**
**	Parameters:
**		m -- the mailer we are reading the status from.
**
**	Returns:
**		status of DATA command
**
**	Side Effects:
**		none
*/
int
smtpstat(m)
	MAILER *m;
{
	int r;

	/* check the status returned after DATA command */
	r = reply(m);
	if (r < 0 || REPLYTYPE(r) == 4)
		return (EX_TEMPFAIL);
	else if (r == 250)
		return (EX_OK);
	else if (r == 552 || r == 554)
		return (EX_UNAVAILABLE);
	else if (r == 550 || r == 551 || r == 553)
		return (EX_NOUSER);
	return (EX_PROTOCOL);
}
# endif /* MAIL11V3 */
#endif /* SMTP */
